; -*- mode: Lisp; fill-column: 400 -*-

; Poseidon Library
;
;    Copyright 2024 Provable Inc.
;
;    Licensed under the Apache License, Version 2.0 (the "License");
;    you may not use this file except in compliance with the License.
;    You may obtain a copy of the License at
;
;      http://www.apache.org/licenses/LICENSE-2.0
;
;    Unless required by applicable law or agreed to in writing, software
;    distributed under the License is distributed on an "AS IS" BASIS,
;    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;    See the License for the specific language governing permissions and
;    limitations under the License.

; Authors: Alessandro Coglio (www.alessandrocoglio.info)
;          Eric McCarthy (bendyarm on GitHub)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "POSEIDON")

(include-book "std/util/defval" :dir :system)
(include-book "kestrel/crypto/primes/bls12-377-prime" :dir :system)
(include-book "kestrel/utilities/strings/chars-codes" :dir :system)
(include-book "kestrel/utilities/digits-any-base/core" :dir :system)
(include-book "std/testing/assert-equal" :dir :system)

(include-book "main-definition")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ poseidon-rate-4-alpha-17
  :parents (poseidon-instantiations)
  :short "Instantiation of Poseidon with rate=4 and alpha=17."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is the instantiation of Poseidon used internally by the
     Aleo Instructions opcode "
    (xdoc::ahref "https://developer.aleo.org/aleo/opcodes#hashpsd4" "hash.psd4")
    ".")
   (xdoc::p
    "Note, there are also some test cases for this instantiation
     in the source file @('rate-4-alpha-17-tests.lisp')."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; ark and mds matrices

; The ark matrix is #rounds rows, and t columns, where t = capacity + rate

(defconst *poseidon-rate-4-ark*  ; taken from .snap
  '((1938618153915392443680844598029810201246194507135996901458264098669274389515 7351729842755538537391125964696321872163870679345690009197692207415990728432 5322144518448602363357169748539649172236272734210869298302105099084309600758 157594388994230771084683648601469840681064412313308311327272295370248347612 5639176816628895275736144020835886044648205703435804661961201905291731964545)
    (1515456138335820402784095676684597756848991357758638903789140848174978116603 2936051880235014750924206093697335836731819189713569794758791546397377103064 3982581465619521168327036341762874184668315537525925987776783752265957181087 2506505397809231573773787628127834226357425846974915238736233851897679282829 7139378454973186236678009358628506840020431728798087188334720652889512319321)
    (2672593735906256483274755668154442470451697810351046267543062576218107414894 2353368021479564160925339643854728606488481970481557070527078401325054539364 5691364736634173285436567852506761312620025325446808579488786684240087041834 136935382297784730137384027603684742037750282653961406753018258777600202766 3619733952002925577526308765183862770871222143611658883115710610166174857456)
    (6669335020756020583589996764039362573936298668797586832439184940226940533437 7224263473890734518293803974695778331551656399854584612082807051474384732497 878240984036144199463468021469884131738324054393558565616921364838796693570 4520219542424737471234608257762615161661305358801424153418498997585660043682 6257385043754857961933710956289215789885093369353337848289903721185972124661)
    (7722630564304610309893791898336718938449688249091203949435297169436356466266 2554814336757862021073797014358571170101576233780808500831626073895431584367 4930759632764889219512688006565462023125202677839065865513077146644786069850 468541112608873801652994277198486757925617982477070849932268935688113034494 4583518460845467045809818528159015881280654822428007456110690588868691254613)
    (6108251737497111515165863780617435271937392447852939470953976370394349961296 7328384035623257574717178746712148304270375502185470258511028780791741060448 2369324994771921885556804522488383544279654736121961270884089539144785033608 2151342666413447238146429940306992237409500980700954880051341448031523631433 863220241038468802365023739369761114023207881141882804926012667186164534622)
    (7777908478055067154391735789968668310060388087280430713858280428728194665376 1995208855540635375089233314180032206712375406736116896528147189148620134319 3898022033572870201154904326008138439102829292912330873622396258301319883942 7721841559280545500253616715965585157608953453705335192411507307323128639245 7565647190655120080497305591587433997402260553239222792121981354427619558583)
    (714498723399445528688990682068639963262041067311215147289115131114496694288 1201400113720339576933992623641675231460532144217840641880145992767466635807 4894209964628724978971841560168369034430131652823257288001514201466224631907 5571575625600181141320663294520967276031024821399490684331740970313983562144 1376228297175617511841126084804452735658593653085585622296114548900886857413)
    (8113173201765304544311621731669294708333042330342242200443628631958786835430 3181202283077993561999902037098810455369029599193517279217633669177161783854 1457312626973841564070163789588432398685784744124958730966967800333674183388 6372113969918197796824457270685801016804679920496549209227270029811911680590 7522581358509145330469860909789542109514209301566817273421127040503709173860)
    (3472525889042953652862718589813143478709904441201932209082108605094254177024 6993289404603321724192176191155104008370468993796236695808240497648234596528 6548545212332686278673925672204546800851726271573314950769535197444758438540 3671446406605645718094291339770807162183644456740535682423131591738653750206 315133824263675918948112617628708663602348888752080999692906512097789043087)
    (7800209345835583355574295774780079600387437339098669103652283981812601728335 7547561662738277852672993297407687470390416724884808579358733069275618554404 4773883145277755181106722075107017010197176923686921219495662978555375573825 4192852815610127808244266301479620401977051262107385097973611519866339008468 4485831831393719753046272899261373805859545917283615732123343572665791052324)
    (6407969893740232104982533118493840811594343386748469571240787199904929885631 2005173139996573326346576800184097559399515741825656144523492599447277166628 7432129259176723417961489699072202470385810823193460235814166894879229006062 8405473886204550244131509466594940813416622357204428734735933166645220716738 4127543586911873806989738861220940866547879625493352425950653014547375926928)
    (949181468914232789290536147124216346019395412280168186950416868369660576528 2918669069767046769589654315210740879624487257618544958600173482216771251960 4748660784715393917784877414347849580386972503643672200367532855060916432403 7314201113723651485987752891007448470789951371096295736286861202558874215499 3326792833828883967764613552296220625711877738402458317133949570037653152487)
    (1196254366965590843491276883001195878006128340124344564285643152653886964122 4941723893786534745387070447063115432834312239175316712453377435494664839610 4267288481904346845505273124914716197553501085254827134316857137621724993526 4405048718379745116629791488238886360259531654308764958975827406613413802565 5806258601096850159401697848417607707297318803276571418857078666098139142922)
    (1142652404983113504959345665774909690676869645037470072262460375365161777529 2469156289998701679405500252853205402773159589590705594173533354274639089676 5520920767289407001822022042403083896368479448002192312691082046217759510513 525038480158609621349927130104503875882748242797637751387345216307103458850 7240206135692535157203300053157292362623229352683029831014846270686341436776)
    (6597259260520012453412799357941073025947719407499471273148435647097893899539 8432950049662506563323423617421840532010141663904656781383942117683961462569 5340220001933804563677331164592538148772473153453872586221349061743986836837 493372773077763485424261132966715366497194757398147747451077310132258777395 334846497659109511902444085009697629184965484227628964917938419603287887403)
    (2021734840341433466981353395003775740524056341773959091921107132401511236713 7913028361910512450009425329286881184369701190298507149110677951284307072892 5318232569439073586919969422645520367401222433979698678905796865755210901979 5996688267414509019348413609673910145079385751325297288045281613032511770877 1002579410398318817257730705125272046089269532068423750698770979964791185831)
    (2705372643444892184580612938717717347838388163636156523658662346705203150312 1704462027493616612564326106284808876408761027201538637672064348889252552692 555127360928775216527971571024579127266028105585051676536492283181344967074 4348609327740355360008617432010868414658570463151336643889730763661272980785 4818770564208602324589770860903787669021194357518974849955012924000660951152)
    (850599884752318299988520861074089681061564127438707245719956659787620596937 2362192310594034731721441581123093976505768250629201620870715986941613064362 2889527598315746509563200105326520800632073285034093762431340987521258588881 947327798182688457223797357128046862733711075480512278828642973210212544633 1144036210215784269087147254749866704140836729025130431432764809802274956218)
    (6553501849005902991200552098908239235292799913107042806218732335761199620342 8025934527878550482582131423044626913769098999930086164975526028725493917584 3049789983894392393292299170967551795890421552729750340563504776554236317539 2652718100309923168894932136480337347412496569871509043575421695746465783853 2892692620314386148518505911228778106240849866723527278204493226898400602119)
    (4247000862961931135186744256215210984154859769385486173483687995912490952141 2499689907492111008178055140600176654342790360785049525469121425773073785114 1116825612183033518568058121311202516791457102168369476372691543648751258327 4620192216404876299434161727614729455412011611197909849981818996301376782387 6200117155226958749643320496764856157895391133828928222285113845946916314306)
    (2553534285822320757467865579925037100529727076354481604215860565224265625763 7394128603792518671184361248310008990735371983827267056141559258356510636462 1234613964317912851906840494699778108017150820337810238167583050065298945527 1768859635142884128160939309516241019302867344334404696587205040505871355221 5989726159927287977407609642400527976647760368632461701582818876520979782701)
    (1219292973343103793718000071811972896623735275315626729876171992922643959481 403199779641232632906552260524240913820131591394483729532698606945228677456 557251576931282129101422080134370453355437210211783254506315241445906799105 2353025239610462745206982082929028046798218737188338140857212371157314631896 3580919253162018862282679291246608562325921138607135256891545058677434128949)
    (4549401701367978889624268488678941870881498911154980497819621451301482553497 4480351501343409974816887943563956067622022222740565263437653650968677815207 6545676944830470479018816792551043304199567666111742442513493248477397468926 2425972693278403085841401382761053804160590006892437925593346089581442906052 2258184811330038805015151163167399793317876095709569826955271186178225796840)
    (784283472002454912586343596637487644480779843031910117919452770638654045023 4333582020609153697174336954090729010239377447136246401227029461342839760237 4835909182628242706107864527691921325093653290285334771102971766201671309289 6164704864045583339634304626602115458373676068790549720191011478788673991449 2921559117716344229039010884108767752617936112353336113450842239424848456826)
    (2678155387149595911101798435772846820280479433295416163982913561739469905820 1483406942551225691321465673211295530012562669022805644366669169277194254124 3213216822889702600812476465015510913964832976232636906338047248991780355259 6691320390844159790493140242140828852785545549550467786944276341266261918113 6270350050959904709265493042490315336402786873089254672294218583544499514794)
    (7017106078220981756525594241393930562774134899407580742068242628475239362417 3987305590933224637460471264974622742124208897747244671074228566639019780778 105360269830260251345018099936836848150201953795020282437450298283971902412 28768629182341016643340760917836967777287799609954030622153320421329869662 1096177310737123536667020945609886600392104144858491426688581659928817169534)
    (3485434642724788092080138080374040834696464777037366106772304848908374013671 304070935614896700795714893597569128680528408455120134198724952315777103319 6715182075840684667404166109407177123404808256739721296373770732556780148612 5373240419739384032529167716189496893702328880837103111109932787478951042302 5954700982236015357691995821035368793174912084564876055500832653786857234799)
    (7453628884172222157610230195392638817139675423066487059662460642684897847415 1250934643351768889406912325067030821248651720515830925108057746894456624970 7562558558509242971209011300463263458351282528662730855585337264519952864135 6884777266313655560716983976779582596433155683678632491850562872829590613889 1426435994731429094755057714577026647564700549794239221174318064053241647324)
    (4718697696585232832254058555625450439215027600827179701357292079371111095444 968222032310589567931845516127800975064323100532117229291162062941005955782 8257927836462015877587925755020317539340359260850652406711240533795709261244 3627707786055012571902258718150492692625194597390304044841293203740351161749 6459955861029881123422771563878472649793354225247416499437695452078640122809)
    (8382057728027982490855584899105323317369353112238223280029043404425186903026 8075421234021802963960125029019339567976041199728157350479378795987910590479 6785669174782078815011277810115644218187873371373625512276267614866754562556 594576217658177877048983021543276128381676596942347646553722906204580617166 1357028126216065618592453537845119588028562797210257916051153693622525448162)
    (1268728026871602653116947688389406592721209658561183371981372092205859606696 2450822316259134428544097962064557374097559227312442154233180332865507872204 5984879519022278834752952255953534710277756772586965995681999900200707610845 8314134236239244039700507053527103895406097435887993881523296399158807904235 5305926835145851291737541628153236752643953880883685203811342443433150224182)
    (6369622090136458777956551199963887877819736910644287042551120222778198024236 8270064914898960396871936593843846156613617820754754191636958462117709267766 8074047548622496822259775011268028495773405300528930802332170314620900442850 3773177164259493386502608859098369555998989692028300423038189592792703271440 3224355514045501265822953252068447663773477137469154316413229418482567960871)
    (2077297725657990210108011265533621423733027406387796151276965617701613762754 7671204003033509453409834862892233870869025764348622790183090345240816843700 1307242050436903920309864835716214995364651539755984695080746895346455724738 6129121335886836095968088472569313034551266305978026561253067791853210407922 6660751762669205386269976325140259706537233897506045722296419208963236650048)
    (6025410611216876921660043157814698670370769549573180785097049111849154490267 2879034513207406300722551387722874123940024511477893980119991020485604334111 7873434927237958329776429419267126264781079363241055621685089858891312474143 4067179546164874962146368835536429691369013885021840539272335225189382526772 7379540440940491301471808991627579895021636500530268455708765738411485103452)
    (1836956214540280652900893672816768705790295898274601183803318869036293505685 2572081759924393942091043729315503264917976108911264716201970132299007643631 1810556740986779921360330339566917216451545415179354120717333942307040757348 6599351112491716463871357902252945533102329163414169280333117368865501830548 1921152951736850675935986677819988308056201199352117054435394594560076112429)
    (6655194612221289821534392480814067994249833646941647976483663447605643216895 5347130057746530685136007928838444602754793948468101847232777508631223836374 311402716634024524248888525171599798612306854505294445783178936491788900458 1000706111233179777477365845872113733720425906341094396024248988167901245041 5995706326122729967329587857402159169764605621082672778821560506731601159816)
    (6604576835629750321261580434422731747043990440247943920270259832177935148172 3848261365107307823304059006891252148156401021825652740877268944891374790373 5567835015363012730097176418581376730292678910065670502635035451809633539793 8205011068960775905037067753373832764332321370611065412234592298480171737790 7842518011152463064529135198301135953088574639551347384804624097508967231276)
    (3229681352867613369487227888546482073992641858866474804011175170774927169355 7463230372047900501667578368327409067813758189527819814651023253082209986807 3679946147402146323532840325045817061654316784146805845579318895180807172372 6650701560897883970058100541536458451564369991282826711293707000526246096581 960263481819073580430660625296237082394673388836804874492991172157780648209)))

; The MDS matrix is t by t, where t = capacity + rate

(defconst *poseidon-rate-4-mds*  ; taken from .snap
  '((8329930521539134039137773392305942487936411634375145971571337914339858953494 278618399552420908809268472238185011462303036479822336003895720847977164374 6268845459389261230018660537668447144447697457946393613953333232198852302660 4793736973433064023122993260797265604206973061643230352208198232828483071003 7010297407853133946224842241325439897662363979865403279371873306755772971989)
    (1148061982653853851068598249615224324504469915643249863017312498536299431197 3834733112939423196050814126244394935155240314078231119823007202778770783538 6731041659203810709196890592188739803945948288312701180512069874327488791023 7945040885943028241700077811804123684851187973082672605799187423678933683097 664113875828674730172077107648407052540591002172341478825704668451795733709)
    (958329444627360057553789790139505520713768597825787905126296764575407842086 1674720908123841295069924925076253392755422630443343951404557347508062476751 7579037503653117732004438867283621215019838810821215969441691059746969237825 4661803141786797747014771083515401817401242280802589352312338052692982089125 2000687660843528256987469891326391023151619051611797488140886363956491692637)
    (1566279381105619741037744131780164213388085730197907125355449381455246435029 5937113098661631221124558176711231482296322706893494182011399472288707098869 7611048118133074344915545590703681762699629288705991157139243067623043401312 965821776879246959795943372802260593723788208223359972604352458788930648590 3183183958806947344092149278940679926778334598736166724628930507444153486021)
    (3254641949675743550586469813187621560773040216186078387819273543796198648127 5934234919226890431372073088080318440843529615216780917716779477738520110526 4321959385295338413565335564489250917518341461988447343488303090937674694131 3934255152967998234825863298301587762651088256904340083647424116555247616707 5720240656559554839605510963541920291004658627897673664359222337962412895726)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Make this a nullary function to make it easier to prevent expansion
; and easier to view in the debugger.
(define rate-4-alpha-17-parameters ()
  :short "Poseidon parameters for Aleo circuit rate 4 implementation."
  (make-param
   :prime primes::*bls12-377-scalar-field-prime*
   :rate 4
   :capacity 1
   :alpha 17
   :full-rounds-half 4
   :partial-rounds 31
   :constants *poseidon-rate-4-ark*
   :mds *poseidon-rate-4-mds*
   :rate-then-capacity-p nil
   :ascending-p t
   :partial-first-p t))

; Make this a nullary function so the meaning is more obvious in the debugger.
(define rate-4-domain-fe ()
  :short  "Domain separation field element for Aleo circuit rate 4 Poseidon hash."
  (acl2::lendian=>nat 256 (acl2::chars=>nats (acl2::explode "Poseidon4"))))

; Regression test, and have the constant recorded.
(assert-equal
 (rate-4-domain-fe)
 967188381303320047440)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Interface functions.

(define hash4-many ((inputs (fe-listp inputs primes::*bls12-377-scalar-field-prime*))
                    (count natp))
  :guard (fep (len inputs) primes::*bls12-377-scalar-field-prime*)
  :returns (outputs (fe-listp outputs 
                              primes::*bls12-377-scalar-field-prime*)
                    :name fe-listp-of-hash4-many
                    :hints (("Goal" :use (:instance fe-listp-of-hash
                                    ;; this :use hint must match the preimage calc below
                                                    (inputs (cons (rate-4-domain-fe)
                                                                  (cons (len inputs)
                                                                        (append (repeat 2 0)
                                                                                inputs))))
                                                    (param (rate-4-alpha-17-parameters)))
                                    :in-theory (disable fe-listp-of-hash))))
  :short "Hash any reasonable number of inputs to any number of outputs using RATE=4"
  :long
  (xdoc::topstring
   (xdoc::p
    "The inputs and outputs are field elements.")
   (xdoc::p
    "This interface function prepends RATE elements to the input to construct the
     preimage: [ DOMAIN || LENGTH(INPUT) || [0; RATE-2] || INPUT ],
     and then calls the main hash function.")
   (xdoc::p
    "The number of inputs must be less than the field size so that the length
     field is expressable."))

  (let ((preimage-and-inputs
         (cons (rate-4-domain-fe)
               (cons (len inputs)
                     (append (repeat 2 0) ; 2 zeros to pad to RATE=4
                             inputs)))))
    (hash preimage-and-inputs (rate-4-alpha-17-parameters) count))
  ///

  (more-returns
    (outputs true-listp
             :rule-classes :type-prescription
             :hints (("Goal"
                      :use fe-listp-of-hash4-many
                      :in-theory (e/d (pfield::true-listp-when-fe-listp) (fe-listp-of-hash4-many))))))

  (defret len-of-hash4-many
    (equal (len outputs)
           (nfix count)))
  )

(define hash4 ((inputs (fe-listp inputs primes::*bls12-377-scalar-field-prime*)))
  :guard (fep (len inputs) primes::*bls12-377-scalar-field-prime*)
  :returns (output (fep output primes::*bls12-377-scalar-field-prime*)
                    :name fep-of-hash4
                    :hints (("Goal" :use (:instance len-of-hash4-many
                                                    (count 1))
                                    :in-theory (disable len-of-hash4-many))))
  :short "Hash any reasonable number of inputs to a single field output using RATE=4"
  :long
  (xdoc::topstring
   (xdoc::p
    "The inputs and output are field elements.")
   (xdoc::p
    "This interface function prepends a domain-separation field element
     and a remaining input-length field element,
     and then calls the main hash function with an output count of 1.")
   (xdoc::p
    "The number of inputs must be less than the field size."))

  (first (hash4-many inputs 1))
  :guard-hints (("Goal" :in-theory (enable hash4-many))))
