/* ----- core/weightsntrup653, derived from supercop/crypto_core/try.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ntruprime.h>
#include "ntruprime_test.h"

#define fail ((ok = 0),printf)
static const char *core_weightsntrup653_checksums[] = {
  "fbbd83b55763dff9eb34f7e04f545b2217b39a45cf6b291c2a241df017978030",
  "2509dd9d3711537b4946ebd79de22310103ffa1c7f48d049011a2384ac152f37",
} ;

static void (*crypto_core)(unsigned char *,const unsigned char *,const unsigned char *,const unsigned char *);
#define crypto_core_OUTPUTBYTES ntruprime_core_weightsntrup653_OUTPUTBYTES
#define crypto_core_INPUTBYTES ntruprime_core_weightsntrup653_INPUTBYTES
#define crypto_core_KEYBYTES ntruprime_core_weightsntrup653_KEYBYTES
#define crypto_core_CONSTBYTES ntruprime_core_weightsntrup653_CONSTBYTES

static void *storage_core_weightsntrup653_h;
static unsigned char *test_core_weightsntrup653_h;
static void *storage_core_weightsntrup653_n;
static unsigned char *test_core_weightsntrup653_n;
static void *storage_core_weightsntrup653_k;
static unsigned char *test_core_weightsntrup653_k;
static void *storage_core_weightsntrup653_c;
static unsigned char *test_core_weightsntrup653_c;
static void *storage_core_weightsntrup653_h2;
static unsigned char *test_core_weightsntrup653_h2;
static void *storage_core_weightsntrup653_n2;
static unsigned char *test_core_weightsntrup653_n2;
static void *storage_core_weightsntrup653_k2;
static unsigned char *test_core_weightsntrup653_k2;
static void *storage_core_weightsntrup653_c2;
static unsigned char *test_core_weightsntrup653_c2;

#define precomputed_core_weightsntrup653_NUM 9

static const unsigned char precomputed_core_weightsntrup653_h[precomputed_core_weightsntrup653_NUM][crypto_core_OUTPUTBYTES] = {
  {50,1},
  {58,1},
  {81,1},
  {56,1},
  {0,0},
  {141,2},
  {31,1},
  {32,1},
  {33,1},
} ;

static const unsigned char precomputed_core_weightsntrup653_n[precomputed_core_weightsntrup653_NUM][crypto_core_INPUTBYTES] = {
  {2,25,191,134,80,22,215,57,96,83,181,36,184,221,54,9,147,89,121,52,110,71,232,247,96,143,82,49,88,43,206,92,135,206,197,216,25,177,27,182,76,188,205,53,247,201,192,153,247,52,25,141,195,98,228,148,109,66,130,123,14,23,158,131,144,107,64,150,45,255,230,170,172,16,240,38,246,70,26,244,142,4,157,114,213,54,145,34,165,69,48,202,77,61,13,84,155,79,58,22,159,164,24,40,15,48,199,185,89,200,77,40,108,238,182,243,4,67,79,242,174,215,130,79,198,103,211,121,54,42,126,239,102,22,167,156,95,29,158,15,109,93,109,53,47,192,133,47,145,244,242,234,250,68,118,33,216,165,26,27,5,82,74,63,144,6,29,14,254,142,248,79,174,239,234,46,125,4,53,132,16,216,142,7,239,182,168,104,185,36,192,129,105,2,228,223,157,1,35,186,225,221,220,60,254,131,242,31,240,91,32,152,54,128,96,102,182,27,6,60,10,149,137,38,188,48,176,241,248,41,180,202,132,22,29,12,200,189,221,233,76,161,18,54,100,104,130,47,114,149,126,184,227,252,188,105,147,192,145,130,34,235,135,83,21,95,98,200,33,141,129,166,247,167,151,93,150,83,81,123,33,137,136,98,244,9,177,175,42,210,55,229,65,80,24,160,1,236,220,146,54,48,116,215,38,4,189,217,64,28,71,251,141,238,246,128,240,215,77,178,153,63,206,86,108,82,42,90,246,168,108,247,68,61,47,23,21,86,121,136,122,141,200,103,41,97,90,88,49,243,204,192,80,6,101,162,218,229,244,111,178,57,43,191,3,202,103,88,224,111,134,1,167,184,152,208,96,2,217,232,251,128,218,211,87,64,70,72,97,118,244,203,143,75,219,122,150,58,243,215,16,58,172,103,135,104,39,213,149,207,121,6,185,40,16,24,162,75,91,148,122,16,116,174,73,160,60,206,64,244,66,58,198,214,26,253,120,34,117,153,190,244,8,7,227,114,119,135,136,80,203,163,153,66,23,83,186,106,238,22,129,183,191,10,253,60,17,45,206,26,55,119,107,148,176,125,242,113,200,75,136,159,253,250,183,115,59,162,26,80,212,19,17,189,185,205,44,160,247,114,166,239,29,124,72,54,169,36,172,0,150,32,147,46,12,14,38,30,116,206,142,83,220,57,67,82,192,184,115,43,170,227,180,194,36,189,236,96,72,1,207,15,240,158,5,185,2,173,91,247,105,182,107,6,151,183,87,105,211,20,25,82,99,180,147,56,103,183,252,125,199,37,216,150,225,239,223,38,11,80,2,104,90,114,174,32,246,170,56,211,149,97,162,49,241,227,15,102,248,180,64,113,212,93,225,71,98,55,155,143,106,197,31,109,176,183,109,174,83,48,51,84,29,52,222,175,225,116,133,62,17,42,113,109,146,201,119,3,114,181,192,59,57,101,28,207,169,136,218,123,127,217,30},
  {208,201,3,34,64,214,74,123,47,70,184,151,119,136,29,188,169,2,66,29,218,201,122,100,208,1,80,34,226,15,163,244,83,104,43,225,24,185,161,91,213,96,181,155,27,212,135,227,59,222,67,197,209,181,152,51,71,215,116,246,79,110,213,223,174,116,78,127,84,207,198,112,213,39,104,67,201,15,151,173,159,177,90,48,97,129,227,201,167,129,122,40,79,238,185,229,61,64,74,94,67,180,6,23,50,191,121,109,219,130,71,160,165,94,214,17,149,115,232,213,32,14,150,85,35,53,175,109,203,5,240,76,184,108,182,142,62,228,178,21,164,13,5,223,210,18,201,202,225,53,114,249,152,244,74,93,109,244,71,227,244,195,190,253,237,82,191,226,24,164,225,150,162,64,40,4,107,66,246,239,244,79,198,222,140,174,75,213,177,255,48,28,136,222,154,98,166,162,128,130,209,220,244,172,201,124,123,64,188,162,48,42,120,142,83,78,150,140,27,60,186,40,32,52,231,207,36,241,160,140,241,116,166,174,67,60,5,86,223,72,4,230,129,236,63,27,28,163,10,66,251,81,144,203,100,207,33,45,224,15,217,223,207,107,146,126,158,114,199,213,227,222,86,122,16,248,80,166,22,80,19,143,133,41,146,81,154,106,205,123,17,196,113,202,85,235,248,159,112,237,114,133,41,121,206,207,0,12,54,170,149,156,13,20,128,133,242,33,226,50,104,205,17,125,25,210,44,35,93,156,202,252,65,45,180,246,8,105,97,59,0,223,201,128,127,40,198,108,167,210,140,25,155,64,156,35,125,172,124,237,35,196,162,135,114,97,31,55,170,170,124,47,75,82,132,182,1,194,107,117,158,115,224,58,254,167,178,241,229,252,191,188,235,50,122,149,39,18,0,35,110,42,174,50,62,38,38,175,237,204,114,30,198,139,94,82,249,32,231,175,223,204,193,239,26,29,208,37,195,216,203,38,96,0,189,189,193,213,135,66,133,82,108,66,233,163,51,116,253,248,14,73,187,203,94,242,222,109,25,226,138,111,172,100,101,129,252,197,5,138,145,93,218,83,41,93,121,32,19,107,244,93,136,160,151,176,246,111,207,232,52,189,36,250,32,33,43,138,24,50,195,100,60,223,130,95,80,95,242,97,118,15,202,108,31,67,127,198,90,178,79,176,178,199,31,86,35,204,40,216,246,35,121,25,5,142,46,197,161,69,51,134,94,73,149,243,60,83,8,42,64,142,116,71,105,115,249,226,240,101,178,55,133,237,140,112,51,103,253,238,100,39,16,82,102,37,223,93,252,229,76,83,103,134,2,101,211,50,214,239,66,200,147,41,198,13,217,184,251,163,104,112,13,213,94,162,123,217,168,55,46,24,129,196,144,159,21,228,228,4,96,253,169,162,14,186,156,206,59,238,5,119,50,213,156,230,5,212,22,113,69,207,46,31,255,225,127,252,161,140,226,189,110},
  {48,14,224,167,129,150,182,199,169,233,105,95,129,107,12,85,26,22,36,65,91,87,17,51,167,202,182,185,65,241,69,113,74,129,20,194,55,151,7,210,121,207,22,229,4,110,254,201,77,209,17,247,51,61,251,254,188,195,63,49,172,57,17,44,145,117,156,238,99,107,211,179,67,11,209,135,157,120,243,2,143,117,201,47,212,120,255,226,125,65,170,96,239,84,122,91,237,212,215,138,251,153,189,168,186,112,170,124,63,94,45,88,165,96,53,61,151,62,134,101,247,79,24,252,242,136,18,178,172,231,53,89,45,185,44,62,163,85,229,183,24,128,216,45,124,121,106,194,66,218,23,36,26,232,87,23,189,27,62,119,56,195,79,138,236,219,36,236,101,99,253,216,122,228,151,222,157,194,222,32,152,10,81,214,172,109,167,134,224,98,7,62,249,35,233,33,167,227,223,214,11,20,182,81,119,18,40,83,33,71,80,126,164,120,112,28,41,37,48,96,11,1,163,128,250,85,186,2,40,49,207,214,97,151,21,130,46,198,57,17,92,192,203,246,234,217,44,231,129,103,145,229,51,230,194,81,83,139,166,48,11,22,239,240,18,0,165,54,187,147,125,120,226,241,45,26,38,93,65,83,237,105,235,190,100,40,1,157,128,27,96,224,255,31,129,93,128,170,168,101,146,191,203,171,241,120,178,162,52,151,75,41,108,29,203,10,250,177,134,216,86,185,101,25,191,93,87,67,101,229,186,131,3,108,206,41,129,37,135,150,66,234,203,1,76,86,49,206,115,245,52,203,190,105,238,204,229,38,191,93,134,152,211,90,120,155,32,63,112,166,239,2,193,29,79,26,18,123,9,68,116,173,161,155,81,62,76,85,50,40,233,219,36,175,129,232,140,17,73,69,120,46,222,216,8,145,123,142,39,208,73,222,206,44,135,177,177,23,124,218,60,32,176,208,162,37,144,12,46,56,183,39,39,246,101,109,182,42,89,196,195,127,188,211,224,102,211,59,78,203,112,26,20,173,193,23,72,163,3,210,219,228,227,239,228,254,227,74,190,29,64,112,138,146,235,32,184,195,93,36,15,44,115,91,131,250,190,104,171,36,69,71,138,251,211,175,0,92,188,94,249,6,1,44,114,6,37,117,110,241,152,245,36,118,204,168,52,52,198,211,36,140,52,21,230,77,16,114,99,45,98,211,241,83,10,145,57,58,25,118,48,27,213,63,5,234,198,19,67,104,35,85,193,169,249,182,236,41,150,183,43,139,115,108,3,164,126,173,4,39,130,79,178,165,240,102,166,107,139,176,168,91,227,145,144,104,181,213,106,28,77,2,54,56,217,69,32,13,163,77,97,58,227,45,67,96,78,112,80,23,86,140,66,188,59,146,135,157,244,122,118,149,35,125,0,179,98,90,254,157,11,98,75,148,170,80,130,121,10,73,124,92,12,8,209,3,66,253,3,61,202,250,66},
  {215,30,199,1,146,179,169,243,31,218,96,53,148,186,63,187,108,80,61,98,222,242,142,98,129,68,77,77,94,67,61,2,97,131,173,94,86,200,163,33,125,98,19,110,143,44,112,140,216,217,90,20,185,47,108,73,207,230,68,158,16,12,199,135,118,247,199,191,11,97,14,231,222,5,151,6,57,50,52,5,254,145,171,207,42,29,180,83,93,86,125,197,21,229,183,212,8,19,156,81,217,217,66,23,109,101,40,245,139,193,140,87,248,228,50,230,98,154,220,133,87,36,21,23,132,46,92,106,46,198,242,129,62,140,143,34,150,153,196,75,188,51,213,113,95,63,126,44,100,170,18,98,14,243,208,106,220,100,118,244,101,221,79,196,198,143,63,251,66,44,68,72,137,188,129,76,97,47,146,154,227,185,15,197,245,74,201,87,249,187,244,92,219,38,109,137,227,251,16,233,183,7,109,49,112,172,212,15,2,153,90,57,192,75,138,118,10,214,50,22,148,121,90,70,50,223,226,94,242,169,61,224,135,220,94,77,197,10,236,109,233,94,123,31,54,109,44,12,236,125,59,150,138,2,85,28,42,26,237,104,86,232,19,175,80,92,167,38,112,154,116,159,72,142,176,177,186,74,151,192,190,164,147,250,40,41,37,114,130,241,16,218,217,110,27,157,226,242,4,188,190,18,23,242,63,135,87,29,224,112,216,137,146,91,245,69,250,254,123,49,178,215,237,199,31,51,58,205,112,140,209,25,0,206,6,26,131,200,207,221,81,124,125,167,118,92,187,138,141,231,9,3,224,232,117,80,240,64,77,130,12,94,250,105,116,184,22,204,121,18,85,63,112,70,15,141,84,219,218,96,160,60,188,6,78,215,28,175,192,226,210,114,77,159,238,107,13,44,191,3,104,82,120,233,31,132,92,25,119,77,21,38,83,42,169,197,39,154,51,63,187,173,240,221,177,130,253,221,45,249,42,211,252,233,119,36,176,156,153,241,196,223,209,128,32,51,166,242,52,145,37,162,45,1,62,51,0,235,94,212,130,130,69,140,232,210,122,135,24,207,236,7,217,185,245,56,52,104,95,70,81,87,95,220,103,253,52,100,116,127,205,50,49,250,153,120,117,18,74,96,99,219,119,8,201,116,14,252,176,40,220,11,64,51,116,46,146,213,197,217,55,226,179,203,153,110,100,5,50,71,49,116,33,73,185,236,160,129,21,72,102,151,128,21,252,241,136,193,70,45,183,183,5,168,178,137,120,48,242,40,175,118,29,90,95,217,136,143,130,59,156,210,74,40,137,167,220,169,93,243,226,41,226,150,47,98,99,20,75,85,170,166,76,206,27,181,184,152,6,13,180,141,69,96,65,136,244,116,12,64,108,146,91,166,94,214,121,161,45,193,73,124,216,102,83,85,187,239,194,244,227,245,164,213,250,163,47,208,113,214,182,238,98,237,165,22,215,159,36,81,47,232,42},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
  {255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255},
  {1,0,1,0,0,1,0,255,0,0,0,255,1,0,1,1,0,0,255,1,0,0,0,0,255,0,1,0,255,0,0,1,1,0,255,0,255,0,0,0,255,0,1,1,255,255,0,0,0,1,255,0,0,0,0,255,255,0,0,0,0,255,0,0,0,0,0,0,0,0,0,255,255,1,0,0,1,0,0,0,1,0,0,0,255,1,0,0,0,1,0,255,0,0,0,0,1,0,1,0,0,255,0,0,255,255,0,255,0,255,1,0,0,0,1,0,255,0,0,255,0,0,255,0,0,1,255,0,255,255,0,0,0,255,255,0,1,0,0,1,0,1,255,1,1,0,0,0,1,0,0,0,0,0,0,1,255,0,0,0,1,0,1,1,0,0,0,0,0,1,0,0,0,0,1,255,1,1,0,1,0,1,255,0,255,0,1,1,255,1,0,0,1,0,255,255,0,255,0,255,0,1,0,0,255,255,0,0,1,0,255,1,0,255,1,1,0,0,0,0,1,0,0,0,1,1,0,0,255,0,0,1,0,0,255,0,1,1,0,1,0,0,1,0,0,255,0,255,1,255,1,255,0,255,255,0,0,1,1,0,0,255,255,0,0,1,0,0,0,0,0,0,255,1,0,255,0,0,0,1,1,255,0,0,255,0,0,255,0,0,0,0,1,1,255,0,1,0,0,255,255,0,0,0,0,0,255,0,1,0,0,1,0,1,0,1,0,0,1,0,0,0,255,1,255,1,0,0,0,0,1,1,1,0,0,1,0,1,0,1,255,1,0,255,0,0,255,1,0,0,255,0,0,0,0,0,0,0,0,0,255,1,1,0,0,255,0,0,255,1,255,255,0,0,0,0,0,1,0,255,0,0,1,255,1,0,255,1,1,0,0,1,0,0,255,255,0,0,0,0,1,0,0,0,255,0,255,1,0,255,0,0,1,255,255,0,1,0,0,1,0,1,0,0,0,255,1,0,1,0,0,1,0,0,1,255,0,255,0,255,1,1,0,1,1,0,0,0,255,0,0,255,255,0,0,255,0,1,0,255,0,1,1,0,0,0,0,0,0,1,1,0,0,0,255,1,0,0,0,0,0,1,0,0,0,255,1,255,255,1,255,255,0,0,255,0,1,0,255,0,1,1,0,255,0,255,0,255,0,1,255,255,255,255,0,1,0,1,255,0,1,0,0,1,0,255,1,0,0,0,0,0,0,0,255,255,0,0,0,255,0,1,0,0,0,0,1,0,1,1,1,0,0,0,255,1,0,1,255,1,0,1,255,0,1,0,255,0,1,255,255,255,0,0,1,255,255,0,0,0,1,255,0,0,0,1,0,1,0,0,0,1,1,1,0,0,255,1,0,1,0,255,0,0,0,0,0,0,255,0,255,0,0,0,0,0,255,0,1,255,1,0,255,0,255,255,0,0,1,255,0,0,0,1,0,0,1,0,255,0,0,0,0,0,0,0,1,0,1,0,255,1,0},
  {0,1,255,255,1,1,0,0,0,255,0,0,0,0,0,1,1,0,255,0,0,255,0,255,0,1,0,0,0,0,0,0,1,1,255,1,0,1,255,255,0,1,255,0,0,0,0,255,0,1,1,0,255,0,0,1,255,1,255,0,0,255,1,0,0,0,0,255,1,0,0,1,0,0,0,255,0,1,0,0,1,1,0,0,255,0,0,1,255,255,0,0,0,255,255,0,1,0,0,1,0,0,0,1,0,0,0,0,0,1,255,255,0,0,0,0,255,1,255,0,0,0,0,1,0,1,255,0,0,255,1,1,0,0,0,0,255,255,0,1,0,0,0,0,0,1,0,1,1,0,0,0,255,0,0,1,1,0,0,0,0,1,0,0,0,0,1,1,255,0,1,0,1,1,0,0,0,1,0,0,0,0,0,255,1,0,1,0,0,1,255,0,0,0,0,0,1,1,255,0,0,0,255,255,0,255,0,1,0,255,0,0,0,255,0,1,1,0,0,0,0,1,0,1,0,0,255,0,255,1,255,0,255,255,1,0,255,1,0,1,0,0,1,0,0,0,0,0,1,1,1,255,0,0,0,1,255,0,0,1,255,255,255,0,255,1,1,1,255,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,1,0,0,0,1,255,0,255,0,255,1,0,255,0,0,0,0,1,0,0,0,0,0,1,255,0,0,1,255,0,1,255,1,255,255,0,0,0,0,0,255,255,255,1,0,0,255,255,255,255,0,255,255,1,0,0,0,0,255,0,1,0,0,255,255,0,0,0,1,1,1,1,0,0,1,255,0,1,255,255,255,255,255,0,1,0,0,0,1,0,1,0,255,1,255,0,0,0,0,0,0,0,0,255,0,0,255,0,0,1,0,1,0,0,0,255,0,1,255,255,1,0,0,0,0,255,0,0,0,0,1,1,255,0,0,0,1,255,255,1,255,255,0,255,0,1,1,255,0,1,255,255,1,1,1,255,1,255,0,0,0,1,255,0,1,1,1,0,1,1,0,0,1,0,1,0,255,0,0,0,0,1,1,1,255,1,0,0,1,0,0,0,1,255,1,0,1,0,255,0,0,255,255,255,0,255,0,0,0,255,0,255,0,1,0,0,0,0,1,1,0,0,1,0,0,1,1,0,0,255,255,0,255,0,0,0,0,255,0,0,255,0,0,1,255,0,0,1,0,255,1,0,0,0,255,0,0,1,255,0,0,255,255,0,1,255,0,0,0,0,0,255,0,0,1,1,1,255,0,1,0,255,0,0,1,0,0,0,0,0,0,0,1,1,0,0,255,0,0,0,1,1,255,0,0,1,0,255,1,0,0,1,0,0,0,0,0,0,0,0,1,1,0,0,1,0,0,255,1,1,0,0,0,0,0,0,0,1,1,1,1,255,0,0,255,0,255,0,0,1,0,1,0,0,255,0,0,0,0,0,0},
  {1,0,0,255,255,0,1,0,1,0,1,0,0,1,1,0,255,0,0,0,0,0,255,0,0,0,255,0,0,255,1,0,0,0,0,255,0,0,0,0,0,255,255,1,1,1,0,0,0,0,0,0,0,0,255,255,1,255,0,0,0,255,1,0,1,255,0,255,0,0,0,0,0,1,255,0,0,0,0,0,1,255,0,0,255,0,0,0,0,255,255,255,255,255,0,0,0,0,1,0,1,0,0,1,1,0,0,255,1,0,255,1,0,1,255,1,1,0,255,0,0,0,255,0,1,0,0,0,1,0,0,0,1,1,255,255,1,1,1,0,255,0,0,0,0,0,1,0,1,255,0,255,1,0,255,0,255,0,1,0,0,0,0,1,255,1,255,1,1,0,0,0,0,0,0,255,1,0,255,0,1,1,0,0,255,1,255,0,255,255,0,1,0,1,1,255,255,255,1,0,0,0,0,0,0,255,0,0,0,255,0,0,0,0,0,0,1,255,0,255,0,1,0,255,0,1,1,255,0,255,1,255,0,0,0,1,1,1,1,0,0,1,0,0,0,1,0,0,0,0,0,255,255,0,0,1,255,0,0,0,0,0,0,1,0,0,0,255,0,255,1,1,0,0,0,1,1,255,0,0,255,0,1,255,0,1,255,0,0,0,0,0,0,1,0,0,0,1,0,255,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,1,0,0,1,0,0,1,0,0,255,0,0,0,0,0,1,1,0,1,0,1,255,255,1,255,0,0,1,0,255,1,1,0,0,255,0,0,255,0,0,0,0,1,1,0,255,0,255,0,0,255,1,0,0,255,0,255,0,0,0,1,255,0,1,0,0,0,0,1,255,255,0,0,0,1,0,0,255,1,1,255,1,0,0,0,0,0,0,255,255,0,0,0,1,0,255,1,0,0,0,0,255,0,1,0,1,0,0,0,0,0,0,1,1,0,255,255,255,0,0,0,255,0,0,0,1,255,0,1,0,0,1,0,1,1,255,1,0,0,0,0,0,255,255,0,1,0,255,0,0,0,1,0,0,255,0,255,1,1,0,0,0,0,0,0,0,0,0,255,255,255,255,0,1,0,0,255,1,1,0,0,0,255,0,1,0,1,255,0,1,0,1,0,0,0,0,0,255,1,0,255,1,0,1,1,1,0,255,0,0,0,0,255,0,0,0,0,255,255,255,0,0,0,0,255,0,1,0,0,0,1,1,1,0,1,255,0,255,0,1,0,1,0,1,1,0,1,1,0,0,0,0,255,255,1,1,0,1,255,1,1,255,0,0,0,255,0,255,0,0,0,255,255,255,1,0,0,0,1,0,255,255,255,1,255,0,1,255,1,255,1,0,0,0,255,255,1,1,0,1,0,0,0,0,0,255,0,0,1,1,1,0,1,1,0,0,0,0,0,0,1,1,0,255,0,1,0,0,0,0,1,255,255,0,255,0},
} ;

static const unsigned char precomputed_core_weightsntrup653_k[precomputed_core_weightsntrup653_NUM][crypto_core_KEYBYTES] = {
  {},
  {},
  {},
  {},
  {},
  {},
  {},
  {},
  {},
} ;

static const unsigned char precomputed_core_weightsntrup653_c[precomputed_core_weightsntrup653_NUM][crypto_core_CONSTBYTES] = {
  {},
  {},
  {},
  {},
  {},
  {},
  {},
  {},
  {},
} ;

static void test_core_weightsntrup653_impl(long long impl)
{
  unsigned char *h = test_core_weightsntrup653_h;
  unsigned char *n = test_core_weightsntrup653_n;
  unsigned char *k = test_core_weightsntrup653_k;
  unsigned char *c = test_core_weightsntrup653_c;
  unsigned char *h2 = test_core_weightsntrup653_h2;
  unsigned char *n2 = test_core_weightsntrup653_n2;
  unsigned char *k2 = test_core_weightsntrup653_k2;
  unsigned char *c2 = test_core_weightsntrup653_c2;
  long long hlen = crypto_core_OUTPUTBYTES;
  long long nlen = crypto_core_INPUTBYTES;
  long long klen = crypto_core_KEYBYTES;
  long long clen = crypto_core_CONSTBYTES;

  if (targeti && strcmp(targeti,".") && strcmp(targeti,ntruprime_dispatch_core_weightsntrup653_implementation(impl))) return;
  if (targetn && atol(targetn) != impl) return;
  if (impl >= 0) {
    crypto_core = ntruprime_dispatch_core_weightsntrup653(impl);
    printf("core_weightsntrup653 %lld implementation %s compiler %s\n",impl,ntruprime_dispatch_core_weightsntrup653_implementation(impl),ntruprime_dispatch_core_weightsntrup653_compiler(impl));
  } else {
    crypto_core = ntruprime_core_weightsntrup653;
    printf("core_weightsntrup653 selected implementation %s compiler %s\n",ntruprime_core_weightsntrup653_implementation(),ntruprime_core_weightsntrup653_compiler());
  }
  for (long long checksumbig = 0;checksumbig < 2;++checksumbig) {
    long long loops = checksumbig ? 4096 : 512;

    checksum_clear();

    for (long long loop = 0;loop < loops;++loop) {

      output_prepare(h2,h,hlen);
      input_prepare(n2,n,nlen);
      input_prepare(k2,k,klen);
      input_prepare(c2,c,clen);
      secret(n,nlen);
      secret(k,klen);
      secret(c,clen);
      crypto_core(h,n,k,c);
      public(n,nlen);
      public(k,klen);
      public(c,clen);
      public(h,hlen);
      checksum(h,hlen);
      output_compare(h2,h,hlen,"crypto_core");
      input_compare(n2,n,nlen,"crypto_core");
      input_compare(k2,k,klen,"crypto_core");
      input_compare(c2,c,clen,"crypto_core");

      double_canary(h2,h,hlen);
      double_canary(n2,n,nlen);
      double_canary(k2,k,klen);
      double_canary(c2,c,clen);
      secret(n2,nlen);
      secret(k2,klen);
      secret(c2,clen);
      crypto_core(h2,n2,k2,c2);
      public(n2,nlen);
      public(k2,klen);
      public(c2,clen);
      public(h2,hlen);
      if (memcmp(h2,h,hlen) != 0) fail("failure: crypto_core is nondeterministic\n");

      double_canary(h2,h,hlen);
      double_canary(n2,n,nlen);
      double_canary(k2,k,klen);
      double_canary(c2,c,clen);
      secret(n2,nlen);
      secret(k,klen);
      secret(c,clen);
      crypto_core(n2,n2,k,c);
      public(n2,hlen);
      public(k,klen);
      public(c,clen);
      if (memcmp(n2,h,hlen) != 0) fail("failure: crypto_core does not handle n=h overlap\n");
      memcpy(n2,n,nlen);
      secret(n,nlen);
      secret(k2,klen);
      secret(c,clen);
      crypto_core(k2,n,k2,c);
      public(k2,hlen);
      public(n,nlen);
      public(c,clen);
      if (memcmp(k2,h,hlen) != 0) fail("failure: crypto_core does not handle k=h overlap\n");
      memcpy(k2,k,klen);
      secret(n,nlen);
      secret(k,klen);
      secret(c2,clen);
      crypto_core(c2,n,k,c2);
      public(c2,hlen);
      public(n,nlen);
      public(k,klen);
      if (memcmp(c2,h,hlen) != 0) fail("failure: crypto_core does not handle c=h overlap\n");
      memcpy(c2,c,clen);
    }
    checksum_expected(core_weightsntrup653_checksums[checksumbig]);
  }
  for (long long precomp = 0;precomp < precomputed_core_weightsntrup653_NUM;++precomp) {
    output_prepare(h2,h,crypto_core_OUTPUTBYTES);
    input_prepare(n2,n,crypto_core_INPUTBYTES);
    memcpy(n,precomputed_core_weightsntrup653_n[precomp],crypto_core_INPUTBYTES);
    memcpy(n2,precomputed_core_weightsntrup653_n[precomp],crypto_core_INPUTBYTES);
    input_prepare(k2,k,crypto_core_KEYBYTES);
    memcpy(k,precomputed_core_weightsntrup653_k[precomp],crypto_core_KEYBYTES);
    memcpy(k2,precomputed_core_weightsntrup653_k[precomp],crypto_core_KEYBYTES);
    input_prepare(c2,c,crypto_core_CONSTBYTES);
    memcpy(c,precomputed_core_weightsntrup653_c[precomp],crypto_core_CONSTBYTES);
    memcpy(c2,precomputed_core_weightsntrup653_c[precomp],crypto_core_CONSTBYTES);
    crypto_core(h,n,k,c);
    if (memcmp(h,precomputed_core_weightsntrup653_h[precomp],crypto_core_OUTPUTBYTES)) {
      fail("failure: crypto_core fails precomputed test vectors\n");
      printf("expected h: ");
      for (long long pos = 0;pos < crypto_core_OUTPUTBYTES;++pos) printf("%02x",((unsigned char *) precomputed_core_weightsntrup653_h[precomp])[pos]);
      printf("\n");
      printf("received h: ");
      for (long long pos = 0;pos < crypto_core_OUTPUTBYTES;++pos) printf("%02x",h[pos]);
      printf("\n");
    }
    output_compare(h2,h,crypto_core_OUTPUTBYTES,"crypto_core");
    input_compare(n2,n,crypto_core_INPUTBYTES,"crypto_core");
    input_compare(k2,k,crypto_core_KEYBYTES,"crypto_core");
    input_compare(c2,c,crypto_core_CONSTBYTES,"crypto_core");
  }
}

void test_core_weightsntrup653(void)
{
  long long maxalloc = 0;
  if (targeto && strcmp(targeto,"core")) return;
  if (targetp && strcmp(targetp,"weightsntrup653")) return;
  storage_core_weightsntrup653_h = callocplus(crypto_core_OUTPUTBYTES);
  test_core_weightsntrup653_h = aligned(storage_core_weightsntrup653_h,crypto_core_OUTPUTBYTES);
  if (crypto_core_OUTPUTBYTES > maxalloc) maxalloc = crypto_core_OUTPUTBYTES;
  storage_core_weightsntrup653_n = callocplus(crypto_core_INPUTBYTES);
  test_core_weightsntrup653_n = aligned(storage_core_weightsntrup653_n,crypto_core_INPUTBYTES);
  if (crypto_core_INPUTBYTES > maxalloc) maxalloc = crypto_core_INPUTBYTES;
  storage_core_weightsntrup653_k = callocplus(crypto_core_KEYBYTES);
  test_core_weightsntrup653_k = aligned(storage_core_weightsntrup653_k,crypto_core_KEYBYTES);
  if (crypto_core_KEYBYTES > maxalloc) maxalloc = crypto_core_KEYBYTES;
  storage_core_weightsntrup653_c = callocplus(crypto_core_CONSTBYTES);
  test_core_weightsntrup653_c = aligned(storage_core_weightsntrup653_c,crypto_core_CONSTBYTES);
  if (crypto_core_CONSTBYTES > maxalloc) maxalloc = crypto_core_CONSTBYTES;
  storage_core_weightsntrup653_h2 = callocplus(maxalloc);
  test_core_weightsntrup653_h2 = aligned(storage_core_weightsntrup653_h2,crypto_core_OUTPUTBYTES);
  storage_core_weightsntrup653_n2 = callocplus(maxalloc);
  test_core_weightsntrup653_n2 = aligned(storage_core_weightsntrup653_n2,crypto_core_INPUTBYTES);
  storage_core_weightsntrup653_k2 = callocplus(maxalloc);
  test_core_weightsntrup653_k2 = aligned(storage_core_weightsntrup653_k2,crypto_core_KEYBYTES);
  storage_core_weightsntrup653_c2 = callocplus(maxalloc);
  test_core_weightsntrup653_c2 = aligned(storage_core_weightsntrup653_c2,crypto_core_CONSTBYTES);

  for (long long offset = 0;offset < 2;++offset) {
    if (targetoffset && atol(targetoffset) != offset) continue;
    if (offset && valgrind) break;
    printf("core_weightsntrup653 offset %lld\n",offset);
    for (long long impl = -1;impl < ntruprime_numimpl_core_weightsntrup653();++impl)
      forked(test_core_weightsntrup653_impl,impl);
    ++test_core_weightsntrup653_h;
    ++test_core_weightsntrup653_n;
    ++test_core_weightsntrup653_k;
    ++test_core_weightsntrup653_c;
    ++test_core_weightsntrup653_h2;
    ++test_core_weightsntrup653_n2;
    ++test_core_weightsntrup653_k2;
    ++test_core_weightsntrup653_c2;
  }
  free(storage_core_weightsntrup653_c2);
  free(storage_core_weightsntrup653_k2);
  free(storage_core_weightsntrup653_n2);
  free(storage_core_weightsntrup653_h2);
  free(storage_core_weightsntrup653_c);
  free(storage_core_weightsntrup653_k);
  free(storage_core_weightsntrup653_n);
  free(storage_core_weightsntrup653_h);
}
#undef crypto_core_OUTPUTBYTES
#undef crypto_core_INPUTBYTES
#undef crypto_core_KEYBYTES
#undef crypto_core_CONSTBYTES

