/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "TintWindow.hpp"

#include "TintDocument.hpp"
#include "../../gui/Window.hpp"
#include "../../gui/WindowMenu.hpp"
#include "../../reg/Components.hpp"
#include "../../image/FileImageResource.hpp"
#include "../../../gnome/gnome.hpp"

#include "wx/config.h"
#include "wx/button.h"
#include "wx/sizer.h"
#include "wx/textdlg.h"
#include "wx/filedlg.h"
#include "wx/msgdlg.h"
#include "wx/dcmemory.h"
#include "wx/cshelp.h"
#include "wx/filename.h"
#include "wx/stdpaths.h"

using namespace indii;

TintWindow::TintWindow(wxWindow* parent, const Mode mode,
    WindowObserver* obs) : Window(parent, mode, obs) {
  init();
  enable();
  if (mode == STANDALONE) {
    infoBar->showStartMessage();
  }
}

TintWindow::TintWindow(wxWindow* parent, ImageResource* res,
    ClusterModel* model, const Mode mode, WindowObserver* obs) :
    Window(parent, mode, obs) {
  if (mode == STANDALONE) {
    TintDocument* doc = new TintDocument(notebook, res, model);
    init(doc);
  } else {
    this->doc = new TintDocument(main, res, model);
    init(this->doc);
  }
  setModel(model);
  enable();
  if (mode == EMBED) {
    infoBar->showInfoMessage(_("Select one or more thumbnails to the right to restore colour to the image, then fine-tune the effect with the sliders."), MSG_INTERFACE);
  }
}

TintWindow::TintWindow(wxWindow* parent, const wxString& path,
    const Mode mode, WindowObserver* obs) :
    Window(parent, mode, obs) {
  Document* doc = open(path);
  init(doc);
  setModel(doc->getModel());
  enable();
  infoBar->showInfoMessage(_("Select one or more thumbnails to the right to restore colour to the image, then fine-tune the effect with the sliders."), MSG_INTERFACE);
}

Document* TintWindow::open(const wxString& file) {
  infoBar->showInfoMessage(_("Select one or more thumbnails to the right to restore colour to the image, then fine-tune the effect with the sliders."), MSG_INTERFACE);
  if (mode == STANDALONE) {
    return new TintDocument(notebook, file);
  } else {
    return new TintDocument(main, file);
  }
}

void TintWindow::layoutDefault() {
  wxAuiPaneInfo& thumbPaneInfo = manager->GetPane(thumbPane);
  thumbPane->Fit();
  thumbPaneInfo.FloatingSize(thumbPane->GetMinSize().GetWidth(), thumbPane->GetMinSize().GetHeight() + EXTRA_PANE_HEIGHT);
  thumbPaneInfo.Right();
  thumbPaneInfo.Layer(2);
  thumbPaneInfo.Dock();
  thumbPaneInfo.Show();
  
  wxAuiPaneInfo& prePaneInfo = manager->GetPane(prePane);
  prePane->Fit();
  prePaneInfo.FloatingSize(prePane->GetMinSize().GetWidth(), prePane->GetMinSize().GetHeight() + EXTRA_PANE_HEIGHT);
  prePaneInfo.Bottom();
  prePaneInfo.Position(2);
  prePaneInfo.Layer(1);
  prePaneInfo.Row(1);
  prePaneInfo.Dock();
  prePaneInfo.Show(false);

  wxAuiPaneInfo& postPaneInfo = manager->GetPane(postPane);
  postPane->Fit();
  postPaneInfo.FloatingSize(postPane->GetMinSize().GetWidth(), postPane->GetMinSize().GetHeight() + EXTRA_PANE_HEIGHT);
  postPaneInfo.Bottom();
  postPaneInfo.Position(1);
  postPaneInfo.Layer(1);
  postPaneInfo.Row(1);
  postPaneInfo.Dock();
  postPaneInfo.Show();

  wxAuiPaneInfo& mixerPaneInfo = manager->GetPane(mixerPane);
  mixerPane->Fit();
  mixerPaneInfo.FloatingSize(mixerPane->GetMinSize().GetWidth(), mixerPane->GetMinSize().GetHeight() + EXTRA_PANE_HEIGHT);
  mixerPaneInfo.Bottom();
  mixerPaneInfo.Position(0);
  mixerPaneInfo.Layer(1);
  mixerPaneInfo.Row(1);
  mixerPaneInfo.Dock();
  mixerPaneInfo.Show();

  manager->Update();
}

void TintWindow::initPanes() {
  int width, height;
  wxAuiPaneInfo paneInfo;
  paneInfo.DefaultPane();
  paneInfo.CloseButton(true);
  //paneInfo.PinButton(true);
  paneInfo.PaneBorder(true);
  paneInfo.CaptionVisible(true);
 
  /* thumbnail pane */
  thumbPane = new ThumbPane(main);
  width = MIN_THUMB_PANE_WIDTH;
  height = 512;
  
  #ifndef WX_28
  paneInfo.Name(_("Thumbs"));
  #else
  paneInfo.Name(wxString("Thumbs", wxConvUTF8));
  #endif
  paneInfo.MinSize(wxSize(width, 110));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.FloatingSize(width + EXTRA_PANE_WIDTH, height + EXTRA_PANE_HEIGHT);
  paneInfo.Caption(_("Thumbs"));
  paneInfo.Right();
  paneInfo.Layer(2);
  manager->AddPane(thumbPane, paneInfo);
  
  /* preprocessing pane */
  prePane = new PreProcessPane(main);
  width = prePane->GetMinSize().GetWidth();
  height = prePane->GetMinSize().GetHeight();
  
  #ifndef WX_28
  paneInfo.Name(_("Preprocessing"));
  #else
  paneInfo.Name(wxString("Preprocessing", wxConvUTF8));
  #endif
  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.FloatingSize(width + EXTRA_PANE_WIDTH, height + EXTRA_PANE_HEIGHT);
  paneInfo.Caption(_("Preprocessing"));
  paneInfo.Bottom();
  paneInfo.Position(2);
  paneInfo.Layer(1);
  paneInfo.Show(false);
  manager->AddPane(prePane, paneInfo);
  paneInfo.Show(true);
  
  /* postprocessing pane */
  postPane = new PostProcessPane(main);
  width = postPane->GetMinSize().GetWidth();
  height = postPane->GetMinSize().GetHeight();
  
  #ifndef WX_28
  paneInfo.Name(_("Postprocessing"));
  #else
  paneInfo.Name(wxString("Postprocessing", wxConvUTF8));
  #endif
  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.FloatingSize(width + EXTRA_PANE_WIDTH, height + EXTRA_PANE_HEIGHT);
  paneInfo.Caption(_("Postprocessing"));
  paneInfo.Bottom();
  paneInfo.Position(1);
  paneInfo.Layer(1);
  manager->AddPane(postPane, paneInfo);
  
  /* channel mixer pane */
  mixerPane = new ChannelMixerPane(main);
  width = mixerPane->GetMinSize().GetWidth();
  height = mixerPane->GetMinSize().GetHeight();
  
  #ifndef WX_28
  paneInfo.Name(_("Mixer"));
  #else
  paneInfo.Name(wxString("Mixer", wxConvUTF8));
  #endif
  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.FloatingSize(width + EXTRA_PANE_WIDTH, height + EXTRA_PANE_HEIGHT);
  paneInfo.Caption(_("Channel mixer"));
  paneInfo.Bottom();
  paneInfo.Position(0);
  paneInfo.Layer(1);
  manager->AddPane(mixerPane, paneInfo);

  /* add panes to window menu */
  addPane(_("Thumbs pane"));
  addPane(_("Preprocessing pane"));
  addPane(_("Postprocessing pane"));
  addPane(_("Channel mixer pane"));
}

void TintWindow::enable(const bool enable) {
  Window::enable(enable);
  if (!enable) {
    thumbPane->setModel(NULL);
    prePane->setModel(NULL);
    postPane->setModel(NULL);
    mixerPane->setModel(NULL);
  }
}

void TintWindow::setModel(Model* model) {
  ClusterModel* cmodel = dynamic_cast<ClusterModel*>(model);

  thumbPane->setModel(cmodel);
  prePane->setModel(cmodel);
  postPane->setModel(cmodel);
  mixerPane->setModel(cmodel);
}

bool TintWindow::isShown(const int i) {
  /* pre-condition */
  assert (i < 4);

  switch (i) {
  case 0:
    return manager->GetPane(thumbPane).IsShown();
  case 1:
    return manager->GetPane(prePane).IsShown();
  case 2:
    return manager->GetPane(postPane).IsShown();
  case 3:
    return manager->GetPane(mixerPane).IsShown();
  default:
    return false;
  }
}

void TintWindow::show(const int i, const bool show) {
  /* pre-condition */
  assert (i < 4);
  
  switch (i) {
  case 0:
    manager->GetPane(thumbPane).Show(show);
    break;
  case 1:
    manager->GetPane(prePane).Show(show);
    break;
  case 2:
    manager->GetPane(postPane).Show(show);
    break;
  case 3:
    manager->GetPane(mixerPane).Show(show);
    break;
  }
  manager->Update();
}

BEGIN_EVENT_TABLE(TintWindow, Window)
END_EVENT_TABLE()
