#!/bin/sh
src=`pwd`
tmp=$src/tests-tmp
rm -rf $tmp
mkdir -p $tmp
PATH="$src:/bin:/usr/bin:/usr/local/bin"
tests_failed=0
tests_count=0
_UID=`id -u`
_GID=`id -g`

usage() {
  echo "usage: sh $0 [-v]"
}

vecho() { :; }
while getopts v flag
do
  case $flag in
    v)      vecho() { echo "$*"; } ;;
	*)      usage; exit 1 ;;
  esac
done
xw=$src/xfer-watch
uid=`id -u`
gid=`id -g`
home=$tmp/home
mkdir -p $home
cvm=$tmp/cvm
TCPLOCALIP=127.0.0.1
TCPLOCALHOST=localhost
TCPREMOTEIP=127.0.0.1
LOGREQUESTS=1
CVM_RANDOM_BYTES=0
TIMEOUT=5
export CVM_RANDOM_BYTES LOGREQUESTS TCPLOCALIP TCPLOCALHOST TCPREMOTEIP TIMEOUT

env="env UID=$uid GID=$gid USER=ftp GROUP=ftp HOME=$home TZ=Canada/Saskatchewan"

# Support functions

strippid() { sed -e 's/\[[0123456789]*\]/[PID]/'; }

echolines() {
  for line in "$@"
  do
    printf "$line\r\n" 2>/dev/null
  done
}

authcmd() { $src/twoftpd-auth "$@" 2>&1 </dev/null | strippid; }

auth() {
  local cvm=$1
  local backend=$2
  shift 2
  echolines "$@" | $src/twoftpd-auth $cvm $backend 2>&1 | strippid
}

makecvm() {
  (
    echo '#!/bin/sh'
    for line in "$@"; do echo "$line"; done
  ) >$cvm
  chmod 755 $cvm
}

fixdir() { sed -e 's/^\(d.*\)    [123456789][0123456789] /\1  4096 /'; }
fixbytes() { sed -e 's/[0123456789]* bytes /### bytes /'; }

anon() { echolines "$@" | $env $src/twoftpd-anon 2>/dev/null | fixdir | tail -n +2; }
drop() { echolines "$@" | $env $src/twoftpd-drop 2>/dev/null | fixdir | tail -n +2; }
xfer() { echolines "$@" | $env $src/twoftpd-xfer 2>/dev/null | fixdir | tail -n +2; }

testparams() {
  local backend=$1
  local command=$2
  shift 2
  local param
  for param in "$@"; do
    $backend "$command $param"
  done
}

xfertest() {
  local backend=$1
  local mode=$2
  local io=$3
  local conv=$4
  shift 4
  $env $src/xfer-test $src/twoftpd-$backend $mode $io $conv "$@" | fixdir | tail -n +2
}


run_compare_test() {
  local name=$1
  shift
  sed -e "s:@SOURCE@:$src:g"   	-e "s:@TMPDIR@:$tmp:g"   	-e "s:@UID@:$_UID:" 	-e "s:@GID@:$_GID:" 	>$tmp/expected
  ( runtest "$@" 2>&1 ) 2>&1 >$tmp/actual-raw
  cat -v $tmp/actual-raw >$tmp/actual
  if ! cmp $tmp/expected $tmp/actual >/dev/null 2>&1
  then
    echo "Test $name $* failed:"
	( cd $tmp; diff -U 9999 expected actual | tail -n +3; echo; )
	tests_failed=$(($tests_failed+1))
  fi
  rm -f $tmp/expected $tmp/actual
  tests_count=$(($tests_count+1))
}

##### Test tests/auth-output #####

runtest() {
( echolines QUIT | $src/twoftpd-auth 1 2 >/dev/null; ) 2>&1 | strippid
echolines QUIT | $src/twoftpd-auth 1 2 2>/dev/null
}
vecho "Running test tests/auth-output "
run_compare_test tests/auth-output  <<END_OF_TEST_RESULTS
twoftpd-auth[PID]: QUIT
220-TwoFTPd server ready.^M
220 Authenticate first.^M
221 Bye.^M
END_OF_TEST_RESULTS


##### Test tests/auth-commands #####

runtest() {
local command="$1"
local parameter="$2"
if [ $parameter = 'no-parameter' ]; then
  cmd=$command
else
  cmd="$command xyzzy"
fi
auth foo foo "$cmd"
}
vecho "Running test tests/auth-commands 'QUIT' 'with-parameter'"
run_compare_test tests/auth-commands 'QUIT' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'QUIT' 'no-parameter'"
run_compare_test tests/auth-commands 'QUIT' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT
221 Bye.^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'QuIt' 'with-parameter'"
run_compare_test tests/auth-commands 'QuIt' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'QuIt' 'no-parameter'"
run_compare_test tests/auth-commands 'QuIt' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT
221 Bye.^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'quit' 'with-parameter'"
run_compare_test tests/auth-commands 'quit' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'quit' 'no-parameter'"
run_compare_test tests/auth-commands 'quit' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT
221 Bye.^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'HELP' 'with-parameter'"
run_compare_test tests/auth-commands 'HELP' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: HELP xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'HELP' 'no-parameter'"
run_compare_test tests/auth-commands 'HELP' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: HELP
502 No help is available.^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'SYST' 'with-parameter'"
run_compare_test tests/auth-commands 'SYST' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: SYST xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'SYST' 'no-parameter'"
run_compare_test tests/auth-commands 'SYST' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: SYST
215 UNIX Type: L8^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'NOOP' 'with-parameter'"
run_compare_test tests/auth-commands 'NOOP' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: NOOP xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'NOOP' 'no-parameter'"
run_compare_test tests/auth-commands 'NOOP' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: NOOP
200 Awaiting your commands, master...^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'USER' 'with-parameter'"
run_compare_test tests/auth-commands 'USER' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER xyzzy
331 Send PASS.^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'USER' 'no-parameter'"
run_compare_test tests/auth-commands 'USER' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'PASS' 'with-parameter'"
run_compare_test tests/auth-commands 'PASS' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: PASS XXXXXXXX
503 Send USER first.^M
END_OF_TEST_RESULTS

vecho "Running test tests/auth-commands 'PASS' 'no-parameter'"
run_compare_test tests/auth-commands 'PASS' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: PASS
504 Command requires a parameter^M
END_OF_TEST_RESULTS


##### Test tests/auth-invocation #####

runtest() {
authcmd
authcmd 1
authcmd 1 2
authcmd 1 2 3
}
vecho "Running test tests/auth-invocation "
run_compare_test tests/auth-invocation  <<END_OF_TEST_RESULTS
421 Configuration error, insufficient paramenters.^M
421 Configuration error, insufficient paramenters.^M
220-TwoFTPd server ready.^M
220 Authenticate first.^M
220-TwoFTPd server ready.^M
220 Authenticate first.^M
END_OF_TEST_RESULTS


##### Test tests/auth-timeout #####

runtest() {
export TIMEOUT
TIMEOUT=60
auth 1 2
TIMEOUT=00
auth 1 2
TIMEOUT=-6
auth 1 2
unset TIMEOUT
}
vecho "Running test tests/auth-timeout "
run_compare_test tests/auth-timeout  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
421 Configuration error, invalid timeout value^M
421 Configuration error, invalid timeout value^M
END_OF_TEST_RESULTS


##### Test tests/backend-greeting #####

runtest() {
local backend="$1"
$env $src/twoftpd-$backend 2>/dev/null
$env BANNER='foo
bar' $src/twoftpd-$backend 2>/dev/null
$env AUTHENTICATED=1 $src/twoftpd-$backend 2>/dev/null
}
vecho "Running test tests/backend-greeting 'anon'"
run_compare_test tests/backend-greeting 'anon' <<END_OF_TEST_RESULTS
220 Ready to transfer files.^M
220-foo^M
220-bar^M
220 Ready to transfer files.^M
230 Ready to transfer files.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-greeting 'drop'"
run_compare_test tests/backend-greeting 'drop' <<END_OF_TEST_RESULTS
220 Ready to transfer files.^M
220-foo^M
220-bar^M
220 Ready to transfer files.^M
230 Ready to transfer files.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-greeting 'xfer'"
run_compare_test tests/backend-greeting 'xfer' <<END_OF_TEST_RESULTS
220 Ready to transfer files.^M
220-foo^M
220-bar^M
220 Ready to transfer files.^M
230 Ready to transfer files.^M
END_OF_TEST_RESULTS


##### Test tests/make-good-cvm #####

runtest() {
makecvm 'cat >/dev/null' \
	'printf "\\0\\0\\001\\010username\\002\\003102\\003\\003103\\004\\011Real Name\\005\\011/home/dir\\006\\012/bin/false\\007\\007grpname\\000" 2>/dev/null' \
	'exit 0'
}
vecho "Running test tests/make-good-cvm "
run_compare_test tests/make-good-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/auth-cvm-good-data #####

runtest() {
(
  echo '#!/bin/sh'
  echo 'echo UID=$UID GID=$GID HOME=$HOME USER=$USER AUTHENTICATED=$AUTHENTICATED'
) >$tmp/env
chmod 755 $tmp/env
auth $cvm $tmp/env 'USER username' 'PASS password'
rm -f $tmp/env
}
vecho "Running test tests/auth-cvm-good-data "
run_compare_test tests/auth-cvm-good-data  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
UID=102 GID=103 HOME=/home/dir USER=username AUTHENTICATED=1
END_OF_TEST_RESULTS


##### Test tests/delete-good-cvm #####

runtest() {
rm -f $cvm
}
vecho "Running test tests/delete-good-cvm "
run_compare_test tests/delete-good-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/make-home #####

runtest() {
mkdir $home/subdir
date >$home/subdir/date
for i in `seq 60`; do
  echo 01234567890123456789012345678901234567890123456789012345678901234567890
done >$home/large
TZ=GMT touch -t 200001020304.05 $home/empty
TZ=GMT touch -t 200102030405.06 $home/subdir
TZ=GMT touch -t 200301180607.08 $home/large
chmod 664 $home/empty
chmod 644 $home/large
chmod 775 $home/subdir
}
vecho "Running test tests/make-home "
run_compare_test tests/make-home  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/backend-appe #####

runtest() {
local backend="$1"
cat $home/large >$home/tmp
xfertest $backend port store bin 'APPE tmp' <$home/large
cksum <$home/tmp
xfertest $backend port store bin 'APPE tmpx' <$home/large
test -e $home/tmpx || echo tmpx was not created
rm -f $home/tmp $home/tmpx
}
vecho "Running test tests/backend-appe 'anon'"
run_compare_test tests/backend-appe 'anon' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
1926267812 4320
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
tmpx was not created
END_OF_TEST_RESULTS

vecho "Running test tests/backend-appe 'drop'"
run_compare_test tests/backend-appe 'drop' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
1926267812 4320
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
tmpx was not created
END_OF_TEST_RESULTS

vecho "Running test tests/backend-appe 'xfer'"
run_compare_test tests/backend-appe 'xfer' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (4320 bytes received).^M
2237341303 8640
200 Transfer mode changed to BINARY.^M
200 OK.^M
550 Could not open output file: No such file or directory^M
tmpx was not created
END_OF_TEST_RESULTS


##### Test tests/backend-messagefile #####

runtest() {
local backend="$1"

echo top line1 >$home/README
echo top line2 >>$home/README
echo subdir >$home/subdir/README

echo CWD subdir \
| $env MESSAGEFILE=README \
  $src/twoftpd-$backend 2>/dev/null

rm -f $home/README $home/subdir/README
}
vecho "Running test tests/backend-messagefile 'anon'"
run_compare_test tests/backend-messagefile 'anon' <<END_OF_TEST_RESULTS
220-top line1^M
220-top line2^M
220 Ready to transfer files.^M
250-subdir^M
250 Changed directory.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-messagefile 'drop'"
run_compare_test tests/backend-messagefile 'drop' <<END_OF_TEST_RESULTS
220-top line1^M
220-top line2^M
220 Ready to transfer files.^M
250-subdir^M
250 Changed directory.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-messagefile 'xfer'"
run_compare_test tests/backend-messagefile 'xfer' <<END_OF_TEST_RESULTS
220-top line1^M
220-top line2^M
220 Ready to transfer files.^M
250-subdir^M
250 Changed directory.^M
END_OF_TEST_RESULTS


##### Test tests/backend-chmod #####

runtest() {
local backend="$1"
touch $home/tmp
chmod 444 $home/tmp
$backend "SITE CHMOD 755 tmp" "SITE CHMOD 755 tmpx"
ls -l $home/tmp | cut -d' ' -f1
rm -f $home/tmp
}
vecho "Running test tests/backend-chmod 'anon'"
run_compare_test tests/backend-chmod 'anon' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
-r--r--r--
END_OF_TEST_RESULTS

vecho "Running test tests/backend-chmod 'drop'"
run_compare_test tests/backend-chmod 'drop' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
-r--r--r--
END_OF_TEST_RESULTS

vecho "Running test tests/backend-chmod 'xfer'"
run_compare_test tests/backend-chmod 'xfer' <<END_OF_TEST_RESULTS
250 File modes changed.^M
550 Could not change modes on file: No such file or directory^M
-rwxr-xr-x
END_OF_TEST_RESULTS


##### Test tests/backend-mode #####

runtest() {
local backend="$1"
testparams $backend MODE S s X x
}
vecho "Running test tests/backend-mode 'anon'"
run_compare_test tests/backend-mode 'anon' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-mode 'drop'"
run_compare_test tests/backend-mode 'drop' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-mode 'xfer'"
run_compare_test tests/backend-mode 'xfer' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS


##### Test tests/backend-store #####

runtest() {
local backend="$1"
local mode="$2"
local conv="$3"
sed -e 's/$/
/' $home/large >$tmp/large
xfertest $backend $mode store $conv 'STOR tmp' <$tmp/large
xfertest $backend $mode store $conv 'STOR tmp' <$tmp/large
if [ -e $home/tmp ]; then
  cksum <$home/tmp
  rm -f $home/tmp
else
  echo tmp was not created
fi
rm -f $tmp/large
}
vecho "Running test tests/backend-store 'anon' 'pasv' 'bin'"
run_compare_test tests/backend-store 'anon' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
502 Command not supported.^M
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
502 Command not supported.^M
tmp was not created
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'anon' 'pasv' 'asc'"
run_compare_test tests/backend-store 'anon' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
502 Command not supported.^M
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
502 Command not supported.^M
tmp was not created
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'anon' 'port' 'bin'"
run_compare_test tests/backend-store 'anon' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
tmp was not created
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'anon' 'port' 'asc'"
run_compare_test tests/backend-store 'anon' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
502 Command not supported.^M
200 Transfer mode changed to ASCII.^M
200 OK.^M
502 Command not supported.^M
tmp was not created
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'drop' 'pasv' 'bin'"
run_compare_test tests/backend-store 'drop' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
550 Could not open output file: File exists^M
3192406476 4380
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'drop' 'pasv' 'asc'"
run_compare_test tests/backend-store 'drop' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
550 Could not open output file: File exists^M
1926267812 4320
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'drop' 'port' 'bin'"
run_compare_test tests/backend-store 'drop' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
200 Transfer mode changed to BINARY.^M
200 OK.^M
550 Could not open output file: File exists^M
3192406476 4380
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'drop' 'port' 'asc'"
run_compare_test tests/backend-store 'drop' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
200 Transfer mode changed to ASCII.^M
200 OK.^M
550 Could not open output file: File exists^M
1926267812 4320
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'xfer' 'pasv' 'bin'"
run_compare_test tests/backend-store 'xfer' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
3192406476 4380
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'xfer' 'pasv' 'asc'"
run_compare_test tests/backend-store 'xfer' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
1926267812 4320
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'xfer' 'port' 'bin'"
run_compare_test tests/backend-store 'xfer' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
3192406476 4380
END_OF_TEST_RESULTS

vecho "Running test tests/backend-store 'xfer' 'port' 'asc'"
run_compare_test tests/backend-store 'xfer' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (4380 bytes received).^M
1926267812 4320
END_OF_TEST_RESULTS


##### Test tests/backend-lockhome #####

runtest() {
LOCKHOME=1
export LOCKHOME
xfertest xfer port retr asc PWD NLST
echo
xfertest xfer port retr asc 'CWD subdir' PWD NLST
echo
xfertest xfer port retr asc 'CWD ..' PWD NLST
unset LOCKHOME
}
vecho "Running test tests/backend-lockhome "
run_compare_test tests/backend-lockhome  <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
257 "@TMPDIR@/home"^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M

200 Transfer mode changed to ASCII.^M
250 Changed directory.^M
257 "@TMPDIR@/home/subdir"^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Transfer completed (6 bytes sent).^M

200 Transfer mode changed to ASCII.^M
550 Permission denied.^M
END_OF_TEST_RESULTS


##### Test tests/backend-size #####

runtest() {
local backend="$1"
$backend 'SIZE empty' 'SIZE large'
}
vecho "Running test tests/backend-size 'anon'"
run_compare_test tests/backend-size 'anon' <<END_OF_TEST_RESULTS
213 0^M
213 4320^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-size 'drop'"
run_compare_test tests/backend-size 'drop' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-size 'xfer'"
run_compare_test tests/backend-size 'xfer' <<END_OF_TEST_RESULTS
213 0^M
213 4320^M
END_OF_TEST_RESULTS


##### Test tests/backend-rest #####

runtest() {
local backend="$1"
xfertest $backend port retr bin 'REST 4000' 'RETR large'
}
vecho "Running test tests/backend-rest 'anon'"
run_compare_test tests/backend-rest 'anon' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
350 Start position for transfer has been set.^M
200 OK.^M
150 Opened data connection.^M
0123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 Transfer completed (320 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-rest 'drop'"
run_compare_test tests/backend-rest 'drop' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-rest 'xfer'"
run_compare_test tests/backend-rest 'xfer' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
350 Start position for transfer has been set.^M
200 OK.^M
150 Opened data connection.^M
0123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 Transfer completed (320 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-list-params #####

runtest() {
local backend="$1"
local param="$2"
xfertest $backend port retr asc "NLST $param" | cut -c 1-43
}
vecho "Running test tests/backend-list-params 'anon' '.'"
run_compare_test tests/backend-list-params 'anon' '.' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '..'"
run_compare_test tests/backend-list-params 'anon' '..' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
553 Paths containing '..' not allowed.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '*'"
run_compare_test tests/backend-list-params 'anon' '*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' 'l*'"
run_compare_test tests/backend-list-params 'anon' 'l*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
large^M
226 Transfer completed (7 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' 's*'"
run_compare_test tests/backend-list-params 'anon' 's*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Transfer completed (6 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' 'subdir'"
run_compare_test tests/backend-list-params 'anon' 'subdir' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Transfer completed (6 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' 'subdir/*'"
run_compare_test tests/backend-list-params 'anon' 'subdir/*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Transfer completed (13 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' 'subdir/d*'"
run_compare_test tests/backend-list-params 'anon' 'subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Transfer completed (13 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '-A'"
run_compare_test tests/backend-list-params 'anon' '-A' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '-a'"
run_compare_test tests/backend-list-params 'anon' '-a' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '-L'"
run_compare_test tests/backend-list-params 'anon' '-L' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '-l'"
run_compare_test tests/backend-list-params 'anon' '-l' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '-alL'"
run_compare_test tests/backend-list-params 'anon' '-alL' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '-a subdir/d*'"
run_compare_test tests/backend-list-params 'anon' '-a subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Transfer completed (13 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'anon' '[.][.]'"
run_compare_test tests/backend-list-params 'anon' '[.][.]' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
550 File or directory does not exist.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '.'"
run_compare_test tests/backend-list-params 'drop' '.' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '..'"
run_compare_test tests/backend-list-params 'drop' '..' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '*'"
run_compare_test tests/backend-list-params 'drop' '*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' 'l*'"
run_compare_test tests/backend-list-params 'drop' 'l*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' 's*'"
run_compare_test tests/backend-list-params 'drop' 's*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' 'subdir'"
run_compare_test tests/backend-list-params 'drop' 'subdir' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' 'subdir/*'"
run_compare_test tests/backend-list-params 'drop' 'subdir/*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' 'subdir/d*'"
run_compare_test tests/backend-list-params 'drop' 'subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '-A'"
run_compare_test tests/backend-list-params 'drop' '-A' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '-a'"
run_compare_test tests/backend-list-params 'drop' '-a' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '-L'"
run_compare_test tests/backend-list-params 'drop' '-L' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '-l'"
run_compare_test tests/backend-list-params 'drop' '-l' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '-alL'"
run_compare_test tests/backend-list-params 'drop' '-alL' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '-a subdir/d*'"
run_compare_test tests/backend-list-params 'drop' '-a subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'drop' '[.][.]'"
run_compare_test tests/backend-list-params 'drop' '[.][.]' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '.'"
run_compare_test tests/backend-list-params 'xfer' '.' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '..'"
run_compare_test tests/backend-list-params 'xfer' '..' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
553 Paths containing '..' not allowed.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '*'"
run_compare_test tests/backend-list-params 'xfer' '*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' 'l*'"
run_compare_test tests/backend-list-params 'xfer' 'l*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
large^M
226 Transfer completed (7 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' 's*'"
run_compare_test tests/backend-list-params 'xfer' 's*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Transfer completed (6 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' 'subdir'"
run_compare_test tests/backend-list-params 'xfer' 'subdir' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Transfer completed (6 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' 'subdir/*'"
run_compare_test tests/backend-list-params 'xfer' 'subdir/*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Transfer completed (13 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' 'subdir/d*'"
run_compare_test tests/backend-list-params 'xfer' 'subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Transfer completed (13 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '-A'"
run_compare_test tests/backend-list-params 'xfer' '-A' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '-a'"
run_compare_test tests/backend-list-params 'xfer' '-a' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '-L'"
run_compare_test tests/backend-list-params 'xfer' '-L' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '-l'"
run_compare_test tests/backend-list-params 'xfer' '-l' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '-alL'"
run_compare_test tests/backend-list-params 'xfer' '-alL' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '-a subdir/d*'"
run_compare_test tests/backend-list-params 'xfer' '-a subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Transfer completed (13 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-params 'xfer' '[.][.]'"
run_compare_test tests/backend-list-params 'xfer' '[.][.]' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
550 File or directory does not exist.^M
END_OF_TEST_RESULTS


##### Test tests/backend-stru #####

runtest() {
local backend="$1"
testparams $backend STRU F f T t
}
vecho "Running test tests/backend-stru 'anon'"
run_compare_test tests/backend-stru 'anon' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-stru 'drop'"
run_compare_test tests/backend-stru 'drop' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-stru 'xfer'"
run_compare_test tests/backend-stru 'xfer' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS


##### Test tests/backend-softchroot #####

runtest() {
SOFTCHROOT=1
export SOFTCHROOT
xfertest xfer port retr asc PWD NLST
echo
xfertest xfer port retr asc 'CWD subdir' PWD NLST
echo
xfertest xfer port retr asc 'CWD ..' PWD NLST
unset SOFTCHROOT
}
vecho "Running test tests/backend-softchroot "
run_compare_test tests/backend-softchroot  <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
257 "/"^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M

200 Transfer mode changed to ASCII.^M
250 Changed directory.^M
257 "/subdir"^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Transfer completed (6 bytes sent).^M

200 Transfer mode changed to ASCII.^M
250 Changed directory.^M
257 "/"^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-cwd-pwd #####

runtest() {
local backend="$1"
$backend PWD 'CWD subdir' PWD 'CWD ..' PWD 'CWD ..' PWD
}
vecho "Running test tests/backend-cwd-pwd 'anon'"
run_compare_test tests/backend-cwd-pwd 'anon' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@/home/subdir"^M
250 Changed directory.^M
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-cwd-pwd 'drop'"
run_compare_test tests/backend-cwd-pwd 'drop' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@/home/subdir"^M
250 Changed directory.^M
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-cwd-pwd 'xfer'"
run_compare_test tests/backend-cwd-pwd 'xfer' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@/home/subdir"^M
250 Changed directory.^M
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@"^M
END_OF_TEST_RESULTS


##### Test tests/backend-retr #####

runtest() {
local backend="$1"
local mode="$2"
local conv="$3"
xfertest $backend $mode retr $conv 'RETR large'
}
vecho "Running test tests/backend-retr 'anon' 'pasv' 'bin'"
run_compare_test tests/backend-retr 'anon' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 Transfer completed (4320 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'anon' 'pasv' 'asc'"
run_compare_test tests/backend-retr 'anon' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
226 Transfer completed (4380 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'anon' 'port' 'bin'"
run_compare_test tests/backend-retr 'anon' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 Transfer completed (4320 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'anon' 'port' 'asc'"
run_compare_test tests/backend-retr 'anon' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
226 Transfer completed (4380 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'drop' 'pasv' 'bin'"
run_compare_test tests/backend-retr 'drop' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'drop' 'pasv' 'asc'"
run_compare_test tests/backend-retr 'drop' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'drop' 'port' 'bin'"
run_compare_test tests/backend-retr 'drop' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'drop' 'port' 'asc'"
run_compare_test tests/backend-retr 'drop' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'xfer' 'pasv' 'bin'"
run_compare_test tests/backend-retr 'xfer' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 Transfer completed (4320 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'xfer' 'pasv' 'asc'"
run_compare_test tests/backend-retr 'xfer' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
226 Transfer completed (4380 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'xfer' 'port' 'bin'"
run_compare_test tests/backend-retr 'xfer' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 Transfer completed (4320 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-retr 'xfer' 'port' 'asc'"
run_compare_test tests/backend-retr 'xfer' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
226 Transfer completed (4380 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-interrupt #####

runtest() {
local mode="$1"
local method="$2"
for i in $( seq 10000 ); do
  echo 012345678901234567890123456789012345678901234567890123456789012
done >$home/testfile
xfertest xfer $mode retr bin 64:$method 'RETR testfile' | fixbytes
rm $home/testfile
}
vecho "Running test tests/backend-interrupt 'pasv' 'close'"
run_compare_test tests/backend-interrupt 'pasv' 'close' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
012345678901234567890123456789012345678901234567890123456789012
426 Transfer interrupted (### bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-interrupt 'pasv' 'abort'"
run_compare_test tests/backend-interrupt 'pasv' 'abort' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
012345678901234567890123456789012345678901234567890123456789012
426 Transfer interrupted (### bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-interrupt 'pasv' 'drop'"
run_compare_test tests/backend-interrupt 'pasv' 'drop' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
012345678901234567890123456789012345678901234567890123456789012
426 Transfer interrupted (### bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-interrupt 'port' 'close'"
run_compare_test tests/backend-interrupt 'port' 'close' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
012345678901234567890123456789012345678901234567890123456789012
426 Transfer interrupted (### bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-interrupt 'port' 'abort'"
run_compare_test tests/backend-interrupt 'port' 'abort' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
012345678901234567890123456789012345678901234567890123456789012
426 Transfer interrupted (### bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-interrupt 'port' 'drop'"
run_compare_test tests/backend-interrupt 'port' 'drop' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
012345678901234567890123456789012345678901234567890123456789012
426 Transfer interrupted (### bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-rnfrto #####

runtest() {
local backend="$1"
$backend 'RNFR subdir/date' 'RNTO date'
ls -1 $home
$backend 'RNFR date' 'RNTO subdir/date' 'RNFR date' 'RNTO subdir/date'
ls -1 $home
}
vecho "Running test tests/backend-rnfrto 'anon'"
run_compare_test tests/backend-rnfrto 'anon' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
empty
large
subdir
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
empty
large
subdir
END_OF_TEST_RESULTS

vecho "Running test tests/backend-rnfrto 'drop'"
run_compare_test tests/backend-rnfrto 'drop' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
empty
large
subdir
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
empty
large
subdir
END_OF_TEST_RESULTS

vecho "Running test tests/backend-rnfrto 'xfer'"
run_compare_test tests/backend-rnfrto 'xfer' <<END_OF_TEST_RESULTS
350 OK, file exists.^M
250 File renamed.^M
date
empty
large
subdir
350 OK, file exists.^M
250 File renamed.^M
450 Could not locate file.^M
425 Send RNFR first.^M
empty
large
subdir
END_OF_TEST_RESULTS


##### Test tests/backend-mdtm #####

runtest() {
local backend="$1"
$backend 'MDTM empty' 'MDTM large'
}
vecho "Running test tests/backend-mdtm 'anon'"
run_compare_test tests/backend-mdtm 'anon' <<END_OF_TEST_RESULTS
213 20000102030405^M
213 20030118060708^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-mdtm 'drop'"
run_compare_test tests/backend-mdtm 'drop' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-mdtm 'xfer'"
run_compare_test tests/backend-mdtm 'xfer' <<END_OF_TEST_RESULTS
213 20000102030405^M
213 20030118060708^M
END_OF_TEST_RESULTS


##### Test tests/backend-list-modes #####

runtest() {
local backend="$1"
local mode="$2"
local conv="$3"
local list="$4"
xfertest $backend $mode retr $conv $list | cut -c 1-43
}
vecho "Running test tests/backend-list-modes 'anon' 'pasv' 'bin' 'NLST'"
run_compare_test tests/backend-list-modes 'anon' 'pasv' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'anon' 'pasv' 'bin' 'LIST'"
run_compare_test tests/backend-list-modes 'anon' 'pasv' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'anon' 'pasv' 'asc' 'NLST'"
run_compare_test tests/backend-list-modes 'anon' 'pasv' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'anon' 'pasv' 'asc' 'LIST'"
run_compare_test tests/backend-list-modes 'anon' 'pasv' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'anon' 'port' 'bin' 'NLST'"
run_compare_test tests/backend-list-modes 'anon' 'port' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'anon' 'port' 'bin' 'LIST'"
run_compare_test tests/backend-list-modes 'anon' 'port' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'anon' 'port' 'asc' 'NLST'"
run_compare_test tests/backend-list-modes 'anon' 'port' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'anon' 'port' 'asc' 'LIST'"
run_compare_test tests/backend-list-modes 'anon' 'port' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'drop' 'pasv' 'bin' 'NLST'"
run_compare_test tests/backend-list-modes 'drop' 'pasv' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'drop' 'pasv' 'bin' 'LIST'"
run_compare_test tests/backend-list-modes 'drop' 'pasv' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'drop' 'pasv' 'asc' 'NLST'"
run_compare_test tests/backend-list-modes 'drop' 'pasv' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'drop' 'pasv' 'asc' 'LIST'"
run_compare_test tests/backend-list-modes 'drop' 'pasv' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'drop' 'port' 'bin' 'NLST'"
run_compare_test tests/backend-list-modes 'drop' 'port' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'drop' 'port' 'bin' 'LIST'"
run_compare_test tests/backend-list-modes 'drop' 'port' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'drop' 'port' 'asc' 'NLST'"
run_compare_test tests/backend-list-modes 'drop' 'port' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'drop' 'port' 'asc' 'LIST'"
run_compare_test tests/backend-list-modes 'drop' 'port' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 Transfer completed (0 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'xfer' 'pasv' 'bin' 'NLST'"
run_compare_test tests/backend-list-modes 'xfer' 'pasv' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'xfer' 'pasv' 'bin' 'LIST'"
run_compare_test tests/backend-list-modes 'xfer' 'pasv' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'xfer' 'pasv' 'asc' 'NLST'"
run_compare_test tests/backend-list-modes 'xfer' 'pasv' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'xfer' 'pasv' 'asc' 'LIST'"
run_compare_test tests/backend-list-modes 'xfer' 'pasv' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'xfer' 'port' 'bin' 'NLST'"
run_compare_test tests/backend-list-modes 'xfer' 'port' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'xfer' 'port' 'bin' 'LIST'"
run_compare_test tests/backend-list-modes 'xfer' 'port' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'xfer' 'port' 'asc' 'NLST'"
run_compare_test tests/backend-list-modes 'xfer' 'port' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Transfer completed (22 bytes sent).^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-list-modes 'xfer' 'port' 'asc' 'LIST'"
run_compare_test tests/backend-list-modes 'xfer' 'port' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 
-rw-r--r--    1 ftp      ftp          4320 
drwxrwxr-x    1 ftp      ftp          4096 
226 Transfer completed (190 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-mkd-rmd #####

runtest() {
local backend="$1"
$backend 'MKD dir' 'MKD dir'
test -d $home/dir && echo dir was created
$backend 'RMD dir' 'RMD dir'
test -d $home/dir && echo dir still exists
$backend 'XMKD dir' 'XRMD dir'
test -d $home/dir && echo dir still exists
}
vecho "Running test tests/backend-mkd-rmd 'anon'"
run_compare_test tests/backend-mkd-rmd 'anon' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-mkd-rmd 'drop'"
run_compare_test tests/backend-mkd-rmd 'drop' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-mkd-rmd 'xfer'"
run_compare_test tests/backend-mkd-rmd 'xfer' <<END_OF_TEST_RESULTS
257 Directory created successfully.^M
550 Could not create directory: File exists^M
dir was created
250 Directory removed successfully.^M
550 Could not remove directory: No such file or directory^M
257 Directory created successfully.^M
250 Directory removed successfully.^M
END_OF_TEST_RESULTS


##### Test tests/backend-list-times #####

runtest() {
currtime=`date +'%b %e %H:%M'`
touch -t `date +%Y%m%d%H%M.00` $home/newfile
xfertest anon pasv retr asc "list empty" | fgrep empty
xfertest anon pasv retr asc "list large" | fgrep large
xfertest anon pasv retr asc "list newfile" | grep "^-rw-rw-r--    1 ftp      ftp             0 $currtime newfile" >/dev/null 2>&1
echo $?
rm -f $home/newfile
}
vecho "Running test tests/backend-list-times "
run_compare_test tests/backend-list-times  <<END_OF_TEST_RESULTS
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18  2003 large^M
0
END_OF_TEST_RESULTS


##### Test tests/backend-list-specials #####

runtest() {
mkdir $home/'[test]'
touch $home/'[test]'/file
xfertest xfer port retr asc "NLST"
echo
xfertest xfer port retr asc "NLST [test]"
rm -r $home/'[test]'
}
vecho "Running test tests/backend-list-specials "
run_compare_test tests/backend-list-specials  <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
[test]^M
empty^M
large^M
subdir^M
226 Transfer completed (30 bytes sent).^M

200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
file^M
226 Transfer completed (6 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-dele #####

runtest() {
local backend="$1"
touch $home/tmp
$backend "DELE tmp"
test -e $home/tmp && echo tmp was not deleted
$backend "DELE tmp"
rm -f $home/tmp
}
vecho "Running test tests/backend-dele 'anon'"
run_compare_test tests/backend-dele 'anon' <<END_OF_TEST_RESULTS
502 Command not supported.^M
tmp was not deleted
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-dele 'drop'"
run_compare_test tests/backend-dele 'drop' <<END_OF_TEST_RESULTS
502 Command not supported.^M
tmp was not deleted
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-dele 'xfer'"
run_compare_test tests/backend-dele 'xfer' <<END_OF_TEST_RESULTS
250 File removed successfully.^M
550 Could not remove file: No such file or directory^M
END_OF_TEST_RESULTS


##### Test tests/backend-type #####

runtest() {
local backend="$1"
testparams $backend TYPE A a "A N" "a n" I i "L 8" "l 8" "AN" "L8"
}
vecho "Running test tests/backend-type 'anon'"
run_compare_test tests/backend-type 'anon' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
501 Unknown transfer type.^M
501 Unknown transfer type.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-type 'drop'"
run_compare_test tests/backend-type 'drop' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
501 Unknown transfer type.^M
501 Unknown transfer type.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-type 'xfer'"
run_compare_test tests/backend-type 'xfer' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
501 Unknown transfer type.^M
501 Unknown transfer type.^M
END_OF_TEST_RESULTS


##### Test tests/backend-mdtm2 #####

runtest() {
local backend="$1"
TZ=UTC touch -t 200101010101.01 $home/newfile
TZ=UTC $backend 'MDTM 20000203040506 newfile' 'MDTM newfile'
( cd $home && echo `TZ=UTC ls -l newfile` )
rm -f $home/newfile
}
vecho "Running test tests/backend-mdtm2 'anon'"
run_compare_test tests/backend-mdtm2 'anon' <<END_OF_TEST_RESULTS
550 Could not determine file time: No such file or directory^M
213 20010101010101^M
-rw-rw-r-- 1 bruce guenter 0 Jan 1 2001 newfile
END_OF_TEST_RESULTS

vecho "Running test tests/backend-mdtm2 'drop'"
run_compare_test tests/backend-mdtm2 'drop' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
-rw-rw-r-- 1 bruce guenter 0 Jan 1 2001 newfile
END_OF_TEST_RESULTS

vecho "Running test tests/backend-mdtm2 'xfer'"
run_compare_test tests/backend-mdtm2 'xfer' <<END_OF_TEST_RESULTS
200 File time set^M
213 20000203040506^M
-rw-rw-r-- 1 bruce guenter 0 Feb 3 2000 newfile
END_OF_TEST_RESULTS


##### Test tests/backend-list-errors #####

runtest() {
xfertest xfer pasv retr asc 'LIST nofile'
}
vecho "Running test tests/backend-list-errors "
run_compare_test tests/backend-list-errors  <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
550 File or directory does not exist.^M
END_OF_TEST_RESULTS


##### Test tests/backend-cdup-pwd #####

runtest() {
local backend="$1"
$backend PWD 'CWD subdir' PWD CDUP PWD CDUP PWD
}
vecho "Running test tests/backend-cdup-pwd 'anon'"
run_compare_test tests/backend-cdup-pwd 'anon' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@/home/subdir"^M
250 Changed directory.^M
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-cdup-pwd 'drop'"
run_compare_test tests/backend-cdup-pwd 'drop' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@/home/subdir"^M
250 Changed directory.^M
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-cdup-pwd 'xfer'"
run_compare_test tests/backend-cdup-pwd 'xfer' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@/home/subdir"^M
250 Changed directory.^M
257 "@TMPDIR@/home"^M
250 Changed directory.^M
257 "@TMPDIR@"^M
END_OF_TEST_RESULTS


##### Test tests/backend-commands #####

runtest() {
local backend="$1"
local command="$2"
local parameter="$3"
if [ $parameter = no ]; then
  cmd=$command
else
  cmd="$command xyzzy"
fi
$backend "$cmd"
}
vecho "Running test tests/backend-commands 'anon' 'QUIT' 'no'"
run_compare_test tests/backend-commands 'anon' 'QUIT' 'no' <<END_OF_TEST_RESULTS
221 Bye.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'QUIT' 'yes'"
run_compare_test tests/backend-commands 'anon' 'QUIT' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'HELP' 'no'"
run_compare_test tests/backend-commands 'anon' 'HELP' 'no' <<END_OF_TEST_RESULTS
502 No help is available.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'HELP' 'yes'"
run_compare_test tests/backend-commands 'anon' 'HELP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'SYST' 'no'"
run_compare_test tests/backend-commands 'anon' 'SYST' 'no' <<END_OF_TEST_RESULTS
215 UNIX Type: L8^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'SYST' 'yes'"
run_compare_test tests/backend-commands 'anon' 'SYST' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'NOOP' 'no'"
run_compare_test tests/backend-commands 'anon' 'NOOP' 'no' <<END_OF_TEST_RESULTS
200 Awaiting your commands, master...^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'NOOP' 'yes'"
run_compare_test tests/backend-commands 'anon' 'NOOP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'USER' 'no'"
run_compare_test tests/backend-commands 'anon' 'USER' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'USER' 'yes'"
run_compare_test tests/backend-commands 'anon' 'USER' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'PASS' 'no'"
run_compare_test tests/backend-commands 'anon' 'PASS' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'PASS' 'yes'"
run_compare_test tests/backend-commands 'anon' 'PASS' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'ACCT' 'no'"
run_compare_test tests/backend-commands 'anon' 'ACCT' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'ACCT' 'yes'"
run_compare_test tests/backend-commands 'anon' 'ACCT' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'TYPE' 'no'"
run_compare_test tests/backend-commands 'anon' 'TYPE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'TYPE' 'yes'"
run_compare_test tests/backend-commands 'anon' 'TYPE' 'yes' <<END_OF_TEST_RESULTS
501 Unknown transfer type.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'STRU' 'no'"
run_compare_test tests/backend-commands 'anon' 'STRU' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'STRU' 'yes'"
run_compare_test tests/backend-commands 'anon' 'STRU' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'MODE' 'no'"
run_compare_test tests/backend-commands 'anon' 'MODE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'MODE' 'yes'"
run_compare_test tests/backend-commands 'anon' 'MODE' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'CWD' 'no'"
run_compare_test tests/backend-commands 'anon' 'CWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'CWD' 'yes'"
run_compare_test tests/backend-commands 'anon' 'CWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'XCWD' 'no'"
run_compare_test tests/backend-commands 'anon' 'XCWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'XCWD' 'yes'"
run_compare_test tests/backend-commands 'anon' 'XCWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'PWD' 'no'"
run_compare_test tests/backend-commands 'anon' 'PWD' 'no' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'PWD' 'yes'"
run_compare_test tests/backend-commands 'anon' 'PWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'XPWD' 'no'"
run_compare_test tests/backend-commands 'anon' 'XPWD' 'no' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'XPWD' 'yes'"
run_compare_test tests/backend-commands 'anon' 'XPWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'CDUP' 'no'"
run_compare_test tests/backend-commands 'anon' 'CDUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'CDUP' 'yes'"
run_compare_test tests/backend-commands 'anon' 'CDUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'XCUP' 'no'"
run_compare_test tests/backend-commands 'anon' 'XCUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'XCUP' 'yes'"
run_compare_test tests/backend-commands 'anon' 'XCUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'SIZE' 'no'"
run_compare_test tests/backend-commands 'anon' 'SIZE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'SIZE' 'yes'"
run_compare_test tests/backend-commands 'anon' 'SIZE' 'yes' <<END_OF_TEST_RESULTS
550 Could not determine file size: No such file or directory^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'MDTM' 'no'"
run_compare_test tests/backend-commands 'anon' 'MDTM' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'anon' 'MDTM' 'yes'"
run_compare_test tests/backend-commands 'anon' 'MDTM' 'yes' <<END_OF_TEST_RESULTS
550 Could not determine file time: No such file or directory^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'QUIT' 'no'"
run_compare_test tests/backend-commands 'drop' 'QUIT' 'no' <<END_OF_TEST_RESULTS
221 Bye.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'QUIT' 'yes'"
run_compare_test tests/backend-commands 'drop' 'QUIT' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'HELP' 'no'"
run_compare_test tests/backend-commands 'drop' 'HELP' 'no' <<END_OF_TEST_RESULTS
502 No help is available.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'HELP' 'yes'"
run_compare_test tests/backend-commands 'drop' 'HELP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'SYST' 'no'"
run_compare_test tests/backend-commands 'drop' 'SYST' 'no' <<END_OF_TEST_RESULTS
215 UNIX Type: L8^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'SYST' 'yes'"
run_compare_test tests/backend-commands 'drop' 'SYST' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'NOOP' 'no'"
run_compare_test tests/backend-commands 'drop' 'NOOP' 'no' <<END_OF_TEST_RESULTS
200 Awaiting your commands, master...^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'NOOP' 'yes'"
run_compare_test tests/backend-commands 'drop' 'NOOP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'USER' 'no'"
run_compare_test tests/backend-commands 'drop' 'USER' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'USER' 'yes'"
run_compare_test tests/backend-commands 'drop' 'USER' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'PASS' 'no'"
run_compare_test tests/backend-commands 'drop' 'PASS' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'PASS' 'yes'"
run_compare_test tests/backend-commands 'drop' 'PASS' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'ACCT' 'no'"
run_compare_test tests/backend-commands 'drop' 'ACCT' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'ACCT' 'yes'"
run_compare_test tests/backend-commands 'drop' 'ACCT' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'TYPE' 'no'"
run_compare_test tests/backend-commands 'drop' 'TYPE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'TYPE' 'yes'"
run_compare_test tests/backend-commands 'drop' 'TYPE' 'yes' <<END_OF_TEST_RESULTS
501 Unknown transfer type.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'STRU' 'no'"
run_compare_test tests/backend-commands 'drop' 'STRU' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'STRU' 'yes'"
run_compare_test tests/backend-commands 'drop' 'STRU' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'MODE' 'no'"
run_compare_test tests/backend-commands 'drop' 'MODE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'MODE' 'yes'"
run_compare_test tests/backend-commands 'drop' 'MODE' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'CWD' 'no'"
run_compare_test tests/backend-commands 'drop' 'CWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'CWD' 'yes'"
run_compare_test tests/backend-commands 'drop' 'CWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'XCWD' 'no'"
run_compare_test tests/backend-commands 'drop' 'XCWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'XCWD' 'yes'"
run_compare_test tests/backend-commands 'drop' 'XCWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'PWD' 'no'"
run_compare_test tests/backend-commands 'drop' 'PWD' 'no' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'PWD' 'yes'"
run_compare_test tests/backend-commands 'drop' 'PWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'XPWD' 'no'"
run_compare_test tests/backend-commands 'drop' 'XPWD' 'no' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'XPWD' 'yes'"
run_compare_test tests/backend-commands 'drop' 'XPWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'CDUP' 'no'"
run_compare_test tests/backend-commands 'drop' 'CDUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'CDUP' 'yes'"
run_compare_test tests/backend-commands 'drop' 'CDUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'XCUP' 'no'"
run_compare_test tests/backend-commands 'drop' 'XCUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'XCUP' 'yes'"
run_compare_test tests/backend-commands 'drop' 'XCUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'SIZE' 'no'"
run_compare_test tests/backend-commands 'drop' 'SIZE' 'no' <<END_OF_TEST_RESULTS
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'SIZE' 'yes'"
run_compare_test tests/backend-commands 'drop' 'SIZE' 'yes' <<END_OF_TEST_RESULTS
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'MDTM' 'no'"
run_compare_test tests/backend-commands 'drop' 'MDTM' 'no' <<END_OF_TEST_RESULTS
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'drop' 'MDTM' 'yes'"
run_compare_test tests/backend-commands 'drop' 'MDTM' 'yes' <<END_OF_TEST_RESULTS
502 Command not supported.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'QUIT' 'no'"
run_compare_test tests/backend-commands 'xfer' 'QUIT' 'no' <<END_OF_TEST_RESULTS
221 Bye.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'QUIT' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'QUIT' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'HELP' 'no'"
run_compare_test tests/backend-commands 'xfer' 'HELP' 'no' <<END_OF_TEST_RESULTS
502 No help is available.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'HELP' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'HELP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'SYST' 'no'"
run_compare_test tests/backend-commands 'xfer' 'SYST' 'no' <<END_OF_TEST_RESULTS
215 UNIX Type: L8^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'SYST' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'SYST' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'NOOP' 'no'"
run_compare_test tests/backend-commands 'xfer' 'NOOP' 'no' <<END_OF_TEST_RESULTS
200 Awaiting your commands, master...^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'NOOP' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'NOOP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'USER' 'no'"
run_compare_test tests/backend-commands 'xfer' 'USER' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'USER' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'USER' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'PASS' 'no'"
run_compare_test tests/backend-commands 'xfer' 'PASS' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'PASS' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'PASS' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'ACCT' 'no'"
run_compare_test tests/backend-commands 'xfer' 'ACCT' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'ACCT' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'ACCT' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'TYPE' 'no'"
run_compare_test tests/backend-commands 'xfer' 'TYPE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'TYPE' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'TYPE' 'yes' <<END_OF_TEST_RESULTS
501 Unknown transfer type.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'STRU' 'no'"
run_compare_test tests/backend-commands 'xfer' 'STRU' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'STRU' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'STRU' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'MODE' 'no'"
run_compare_test tests/backend-commands 'xfer' 'MODE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'MODE' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'MODE' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'CWD' 'no'"
run_compare_test tests/backend-commands 'xfer' 'CWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'CWD' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'CWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'XCWD' 'no'"
run_compare_test tests/backend-commands 'xfer' 'XCWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'XCWD' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'XCWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'PWD' 'no'"
run_compare_test tests/backend-commands 'xfer' 'PWD' 'no' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'PWD' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'PWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'XPWD' 'no'"
run_compare_test tests/backend-commands 'xfer' 'XPWD' 'no' <<END_OF_TEST_RESULTS
257 "@TMPDIR@/home"^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'XPWD' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'XPWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'CDUP' 'no'"
run_compare_test tests/backend-commands 'xfer' 'CDUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'CDUP' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'CDUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'XCUP' 'no'"
run_compare_test tests/backend-commands 'xfer' 'XCUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'XCUP' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'XCUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'SIZE' 'no'"
run_compare_test tests/backend-commands 'xfer' 'SIZE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'SIZE' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'SIZE' 'yes' <<END_OF_TEST_RESULTS
550 Could not determine file size: No such file or directory^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'MDTM' 'no'"
run_compare_test tests/backend-commands 'xfer' 'MDTM' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

vecho "Running test tests/backend-commands 'xfer' 'MDTM' 'yes'"
run_compare_test tests/backend-commands 'xfer' 'MDTM' 'yes' <<END_OF_TEST_RESULTS
550 Could not determine file time: No such file or directory^M
END_OF_TEST_RESULTS


##### Test tests/delete-home #####

runtest() {
rm -r $home/*
}
vecho "Running test tests/delete-home "
run_compare_test tests/delete-home  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/make-error-cvm #####

runtest() {
makecvm 'echo SERVICE=$SERVICE >&2' 'exit 111'
}
vecho "Running test tests/make-error-cvm "
run_compare_test tests/make-error-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/auth-cvm-invocation #####

runtest() {
auth $cvm printenv 'USER username' 'PASS password'
}
vecho "Running test tests/auth-cvm-invocation "
run_compare_test tests/auth-cvm-invocation  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
SERVICE=ftp
530 Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/auth-cvm-service #####

runtest() {
SERVICE=noftp
export SERVICE
auth $cvm printenv 'USER username' 'PASS password'
unset SERVICE
}
vecho "Running test tests/auth-cvm-service "
run_compare_test tests/auth-cvm-service  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
SERVICE=noftp
530 Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/delete-error-cvm #####

runtest() {
rm -f $cvm
}
vecho "Running test tests/delete-error-cvm "
run_compare_test tests/delete-error-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/make-bad-cvm #####

runtest() {
makecvm 'exit 0'
}
vecho "Running test tests/make-bad-cvm "
run_compare_test tests/make-bad-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/auth-cvm-bad-data #####

runtest() {
auth $cvm 'printf Help' 'USER username' 'PASS password'
}
vecho "Running test tests/auth-cvm-bad-data "
run_compare_test tests/auth-cvm-bad-data  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
530 Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/auth-attempts #####

runtest() {

AUTH_ATTEMPTS=1
export AUTH_ATTEMPTS

auth $cvm 'printf Help' \
	'USER username' 'PASS password' \
	'USER username' 'PASS password'

AUTH_ATTEMPTS=0
echo
auth $cvm 'printf Help' \
	'USER username' 'PASS password' \
	'USER username' 'PASS password'

unset AUTH_ATTEMPTS
echo
auth $cvm 'printf Help' \
	'USER username' 'PASS password' \
	'USER username' 'PASS password'
}
vecho "Running test tests/auth-attempts "
run_compare_test tests/auth-attempts  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
530 Authentication failed.^M
421 Too many authentication failures.^M

220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
530 Authentication failed.^M
421 Too many authentication failures.^M

220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
530 Authentication failed.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
530 Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/delete-bad-cvm #####

runtest() {
rm -f $cvm
}
vecho "Running test tests/delete-bad-cvm "
run_compare_test tests/delete-bad-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


rm -rf $tmp
echo $tests_count tests executed, $tests_failed failures
if [ $tests_failed != 0 ]; then exit 1; fi
