//----------------------------------------------------------------------------
// AGG Contribution Pack - Gradients 1 (AGG CP - Gradients 1)
// http://milan.marusinec.sk/aggcp
//
// For Anti-Grain Geometry - Version 2.4 
// http://www.antigrain.org
//
// Contribution Created By:
//  Milan Marusinec alias Milano
//  milan@marusinec.sk
//  Copyright (c) 2007-2008
//
// Permission to copy, use, modify, sell and distribute this software
// is granted provided this copyright notice appears in all copies.
// This software is provided "as is" without express or implied
// warranty, and with no claim as to its suitability for any purpose.
//
// [History] -----------------------------------------------------------------
//
// 03.02.2008-Milano: Ported from Object Pascal code of AggPas
//
#include <stdio.h>

#include "agg_span_gradient_image.h"
#include "agg_span_gradient_contour.h"

#include "agg_basics.h"
#include "agg_array.h"
#include "agg_rendering_buffer.h"
#include "agg_rasterizer_scanline_aa.h"
#include "agg_scanline_p.h"
#include "agg_renderer_scanline.h"
#include "agg_renderer_primitives.h"
#include "agg_conv_curve.h"
#include "agg_conv_stroke.h"
#include "agg_gsv_text.h"
#include "agg_pixfmt_rgb.h"
#include "agg_pixfmt_gray.h"
#include "agg_pixfmt_amask_adaptor.h"
#include "agg_span_allocator.h"
#include "agg_span_gradient.h"
#include "agg_span_interpolator_linear.h"
#include "agg_span_interpolator_trans.h"
#include "agg_alpha_mask_u8.h"
#include "agg_path_storage.h"
#include "agg_gradient_lut.h"
#include "agg_bounding_rect.h"
#include "agg_trans_perspective.h"

#include "platform/agg_platform_support.h"

#include "ctrl/agg_slider_ctrl.h"
#include "ctrl/agg_cbox_ctrl.h"
#include "ctrl/agg_rbox_ctrl.h"
#include "ctrl/agg_polygon_ctrl.h"

#define AGG_BGR24
#include "pixel_formats.h"

enum flip_y_e { flip_y = true };

#define angle 10
#define zoom_up 1.1
#define zoom_down 0.9

//==========================================================gradient_conic
class gradient_conic_angle
{
public:
    static AGG_INLINE int calculate(int x, int y, int d) 
    { 
        double res = atan2(double(y), double(x));
		
		if (res < 0)
		{
			return abs(1600 - agg::uround(fabs(res) * double(d) / agg::pi / 2  ) );
		}
		else;
		{
			return agg::uround(res * double(d) / agg::pi / 2 );
		}

    }
};

//==========================================================spiral
class spiral
{
public:
    spiral(double x, double y, double r1, double r2, double step, double start_angle=0) :
        m_x(x), 
        m_y(y), 
        m_r1(r1), 
        m_r2(r2), 
        m_step(step), 
        m_start_angle(start_angle),
        m_angle(start_angle),
        m_da(agg::deg2rad(4.0)),
        m_dr(m_step / 90.0)
    {
    }

    void rewind(unsigned) 
    { 
        m_angle = m_start_angle; 
        m_curr_r = m_r1; 
        m_start = true; 
    }

    unsigned vertex(double* x, double* y)
    {
        if(m_curr_r > m_r2) return agg::path_cmd_stop;

        *x = m_x + cos(m_angle) * m_curr_r;
        *y = m_y + sin(m_angle) * m_curr_r;
        m_curr_r += m_dr;
        m_angle += m_da;
        if(m_start) 
        {
            m_start = false;
            return agg::path_cmd_move_to;
        }
        return agg::path_cmd_line_to;
    }

private:
    double m_x;
    double m_y;
    double m_r1;
    double m_r2;
    double m_step;
    double m_start_angle;

    double m_angle;
    double m_curr_r;
    double m_da;
    double m_dr;
    bool   m_start;
};

namespace agg
{

const int8u puzzle[] =
{
  0x02,0x51,0xAE,0xFF,0x00,0x5C,0xB0,0xFF,0x0A,0x85,0xFF,0xFF,0x3B,0x87,0x95,0xFF,0x28,0x27,0x12,0xFF,
  0x5C,0x6F,0x03,0xFF,0x88,0xAD,0x08,0xFF,0x90,0xAC,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8B,0xAE,0x00,0xFF,
  0x8C,0xAD,0x00,0xFF,0x9E,0xBE,0x05,0xFF,0xA5,0xC7,0x2B,0xFF,0x5A,0x89,0x67,0xFF,0x17,0x45,0x52,0xFF,
  0x00,0x36,0x80,0xFF,0x03,0x53,0xAA,0xFF,0x00,0x54,0xA4,0xFF,0x00,0x52,0xB1,0xFF,0x04,0x55,0xB1,0xFF,
  0x00,0x50,0xAC,0xFF,0x00,0x52,0xA9,0xFF,0x00,0x4F,0x98,0xFF,0x00,0x43,0x84,0xFF,0x00,0x48,0x9C,0xFF,
  0x06,0x55,0xA4,0xFF,0x03,0x4F,0xA5,0xFF,0x01,0x50,0xAC,0xFF,0x00,0x54,0xAE,0xFF,0x05,0x4E,0xAD,0xFF,
  0x01,0x54,0xBA,0xFF,0x00,0x51,0xAC,0xFF,0x00,0x51,0xAC,0xFF,0x03,0x52,0xAF,0xFF,0x00,0x50,0xB1,0xFF,
  0x03,0x4F,0xAF,0xFF,0x04,0x4F,0xAA,0xFF,0x00,0x56,0xA7,0xFF,0x00,0x55,0xAE,0xFF,0x05,0x4E,0xB5,0xFF,
  0x00,0x53,0xA9,0xFF,0x00,0x52,0xA4,0xFF,0x00,0x51,0xAF,0xFF,0x01,0x53,0xA7,0xFF,0x01,0x53,0xA9,0xFF,
  0x04,0x51,0xAB,0xFF,0x03,0x58,0xB2,0xFF,0x00,0x74,0xF6,0xFF,0x14,0x79,0xD1,0xFF,0x34,0x49,0x22,0xFF,
  0x5D,0x75,0x00,0xFF,0x8A,0xB0,0x01,0xFF,0x8F,0xA4,0x00,0xFF,0x8E,0xB0,0x02,0xFF,0x8B,0xAC,0x07,0xFF,
  0x8D,0xAC,0x00,0xFF,0xC0,0xD9,0x04,0xFF,0x91,0xCF,0x2E,0xFF,0x11,0x39,0x43,0xFF,0x00,0x28,0x58,0xFF,
  0x00,0x4C,0x99,0xFF,0x00,0x54,0xAA,0xFF,0x00,0x50,0xA9,0xFF,0x03,0x53,0xAA,0xFF,0x05,0x52,0xAE,0xFF,
  0x00,0x59,0xB0,0xFF,0x00,0x72,0xEE,0xFF,0x35,0x76,0x92,0xFF,0x2C,0x2C,0x12,0xFF,0x64,0x7A,0x00,0xFF,
  0x89,0xB0,0x00,0xFF,0x88,0xAE,0x00,0xFF,0x91,0xAD,0x00,0xFF,0x84,0xAD,0x06,0xFF,0x8F,0xAE,0x0A,0xFF,
  0x9D,0xBB,0x00,0xFF,0xBE,0xF0,0x13,0xFF,0x57,0x8D,0x76,0xFF,0x08,0x2E,0x41,0xFF,0x01,0x44,0x8A,0xFF,
  0x00,0x52,0xA8,0xFF,0x01,0x4F,0xB3,0xFF,0x05,0x4D,0x95,0xFF,0x00,0x40,0x7E,0xFF,0x04,0x3F,0x8F,0xFF,
  0x0C,0x41,0x8F,0xFF,0x00,0x20,0x3E,0xFF,0x00,0x07,0x12,0xFF,0x08,0x23,0x4E,0xFF,0x00,0x3D,0x8C,0xFF,
  0x00,0x58,0xA8,0xFF,0x03,0x56,0xA6,0xFF,0x04,0x50,0xB0,0xFF,0x03,0x51,0xA6,0xFF,0x04,0x51,0xAB,0xFF,
  0x00,0x53,0xA6,0xFF,0x00,0x52,0xAC,0xFF,0x04,0x56,0xA2,0xFF,0x04,0x55,0xAE,0xFF,0x02,0x53,0xAE,0xFF,
  0x00,0x50,0xAC,0xFF,0x00,0x53,0xA6,0xFF,0x06,0x50,0xA7,0xFF,0x06,0x50,0xA5,0xFF,0x01,0x52,0xAE,0xFF,
  0x01,0x55,0xB5,0xFF,0x00,0x55,0xB1,0xFF,0x02,0x4F,0xAB,0xFF,0x07,0x52,0xB7,0xFF,0x00,0x4F,0xAC,0xFF,
  0x00,0x51,0xA5,0xFF,0x01,0x6D,0xD0,0xFF,0x01,0x72,0xD8,0xFF,0x2B,0x45,0x20,0xFF,0x3E,0x4C,0x00,0xFF,
  0x82,0xA6,0x08,0xFF,0x8E,0xAA,0x00,0xFF,0x8B,0xAD,0x01,0xFF,0x8C,0xAD,0x00,0xFF,0x8D,0xAE,0x00,0xFF,
  0xA3,0xCD,0x00,0xFF,0xB6,0xF7,0x17,0xFF,0x34,0x6C,0x6B,0xFF,0x01,0x16,0x2B,0xFF,0x04,0x31,0x6C,0xFF,
  0x00,0x47,0x9B,0xFF,0x00,0x58,0xA9,0xFF,0x06,0x50,0xA9,0xFF,0x00,0x45,0x87,0xFF,0x00,0x4D,0x98,0xFF,
  0x03,0x56,0xB2,0xFF,0x31,0x60,0x6A,0xFF,0x44,0x55,0x0F,0xFF,0x70,0x97,0x00,0xFF,0x8B,0xB1,0x00,0xFF,
  0x8D,0xAA,0x02,0xFF,0x8E,0xAB,0x03,0xFF,0x8D,0xAD,0x02,0xFF,0x89,0xAB,0x0C,0xFF,0x93,0xBA,0x09,0xFF,
  0xC2,0xEB,0x37,0xFF,0x3B,0x77,0x91,0xFF,0x05,0x3A,0x6E,0xFF,0x00,0x4F,0x99,0xFF,0x00,0x53,0xA7,0xFF,
  0x00,0x51,0xA0,0xFF,0x07,0x37,0x83,0xFF,0x0C,0x26,0x41,0xFF,0x00,0x21,0x46,0xFF,0x16,0x56,0x71,0xFF,
  0x32,0x6B,0x65,0xFF,0x14,0x27,0x2D,0xFF,0x00,0x08,0x02,0xFF,0x00,0x1B,0x29,0xFF,0x00,0x38,0x82,0xFF,
  0x00,0x50,0xA8,0xFF,0x01,0x55,0xAB,0xFF,0x07,0x53,0xB3,0xFF,0x00,0x51,0xA3,0xFF,0x06,0x52,0xB4,0xFF,
  0x00,0x58,0xA8,0xFF,0x00,0x4F,0xAB,0xFF,0x00,0x4D,0xB9,0xFF,0x07,0x51,0xB2,0xFF,0x06,0x52,0xAA,0xFF,
  0x03,0x50,0xB8,0xFF,0x09,0x49,0xA0,0xFF,0x00,0x3C,0x7B,0xFF,0x00,0x2F,0x62,0xFF,0x02,0x27,0x5E,0xFF,
  0x04,0x30,0x5D,0xFF,0x00,0x38,0x74,0xFF,0x00,0x48,0x9D,0xFF,0x04,0x56,0xAC,0xFF,0x02,0x51,0xAE,0xFF,
  0x06,0x67,0xD0,0xFF,0x00,0x67,0xE5,0xFF,0x40,0x5C,0x45,0xFF,0x49,0x54,0x06,0xFF,0x78,0x9D,0x00,0xFF,
  0x8E,0xB0,0x00,0xFF,0x8B,0xAB,0x00,0xFF,0x8B,0xA9,0x00,0xFF,0x8E,0xAD,0x09,0xFF,0x93,0xB3,0x0A,0xFF,
  0xCD,0xFD,0x07,0xFF,0xAB,0xE7,0x4B,0xFF,0x29,0x49,0x58,0xFF,0x01,0x0E,0x17,0xFF,0x0A,0x23,0x42,0xFF,
  0x00,0x30,0x69,0xFF,0x00,0x46,0x90,0xFF,0x01,0x12,0x2C,0xFF,0x00,0x24,0x56,0xFF,0x19,0x52,0x70,0xFF,
  0x4A,0x72,0x3E,0xFF,0x74,0x92,0x0C,0xFF,0x84,0xA9,0x00,0xFF,0x8D,0xA9,0x00,0xFF,0x91,0xAC,0x05,0xFF,
  0x8D,0xAE,0x05,0xFF,0x8A,0xA7,0x00,0xFF,0x90,0xB5,0x05,0xFF,0x9C,0xC9,0x0A,0xFF,0xA8,0xC3,0x34,0xFF,
  0x13,0x4B,0x70,0xFF,0x00,0x35,0x5D,0xFF,0x00,0x49,0x92,0xFF,0x02,0x4D,0x9E,0xFF,0x07,0x51,0x98,0xFF,
  0x16,0x57,0x7F,0xFF,0x29,0x59,0x5D,0xFF,0x47,0x6C,0x39,0xFF,0x93,0xBC,0x2F,0xFF,0xD0,0xFF,0x15,0xFF,
  0xCF,0xF7,0x36,0xFF,0x5E,0x85,0x59,0xFF,0x07,0x10,0x1F,0xFF,0x07,0x0A,0x29,0xFF,0x00,0x43,0x84,0xFF,
  0x03,0x56,0xB2,0xFF,0x04,0x49,0xA4,0xFF,0x00,0x55,0xA9,0xFF,0x00,0x55,0xB6,0xFF,0x01,0x50,0xAF,0xFF,
  0x08,0x55,0xAF,0xFF,0x00,0x58,0xA6,0xFF,0x02,0x56,0xAE,0xFF,0x00,0x50,0xAA,0xFF,0x00,0x53,0xB6,0xFF,
  0x0A,0x40,0x6C,0xFF,0x00,0x18,0x2C,0xFF,0x02,0x09,0x0F,0xFF,0x04,0x00,0x02,0xFF,0x04,0x02,0x05,0xFF,
  0x04,0x0B,0x15,0xFF,0x00,0x18,0x37,0xFF,0x03,0x42,0x85,0xFF,0x00,0x55,0xAF,0xFF,0x00,0x6E,0xCE,0xFF,
  0x17,0x68,0xA9,0xFF,0x42,0x5D,0x28,0xFF,0x69,0x7D,0x00,0xFF,0x82,0xA7,0x01,0xFF,0x8B,0xAD,0x00,0xFF,
  0x8A,0xAC,0x00,0xFF,0x90,0xAD,0x05,0xFF,0x8A,0xAC,0x00,0xFF,0x8B,0xAE,0x00,0xFF,0xA3,0xCE,0x00,0xFF,
  0xCF,0xF8,0x30,0xFF,0xAE,0xD0,0x57,0xFF,0x31,0x5F,0x45,0xFF,0x01,0x15,0x2D,0xFF,0x00,0x14,0x1F,0xFF,
  0x03,0x0C,0x2B,0xFF,0x28,0x5E,0x68,0xFF,0x3E,0x78,0x50,0xFF,0x74,0xAA,0x3A,0xFF,0x82,0xA2,0x19,0xFF,
  0x85,0xA5,0x02,0xFF,0x87,0xB0,0x00,0xFF,0x8E,0xAF,0x00,0xFF,0x8B,0xA8,0x02,0xFF,0x8B,0xAE,0x00,0xFF,
  0x8B,0xB0,0x01,0xFF,0x94,0xB8,0x00,0xFF,0xB1,0xD8,0x01,0xFF,0xAA,0xC5,0x2A,0xFF,0x1E,0x46,0x5F,0xFF,
  0x0C,0x2D,0x4C,0xFF,0x07,0x3D,0x85,0xFF,0x16,0x56,0x79,0xFF,0x1B,0x59,0x72,0xFF,0x4B,0x83,0x52,0xFF,
  0x93,0xB9,0x2E,0xFF,0xAD,0xCA,0x10,0xFF,0xA8,0xCB,0x00,0xFF,0xB0,0xD8,0x06,0xFF,0xCD,0xFF,0x03,0xFF,
  0xD1,0xFF,0x11,0xFF,0x7D,0xA2,0x42,0xFF,0x04,0x08,0x14,0xFF,0x00,0x24,0x51,0xFF,0x00,0x53,0xAE,0xFF,
  0x06,0x54,0xA7,0xFF,0x05,0x51,0xB1,0xFF,0x00,0x53,0xA1,0xFF,0x00,0x50,0xAC,0xFF,0x00,0x52,0xB2,0xFF,
  0x00,0x51,0xAB,0xFF,0x03,0x53,0xA8,0xFF,0x01,0x56,0xB0,0xFF,0x0B,0x5A,0x95,0xFF,0x21,0x49,0x48,0xFF,
  0x36,0x4A,0x2F,0xFF,0x5F,0x78,0x25,0xFF,0x6C,0xA2,0x42,0xFF,0x53,0x8F,0x5C,0xFF,0x33,0x4A,0x36,0xFF,
  0x00,0x05,0x00,0xFF,0x00,0x21,0x40,0xFF,0x07,0x43,0x81,0xFF,0x00,0x4B,0x9D,0xFF,0x1A,0x41,0x3E,0xFF,
  0x3D,0x4F,0x05,0xFF,0x7B,0x98,0x00,0xFF,0x89,0xAD,0x05,0xFF,0x8E,0xAA,0x0A,0xFF,0x8B,0xAE,0x00,0xFF,
  0x8D,0xAD,0x04,0xFF,0x8A,0xAB,0x00,0xFF,0x90,0xAE,0x00,0xFF,0x84,0xB6,0x00,0xFF,0xAC,0xD7,0x03,0xFF,
  0xDA,0xFA,0x1D,0xFF,0xB7,0xF7,0x49,0xFF,0x8C,0xBA,0x72,0xFF,0x4D,0x78,0x6F,0xFF,0x3B,0x6D,0x54,0xFF,
  0xD6,0xFF,0x1A,0xFF,0xB4,0xE0,0x11,0xFF,0xA3,0xD0,0x03,0xFF,0x91,0xB4,0x00,0xFF,0x8B,0xAC,0x00,0xFF,
  0x8F,0xAC,0x04,0xFF,0x8D,0xA9,0x00,0xFF,0x8F,0xAE,0x00,0xFF,0x8B,0xAB,0x02,0xFF,0x8B,0xAA,0x06,0xFF,
  0x8D,0xB3,0x00,0xFF,0xA9,0xD4,0x00,0xFF,0xD3,0xFE,0x2A,0xFF,0x72,0xA9,0x43,0xFF,0x05,0x33,0x40,0xFF,
  0x0D,0x40,0x6D,0xFF,0x33,0x74,0x8A,0xFF,0x5B,0x82,0x53,0xFF,0x95,0xB6,0x0F,0xFF,0x9C,0xC4,0x0B,0xFF,
  0x87,0xB7,0x00,0xFF,0x8A,0xB3,0x00,0xFF,0x89,0xA9,0x00,0xFF,0x92,0xB4,0x00,0xFF,0xCE,0xF4,0x03,0xFF,
  0xD0,0xFF,0x17,0xFF,0x0F,0x29,0x28,0xFF,0x05,0x1D,0x39,0xFF,0x00,0x48,0xA0,0xFF,0x00,0x55,0xAD,0xFF,
  0x09,0x50,0xAC,0xFF,0x03,0x52,0xAF,0xFF,0x08,0x53,0xAE,0xFF,0x00,0x52,0xAE,0xFF,0x00,0x4D,0xB5,0xFF,
  0x00,0x5E,0xB2,0xFF,0x11,0x77,0xCB,0xFF,0x33,0x71,0x6E,0xFF,0x57,0x6C,0x0F,0xFF,0x87,0xA5,0x00,0xFF,
  0x9E,0xCF,0x02,0xFF,0xC4,0xF7,0x06,0xFF,0xCA,0xFF,0x0E,0xFF,0xDA,0xFE,0x22,0xFF,0x7C,0x9C,0x31,0xFF,
  0x0E,0x40,0x73,0xFF,0x00,0x3C,0x83,0xFF,0x1A,0x5D,0x65,0xFF,0x4C,0x5B,0x06,0xFF,0x66,0x79,0x05,0xFF,
  0x87,0xA6,0x02,0xFF,0x89,0xAB,0x00,0xFF,0x8F,0xAC,0x04,0xFF,0x8E,0xB0,0x00,0xFF,0x87,0xA9,0x00,0xFF,
  0x8E,0xAD,0x07,0xFF,0x8F,0xAB,0x09,0xFF,0x90,0xAC,0x00,0xFF,0x93,0xAF,0x02,0xFF,0xA8,0xCE,0x00,0xFF,
  0xBF,0xF6,0x07,0xFF,0xCD,0xFF,0x06,0xFF,0xDA,0xFB,0x14,0xFF,0xD2,0xFF,0x20,0xFF,0x99,0xBF,0x00,0xFF,
  0x98,0xB3,0x02,0xFF,0x92,0xAA,0x00,0xFF,0x89,0xA8,0x04,0xFF,0x8D,0xAF,0x00,0xFF,0x8A,0xAF,0x00,0xFF,
  0x89,0xAA,0x03,0xFF,0x90,0xA9,0x07,0xFF,0x8E,0xAF,0x00,0xFF,0x8F,0xA9,0x00,0xFF,0x89,0xAF,0x00,0xFF,
  0x92,0xBB,0x05,0xFF,0xB7,0xE3,0x08,0xFF,0xC0,0xF7,0x28,0xFF,0x76,0xB8,0x7B,0xFF,0x55,0x7F,0x73,0xFF,
  0x8E,0xBA,0x27,0xFF,0x93,0xBF,0x0A,0xFF,0x88,0xA9,0x00,0xFF,0x8D,0xA8,0x01,0xFF,0x8B,0xAA,0x04,0xFF,
  0x89,0xAE,0x00,0xFF,0x90,0xB1,0x0A,0xFF,0x8B,0xA8,0x00,0xFF,0x9C,0xD1,0x00,0xFF,0xCE,0xFB,0x00,0xFF,
  0x3A,0x59,0x4A,0xFF,0x00,0x07,0x00,0xFF,0x04,0x3C,0x79,0xFF,0x01,0x53,0xA9,0xFF,0x00,0x51,0xAB,0xFF,
  0x00,0x54,0xB1,0xFF,0x00,0x4F,0xA9,0xFF,0x01,0x56,0xA7,0xFF,0x04,0x53,0xB2,0xFF,0x00,0x62,0xC8,0xFF,
  0x28,0x7A,0xB4,0xFF,0x47,0x66,0x2D,0xFF,0x6A,0x80,0x04,0xFF,0x81,0xA6,0x00,0xFF,0x8E,0xB0,0x00,0xFF,
  0x9C,0xB0,0x03,0xFF,0x95,0xB9,0x01,0xFF,0xC0,0xEF,0x01,0xFF,0xCD,0xF7,0x00,0xFF,0xB8,0xDF,0x20,0xFF,
  0xAA,0xCE,0x26,0xFF,0x9E,0xC5,0x14,0xFF,0x9D,0xBB,0x01,0xFF,0x8D,0xAC,0x00,0xFF,0x8D,0xAB,0x00,0xFF,
  0x8E,0xAF,0x00,0xFF,0x8B,0xAD,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8D,0xAD,0x00,0xFF,
  0x84,0xAC,0x00,0xFF,0x93,0xB1,0x04,0xFF,0x83,0xB0,0x00,0xFF,0x8D,0xA4,0x00,0xFF,0x85,0xB2,0x00,0xFF,
  0x92,0xB6,0x00,0xFF,0x99,0xC1,0x00,0xFF,0x99,0xBC,0x00,0xFF,0x88,0xAF,0x08,0xFF,0x8E,0xAB,0x01,0xFF,
  0x8E,0xAD,0x00,0xFF,0x8A,0xAE,0x02,0xFF,0x8E,0xAE,0x03,0xFF,0x89,0xA7,0x00,0xFF,0x8D,0xB4,0x01,0xFF,
  0x87,0xAC,0x00,0xFF,0x8C,0xAD,0x00,0xFF,0x89,0xAD,0x05,0xFF,0x8C,0xAD,0x00,0xFF,0x94,0xA8,0x09,0xFF,
  0x96,0xB6,0x00,0xFF,0xBE,0xDA,0x00,0xFF,0xBE,0xFB,0x00,0xFF,0xB5,0xDE,0x00,0xFF,0x97,0xBF,0x00,0xFF,
  0x8E,0xB8,0x00,0xFF,0x89,0xB0,0x00,0xFF,0x8F,0xAE,0x08,0xFF,0x90,0xAB,0x00,0xFF,0x8F,0xAD,0x00,0xFF,
  0x8A,0xAA,0x00,0xFF,0x8C,0xAD,0x04,0xFF,0x99,0xB2,0x00,0xFF,0xCF,0xFF,0x0C,0xFF,0x48,0x78,0x46,0xFF,
  0x02,0x07,0x0B,0xFF,0x00,0x36,0x7F,0xFF,0x07,0x51,0xB0,0xFF,0x00,0x54,0xA9,0xFF,0x03,0x4F,0xAD,0xFF,
  0x01,0x52,0xAD,0xFF,0x05,0x51,0xAF,0xFF,0x00,0x55,0xB1,0xFF,0x02,0x70,0xEB,0xFF,0x38,0x74,0x98,0xFF,
  0x3D,0x48,0x03,0xFF,0x76,0x9C,0x08,0xFF,0x97,0xB7,0x00,0xFF,0x88,0xAE,0x00,0xFF,0x91,0xA8,0x00,0xFF,
  0x8C,0xB1,0x01,0xFF,0x88,0xAF,0x00,0xFF,0x91,0xB4,0x00,0xFF,0xAB,0xD9,0x00,0xFF,0xB0,0xDE,0x00,0xFF,
  0xA5,0xCE,0x06,0xFF,0x8E,0xB7,0x05,0xFF,0x8D,0xAE,0x09,0xFF,0x91,0xAC,0x00,0xFF,0x8A,0xA9,0x03,0xFF,
  0x87,0xAF,0x01,0xFF,0x8E,0xAB,0x01,0xFF,0x8F,0xA8,0x06,0xFF,0x8B,0xAD,0x00,0xFF,0x90,0xAD,0x00,0xFF,
  0x86,0xAC,0x00,0xFF,0x90,0xAC,0x02,0xFF,0x94,0xAC,0x00,0xFF,0x8B,0xAB,0x00,0xFF,0x90,0xAD,0x03,0xFF,
  0x89,0xAB,0x00,0xFF,0x94,0xAB,0x03,0xFF,0x8A,0xB0,0x00,0xFF,0x90,0xAC,0x02,0xFF,0x91,0xAC,0x00,0xFF,
  0x8B,0xA6,0x00,0xFF,0x8E,0xAF,0x00,0xFF,0x8C,0xAF,0x00,0xFF,0x8A,0xAD,0x00,0xFF,0x8A,0xAA,0x07,0xFF,
  0x8D,0xAF,0x00,0xFF,0x8E,0xAB,0x01,0xFF,0x8E,0xAA,0x00,0xFF,0x88,0xAC,0x04,0xFF,0x89,0xAE,0x00,0xFF,
  0x8B,0xAC,0x00,0xFF,0x91,0xAD,0x03,0xFF,0x8B,0xAB,0x00,0xFF,0x83,0xA4,0x00,0xFF,0x86,0xA2,0x02,0xFF,
  0x89,0xA3,0x02,0xFF,0x85,0xA4,0x00,0xFF,0x84,0xA5,0x00,0xFF,0x8D,0xAE,0x00,0xFF,0x8D,0xAA,0x00,0xFF,
  0x8C,0xAC,0x09,0xFF,0x8F,0xB8,0x00,0xFF,0xC2,0xED,0x08,0xFF,0x49,0x63,0x4A,0xFF,0x00,0x1A,0x2D,0xFF,
  0x00,0x44,0x85,0xFF,0x01,0x4D,0xAD,0xFF,0x00,0x55,0xAB,0xFF,0x01,0x52,0xAB,0xFF,0x04,0x4F,0xB5,0xFF,
  0x03,0x51,0xA6,0xFF,0x00,0x5D,0xAA,0xFF,0x0D,0x84,0xFF,0xFF,0x4A,0x79,0x73,0xFF,0x1D,0x29,0x00,0xFF,
  0x6B,0x80,0x07,0xFF,0x8F,0xAF,0x02,0xFF,0x84,0xB1,0x00,0xFF,0x8F,0xAC,0x04,0xFF,0x90,0xAA,0x00,0xFF,
  0x8E,0xAE,0x01,0xFF,0x8A,0xAD,0x00,0xFF,0x8B,0xAC,0x05,0xFF,0x8D,0xB0,0x00,0xFF,0x8A,0xAE,0x00,0xFF,
  0x8A,0xAD,0x00,0xFF,0x8C,0xAD,0x04,0xFF,0x8E,0xAC,0x00,0xFF,0x8A,0xAB,0x04,0xFF,0x8E,0xB1,0x00,0xFF,
  0x8E,0xA8,0x06,0xFF,0x91,0xAD,0x03,0xFF,0x89,0xAB,0x00,0xFF,0x8D,0xA9,0x00,0xFF,0x8F,0xAD,0x00,0xFF,
  0x8D,0xAC,0x08,0xFF,0x8A,0xAB,0x06,0xFF,0x8C,0xAD,0x00,0xFF,0x8D,0xAE,0x00,0xFF,0x8A,0xAA,0x00,0xFF,
  0x8E,0xAD,0x07,0xFF,0x88,0xAE,0x03,0xFF,0x8D,0xAD,0x02,0xFF,0x8B,0xA8,0x02,0xFF,0x8E,0xB0,0x04,0xFF,
  0x8A,0xAF,0x00,0xFF,0x89,0xA6,0x00,0xFF,0x93,0xAC,0x08,0xFF,0x93,0xAB,0x00,0xFF,0x8E,0xAD,0x07,0xFF,
  0x84,0xAA,0x00,0xFF,0x8C,0xB1,0x00,0xFF,0x91,0xAE,0x08,0xFF,0x8F,0xAB,0x00,0xFF,0x83,0xA4,0x0B,0xFF,
  0x7A,0x9C,0x00,0xFF,0x69,0x85,0x00,0xFF,0x52,0x6B,0x03,0xFF,0x47,0x5E,0x04,0xFF,0x54,0x70,0x00,0xFF,
  0x62,0x7E,0x05,0xFF,0x75,0x90,0x03,0xFF,0x8C,0xAD,0x00,0xFF,0x89,0xAA,0x03,0xFF,0x82,0xA9,0x00,0xFF,
  0x90,0xA9,0x0F,0xFF,0x85,0xA6,0x0F,0xFF,0x25,0x47,0x49,0xFF,0x00,0x29,0x5D,0xFF,0x03,0x4C,0x99,0xFF,
  0x04,0x51,0xAD,0xFF,0x01,0x55,0xAB,0xFF,0x04,0x4F,0xB4,0xFF,0x00,0x56,0xB0,0xFF,0x03,0x4F,0xA7,0xFF,
  0x04,0x5A,0xBD,0xFF,0x08,0x7F,0xFF,0xFF,0x46,0x7E,0x71,0xFF,0x1B,0x19,0x04,0xFF,0x5D,0x7C,0x00,0xFF,
  0x8A,0xAB,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8B,0xAA,0x04,0xFF,0x8C,0xB2,0x00,0xFF,0x86,0xAA,0x00,0xFF,
  0x8B,0xAC,0x05,0xFF,0x8C,0xAD,0x06,0xFF,0x8E,0xAD,0x00,0xFF,0x93,0xA8,0x0D,0xFF,0x8A,0xAF,0x07,0xFF,
  0x85,0xAC,0x00,0xFF,0x8C,0xAD,0x00,0xFF,0x91,0xB0,0x00,0xFF,0x89,0xAA,0x00,0xFF,0x8A,0xAD,0x00,0xFF,
  0x89,0xAB,0x00,0xFF,0x8F,0xB0,0x01,0xFF,0x8C,0xAC,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8E,0xAB,0x03,0xFF,
  0x8B,0xAB,0x00,0xFF,0x93,0xAE,0x01,0xFF,0x8B,0xA8,0x02,0xFF,0x8D,0xB0,0x00,0xFF,0x8A,0xAB,0x00,0xFF,
  0x8B,0xAB,0x00,0xFF,0x88,0xAA,0x00,0xFF,0x90,0xB1,0x00,0xFF,0x8F,0xA8,0x02,0xFF,0x89,0xAA,0x00,0xFF,
  0x8A,0xB5,0x06,0xFF,0x87,0xB1,0x00,0xFF,0x8C,0xAB,0x00,0xFF,0x93,0xA9,0x00,0xFF,0x8C,0xA9,0x00,0xFF,
  0x8C,0xAB,0x05,0xFF,0x8A,0xA8,0x00,0xFF,0x86,0xA9,0x00,0xFF,0x71,0x93,0x01,0xFF,0x5F,0x7E,0x00,0xFF,
  0x3D,0x49,0x0B,0xFF,0x35,0x40,0x16,0xFF,0x3F,0x57,0x0B,0xFF,0x37,0x4D,0x0D,0xFF,0x1C,0x28,0x00,0xFF,
  0x3C,0x48,0x0A,0xFF,0x56,0x68,0x04,0xFF,0x64,0x7C,0x04,0xFF,0x52,0x6B,0x10,0xFF,0x57,0x7E,0x2F,0xFF,
  0x46,0x71,0x43,0xFF,0x1D,0x55,0x70,0xFF,0x08,0x46,0x93,0xFF,0x01,0x4F,0xA1,0xFF,0x05,0x53,0xA5,0xFF,
  0x03,0x56,0xA4,0xFF,0x00,0x4D,0xAC,0xFF,0x02,0x4D,0xB2,0xFF,0x00,0x54,0xAB,0xFF,0x03,0x59,0xB0,0xFF,
  0x05,0x85,0xFF,0xFF,0x44,0x8B,0x79,0xFF,0x01,0x12,0x00,0xFF,0x53,0x6D,0x00,0xFF,0x88,0xA3,0x00,0xFF,
  0x93,0xA9,0x06,0xFF,0x8E,0xAC,0x00,0xFF,0x8D,0xAB,0x00,0xFF,0x90,0xAC,0x00,0xFF,0x90,0xAC,0x09,0xFF,
  0x89,0xB1,0x01,0xFF,0x73,0x98,0x00,0xFF,0x75,0x95,0x03,0xFF,0x7F,0x99,0x00,0xFF,0x85,0xA3,0x05,0xFF,
  0x8B,0xAF,0x07,0xFF,0x93,0xB2,0x00,0xFF,0x8D,0xA9,0x06,0xFF,0x8B,0xB2,0x01,0xFF,0x89,0xAE,0x00,0xFF,
  0x8D,0xAA,0x00,0xFF,0x83,0xB3,0x00,0xFF,0x8C,0xAD,0x00,0xFF,0x90,0xAC,0x02,0xFF,0x8A,0xAA,0x00,0xFF,
  0x90,0xAA,0x09,0xFF,0x89,0xAF,0x00,0xFF,0x88,0xAE,0x01,0xFF,0x91,0xAA,0x06,0xFF,0x86,0xA5,0x02,0xFF,
  0x83,0xA9,0x00,0xFF,0x85,0xAE,0x00,0xFF,0x88,0xAD,0x05,0xFF,0x8E,0xAF,0x00,0xFF,0x8D,0xA8,0x03,0xFF,
  0x8F,0xAC,0x06,0xFF,0x8D,0xAC,0x06,0xFF,0x8B,0xAE,0x0A,0xFF,0x8C,0xAD,0x00,0xFF,0x8B,0xB1,0x02,0xFF,
  0x89,0xAC,0x00,0xFF,0x74,0x94,0x0D,0xFF,0x56,0x6E,0x00,0xFF,0x56,0x7A,0x20,0xFF,0x31,0x59,0x50,0xFF,
  0x00,0x59,0xB3,0xFF,0x00,0x76,0xFF,0xFF,0x2C,0x89,0xB5,0xFF,0x3F,0x5E,0x3F,0xFF,0x24,0x34,0x19,0xFF,
  0x22,0x2E,0x00,0xFF,0x2E,0x3D,0x14,0xFF,0x35,0x64,0x5A,0xFF,0x19,0x67,0x8D,0xFF,0x0D,0x5A,0x92,0xFF,
  0x05,0x53,0x93,0xFF,0x04,0x4D,0xAB,0xFF,0x03,0x4F,0xAF,0xFF,0x02,0x52,0xAB,0xFF,0x00,0x52,0xAA,0xFF,
  0x02,0x55,0xB5,0xFF,0x08,0x54,0xAA,0xFF,0x00,0x50,0xB0,0xFF,0x00,0x56,0xAE,0xFF,0x01,0x77,0xFB,0xFF,
  0x3A,0x94,0xD2,0xFF,0x22,0x2B,0x10,0xFF,0x2E,0x2B,0x00,0xFF,0x57,0x78,0x03,0xFF,0x7A,0x99,0x00,0xFF,
  0x7D,0xA6,0x00,0xFF,0x8B,0xB1,0x0C,0xFF,0x86,0xA5,0x02,0xFF,0x80,0x9A,0x00,0xFF,0x61,0x75,0x00,0xFF,
  0x4E,0x62,0x0D,0xFF,0x52,0x75,0x1B,0xFF,0x46,0x51,0x03,0xFF,0x4A,0x51,0x0B,0xFF,0x6E,0x8E,0x00,0xFF,
  0x89,0xB0,0x00,0xFF,0x8F,0xAD,0x00,0xFF,0x8C,0xAC,0x00,0xFF,0x90,0xAF,0x00,0xFF,0x8D,0xAB,0x00,0xFF,
  0x86,0xAC,0x00,0xFF,0x91,0xAA,0x04,0xFF,0x8A,0xAC,0x00,0xFF,0x8A,0xAE,0x02,0xFF,0x85,0xA4,0x00,0xFF,
  0x7B,0x9E,0x00,0xFF,0x74,0x98,0x00,0xFF,0x84,0x9B,0x05,0xFF,0x3C,0x45,0x06,0xFF,0x54,0x68,0x07,0xFF,
  0x74,0x92,0x00,0xFF,0x85,0xA1,0x00,0xFF,0x90,0xAB,0x06,0xFF,0x8B,0xAB,0x00,0xFF,0x8B,0xAE,0x00,0xFF,
  0x8B,0xAB,0x00,0xFF,0x84,0xAA,0x00,0xFF,0x91,0xB0,0x00,0xFF,0x90,0xB4,0x0A,0xFF,0x90,0xA8,0x08,0xFF,
  0x6D,0x8A,0x1C,0xFF,0x2A,0x48,0x48,0xFF,0x0D,0x59,0xAF,0xFF,0x00,0x73,0xD8,0xFF,0x00,0x5C,0xB6,0xFF,
  0x07,0x71,0xE9,0xFF,0x00,0x86,0xF9,0xFF,0x0A,0x8E,0xF2,0xFF,0x24,0x78,0xD2,0xFF,0x2A,0x5C,0x8F,0xFF,
  0x14,0x5E,0x8F,0xFF,0x00,0x64,0xCA,0xFF,0x09,0x72,0xDB,0xFF,0x02,0x63,0xC2,0xFF,0x00,0x53,0xAD,0xFF,
  0x02,0x51,0xAD,0xFF,0x06,0x55,0xB4,0xFF,0x00,0x53,0xAD,0xFF,0x00,0x4E,0xAA,0xFF,0x00,0x53,0xAF,0xFF,
  0x02,0x4C,0xAB,0xFF,0x06,0x55,0xB1,0xFF,0x03,0x52,0xB1,0xFF,0x02,0x75,0xF2,0xFF,0x29,0x91,0xFF,0xFF,
  0x40,0x84,0x99,0xFF,0x38,0x42,0x1D,0xFF,0x20,0x30,0x03,0xFF,0x3D,0x49,0x0B,0xFF,0x4F,0x6A,0x00,0xFF,
  0x56,0x71,0x00,0xFF,0x5B,0x73,0x03,0xFF,0x4B,0x5D,0x13,0xFF,0x37,0x47,0x20,0xFF,0x23,0x46,0x6E,0xFF,
  0x1F,0x74,0xAB,0xFF,0x30,0x5B,0x3F,0xFF,0x17,0x22,0x00,0xFF,0x47,0x5A,0x09,0xFF,0x7B,0x9E,0x02,0xFF,
  0x8B,0xAC,0x00,0xFF,0x8B,0xAE,0x00,0xFF,0x8F,0xAA,0x00,0xFF,0x8C,0xAA,0x0A,0xFF,0x94,0xA9,0x04,0xFF,
  0x94,0xAB,0x05,0xFF,0x81,0xA5,0x00,0xFF,0x6F,0x95,0x02,0xFF,0x51,0x68,0x00,0xFF,0x38,0x46,0x08,0xFF,
  0x2B,0x38,0x0A,0xFF,0x28,0x2F,0x05,0xFF,0x2C,0x35,0x1A,0xFF,0x18,0x20,0x0B,0xFF,0x38,0x4C,0x05,0xFF,
  0x61,0x81,0x06,0xFF,0x81,0xA5,0x00,0xFF,0x90,0xAD,0x07,0xFF,0x8C,0xAA,0x00,0xFF,0x8A,0xAF,0x00,0xFF,
  0x93,0xAF,0x05,0xFF,0x89,0xA7,0x00,0xFF,0x9F,0xD4,0x00,0xFF,0xA7,0xCF,0x31,0xFF,0x42,0x77,0x4B,0xFF,
  0x08,0x40,0x6F,0xFF,0x00,0x54,0xAA,0xFF,0x00,0x56,0xB7,0xFF,0x00,0x50,0xAB,0xFF,0x00,0x58,0xB9,0xFF,
  0x06,0x68,0xC9,0xFF,0x00,0x68,0xDE,0xFF,0x09,0x86,0xFF,0xFF,0x00,0x6E,0xDE,0xFF,0x02,0x55,0xBB,0xFF,
  0x03,0x56,0xB6,0xFF,0x00,0x58,0xBE,0xFF,0x00,0x59,0xB2,0xFF,0x02,0x50,0xB2,0xFF,0x00,0x52,0xA3,0xFF,
  0x01,0x51,0xA8,0xFF,0x00,0x55,0xA7,0xFF,0x02,0x54,0xAA,0xFF,0x01,0x51,0xA8,0xFF,0x02,0x54,0xA8,0xFF,
  0x05,0x4E,0xAA,0xFF,0x00,0x54,0xA4,0xFF,0x00,0x57,0xB2,0xFF,0x00,0x76,0xF1,0xFF,0x0A,0x88,0xEE,0xFF,
  0x3C,0x94,0xD3,0xFF,0x46,0x7E,0x71,0xFF,0x44,0x61,0x31,0xFF,0x40,0x5C,0x29,0xFF,0x49,0x65,0x28,0xFF,
  0x54,0x84,0x54,0xFF,0x4D,0x94,0x82,0xFF,0x20,0x67,0x93,0xFF,0x06,0x60,0xC0,0xFF,0x00,0x87,0xFF,0xFF,
  0x36,0x85,0xC8,0xFF,0x32,0x50,0x2C,0xFF,0x47,0x58,0x11,0xFF,0x7B,0x9B,0x00,0xFF,0x8B,0xAC,0x00,0xFF,
  0x8A,0xB0,0x05,0xFF,0x90,0xAC,0x00,0xFF,0x8B,0xAC,0x00,0xFF,0x87,0xAB,0x00,0xFF,0x8B,0xB0,0x00,0xFF,
  0x8D,0xB3,0x06,0xFF,0x52,0x74,0x38,0xFF,0x30,0x5D,0x48,0xFF,0x33,0x61,0x5E,0xFF,0x35,0x5A,0x52,0xFF,
  0x2E,0x4F,0x32,0xFF,0x05,0x7C,0xDA,0xFF,0x2B,0x64,0x81,0xFF,0x2B,0x33,0x1E,0xFF,0x38,0x40,0x05,0xFF,
  0x68,0x88,0x00,0xFF,0x90,0xAE,0x01,0xFF,0x8D,0xA9,0x00,0xFF,0x8C,0xB1,0x01,0xFF,0x88,0xAF,0x00,0xFF,
  0x90,0xAB,0x00,0xFF,0xB0,0xD7,0x00,0xFF,0xC7,0xE2,0x35,0xFF,0x2E,0x5A,0x4B,0xFF,0x00,0x33,0x62,0xFF,
  0x01,0x53,0xA5,0xFF,0x02,0x51,0xAD,0xFF,0x09,0x50,0xA8,0xFF,0x04,0x50,0xB2,0xFF,0x01,0x4F,0xA1,0xFF,
  0x02,0x5A,0xB4,0xFF,0x00,0x63,0xCA,0xFF,0x00,0x67,0xBD,0xFF,0x03,0x5D,0xB5,0xFF,0x00,0x4D,0xA8,0xFF,
  0x05,0x54,0xB1,0xFF,0x00,0x54,0xAE,0xFF,0x02,0x4D,0xB2,0xFF,0x00,0x53,0xAA,0xFF,0x04,0x50,0xB0,0xFF,
  0x00,0x51,0xAF,0xFF,0x04,0x51,0xAB,0xFF,0x01,0x51,0xB0,0xFF,0x00,0x51,0xAB,0xFF,0x04,0x59,0xAC,0xFF,
  0x00,0x52,0xA8,0xFF,0x00,0x52,0xB7,0xFF,0x01,0x5E,0xAE,0xFF,0x0A,0x76,0xDB,0xFF,0x00,0x6E,0xE9,0xFF,
  0x04,0x7B,0xFD,0xFF,0x00,0x74,0xE7,0xFF,0x00,0x76,0xD7,0xFF,0x00,0x68,0xC8,0xFF,0x00,0x6C,0xDF,0xFF,
  0x00,0x76,0xEE,0xFF,0x00,0x6C,0xC9,0xFF,0x00,0x5A,0xBA,0xFF,0x01,0x69,0xE6,0xFF,0x25,0x73,0xB1,0xFF,
  0x41,0x5F,0x2B,0xFF,0x5B,0x74,0x18,0xFF,0x87,0xA6,0x02,0xFF,0x8F,0xA8,0x06,0xFF,0x87,0xAC,0x04,0xFF,
  0x8F,0xAC,0x04,0xFF,0x8C,0xAF,0x00,0xFF,0x89,0xAD,0x03,0xFF,0x90,0xBD,0x08,0xFF,0x7D,0xAA,0x0F,0xFF,
  0x19,0x48,0x72,0xFF,0x00,0x5D,0xAD,0xFF,0x02,0x7C,0xFF,0xFF,0x01,0x7E,0xFF,0xFF,0x02,0x82,0xFF,0xFF,
  0x00,0x7A,0xFF,0xFF,0x17,0x89,0xF9,0xFF,0x29,0x71,0xBB,0xFF,0x32,0x52,0x21,0xFF,0x43,0x4D,0x08,0xFF,
  0x7A,0x9D,0x00,0xFF,0x88,0xAD,0x08,0xFF,0x92,0xAA,0x00,0xFF,0x88,0xAB,0x06,0xFF,0x93,0xAF,0x00,0xFF,
  0x9D,0xCD,0x00,0xFF,0xCB,0xF8,0x2B,0xFF,0x29,0x55,0x58,0xFF,0x01,0x28,0x51,0xFF,0x08,0x4E,0xA6,0xFF,
  0x00,0x54,0xAD,0xFF,0x00,0x52,0xAC,0xFF,0x02,0x56,0xB0,0xFF,0x00,0x52,0xAC,0xFF,0x00,0x50,0xB4,0xFF,
  0x02,0x55,0xB5,0xFF,0x00,0x59,0xB1,0xFF,0x00,0x51,0xB4,0xFF,0x09,0x51,0xB7,0xFF,0x02,0x4E,0xA4,0xFF,
  0x00,0x55,0xB0,0xFF,0x00,0x53,0xA9,0xFF,0x00,0x54,0xA3,0xFF,0x07,0x4F,0xB1,0xFF,0x03,0x51,0xB5,0xFF,
  0x02,0x50,0xA3,0xFF,0x00,0x54,0xAD,0xFF,0x03,0x51,0xB3,0xFF,0x00,0x50,0xAD,0xFF,0x06,0x53,0xAD,0xFF,
  0x07,0x52,0xAF,0xFF,0x00,0x51,0xAE,0xFF,0x01,0x4D,0xAB,0xFF,0x00,0x54,0xAE,0xFF,0x00,0x5A,0xB4,0xFF,
  0x05,0x62,0xCA,0xFF,0x00,0x64,0xC8,0xFF,0x02,0x5E,0xB5,0xFF,0x00,0x4F,0xB2,0xFF,0x00,0x55,0xAF,0xFF,
  0x05,0x4E,0xB7,0xFF,0x00,0x4F,0xB4,0xFF,0x00,0x6F,0xD8,0xFF,0x37,0x62,0x8C,0xFF,0x27,0x32,0x10,0xFF,
  0x5C,0x7E,0x06,0xFF,0x8B,0xAD,0x00,0xFF,0x94,0xA7,0x00,0xFF,0x8F,0xB2,0x00,0xFF,0x8A,0xA7,0x00,0xFF,
  0x8A,0xAF,0x00,0xFF,0x99,0xB6,0x00,0xFF,0xA2,0xD5,0x14,0xFF,0x5A,0x76,0x4E,0xFF,0x06,0x24,0x56,0xFF,
  0x04,0x46,0x94,0xFF,0x00,0x53,0xB1,0xFF,0x00,0x59,0xB2,0xFF,0x00,0x73,0xD6,0xFF,0x00,0x59,0xB9,0xFF,
  0x01,0x72,0xDA,0xFF,0x0B,0x84,0xF9,0xFF,0x30,0x72,0x92,0xFF,0x25,0x32,0x00,0xFF,0x60,0x72,0x06,0xFF,
  0x82,0xA4,0x05,0xFF,0x8B,0xAD,0x00,0xFF,0x8C,0xAA,0x00,0xFF,0x8D,0xAE,0x05,0xFF,0xA6,0xCD,0x00,0xFF,
  0xAE,0xF9,0x2A,0xFF,0x43,0x72,0x62,0xFF,0x07,0x28,0x5D,0xFF,0x00,0x53,0x9E,0xFF,0x05,0x51,0xB1,0xFF,
  0x00,0x55,0xAB,0xFF,0x03,0x52,0xAE,0xFF,0x05,0x55,0xAE,0xFF,0x00,0x53,0xAB,0xFF,0x00,0x51,0xAC,0xFF,
  0x00,0x53,0xAF,0xFF,0x00,0x52,0xAE,0xFF,0x07,0x4F,0xB1,0xFF,0x00,0x53,0xB5,0xFF,0x01,0x53,0xA7,0xFF,
  0x00,0x54,0xA5,0xFF,0x00,0x51,0xAB,0xFF,0x00,0x54,0xB3,0xFF,0x02,0x52,0xB1,0xFF,0x03,0x54,0xB0,0xFF,
  0x00,0x53,0xA8,0xFF,0x00,0x54,0xAE,0xFF,0x02,0x51,0xAD,0xFF,0x01,0x51,0xAA,0xFF,0x00,0x54,0xAB,0xFF,
  0x00,0x52,0xAC,0xFF,0x07,0x54,0xB0,0xFF,0x00,0x4E,0xAB,0xFF,0x00,0x55,0xB1,0xFF,0x05,0x54,0xB1,0xFF,
  0x04,0x4D,0xA9,0xFF,0x00,0x54,0xB0,0xFF,0x04,0x51,0xAD,0xFF,0x07,0x4F,0xB3,0xFF,0x00,0x53,0xB1,0xFF,
  0x03,0x58,0xB5,0xFF,0x00,0x74,0xE8,0xFF,0x2D,0x63,0x87,0xFF,0x2C,0x32,0x18,0xFF,0x69,0x82,0x03,0xFF,
  0x8B,0xAC,0x00,0xFF,0x91,0xAE,0x00,0xFF,0x87,0xB0,0x00,0xFF,0x8D,0xAB,0x00,0xFF,0x8D,0xAC,0x08,0xFF,
  0xA9,0xCE,0x0E,0xFF,0xBF,0xE6,0x35,0xFF,0x59,0x69,0x4C,0xFF,0x0A,0x2B,0x56,0xFF,0x00,0x47,0x94,0xFF,
  0x00,0x53,0xAA,0xFF,0x03,0x54,0xB0,0xFF,0x02,0x4F,0xA7,0xFF,0x00,0x54,0xAB,0xFF,0x00,0x52,0xB1,0xFF,
  0x00,0x79,0xEC,0xFF,0x1D,0x7D,0xC7,0xFF,0x33,0x40,0x15,0xFF,0x40,0x52,0x00,0xFF,0x78,0x9D,0x00,0xFF,
  0x90,0xAB,0x00,0xFF,0x8B,0xB1,0x00,0xFF,0x8D,0xA2,0x05,0xFF,0x9E,0xC3,0x04,0xFF,0xC0,0xED,0x16,0xFF,
  0x60,0x8F,0x5B,0xFF,0x04,0x37,0x79,0xFF,0x00,0x58,0xA7,0xFF,0x05,0x51,0xB1,0xFF,0x02,0x51,0xAE,0xFF,
  0x00,0x50,0xA9,0xFF,0x00,0x50,0xAB,0xFF,0x00,0x51,0xAB,0xFF,0x01,0x53,0xA9,0xFF,0x02,0x53,0xAE,0xFF,
  0x00,0x52,0xAF,0xFF,0x00,0x52,0xAD,0xFF,0x02,0x51,0xAE,0xFF,0x02,0x51,0xB0,0xFF,0x04,0x51,0xA9,0xFF,
  0x08,0x54,0xAA,0xFF,0x06,0x4E,0xB0,0xFF,0x06,0x4C,0xAC,0xFF,0x02,0x4F,0xA7,0xFF,0x00,0x53,0xB1,0xFF,
  0x00,0x50,0xAC,0xFF,0x06,0x53,0xAF,0xFF,0x06,0x51,0xAE,0xFF,0x01,0x52,0xAD,0xFF,0x01,0x52,0xAE,0xFF,
  0x03,0x4D,0xAC,0xFF,0x04,0x4E,0xAD,0xFF,0x01,0x52,0xAD,0xFF,0x00,0x52,0xB3,0xFF,0x00,0x52,0xAC,0xFF,
  0x02,0x53,0xAF,0xFF,0x00,0x52,0xA8,0xFF,0x00,0x54,0xAE,0xFF,0x00,0x53,0xB0,0xFF,0x00,0x5D,0xBA,0xFF,
  0x0B,0x7D,0xD5,0xFF,0x3F,0x7C,0x5B,0xFF,0x38,0x47,0x06,0xFF,0x76,0x93,0x05,0xFF,0x8D,0xAD,0x00,0xFF,
  0x8D,0xA9,0x00,0xFF,0x88,0xAF,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8D,0xAD,0x00,0xFF,0xA3,0xC9,0x00,0xFF,
  0xC6,0xED,0x38,0xFF,0x4E,0x6D,0x4D,0xFF,0x00,0x1D,0x39,0xFF,0x04,0x4B,0x99,0xFF,0x00,0x51,0xAA,0xFF,
  0x02,0x56,0xAE,0xFF,0x00,0x50,0xAF,0xFF,0x08,0x55,0xAF,0xFF,0x03,0x50,0xAC,0xFF,0x00,0x73,0xDC,0xFF,
  0x13,0x7A,0xD6,0xFF,0x44,0x58,0x27,0xFF,0x4E,0x69,0x02,0xFF,0x80,0x9A,0x01,0xFF,0x8E,0xAF,0x00,0xFF,
  0x8B,0xB1,0x00,0xFF,0x90,0xAA,0x08,0xFF,0x92,0xB1,0x00,0xFF,0xB3,0xDD,0x0D,0xFF,0x60,0x8D,0x4A,0xFF,
  0x01,0x41,0x71,0xFF,0x00,0x49,0x9A,0xFF,0x01,0x51,0xAA,0xFF,0x07,0x4F,0xB3,0xFF,0x02,0x53,0xAF,0xFF,
  0x00,0x51,0xB4,0xFF,0x04,0x52,0xB7,0xFF,0x06,0x50,0xAF,0xFF,0x05,0x4E,0xAC,0xFF,0x05,0x50,0xAD,0xFF,
  0x00,0x54,0xA5,0xFF,0x04,0x4F,0xAA,0xFF,0x00,0x52,0xAE,0xFF,0x00,0x54,0xB0,0xFF,0x00,0x55,0xAD,0xFF,
  0x00,0x55,0xA5,0xFF,0x01,0x56,0xA9,0xFF,0x02,0x53,0xAE,0xFF,0x01,0x50,0xAF,0xFF,0x00,0x53,0xAD,0xFF,
  0x00,0x4E,0xAB,0xFF,0x06,0x52,0xB4,0xFF,0x00,0x4F,0xAD,0xFF,0x00,0x55,0xAC,0xFF,0x00,0x55,0xA8,0xFF,
  0x00,0x53,0xA9,0xFF,0x00,0x55,0xAC,0xFF,0x00,0x52,0xAE,0xFF,0x00,0x55,0xA9,0xFF,0x00,0x52,0xB0,0xFF,
  0x02,0x53,0xAC,0xFF,0x03,0x55,0xA9,0xFF,0x01,0x4F,0xB3,0xFF,0x02,0x5C,0xB4,0xFF,0x10,0x78,0xCD,0xFF,
  0x33,0x65,0x5A,0xFF,0x36,0x45,0x0A,0xFF,0x75,0x93,0x01,0xFF,0x8C,0xAE,0x00,0xFF,0x8E,0xA9,0x02,0xFF,
  0x8A,0xB0,0x05,0xFF,0x8E,0xAB,0x01,0xFF,0x8C,0xAD,0x00,0xFF,0xAA,0xCF,0x03,0xFF,0xCB,0xF4,0x2E,0xFF,
  0x54,0x83,0x59,0xFF,0x01,0x1B,0x3E,0xFF,0x01,0x42,0x82,0xFF,0x04,0x54,0xAD,0xFF,0x00,0x4D,0xAF,0xFF,
  0x01,0x55,0xAF,0xFF,0x03,0x4F,0xAF,0xFF,0x01,0x55,0xAD,0xFF,0x00,0x69,0xDC,0xFF,0x16,0x68,0xA0,0xFF,
  0x47,0x56,0x11,0xFF,0x75,0x97,0x02,0xFF,0x84,0xA1,0x09,0xFF,0x88,0xAE,0x01,0xFF,0x87,0xAC,0x00,0xFF,
  0x8F,0xAC,0x00,0xFF,0x8D,0xB9,0x02,0xFF,0x9C,0xC5,0x13,0xFF,0x68,0xB0,0x40,0xFF,0x0F,0x3C,0x65,0xFF,
  0x00,0x44,0x85,0xFF,0x04,0x52,0xA7,0xFF,0x01,0x51,0xA8,0xFF,0x00,0x55,0xA7,0xFF,0x00,0x53,0xA5,0xFF,
  0x00,0x53,0xA6,0xFF,0x00,0x53,0xA6,0xFF,0x00,0x51,0xA9,0xFF,0x00,0x53,0xAF,0xFF,0x00,0x55,0xB1,0xFF,
  0x00,0x53,0xAD,0xFF,0x00,0x52,0xA8,0xFF,0x02,0x53,0xAE,0xFF,0x00,0x4E,0xAB,0xFF,0x02,0x54,0xAA,0xFF,
  0x00,0x53,0xAD,0xFF,0x00,0x53,0xB3,0xFF,0x00,0x50,0xA7,0xFF,0x00,0x57,0xAA,0xFF,0x00,0x50,0xAC,0xFF,
  0x04,0x54,0xB5,0xFF,0x01,0x52,0xAE,0xFF,0x02,0x54,0xAA,0xFF,0x02,0x53,0xA5,0xFF,0x03,0x50,0xA8,0xFF,
  0x05,0x4F,0xAE,0xFF,0x0B,0x51,0xB0,0xFF,0x09,0x50,0xA8,0xFF,0x00,0x55,0xAA,0xFF,0x00,0x53,0xAD,0xFF,
  0x03,0x54,0xA6,0xFF,0x00,0x58,0xBD,0xFF,0x0D,0x62,0xA5,0xFF,0x1E,0x62,0xA3,0xFF,0x34,0x5A,0x5D,0xFF,
  0x50,0x63,0x14,0xFF,0x79,0x97,0x01,0xFF,0x8D,0xAF,0x01,0xFF,0x90,0xAE,0x01,0xFF,0x8A,0xAE,0x06,0xFF,
  0x8C,0xA9,0x00,0xFF,0x8B,0xAC,0x00,0xFF,0x9A,0xBF,0x00,0xFF,0xC9,0xF2,0x2C,0xFF,0x63,0x96,0x4B,0xFF,
  0x03,0x1A,0x3C,0xFF,0x04,0x37,0x76,0xFF,0x00,0x51,0xAF,0xFF,0x03,0x54,0xAD,0xFF,0x00,0x50,0xAB,0xFF,
  0x00,0x52,0xAE,0xFF,0x00,0x56,0xAB,0xFF,0x08,0x65,0xE6,0xFF,0x30,0x68,0x69,0xFF,0x53,0x5A,0x00,0xFF,
  0x82,0xA7,0x00,0xFF,0x82,0xAE,0x01,0xFF,0x90,0xAE,0x00,0xFF,0x8D,0xAE,0x09,0xFF,0x87,0xAA,0x00,0xFF,
  0x91,0xB8,0x03,0xFF,0x89,0xB5,0x08,0xFF,0x89,0xD7,0x12,0xFF,0x18,0x4C,0x71,0xFF,0x00,0x3C,0x73,0xFF,
  0x02,0x51,0xB6,0xFF,0x00,0x50,0xA9,0xFF,0x05,0x51,0xB1,0xFF,0x04,0x53,0xB0,0xFF,0x01,0x55,0xAF,0xFF,
  0x00,0x53,0xB0,0xFF,0x00,0x53,0xAD,0xFF,0x00,0x53,0xA9,0xFF,0x03,0x50,0xAC,0xFF,0x00,0x58,0xA3,0xFF,
  0x00,0x4F,0xA8,0xFF,0x03,0x4D,0xA6,0xFF,0x04,0x4F,0xAA,0xFF,0x01,0x54,0xB4,0xFF,0x00,0x53,0xAF,0xFF,
  0x00,0x54,0xAB,0xFF,0x05,0x51,0xAF,0xFF,0x02,0x53,0xAC,0xFF,0x01,0x52,0xAB,0xFF,0x00,0x4E,0xA3,0xFF,
  0x03,0x53,0xA8,0xFF,0x01,0x51,0xA6,0xFF,0x00,0x53,0xAB,0xFF,0x00,0x57,0xB1,0xFF,0x00,0x52,0xB1,0xFF,
  0x00,0x51,0xAF,0xFF,0x06,0x50,0xA9,0xFF,0x01,0x52,0xAD,0xFF,0x04,0x4D,0xB3,0xFF,0x06,0x50,0xBB,0xFF,
  0x00,0x70,0xD3,0xFF,0x16,0x81,0xB7,0xFF,0x34,0x68,0x5C,0xFF,0x46,0x69,0x27,0xFF,0x7C,0x9B,0x02,0xFF,
  0x88,0xA9,0x02,0xFF,0x8E,0xAF,0x08,0xFF,0x8C,0xAD,0x00,0xFF,0x89,0xAB,0x00,0xFF,0x8E,0xAC,0x00,0xFF,
  0x8E,0xAE,0x03,0xFF,0x90,0xB5,0x03,0xFF,0xC1,0xEA,0x14,0xFF,0x8D,0xB9,0x2E,0xFF,0x00,0x17,0x2F,0xFF,
  0x00,0x28,0x56,0xFF,0x00,0x51,0xAB,0xFF,0x04,0x51,0xA9,0xFF,0x07,0x54,0xAE,0xFF,0x00,0x50,0xAC,0xFF,
  0x00,0x55,0xAC,0xFF,0x1A,0x60,0xA8,0xFF,0x47,0x68,0x31,0xFF,0x66,0x7B,0x00,0xFF,0x7E,0xA7,0x00,0xFF,
  0x87,0xAD,0x00,0xFF,0x92,0xAA,0x00,0xFF,0x8E,0xB0,0x00,0xFF,0x8A,0xB4,0x00,0xFF,0x97,0xAF,0x01,0xFF,
  0x89,0xB1,0x12,0xFF,0xA3,0xD0,0x04,0xFF,0x30,0x84,0x6C,0xFF,0x00,0x32,0x69,0xFF,0x00,0x51,0xA5,0xFF,
  0x00,0x57,0xB0,0xFF,0x02,0x4E,0xAE,0xFF,0x00,0x4F,0xB0,0xFF,0x00,0x51,0xB1,0xFF,0x06,0x53,0xAF,0xFF,
  0x05,0x50,0xA3,0xFF,0x00,0x43,0x8A,0xFF,0x04,0x34,0x7C,0xFF,0x04,0x26,0x4B,0xFF,0x05,0x22,0x40,0xFF,
  0x01,0x22,0x41,0xFF,0x00,0x34,0x66,0xFF,0x00,0x43,0x8E,0xFF,0x04,0x50,0xA4,0xFF,0x01,0x51,0xA8,0xFF,
  0x00,0x52,0xAC,0xFF,0x07,0x4F,0xB3,0xFF,0x00,0x50,0xA9,0xFF,0x02,0x51,0x9C,0xFF,0x00,0x43,0x88,0xFF,
  0x00,0x34,0x7A,0xFF,0x01,0x41,0x88,0xFF,0x00,0x42,0x8D,0xFF,0x00,0x4A,0x98,0xFF,0x01,0x54,0xB4,0xFF,
  0x03,0x54,0xAF,0xFF,0x05,0x50,0xAD,0xFF,0x02,0x52,0xA9,0xFF,0x00,0x50,0xB3,0xFF,0x00,0x6A,0xC3,0xFF,
  0x0C,0x7A,0xD1,0xFF,0x35,0x5F,0x49,0xFF,0x3A,0x52,0x04,0xFF,0x7F,0x9F,0x00,0xFF,0x8A,0xAC,0x00,0xFF,
  0x8D,0xAD,0x04,0xFF,0x8B,0xAD,0x00,0xFF,0x8C,0xAC,0x01,0xFF,0x8D,0xAD,0x04,0xFF,0x8B,0xAA,0x04,0xFF,
  0x8F,0xB4,0x04,0xFF,0xB3,0xDC,0x00,0xFF,0xD7,0xFF,0x31,0xFF,0x12,0x30,0x2E,0xFF,0x00,0x24,0x3E,0xFF,
  0x00,0x52,0xA3,0xFF,0x09,0x55,0xB3,0xFF,0x00,0x4E,0xA7,0xFF,0x00,0x55,0xB2,0xFF,0x0E,0x61,0xA7,0xFF,
  0x3E,0x66,0x44,0xFF,0x64,0x7D,0x08,0xFF,0x80,0xA2,0x03,0xFF,0x89,0xB1,0x00,0xFF,0x92,0xAA,0x0A,0xFF,
  0x90,0xAE,0x01,0xFF,0x8A,0xA9,0x05,0xFF,0x8C,0xAD,0x04,0xFF,0x93,0xB5,0x00,0xFF,0x90,0xAC,0x00,0xFF,
  0xA4,0xCD,0x05,0xFF,0x6A,0xAA,0x38,0xFF,0x05,0x31,0x60,0xFF,0x01,0x40,0x8D,0xFF,0x00,0x4A,0x97,0xFF,
  0x04,0x48,0x91,0xFF,0x05,0x55,0xAA,0xFF,0x00,0x51,0xA6,0xFF,0x01,0x42,0x82,0xFF,0x03,0x40,0x6D,0xFF,
  0x0A,0x45,0x65,0xFF,0x2D,0x52,0x5B,0xFF,0x27,0x4F,0x51,0xFF,0x22,0x3D,0x38,0xFF,0x19,0x2F,0x3C,0xFF,
  0x08,0x2B,0x4B,0xFF,0x00,0x2C,0x4B,0xFF,0x00,0x2A,0x4F,0xFF,0x01,0x3B,0x7B,0xFF,0x04,0x52,0xA5,0xFF,
  0x00,0x4F,0xB2,0xFF,0x00,0x53,0xB4,0xFF,0x03,0x51,0xA4,0xFF,0x00,0x32,0x69,0xFF,0x00,0x1B,0x30,0xFF,
  0x0C,0x24,0x28,0xFF,0x07,0x22,0x35,0xFF,0x01,0x1D,0x45,0xFF,0x00,0x28,0x5A,0xFF,0x00,0x3B,0x75,0xFF,
  0x00,0x43,0x8C,0xFF,0x01,0x52,0xA4,0xFF,0x03,0x53,0xB2,0xFF,0x03,0x57,0xAD,0xFF,0x0D,0x56,0x9A,0xFF,
  0x27,0x36,0x23,0xFF,0x3F,0x4A,0x06,0xFF,0x67,0x7F,0x03,0xFF,0x7F,0x9E,0x00,0xFF,0x84,0xA3,0x00,0xFF,
  0x85,0xA6,0x00,0xFF,0x8D,0xAE,0x00,0xFF,0x8E,0xAE,0x01,0xFF,0x8C,0xAE,0x00,0xFF,0x86,0xAB,0x00,0xFF,
  0x9C,0xBF,0x03,0xFF,0xCE,0xFE,0x08,0xFF,0x5E,0x84,0x5B,0xFF,0x05,0x28,0x48,0xFF,0x01,0x49,0x93,0xFF,
  0x00,0x50,0xA2,0xFF,0x00,0x54,0xB4,0xFF,0x04,0x6C,0xC1,0xFF,0x2F,0x71,0x93,0xFF,0x6A,0x79,0x10,0xFF,
  0x7F,0xA4,0x00,0xFF,0x8F,0xAB,0x01,0xFF,0x91,0xAF,0x01,0xFF,0x8B,0xA8,0x00,0xFF,0x87,0xAE,0x00,0xFF,
  0x92,0xAE,0x01,0xFF,0x80,0xA3,0x00,0xFF,0x80,0xA5,0x00,0xFF,0x77,0x9D,0x00,0xFF,0x8F,0xB4,0x02,0xFF,
  0x87,0xA4,0x16,0xFF,0x01,0x21,0x30,0xFF,0x00,0x2D,0x52,0xFF,0x00,0x21,0x3D,0xFF,0x02,0x1C,0x35,0xFF,
  0x00,0x30,0x6E,0xFF,0x02,0x3D,0x83,0xFF,0x26,0x4E,0x5A,0xFF,0x47,0x75,0x47,0xFF,0x65,0x9F,0x39,0xFF,
  0x9E,0xCC,0x23,0xFF,0xC0,0xED,0x20,0xFF,0xB7,0xE0,0x00,0xFF,0xB4,0xE5,0x0D,0xFF,0x85,0xC1,0x2E,0xFF,
  0x50,0x7E,0x4F,0xFF,0x08,0x23,0x34,0xFF,0x00,0x27,0x5C,0xFF,0x00,0x4C,0x9A,0xFF,0x00,0x56,0xB2,0xFF,
  0x02,0x61,0xCB,0xFF,0x00,0x52,0xB3,0xFF,0x1D,0x59,0x7E,0xFF,0x4A,0x71,0x45,0xFF,0x5D,0x83,0x2E,0xFF,
  0x62,0x8B,0x4F,0xFF,0x30,0x5C,0x4B,0xFF,0x0A,0x29,0x48,0xFF,0x14,0x3B,0x5C,0xFF,0x0A,0x32,0x4B,0xFF,
  0x00,0x29,0x5B,0xFF,0x00,0x31,0x6C,0xFF,0x03,0x3E,0x84,0xFF,0x1B,0x46,0x4D,0xFF,0x1F,0x26,0x16,0xFF,
  0x2C,0x2E,0x00,0xFF,0x27,0x33,0x05,0xFF,0x43,0x51,0x04,0xFF,0x43,0x54,0x06,0xFF,0x47,0x59,0x03,0xFF,
  0x59,0x6F,0x02,0xFF,0x63,0x79,0x0A,0xFF,0x6A,0x85,0x00,0xFF,0x7D,0x9C,0x00,0xFF,0x89,0xA4,0x07,0xFF,
  0xBD,0xF1,0x00,0xFF,0x60,0x88,0x54,0xFF,0x00,0x25,0x46,0xFF,0x05,0x40,0x84,0xFF,0x00,0x53,0xA8,0xFF,
  0x00,0x5A,0xB4,0xFF,0x00,0x77,0xD5,0xFF,0x38,0x74,0x36,0xFF,0x45,0x4C,0x00,0xFF,0x6F,0x85,0x07,0xFF,
  0x77,0xA1,0x00,0xFF,0x80,0xA5,0x00,0xFF,0x86,0xA7,0x00,0xFF,0x85,0xA4,0x00,0xFF,0x79,0x9A,0x01,0xFF,
  0x6A,0x7D,0x07,0xFF,0x4F,0x69,0x08,0xFF,0x53,0x69,0x00,0xFF,0x5B,0x74,0x0C,0xFF,0x60,0x88,0x28,0xFF,
  0x0D,0x25,0x27,0xFF,0x0B,0x34,0x50,0xFF,0x22,0x4B,0x47,0xFF,0x0F,0x2B,0x2F,0xFF,0x3B,0x5A,0x21,0xFF,
  0x6D,0x8F,0x42,0xFF,0x97,0xBC,0x1E,0xFF,0xB2,0xD7,0x0D,0xFF,0xA9,0xD9,0x00,0xFF,0xA1,0xCD,0x00,0xFF,
  0xA7,0xCC,0x00,0xFF,0x92,0xC1,0x00,0xFF,0xA4,0xCE,0x00,0xFF,0xC9,0xF1,0x05,0xFF,0xC1,0xF0,0x02,0xFF,
  0x4A,0x75,0x25,0xFF,0x03,0x24,0x4D,0xFF,0x00,0x4B,0x9D,0xFF,0x03,0x53,0xAA,0xFF,0x00,0x69,0xE0,0xFF,
  0x17,0x65,0xAD,0xFF,0x73,0x9B,0x16,0xFF,0xA6,0xCC,0x05,0xFF,0xC3,0xEE,0x0B,0xFF,0xCB,0xFF,0x00,0xFF,
  0xCD,0xFE,0x02,0xFF,0xA9,0xD6,0x17,0xFF,0x87,0xBF,0x20,0xFF,0x66,0x97,0x56,0xFF,0x31,0x83,0x6D,0xFF,
  0x31,0x6D,0x6E,0xFF,0x4B,0x8D,0x7F,0xFF,0x74,0x99,0x25,0xFF,0x53,0x89,0x2B,0xFF,0x1C,0x3D,0x20,0xFF,
  0x16,0x31,0x3A,0xFF,0x16,0x5E,0x77,0xFF,0x36,0x71,0x6F,0xFF,0x42,0x62,0x31,0xFF,0x3E,0x47,0x1C,0xFF,
  0x1E,0x29,0x09,0xFF,0x29,0x30,0x00,0xFF,0x4B,0x58,0x09,0xFF,0x60,0x76,0x00,0xFF,0x75,0x8E,0x0E,0xFF,
  0x35,0x4D,0x19,0xFF,0x00,0x22,0x48,0xFF,0x04,0x43,0x93,0xFF,0x00,0x4F,0xAA,0xFF,0x01,0x58,0xB7,0xFF,
  0x10,0x8D,0xF9,0xFF,0x34,0x77,0xA4,0xFF,0x33,0x46,0x19,0xFF,0x3B,0x48,0x03,0xFF,0x4C,0x69,0x00,0xFF,
  0x59,0x74,0x00,0xFF,0x5B,0x6E,0x00,0xFF,0x64,0x7E,0x00,0xFF,0x54,0x6B,0x10,0xFF,0x41,0x5A,0x18,0xFF,
  0x49,0x69,0x3A,0xFF,0x54,0x8F,0x63,0xFF,0x48,0x96,0x86,0xFF,0x2E,0x92,0xCD,0xFF,0x33,0x76,0x90,0xFF,
  0x53,0x83,0x37,0xFF,0x99,0xCA,0x30,0xFF,0xA0,0xD0,0x24,0xFF,0x9F,0xCF,0x0B,0xFF,0xAA,0xDC,0x00,0xFF,
  0x99,0xC1,0x00,0xFF,0x8F,0xB0,0x01,0xFF,0x8B,0xB1,0x00,0xFF,0x8B,0xAE,0x0C,0xFF,0x92,0xA8,0x00,0xFF,
  0x8E,0xAA,0x00,0xFF,0x8D,0xB2,0x00,0xFF,0x91,0xB4,0x02,0xFF,0x7D,0xAB,0x0A,0xFF,0x1E,0x3C,0x1A,0xFF,
  0x03,0x28,0x5C,0xFF,0x00,0x53,0xA5,0xFF,0x01,0x50,0xAD,0xFF,0x00,0x74,0xEF,0xFF,0x10,0x64,0xAA,0xFF,
  0x3D,0x4A,0x2E,0xFF,0x5C,0x6D,0x03,0xFF,0x81,0xA9,0x00,0xFF,0x94,0xB7,0x01,0xFF,0x99,0xBF,0x00,0xFF,
  0xA4,0xC9,0x00,0xFF,0xB7,0xE0,0x00,0xFF,0xC8,0xEF,0x02,0xFF,0xBB,0xF2,0x01,0xFF,0xBF,0xF0,0x00,0xFF,
  0xC2,0xFB,0x00,0xFF,0xCF,0xF6,0x00,0xFF,0xA4,0xDB,0x35,0xFF,0x32,0x7A,0x86,0xFF,0x02,0x46,0x99,0xFF,
  0x00,0x70,0xEB,0xFF,0x00,0x7F,0xFF,0xFF,0x15,0x88,0xFB,0xFF,0x42,0x97,0xCE,0xFF,0x4C,0x8B,0x78,0xFF,
  0x35,0x60,0x2B,0xFF,0x43,0x59,0x10,0xFF,0x4C,0x61,0x26,0xFF,0x27,0x47,0x30,0xFF,0x03,0x37,0x4F,0xFF,
  0x00,0x45,0x81,0xFF,0x00,0x4F,0x9E,0xFF,0x00,0x58,0xB1,0xFF,0x01,0x64,0xC2,0xFF,0x00,0x72,0xE5,0xFF,
  0x09,0x85,0xFF,0xFF,0x1E,0x77,0xD1,0xFF,0x48,0x73,0x48,0xFF,0x4F,0x5C,0x16,0xFF,0x4C,0x59,0x24,0xFF,
  0x42,0x59,0x25,0xFF,0x54,0x73,0x3A,0xFF,0x4C,0x85,0x5A,0xFF,0x1F,0x73,0x98,0xFF,0x0E,0x6D,0xCB,0xFF,
  0x00,0x75,0xEF,0xFF,0x03,0x77,0xFE,0xFF,0x00,0x7C,0xF8,0xFF,0x4D,0xAD,0xA2,0xFF,0x70,0x8D,0x17,0xFF,
  0x85,0xA4,0x09,0xFF,0x8C,0xB4,0x00,0xFF,0x8B,0xB2,0x01,0xFF,0x86,0xA5,0x01,0xFF,0x90,0xAF,0x00,0xFF,
  0x8F,0xAF,0x06,0xFF,0x8F,0xAC,0x06,0xFF,0x8A,0xAE,0x00,0xFF,0x8C,0xB0,0x06,0xFF,0x93,0xAB,0x00,0xFF,
  0x92,0xB0,0x02,0xFF,0x8B,0xAB,0x0C,0xFF,0x4A,0x85,0x55,0xFF,0x12,0x4A,0x6F,0xFF,0x0A,0x47,0x98,0xFF,
  0x00,0x54,0xA9,0xFF,0x00,0x53,0xA9,0xFF,0x00,0x71,0xE3,0xFF,0x00,0x6B,0xBE,0xFF,0x25,0x25,0x0D,0xFF,
  0x27,0x35,0x00,0xFF,0x71,0x92,0x00,0xFF,0x8B,0xB0,0x0A,0xFF,0x90,0xAE,0x00,0xFF,0x8B,0xAB,0x08,0xFF,
  0x8F,0xAB,0x00,0xFF,0x96,0xB4,0x00,0xFF,0x98,0xB6,0x08,0xFF,0x9B,0xBE,0x02,0xFF,0x98,0xBB,0x00,0xFF,
  0xAF,0xD1,0x08,0xFF,0xAA,0xEA,0x1A,0xFF,0x3A,0x5F,0x4D,0xFF,0x00,0x30,0x6C,0xFF,0x00,0x58,0xAA,0xFF,
  0x00,0x5C,0xB0,0xFF,0x00,0x69,0xDF,0xFF,0x06,0x83,0xFF,0xFF,0x12,0x90,0xF7,0xFF,0x0A,0x76,0xD8,0xFF,
  0x37,0x98,0xCF,0xFF,0x3D,0x99,0xD8,0xFF,0x1F,0x72,0xB4,0xFF,0x02,0x4D,0x90,0xFF,0x00,0x44,0x8F,0xFF,
  0x00,0x48,0x9F,0xFF,0x00,0x53,0xAB,0xFF,0x00,0x5B,0xAE,0xFF,0x02,0x56,0xB6,0xFF,0x07,0x6B,0xCB,0xFF,
  0x03,0x76,0xF3,0xFF,0x09,0x86,0xFC,0xFF,0x1C,0x98,0xBE,0xFF,0x20,0x7A,0xB9,0xFF,0x20,0x79,0xB1,0xFF,
  0x0E,0x78,0xCE,0xFF,0x0A,0x6B,0xC9,0xFF,0x01,0x65,0xBD,0xFF,0x0F,0x55,0xB4,0xFF,0x03,0x52,0xB1,0xFF,
  0x00,0x54,0xB3,0xFF,0x04,0x76,0xF0,0xFF,0x25,0x89,0xE1,0xFF,0x4D,0x71,0x33,0xFF,0x41,0x4F,0x00,0xFF,
  0x6A,0x86,0x00,0xFF,0x82,0xA5,0x03,0xFF,0x91,0xAC,0x00,0xFF,0x8D,0xA9,0x00,0xFF,0x89,0xAB,0x00,0xFF,
  0x91,0xAE,0x00,0xFF,0x8A,0xAC,0x00,0xFF,0x84,0xB1,0x00,0xFF,0x93,0xB3,0x00,0xFF,0x9E,0xBE,0x00,0xFF,
  0x70,0x8D,0x16,0xFF,0x13,0x55,0x6D,0xFF,0x00,0x56,0xA2,0xFF,0x00,0x4F,0xB0,0xFF,0x03,0x50,0xAA,0xFF,
  0x00,0x52,0xB6,0xFF,0x02,0x69,0xE0,0xFF,0x00,0x7A,0xF8,0xFF,0x32,0x48,0x31,0xFF,0x07,0x15,0x00,0xFF,
  0x61,0x79,0x00,0xFF,0x87,0xAB,0x03,0xFF,0x8D,0xA8,0x03,0xFF,0x8E,0xAF,0x00,0xFF,0x8C,0xA6,0x04,0xFF,
  0x8A,0xAD,0x00,0xFF,0x87,0xAE,0x05,0xFF,0x87,0xAD,0x08,0xFF,0x8C,0xB3,0x00,0xFF,0x9F,0xC3,0x00,0xFF,
  0x84,0xBE,0x2C,0xFF,0x1C,0x4B,0x69,0xFF,0x00,0x3C,0x79,0xFF,0x00,0x52,0x9E,0xFF,0x04,0x4E,0xA3,0xFF,
  0x00,0x52,0xB2,0xFF,0x00,0x5C,0xB7,0xFF,0x00,0x67,0xC2,0xFF,0x01,0x73,0xE5,0xFF,0x00,0x73,0xDD,0xFF,
  0x00,0x73,0xEB,0xFF,0x07,0x76,0xEC,0xFF,0x04,0x62,0xC2,0xFF,0x00,0x51,0xA0,0xFF,0x05,0x53,0xA5,0xFF,
  0x03,0x52,0xAF,0xFF,0x05,0x50,0xB5,0xFF,0x00,0x55,0xAC,0xFF,0x00,0x51,0xAE,0xFF,0x00,0x59,0xAE,0xFF,
  0x06,0x71,0xD3,0xFF,0x00,0x67,0xDE,0xFF,0x00,0x68,0xE1,0xFF,0x08,0x78,0xDA,0xFF,0x00,0x6C,0xCB,0xFF,
  0x03,0x55,0xB9,0xFF,0x00,0x55,0xB3,0xFF,0x00,0x51,0xA7,0xFF,0x02,0x4C,0xAF,0xFF,0x04,0x52,0xB7,0xFF,
  0x00,0x58,0xB0,0xFF,0x00,0x78,0xEC,0xFF,0x20,0x76,0xB3,0xFF,0x26,0x38,0x28,0xFF,0x40,0x55,0x00,0xFF,
  0x73,0x94,0x00,0xFF,0x91,0xAA,0x08,0xFF,0x90,0xAA,0x08,0xFF,0x8B,0xAF,0x07,0xFF,0x8F,0xAD,0x00,0xFF,
  0x89,0xAB,0x00,0xFF,0x8B,0xA8,0x02,0xFF,0x9D,0xC3,0x00,0xFF,0xA0,0xCA,0x10,0xFF,0x33,0x52,0x42,0xFF,
  0x00,0x34,0x63,0xFF,0x00,0x50,0xA0,0xFF,0x02,0x4F,0xA7,0xFF,0x06,0x55,0xB4,0xFF,0x01,0x50,0xAD,0xFF,
  0x00,0x63,0xBE,0xFF,0x00,0x81,0xF9,0xFF,0x38,0x78,0x9C,0xFF,0x10,0x15,0x00,0xFF,0x51,0x6D,0x00,0xFF,
  0x8C,0xA8,0x00,0xFF,0x8D,0xAF,0x00,0xFF,0x8A,0xAD,0x00,0xFF,0x8D,0xAC,0x06,0xFF,0x91,0xAE,0x00,0xFF,
  0x89,0xAD,0x03,0xFF,0x8E,0xA9,0x04,0xFF,0x91,0xB9,0x00,0xFF,0xA2,0xCB,0x05,0xFF,0x4E,0x82,0x5C,0xFF,
  0x13,0x37,0x5B,0xFF,0x00,0x46,0x8A,0xFF,0x00,0x51,0xAF,0xFF,0x04,0x53,0xAF,0xFF,0x03,0x51,0xA6,0xFF,
  0x03,0x4F,0xA7,0xFF,0x04,0x4D,0xA9,0xFF,0x03,0x53,0xAA,0xFF,0x01,0x54,0xB2,0xFF,0x00,0x53,0xB3,0xFF,
  0x00,0x56,0xB4,0xFF,0x00,0x5B,0xB7,0xFF,0x00,0x56,0xB8,0xFF,0x02,0x4F,0xB7,0xFF,0x02,0x51,0xB0,0xFF,
  0x00,0x53,0xA3,0xFF,0x00,0x57,0xA5,0xFF,0x02,0x51,0xAD,0xFF,0x00,0x54,0xB6,0xFF,0x00,0x4F,0xA9,0xFF,
  0x08,0x55,0xA7,0xFF,0x00,0x4C,0xAE,0xFF,0x00,0x55,0xB3,0xFF,0x00,0x55,0xA4,0xFF,0x04,0x56,0xAA,0xFF,
  0x00,0x50,0xAA,0xFF,0x00,0x56,0xAC,0xFF,0x05,0x58,0xB6,0xFF,0x00,0x53,0xAC,0xFF,0x07,0x52,0xAD,0xFF,
  0x00,0x6F,0xD5,0xFF,0x21,0x89,0xE2,0xFF,0x36,0x68,0x5F,0xFF,0x2F,0x40,0x14,0xFF,0x67,0x85,0x00,0xFF,
  0x84,0x9F,0x02,0xFF,0x90,0xAD,0x00,0xFF,0x88,0xB1,0x00,0xFF,0x8C,0xAA,0x00,0xFF,0x8E,0xAE,0x0F,0xFF,
  0x8D,0xAD,0x00,0xFF,0xA1,0xCB,0x07,0xFF,0x8D,0xB9,0x30,0xFF,0x10,0x3A,0x60,0xFF,0x05,0x38,0x79,0xFF,
  0x05,0x52,0xAA,0xFF,0x08,0x52,0xAB,0xFF,0x00,0x53,0xB1,0xFF,0x05,0x53,0xB5,0xFF,0x00,0x58,0xAC,0xFF,
  0x00,0x7B,0xFA,0xFF,0x28,0x7C,0xD2,0xFF,0x28,0x32,0x1A,0xFF,0x44,0x5B,0x03,0xFF,0x88,0xA2,0x01,0xFF,
  0x87,0xA9,0x00,0xFF,0x8A,0xAF,0x00,0xFF,0x8B,0xAD,0x01,0xFF,0x8D,0xA9,0x00,0xFF,0x87,0xAD,0x00,0xFF,
  0x8F,0xAC,0x00,0xFF,0x8E,0xB7,0x03,0xFF,0xA2,0xCC,0x20,0xFF,0x32,0x6C,0x60,0xFF,0x00,0x2A,0x5A,0xFF,
  0x00,0x4B,0x92,0xFF,0x02,0x53,0xAF,0xFF,0x02,0x50,0xB2,0xFF,0x03,0x52,0xB9,0xFF,0x00,0x52,0xB3,0xFF,
  0x00,0x56,0xAD,0xFF,0x00,0x50,0xA9,0xFF,0x00,0x4E,0xAA,0xFF,0x05,0x52,0xAC,0xFF,0x05,0x52,0xAC,0xFF,
  0x05,0x52,0xAC,0xFF,0x05,0x52,0xAC,0xFF,0x04,0x51,0xA9,0xFF,0x01,0x55,0xAB,0xFF,0x00,0x56,0xA7,0xFF,
  0x04,0x50,0xB0,0xFF,0x04,0x4C,0xB0,0xFF,0x08,0x53,0xAE,0xFF,0x02,0x51,0xAE,0xFF,0x00,0x4F,0xAB,0xFF,
  0x03,0x55,0xAB,0xFF,0x00,0x4E,0xAF,0xFF,0x03,0x4F,0xAF,0xFF,0x01,0x50,0xAC,0xFF,0x00,0x52,0xAA,0xFF,
  0x00,0x53,0xAD,0xFF,0x00,0x4F,0xA9,0xFF,0x00,0x59,0xA4,0xFF,0x01,0x4C,0xA9,0xFF,0x03,0x6F,0xCA,0xFF,
  0x13,0x7B,0xDE,0xFF,0x31,0x7E,0x9C,0xFF,0x3C,0x55,0x11,0xFF,0x51,0x6A,0x05,0xFF,0x88,0xA8,0x05,0xFF,
  0x8F,0xAD,0x00,0xFF,0x88,0xAF,0x00,0xFF,0x8B,0xAB,0x02,0xFF,0x8F,0xAD,0x00,0xFF,0x8B,0xB5,0x00,0xFF,
  0xAB,0xD8,0x03,0xFF,0x79,0xA6,0x41,0xFF,0x09,0x34,0x47,0xFF,0x00,0x37,0x7D,0xFF,0x00,0x4F,0xB2,0xFF,
  0x01,0x54,0xB0,0xFF,0x00,0x51,0xAC,0xFF,0x01,0x51,0xB4,0xFF,0x02,0x52,0xA9,0xFF,0x00,0x75,0xFB,0xFF,
  0x2A,0x7B,0xCC,0xFF,0x35,0x4C,0x2F,0xFF,0x38,0x49,0x03,0xFF,0x7E,0x9E,0x00,0xFF,0x8C,0xAC,0x00,0xFF,
  0x8E,0xAB,0x01,0xFF,0x91,0xA9,0x09,0xFF,0x8F,0xAC,0x02,0xFF,0x8A,0xAC,0x00,0xFF,0x8D,0xAF,0x03,0xFF,
  0xA2,0xCA,0x07,0xFF,0x96,0xBB,0x37,0xFF,0x23,0x54,0x74,0xFF,0x00,0x39,0x82,0xFF,0x01,0x4C,0x9F,0xFF,
  0x0A,0x4C,0xA3,0xFF,0x00,0x53,0xA4,0xFF,0x00,0x54,0xAE,0xFF,0x00,0x51,0xAA,0xFF,0x02,0x53,0xA5,0xFF,
  0x00,0x53,0xA9,0xFF,0x06,0x55,0xB1,0xFF,0x00,0x4E,0xAE,0xFF,0x00,0x52,0xB2,0xFF,0x03,0x52,0xAE,0xFF,
  0x00,0x51,0xA5,0xFF,0x00,0x55,0xA6,0xFF,0x00,0x53,0xAC,0xFF,0x04,0x53,0xB2,0xFF,0x02,0x53,0xAE,0xFF,
  0x00,0x52,0xA7,0xFF,0x03,0x52,0xB7,0xFF,0x00,0x52,0xA8,0xFF,0x00,0x55,0xAE,0xFF,0x01,0x54,0xA4,0xFF,
  0x03,0x50,0xAC,0xFF,0x00,0x55,0xB0,0xFF,0x01,0x50,0xB5,0xFF,0x00,0x54,0xA5,0xFF,0x07,0x52,0xAF,0xFF,
  0x02,0x4D,0xB3,0xFF,0x04,0x53,0xB0,0xFF,0x04,0x4E,0xAD,0xFF,0x00,0x5A,0xB8,0xFF,0x06,0x73,0xD2,0xFF,
  0x1E,0x7C,0xBA,0xFF,0x35,0x4F,0x22,0xFF,0x47,0x5F,0x00,0xFF,0x78,0x9A,0x00,0xFF,0x8F,0xA8,0x04,0xFF,
  0x8A,0xB1,0x00,0xFF,0x8B,0xAC,0x00,0xFF,0x90,0xAB,0x04,0xFF,0x95,0xB2,0x02,0xFF,0xC1,0xEE,0x00,0xFF,
  0x77,0xA7,0x44,0xFF,0x05,0x21,0x2D,0xFF,0x00,0x35,0x62,0xFF,0x02,0x4F,0xA9,0xFF,0x00,0x57,0xA7,0xFF,
  0x05,0x4E,0xAA,0xFF,0x00,0x4F,0xAB,0xFF,0x00,0x57,0xA6,0xFF,0x00,0x68,0xE3,0xFF,0x34,0x81,0xD3,0xFF,
  0x3A,0x4C,0x34,0xFF,0x32,0x4C,0x0B,0xFF,0x75,0x91,0x00,0xFF,0x8F,0xAD,0x00,0xFF,0x89,0xB0,0x00,0xFF,
  0x8C,0xA9,0x00,0xFF,0x87,0xAF,0x00,0xFF,0x8F,0xAB,0x00,0xFF,0x87,0xB2,0x00,0xFF,0xAF,0xD9,0x09,0xFF,
  0x6B,0x95,0x43,0xFF,0x06,0x40,0x65,0xFF,0x01,0x49,0x9B,0xFF,0x00,0x53,0xA7,0xFF,0x05,0x4F,0xAE,0xFF,
  0x00,0x59,0xAF,0xFF,0x03,0x4F,0xA7,0xFF,0x05,0x51,0xAF,0xFF,0x01,0x51,0xB2,0xFF,0x02,0x51,0xB0,0xFF,
  0x00,0x53,0xA8,0xFF,0x05,0x51,0xB1,0xFF,0x04,0x4E,0xAF,0xFF,0x01,0x52,0xAB,0xFF,0x00,0x56,0xAA,0xFF,
  0x00,0x52,0xB0,0xFF,0x04,0x50,0xB2,0xFF,0x05,0x50,0xAD,0xFF,0x00,0x51,0xA9,0xFF,0x07,0x53,0xAB,0xFF,
  0x00,0x52,0xA6,0xFF,0x00,0x53,0xB4,0xFF,0x01,0x4F,0xB1,0xFF,0x05,0x53,0xA8,0xFF,0x00,0x54,0xAF,0xFF,
  0x00,0x52,0xB2,0xFF,0x00,0x50,0xB0,0xFF,0x03,0x52,0xAE,0xFF,0x04,0x4F,0xAC,0xFF,0x00,0x54,0xAE,0xFF,
  0x00,0x53,0xAD,0xFF,0x02,0x51,0xB6,0xFF,0x00,0x5A,0xB7,0xFF,0x0C,0x67,0xAD,0xFF,0x13,0x75,0xD6,0xFF,
  0x32,0x58,0x4B,0xFF,0x3C,0x47,0x03,0xFF,0x78,0x9D,0x00,0xFF,0x8B,0xAC,0x05,0xFF,0x8E,0xAA,0x00,0xFF,
  0x8A,0xAC,0x00,0xFF,0x8E,0xAF,0x06,0xFF,0x93,0xB4,0x01,0xFF,0xB7,0xDC,0x00,0xFF,0xAC,0xDB,0x3D,0xFF,
  0x20,0x46,0x39,0xFF,0x00,0x25,0x45,0xFF,0x04,0x3E,0x90,0xFF,0x01,0x53,0xA5,0xFF,0x03,0x51,0xB6,0xFF,
  0x03,0x50,0xAC,0xFF,0x02,0x57,0xA8,0xFF,0x01,0x68,0xDF,0xFF,0x29,0x87,0xCF,0xFF,0x44,0x63,0x44,0xFF,
  0x4F,0x59,0x12,0xFF,0x79,0x9B,0x00,0xFF,0x91,0xAE,0x04,0xFF,0x8B,0xAC,0x00,0xFF,0x8B,0xAD,0x00,0xFF,
  0x88,0xAF,0x00,0xFF,0x90,0xAD,0x05,0xFF,0x99,0xAF,0x02,0xFF,0xAF,0xE2,0x0B,0xFF,0x2E,0x5E,0x46,0xFF,
  0x00,0x29,0x5A,0xFF,0x00,0x53,0xA6,0xFF,0x02,0x51,0xBA,0xFF,0x00,0x54,0xAF,0xFF,0x00,0x51,0xAC,0xFF,
  0x05,0x51,0xAF,0xFF,0x00,0x53,0xAB,0xFF,0x00,0x54,0xAC,0xFF,0x04,0x50,0xAE,0xFF,0x04,0x4E,0xAD,0xFF,
  0x04,0x51,0xAB,0xFF,0x00,0x4F,0xAC,0xFF,0x00,0x56,0xAF,0xFF,0x02,0x4F,0xBB,0xFF,0x00,0x59,0xA9,0xFF,
  0x06,0x52,0xAA,0xFF,0x04,0x50,0xB2,0xFF,0x00,0x51,0xAC,0xFF,0x00,0x52,0xAC,0xFF,0x03,0x55,0xA9,0xFF,
  0x00,0x53,0xA3,0xFF,0x01,0x51,0xAA,0xFF,0x04,0x50,0xB0,0xFF,0x00,0x53,0xAA,0xFF,0x00,0x51,0xA3,0xFF,
  0x00,0x53,0xAF,0xFF,0x00,0x52,0xA9,0xFF,0x04,0x4D,0xA9,0xFF,0x00,0x50,0xAC,0xFF,0x00,0x54,0xAE,0xFF,
  0x02,0x5C,0xBC,0xFF,0x0A,0x60,0xB7,0xFF,0x0E,0x59,0x92,0xFF,0x00,0x77,0xD3,0xFF,0x2B,0x64,0x5E,0xFF,
  0x29,0x37,0x02,0xFF,0x73,0x93,0x02,0xFF,0x8D,0xAD,0x02,0xFF,0x91,0xAD,0x01,0xFF,0x88,0xAD,0x00,0xFF,
  0x8B,0xAE,0x00,0xFF,0x91,0xAD,0x01,0xFF,0xA3,0xCC,0x02,0xFF,0xD1,0xFD,0x20,0xFF,0x70,0x99,0x57,0xFF,
  0x0B,0x21,0x1E,0xFF,0x00,0x27,0x48,0xFF,0x00,0x31,0x59,0xFF,0x04,0x3F,0x8F,0xFF,0x00,0x48,0x9F,0xFF,
  0x00,0x52,0xA1,0xFF,0x04,0x5B,0xBC,0xFF,0x08,0x6A,0xC5,0xFF,0x43,0x59,0x44,0xFF,0x52,0x76,0x0A,0xFF,
  0x86,0xA3,0x00,0xFF,0x85,0xAB,0x00,0xFF,0x8F,0xAC,0x02,0xFF,0x8F,0xAD,0x00,0xFF,0x8A,0xA9,0x03,0xFF,
  0x89,0xB1,0x00,0xFF,0x95,0xC0,0x11,0xFF,0x91,0xBA,0x36,0xFF,0x20,0x41,0x50,0xFF,0x08,0x33,0x81,0xFF,
  0x04,0x56,0xAC,0xFF,0x05,0x4D,0xA1,0xFF,0x04,0x4F,0xAA,0xFF,0x05,0x55,0xAA,0xFF,0x02,0x4A,0x9F,0xFF,
  0x00,0x46,0x8B,0xFF,0x03,0x4E,0x92,0xFF,0x00,0x42,0x89,0xFF,0x00,0x46,0x8B,0xFF,0x00,0x4E,0x9D,0xFF,
  0x01,0x52,0xAB,0xFF,0x0B,0x52,0xAE,0xFF,0x02,0x52,0xB1,0xFF,0x00,0x4A,0xA2,0xFF,0x01,0x51,0xA6,0xFF,
  0x00,0x54,0xAA,0xFF,0x05,0x55,0xB4,0xFF,0x00,0x51,0xAE,0xFF,0x00,0x50,0xAE,0xFF,0x00,0x50,0xA9,0xFF,
  0x07,0x53,0xAB,0xFF,0x01,0x51,0xAA,0xFF,0x00,0x52,0xAA,0xFF,0x0A,0x45,0xA1,0xFF,0x02,0x37,0x87,0xFF,
  0x00,0x37,0x5C,0xFF,0x07,0x39,0x78,0xFF,0x04,0x3C,0x86,0xFF,0x03,0x4E,0x88,0xFF,0x02,0x56,0xB0,0xFF,
  0x0D,0x64,0xC5,0xFF,0x0D,0x4E,0x8E,0xFF,0x01,0x77,0xD7,0xFF,0x31,0x6B,0x5D,0xFF,0x35,0x48,0x06,0xFF,
  0x7D,0x99,0x08,0xFF,0x8D,0xAE,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8D,0xAF,0x03,0xFF,0x8A,0xAB,0x00,0xFF,
  0x87,0xAD,0x00,0xFF,0x93,0xB4,0x03,0xFF,0xB7,0xE9,0x0C,0xFF,0xDA,0xFF,0x2C,0xFF,0x6B,0x91,0x6C,0xFF,
  0x1A,0x2C,0x3A,0xFF,0x00,0x17,0x26,0xFF,0x00,0x10,0x22,0xFF,0x00,0x13,0x2E,0xFF,0x00,0x1C,0x34,0xFF,
  0x01,0x26,0x5A,0xFF,0x04,0x34,0x64,0xFF,0x45,0x52,0x26,0xFF,0x71,0x92,0x00,0xFF,0x94,0xB2,0x02,0xFF,
  0x88,0xAF,0x00,0xFF,0x8C,0xA9,0x03,0xFF,0x86,0xAD,0x00,0xFF,0x90,0xB4,0x00,0xFF,0x91,0xB4,0x00,0xFF,
  0xAB,0xD4,0x14,0xFF,0x7B,0x94,0x5D,0xFF,0x08,0x27,0x44,0xFF,0x00,0x39,0x78,0xFF,0x03,0x52,0xB1,0xFF,
  0x00,0x50,0xAC,0xFF,0x04,0x55,0xAE,0xFF,0x00,0x41,0x7C,0xFF,0x00,0x25,0x40,0xFF,0x05,0x1F,0x52,0xFF,
  0x04,0x20,0x45,0xFF,0x00,0x19,0x30,0xFF,0x00,0x0C,0x13,0xFF,0x00,0x1D,0x44,0xFF,0x00,0x50,0x84,0xFF,
  0x00,0x4C,0xAE,0xFF,0x00,0x53,0xA3,0xFF,0x00,0x59,0xAD,0xFF,0x00,0x57,0xB7,0xFF,0x03,0x4F,0xAD,0xFF,
  0x00,0x4F,0xA2,0xFF,0x08,0x54,0xB6,0xFF,0x00,0x55,0xB2,0xFF,0x00,0x52,0xA8,0xFF,0x05,0x52,0xAA,0xFF,
  0x00,0x50,0xAB,0xFF,0x00,0x47,0x8C,0xFF,0x01,0x2C,0x3F,0xFF,0x00,0x10,0x1A,0xFF,0x04,0x09,0x0D,0xFF,
  0x00,0x03,0x14,0xFF,0x00,0x0E,0x1D,0xFF,0x00,0x1A,0x36,0xFF,0x00,0x42,0x8D,0xFF,0x05,0x6C,0xC8,0xFF,
  0x08,0x53,0x97,0xFF,0x0B,0x4D,0x9B,0xFF,0x3D,0x64,0x2F,0xFF,0x62,0x75,0x02,0xFF,0x85,0xA4,0x01,0xFF,
  0x8B,0xAD,0x00,0xFF,0x89,0xA9,0x00,0xFF,0x8D,0xAD,0x02,0xFF,0x90,0xAC,0x00,0xFF,0x94,0xAB,0x03,0xFF,
  0x88,0xB1,0x00,0xFF,0x97,0xB3,0x00,0xFF,0xBA,0xE8,0x00,0xFF,0xDA,0xFE,0x2A,0xFF,0xB1,0xD9,0x52,0xFF,
  0x78,0x91,0x58,0xFF,0x2F,0x5B,0x44,0xFF,0x24,0x3F,0x38,0xFF,0x18,0x36,0x38,0xFF,0x21,0x49,0x4B,0xFF,
  0x4E,0x6F,0x40,0xFF,0x74,0xA0,0x05,0xFF,0x8A,0xA5,0x00,0xFF,0x8C,0xA8,0x00,0xFF,0x8F,0xA9,0x0A,0xFF,
  0x8B,0xAC,0x03,0xFF,0x8C,0xAD,0x06,0xFF,0x8D,0xA7,0x00,0xFF,0x9C,0xBA,0x00,0xFF,0xB2,0xE9,0x1E,0xFF,
  0x6A,0x9C,0x5D,0xFF,0x00,0x29,0x28,0xFF,0x05,0x25,0x5E,0xFF,0x00,0x42,0x88,0xFF,0x00,0x4C,0x97,0xFF,
  0x00,0x36,0x81,0xFF,0x21,0x45,0x55,0xFF,0x32,0x4A,0x3A,0xFF,0x3C,0x64,0x49,0xFF,0x40,0x6E,0x54,0xFF,
  0x4D,0x73,0x4C,0xFF,0x06,0x07,0x0B,0xFF,0x00,0x01,0x02,0xFF,0x05,0x21,0x49,0xFF,0x00,0x47,0x93,0xFF,
  0x04,0x54,0xA9,0xFF,0x05,0x4F,0xA6,0xFF,0x00,0x4F,0xAE,0xFF,0x00,0x51,0xAF,0xFF,0x00,0x52,0xAA,0xFF,
  0x07,0x51,0xAA,0xFF,0x00,0x51,0xA8,0xFF,0x00,0x51,0xA7,0xFF,0x00,0x58,0xB9,0xFF,0x01,0x5C,0xAD,0xFF,
  0x14,0x40,0x67,0xFF,0x1C,0x2F,0x3D,0xFF,0x39,0x5A,0x2D,0xFF,0x5D,0x85,0x49,0xFF,0x49,0x76,0x4B,0xFF,
  0x2B,0x4A,0x4C,0xFF,0x00,0x07,0x0E,0xFF,0x00,0x23,0x3D,0xFF,0x05,0x57,0xA0,0xFF,0x0A,0x59,0x92,0xFF,
  0x23,0x41,0x39,0xFF,0x5C,0x77,0x04,0xFF,0x86,0x9F,0x03,0xFF,0x88,0xAB,0x00,0xFF,0x8E,0xAF,0x00,0xFF,
  0x8D,0xAA,0x00,0xFF,0x8A,0xAA,0x01,0xFF,0x8E,0xAE,0x01,0xFF,0x8A,0xAB,0x06,0xFF,0x89,0xAF,0x00,0xFF,
  0x8C,0xAE,0x02,0xFF,0x96,0xB4,0x06,0xFF,0xA4,0xDC,0x00,0xFF,0xD0,0xFF,0x07,0xFF,0xD2,0xFC,0x16,0xFF,
  0xDC,0xFE,0x35,0xFF,0xDA,0xFB,0x38,0xFF,0xD0,0xFF,0x21,0xFF,0xB9,0xE1,0x1E,0xFF,0x9B,0xC6,0x14,0xFF,
  0x8E,0xB5,0x00,0xFF,0x88,0xAD,0x07,0xFF,0x8E,0xAD,0x00,0xFF,0x8B,0xAF,0x03,0xFF,0x89,0xAD,0x03,0xFF,
  0x8F,0xAB,0x01,0xFF,0x91,0xAE,0x00,0xFF,0x9A,0xBC,0x03,0xFF,0xBA,0xF0,0x16,0xFF,0xA8,0xE8,0x4B,0xFF,
  0x1F,0x46,0x57,0xFF,0x04,0x1C,0x38,0xFF,0x00,0x1D,0x45,0xFF,0x06,0x3B,0x7F,0xFF,0x2D,0x70,0x76,0xFF,
  0x69,0x9F,0x41,0xFF,0xB6,0xE3,0x28,0xFF,0xC2,0xEF,0x22,0xFF,0xCC,0xFB,0x23,0xFF,0xD5,0xFF,0x19,0xFF,
  0xCB,0xE6,0x39,0xFF,0x2B,0x45,0x2C,0xFF,0x00,0x01,0x00,0xFF,0x07,0x1D,0x44,0xFF,0x00,0x47,0x92,0xFF,
  0x02,0x51,0xB6,0xFF,0x01,0x55,0xAB,0xFF,0x03,0x52,0xB1,0xFF,0x02,0x52,0xAB,0xFF,0x00,0x4F,0xAC,0xFF,
  0x00,0x53,0xB1,0xFF,0x01,0x56,0xA7,0xFF,0x00,0x61,0xCC,0xFF,0x1B,0x64,0xA9,0xFF,0x41,0x58,0x20,0xFF,
  0x5F,0x7E,0x07,0xFF,0x8D,0xB3,0x00,0xFF,0xBD,0xEF,0x02,0xFF,0xD0,0xFD,0x0C,0xFF,0xD0,0xFF,0x1C,0xFF,
  0x88,0xBD,0x2D,0xFF,0x18,0x49,0x74,0xFF,0x02,0x44,0x98,0xFF,0x4A,0x9C,0x6B,0xFF,0x6C,0x8C,0x11,0xFF,
  0x80,0xA3,0x00,0xFF,0x8B,0xAF,0x05,0xFF,0x8D,0xB0,0x00,0xFF,0x8E,0xAD,0x00,0xFF,0x90,0xAB,0x04,0xFF,
  0x8C,0xAC,0x01,0xFF,0x87,0xAE,0x00,0xFF,0x90,0xAC,0x00,0xFF,0x8B,0xAD,0x00,0xFF,0x8A,0xAC,0x00,0xFF,
  0x91,0xAA,0x06,0xFF,0x8D,0xA7,0x00,0xFF,0x97,0xB4,0x00,0xFF,0x9B,0xBD,0x04,0xFF,0xA1,0xCF,0x0D,0xFF,
  0xA9,0xCE,0x02,0xFF,0xA4,0xD0,0x01,0xFF,0x96,0xB3,0x03,0xFF,0x89,0xB1,0x00,0xFF,0x8D,0xAB,0x00,0xFF,
  0x89,0xAF,0x02,0xFF,0x90,0xAB,0x06,0xFF,0x89,0xAB,0x00,0xFF,0x8C,0xB1,0x01,0xFF,0x8F,0xAB,0x00,0xFF,
  0x8A,0xAC,0x00,0xFF,0x90,0xBA,0x01,0xFF,0xAD,0xD0,0x02,0xFF,0xD5,0xFF,0x11,0xFF,0xB2,0xE6,0x46,0xFF,
  0x6C,0x99,0x5E,0xFF,0x57,0x73,0x4B,0xFF,0x70,0x96,0x41,0xFF,0xA8,0xCE,0x2B,0xFF,0xA9,0xCA,0x00,0xFF,
  0x9C,0xB9,0x00,0xFF,0x9E,0xBB,0x00,0xFF,0x99,0xB8,0x00,0xFF,0xB6,0xDD,0x06,0xFF,0xD4,0xFD,0x01,0xFF,
  0xC2,0xE6,0x3C,0xFF,0x07,0x22,0x35,0xFF,0x06,0x0F,0x36,0xFF,0x00,0x44,0x8A,0xFF,0x00,0x52,0xA8,0xFF,
  0x05,0x50,0xAB,0xFF,0x00,0x53,0xB0,0xFF,0x02,0x53,0xAE,0xFF,0x01,0x51,0xB2,0xFF,0x00,0x4E,0xB0,0xFF,
  0x00,0x5A,0xBB,0xFF,0x19,0x79,0xCF,0xFF,0x49,0x76,0x7B,0xFF,0x5A,0x6C,0x00,0xFF,0x70,0x93,0x00,0xFF,
  0x8E,0xB3,0x03,0xFF,0x8A,0xB4,0x00,0xFF,0xA1,0xB9,0x09,0xFF,0xB9,0xDE,0x00,0xFF,0xC4,0xF4,0x00,0xFF,
  0xB3,0xDE,0x13,0xFF,0xA7,0xD3,0x0A,0xFF,0xAD,0xD4,0x00,0xFF,0x9B,0xBE,0x0A,0xFF,0x8B,0xB2,0x00,0xFF,
  0x87,0xAB,0x01,0xFF,0x8E,0xAE,0x01,0xFF,0x8A,0xA8,0x00,0xFF,0x8D,0xAD,0x04,0xFF,0x8E,0xB0,0x01,0xFF,
  0x89,0xAE,0x00,0xFF,0x8A,0xAE,0x02,0xFF,0x8A,0xAA,0x00,0xFF,0x91,0xAE,0x08,0xFF,0x8A,0xAF,0x00,0xFF,
  0x8F,0xAC,0x06,0xFF,0x8B,0xAB,0x00,0xFF,0x8E,0xA8,0x00,0xFF,0x97,0xA8,0x02,0xFF,0x87,0xAC,0x00,0xFF,
  0x8E,0xAB,0x05,0xFF,0x89,0xA9,0x00,0xFF,0x92,0xAB,0x0F,0xFF,0x8F,0xAD,0x00,0xFF,0x8A,0xA6,0x00,0xFF,
  0x8D,0xAA,0x02,0xFF,0x94,0xAF,0x02,0xFF,0x8B,0xA8,0x00,0xFF,0x92,0xAB,0x09,0xFF,0x8E,0xAC,0x00,0xFF,
  0x82,0xAA,0x00,0xFF,0x90,0xBB,0x00,0xFF,0xA3,0xCD,0x00,0xFF,0xCB,0xF8,0x05,0xFF,0xD1,0xFF,0x14,0xFF,
  0xBE,0xEA,0x11,0xFF,0xA0,0xCB,0x09,0xFF,0x8F,0xB5,0x00,0xFF,0x8F,0xAD,0x00,0xFF,0x8E,0xAF,0x00,0xFF,
  0x93,0xAF,0x03,0xFF,0x8F,0xAD,0x00,0xFF,0x86,0xB6,0x00,0xFF,0xB1,0xE2,0x00,0xFF,0xCC,0xFF,0x04,0xFF,
  0x41,0x87,0x51,0xFF,0x00,0x07,0x11,0xFF,0x02,0x30,0x61,0xFF,0x00,0x54,0xAA,0xFF,0x04,0x4E,0xAD,0xFF,
  0x02,0x52,0xA7,0xFF,0x00,0x54,0xA4,0xFF,0x04,0x52,0xB4,0xFF,0x01,0x51,0xA6,0xFF,0x00,0x73,0xCC,0xFF,
  0x33,0x8A,0xB7,0xFF,0x4A,0x61,0x2B,0xFF,0x51,0x69,0x00,0xFF,0x82,0xA7,0x02,0xFF,0x8B,0xB2,0x00,0xFF,
  0x8C,0xAF,0x0A,0xFF,0x8B,0xAD,0x00,0xFF,0x8B,0xB3,0x03,0xFF,0x93,0xB7,0x00,0xFF,0xAB,0xCD,0x00,0xFF,
  0x93,0xBD,0x00,0xFF,0x94,0xB5,0x02,0xFF,0x92,0xAB,0x00,0xFF,0x89,0xAA,0x00,0xFF,0x8E,0xAE,0x01,0xFF,
  0x8E,0xAB,0x01,0xFF,0x8B,0xAD,0x01,0xFF,0x8C,0xB0,0x06,0xFF,0x89,0xAA,0x00,0xFF,0x8F,0xAA,0x03,0xFF,
  0x8C,0xAB,0x05,0xFF,0x91,0xAC,0x00,0xFF,0x89,0xAB,0x00,0xFF,0x89,0xAE,0x00,0xFF,0x8E,0xAB,0x01,0xFF,
  0x8C,0xAD,0x00,0xFF,0x88,0xB3,0x02,0xFF,0x88,0xAD,0x00,0xFF,0x89,0xAF,0x00,0xFF,0x8E,0xAD,0x00,0xFF,
  0x8C,0xAE,0x02,0xFF,0x8A,0xAA,0x00,0xFF,0x8D,0xAE,0x00,0xFF,0x8D,0xAF,0x00,0xFF,0x8A,0xB1,0x06,0xFF,
  0x88,0xAA,0x00,0xFF,0x92,0xAD,0x00,0xFF,0x87,0xAD,0x00,0xFF,0x8D,0xAF,0x00,0xFF,0x95,0xAB,0x0A,0xFF,
  0x8C,0xAB,0x00,0xFF,0x8B,0xAA,0x06,0xFF,0x94,0xB1,0x00,0xFF,0x9D,0xC1,0x00,0xFF,0x93,0xB4,0x01,0xFF,
  0x8C,0xB7,0x00,0xFF,0x8D,0xAE,0x05,0xFF,0x8F,0xAB,0x01,0xFF,0x86,0xAD,0x00,0xFF,0x87,0xAD,0x02,0xFF,
  0x8D,0xAA,0x00,0xFF,0x8E,0xAF,0x00,0xFF,0x92,0xB3,0x00,0xFF,0xD0,0xFE,0x05,0xFF,0x7F,0xA1,0x40,0xFF,
  0x04,0x09,0x0F,0xFF,0x00,0x33,0x60,0xFF,0x02,0x56,0xB8,0xFF,0x00,0x50,0xA9,0xFF,0x04,0x58,0xBA,0xFF,
  0x00,0x52,0xAE,0xFF,0x03,0x53,0xAC,0xFF,0x03,0x51,0xB5,0xFF,0x01,0x77,0xF5,0xFF,0x40,0x99,0xBB,0xFF,
  0x33,0x3E,0x1E,0xFF,0x41,0x55,0x00,0xFF,0x7C,0x9B,0x02,0xFF,0x8C,0xAF,0x00,0xFF,0x91,0xAB,0x02,0xFF,
  0x86,0xAC,0x00,0xFF,0x95,0xAA,0x03,0xFF,0x82,0xAF,0x00,0xFF,0x8D,0xAE,0x00,0xFF,0x8A,0xAB,0x00,0xFF,
  0x8C,0xAC,0x00,0xFF,0x8E,0xAB,0x03,0xFF,0x89,0xB1,0x01,0xFF,0x8E,0xA9,0x00,0xFF,0x8E,0xAF,0x06,0xFF,
  0x8B,0xAC,0x00,0xFF,0x93,0xAA,0x04,0xFF,0x84,0xAF,0x00,0xFF,0x8D,0xAA,0x04,0xFF,0x8D,0xAE,0x00,0xFF,
  0x8D,0xAE,0x00,0xFF,0x8D,0xA8,0x01,0xFF,0x8E,0xA8,0x06,0xFF,0x8A,0xB0,0x03,0xFF,0x92,0xA9,0x01,0xFF,
  0x84,0xB0,0x00,0xFF,0x8A,0xAC,0x00,0xFF,0x8B,0xAA,0x04,0xFF,0x8B,0xB1,0x06,0xFF,0x86,0xAC,0x01,0xFF,
  0x8E,0xAD,0x00,0xFF,0x8F,0xAA,0x03,0xFF,0x86,0xAA,0x02,0xFF,0x8C,0xAF,0x00,0xFF,0x92,0xAA,0x00,0xFF,
  0x87,0xAD,0x00,0xFF,0x8F,0xAB,0x01,0xFF,0x92,0xAD,0x00,0xFF,0x89,0xAE,0x00,0xFF,0x87,0xAC,0x06,0xFF,
  0x94,0xA7,0x00,0xFF,0x8D,0xA6,0x00,0xFF,0x72,0x91,0x07,0xFF,0x5C,0x74,0x00,0xFF,0x5D,0x6F,0x03,0xFF,
  0x70,0x96,0x05,0xFF,0x89,0xB1,0x00,0xFF,0x8E,0xAB,0x03,0xFF,0x8F,0xAC,0x04,0xFF,0x8D,0xA8,0x00,0xFF,
  0x88,0xAF,0x00,0xFF,0x8F,0xB3,0x00,0xFF,0xBC,0xF0,0x07,0xFF,0x7F,0x99,0x36,0xFF,0x00,0x10,0x1F,0xFF,
  0x00,0x32,0x5F,0xFF,0x07,0x53,0xB5,0xFF,0x00,0x51,0xAB,0xFF,0x00,0x57,0xB8,0xFF,0x00,0x50,0xA5,0xFF,
  0x00,0x53,0xAE,0xFF,0x00,0x50,0xA5,0xFF,0x02,0x7C,0xFB,0xFF,0x3C,0x95,0xB3,0xFF,0x2C,0x31,0x00,0xFF,
  0x40,0x54,0x00,0xFF,0x74,0x90,0x07,0xFF,0x8A,0xAA,0x00,0xFF,0x8F,0xA8,0x04,0xFF,0x8E,0xAA,0x00,0xFF,
  0x89,0xAF,0x00,0xFF,0x8C,0xAD,0x04,0xFF,0x90,0xA6,0x00,0xFF,0x8A,0xB3,0x0C,0xFF,0x8D,0xAB,0x00,0xFF,
  0x90,0xAC,0x02,0xFF,0x84,0xAB,0x00,0xFF,0x91,0xB0,0x00,0xFF,0x8D,0xAC,0x00,0xFF,0x87,0xA9,0x00,0xFF,
  0x8C,0xAE,0x02,0xFF,0x8D,0xAE,0x07,0xFF,0x8F,0xAC,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8B,0xAA,0x04,0xFF,
  0x8E,0xAD,0x00,0xFF,0x8F,0xAC,0x02,0xFF,0x86,0xB0,0x00,0xFF,0x90,0xA7,0x01,0xFF,0x8C,0xB1,0x01,0xFF,
  0x91,0xAC,0x00,0xFF,0x8E,0xAC,0x00,0xFF,0x8F,0xA6,0x00,0xFF,0x91,0xAE,0x00,0xFF,0x8B,0xAD,0x00,0xFF,
  0x8A,0xAC,0x00,0xFF,0x8C,0xAE,0x00,0xFF,0x89,0xAD,0x03,0xFF,0x8D,0xAB,0x00,0xFF,0x8C,0xAD,0x00,0xFF,
  0x8D,0xA9,0x00,0xFF,0x8F,0xA9,0x00,0xFF,0x8D,0xAD,0x02,0xFF,0x86,0xB2,0x00,0xFF,0x7D,0xA7,0x00,0xFF,
  0x59,0x72,0x17,0xFF,0x34,0x3F,0x15,0xFF,0x14,0x17,0x02,0xFF,0x00,0x02,0x07,0xFF,0x2B,0x30,0x00,0xFF,
  0x6A,0x7D,0x00,0xFF,0x84,0xA8,0x00,0xFF,0x8A,0xB3,0x01,0xFF,0x8C,0xAE,0x02,0xFF,0x91,0xAC,0x05,0xFF,
  0x92,0xB6,0x00,0xFF,0xB2,0xE3,0x15,0xFF,0x47,0x5F,0x47,0xFF,0x02,0x22,0x39,0xFF,0x02,0x40,0x89,0xFF,
  0x00,0x50,0xA1,0xFF,0x02,0x56,0xB0,0xFF,0x03,0x53,0xAA,0xFF,0x07,0x51,0xB0,0xFF,0x01,0x52,0xAD,0xFF,
  0x03,0x51,0xA6,0xFF,0x00,0x7B,0xFF,0xFF,0x2A,0x92,0xC7,0xFF,0x22,0x2A,0x13,0xFF,0x1D,0x29,0x00,0xFF,
  0x6E,0x86,0x00,0xFF,0x8B,0xB3,0x03,0xFF,0x87,0xAD,0x00,0xFF,0x8D,0xAE,0x00,0xFF,0x8F,0xAB,0x01,0xFF,
  0x8D,0xAA,0x00,0xFF,0x8B,0xB1,0x02,0xFF,0x81,0xA2,0x00,0xFF,0x7A,0x9B,0x06,0xFF,0x78,0x92,0x00,0xFF,
  0x6F,0x85,0x00,0xFF,0x72,0x8F,0x01,0xFF,0x82,0xA4,0x00,0xFF,0x91,0xAD,0x0B,0xFF,0x91,0xAD,0x00,0xFF,
  0x8F,0xAC,0x00,0xFF,0x88,0xAB,0x00,0xFF,0x88,0xAC,0x02,0xFF,0x8E,0xB1,0x00,0xFF,0x8D,0xAA,0x00,0xFF,
  0x92,0xAD,0x00,0xFF,0x8E,0xAF,0x00,0xFF,0x87,0xA0,0x07,0xFF,0x76,0x9D,0x00,0xFF,0x7C,0x9B,0x02,0xFF,
  0x75,0xA3,0x00,0xFF,0x83,0xA2,0x00,0xFF,0x90,0xAC,0x00,0xFF,0x8C,0xAA,0x0A,0xFF,0x8B,0xAE,0x00,0xFF,
  0x8A,0xB0,0x05,0xFF,0x89,0xAB,0x00,0xFF,0x93,0xA7,0x07,0xFF,0x91,0xAC,0x05,0xFF,0x8E,0xB0,0x00,0xFF,
  0x8A,0xAD,0x09,0xFF,0x88,0xA7,0x01,0xFF,0x93,0xB6,0x00,0xFF,0x6F,0x9A,0x2F,0xFF,0x3B,0x72,0x52,0xFF,
  0x1D,0x60,0x7A,0xFF,0x22,0x61,0x8A,0xFF,0x1D,0x43,0x56,0xFF,0x0C,0x1B,0x00,0xFF,0x28,0x37,0x0C,0xFF,
  0x47,0x58,0x00,0xFF,0x6E,0x83,0x00,0xFF,0x75,0x9C,0x01,0xFF,0x79,0x96,0x00,0xFF,0x8B,0xAA,0x06,0xFF,
  0x80,0xA8,0x13,0xFF,0x18,0x34,0x37,0xFF,0x01,0x2C,0x4F,0xFF,0x03,0x48,0x93,0xFF,0x00,0x55,0xA7,0xFF,
  0x00,0x52,0xAE,0xFF,0x02,0x50,0xA5,0xFF,0x02,0x50,0xB2,0xFF,0x00,0x52,0xA6,0xFF,0x01,0x56,0xA9,0xFF,
  0x00,0x7A,0xED,0xFF,0x31,0x93,0xEE,0xFF,0x43,0x56,0x38,0xFF,0x0C,0x0F,0x04,0xFF,0x3E,0x4F,0x01,0xFF,
  0x7F,0x96,0x00,0xFF,0x81,0xA5,0x00,0xFF,0x88,0xB1,0x00,0xFF,0x95,0xA7,0x07,0xFF,0x90,0xAC,0x00,0xFF,
  0x7B,0x9C,0x07,0xFF,0x65,0x77,0x07,0xFF,0x63,0x7D,0x10,0xFF,0x35,0x5D,0x21,0xFF,0x43,0x55,0x17,0xFF,
  0x44,0x4F,0x0A,0xFF,0x67,0x84,0x04,0xFF,0x8F,0xAD,0x00,0xFF,0x8B,0xA6,0x00,0xFF,0x8A,0xAA,0x00,0xFF,
  0x8F,0xB2,0x00,0xFF,0x8D,0xA9,0x00,0xFF,0x8E,0xAE,0x03,0xFF,0x89,0xAA,0x01,0xFF,0x89,0xAC,0x00,0xFF,
  0x73,0x95,0x0D,0xFF,0x52,0x6C,0x07,0xFF,0x28,0x34,0x0E,0xFF,0x23,0x22,0x03,0xFF,0x26,0x25,0x07,0xFF,
  0x42,0x54,0x00,0xFF,0x5D,0x78,0x03,0xFF,0x7D,0x9D,0x00,0xFF,0x8F,0xAB,0x08,0xFF,0x91,0xA9,0x00,0xFF,
  0x91,0xAD,0x01,0xFF,0x88,0xAD,0x05,0xFF,0x89,0xAC,0x00,0xFF,0x8A,0xAC,0x00,0xFF,0x89,0xAF,0x04,0xFF,
  0x8D,0xB2,0x00,0xFF,0xAD,0xD0,0x00,0xFF,0x58,0x7E,0x4D,0xFF,0x19,0x5C,0x86,0xFF,0x00,0x6A,0xDF,0xFF,
  0x00,0x88,0xFD,0xFF,0x0E,0x8B,0xF7,0xFF,0x32,0x77,0xA0,0xFF,0x20,0x39,0x1C,0xFF,0x00,0x01,0x0E,0xFF,
  0x0E,0x17,0x02,0xFF,0x3A,0x49,0x12,0xFF,0x46,0x55,0x02,0xFF,0x50,0x5C,0x14,0xFF,0x30,0x48,0x28,0xFF,
  0x11,0x38,0x59,0xFF,0x08,0x40,0x8A,0xFF,0x01,0x4E,0x9E,0xFF,0x02,0x57,0xB1,0xFF,0x00,0x51,0xAF,0xFF,
  0x03,0x53,0xAA,0xFF,0x00,0x53,0xA8,0xFF,0x01,0x50,0xAC,0xFF,0x00,0x51,0x9F,0xFF,0x04,0x70,0xD5,0xFF,
  0x28,0x92,0xFF,0xFF,0x4B,0x90,0xC9,0xFF,0x2D,0x52,0x27,0xFF,0x24,0x2B,0x00,0xFF,0x3C,0x4B,0x08,0xFF,
  0x54,0x67,0x0B,0xFF,0x56,0x74,0x02,0xFF,0x5A,0x72,0x04,0xFF,0x65,0x86,0x04,0xFF,0x37,0x56,0x06,0xFF,
  0x19,0x59,0x7C,0xFF,0x32,0x7C,0x85,0xFF,0x16,0x59,0x90,0xFF,0x20,0x56,0x78,0xFF,0x26,0x33,0x17,0xFF,
  0x44,0x52,0x00,0xFF,0x80,0xA7,0x00,0xFF,0x8D,0xAE,0x07,0xFF,0x89,0xAC,0x08,0xFF,0x8F,0xAA,0x00,0xFF,
  0x8D,0xAE,0x00,0xFF,0x89,0xAA,0x03,0xFF,0x86,0xA6,0x05,0xFF,0x77,0x95,0x03,0xFF,0x53,0x6F,0x18,0xFF,
  0x45,0x6C,0x35,0xFF,0x2B,0x47,0x4B,0xFF,0x20,0x3A,0x1D,0xFF,0x10,0x13,0x00,0xFF,0x01,0x12,0x02,0xFF,
  0x15,0x1B,0x01,0xFF,0x4F,0x55,0x00,0xFF,0x7C,0x9B,0x02,0xFF,0x8A,0xAF,0x00,0xFF,0x8C,0xAA,0x00,0xFF,
  0x87,0xB0,0x00,0xFF,0x8C,0xAF,0x0A,0xFF,0x8E,0xAA,0x00,0xFF,0x8D,0xAB,0x00,0xFF,0x9A,0xC1,0x04,0xFF,
  0xA6,0xCC,0x27,0xFF,0x26,0x4F,0x7B,0xFF,0x04,0x3C,0x77,0xFF,0x00,0x56,0xB0,0xFF,0x00,0x6D,0xD2,0xFF,
  0x00,0x6F,0xEC,0xFF,0x0A,0x84,0xFF,0xFF,0x3A,0x96,0xD7,0xFF,0x49,0x73,0x4B,0xFF,0x18,0x22,0x0A,0xFF,
  0x0E,0x0F,0x00,0xFF,0x1B,0x1C,0x0C,0xFF,0x19,0x2F,0x3A,0xFF,0x01,0x26,0x52,0xFF,0x03,0x41,0x8A,0xFF,
  0x00,0x51,0xAD,0xFF,0x00,0x52,0xAD,0xFF,0x00,0x51,0xA3,0xFF,0x01,0x52,0xAE,0xFF,0x00,0x50,0xAE,0xFF,
  0x00,0x53,0xB3,0xFF,0x01,0x53,0xA7,0xFF,0x00,0x55,0xA8,0xFF,0x02,0x59,0xC2,0xFF,0x00,0x6F,0xF1,0xFF,
  0x11,0x93,0xF3,0xFF,0x38,0x93,0xE6,0xFF,0x47,0x89,0x87,0xFF,0x3D,0x61,0x31,0xFF,0x4E,0x5B,0x16,0xFF,
  0x3E,0x52,0x0B,0xFF,0x49,0x63,0x22,0xFF,0x54,0x90,0x76,0xFF,0x2E,0x6F,0x69,0xFF,0x05,0x61,0xC4,0xFF,
  0x0A,0x7B,0xD7,0xFF,0x0F,0x64,0xC1,0xFF,0x14,0x79,0xD3,0xFF,0x30,0x5D,0x48,0xFF,0x39,0x45,0x05,0xFF,
  0x70,0x96,0x05,0xFF,0x8B,0xAB,0x00,0xFF,0x8D,0xAD,0x00,0xFF,0x8B,0xAB,0x00,0xFF,0x8D,0xAD,0x04,0xFF,
  0x91,0xB8,0x00,0xFF,0x83,0xA7,0x11,0xFF,0x3F,0x60,0x35,0xFF,0x10,0x45,0x67,0xFF,0x11,0x74,0xCF,0xFF,
  0x0C,0x7F,0xFF,0xFF,0x06,0x87,0xFC,0xFF,0x2A,0x8C,0xE3,0xFF,0x47,0x93,0x91,0xFF,0x2E,0x4C,0x26,0xFF,
  0x18,0x1C,0x0D,0xFF,0x43,0x61,0x00,0xFF,0x7E,0xA0,0x03,0xFF,0x91,0xA8,0x02,0xFF,0x89,0xAC,0x00,0xFF,
  0x8D,0xAA,0x02,0xFF,0x8C,0xAC,0x00,0xFF,0x8E,0xAE,0x03,0xFF,0xAC,0xD7,0x01,0xFF,0x9B,0xC7,0x2C,0xFF,
  0x06,0x42,0x88,0xFF,0x01,0x3E,0x7F,0xFF,0x08,0x50,0xB4,0xFF,0x01,0x51,0xB4,0xFF,0x07,0x5A,0xC0,0xFF,
  0x02,0x6F,0xEE,0xFF,0x05,0x87,0xFF,0xFF,0x0C,0x8C,0xFA,0xFF,0x49,0xA8,0xDE,0xFF,0x4C,0x8D,0x91,0xFF,
  0x24,0x73,0x84,0xFF,0x10,0x59,0x9D,0xFF,0x02,0x4D,0x90,0xFF,0x00,0x50,0xA0,0xFF,0x00,0x56,0xA9,0xFF,
  0x06,0x53,0xAF,0xFF,0x07,0x50,0xAE,0xFF,0x02,0x58,0xAF,0xFF,0x00,0x59,0xBB,0xFF,0x03,0x53,0xB2,0xFF,
  0x00,0x50,0xAA,0xFF,0x00,0x51,0xA6,0xFF,0x04,0x52,0xA7,0xFF,0x00,0x5C,0xBB,0xFF,0x00,0x68,0xC4,0xFF,
  0x02,0x71,0xF6,0xFF,0x03,0x7D,0xFF,0xFF,0x08,0x85,0xFD,0xFF,0x0B,0x72,0xCE,0xFF,0x02,0x64,0xBD,0xFF,
  0x0A,0x65,0xB8,0xFF,0x02,0x72,0xFA,0xFF,0x00,0x6C,0xD9,0xFF,0x00,0x56,0xBA,0xFF,0x0B,0x66,0xC3,0xFF,
  0x09,0x61,0xB8,0xFF,0x18,0x7A,0xB9,0xFF,0x3B,0x6D,0x52,0xFF,0x4C,0x5C,0x1B,0xFF,0x77,0x9A,0x00,0xFF,
  0x90,0xAD,0x00,0xFF,0x90,0xAC,0x02,0xFF,0x85,0xAE,0x00,0xFF,0x8F,0xA8,0x02,0xFF,0xA8,0xD2,0x02,0xFF,
  0x98,0xC6,0x32,0xFF,0x27,0x4B,0x57,0xFF,0x00,0x34,0x77,0xFF,0x00,0x6A,0xC7,0xFF,0x00,0x6E,0xE4,0xFF,
  0x01,0x76,0xE0,0xFF,0x04,0x7A,0xF5,0xFF,0x00,0x85,0xFE,0xFF,0x28,0x93,0xF3,0xFF,0x2A,0x54,0x60,0xFF,
  0x29,0x2E,0x00,0xFF,0x58,0x70,0x00,0xFF,0x7E,0xA5,0x00,0xFF,0x98,0xAE,0x01,0xFF,0x93,0xAE,0x07,0xFF,
  0x8C,0xAD,0x06,0xFF,0x86,0xAC,0x00,0xFF,0xA7,0xD4,0x00,0xFF,0x9C,0xC2,0x1F,0xFF,0x01,0x46,0x67,0xFF,
  0x00,0x42,0x83,0xFF,0x00,0x56,0xAF,0xFF,0x02,0x50,0xB2,0xFF,0x06,0x54,0x9F,0xFF,0x00,0x58,0xAF,0xFF,
  0x00,0x67,0xC6,0xFF,0x03,0x74,0xEC,0xFF,0x02,0x7F,0xFF,0xFF,0x00,0x80,0xF8,0xFF,0x00,0x76,0xF1,0xFF,
  0x00,0x62,0xD1,0xFF,0x00,0x57,0xB6,0xFF,0x00,0x55,0xAF,0xFF,0x00,0x53,0xAD,0xFF,0x04,0x4D,0xA9,0xFF,
  0x05,0x4C,0xB4,0xFF,0x00,0x52,0xA9,0xFF,0x00,0x5C,0xB6,0xFF,0x00,0x4F,0xAE,0xFF,0x09,0x54,0xA5,0xFF,
  0x05,0x45,0x9F,0xFF,0x00,0x52,0xA6,0xFF,0x00,0x59,0xA6,0xFF,0x01,0x50,0xAF,0xFF,0x09,0x53,0xAC,0xFF,
  0x00,0x58,0xB0,0xFF,0x00,0x6E,0xCF,0xFF,0x00,0x6A,0xCF,0xFF,0x04,0x64,0xCA,0xFF,0x02,0x58,0xAF,0xFF,
  0x03,0x4F,0xB1,0xFF,0x00,0x52,0xB9,0xFF,0x00,0x5E,0xB2,0xFF,0x0B,0x57,0xAB,0xFF,0x01,0x5E,0xAC,0xFF,
  0x32,0x66,0x8B,0xFF,0x50,0x66,0x37,0xFF,0x70,0x87,0x15,0xFF,0x82,0xAA,0x00,0xFF,0x88,0xAD,0x00,0xFF,
  0x8D,0xAB,0x00,0xFF,0x8D,0xAC,0x08,0xFF,0x8F,0xAD,0x00,0xFF,0xAE,0xDD,0x01,0xFF,0xAA,0xD2,0x60,0xFF,
  0x25,0x49,0x49,0xFF,0x00,0x2A,0x54,0xFF,0x00,0x54,0xA8,0xFF,0x03,0x53,0xB6,0xFF,0x00,0x4F,0xAC,0xFF,
  0x04,0x53,0xBC,0xFF,0x00,0x69,0xCB,0xFF,0x03,0x85,0xFF,0xFF,0x34,0x90,0xF7,0xFF,0x29,0x49,0x22,0xFF,
  0x2B,0x2D,0x08,0xFF,0x7A,0x9E,0x00,0xFF,0x87,0xAB,0x03,0xFF,0x85,0xAA,0x00,0xFF,0x8F,0xB0,0x00,0xFF,
  0x8C,0xAF,0x00,0xFF,0xA4,0xCE,0x08,0xFF,0xA9,0xC9,0x28,0xFF,0x0B,0x3E,0x69,0xFF,0x05,0x38,0x77,0xFF,
  0x00,0x51,0xA9,0xFF,0x00,0x53,0xAD,0xFF,0x00,0x55,0xAE,0xFF,0x06,0x50,0xB1,0xFF,0x02,0x51,0xB0,0xFF,
  0x00,0x59,0xB3,0xFF,0x02,0x69,0xD0,0xFF,0x00,0x6E,0xD4,0xFF,0x01,0x61,0xC5,0xFF,0x00,0x5E,0xB0,0xFF,
  0x02,0x4D,0xA8,0xFF,0x02,0x51,0xAE,0xFF,0x00,0x56,0xAF,0xFF,0x00,0x53,0xAE,0xFF,0x02,0x54,0xAA,0xFF,
  0x01,0x52,0xAE,0xFF,0x04,0x50,0xA6,0xFF,0x00,0x52,0xB2,0xFF,0x09,0x55,0xB3,0xFF,0x01,0x4C,0x9F,0xFF,
  0x00,0x4A,0x9A,0xFF,0x01,0x50,0xAC,0xFF,0x01,0x51,0xB0,0xFF,0x00,0x53,0xAF,0xFF,0x01,0x51,0xB4,0xFF,
  0x00,0x5B,0xB8,0xFF,0x03,0x58,0xB5,0xFF,0x02,0x52,0xB3,0xFF,0x03,0x54,0xAD,0xFF,0x01,0x53,0xA7,0xFF,
  0x00,0x57,0xBE,0xFF,0x05,0x57,0xBA,0xFF,0x12,0x58,0x96,0xFF,0x1E,0x64,0x98,0xFF,0x2E,0x4A,0x4D,0xFF,
  0x4F,0x67,0x0F,0xFF,0x80,0xA7,0x00,0xFF,0x91,0xAC,0x05,0xFF,0x8D,0xA9,0x00,0xFF,0x8A,0xAE,0x04,0xFF,
  0x8F,0xAD,0x00,0xFF,0x8F,0xAB,0x01,0xFF,0xB5,0xDE,0x06,0xFF,0xC6,0xE1,0x56,0xFF,0x18,0x40,0x4C,0xFF,
  0x00,0x2B,0x51,0xFF,0x01,0x53,0xA7,0xFF,0x05,0x4F,0xAE,0xFF,0x00,0x52,0xAC,0xFF,0x00,0x55,0xAE,0xFF,
  0x04,0x51,0xAD,0xFF,0x02,0x5F,0xC6,0xFF,0x02,0x78,0xF6,0xFF,0x32,0x6E,0x66,0xFF,0x2B,0x35,0x01,0xFF,
  0x6B,0x94,0x00,0xFF,0x90,0xAC,0x02,0xFF,0x88,0xAD,0x00,0xFF,0x92,0xB0,0x00,0xFF,0x85,0xAA,0x02,0xFF,
  0x9D,0xBC,0x06,0xFF,0xAB,0xE8,0x0F,0xFF,0x1C,0x51,0x71,0xFF,0x00,0x31,0x63,0xFF,0x01,0x54,0xA4,0xFF,
  0x00,0x53,0xA6,0xFF,0x00,0x54,0xAE,0xFF,0x00,0x52,0xB2,0xFF,0x00,0x55,0xA8,0xFF,0x01,0x52,0xAD,0xFF,
  0x01,0x50,0xAC,0xFF,0x04,0x52,0xB4,0xFF,0x00,0x51,0xAC,0xFF,0x00,0x54,0xAB,0xFF,0x00,0x51,0xAE,0xFF,
  0x04,0x55,0xB0,0xFF,0x03,0x52,0xB1,0xFF,0x00,0x4E,0xAF,0xFF,0x02,0x53,0xAE,0xFF,0x00,0x53,0xB3,0xFF,
  0x00,0x52,0xAC,0xFF,0x00,0x55,0xAD,0xFF,0x00,0x58,0xB4,0xFF,0x00,0x51,0xA9,0xFF,0x01,0x52,0xA3,0xFF,
  0x04,0x54,0xAD,0xFF,0x02,0x53,0xAE,0xFF,0x04,0x51,0xA9,0xFF,0x09,0x50,0xAC,0xFF,0x00,0x53,0xA8,0xFF,
  0x03,0x55,0xAB,0xFF,0x00,0x51,0xA4,0xFF,0x00,0x54,0xA7,0xFF,0x03,0x54,0xAF,0xFF,0x00,0x50,0xA1,0xFF,
  0x07,0x5E,0xA1,0xFF,0x1B,0x66,0xA7,0xFF,0x36,0x73,0x74,0xFF,0x2E,0x49,0x16,0xFF,0x64,0x7B,0x06,0xFF,
  0x83,0xA8,0x02,0xFF,0x8D,0xAA,0x00,0xFF,0x8F,0xAD,0x00,0xFF,0x89,0xAD,0x05,0xFF,0x8A,0xAB,0x00,0xFF,
  0x8E,0xAD,0x00,0xFF,0xAB,0xD4,0x00,0xFF,0xCF,0xEC,0x46,0xFF,0x1F,0x50,0x4C,0xFF,0x00,0x14,0x46,0xFF,
  0x00,0x4C,0x99,0xFF,0x02,0x4E,0xAC,0xFF,0x00,0x54,0xAC,0xFF,0x00,0x50,0xA5,0xFF,0x00,0x54,0xB1,0xFF,
  0x00,0x55,0xB6,0xFF,0x01,0x7B,0xFF,0xFF,0x30,0x65,0x87,0xFF,0x3E,0x47,0x10,0xFF,0x78,0x9A,0x07,0xFF,
  0x90,0xAD,0x03,0xFF,0x8E,0xAC,0x00,0xFF,0x8D,0xA3,0x02,0xFF,0x8D,0xAF,0x00,0xFF,0x97,0xB5,0x07,0xFF,
  0xB3,0xE4,0x0C,0xFF,0x22,0x4F,0x79,0xFF,0x00,0x28,0x58,0xFF,0x00,0x4B,0x9D,0xFF,0x07,0x51,0xAA,0xFF,
  0x04,0x50,0xB0,0xFF,0x00,0x51,0xAD,0xFF,0x00,0x51,0xB1,0xFF,0x00,0x51,0xAB,0xFF,0x00,0x51,0xB1,0xFF,
  0x03,0x54,0xB0,0xFF,0x03,0x54,0xB0,0xFF,0x09,0x4F,0xB1,0xFF,0x01,0x51,0xAA,0xFF,0x00,0x51,0xA3,0xFF,
  0x00,0x52,0xA8,0xFF,0x00,0x55,0xAA,0xFF,0x00,0x55,0xA6,0xFF,0x00,0x51,0xAA,0xFF,0x00,0x55,0xB2,0xFF,
  0x00,0x58,0xA1,0xFF,0x00,0x5A,0xB0,0xFF,0x00,0x53,0xAB,0xFF,0x06,0x54,0xA9,0xFF,0x01,0x50,0xAC,0xFF,
  0x00,0x52,0xAE,0xFF,0x00,0x54,0xAD,0xFF,0x00,0x55,0xAF,0xFF,0x05,0x50,0xB5,0xFF,0x08,0x50,0xB6,0xFF,
  0x01,0x51,0xB0,0xFF,0x00,0x53,0xB3,0xFF,0x04,0x4E,0xAF,0xFF,0x0F,0x59,0xA0,0xFF,0x0E,0x5F,0x8E,0xFF,
  0x2C,0x74,0xAE,0xFF,0x43,0x70,0x51,0xFF,0x4B,0x61,0x0F,0xFF,0x7B,0x91,0x02,0xFF,0x8A,0xAB,0x02,0xFF,
  0x90,0xAA,0x0B,0xFF,0x8E,0xAD,0x00,0xFF,0x8B,0xB0,0x01,0xFF,0x8C,0xAC,0x03,0xFF,0x8D,0xAB,0x00,0xFF,
  0xA7,0xCF,0x00,0xFF,0xDA,0xFD,0x2D,0xFF,0x1B,0x4A,0x52,0xFF,0x06,0x0B,0x1F,0xFF,0x00,0x43,0x8A,0xFF,
  0x04,0x52,0xB4,0xFF,0x00,0x52,0xA8,0xFF,0x05,0x50,0xAB,0xFF,0x00,0x55,0xA5,0xFF,0x00,0x50,0xB3,0xFF,
  0x00,0x72,0xF8,0xFF,0x3A,0x68,0x5D,0xFF,0x48,0x59,0x00,0xFF,0x8D,0xAE,0x00,0xFF,0x8A,0xAB,0x00,0xFF,
  0x8A,0xB0,0x03,0xFF,0x8C,0xAE,0x02,0xFF,0x8C,0xAF,0x00,0xFF,0x90,0xB2,0x06,0xFF,0xB8,0xDA,0x07,0xFF,
  0x23,0x86,0x67,0xFF,0x00,0x28,0x50,0xFF,0x02,0x47,0x94,0xFF,0x02,0x51,0xB0,0xFF,0x01,0x50,0xAF,0xFF,
  0x04,0x54,0xAD,0xFF,0x04,0x53,0xB2,0xFF,0x03,0x55,0xAB,0xFF,0x03,0x54,0xAF,0xFF,0x00,0x53,0xA6,0xFF,
  0x00,0x52,0xAC,0xFF,0x01,0x53,0xA7,0xFF,0x00,0x57,0xAA,0xFF,0x00,0x54,0xAE,0xFF,0x02,0x53,0xAF,0xFF,
  0x00,0x52,0xAE,0xFF,0x01,0x4D,0xAB,0xFF,0x06,0x4D,0xA7,0xFF,0x03,0x53,0xA8,0xFF,0x03,0x56,0xB2,0xFF,
  0x01,0x53,0xB6,0xFF,0x00,0x4F,0xAE,0xFF,0x01,0x50,0xAC,0xFF,0x03,0x51,0xB3,0xFF,0x05,0x54,0xB3,0xFF,
  0x03,0x53,0xAA,0xFF,0x00,0x50,0xA9,0xFF,0x00,0x51,0xAC,0xFF,0x00,0x50,0xA7,0xFF,0x00,0x53,0xAE,0xFF,
  0x01,0x5B,0xBB,0xFF,0x04,0x55,0xAE,0xFF,0x14,0x5F,0xA3,0xFF,0x0F,0x57,0x89,0xFF,0x27,0x67,0x8D,0xFF,
  0x3F,0x5C,0x3E,0xFF,0x6B,0x81,0x03,0xFF,0x88,0xA1,0x00,0xFF,0x8C,0xAD,0x06,0xFF,0x8E,0xAB,0x01,0xFF,
  0x8D,0xAC,0x00,0xFF,0x8A,0xAE,0x02,0xFF,0x8C,0xAE,0x00,0xFF,0x90,0xAD,0x07,0xFF,0x96,0xBF,0x00,0xFF,
  0xD5,0xFF,0x13,0xFF,0x4C,0x72,0x65,0xFF,0x02,0x04,0x00,0xFF,0x00,0x2F,0x65,0xFF,0x03,0x4F,0xAF,0xFF,
  0x01,0x52,0xAB,0xFF,0x03,0x53,0xAC,0xFF,0x04,0x52,0xB6,0xFF,0x01,0x5C,0xAB,0xFF,0x0E,0x6B,0xC8,0xFF,
  0x3F,0x58,0x3A,0xFF,0x58,0x73,0x00,0xFF,0x8F,0xAC,0x02,0xFF,0x8B,0xAC,0x03,0xFF,0x8D,0xAA,0x02,0xFF,
  0x8B,0xAC,0x00,0xFF,0x8C,0xAD,0x00,0xFF,0x87,0xAC,0x07,0xFF,0xBD,0xED,0x0B,0xFF,0x71,0xB9,0x65,0xFF,
  0x0D,0x41,0x66,0xFF,0x01,0x46,0x8B,0xFF,0x00,0x55,0xB2,0xFF,0x00,0x55,0xA5,0xFF,0x02,0x54,0xA8,0xFF,
  0x02,0x50,0xA3,0xFF,0x02,0x4E,0xAC,0xFF,0x03,0x54,0xA6,0xFF,0x00,0x51,0xAC,0xFF,0x00,0x53,0xB3,0xFF,
  0x01,0x51,0xB4,0xFF,0x00,0x52,0xAF,0xFF,0x01,0x4F,0xB1,0xFF,0x07,0x51,0xB0,0xFF,0x00,0x54,0xAF,0xFF,
  0x01,0x54,0xB2,0xFF,0x06,0x55,0xB1,0xFF,0x00,0x51,0xAB,0xFF,0x04,0x56,0xBA,0xFF,0x01,0x51,0xB0,0xFF,
  0x00,0x53,0xA4,0xFF,0x00,0x52,0xA2,0xFF,0x00,0x53,0xAD,0xFF,0x00,0x4E,0xAA,0xFF,0x03,0x50,0xA8,0xFF,
  0x03,0x53,0xB2,0xFF,0x00,0x57,0xB1,0xFF,0x03,0x53,0xA8,0xFF,0x00,0x52,0xAC,0xFF,0x02,0x5B,0xB7,0xFF,
  0x0D,0x5F,0xB1,0xFF,0x0D,0x58,0x92,0xFF,0x22,0x5D,0x7D,0xFF,0x29,0x50,0x57,0xFF,0x4A,0x63,0x2A,0xFF,
  0x7A,0x99,0x00,0xFF,0x8F,0xB0,0x01,0xFF,0x8B,0xB0,0x01,0xFF,0x8C,0xAE,0x00,0xFF,0x8E,0xAF,0x00,0xFF,
  0x8D,0xAE,0x07,0xFF,0x8A,0xAB,0x00,0xFF,0x8F,0xAB,0x01,0xFF,0x8B,0xB4,0x00,0xFF,0xCD,0xFF,0x06,0xFF,
  0xAA,0xCF,0x5D,0xFF,0x00,0x07,0x07,0xFF,0x00,0x1E,0x3B,0xFF,0x01,0x4C,0x9F,0xFF,0x00,0x51,0xAF,0xFF,
  0x02,0x52,0xA7,0xFF,0x00,0x51,0xAE,0xFF,0x0D,0x59,0xA4,0xFF,0x15,0x49,0x5F,0xFF,0x43,0x57,0x10,0xFF,
  0x77,0x93,0x02,0xFF,0x8B,0xA8,0x00,0xFF,0x8B,0xB1,0x00,0xFF,0x89,0xAE,0x00,0xFF,0x89,0xB0,0x00,0xFF,
  0x88,0xAD,0x00,0xFF,0x89,0xAB,0x00,0xFF,0xA5,0xD1,0x02,0xFF,0x75,0xDF,0x3F,0xFF,0x16,0x52,0x6C,0xFF,
  0x00,0x35,0x75,0xFF,0x00,0x4F,0xAE,0xFF,0x05,0x53,0xA8,0xFF,0x06,0x53,0xAD,0xFF,0x02,0x51,0xAE,0xFF,
  0x02,0x51,0xB8,0xFF,0x01,0x55,0xAF,0xFF,0x02,0x51,0xAE,0xFF,0x04,0x50,0xAE,0xFF,0x04,0x52,0xA5,0xFF,
  0x01,0x53,0x9F,0xFF,0x04,0x4F,0xA0,0xFF,0x05,0x4E,0xAA,0xFF,0x00,0x52,0xAB,0xFF,0x00,0x53,0xA8,0xFF,
  0x05,0x52,0xAC,0xFF,0x04,0x50,0xB0,0xFF,0x00,0x58,0xB2,0xFF,0x00,0x51,0xA4,0xFF,0x00,0x49,0x8C,0xFF,
  0x02,0x44,0x80,0xFF,0x07,0x44,0x87,0xFF,0x05,0x45,0x8D,0xFF,0x00,0x49,0x96,0xFF,0x00,0x51,0xAA,0xFF,
  0x03,0x4E,0xAB,0xFF,0x04,0x51,0xA9,0xFF,0x00,0x55,0xAC,0xFF,0x03,0x5B,0xB4,0xFF,0x13,0x66,0xB2,0xFF,
  0x23,0x72,0x99,0xFF,0x35,0x70,0x6C,0xFF,0x32,0x4D,0x2E,0xFF,0x60,0x7C,0x0D,0xFF,0x85,0xAA,0x00,0xFF,
  0x89,0xAD,0x01,0xFF,0x84,0xA9,0x00,0xFF,0x8D,0xAF,0x01,0xFF,0x89,0xA9,0x00,0xFF,0x8E,0xAE,0x03,0xFF,
  0x8C,0xAB,0x07,0xFF,0x8C,0xAA,0x00,0xFF,0x8F,0xB6,0x01,0xFF,0xC4,0xF8,0x02,0xFF,0xDA,0xFF,0x34,0xFF,
  0x29,0x47,0x47,0xFF,0x02,0x1A,0x32,0xFF,0x01,0x46,0x91,0xFF,0x00,0x52,0xAD,0xFF,0x03,0x50,0xBA,0xFF,
  0x03,0x6D,0xC3,0xFF,0x36,0x63,0x8D,0xFF,0x3B,0x4D,0x23,0xFF,0x62,0x81,0x00,0xFF,0x89,0xA4,0x00,0xFF,
  0x8F,0xAC,0x02,0xFF,0x89,0xAE,0x00,0xFF,0x91,0xA7,0x00,0xFF,0x95,0xAD,0x00,0xFF,0x8F,0xAB,0x08,0xFF,
  0x92,0xAD,0x00,0xFF,0x89,0xB2,0x00,0xFF,0xB6,0xE5,0x09,0xFF,0x40,0x7A,0x51,0xFF,0x09,0x31,0x65,0xFF,
  0x00,0x4B,0x91,0xFF,0x01,0x4E,0x96,0xFF,0x00,0x4A,0x8E,0xFF,0x00,0x47,0x98,0xFF,0x00,0x46,0x8D,0xFF,
  0x00,0x39,0x7C,0xFF,0x00,0x2F,0x60,0xFF,0x02,0x28,0x57,0xFF,0x00,0x25,0x3E,0xFF,0x00,0x27,0x55,0xFF,
  0x01,0x35,0x67,0xFF,0x02,0x43,0x85,0xFF,0x04,0x52,0x9D,0xFF,0x01,0x57,0xAA,0xFF,0x00,0x50,0xB1,0xFF,
  0x00,0x53,0xAD,0xFF,0x00,0x5B,0xB7,0xFF,0x00,0x58,0xBA,0xFF,0x00,0x3F,0x87,0xFF,0x03,0x20,0x42,0xFF,
  0x00,0x0C,0x1F,0xFF,0x00,0x19,0x32,0xFF,0x00,0x1F,0x43,0xFF,0x05,0x29,0x5B,0xFF,0x00,0x39,0x7E,0xFF,
  0x02,0x4A,0x96,0xFF,0x01,0x58,0xB5,0xFF,0x06,0x5F,0xBD,0xFF,0x0E,0x62,0xAB,0xFF,0x24,0x75,0xAC,0xFF,
  0x1C,0x59,0x6C,0xFF,0x1F,0x39,0x1E,0xFF,0x53,0x69,0x00,0xFF,0x67,0x85,0x03,0xFF,0x66,0x82,0x07,0xFF,
  0x65,0x7E,0x08,0xFF,0x65,0x7D,0x01,0xFF,0x69,0x85,0x00,0xFF,0x7B,0x97,0x04,0xFF,0x83,0xA7,0x00,0xFF,
  0x87,0xA6,0x02,0xFF,0x8A,0xAC,0x00,0xFF,0xA4,0xCE,0x00,0xFF,0xD0,0xFA,0x1A,0xFF,0x52,0x82,0x68,0xFF,
  0x06,0x17,0x35,0xFF,0x00,0x40,0x89,0xFF,0x03,0x57,0xAF,0xFF,0x00,0x5B,0xB5,0xFF,0x1C,0x82,0xD6,0xFF,
  0x62,0x83,0x32,0xFF,0x7B,0x91,0x12,0xFF,0x84,0xA9,0x00,0xFF,0x8B,0xAC,0x00,0xFF,0x8C,0xAC,0x01,0xFF,
  0x8A,0xAC,0x00,0xFF,0x8C,0xAE,0x00,0xFF,0x8C,0xA5,0x00,0xFF,0x87,0xAF,0x00,0xFF,0x86,0x9F,0x05,0xFF,
  0x87,0xA0,0x00,0xFF,0x9E,0xD7,0x00,0xFF,0x54,0x81,0x24,0xFF,0x00,0x18,0x40,0xFF,0x00,0x27,0x5A,0xFF,
  0x00,0x27,0x5E,0xFF,0x00,0x25,0x53,0xFF,0x08,0x36,0x67,0xFF,0x13,0x49,0x6D,0xFF,0x16,0x48,0x6B,0xFF,
  0x21,0x4F,0x5F,0xFF,0x2D,0x52,0x58,0xFF,0x24,0x3F,0x36,0xFF,0x1A,0x32,0x3E,0xFF,0x0F,0x3B,0x3E,0xFF,
  0x00,0x2E,0x4F,0xFF,0x00,0x22,0x5F,0xFF,0x03,0x3A,0x7B,0xFF,0x01,0x4A,0xA8,0xFF,0x07,0x54,0xAE,0xFF,
  0x02,0x5A,0xB4,0xFF,0x05,0x65,0xCB,0xFF,0x06,0x53,0x97,0xFF,0x26,0x49,0x4B,0xFF,0x39,0x4C,0x2C,0xFF,
  0x41,0x63,0x4B,0xFF,0x17,0x36,0x38,0xFF,0x02,0x0D,0x1F,0xFF,0x00,0x10,0x28,0xFF,0x00,0x2B,0x52,0xFF,
  0x00,0x42,0x8D,0xFF,0x0D,0x52,0x97,0xFF,0x19,0x5A,0x76,0xFF,0x1E,0x5D,0x7E,0xFF,0x25,0x51,0x5C,0xFF,
  0x3D,0x49,0x09,0xFF,0x2B,0x32,0x00,0xFF,0x27,0x30,0x1D,0xFF,0x3C,0x4B,0x20,0xFF,0x3B,0x46,0x1B,0xFF,
  0x1F,0x27,0x10,0xFF,0x43,0x54,0x06,0xFF,0x5B,0x73,0x01,0xFF,0x71,0x93,0x00,0xFF,0x74,0x95,0x00,0xFF,
  0x7C,0x99,0x03,0xFF,0x84,0xA4,0x05,0xFF,0xBD,0xEB,0x16,0xFF,0x3D,0x74,0x4D,0xFF,0x04,0x15,0x31,0xFF,
  0x01,0x44,0x8A,0xFF,0x00,0x51,0xAC,0xFF,0x00,0x69,0xCC,0xFF,0x10,0x64,0xAD,0xFF,0x5D,0x75,0x07,0xFF,
  0x78,0x9A,0x08,0xFF,0x90,0xAB,0x00,0xFF,0x86,0xAF,0x00,0xFF,0x8C,0xAC,0x03,0xFF,0x8E,0xAC,0x00,0xFF,
  0x84,0xAE,0x04,0xFF,0x8A,0xA6,0x00,0xFF,0x79,0xAB,0x00,0xFF,0x7A,0x9C,0x00,0xFF,0x6F,0x84,0x00,0xFF,
  0x81,0x98,0x12,0xFF,0x49,0x63,0x24,0xFF,0x07,0x22,0x37,0xFF,0x18,0x43,0x53,0xFF,0x24,0x48,0x46,0xFF,
  0x30,0x4E,0x32,0xFF,0x58,0x7D,0x2D,0xFF,0x7C,0xA8,0x39,0xFF,0x87,0xBB,0x2A,0xFF,0x9F,0xD3,0x28,0xFF,
  0xAF,0xE3,0x1C,0xFF,0xA6,0xD7,0x19,0xFF,0x9C,0xC5,0x13,0xFF,0x8B,0xC2,0x1F,0xFF,0x48,0x6E,0x3D,0xFF,
  0x08,0x0E,0x24,0xFF,0x00,0x1B,0x28,0xFF,0x00,0x45,0x83,0xFF,0x03,0x52,0xB1,0xFF,0x00,0x5C,0xB1,0xFF,
  0x02,0x6C,0xD9,0xFF,0x31,0x7B,0x7E,0xFF,0x73,0x9E,0x2A,0xFF,0xAD,0xE7,0x16,0xFF,0xBA,0xF1,0x22,0xFF,
  0xA3,0xE1,0x38,0xFF,0x46,0x9E,0x64,0xFF,0x1E,0x4A,0x4B,0xFF,0x13,0x35,0x37,0xFF,0x16,0x38,0x54,0xFF,
  0x17,0x54,0x70,0xFF,0x30,0x6C,0x6D,0xFF,0x4B,0x74,0x4C,0xFF,0x4A,0x71,0x32,0xFF,0x59,0x75,0x0E,0xFF,
  0x25,0x34,0x15,0xFF,0x0D,0x2B,0x29,0xFF,0x1A,0x58,0x7F,0xFF,0x2C,0x82,0xA3,0xFF,0x2F,0x76,0x78,0xFF,
  0x32,0x52,0x2D,0xFF,0x3A,0x47,0x01,0xFF,0x37,0x47,0x08,0xFF,0x3E,0x55,0x00,0xFF,0x55,0x64,0x09,0xFF,
  0x54,0x6A,0x06,0xFF,0x69,0x8D,0x11,0xFF,0x22,0x41,0x3C,0xFF,0x01,0x34,0x61,0xFF,0x06,0x50,0xA7,0xFF,
  0x00,0x52,0xA5,0xFF,0x00,0x6A,0xE4,0xFF,0x26,0x75,0x9E,0xFF,0x36,0x46,0x07,0xFF,0x50,0x59,0x0A,0xFF,
  0x6F,0x97,0x00,0xFF,0x87,0xA3,0x01,0xFF,0x81,0xA3,0x04,0xFF,0x79,0x9E,0x00,0xFF,0x80,0xA8,0x0A,0xFF,
  0x74,0x90,0x00,0xFF,0x5E,0x73,0x00,0xFF,0x4F,0x6B,0x00,0xFF,0x42,0x68,0x1D,0xFF,0x66,0x8A,0x40,0xFF,
  0x45,0x70,0x66,0xFF,0x36,0x5D,0x3E,0xFF,0x58,0x84,0x25,0xFF,0x74,0xA8,0x44,0xFF,0x7C,0xA7,0x19,0xFF,
  0x90,0xB7,0x0C,0xFF,0x9A,0xCA,0x12,0xFF,0x92,0xBB,0x07,0xFF,0x98,0xBD,0x00,0xFF,0x9D,0xC6,0x14,0xFF,
  0x9F,0xBE,0x00,0xFF,0x98,0xBB,0x00,0xFF,0xBD,0xEE,0x06,0xFF,0xBB,0xF4,0x2B,0xFF,0x64,0x8A,0x4D,0xFF,
  0x0A,0x2F,0x49,0xFF,0x00,0x35,0x81,0xFF,0x03,0x53,0xAA,0xFF,0x00,0x62,0xC7,0xFF,0x03,0x73,0xF3,0xFF,
  0x42,0x74,0x57,0xFF,0x52,0x71,0x00,0xFF,0x8F,0xB8,0x02,0xFF,0xAE,0xD8,0x00,0xFF,0xC0,0xED,0x00,0xFF,
  0xB6,0xFA,0x29,0xFF,0x98,0xE1,0x32,0xFF,0x88,0xBC,0x34,0xFF,0x79,0xA3,0x33,0xFF,0x83,0xBD,0x33,0xFF,
  0x8B,0xC5,0x33,0xFF,0x94,0xC2,0x2B,0xFF,0x99,0xD4,0x20,0xFF,0x67,0x9A,0x35,0xFF,0x42,0x72,0x62,0xFF,
  0x0C,0x41,0x6D,0xFF,0x00,0x4F,0xAB,0xFF,0x02,0x78,0xE4,0xFF,0x0C,0x8B,0xF6,0xFF,0x2D,0x91,0xCC,0xFF,
  0x4D,0x92,0x80,0xFF,0x48,0x78,0x46,0xFF,0x48,0x6D,0x39,0xFF,0x5A,0x7C,0x26,0xFF,0x4D,0x75,0x2C,0xFF,
  0x17,0x41,0x4D,0xFF,0x00,0x32,0x5C,0xFF,0x00,0x44,0x99,0xFF,0x00,0x53,0xB1,0xFF,0x00,0x51,0xB5,0xFF,
  0x00,0x6D,0xE0,0xFF,0x0C,0x8B,0xF4,0xFF,0x2D,0x6F,0x63,0xFF,0x3D,0x49,0x0B,0xFF,0x41,0x4F,0x00,0xFF,
  0x48,0x52,0x09,0xFF,0x52,0x67,0x00,0xFF,0x60,0x72,0x02,0xFF,0x5C,0x71,0x00,0xFF,0x54,0x72,0x1C,0xFF,
  0x56,0x83,0x40,0xFF,0x58,0x8F,0x7C,0xFF,0x2B,0x81,0x9C,0xFF,0x21,0x8B,0xE3,0xFF,0x2B,0xA1,0xD3,0xFF,
  0x43,0x8E,0x77,0xFF,0x4F,0x73,0x29,0xFF,0x69,0x8B,0x12,0xFF,0x7D,0x9A,0x02,0xFF,0x8C,0xAC,0x0B,0xFF,
  0x8B,0xB0,0x00,0xFF,0x94,0xB7,0x01,0xFF,0x91,0xB5,0x0B,0xFF,0x91,0xB7,0x00,0xFF,0x93,0xB3,0x00,0xFF,
  0x97,0xBB,0x01,0xFF,0x9B,0xBE,0x00,0xFF,0xBA,0xE0,0x01,0xFF,0x8A,0xB0,0x0B,0xFF,0x12,0x3A,0x46,0xFF,
  0x00,0x39,0x7C,0xFF,0x03,0x52,0xAF,0xFF,0x00,0x64,0xC2,0xFF,0x00,0x70,0xF0,0xFF,0x34,0x4F,0x40,0xFF,
  0x20,0x24,0x01,0xFF,0x75,0x91,0x08,0xFF,0x91,0xB5,0x00,0xFF,0x9C,0xBA,0x04,0xFF,0xA0,0xCC,0x00,0xFF,
  0xAC,0xDB,0x01,0xFF,0xAB,0xD1,0x08,0xFF,0xAB,0xD5,0x09,0xFF,0xA4,0xD0,0x0B,0xFF,0xA2,0xCE,0x09,0xFF,
  0xAB,0xD6,0x00,0xFF,0xC6,0xF8,0x00,0xFF,0x76,0xA6,0x43,0xFF,0x12,0x59,0x79,0xFF,0x0C,0x50,0x91,0xFF,
  0x03,0x4F,0xA3,0xFF,0x00,0x59,0xB3,0xFF,0x00,0x6F,0xD3,0xFF,0x00,0x76,0xF0,0xFF,0x00,0x77,0xFE,0xFF,
  0x05,0x81,0xFB,0xFF,0x11,0x79,0xD2,0xFF,0x2A,0x8C,0xE5,0xFF,0x2F,0x8F,0xE5,0xFF,0x0C,0x62,0xC7,0xFF,
  0x00,0x4C,0x84,0xFF,0x00,0x4B,0x91,0xFF,0x00,0x53,0x9E,0xFF,0x00,0x55,0xAF,0xFF,0x07,0x5B,0xBB,0xFF,
  0x00,0x72,0xEB,0xFF,0x0A,0x82,0xFC,0xFF,0x01,0x74,0xCF,0xFF,0x2B,0x7D,0x79,0xFF,0x3D,0x62,0x2F,0xFF,
  0x41,0x60,0x25,0xFF,0x4A,0x6E,0x26,0xFF,0x3D,0x71,0x4B,0xFF,0x15,0x5B,0xA1,0xFF,0x00,0x72,0xD9,0xFF,
  0x00,0x78,0xF0,0xFF,0x00,0x70,0xDC,0xFF,0x00,0x6F,0xEF,0xFF,0x08,0x86,0xFF,0xFF,0x1F,0x7F,0xB9,0xFF,
  0x4F,0x73,0x35,0xFF,0x68,0x80,0x00,0xFF,0x92,0xAF,0x07,0xFF,0x87,0xA8,0x00,0xFF,0x8E,0xA8,0x06,0xFF,
  0x90,0xAE,0x00,0xFF,0x8A,0xAE,0x04,0xFF,0x8C,0xAB,0x00,0xFF,0x8D,0xAF,0x00,0xFF,0x85,0xA8,0x04,0xFF,
  0x8F,0xB4,0x02,0xFF,0x7A,0x9A,0x08,0xFF,0x38,0x5C,0x2C,0xFF,0x00,0x35,0x49,0xFF,0x00,0x47,0x88,0xFF,
  0x02,0x50,0xB2,0xFF,0x00,0x5B,0xBA,0xFF,0x04,0x76,0xFD,0xFF,0x33,0x62,0x5C,0xFF,0x0C,0x0E,0x00,0xFF,
  0x5C,0x74,0x04,0xFF,0x83,0xAC,0x00,0xFF,0x8B,0xB2,0x01,0xFF,0x8F,0xB6,0x05,0xFF,0x9E,0xB0,0x04,0xFF,
  0x96,0xB0,0x00,0xFF,0x95,0xB7,0x01,0xFF,0x96,0xB6,0x00,0xFF,0x90,0xB3,0x00,0xFF,0x9D,0xBF,0x08,0xFF,
  0xC5,0xEB,0x04,0xFF,0x4D,0x76,0x30,0xFF,0x0B,0x35,0x4B,0xFF,0x05,0x3F,0x89,0xFF,0x00,0x4F,0xA4,0xFF,
  0x00,0x53,0xA6,0xFF,0x01,0x55,0xAF,0xFF,0x01,0x56,0xB0,0xFF,0x02,0x56,0xB8,0xFF,0x06,0x5A,0xBA,0xFF,
  0x00,0x6D,0xD4,0xFF,0x00,0x6B,0xE8,0xFF,0x00,0x69,0xE4,0xFF,0x00,0x71,0xDB,0xFF,0x00,0x54,0xBB,0xFF,
  0x00,0x4B,0xA8,0xFF,0x02,0x52,0xA9,0xFF,0x01,0x4F,0xB1,0xFF,0x00,0x4F,0xA8,0xFF,0x00,0x5B,0xBB,0xFF,
  0x07,0x62,0xCD,0xFF,0x02,0x74,0xF1,0xFF,0x02,0x87,0xFD,0xFF,0x24,0x92,0xF3,0xFF,0x1E,0x7A,0xCD,0xFF,
  0x22,0x7D,0xB6,0xFF,0x0A,0x6D,0xCA,0xFF,0x04,0x55,0xB1,0xFF,0x00,0x54,0xBB,0xFF,0x00,0x50,0xAA,0xFF,
  0x05,0x50,0xB5,0xFF,0x06,0x56,0xB5,0xFF,0x00,0x69,0xDF,0xFF,0x09,0x61,0xB8,0xFF,0x44,0x62,0x2E,0xFF,
  0x73,0x93,0x01,0xFF,0x88,0xB4,0x00,0xFF,0x8D,0xB6,0x02,0xFF,0x8E,0xA9,0x04,0xFF,0x8A,0xAA,0x01,0xFF,
  0x8C,0xAD,0x00,0xFF,0x8F,0xAB,0x00,0xFF,0x85,0xAF,0x05,0xFF,0x93,0xB1,0x03,0xFF,0x86,0xA7,0x00,0xFF,
  0x68,0x87,0x2A,0xFF,0x2A,0x54,0x62,0xFF,0x02,0x33,0x76,0xFF,0x0A,0x51,0xA9,0xFF,0x00,0x52,0xA9,0xFF,
  0x00,0x5B,0xB5,0xFF,0x00,0x7C,0xF6,0xFF,0x2C,0x8C,0xB2,0xFF,0x17,0x26,0x05,0xFF,0x41,0x51,0x00,0xFF,
  0x86,0xA7,0x00,0xFF,0x8A,0xAB,0x00,0xFF,0x8A,0xAA,0x01,0xFF,0x8C,0xAE,0x00,0xFF,0x8B,0xB0,0x00,0xFF,
  0x8A,0xB0,0x05,0xFF,0x8E,0xA9,0x00,0xFF,0x8F,0xAD,0x00,0xFF,0x98,0xC1,0x03,0xFF,0x92,0xBC,0x28,0xFF,
  0x2D,0x5F,0x66,0xFF,0x05,0x31,0x4E,0xFF,0x00,0x37,0x7D,0xFF,0x00,0x4B,0x99,0xFF,0x07,0x55,0xAA,0xFF,
  0x00,0x4D,0xAA,0xFF,0x00,0x51,0xA6,0xFF,0x00,0x55,0xAF,0xFF,0x00,0x4F,0xAB,0xFF,0x02,0x4F,0xAB,0xFF,
  0x01,0x4E,0xAA,0xFF,0x00,0x58,0xA7,0xFF,0x00,0x5C,0xB0,0xFF,0x03,0x58,0xB5,0xFF,0x01,0x55,0xB8,0xFF,
  0x00,0x54,0xAD,0xFF,0x00,0x51,0xAC,0xFF,0x01,0x52,0xAE,0xFF,0x00,0x57,0xAF,0xFF,0x07,0x51,0xB0,0xFF,
  0x00,0x4E,0xB3,0xFF,0x00,0x60,0xC6,0xFF,0x03,0x74,0xD4,0xFF,0x00,0x71,0xDB,0xFF,0x00,0x6F,0xCF,0xFF,
  0x01,0x56,0xB0,0xFF,0x06,0x53,0xA3,0xFF,0x00,0x51,0xAC,0xFF,0x03,0x52,0xAF,0xFF,0x07,0x51,0xA6,0xFF,
  0x00,0x55,0xA5,0xFF,0x00,0x73,0xDD,0xFF,0x11,0x65,0xAE,0xFF,0x31,0x40,0x19,0xFF,0x63,0x7D,0x00,0xFF,
  0x88,0xBA,0x01,0xFF,0x87,0xAD,0x00,0xFF,0x8E,0xAD,0x00,0xFF,0x8B,0xB0,0x01,0xFF,0x8A,0xA7,0x00,0xFF,
  0x91,0xAF,0x01,0xFF,0x8E,0xB4,0x00,0xFF,0xA7,0xD1,0x03,0xFF,0x6B,0xA1,0x31,0xFF,0x25,0x62,0x5D,0xFF,
  0x06,0x56,0xAD,0xFF,0x06,0x54,0xA9,0xFF,0x01,0x55,0xAD,0xFF,0x01,0x51,0xAA,0xFF,0x01,0x56,0xB3,0xFF,
  0x00,0x73,0xFE,0xFF,0x0C,0x86,0xFF,0xFF,0x24,0x43,0x31,0xFF,0x22,0x23,0x01,0xFF,0x78,0x8D,0x00,0xFF,
  0x91,0xAC,0x00,0xFF,0x90,0xAD,0x05,0xFF,0x89,0xB1,0x01,0xFF,0x82,0xA9,0x00,0xFF,0x89,0xB0,0x00,0xFF,
  0x8F,0xA9,0x08,0xFF,0x90,0xAD,0x00,0xFF,0xA8,0xD8,0x08,0xFF,0x6E,0x97,0x5D,0xFF,0x16,0x4B,0x75,0xFF,
  0x02,0x38,0x74,0xFF,0x00,0x44,0x8D,0xFF,0x00,0x48,0x9E,0xFF,0x04,0x4C,0xB0,0xFF,0x09,0x52,0xB1,0xFF,
  0x03,0x55,0xAB,0xFF,0x00,0x51,0xAB,0xFF,0x02,0x52,0xAB,0xFF,0x04,0x56,0xAC,0xFF,0x04,0x53,0xB2,0xFF,
  0x00,0x53,0xAF,0xFF,0x00,0x4F,0xA9,0xFF,0x05,0x50,0xAD,0xFF,0x06,0x51,0xB6,0xFF,0x00,0x52,0xAC,0xFF,
  0x00,0x52,0xA2,0xFF,0x00,0x55,0xA6,0xFF,0x04,0x54,0xA9,0xFF,0x00,0x52,0xA5,0xFF,0x00,0x57,0xB0,0xFF,
  0x01,0x4F,0xB4,0xFF,0x02,0x52,0xA9,0xFF,0x00,0x51,0xA3,0xFF,0x07,0x4F,0xB1,0xFF,0x00,0x55,0xAD,0xFF,
  0x05,0x55,0xB6,0xFF,0x00,0x51,0xAC,0xFF,0x01,0x50,0xAF,0xFF,0x05,0x50,0xAB,0xFF,0x00,0x52,0xB3,0xFF,
  0x00,0x6B,0xE8,0xFF,0x1D,0x7B,0xB8,0xFF,0x1B,0x23,0x0E,0xFF,0x4B,0x54,0x03,0xFF,0x85,0xAE,0x00,0xFF,
  0x8F,0xAF,0x02,0xFF,0x8E,0xAB,0x00,0xFF,0x89,0xB0,0x00,0xFF,0x8F,0xAA,0x03,0xFF,0x89,0xAB,0x00,0xFF,
  0x95,0xB3,0x03,0xFF,0xB1,0xE6,0x00,0xFF,0x4A,0x79,0x73,0xFF,0x0E,0x41,0x6C,0xFF,0x03,0x53,0xAA,0xFF,
  0x00,0x51,0xB1,0xFF,0x00,0x56,0xA9,0xFF,0x06,0x4F,0xAD,0xFF,0x05,0x52,0xAE,0xFF,0x02,0x6C,0xD0,0xFF,
  0x00,0x78,0xFB,0xFF,0x39,0x6B,0x6A,0xFF,0x1C,0x20,0x05,0xFF,0x66,0x7B,0x02,0xFF,0x90,0xAA,0x09,0xFF,
  0x85,0xAC,0x00,0xFF,0x8B,0xA7,0x00,0xFF,0x95,0xB3,0x05,0xFF,0x89,0xA8,0x02,0xFF,0x8C,0xAF,0x00,0xFF,
  0x91,0xB8,0x00,0xFF,0x91,0xBC,0x24,0xFF,0x45,0x71,0x62,0xFF,0x0A,0x39,0x4B,0xFF,0x00,0x3B,0x79,0xFF,
  0x00,0x4B,0x9F,0xFF,0x00,0x52,0xAB,0xFF,0x00,0x59,0xAC,0xFF,0x00,0x53,0xB0,0xFF,0x00,0x52,0xAC,0xFF,
  0x00,0x53,0xA4,0xFF,0x00,0x52,0xB8,0xFF,0x02,0x4C,0xAD,0xFF,0x05,0x50,0xAB,0xFF,0x00,0x51,0xA8,0xFF,
  0x00,0x5B,0xAC,0xFF,0x04,0x4F,0xB5,0xFF,0x04,0x4D,0xAB,0xFF,0x04,0x50,0xAE,0xFF,0x08,0x51,0xB0,0xFF,
  0x05,0x4E,0xAD,0xFF,0x00,0x51,0xB1,0xFF,0x00,0x51,0xAF,0xFF,0x02,0x4C,0xA5,0xFF,0x07,0x56,0xA5,0xFF,
  0x00,0x50,0xA5,0xFF,0x06,0x56,0xAD,0xFF,0x00,0x52,0xAC,0xFF,0x00,0x50,0xB1,0xFF,0x04,0x54,0xA9,0xFF,
  0x01,0x54,0xB2,0xFF,0x00,0x52,0xAD,0xFF,0x03,0x52,0xAF,0xFF,0x07,0x53,0xA9,0xFF,0x00,0x68,0xD7,0xFF,
  0x15,0x8E,0xF9,0xFF,0x19,0x3F,0x42,0xFF,0x31,0x31,0x00,0xFF,0x84,0x9F,0x04,0xFF,0x87,0xA8,0x00,0xFF,
  0x92,0xAB,0x07,0xFF,0x89,0xAC,0x00,0xFF,0x91,0xAE,0x00,0xFF,0x88,0xAC,0x02,0xFF,0x9A,0xB9,0x03,0xFF,
  0xA4,0xD2,0x27,0xFF,0x1B,0x53,0x62,0xFF,0x00,0x34,0x67,0xFF,0x00,0x5A,0xA0,0xFF,0x00,0x52,0xAF,0xFF,
  0x00,0x57,0xA7,0xFF,0x05,0x4D,0xB9,0xFF,0x00,0x50,0xAC,0xFF,0x00,0x66,0xCB,0xFF,0x08,0x86,0xF4,0xFF,
  0x48,0x8E,0x96,0xFF,0x1D,0x29,0x13,0xFF,0x55,0x71,0x02,0xFF,0x8C,0xAA,0x00,0xFF,0x8C,0xB0,0x06,0xFF,
  0x8F,0xAD,0x00,0xFF,0x89,0xA9,0x00,0xFF,0x8F,0xAD,0x00,0xFF,0x8A,0xB6,0x00,0xFF,0x94,0xBB,0x24,0xFF,
  0x7B,0xA7,0x48,0xFF,0x2B,0x65,0x67,0xFF,0x11,0x42,0x60,0xFF,0x00,0x42,0x8A,0xFF,0x09,0x52,0xA1,0xFF,
  0x03,0x4D,0xAC,0xFF,0x01,0x51,0xA8,0xFF,0x01,0x52,0xAD,0xFF,0x03,0x53,0xB6,0xFF,0x00,0x52,0xA6,0xFF,
  0x02,0x52,0xAB,0xFF,0x00,0x57,0xAF,0xFF,0x04,0x54,0xA9,0xFF,0x04,0x51,0xAD,0xFF,0x00,0x4E,0xAB,0xFF,
  0x03,0x53,0xAA,0xFF,0x00,0x57,0xA9,0xFF,0x00,0x54,0xAC,0xFF,0x00,0x51,0xAA,0xFF,0x00,0x51,0xAF,0xFF,
  0x08,0x53,0xAE,0xFF,0x00,0x55,0xA8,0xFF,0x00,0x56,0xB0,0xFF,0x00,0x53,0xAE,0xFF,0x00,0x4E,0xB9,0xFF,
  0x00,0x50,0xAF,0xFF,0x01,0x53,0xA9,0xFF,0x03,0x54,0xA6,0xFF,0x06,0x4E,0xB4,0xFF,0x00,0x4F,0xA4,0xFF,
  0x01,0x56,0xB1,0xFF,0x00,0x53,0xAE,0xFF,0x03,0x4F,0xA5,0xFF,0x01,0x6D,0xDC,0xFF,0x07,0x82,0xF9,0xFF,
  0x2F,0x7F,0x98,0xFF,0x3B,0x46,0x02,0xFF,0x6E,0x83,0x00,0xFF,0x88,0xB0,0x02,0xFF,0x8D,0xA8,0x01,0xFF,
  0x8E,0xAE,0x05,0xFF,0x8F,0xAB,0x00,0xFF,0x8C,0xAE,0x02,0xFF,0xAD,0xDA,0x03,0xFF,0x94,0xB9,0x1C,0xFF,
  0x17,0x3C,0x73,0xFF,0x05,0x34,0x7A,0xFF,0x03,0x52,0xA1,0xFF,0x0B,0x4C,0xB0,0xFF,0x00,0x54,0xAE,0xFF,
  0x00,0x53,0xA6,0xFF,

};

}

void make_gb_poly(agg::path_storage& ps);
void make_arrows(agg::path_storage& ps);

//==========================================================the_application
class the_application : public agg::platform_support
{
	typedef agg::gradient_lut<agg::color_interpolator<color_type>, 1024> color_func_type;

    agg::rbox_ctrl<color_type> m_polygons;
    agg::rbox_ctrl<color_type> m_gradient;

	agg::cbox_ctrl<color_type>   m_stroke;
	agg::cbox_ctrl<color_type>   m_refl;

	agg::slider_ctrl<color_type> m_c1;
	agg::slider_ctrl<color_type> m_c2;
	agg::slider_ctrl<color_type> m_d1;
	agg::slider_ctrl<color_type> m_d2;
	agg::slider_ctrl<color_type> m_clrs;

	agg::polygon_ctrl<color_type> m_persp;

    agg::rasterizer_scanline_aa<>   m_ras;
    agg::scanline_p8                m_sl;
	agg::span_allocator<color_type> m_alloc;

	double m_gamma;
	bool   m_init;
	int    m_last;

	color_func_type m_colors02;
	color_func_type m_colors03;
	color_func_type m_colors04;
	color_func_type m_colors05;
	color_func_type m_colors06;
	color_func_type m_colors07;
	color_func_type m_colors08;
	color_func_type m_colors09;
	color_func_type m_colors10;
	color_func_type m_colors11;

public:
    ~the_application()
    {
    }
    
    the_application(agg::pix_format_e format, bool flip_y) :
        agg::platform_support(format, flip_y),
        m_polygons(5.0,     5.0, 5.0+130.0,  90.0,  !flip_y),
        m_gradient(145.0,   5.0, 220.0+80.0, 90.0,  !flip_y),
		m_stroke(305.0 ,77.0 ,"Bitmap Gradient" , !flip_y ),
		m_refl(440.0 ,77.0 ,"Reflect" ,!flip_y ),
		m_c1(310.0 ,10.0 ,400.0 ,16.0 ,!flip_y ),
		m_c2(310.0 ,30.0 ,400.0 ,36.0 ,!flip_y ),
		m_d1(410.0 ,10.0 ,500.0 ,16.0 ,!flip_y ),
		m_d2(410.0 ,30.0 ,500.0 ,36.0 ,!flip_y ),
		m_clrs(310.0 ,50.0 ,500.0 ,58.0 ,!flip_y ),
		m_persp(4 )
    {
        m_init = true;
		
       // Controls
        m_gradient.add_item("Contour");
		m_gradient.add_item("Auto Contour");
		m_gradient.add_item("Assymetric Conic");
		m_gradient.add_item("Flat Fill");
        m_gradient.cur_item(1);
        add_ctrl(m_gradient);
        m_gradient.no_transform();

        m_polygons.add_item("Simple Path");
        m_polygons.add_item("Great Britain");
        m_polygons.add_item("Spiral");
        m_polygons.add_item("Glyph");
        m_polygons.cur_item(0);
        add_ctrl(m_polygons);
        m_polygons.no_transform();

		m_last = m_polygons.cur_item();

        add_ctrl(m_stroke );
        m_stroke.status(false );
        m_stroke.no_transform();

        add_ctrl(m_refl );
        m_refl.status(true );
        m_refl.no_transform();

		add_ctrl(m_c1 );

		m_c1.label("C1=%1.0f" );
		m_c1.range(0.0 ,512.0 );
		m_c1.value(0.0 );
		m_c1.no_transform();

		add_ctrl(m_c2 );

		m_c2.label("C2=%1.0f" );
		m_c2.range(0.0 ,512.0 );
		m_c2.value(512.0 );
		m_c2.no_transform();

		add_ctrl(m_d1 );

		m_d1.label("D1=%1.0f" );
		m_d1.range(0.0 ,512.0 );
		m_d1.value(0.0 );
		m_d1.no_transform();

		add_ctrl(m_d2 );

		m_d2.label("D2=%1.0f" );
		m_d2.range(0.1 ,512.0 );
		m_d2.value(100.0 );
		m_d2.no_transform();

		add_ctrl(m_clrs );

		m_clrs.label("Colors=%1.0f" );
		m_clrs.range(2.0 ,11.0 );
		m_clrs.value(2.0 );
		m_clrs.num_steps(9 );
		m_clrs.no_transform();

		add_ctrl(m_persp );

		m_persp.in_polygon_check(true );
		m_persp.no_transform();
		m_persp.line_color(agg::rgba(0, 0.3, 0.5, 0.3));

	   // 2 colors gradient
        m_colors02.remove_all();
		m_colors02.add_color(0.0, agg::srgba8(178 ,34 ,34 ) );
		m_colors02.add_color(1.0, agg::srgba8(255 ,255 ,0 ) );
		m_colors02.build_lut();

	   // 3 colors gradient
        m_colors03.remove_all();
		m_colors03.add_color(0.0, agg::srgba8(245 ,233 ,131 ) );
		m_colors03.add_color(0.5, agg::srgba8(146 ,35 ,219 ) );
		m_colors03.add_color(1.0, agg::srgba8(255 ,35 ,0 ) );
		m_colors03.build_lut();

	   // 4 colors gradient
        m_colors04.remove_all();
		m_colors04.add_color(1.0, agg::srgba8(0 ,255 ,0 ) );
		m_colors04.add_color(0.7, agg::srgba8(120 ,0 ,0 ) );
		m_colors04.add_color(0.2, agg::srgba8(120 ,120 ,0 ) );
		m_colors04.add_color(0.0, agg::srgba8(0 ,0 ,255 ) );
		m_colors04.build_lut();

	   // 5 colors gradient
        m_colors05.remove_all();
		m_colors05.add_color(0.2, agg::srgba8(230 ,188 ,106 ) );
		m_colors05.add_color(0.4, agg::srgba8(207 ,148 ,31 ) );
		m_colors05.add_color(0.6, agg::srgba8(69 ,56 ,30 ) );
		m_colors05.add_color(0.8, agg::srgba8(43 ,33 ,13 ) );
		m_colors05.add_color(1.0, agg::srgba8(227 ,221 ,209 ) );
		m_colors05.build_lut();

	   // 6 colors gradient
        m_colors06.remove_all();
		m_colors06.add_color(0.0, agg::srgba8(125 ,99 ,255 ) );
		m_colors06.add_color(0.2, agg::srgba8(118 ,79 ,210 ) );
		m_colors06.add_color(0.4, agg::srgba8(105 ,58 ,81 ) );
		m_colors06.add_color(0.6, agg::srgba8(217 ,74 ,102 ) );
		m_colors06.add_color(0.8, agg::srgba8(242 ,148 ,90 ) );
		m_colors06.add_color(1.0, agg::srgba8(242 ,200 ,102 ) );
		m_colors06.build_lut();

	   // 7 colors gradient
        m_colors07.remove_all();
		m_colors07.add_color(0.00, agg::srgba8(216 ,237 ,232 ) );
		m_colors07.add_color(0.16, agg::srgba8(196 ,214 ,226 ) );
		m_colors07.add_color(0.32, agg::srgba8(175 ,194 ,217 ) );
		m_colors07.add_color(0.48, agg::srgba8(155 ,176 ,210 ) );
		m_colors07.add_color(0.64, agg::srgba8(140 ,162 ,202 ) );
		m_colors07.add_color(0.80, agg::srgba8(130 ,149 ,193 ) );
		m_colors07.add_color(1.00, agg::srgba8(72 ,102 ,165 ) );
		m_colors07.build_lut();

	   // 8 colors gradient
        m_colors08.remove_all();
		m_colors08.add_color(0.00, agg::srgba8(255 ,223 ,168 ) );
		m_colors08.add_color(0.14, agg::srgba8(255 ,199 ,162 ) );
		m_colors08.add_color(0.28, agg::srgba8(255 ,175 ,156 ) );
		m_colors08.add_color(0.42, agg::srgba8(255 ,151 ,151 ) );
		m_colors08.add_color(0.56, agg::srgba8(255 ,127 ,145 ) );
		m_colors08.add_color(0.70, agg::srgba8(255 ,104 ,140 ) );
		m_colors08.add_color(0.84, agg::srgba8(255 ,80 ,133 ) );
		m_colors08.add_color(1.00, agg::srgba8(255 ,56 ,128 ) );
		m_colors08.build_lut();

	   // 9 colors gradient
        m_colors09.remove_all();
		m_colors09.add_color(0.000, agg::srgba8(255 ,4 ,163 ) );
		m_colors09.add_color(0.125, agg::srgba8(255 ,4 ,109 ) );
		m_colors09.add_color(0.250, agg::srgba8(255 ,4 ,46 ) );
		m_colors09.add_color(0.375, agg::srgba8(255 ,75 ,75 ) );
		m_colors09.add_color(0.500, agg::srgba8(255 ,120 ,83 ) );
		m_colors09.add_color(0.625, agg::srgba8(255 ,143 ,83 ) );
		m_colors09.add_color(0.750, agg::srgba8(255 ,180 ,83 ) );
		m_colors09.add_color(0.875, agg::srgba8(255 ,209 ,83 ) );
		m_colors09.add_color(1.000, agg::srgba8(255 ,246 ,83 ) );
		m_colors09.build_lut();

	   // 10 colors gradient
        m_colors10.remove_all();
		m_colors10.add_color(0.00, agg::srgba8(255 ,0 ,0 ) );
		m_colors10.add_color(0.11, agg::srgba8(255 ,198 ,198 ) );
		m_colors10.add_color(0.22, agg::srgba8(255 ,255 ,0 ) );
		m_colors10.add_color(0.33, agg::srgba8(255 ,255 ,226 ) );
		m_colors10.add_color(0.44, agg::srgba8(85 ,85 ,255 ) );
		m_colors10.add_color(0.55, agg::srgba8(226 ,226 ,255 ) );
		m_colors10.add_color(0.66, agg::srgba8(28 ,255 ,28 ) );
		m_colors10.add_color(0.77, agg::srgba8(226 ,255 ,226 ) );
		m_colors10.add_color(0.88, agg::srgba8(255 ,72 ,255 ) );
		m_colors10.add_color(1.00, agg::srgba8(255 ,227 ,255 ) );
		m_colors10.build_lut();

	   // 11 colors gradient
        m_gamma = 1.8;

        m_colors11.remove_all();
		m_colors11.add_color(0.0, agg::srgba8::from_wavelength(380, m_gamma ) );
		m_colors11.add_color(0.1, agg::srgba8::from_wavelength(420, m_gamma ) );
		m_colors11.add_color(0.2, agg::srgba8::from_wavelength(460, m_gamma ) );
		m_colors11.add_color(0.3, agg::srgba8::from_wavelength(500, m_gamma ) );
		m_colors11.add_color(0.4, agg::srgba8::from_wavelength(540, m_gamma ) );
		m_colors11.add_color(0.5, agg::srgba8::from_wavelength(580, m_gamma ) );
		m_colors11.add_color(0.6, agg::srgba8::from_wavelength(620, m_gamma ) );
		m_colors11.add_color(0.7, agg::srgba8::from_wavelength(660, m_gamma ) );
		m_colors11.add_color(0.8, agg::srgba8::from_wavelength(700, m_gamma ) );
		m_colors11.add_color(0.9, agg::srgba8::from_wavelength(740, m_gamma ) );
		m_colors11.add_color(1.0, agg::srgba8::from_wavelength(780, m_gamma ) );
		m_colors11.build_lut();

    }


    void draw_text(double x, double y, const char* str)
    {
        pixfmt pf(rbuf_window());
        agg::renderer_base<pixfmt> rb(pf);
        agg::renderer_scanline_aa_solid<agg::renderer_base<pixfmt> > ren(rb);

        agg::gsv_text txt;
        agg::conv_stroke<agg::gsv_text> txt_stroke(txt);
        txt_stroke.width(1.5);
        txt_stroke.line_cap(agg::round_cap);
        txt.size(10.0);
        txt.start_point(x, y);
        txt.text(str);
        m_ras.add_path(txt_stroke);
        ren.color(agg::rgba(0.0, 0.0, 0.0));
        agg::render_scanlines(m_ras, m_sl, ren);
    }

    template<class VertexSource>
	void perform_rendering(VertexSource& vs, agg::path_storage& contour)
    {
		typedef agg::renderer_base<pixfmt> ren_base;
		
		pixfmt pf(rbuf_window());
		ren_base rb(pf);
		agg::renderer_scanline_aa_solid<agg::renderer_base<pixfmt> > rs(rb);

		double x1,y1,x2,y2;

		if (agg::bounding_rect_single(vs ,0 ,&x1 ,&y1 ,&x2 ,&y2 ))
		{
		   // Init Basic Transformations
			double scale = (width() - 120 ) / (x2 - x1 );

			if (scale > (height() - 120 ) / (y2 - y1 ) )
			{
				scale = (height() - 120 ) / (y2 - y1 );
			}

			agg::trans_affine mtx;
			mtx *= agg::trans_affine_translation(-x1, -y1);
			mtx *= agg::trans_affine_scaling(scale, scale);

			agg::conv_transform<VertexSource> t1(vs, mtx);

			agg::trans_affine_translation tat(100, 105);

		   // Init perspective control
			double px1 = x1 ,py1 = y1 ,px2 = x2 ,py2 = y2;

			mtx.transform(&px1 ,&py1 );
			mtx.transform(&px2 ,&py2 );

			if (m_init)
			{
				m_persp.xn(0) = px1;
				m_persp.yn(0) = py1;
				m_persp.xn(1) = px2;
				m_persp.yn(1) = py1;
				m_persp.xn(2) = px2;
				m_persp.yn(2) = py2;
				m_persp.xn(3) = px1;
				m_persp.yn(3) = py2;

				tat.transform(&m_persp.xn(0) ,&m_persp.yn(0) );
				tat.transform(&m_persp.xn(1) ,&m_persp.yn(1) );
				tat.transform(&m_persp.xn(2) ,&m_persp.yn(2) );
				tat.transform(&m_persp.xn(3) ,&m_persp.yn(3) );

				m_init = false;

			}

		   // Add Perspective Transformation
			agg::trans_perspective trpp(px1, py1, px2, py2, m_persp.polygon());
			agg::trans_perspective trpg(m_persp.polygon(), px1, py1, px2, py2);

            if (!trpp.is_valid())
			{
				return;
			}

			if (!trpg.is_valid())
			{
				return;
			}

			typedef agg::conv_transform<agg::conv_transform<VertexSource>,agg::trans_perspective> t2_type;

			t2_type t2(t1, trpp);

		   // Create Path
			agg::path_storage path;
			path.concat_path(t1 );

		   // Color function
			color_func_type* cfunction = NULL;
	
			if ((int)m_clrs.value() == 2 )
			{
					cfunction = &m_colors02;
			}

			if ((int)m_clrs.value() == 3 )
			{
					cfunction = &m_colors03;
			}

			if ((int)m_clrs.value() == 4 )
			{
					cfunction = &m_colors04;
			}

			if ((int)m_clrs.value() == 5 )
			{
					cfunction = &m_colors05;
			}

			if ((int)m_clrs.value() == 6 )
			{
					cfunction = &m_colors06;
			}

			if ((int)m_clrs.value() == 7 )
			{
					cfunction = &m_colors07;
			}

			if ((int)m_clrs.value() == 8 )
			{
					cfunction = &m_colors08;
			}

			if ((int)m_clrs.value() == 9 )
			{
					cfunction = &m_colors09;
			}

			if ((int)m_clrs.value() == 10 )
			{
					cfunction = &m_colors10;
			}

			if (!cfunction)
			{
					cfunction = &m_colors11;
			}

		   // Render 
			double d1 = 0;
			double d2 = 100;

			switch(m_gradient.cur_item())
			{
			   // contour {..}
			case 0:
			case 1:
				{
					agg::gradient_contour gradient_func;

					gradient_func.frame(0 );
					gradient_func.d1(m_c1.value());
					gradient_func.d2(m_c2.value());

					switch(m_gradient.cur_item())
					{
					case 0:
						if (!gradient_func.contour_create(&contour ))
						{
							return;
						}
						break;
					case 1:
						if (!gradient_func.contour_create(&path ))
						{
							return;
						}
						break;

					}

					d1 = m_d1.value();
					d2 = m_d2.value();

					agg::span_interpolator_trans<agg::trans_perspective> span_interpolator(trpg );

					if (m_refl.status())
					{
						agg::gradient_reflect_adaptor<agg::gradient_contour> func_reflect(gradient_func);

						typedef agg::span_gradient<color_type, 
                               agg::span_interpolator_trans<agg::trans_perspective>, 
							   agg::gradient_reflect_adaptor<agg::gradient_contour>, 
                               color_func_type> span_gradient_type;

						span_gradient_type span_gradient(
						       span_interpolator, 
                               func_reflect, 
                               *cfunction, 
                               d1, d2);

						m_ras.reset();
						m_ras.add_path(t2 );

						agg::render_scanlines_aa(m_ras, m_sl, rb, m_alloc, span_gradient);

					}
					else
					{
						typedef agg::span_gradient<color_type, 
                               agg::span_interpolator_trans<agg::trans_perspective>, 
							   agg::gradient_contour, 
                               color_func_type> span_gradient_type;

						span_gradient_type span_gradient(
						       span_interpolator, 
                               gradient_func, 
                               *cfunction, 
                               d1, d2);

						m_ras.reset();
						m_ras.add_path(t2 );

						agg::render_scanlines_aa(m_ras, m_sl, rb, m_alloc, span_gradient);
					}
				}
				break;

			   // assymetric conic
				case 2:
				{
					gradient_conic_angle gradient_func;

					agg::trans_affine_translation gmt(270 ,300);
					gmt.invert();

					agg::span_interpolator_linear<> span_interpolator(gmt);

					typedef agg::span_gradient<color_type, 
                               agg::span_interpolator_linear<>, 
                               gradient_conic_angle, 
                               color_func_type> span_gradient_type;

					span_gradient_type span_gradient(
						           span_interpolator, 
                                   gradient_func, 
                                   *cfunction, 
                                   d1, d2);

					m_ras.reset();
					m_ras.add_path(t2 );

					agg::render_scanlines_aa(m_ras, m_sl, rb, m_alloc, span_gradient);

				}
				break;

		       // Flat fill
				default:
					m_ras.reset();
					m_ras.add_path(t2 );
					rs.color(agg::rgba(0, 0.6, 0, 0.5));
					agg::render_scanlines(m_ras, m_sl, rs);

			}

		   // Stroke
			if (m_stroke.status())
			{
				agg::gradient_image<agg::srgba8> gradient_func;

				void* bitmap = gradient_func.image_create(64 ,64 );

				if (bitmap)
				{
					memmove(bitmap, agg::puzzle, 64 * 64 * 4 );

					agg::conv_stroke<t2_type> stroke(t2 );
					stroke.width(10.0 );

					agg::span_interpolator_trans<agg::trans_perspective> span_interpolator(trpg );

					typedef agg::one_color_function<agg::srgba8> color_func_type;
					typedef agg::span_gradient<color_type, 
                            agg::span_interpolator_trans<agg::trans_perspective>, 
							agg::gradient_image<agg::srgba8>, 
							const color_func_type> span_gradient_type;

					span_gradient_type span_gradient(
						    span_interpolator, 
                            gradient_func, 
                            gradient_func.color_function(), 
                            d1, d2);

					m_ras.reset();
					m_ras.add_path(stroke );

					agg::render_scanlines_aa(m_ras, m_sl, rb, m_alloc, span_gradient);

				}

			}

		}

    }

    unsigned render()
    {
		agg::path_storage star;

		star.move_to(12.0 ,40.0 );
		star.line_to(52.0 ,40.0 );
		star.line_to(72.0 ,6.0 );
		star.line_to(92.0 ,40.0 );
		star.line_to(132.0 ,40.0 );
		star.line_to(112.0 ,76.0 );
		star.line_to(132.0 ,112.0 );
		star.line_to(92.0 ,112.0 );
		star.line_to(72.0 ,148.0 );
		star.line_to(52.0 ,112.0 );
		star.line_to(12.0 ,112.0 );
		star.line_to(32.0 ,76.0 );
		star.close_polygon();

		switch(m_polygons.cur_item())
		{
		   // Simple path
			case 0:
			{
				perform_rendering(star ,star );

			}
			break;

		   // Great Britain
			case 1:
			{
				agg::path_storage gb_poly;
				make_gb_poly(gb_poly);
				perform_rendering(gb_poly ,star );

			}
			break;

		   // Spiral
			case 2:
			{
                spiral sp(0, 0, 10, 150, 30, 0.0);
                agg::conv_stroke<spiral> stroke(sp);
                stroke.width(22.0);
				perform_rendering(stroke ,star );

			}
			break;

		   // Glyph
			case 3:
			{
                agg::path_storage glyph;
                glyph.move_to(28.47, 6.45);
                glyph.curve3(21.58, 1.12, 19.82, 0.29);
                glyph.curve3(17.19, -0.93, 14.21, -0.93);
                glyph.curve3(9.57, -0.93, 6.57, 2.25);
                glyph.curve3(3.56, 5.42, 3.56, 10.60);
                glyph.curve3(3.56, 13.87, 5.03, 16.26);
                glyph.curve3(7.03, 19.58, 11.99, 22.51);
                glyph.curve3(16.94, 25.44, 28.47, 29.64);
                glyph.line_to(28.47, 31.40);
                glyph.curve3(28.47, 38.09, 26.34, 40.58);
                glyph.curve3(24.22, 43.07, 20.17, 43.07);
                glyph.curve3(17.09, 43.07, 15.28, 41.41);
                glyph.curve3(13.43, 39.75, 13.43, 37.60);
                glyph.line_to(13.53, 34.77);
                glyph.curve3(13.53, 32.52, 12.38, 31.30);
                glyph.curve3(11.23, 30.08, 9.38, 30.08);
                glyph.curve3(7.57, 30.08, 6.42, 31.35);
                glyph.curve3(5.27, 32.62, 5.27, 34.81);
                glyph.curve3(5.27, 39.01, 9.57, 42.53);
                glyph.curve3(13.87, 46.04, 21.63, 46.04);
                glyph.curve3(27.59, 46.04, 31.40, 44.04);
                glyph.curve3(34.28, 42.53, 35.64, 39.31);
                glyph.curve3(36.52, 37.21, 36.52, 30.71);
                glyph.line_to(36.52, 15.53);
                glyph.curve3(36.52, 9.13, 36.77, 7.69);
                glyph.curve3(37.01, 6.25, 37.57, 5.76);
                glyph.curve3(38.13, 5.27, 38.87, 5.27);
                glyph.curve3(39.65, 5.27, 40.23, 5.62);
                glyph.curve3(41.26, 6.25, 44.19, 9.18);
                glyph.line_to(44.19, 6.45);
                glyph.curve3(38.72, -0.88, 33.74, -0.88);
                glyph.curve3(31.35, -0.88, 29.93, 0.78);
                glyph.curve3(28.52, 2.44, 28.47, 6.45);
                glyph.close_polygon();

                glyph.move_to(28.47, 9.62);
                glyph.line_to(28.47, 26.66);
                glyph.curve3(21.09, 23.73, 18.95, 22.51);
                glyph.curve3(15.09, 20.36, 13.43, 18.02);
                glyph.curve3(11.77, 15.67, 11.77, 12.89);
                glyph.curve3(11.77, 9.38, 13.87, 7.06);
                glyph.curve3(15.97, 4.74, 18.70, 4.74);
                glyph.curve3(22.41, 4.74, 28.47, 9.62);
                glyph.close_polygon();

				agg::conv_curve<agg::path_storage> curve(glyph);
				curve.approximation_scale(10 );

				perform_rendering(curve ,star );

			}
			break;

		}

        return 0;
    }

    virtual void on_draw()
    {
        typedef agg::renderer_base<pixfmt> base_ren_type;

        pixfmt pf(rbuf_window());
        base_ren_type ren_base(pf);
        ren_base.clear(agg::rgba(1,1,1));

        render();

        agg::render_ctrl(m_ras, m_sl, ren_base, m_persp);
		agg::render_ctrl(m_ras, m_sl, ren_base, m_polygons);
        agg::render_ctrl(m_ras, m_sl, ren_base, m_gradient );
		agg::render_ctrl(m_ras, m_sl, ren_base, m_stroke );
		agg::render_ctrl(m_ras, m_sl, ren_base, m_refl );
		agg::render_ctrl(m_ras, m_sl, ren_base, m_c1 );
		agg::render_ctrl(m_ras, m_sl, ren_base, m_c2 );
		agg::render_ctrl(m_ras, m_sl, ren_base, m_d1 );
		agg::render_ctrl(m_ras, m_sl, ren_base, m_d2 );
		agg::render_ctrl(m_ras, m_sl, ren_base, m_clrs );
    }

	virtual void on_ctrl_change()
	{
		if (m_last != m_polygons.cur_item())
		{
			m_last = m_polygons.cur_item();
			m_init = true;
		}
	}

	virtual void on_resize(int cx, int cy)
	{
		m_init = true;
	}

#define bbox double x1, y1, x2, y2;\
x1 = m_persp.xn(0 );\
y1 = m_persp.yn(0 );\
x2 = m_persp.xn(1 );\
y2 = m_persp.yn(2 );\
if (m_persp.yn(1 ) < y1 )\
{\
    y1 = m_persp.yn(1 );\
}\
if (m_persp.xn(2 ) > x2 )\
{\
    x2 = m_persp.xn(2 );\
}\
if (m_persp.xn(3 ) < x1 )\
{\
    x1 = m_persp.xn(3 );\
}\
if (m_persp.yn(3 ) > y2 )\
{\
    y2 = m_persp.yn(3 );\
}

#define rotate(a) agg::trans_affine_translation tat(-(x1 + (x2 - x1 ) / 2 ) ,-(y1 + (y2 - y1 ) / 2 ) );\
tat.transform(&m_persp.xn(0) ,&m_persp.yn(0) );\
tat.transform(&m_persp.xn(1) ,&m_persp.yn(1) );\
tat.transform(&m_persp.xn(2) ,&m_persp.yn(2) );\
tat.transform(&m_persp.xn(3) ,&m_persp.yn(3) );\
agg::trans_affine_rotation tar(agg::deg2rad(a ) );\
tar.transform(&m_persp.xn(0) ,&m_persp.yn(0) );\
tar.transform(&m_persp.xn(1) ,&m_persp.yn(1) );\
tar.transform(&m_persp.xn(2) ,&m_persp.yn(2) );\
tar.transform(&m_persp.xn(3) ,&m_persp.yn(3) );\
agg::trans_affine_translation tt2(x1 + (x2 - x1 ) / 2 ,y1 + (y2 - y1 ) / 2 );\
tt2.transform(&m_persp.xn(0) ,&m_persp.yn(0) );\
tt2.transform(&m_persp.xn(1) ,&m_persp.yn(1) );\
tt2.transform(&m_persp.xn(2) ,&m_persp.yn(2) );\
tt2.transform(&m_persp.xn(3) ,&m_persp.yn(3) );\
force_redraw();

#define zoom(z) agg::trans_affine_translation tat(-(x1 + (x2 - x1 ) / 2 ) ,-(y1 + (y2 - y1 ) / 2 ) );\
tat.transform(&m_persp.xn(0) ,&m_persp.yn(0) );\
tat.transform(&m_persp.xn(1) ,&m_persp.yn(1) );\
tat.transform(&m_persp.xn(2) ,&m_persp.yn(2) );\
tat.transform(&m_persp.xn(3) ,&m_persp.yn(3) );\
agg::trans_affine_scaling tas(z );\
tas.transform(&m_persp.xn(0) ,&m_persp.yn(0) );\
tas.transform(&m_persp.xn(1) ,&m_persp.yn(1) );\
tas.transform(&m_persp.xn(2) ,&m_persp.yn(2) );\
tas.transform(&m_persp.xn(3) ,&m_persp.yn(3) );\
agg::trans_affine_translation tt2(x1 + (x2 - x1 ) / 2 ,y1 + (y2 - y1 ) / 2 );\
tt2.transform(&m_persp.xn(0) ,&m_persp.yn(0) );\
tt2.transform(&m_persp.xn(1) ,&m_persp.yn(1) );\
tt2.transform(&m_persp.xn(2) ,&m_persp.yn(2) );\
tt2.transform(&m_persp.xn(3) ,&m_persp.yn(3) );\
force_redraw();

	virtual void on_key(int x, int y, unsigned key, unsigned flags )
	{
		if (key == agg::key_kp_plus )
		{
			bbox
			rotate(-angle)
		}

		if (key == agg::key_kp_minus )
		{
			bbox
			rotate(angle)
		}

		if (key == agg::key_page_up )
		{
			bbox
			zoom(zoom_up)
		}

		if (key == agg::key_page_down )
		{
			bbox
			zoom(zoom_down)
		}

		if (key == agg::key_f1)
		{
			message(
				"Here are some more gradient functions. Firstly the Contour gradient, makes\n"
				"color transitions from shape defined by an arbitrary (in fact any) path.\n"
				"It computes so called Distance Transform (DT) from image produced by only\n"
				"stroking the path, which is then the source of color level in the underlying\n"
				"gradient function. Contour gradient can be used in two forms. One is to define   \n"
				"shape for contour different from shape of object being drawn. Second is to\n"
				"use the same shape for contour and for drawing (AutoContour).\n\n"

				"Assymetric conic gradient (also called angle) is the same as conic, but the ray\n"
				"of light with color transitions going counter clockwise travels whole circle\n"
				"instead of just half (as with conic).\n\n"

				"Bitmap gradient is very similar to pattern fill, but works in the framework of\n"
				"gradient functions interfaces. Because of that all interpolator transformations\n"
				"from gradient span generator can be applied to this kind of fill.\n\n"

				"How to play with:\n\n"

				"Try to fiddle with C1,C2 & D1,D2 parameters to see, how interestingly\n"
				"the gradient transitions changes (each time a new DT is computed).\n"
				"DT is reused when the color ramp changes.\n\n"

				"Use +/- keys to Rotate polygon\n"
				"Use PgUp/PdDn keys to Zoom polygon\n\n"

				"Note: F2 key saves current 'screenshot' file in this demo's directory.  " );

		}

	}

};



int agg_main(int argc, char* argv[])
{
    the_application app(pix_format, flip_y);
    app.caption("AGG Example. More gradients: Contour, Bitmap & Assymetric Conic (F1-Help)");

    if(app.init(520, 520, agg::window_resize))
    {
        return app.run();
    }
    return 1;
}


