# -*- coding: utf-8 -*-
###########################################
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published
#  by the Free Software Foundation; version 2 only.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
###########################################

import os
import statvfs
import shutil
import stat
import time
import gtk
import gobject
import urllib
import bz2
import gzip

# path - file procedures
def normalizePath(path):
    return os.path.abspath(path)

def get_free_space(path):
    wheretolook = "."
    if path !="":
        wheretolook = path
    s = os.statvfs(wheretolook)
    freebytes = (s[statvfs.F_BSIZE] * s[statvfs.F_BAVAIL])
    return freebytes
    
def get_burn_applications(burnfile):
    """
        Get cd burn applications to be used with aptoncd.
    """
    burns = []
    try:
        file = open(burnfile)
        for line in file.readlines():
            app = line.split(';')
            burns.append([app[0].replace('\n',''),app[1].replace('\n','')])
    except:
        burns.append(['nautilus-cd-burn','--source-iso='])

    return burns

def get_home_folder():
    """
        Gets user home folder from environment.
    """
    return os.environ['HOME']

def whereis(filename):
    val = os.popen('whereis %s' % filename).readlines()
    #remove break in dots
    val2 = val[0].split(" ")

    # delete first element (name of app used in search)
    del val2[0]

    if len(val2) > 0:
        return val2[0]
    else:
        return None

def checkAccess(path):
    """
        Checks file/folder permissions and
        returns True if the file/folder could be accessed
    """
    mRet = False
    if pathExists(path):
        if os.access(path, os.R_OK | os.W_OK | os.X_OK):
            mRet = True
    return mRet

def fileExist( path):
    """Returns True if a file exist."""
    return os.path.isfile(path)

def pathExists( path):
    """Returns True if a path exists."""
    return os.path.exists(path)

def removePath( path):
    """ Removes a directory path """
    try:
        shutil.rmtree(path, ignore_errors=False)
    except Exception , e:
        print str(e)
        pass

def walktree(folder_name = ".", depthfirst = True):
    """
        Navigates recursively through folders.
    """
    names = os.listdir(folder_name)
    if not depthfirst:
        yield folder_name, names
    for name in names:
        try:
            st = os.lstat(os.path.join(folder_name, name))
        except os.error:
            continue
        if stat.S_ISDIR(st.st_mode):
            for (newtop, children) in walktree (os.path.join(folder_name, name), depthfirst):
                yield newtop, children
    if depthfirst:
        yield folder_name, names
    return

def get_deb_from_folder(path, recursive = False):
    """ """
    deb_files = []

    if recursive:
        for top_folder, names in walktree(path, recursive):
            for name in names:
                if ( name.endswith(".deb") and "aptoncd-metapackage" not in name ):
                    deb_files.append(os.path.join(top_folder,name))
    else:
        deb_files = [ filename for filename in os.listdir(path) if ( filename.endswith(".deb") and "aptoncd-metapackage" not in filename ) ]
    return deb_files

def mkdir( path, removeExisting = False):
    """ 
        Create a full path, directory by directory
        if removeExisting is set it will remove main folder and contents before creation.
    """

    #remove directiory if already exists
    if removeExisting:
        if pathExists(path):
            removePath(path)

    if not pathExists(path):
        os.makedirs(path)

    return path

def del_file( path):
    """ Deletes a file if it exists"""
    if fileExist(path):
        os.remove(path)

def get_current_dir():
    return os.getcwd()

def run_command(command):
    return os.system(command)

def change_dir(path):
    os.chdir(path)

def list_dir(path, pattern = ''):
    if pattern != '':
        return [nFile for nFile in os.listdir(path) if ( pattern in nFile  )]
    else:
        return os.listdir(path)

def join_path(path, filename = ''):
    """
        Joins path and filename to make one path.
    """
    if path[-1] == '/':
        return path + filename
    else:
        return os.path.join(path, filename)
def compress(source, dest = ''):
    """
        Compress a file using bz2 compression.
    """
    if dest == '':
        dest = source

    dest_ext = '.bz2'
    arcname = os.path.basename (source)
    dest_name = '%s%s' % (dest, dest_ext)
    dest_path = os.path.join(dest, dest_name)

    try:
        input = bz2.compress(file(source, 'r').read())
        out = file(dest_path,'w')
        out.write(input)
        out.close()
    except Exception , e:
        return False, str(e)

    return True, dest_path

def zip_file(source, dest):
    try:
        fileObj = gzip.GzipFile(os.path.join(source), 'wb');
        fileObj.write(file(os.path.join(dest), 'rb').read())
        fileObj.close()
        return True, ''
    except Exception, e:
        return False, str(e)

def parse_url( uri):
     """ helper to get a useful path from a drop uri"""
     path = urllib.url2pathname(uri) # escape special chars
     path = path.strip('\r\n\x00') # remove \r\n and NULL
     # get the path to file
     if path.startswith('file:\\\\\\'): # windows
         path = path[8:] # 8 is len('file:///')
     elif path.startswith('file://'): # nautilus, rox
         path = path[7:] # 7 is len('file://')
     elif path.startswith('file:'): # xffm
         path = path[5:] # 5 is len('file:')

     return path
# functions

def split_path(path):
    """Helper to split a path/filename from a url"""
    return os.path.split(path)

def grayscale(img , value = 0.0, dark_option = False):
    """
        Grayscale an image or a pixbuff using a factor.

        @img a gtk.Image or a gtk.gdk.Pixbuf object
            if @img is an image it will return a grayscale image object.
            if @img is a pixbuf it will return a pixbuf object.
        @value a factor to grayscale the imagem/pixbuf
    """
    if isinstance(img, gtk.Image):
        pixbuf = image.get_pixbuf()
        pixbuf2 = pixbuf.copy()
        pixbuf.saturate_and_pixelate(pixbuf2, value, dark_option)
        temp_img = gtk.Image()
        temp_img.set_from_pixbuf(pixbuf2)
        return temp_img
    elif isinstance(img, gtk.gdk.Pixbuf):
        pixbuf2 = img.copy()
        img.saturate_and_pixelate(pixbuf2, value, dark_option)
        return pixbuf2
    else:
        return None

def get_icon(icon_name, icon_size = 32, returnNone = False):
    icon_theme = gtk.icon_theme_get_default()
    try:
        return icon_theme.load_icon(icon_name, icon_size, 0)
    except gobject.GError, exc:
        try :
            if returnNone:
                return None
            else:
                return icon_theme.load_icon('deb', icon_size, 0)
        except gobject.GError, exc:    
            return None

def humanize_file_size( i):
    """
        Taken from jinja
        Format the value like a 'human-readable' file size (i.e. 13 KB, 4.1 MB,
        102 bytes, etc).
    """
    try:
        bytes = float(i)
    except:
        bytes = 0

    if bytes < 1024:
        return u"%d Byte%s" % (bytes, bytes != 1 and u's' or u'')
    elif bytes < 1024 * 1024:
        return u"%.1f KB" % (bytes / 1024)
    elif bytes < 1024 * 1024 * 1024:
        return u"%.1f MB" % (bytes / (1024 * 1024))
    else:
        return u"%.1f GB" % (bytes / (1024 * 1024 * 1024)) 

def escape(text):
    return text.replace("&", "&amp;" )

class SystemInfo:
    """
        This class helps to get system information.
    """
    def __init__(self):
        """
            Constructor
        """
        self.distro = '' #distribution name
        self.architecture = '' #System architecture
        self.codename = '' #distribution code name
        self.release = '' #realease info
        self.read_release_info()

    def read_release_info(self):
        """
            Get release info and fill variables.
        """
        self.get_infos()
        self.architecture = self.__architecture()

    def get_infos(self):
        """
            Get system info values.
        """
        try:
            p = open('/etc/lsb-release','r')
            for line in p:
                if 'DISTRIB_ID' in line:
                    self.distro = line.split('=')[-1].replace('\n','').lower()
                elif 'DISTRIB_CODENAME' in line:
                    self.codename = line.split('=')[-1].replace('\n','').lower()
                elif 'DISTRIB_RELEASE' in line:
                    self.release = line.split('=')[-1].replace('\n','').lower()
            p.close()
        except:
            pass

    def __architecture(self):
        """
        Detect whether the architecture is x86/ppc/amd64
        """
        arch = os.uname()[-1]
        if arch in ('ppc', 'ppc64'):
            arch = 'powerpc'
        elif arch in ('sparc32','sparc64','sparc'):
            arch = 'sparc'
        elif arch =='x86_64':
            if self.distro in ('debian'):
                arch = 'ia64'
            else:
                arch = 'amd64'
        elif arch in ('i386','i686','i586','k7'):
            arch = 'i386'

        return arch
