//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Model/ParameterTreeUtil.h
//! @brief     Defines namespace GUI::Model::ParameterTreeUtil
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_MODEL_PARAMETERTREEUTIL_H
#define BORNAGAIN_GUI_MODEL_MODEL_PARAMETERTREEUTIL_H

#include <QPair>
#include <QString>
#include <functional>
#include <variant>

class BackgroundItem;
class BeamDistributionItem;
class BeamItem;
class DetectorItem;
class DoubleProperty;
class InstrumentItem;
class Interference2DAbstractLatticeItem;
class ItemWithParticles;
class JobItem;
class OffspecDetectorItem;
class ParameterContainerItem;
class ParameterItem;
class ParameterLabelItem;
class ParticleLayoutItem;
class VectorProperty;

//! The ParameterTreeBuilder contains helper functions to create container
//! with ParameterItems. The ParameterItem appears in RealTimeView and provides real
//! time tuning of SampleItem and InstrumentItem.

class ParameterTreeBuilder {
public:
    ParameterTreeBuilder(JobItem* jobItem, bool recreateBackupValues);

    void build();

private:
    //! add the job's materials
    void addMaterials();

    //! add the job's sample
    void addSample();
    void addInstrument();
    void addParameterItem(ParameterLabelItem* parent, DoubleProperty& d,
                          const QString& label = QString());
    void addParameterItem(ParameterLabelItem* parent, VectorProperty& d);
    void addMagnetizationNoZ(ParameterLabelItem* parent, VectorProperty& d);
    ParameterContainerItem* parameterContainerItem();
    bool allowMagneticFields() const;

    void addInterference(ParameterLabelItem* layoutLabel, const ParticleLayoutItem* layout);

    //! Returns the top label which was created for the particle
    ParameterLabelItem* addItemWithParticles(ParameterLabelItem* parentLabel, ItemWithParticles* p,
                                             bool enableAbundance, bool enablePosition = true);
    void addLattice(ParameterLabelItem* parentLabel, const Interference2DAbstractLatticeItem* itf);
    void addRotation(ParameterLabelItem* parentLabel, ItemWithParticles* p);

    void addBeamDistribution(ParameterLabelItem* parentLabel,
                             BeamDistributionItem* distributionItem, const QString& label,
                             bool withMean = true);

    void addDetector(ParameterLabelItem* parentLabel, DetectorItem* detector);
    void addOffspecDetector(ParameterLabelItem* parentLabel, OffspecDetectorItem* detector);
    void addBackground(ParameterLabelItem* instrumentLabel, BackgroundItem* backgroundItem);
    void addPolarization(ParameterLabelItem* instrumentLabel, InstrumentItem* instrument);

private:
    JobItem* m_jobItem;
    bool m_recreateBackupValues;
};

#endif // BORNAGAIN_GUI_MODEL_MODEL_PARAMETERTREEUTIL_H
