-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-hostvariable.adb,v $
--  Description     : Ada Database Objects - Hostvariable                    --
--  Author          : Michael Erdmann                                        --
--  Created         : 29.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/02/14 20:46:59 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This package implements the kernel features of hostvariables             --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
with System;                             use System;
with System.Storage_Elements;            use System.Storage_Elements;
with Unchecked_Deallocation;

package body GNU.DB.ADBC.Hostvariable  is

   Version : constant  String :=
      "$Id: gnu-db-adbc-hostvariable.adb,v 1.1 2002/02/14 20:46:59 merdmann Exp $";

   type Storage_Access is access Storage_Array;

   type Object_Data is record
         Is_Null  : Boolean        := False;
         Self     : Handle         := null;
         Length   : Natural        := 0;
         Hostdata : Storage_Access := null;
         Stmt     : Statement_ID;
      end record;

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      This : in out Object ) is
      Data : Object_Data_Access := new Object_Data;
   begin
      Data.Is_Null := False;
      Data.Self    := This'Unrestricted_Access;

      This.Data    := Data;
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      This : in out Object ) is

      procedure Free is
            new Unchecked_Deallocation( Object_Data, Object_Data_Access);
   begin
      if This.Data /= null then
         Deallocate( This );
         Free( This.Data );
      end if;
   end Finalize;

   -------------
   -- Is_Null --
   -------------
   function Is_Null(
      This : in Object'Class ) return Boolean is
      Data : Object_Data_Access := This.Data;
   begin
      return Data.Is_Null;
   end Is_Null;

   ----------
   -- Self --
   ----------
   function Self(
      This : in Object'Class ) return Handle is
   begin
      return This.Data.Self;
   end Self;

   --------------
   -- Allocate --
   --------------
   function Allocate(
      This   : in Object'Class;
      Size   : in Natural ) return System.Address is
      -- allocate a piece of memory
      Data   : Object_Data_Access renames This.Data;
      Length : Natural := Size / Storage_Unit;
      Result : Storage_Access :=
                  new Storage_Array( 1..Storage_Offset(Length) );
   begin
      if Data.Hostdata /= null then
         Deallocate( This );
      end if;
      Data.Hostdata := Result;
      Data.Length   := Length;
      return Result(1)'Address;
   end Allocate;

   ----------------
   -- Deallocate --
   ----------------
   procedure Deallocate(
      This  : in Object'Class ) is

      procedure Free is
            new Unchecked_Deallocation( Storage_Array, Storage_Access);
   begin
      Free( This.Data.Hostdata );
   end Deallocate;

   --------------
   -- Hostdata --
   --------------
   function Hostdata(
      This   : in Object'Class ) return System.Address is
   begin
      return This.Data.Hostdata(1)'Address;
   end Hostdata;

   ------------
   -- Length --
   ------------
   function Length(
      This   : in Object'Class ) return Natural is
   begin
      return This.Data.Length ;
   end Length;

   -------------------------
   -- Assign_To_Statement --
   -------------------------
   procedure Assign_To_Statement(
      This : in Object'Class;
      Stmt : in Statement_ID ) is
   begin
      This.Data.Stmt := Stmt;
   end Assign_To_Statement;

end GNU.DB.ADBC.Hostvariable ;

