///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ViewportInputManager.h
 * \brief Contains the definition of the Core::ViewportInputManager class.
 */

#ifndef __OVITO_VIEWPORT_INPUT_MANAGER_H
#define __OVITO_VIEWPORT_INPUT_MANAGER_H

#include <core/Core.h>
#include "ViewportInputHandler.h"

namespace Core {

class Viewport; // defined in Viewport.h

/// \def VIEWPORT_INPUT_MANAGER
/// \brief The predefined instance of the Core::ViewportInputManager class.
///
/// Always use this macro to access the Core::ViewportInputManager class instance.
#define VIEWPORT_INPUT_MANAGER		(*ViewportInputManager::getSingletonInstance())

/**
 * \brief Manages a stack of viewport input handlers.
 *
 * This is a singleton class with only one predefined instance of this class.
 * You can access the instance of this class using the VIEWPORT_INPUT_MANAGER macro.
 *
 * \author Alexander Stukowski
 * \sa ViewportInputHandler
 */
class CORE_DLLEXPORT ViewportInputManager : public QObject
{
	Q_OBJECT
public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the ViewportInputManager singleton class.
	/// \note You should use the VIEWPORT_INPUT_MANAGER macro to access the ViewportInputManager instance instead
	///       of this method.
	inline static ViewportInputManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "ViewportInputManager::getSingletonInstance", "ViewportInputManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Returns the currently active ViewportInputHandler that handles the mouse events in viewports.
	/// \return The handler implementation that is responsible for mouse event processing. Can be \c NULL when the handler stack is empty.
	ViewportInputHandler* currentHandler() const;

	/// \brief Returns the current stack of input handlers.
	/// \return The stack of input handlers. The topmost handler is the active one.
	const QVector<ViewportInputHandler::SmartPtr>& stack() { return inputHandlerStack; }

	/// \brief Pushes a handler onto the stack and makes it active.
	/// \param handler The handler to be made active.
	/// \sa removeInputHandler()
	void pushInputHandler(const ViewportInputHandler::SmartPtr& handler);

	/// \brief Removes a handler from the stack and deactivates it if it is currently active.
	/// \param handler The handler to remove from the stack. It is not deleted by this method.
	/// \sa pushInputHandler()
	void removeInputHandler(ViewportInputHandler* handler);

public Q_SLOTS:

	/// \brief Resets the input mode stack to its initial state on application startup.
	///
	/// All input handlers are removed from the handler stack and the default input handler
	/// is activated.
	void reset();

Q_SIGNALS:

	/// \brief This signal is sent when the active viewport input mode has changed.
	/// \param oldMode The previous input handler (can be \c NULL).
	/// \param newMode The new input handler that is now active (can be \c NULL).
	void inputModeChanged(ViewportInputHandler* oldMode, ViewportInputHandler* newMode);

private:

	/// Stack of input handlers.
	/// The topmost handler is the active one.
	QVector< ViewportInputHandler::SmartPtr > inputHandlerStack;

private:

	/// Private constructor.
	/// This is a singleton class; no public instances are allowed.
	ViewportInputManager();

	/// Initializes the ViewportInputManager class.
	/// This is called at program startup.
	static void initialize() {
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new ViewportInputManager();
	}

	/// ViewportInputManager shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}

	/// The singleton instance of this class.
	static ViewportInputManager* _singletonInstance;

	friend class ApplicationManager;
};

};

#endif // __OVITO_VIEWPORT_INPUT_MANAGER_H
