
##############################################################################
#
# Copyright (c) 2003-2017 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2017 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
test for binary operations with tagged data

:remark: use see `test_util`
:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement 
:var __url__: url entry point on documentation 
:var __version__: version
:var __date__: date of the version
"""

__author__="Lutz Gross, l.gross@uq.edu.au"

import esys.escriptcore.utestselect as unittest
import numpy
from esys.escript import *
from test_util_base import Test_util_base

class Test_util_binary_with_tagged_data(Test_util_base):
   """
   test for the operations inner, outer ...  with tagged data
   """
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_float_rank0_taggedData_rank0(self):
      arg0=0.302372465341
      arg1=Data(-0.685671977824,self.functionspace)
      arg1.setTaggedValue(1,-0.000993700388667)
      res=inner(arg0,arg1)
      ref=Data(-0.20732832634981449,self.functionspace)
      ref.setTaggedValue(1,-0.00030046763633121754)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=inner(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank0_taggedData_rank0(self):
      arg0=numpy.array(0.0765648367337)
      arg1=Data(0.472158944777,self.functionspace)
      arg1.setTaggedValue(1,-0.230201498481)
      res=inner(arg0,arg1)
      ref=Data(0.036150772519180453,self.functionspace)
      ref.setTaggedValue(1,-0.017625340147041275)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(arg0*1j)
      arg1=arg1*1j
      ref=-ref
      res=inner(arg0,arg1)     
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank1_taggedData_rank1(self):
      arg0=numpy.array([-0.86109887228266446, 0.80664771143714975])
      arg1=Data(numpy.array([-0.53533967994450604, 0.45200717986751293]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.92533923599714352, 0.85965743120429949]))
      res=inner(arg0,arg1)
      ref=Data(0.825590951882,self.functionspace)
      ref.setTaggedValue(1,1.4902492721)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=inner(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank2_taggedData_rank2(self):
      arg0=numpy.array([[0.28546214116183566, -0.25897102118715942, 0.22146243895717355, -0.2341748097231191, 
0.55062645551197886], [0.15477233395592593, 0.6983214710376231, 0.082312765100553964, 0.88830487348434883, 
-0.35263700545380994], [-0.30422131569008792, -0.6752995720182402, -0.89033601042250776, -0.27533901644197178, 
-0.85228273014231459], [-0.27157032271746062, 0.22782516604067693, 0.69643942469578146, 0.21476478403738941, 
-0.95372841622315097]])
      arg1=Data(numpy.array([[-0.12475183486978714, -0.59085384957661913, -0.57768826874218804, 0.69002921217887692, 
0.6146136290800055], [-0.76675274595793108, 0.096016036368918867, -0.72098517522278427, 0.60739975677450486, 
0.20039340509437475], [-0.53106917471571125, 0.66808112301286671, -0.63653755160321879, -0.42121825850964556, 
-0.86625666223079612], [-0.14889283597580261, 0.61453496781922712, -0.037562334932733421, 0.59405710847032323, 
-0.15339609068293081]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.74276855773761907, -0.64412463316166813, -0.59462268478785418, 0.49086072608194375, 
0.12177039971317694], [-0.72919475767344633, 0.93923067996533427, -0.95425250771711911, 0.46032229575023731, 
-0.30203572196549966], [0.56279562163448782, 0.23249492420872153, 0.021543867282588725, -0.30407015194693954, 
-0.67749837500751253], [-0.48490284800976879, -0.20505188217693404, 0.49270997431831609, -0.60447551176070791, 
0.24761094681266882]]))
      res=inner(arg0,arg1)
      ref=Data(2.0837982665,self.functionspace)
      ref.setTaggedValue(1,1.55503359669)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=inner(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank3_taggedData_rank3(self):
      arg0=numpy.array([[[-0.36965911393496653, 0.98091128998261667], [-0.77339818424681805, -0.12759468235415383]], 
[[0.85746330178302221, 0.53597813647549519], [0.42784310969022865, 0.66625125398782981]], [[-0.3272902988529407, 
-0.045146665394883767], [0.010477863144669808, 0.36591153971242574]], [[0.78418331466442814, -0.2680867565180145], 
[-0.82326919338393179, 0.331955227753574]], [[-0.73241891335179088, 0.13873093720146068], [-0.55672796708761174, 
-0.65553307432364538]], [[-0.91098505655985895, -0.041178931836881594], [-0.84850305096137357, 0.075790396228343937]]])
      arg1=Data(numpy.array([[[-0.045801260013890621, 0.050011452068426498], [0.12900776296788585, -0.72261296995481095]], 
[[-0.50810328798679771, -0.39652163006393115], [0.16632033579522254, -0.086016092943626221]], [[-0.52076314263072998, 
-0.28555894670684556], [-0.032774218742086081, -0.26414862258056804]], [[0.78155149305534199, -0.19538864556833069], 
[0.28091162566157446, 0.73222242328111409]], [[0.6600836851486287, -0.42668876810698309], [-0.79194290839233084, 
0.96451261473614314]], [[-0.95480400656677245, 0.31956525371280509], [0.78360682059206876, 
0.64161295312382882]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.65422428312190717, -0.53976359988965394], [-0.052928480807738376, 
0.24124608151737514]], [[0.034172873530292192, -0.43901996828862244], [0.48931220600808989, -0.41404722086870471]], 
[[-0.30993774327062162, -0.78232475467205198], [-0.90686194430041378, -0.21480804837303458]], [[0.8222242674345126, 
0.53139548519319901], [0.70824891290441272, -0.82344959541323637]], [[0.87193618834414788, 0.69581301129047568], 
[-0.70465587143016051, 0.88104441331923589]], [[-0.79952576914512608, 0.26866616537225418], [0.30935236842782232, 
0.7773523878110058]]]))
      res=inner(arg0,arg1)
      ref=Data(-0.306184779423,self.functionspace)
      ref.setTaggedValue(1,-1.06905993572)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=inner(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank4_taggedData_rank4(self):
      arg0=numpy.array([[[[0.7604938557994263, -0.2064171490048079, 0.0082224477660097772, 0.47661474451358177], 
[-0.016147095333721051, 0.92510969195222525, 0.36963882069134235, 0.7313344332761289], [-0.44349149295001022, 
0.50038611389493015, 0.073989698187160835, 0.30213127319358435]], [[-0.085779312355246162, -0.6516617502626687, 
-0.58552435465049935, -0.54670802040975519], [-0.97149839070701294, 0.41188158744522307, 0.25718202606929053, 
-0.24291708491392328], [0.64570117640655855, -0.83614561582065616, 0.28699854951478798, -0.93312346950685754]]], 
[[[-0.0018734537112383709, 0.1051474428933783, -0.11005886066716353, 0.37962929610600193], [0.3126099665104769, 
0.23100822254020681, -0.96283590668441099, -0.41440492034589926], [-0.42530572477680773, -0.79313343846577156, 
-0.89038637212946026, -0.42387473920202057]], [[0.27908258076110259, -0.7068815739396217, 0.53637477002212108, 
0.24726832002590049], [-0.13013211223246435, -0.78735407545299374, -0.58836851126137502, -0.36234941534751197], 
[-0.90815573867674626, -0.88313365509272446, -0.68284207012769271, 0.31975112384448079]]], [[[0.77142356827210001, 
0.91191967036317712, 0.55855716445627746, 0.40827639976213281], [0.46021247517969788, 0.7343548047142987, -0.48088350560507598, 
0.37098064486240512], [-0.091697475899351177, -0.40781182998553556, 0.45547669848377237, -0.087366870549572706]], 
[[0.98648280728299209, 0.41378759479398819, 0.32888151537143351, -0.43752490453037263], [0.88300345537397695, 
0.29494171299568217, -0.7713450313602781, -0.29281904047584706], [-0.82614050614132317, -0.3971654638950386, 
-0.71539168791654806, 0.15347445192877696]]]])
      arg1=Data(numpy.array([[[[-0.46437123780262635, 0.74185133523265634, 0.49951558233663884, -0.71118985538663848], 
[0.56245719755571622, -0.50427973433709905, -0.12435103652047963, 0.66980870844438223], [0.78155494433345329, 
-0.85062663906351754, 0.13031784376377731, -0.88502969392544162]], [[-0.97920484738866742, -0.0013183327286987456, 
0.05788815970797101, -0.25450310079245542], [-0.10117905674552485, -0.66776086342060137, -0.064655167154714199, 
0.78055990504131412], [0.7964676623185134, -0.46711576410747857, -0.50544931921196135, -0.71948312823243077]]], 
[[[-0.95339111079344985, 0.68081818892697377, -0.3780663950715053, -0.21119065883350752], [0.65254047523120917, 
0.76728475834565746, 0.87803824658903107, -0.8853243700127682], [-0.3733648785675967, 0.96328693805533505, 0.33135974940990853, 
-0.20617756533910181]], [[-0.58488374063355431, 0.16307637443378087, 0.010782110100696851, -0.29738192411321873], 
[-0.30031127943058711, 0.16916161988008671, -0.58277726739394664, -0.23461236749957237], [-0.99472730025633349, 
-0.59721917380688838, 0.023364804513121396, 0.75019537702359584]]], [[[-0.58798021179709536, -0.68699103286410312, 
0.074816868000341152, -0.23675859949810363], [-0.83143406378762141, 0.26888961413208534, 0.49310597144176938, 
0.7977151216226257], [-0.35221825673156193, 0.088975930385955859, 0.28584204161524585, 0.43347032440004152]], 
[[-0.043228877706345203, 0.49224430088901383, 0.041270615090814422, -0.93931728505799428], [-0.62438052044546866, 
-0.1555347219808787, -0.16005471087012535, -0.094995277206445294], [0.32864350845283452, 0.0018525108486773245, 
-0.52778818417274587, -0.44217899482475032]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.60823100775411576, 0.066842557698393756, 0.79566528979669782, 
0.82868562207053897], [0.78613498132146642, -0.67304720757061642, 0.37317140593547271, -0.42023355638056614], 
[-0.89334001862300805, -0.3484455219637248, -0.81915812288585266, -0.17186157363295962]], [[0.77190239187530096, 
-0.27466010315273626, -0.047069441496553344, 0.6406589001720695], [0.26517088784363785, 0.25594718420441964, 
-0.019799536003160423, -0.99591510500911018], [-0.30057831255741507, -0.097997744185298163, -0.33402571643961587, 
0.38899883894704934]]], [[[-0.60746130055235414, -0.86937617334659922, 0.4464080214266859, -0.65061460647358249], 
[-0.83500521257541127, -0.40467215791490863, -0.36884962773909358, -0.017581713186127779], [-0.78071757387893581, 
-0.7555378173021392, 0.16384580135655602, 0.39981324416993913]], [[-0.057920994167991235, -0.30947075713623273, 
-0.010489307190590269, 0.26827197750564236], [0.64537209540712182, -0.99304444119759605, -0.43536243190966828, 
-0.38845524275675447], [0.54072689740927937, 0.39536658441920691, -0.059113176182096883, 0.2365975100088995]]], 
[[[-0.71444398327660918, -0.10601210256168847, -0.35082430378728535, 0.53048714028601296], [0.33012650050062309, 
0.2233872473156, 0.72533907055290014, 0.54823462243745014], [0.64364908972318724, 0.97764737901583798, 0.85636778064605723, 
0.58032227689387805]], [[0.63842167202707722, 0.90385415304382066, -0.68593462018012064, -0.99721884955912099], 
[0.97096634999990328, 0.19065950109102614, 0.99287947652913089, -0.17178598348397101], [-0.37020715241965951, 
0.46519340045957036, 0.78203721578950969, 0.85742325776521233]]]]))
      res=inner(arg0,arg1)
      ref=Data(-0.893510369763,self.functionspace)
      ref.setTaggedValue(1,0.860688349328)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=inner(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank0_taggedData_rank0(self):
      arg0=Data(0.124403467791,self.functionspace)
      arg1=Data(-0.874425347455,self.functionspace)
      arg1.setTaggedValue(1,-0.671864318423)
      res=inner(arg0,arg1)
      ref=Data(-0.10878154554764913,self.functionspace)
      ref.setTaggedValue(1,-0.083582251096861085)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=inner(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([-0.87777216173835626, 0.84139785859967287]),self.functionspace)
      arg1=Data(numpy.array([0.19381057351148212, -0.80556320012388016]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.4071581862361735, 0.33773358004973697]))
      res=inner(arg0,arg1)
      ref=Data(-0.84792067763,self.functionspace)
      ref.setTaggedValue(1,-0.0732238102709)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[-0.53558219507023308, -0.61589866611438326, 0.45097856319098573, 0.25388993191306808, 
-0.30235062231871401], [-0.67843710959171544, 0.82102539341457237, -0.78663863825580416, -0.91255907508038492, 
0.84592712378626111], [-0.66629771965647944, 0.81619185346418366, -0.29532207464519744, 0.10768209234500614, 
-0.37659176789130555], [0.38297194424580261, 0.94435053259958179, -0.34406439315377391, 0.85228922849010402, 
0.31069471022979522]]),self.functionspace)
      arg1=Data(numpy.array([[-0.45063209573283625, 0.31906325025990867, -0.79996649582760315, -0.8719549142163685, 
-0.61116792027123434], [0.93342285423671356, 0.29309923039382113, -0.052346940956415011, -0.97354021791165568, 
-0.8166119312763509], [-0.81355518459002574, 0.8340183539915853, 0.41066261037050622, 0.72070919852118775, 
-0.91321592734799917], [-0.30398456692915921, -0.67967853737521899, 0.12078859764692362, 0.63256165021455679, 
-0.86135674027860887]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.070133972784780507, 0.79179455827162326, 0.22559047526462406, 
-0.15487582879118866, 0.086173179749749584], [-0.49099264874725934, 0.15718387953997537, 0.083557885763687167, 
0.36624713245990304, -0.9879782326380584], [0.8437413918981016, 0.78411031814521648, -0.92616444903525719, 0.52449912816715782, 
-0.88884568120597884], [0.4188503855795338, -0.83384284488763516, -0.1000450282773846, 0.014481499256485453, 
0.56405225419936711]]))
      res=inner(arg0,arg1)
      ref=Data(0.488352080172,self.functionspace)
      ref.setTaggedValue(1,-0.849782319222)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank3_taggedData_rank3(self):
      arg0=Data(numpy.array([[[0.32367315887079973, -0.30263048303176299], [0.82812191977634741, 0.3119002376452662]], 
[[0.22813257746868776, 0.97402972533071819], [-0.54233179929165987, -0.80680561598116007]], [[0.1590130931651037, 
-0.49426459739079198], [-0.11746988775232903, -0.9842239132556978]], [[-0.35695911128143742, -0.18249297127593178], 
[-0.92278154418741476, 0.87912080799490733]], [[-0.89586731890167859, -0.27524189410374422], [0.29646129579927738, 
-0.77345986616047147]], [[0.85402043641717662, -0.64082396428826294], [-0.69850050662519703, 
-0.27427052888689873]]]),self.functionspace)
      arg1=Data(numpy.array([[[-0.038373127004912577, 0.69854132693489612], [-0.22144804152724884, 0.867918531620258]], 
[[-0.2448082760358512, 0.039516040072887337], [0.60769318427537278, 0.59775962217911571]], [[-0.35136554812506349, 
-0.41786540808521555], [0.5577102184541467, -0.48219353351558403]], [[0.20679395339756135, -0.14203992627417339], 
[0.22347465879961881, 0.83469308762511529]], [[-0.71658361378345203, 0.73006344811297619], [0.18648331456071809, 
0.6070411780792635]], [[0.72379549609007632, 0.32971891184586433], [-0.89844536138693565, 
-0.047994818053717747]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.71218579508520241, 0.96010233443335546], [0.55507508072781708, 
-0.36831038492539214]], [[0.17528688575580786, 0.61695742846923629], [-0.82195672718749679, 0.17075920275894552]], 
[[0.55270422620308191, -0.74553477790604727], [-0.52581399960599584, -0.78115864647411315]], [[0.44805219775260041, 
-0.84839997965754477], [0.97200994593220913, 0.13649532133838327]], [[-0.46081677336839921, 0.94527111248512652], 
[-0.27412448981019244, -0.80029609067999363]], [[-0.50156035441383384, -0.2751471867242814], [-0.92982284092982637, 
-0.62847911902206222]]]))
      res=inner(arg0,arg1)
      ref=Data(1.14806570384,self.functionspace)
      ref.setTaggedValue(1,2.53777936588)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank4_taggedData_rank4(self):
      arg0=Data(numpy.array([[[[-0.93310409164839814, -0.0083882985931100773, 0.12139394793260316, -0.56134742619926215], 
[-0.50601897862936562, 0.42004575170481706, 0.35218268492545701, -0.77475021010461242], [0.79254311760208584, 
-0.054641867810987677, -0.050758183802747769, 0.0074782301973994603]], [[0.91239799146073008, -0.48935649243627699, 
0.76571554532417485, 0.76703850687451669], [0.011962595183448332, -0.77077827212134187, -0.6175254836025883, 
0.11718915795745821], [-0.61288264931728231, 0.2985764005034468, 0.80408183577228676, 0.37742071090166762]]], 
[[[0.31418618666015652, -0.25150031133868356, 0.89405784870817828, -0.39871636755948159], [0.17969596554515888, 
-0.29519251397439228, 0.28910118378066385, 0.71448980221461444], [0.21547295413418199, -0.72427927587438945, 
0.53536962343891958, -0.069690466431044262]], [[0.49000233467877341, -0.9141946674025565, 0.92356537646173953, 
0.36258771674787926], [0.98025031837109422, 0.97873403528715097, 0.34718708482623573, -0.61276091766631979], 
[0.86008273383772971, 0.21610382707976461, 0.76694442038930077, -0.23755335610017037]]], [[[0.88937859655236817, 
0.24344071269860978, 0.52899051979898015, -0.28696538419127982], [-0.46966277072662743, -0.30757480861305031, 
0.22839605612058311, 0.33192174295396026], [0.29142762132462985, -0.80630814389193173, -0.5751934892839492, 
0.65201665285212473]], [[-0.039813033511368889, 0.96529940590693175, 0.37632722676676567, -0.26877197607752779], 
[0.92745644548969675, -0.96045351917051502, 0.84375073169420811, -0.23099749192911356], [0.95561345473137993, 
-0.48726431358889988, -0.65953167042628724, 0.81239964832698797]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[0.51489134479347221, -0.54469574861105707, -0.037597216309791293, 0.7019466835656738], 
[-0.76731432871536542, -0.90127415409496514, -0.76987601868586997, 0.77010818157958871], [-0.69384686719896749, 
0.62625639111767351, -0.23308748573662963, -0.42652594480915629]], [[-0.18811542803666326, 0.1486445294791845, 
0.68173290208899284, 0.26969452098008007], [-0.45374750053407364, -0.64551896224527017, -0.16802446539309601, 
-0.24685031958026249], [-0.51848308917304098, -0.11861062085946261, 0.48934696254962851, 0.86517643359480934]]], 
[[[0.28086007546524328, -0.39936059062910068, -0.5067954326563715, -0.036782766435768277], [0.82560090016901788, 
0.27765920129644805, -0.018567093455775119, 0.47207891606182795], [-0.15593188747309594, 0.25529482052812713, 
0.032605315239380817, 0.27847952618974148]], [[-0.88227887390980642, 0.18427564516392758, -0.83294415177359493, 
-0.60388456873328455], [0.077171137101491372, -0.33680043732321585, 0.18420167589764103, 0.58052785166190968], 
[-0.64455700116665571, -0.79387638916683878, -0.64975007681278663, -0.84828567153880963]]], [[[0.32663262132021531, 
0.73218893318281197, -0.64345432791879675, 0.59965232226385523], [0.17131035254817917, -0.23132662830487472, 
0.6961546550573714, -0.53911393403379737], [-0.42666255251476404, 0.21530510545039738, 0.70401037578990699, 
0.095099886800676936]], [[0.5003184293580889, -0.098105108952792008, 0.50584889025171464, 0.38957363370207543], 
[0.09659533288820743, 0.080361254062045306, 0.030534457326652831, 0.93489890357882199], [0.41933737505983215, 
-0.063776648440711181, 0.66491991277606899, 0.44959720299084727]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.54418366876642899, -0.60526479467916716, 0.11342547329983055, 
0.42383191451312063], [0.39135815093342252, -0.44513901404757261, -0.23322235658664381, 0.3659994446410908], 
[0.66492888300362529, 0.62597290753537571, 0.31094112895184667, 0.10489580918025698]], [[-0.97047212620613776, 
-0.52791568763202812, 0.10995307784755948, -0.44515315931199395], [-0.47335144203548563, -0.97740483862239791, 
-0.9195838099615341, 0.50299939265672822], [0.049124924721247964, 0.15451147714094904, -0.60556101876844037, 
0.45416140786187498]]], [[[-0.47226380688055336, 0.31327143015943948, 0.54278284165551072, 0.24625540866446793], 
[0.89377751481793477, 0.83104905257937856, -0.13520853421266454, 0.95299778262344459], [-0.88482243081956935, 
-0.73274023636789032, -0.38931996372876387, 0.83062700338877016]], [[0.25938263990154242, -0.11989036805274944, 
-0.55654108615691134, 0.32775693391444682], [-0.67941981556523157, -0.46265473435061355, 0.57748311018645127, 
0.23668950450902027], [0.39174804246501505, 0.42180015848082486, -0.059846390834348728, -0.027940625115480566]]], 
[[[-0.75728289481046129, 0.8913423331446626, 0.36920972475668301, -0.49776833255035058], [-0.53392067106880292, 
-0.97021106812295321, 0.4389054023008272, -0.15891418785010325], [0.46855458984212395, -0.49678451406819368, 
0.97600572013840603, -0.1580943485209958]], [[0.54841118568796965, 0.95897485982412523, -0.31049169056675874, 
-0.048522087215220111], [-0.36218772131334931, -0.94678023203297856, 0.65259361536023497, 0.16207173456180546], 
[0.44765019783968674, 0.4212618926464633, -0.070340580015930243, -0.70874348922942909]]]]))
      res=inner(arg0,arg1)
      ref=Data(-4.04571518118,self.functionspace)
      ref.setTaggedValue(1,1.07939094218)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank0_float_rank0(self):
      arg0=Data(-0.208527599648,self.functionspace)
      arg0.setTaggedValue(1,-0.767468244961)
      arg1=-0.178137616236
      res=inner(arg0,arg1)
      ref=Data(0.037146609520777384,self.functionspace)
      ref.setTaggedValue(1,0.13671496369433997)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank0_array_rank0(self):
      arg0=Data(-0.857100156979,self.functionspace)
      arg0.setTaggedValue(1,0.225119730149)
      arg1=numpy.array(-0.948772846918)
      res=inner(arg0,arg1)
      ref=Data(0.81319335603092824,self.functionspace)
      ref.setTaggedValue(1,-0.2135874872707175)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank1_array_rank1(self):
      arg0=Data(numpy.array([-0.10929214686781652, -0.41794109094332277]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.59014609402756646, -0.54105713874983774]))
      arg1=numpy.array([0.92461235858287183, 0.22726606102559077])
      res=inner(arg0,arg1)
      ref=Data(-0.196036695169,self.functionspace)
      ref.setTaggedValue(1,-0.668620296621)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[0.50432055496183503, -0.25926507266256782, 0.90218730806132474, -0.65644592639875099, 
-0.059191757078101759], [0.66359334654734159, 0.37719497986085426, 0.40048540726169235, -0.58043861122321316, 
-0.87561593959589334], [0.78558772536960753, 0.17605040095994906, -0.78505148254062984, 0.40642948202591578, 
0.45654533196536051], [-0.32942377362749764, 0.1874644453300931, 0.71344685195895496, -0.40528138931671442, 
0.88857878462404827]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.22105602264554003, -0.63863190060808672, 0.50266503449971722, -0.7925888054932555, 
-0.83308060277192775], [0.63703697336208776, -0.032840067023147679, 0.69911446681021494, 0.21040364385932797, 
0.29404918846609007], [-0.070863199153732293, -0.18543571622725263, -0.82063526347944293, 0.37192977688927753, 
0.44803860869320022], [-0.86741289381545505, -0.89820931814083838, -0.37820158569229401, 0.38847596974859422, 
0.68390172357881429]]))
      arg1=numpy.array([[0.12749341798106895, 0.45788765153576771, -0.6039619754705059, -0.49054005474816376, 
0.86031738019555126], [0.24594646259761133, -0.095457475650316814, -0.24118419741163821, 0.029858526013547104, 
-0.92932795231554488], [0.74101806194544184, -0.72015049555575827, 0.86099611539654819, -0.49626581292352223, 
-0.69390917864289525], [-0.43069226988820031, 0.96152849074742819, 0.29564054323986033, -0.80384598311832089, 
0.0090587338159839881]])
      res=inner(arg0,arg1)
      ref=Data(0.626617989575,self.functionspace)
      ref.setTaggedValue(1,-3.25686621688)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank3_array_rank3(self):
      arg0=Data(numpy.array([[[-0.76171304226146686, -0.87057568254167039], [-0.95783872561735039, 0.028253142239920814]], 
[[0.025144394665756575, -0.98991849326909009], [-0.39520559134097222, -0.19067238233979156]], [[-0.26218857207856705, 
0.49187562788004735], [0.85284340674467396, -0.56038962999128583]], [[-0.089703826668767528, -0.49929995985201558], 
[0.96751784102390959, -0.27164346220044866]], [[-0.14802828669590218, -0.75545607370279977], [-0.1617736717280458, 
-0.78808683888556974]], [[0.62984126406968532, 0.68468379288019787], [-0.44931109913842593, 
0.50509563243826583]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.22178901142117713, 0.78220205503048157], [-0.19432691201316632, 
-0.47599717633424965]], [[0.9604400709781753, -0.11235119971091745], [0.48115210800379971, -0.23549858583246408]], 
[[0.29759835928567369, -0.064835513256836785], [-0.40034926455202524, 0.6943561275363106]], [[-0.40837922199624632, 
-0.51911834236369581], [0.05077660647839588, -0.43451524712304113]], [[0.045587680552171106, 0.85952135991496204], 
[-0.95922963902852687, -0.24468246886520362]], [[0.024517678013268096, -0.11713897745648127], [-0.89584545461258402, 
0.34464557693057518]]]))
      arg1=numpy.array([[[-0.72174377779951682, 0.059081282269708657], [-0.28060648056160131, -0.34472903191770143]], 
[[0.45000848225058454, -0.44868290520037823], [0.033178078850591763, -0.55936996337690692]], [[0.73415989282999417, 
-0.060548680808505173], [-0.2633226210711046, -0.76836425513687123]], [[-0.54078705163433893, -0.28232559270260049], 
[0.86058019812413189, 0.38112955079541067]], [[0.81487042267811938, -0.10908664215490793], [0.21460784231934671, 
0.82135632704872363]], [[-0.12719008299748169, -0.69803730453549195], [0.58370363315997809, -0.39636106074606903]]])
      res=inner(arg0,arg1)
      ref=Data(0.467951879016,self.functionspace)
      ref.setTaggedValue(1,0.0507216527758)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=inner(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank4_array_rank4(self):
      arg0=Data(numpy.array([[[[-0.31697817736254796, -0.78512493043425224, 0.86570946770719948, -0.30099839115644333], 
[0.79216701686370627, -0.64757590962041389, 0.22166516295717975, -0.78151604833143207], [0.9299270318224977, 
0.37583789551257785, -0.27795332747002455, 0.030321789068909588]], [[-0.63557700398321737, 0.49708480036124181, 
-0.031456189129278167, 0.64744305850115391], [0.43557945873823445, -0.15913175846807137, -0.87379654553154551, 
-0.99635968128942665], [-0.74381743963289271, 0.0092210227067150985, -0.079122620124246534, 0.69175955942241041]]], 
[[[0.32758830606304534, 0.41050809422726497, 0.30475810003981429, 0.0076999648159608025], [-0.31049044422641492, 
0.76708837064764257, 0.36989303524440942, 0.732540298679071], [0.080394000245085362, -0.74137993588395035, 0.12037495618924576, 
0.9701774972840369]], [[0.86110281973938529, -0.10207435421952815, 0.599579739814319, 0.21391827887167825], 
[0.59258637353077659, 0.8743607189938456, -0.47242492532228542, 0.18241942826352853], [-0.32614433023749489, 
0.57117501379725066, -0.60148954891169515, 0.14186407850891891]]], [[[-0.090017617326749066, -0.81787757023927554, 
0.50194850135954949, 0.43729568597122648], [0.30001329703403812, 0.52685074553229927, -0.41448659468824522, 
-0.63884205886593537], [-0.81699433223447882, 0.33297526517646991, -0.65679546631436336, -0.99209250737241605]], 
[[-0.025407106832654902, 0.71882668575776743, 0.029164177994470242, 0.4335945896003035], [0.62615882293748926, 
-0.064641777287617552, 0.80949837899905441, -0.71257069800284278], [-0.66332513247846947, 0.46790565702256259, 
0.32238324548727038, 0.33792929708990704]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.45871757104084354, -0.92390424723128861, 0.70981795882027887, 
0.9122755330607204], [0.87651247288139134, -0.74201533966259059, 0.80923968843463534, 0.51021396948288089], 
[-0.50695332539024207, 0.22518571002521437, 0.86950505435134118, -0.70150424545332579]], [[-0.81653727987121427, 
0.99419500045760567, -0.9256942558562582, -0.2809423747068025], [-0.68848972363103766, 0.59520382606324596, 
0.78162105313584718, -0.13471258450578305], [-0.083630144001856532, 0.2947800693912872, 0.73753114211410309, 
-0.59612943790680584]]], [[[0.53804573061241956, -0.79508469152189587, -0.25014912391198441, 0.61392171013653418], 
[0.35306361411337561, -0.50804638007036051, -0.93226868699574816, -0.542574075476699], [0.86804064605140496, 
0.69368974773301728, -0.82547422824901906, 0.81553076863194018]], [[-0.76847923433434229, -0.53782577943601462, 
0.49816996275629433, -0.5478800397095891], [-0.52585732515942563, 0.013413299233611786, 0.92152733999531944, 
0.63154486570628654], [0.017101794716768426, 0.56950843544483454, -0.64774883015845686, -0.32021485467707511]]], 
[[[0.16293941619247643, 0.93395993437953018, -0.67608001806679852, -0.21085045980156747], [0.67027194252273992, 
0.49681751143982211, 0.79483357499627028, -0.11592472008619725], [-0.86129930437499036, -0.094583161957032313, 
-0.28432678466010763, -0.19049019692072577]], [[0.48364331545344297, 0.64759901158970523, 0.44158479096669567, 
0.91087816973335878], [0.55563207239419632, 0.89622760731857443, -0.77462076511970146, -0.55008113288067562], 
[-0.062702021637686789, -0.36614149730023793, -0.42024403056517046, 0.96091706517741415]]]]))
      arg1=numpy.array([[[[0.83204513377078149, -0.76761809303627948, 0.066787973351226215, 0.36538735059677441], 
[0.45589406391915777, 0.11700612355517404, 0.073381588375790896, -0.53159443168868004], [0.83486043858200754, 
-0.68086018885446764, 0.92550299371806477, 0.9480540815873697]], [[0.10151187608193468, 0.59095793696164689, 
0.28464761333376432, -0.32198775660949286], [-0.34983450636613389, 0.70293514495324416, -0.59006734629729984, 
-0.81306388188302225], [-0.81006578456865874, 0.064172736407334341, 0.16201739205143673, -0.51828853170275346]]], 
[[[-0.47991938987604277, -0.54751978313822169, -0.72721529033955079, -0.34570482811971059], [0.6715090631030658, 
-0.12670214080509767, -0.5423251791639756, -0.7703681239848823], [-0.30093583009759417, -0.63039091832447824, 
0.79082639288724321, -0.39133183119861781]], [[0.12960762045331387, -0.76038570255245674, -0.8769453751398788, 
-0.058169497772617262], [-0.086742262891742028, -0.44787983557982747, 0.27402763392760576, -0.9577235556573549], 
[0.087831250199435384, 0.18699861019026653, -0.85863725074202524, -0.60671857828320452]]], [[[0.76474343080406926, 
-0.73937248325562943, 0.4126559341378806, 0.094701247265092769], [-0.34730486532097005, 0.44334435178042852, 
-0.13423417115904335, 0.87043635234329342], [0.92676880819030205, 0.068160974047376088, 0.773410124229297, 
-0.53021280058437847]], [[-0.67131286948032898, -0.50541554507410313, 0.099774848887049572, -0.78109306352151386], 
[-0.35850590401163052, 0.5151123022771662, -0.0094539490115992386, 0.52174949414465055], [0.47020215680787247, 
-0.2845955605437942, 0.96430493828063502, 0.76221537591057897]]]])
      res=inner(arg0,arg1)
      ref=Data(-1.00383159382,self.functionspace)
      ref.setTaggedValue(1,-0.501913859213)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank0_constData_rank0(self):
      arg0=Data(-0.33268967754,self.functionspace)
      arg0.setTaggedValue(1,-0.815887580644)
      arg1=Data(0.364342327659,self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-0.12121293150329328,self.functionspace)
      ref.setTaggedValue(1,-0.29726238024033302)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([0.63733954801001458, -0.1137471801817509]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.24217122901848853, 0.28636999575661304]))
      arg1=Data(numpy.array([0.91539698377449619, -0.8268533778075331]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(0.677470940038,self.functionspace)
      ref.setTaggedValue(1,-0.0151031856936)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[0.58101836138163843, -0.4031750373694778, -0.23503877448260058, -0.53545018667304833, 
-0.56796733494055696], [0.76908984102446798, 0.27761452390094443, -0.56927162610601734, -0.31006302496106986, 
0.9664514293349693], [-0.50501017600427422, 0.41509818136608456, 0.093030074977553001, 0.48055231142577104, 
-0.99024450887323012], [0.92694069487505826, -0.65376988489792742, 0.44586065789300267, -0.83207646690634207, 
0.90668830552056279]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.45242936740850692, -0.88425836413059411, -0.024383780439757086, 
-0.96104048751697624, 0.51294061735355889], [0.043140504291506954, -0.78732099987146453, 0.95385490933785055, 
0.92795434196987747, -0.69527014896555839], [0.95526237819676552, -0.21077464553879066, 0.49323865221604857, 
-0.041671077519269017, -0.22937852253392421], [-0.095686031937835425, 0.5987587697331358, 0.41046872811285939, 
-0.34445951383335838, -0.52742604097793544]]))
      arg1=Data(numpy.array([[-0.27313386322466293, -0.22369424816845096, -0.58131494558876606, -0.94422708653680609, 
-0.11992632854747698], [0.96519954085287463, -0.26329887152840104, 0.59336450249171269, 0.93173643027628006, 
-0.71980388995820377], [-0.16679146346425555, -0.27614896464910932, 0.45998946501843396, 0.23809766214003836, 
0.70928429028903195], [-0.64045594106384507, 0.93869035601866679, -0.3322685784234225, 0.74415562371651656, 
-0.63743967955106995]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-3.13948632963,self.functionspace)
      ref.setTaggedValue(1,3.8814284317)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank3_constData_rank3(self):
      arg0=Data(numpy.array([[[0.96700564047927617, 0.95973772195526807], [0.0038308424877011582, 0.14196432362056899]], 
[[-0.73442868081424462, 0.2914322080354379], [0.019152009062897601, -0.78669189639973292]], [[-0.21798022996263944, 
-0.66732870929782129], [-0.14072594770292168, -0.83660203892684848]], [[0.67854000860556196, -0.13205985286041999], 
[0.74324138004734475, 0.86693037011480967]], [[0.50225303577765179, -0.5125810862302953], [0.56419197774058039, 
0.98421771434514516]], [[0.85221882222233547, 0.12249271224959801], [0.23820117654345196, 
0.75701957477272619]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.95560913385424739, -0.23797641808955738], [-0.80390789628183423, 
-0.6740055973133916]], [[-0.22922142447992711, -0.62239349187028314], [-0.4633862407522531, -0.31239129097033791]], 
[[-0.74813806971554908, -0.71477636183356119], [0.1468590807819341, 0.062926800112239389]], [[-0.29113130234083839, 
-0.62094920148047472], [-0.90516079338793332, 0.032339374252750241]], [[-0.31875715830740003, -0.064730138241973822], 
[0.98861925693113739, -0.18379179819802194]], [[-0.51191213285665382, 0.72382128089652298], [0.14874983242976714, 
-0.15683334876829691]]]))
      arg1=Data(numpy.array([[[0.68671044602605691, -0.61601906334393397], [-0.76160842339980817, -0.45795678830979147]], 
[[-0.65697615267992049, -0.33993292529940944], [0.13592434378357754, 0.73679015827871708]], [[-0.025809198391473442, 
0.23813509535487287], [-0.26430892494774616, -0.28498634143108248]], [[0.64517026622993323, 0.68363206376295471], 
[0.91912113331149525, 0.68666159866040899]], [[0.090269098889840027, 0.10779578098809472], [0.48326192604785945, 
0.27363361595817692]], [[-0.13146302737663196, 0.4226421948769854], [-0.24173240888320091, 
-0.37698665021687172]]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(1.68837255608,self.functionspace)
      ref.setTaggedValue(1,-0.361317507289)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank4_constData_rank4(self):
      arg0=Data(numpy.array([[[[0.29901440567790405, 0.9031194922891097, 0.025999982863851301, -0.0085334211021907969], 
[-0.36205096754265065, 0.11915636002224073, -0.58711429493738776, -0.84298737566654913], [-0.40963741898674244, 
0.94032728357639006, 0.89349879125861742, 0.53171776117860903]], [[0.84500845801135704, -0.23633741039288836, 
-0.084149210804974395, -0.21024329844301426], [0.96569229606788176, 0.58697168799131205, 0.9558971504815712, 
0.15021962201553118], [-0.60704011942235114, -0.57407911773795894, -0.94662067720255694, -0.043173494156841796]]], 
[[[0.77613103784608706, 0.35358500911921165, -0.64043610088055591, -0.33703728044364478], [-0.87655602692675738, 
0.2784735402321028, 0.57934321601059002, 0.7518380843530359], [-0.92039899983598406, -0.99759406430027497, 
-0.80885972946757856, 0.25257539420473907]], [[0.28771691902278484, -0.81145553520462999, -0.97338059936466825, 
0.052548984154607403], [-0.71838493503732237, -0.13341829480990519, 0.38414702381658983, -0.73867420481304014], 
[-0.76458975420853692, -0.87911036052671898, -0.41932776165557506, -0.92719117870187762]]], [[[0.87632105601352883, 
-0.40754902794606274, 0.57932627725153818, 0.29700291103489329], [0.27696132610243129, -0.55764776255987769, 
-0.80971730885116755, -0.78287436487835649], [0.027536084323056764, 0.58700948560074862, -0.20350017878627513, 
0.88550540356302698]], [[-0.43064219119292035, -0.66944031639992363, 0.54299150163346144, 0.64204612517360138], 
[0.90511690197912498, -0.92495620219436114, 0.53906581803937281, 0.30984732682034877], [-0.14688079983411328, 
-0.42649414272498531, 0.7311815068546168, 0.78148619773319283]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.93173347253756544, -0.65109717971547276, -0.46341530924930274, 
-0.84302490867508673], [0.90803919328439431, 0.97288378452257995, -0.73580808213542048, -0.98913655693295088], 
[-0.77753407695080567, 0.39219163958162784, 0.58652744665213885, 0.21157389070092236]], [[0.99896324273859882, 
-0.76755957208574754, 0.9275294804062808, 0.24783112362819915], [-0.81680122911144815, 0.85947950762483161, 
0.88629908134471846, 0.85527445337149599], [-0.9226219031005749, 0.25576031550731071, 0.97929940705262153, 
0.59813792024805856]]], [[[-0.18971902341240088, -0.16444486646712542, -0.13914511487771253, -0.33839969204684128], 
[0.59040989308934799, -0.087931156017434731, 0.91795042600887933, -0.72442856908461484], [-0.40369931243657575, 
-0.24305058590211037, 0.49222789473168049, -0.31610862233959924]], [[0.47238570938209534, -0.62393951485012589, 
0.50716893317002776, -0.021062968036011931], [0.78595074058645031, -0.73896679371025842, -0.406942871003803, 
-0.065571750030541276], [-0.44505109460536474, 0.58249103491574195, -0.22096363485769421, -0.066782452139382276]]], 
[[[-0.85248800668164848, 0.84790833155622147, -0.80105414038193956, 0.39156841745558957], [0.31858132709713027, 
0.56258056735771822, 0.059858200683915452, -0.34747710076497174], [-0.36461217957625736, 0.0066746374399917485, 
-0.2237271279211821, 0.49797443129070662]], [[0.40135001265003312, -0.64389635553832347, -0.69660568862034666, 
0.89522800033009875], [-0.10890875868678651, -0.4599840617941251, -0.20008830321673532, -0.082680996669195039], 
[-0.82118227598991367, 0.122629195898597, 0.24040621816356422, 0.29228571959435956]]]]))
      arg1=Data(numpy.array([[[[0.96771549795725864, 0.037359900176048289, -0.5954350987864907, -0.83698211107104714], 
[-0.57357279902264424, 0.5669393140813257, -0.34450077580311644, -0.74632845089441013], [-0.35478224007976622, 
-0.42531507718140471, -0.99815520673525593, 0.18387384340348878]], [[-0.44228171189793097, 0.19306845870863198, 
-0.81023238664850017, 0.44555560829611918], [0.042488860812775808, -0.36751275480234336, 0.75034399621512105, 
0.30834797293925931], [-0.099236591307724442, 0.0088709277917524698, 0.88007289887288764, -0.37597231686200927]]], 
[[[0.71897129043937458, -0.86597853444809192, 0.39941642297585389, 0.28932696715234885], [-0.6778465086312746, 
0.53164607266468056, 0.15258719511805108, 0.25509762688165627], [0.60791914995828167, -0.14276691775636396, 
0.58831735495593973, 0.74986407406594902]], [[0.28179048693667386, -0.33247434289894673, 0.39762153813817003, 
-0.97392960404583162], [-0.1609102007698251, 0.072438246693509178, -0.72269427573500011, 0.65969814794193393], 
[0.65965274024402065, 0.87024164771116408, -0.76682432225484431, 0.22224883928496397]]], [[[-0.98355907454056024, 
-0.42568711233801926, 0.25071877872545989, 0.75918366253916481], [-0.30179748821004604, 0.87787646961536159, 
0.10783491615373664, 0.37882286090190864], [-0.77004123634346344, 0.91064981370481912, -0.85282374653466331, 
-0.40315177036510152]], [[-0.31227172749684762, -0.21296591311923052, 0.45444771231558789, 0.85637856278212743], 
[0.40638168645853034, 0.37810190214103834, 0.81497970055887348, -0.36477796125600181], [-0.40072222126590873, 
0.02914156830482062, 0.085533392822153775, -0.62286801487541354]]]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-1.83246345289,self.functionspace)
      ref.setTaggedValue(1,4.13118192843)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank0_taggedData_rank0(self):
      arg0=Data(0.881455398501,self.functionspace)
      arg0.setTaggedValue(1,-0.308379780566)
      arg1=Data(-0.931432982036,self.functionspace)
      arg1.setTaggedValue(1,0.254621766347)
      res=inner(arg0,arg1)
      ref=Data(-0.82101663035753247,self.functionspace)
      ref.setTaggedValue(1,-0.078520204433200722)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([0.16709909048565597, -0.32042427060900747]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.94084662320929469, 0.42620451688374561]))
      arg1=Data(numpy.array([-0.0067239622503290075, -0.095741488158775256]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.84727991466252495, -0.72671493273723287]))
      res=inner(arg0,arg1)
      ref=Data(0.0295543285338,self.functionspace)
      ref.setTaggedValue(1,0.487431259804)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[-0.59285507566264051, -0.47520002555899565, -0.68483564116719342, -0.35716593649665862, 
0.13204124253991267], [-0.80815584374526939, -0.80340852882116365, 0.32081938942243848, 0.69658503961797735, 
-0.18749966448003064], [0.060119105026210207, 0.90726525358976251, -0.075765812458028048, 0.10866642153023043, 
-0.86481564258563037], [-0.23574277109004438, -0.65453562495164008, -0.95937246178106372, -0.17949404404798064, 
-0.47210068514118397]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.79669530621211937, -0.11900964643742329, -0.96003323961397058, 0.7871084173611913, 
-0.52059390752311052], [-0.68122534260154244, -0.84754682997019004, -0.65024778196675093, 0.2504267717604276, 
0.74116767365785963], [-0.40421446705476338, 0.53968966639394922, 0.48233473899696744, 0.1831776910000269, 
-0.46084692232247892], [-0.51015778935582534, 0.65710366347273963, 0.55467804635148021, -0.44099861336485735, 
-0.049752088645277803]]))
      arg1=Data(numpy.array([[0.27811978974420604, -0.34574572935507741, -0.072537995212597295, -0.0026819717578348001, 
0.45862131080365187], [-0.81571756275732832, -0.85843196798903798, -0.46746207611434909, 0.87189862294837606, 
-0.67777784569729249], [-0.6022914410958089, -0.3295668496915769, 0.64292115598117272, 0.6062781034541076, 
-0.2580700009527237], [0.058346007481130613, 0.11124458583850361, 0.20428599823424465, -0.90805616312429382, 
-0.83047458573595723]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.31568261499313421, -0.51379781398090363, -0.77765719558176261, 
-0.64830981854176728, 0.74388439618332192], [-0.56443373296554111, 0.048160305525875913, -0.39797487278496946, 
0.9773016060932691, 0.93838869754644461], [-0.99428542819784149, 0.38601068041332054, -0.30908598679465493, 
-0.26361139513097553, 0.52705552454466664], [0.40498164041729412, -0.14360398560872945, 0.71130467268539221, 
-0.0038535998638467017, 0.11782382233153932]]))
      res=inner(arg0,arg1)
      ref=Data(2.22161089579,self.functionspace)
      ref.setTaggedValue(1,1.96377456382)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank3_taggedData_rank3(self):
      arg0=Data(numpy.array([[[-0.63099897684983297, -0.56489482197356278], [0.87576068679994878, 0.57635076518298689]], 
[[-0.025656634022058089, 0.34281663660139361], [0.63860958454205163, -0.57576006027474724]], [[0.77430471641108456, 
0.20897079849639777], [-0.93671591554694755, -0.31542122668115646]], [[-0.57230787874269962, -0.68996602427383191], 
[0.30925657710009347, -0.03444930304938687]], [[-0.61158288831097773, -0.087306503756426901], [-0.11486359350471487, 
0.063503411374259411]], [[0.85134467175787698, -0.14556230835185446], [-0.56337662681433032, 
0.30279662287750519]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.45520052862702776, 0.92668278477422872], [0.94323760968945458, 
-0.52005237383246561]], [[-0.32678083302773642, -0.54888597470017242], [-0.00060725574578079922, 0.87555709700271667]], 
[[0.80419737911843492, -0.53047314352854724], [0.83270188932360933, 0.83066709908869263]], [[0.19151836066385552, 
0.19688232230826075], [0.62209069061508515, -0.98427703050341275]], [[-0.61985880706926877, 0.23056481696150444], 
[0.92576528848433015, -0.53736632833367026]], [[-0.24828334391352547, 0.11617811407597545], [0.68051988188473556, 
-0.44481686842644508]]]))
      arg1=Data(numpy.array([[[0.3624297843769877, 0.24162259052245405], [0.56010899163227124, -0.17364424406575552]], 
[[-0.36104068262474343, -0.79823476718227737], [-0.90991851631562293, 0.26592778488442947]], [[0.80285485331432982, 
-0.7520192183400265], [-0.66568659982073708, 0.46258338899654561]], [[-0.54938027450769233, 0.39552092225871083], 
[-0.9719272769837084, 0.014499930391750793]], [[-0.15413176943367746, -0.73343318088535692], [-0.30653831076426186, 
0.028183265606130048]], [[0.34991669521925917, -0.43784659634409429], [0.11910057220329651, 
0.10598952402746575]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.86907204319284204, -0.39706660589599441], [-0.91393853231426792, 
0.40971196863579329]], [[-0.63120886266547527, 0.75156060212798526], [-0.41086531054703257, -0.34660332161981211]], 
[[0.68511982467319288, -0.7837293681559756], [0.81881601067748, 0.6062833136506558]], [[0.22883779528070702, 
-0.51196618625056556], [0.079972284998143683, 0.77884964075776497]], [[-0.22589859699003689, -0.062651417883206939], 
[0.82455276325167981, 0.7756546399164439]], [[-0.82260777179176725, 0.21182506900096332], [0.48899508942132797, 
-0.78748367970741806]]]))
      res=inner(arg0,arg1)
      ref=Data(0.231204149557,self.functionspace)
      ref.setTaggedValue(1,1.20540052616)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank4_taggedData_rank4(self):
      arg0=Data(numpy.array([[[[-0.71948751414580037, -0.64781588667774659, -0.088032923972571897, 0.054079744210093317], 
[-0.64775883661869282, -0.54486598614577897, -0.0019816261638903043, -0.84199073796448354], [-0.55027210883717981, 
-0.40280920665623388, -0.8673821158295103, -0.18432839999127437]], [[-0.3408271613641829, 0.50574412143448799, 
0.059464953272122978, 0.9675736205602492], [-0.65590706272406174, 0.53037732687981065, -0.51596692546482092, 
0.33909732806059179], [-0.29029794622158156, 0.55191338400453205, 0.30287253189259711, 0.94497734413693113]]], 
[[[0.92184474824529339, 0.72491675260537058, 0.49949022044145197, -0.87315675263095671], [-0.53164765916718748, 
-0.9138962661714165, -0.32585173514235644, 0.053860995934299938], [0.22204320693690738, 0.61326278789361099, 
0.97162159232513989, -0.0046997300297617617]], [[-0.13056640348089199, -0.71505402281774288, 0.20927319679314071, 
0.36386054817114721], [-0.67572006875081825, -0.94312980857763828, 0.50951874798691454, -0.094459509225578886], 
[-0.88334465053270694, -0.18710291810863389, 0.23413554940078574, -0.64741492323569028]]], [[[0.76943810054018402, 
-0.4977166800238535, 0.99343345931789417, -0.92268077779255164], [-0.30687214928582351, -0.16938779390059655, 
-0.5923552823543683, 0.19555802252634669], [-0.49409987795201205, -0.68189354687907833, 0.10148799526762642, 
0.43762912889100924]], [[0.21330953271871644, -0.86281038300098079, 0.55281060980824859, -0.26208773952328013], 
[-0.048334510182155288, -0.99019303342560305, -0.81418813459348294, -0.18561915813057617], [-0.16737744524309939, 
-0.42892713904953328, -0.479182258539395, -0.9143551786234243]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.03959829715553842, 0.03206020292258227, -0.042319158754688058, 
-0.38958679047303701], [0.56661911363801054, -0.039128056995030081, 0.84608017122893053, -0.65540424958151067], 
[-0.036702649306371793, -0.64902506672584415, 0.56737244475128112, -0.24009348542488085]], [[0.89968083834723278, 
-0.30530432784448647, -0.69894302577411183, -0.25158939138697201], [0.16065434957059921, 0.96387165874786085, 
0.91252381153183837, -0.058815691077375831], [0.85705094158458639, 0.26961554961272638, 0.034269667345734778, 
0.55409192189872125]]], [[[-0.31195338416821694, -0.021823979889771472, 0.42167375872207979, 0.88427101866238367], 
[-0.54686992122438105, 0.89571500114111835, -0.84064098700206435, -0.41567856327135488], [-0.12855262867354345, 
-0.23040118503656126, 0.55179419467831692, -0.21537636882255429]], [[0.75773586034712292, -0.85240514106722798, 
-0.11863725681909409, 0.17084848248916762], [0.18824768436294281, -0.79883126927693882, 0.99681510160602982, 
-0.39530428502963511], [0.6816256286594744, 0.24227154919409633, 0.26379431003658138, 0.56490989421835569]]], 
[[[-0.39704563818569771, 0.78730977171969618, -0.69079582594092925, 0.19189128195849259], [-0.9497781597720647, 
0.066726644605307639, 0.78629697089477624, -0.079938536408641125], [-0.015828071684629741, -0.72832510900241743, 
-0.15571479670578059, -0.42344871341450974]], [[-0.82302900024278269, -0.99641257125367311, 0.90133253519800971, 
-0.34491520623514904], [0.48521882511753556, 0.99595220501109005, -0.60051252255359677, -0.19778582998287031], 
[-0.72707202883707533, -0.19129846963009722, -0.33819109789849344, -0.23324369379622434]]]]))
      arg1=Data(numpy.array([[[[0.72322433571756628, 0.52813398097811026, 0.75264927586505981, -0.96759706336385798], 
[-0.13506154883816612, -0.2911797036987176, 0.50932331677321829, -0.19649761526093834], [-0.50954953754172583, 
0.35616462121150994, 0.87524942312308185, 0.49599073575056951]], [[0.76983479571548696, -0.17545934753128578, 
0.50377725683895935, 0.28631997036438239], [-0.61551784906673856, 0.68538123829428721, 0.39202754006468155, 
-0.604088134739454], [-0.64700888595967099, 0.11889396368766336, -0.062392289770082154, 0.70214215098075194]]], 
[[[-0.42307398665297735, -0.5357279563431403, 0.81136625962235964, -0.19318176719443447], [0.6300522992970059, 
0.24619600335789804, 0.85548841887729266, 0.9880069925635746], [-0.044494129332561139, 0.43313315777013961, 
-0.5307913979443708, 0.13393668624930566]], [[-0.19708084293410821, -0.19480708208471609, 0.24157893643900707, 
0.49968205486697381], [-0.47579868179502993, 0.99926234424174432, -0.72946288247806157, -0.93301550882543283], 
[0.26613750470382258, 0.070150542673697913, -0.14805872695411804, -0.90248617680211241]]], [[[0.89874171645780998, 
-0.86569451176748324, -0.18210015358902831, 0.88062483996876129], [-0.56333783885140187, -0.3167602108978631, 
0.73926967443023095, -0.45990916601129439], [0.66571371183380457, -0.9307496933129793, 0.072044702559182161, 
0.71602956349901792]], [[0.039966388538660969, 0.99635807561281386, 0.249412766853228, -0.14000007463395892], 
[0.57620787245708183, -0.13295201607213314, -0.9881374355679875, -0.80523611225659231], [0.47975776045195651, 
0.6121401752847857, 0.44743784755963079, -0.51286431779566]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.11660132854414718, 0.76551438653868575, 0.45165217791484524, 
0.39891469653591205], [-0.46674121385468359, -0.2266240009703766, -0.69557886350899234, 0.64151488786120692], 
[0.038555555357479676, 0.48145240590380145, -0.31631093841892688, 0.42092526904365379]], [[0.71817896538735249, 
-0.76113031768184825, 0.7368269484389729, -0.58130101635246634], [0.96279927397144882, -0.10975832598306723, 
0.032676463171715131, 0.57147873310322961], [0.9300459812828914, 0.30211993558246464, 0.98967760125242288, 
0.10894115879188382]]], [[[0.29238488322175304, -0.28208487685506345, -0.19967608765976874, 0.22062245724995866], 
[-0.63439712392920833, -0.18854322697625014, -0.045485050115293024, 0.81695318081607149], [0.27421677698727098, 
-0.95097960149364602, 0.12570413607312925, 0.21013648358932602]], [[0.74810615394730418, 0.50953042705177154, 
-0.41037826258966992, -0.65480548162821517], [0.11274082235793803, -0.11803620030634243, 0.38685100292541907, 
-0.85993327141888476], [0.17075030304707739, 0.072417695665841997, 0.73975432270814845, 0.49746237446166286]]], 
[[[-0.57242534664272093, -0.32236428554701702, -0.003192002502402147, -0.42991257848940667], [-0.1653132722502908, 
0.15975009875641444, -0.66375512279297721, -0.55959172674133595], [0.73394120398754614, -0.72514934767582373, 
0.60456422834647205, -0.82859775418142156]], [[0.50913648818921819, -0.33080185257139827, -0.77644970547352732, 
-0.67429987359265975], [0.60278325007821287, 0.48425140340277983, 0.9670782068086119, -0.17447848176492564], 
[-0.34635473356842161, 0.36222881689712927, -0.75534713009921939, -0.50105783084562305]]]]))
      res=inner(arg0,arg1)
      ref=Data(-0.77939206013,self.functionspace)
      ref.setTaggedValue(1,1.72876212013)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank0_expandedData_rank0(self):
      arg0=Data(-0.0187655677641,self.functionspace)
      arg0.setTaggedValue(1,-0.275610243781)
      arg1=Data(-0.647076500109,self.functionspace)
      arg1.setTaggedValue(1,-0.270112545792)
      arg1.expand()
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array(0.012142757911322563)+(1.-msk_ref)*numpy.array(0.074445784594044939)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([0.40133674020326593, 0.28355033879731639]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.027395527514068707, -0.45154247492764021]))
      arg1=Data(numpy.array([0.53573886017388173, 0.32925511671867969]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.52583130208736062, 0.30465341427949566]))
      arg1.expand()
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(0.308372087639)+(1.-msk_ref)*(-0.151969382583)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[-0.74015282640300795, 0.34260538181516598, 0.35578315276374028, -0.95859180704925939, 
0.52581097408891986], [-0.656480805888358, 0.91643376274250121, 0.74255171441524648, 0.19624447830454428, 0.94146447698777314], 
[-0.37655932383524138, -0.57865376055953632, -0.80120219108716451, 0.45945090909136499, 0.11656067726762531], 
[-0.89582199378009797, 0.82514048047505817, -0.36753370461031332, 0.13609522362061499, 
0.76483029342018449]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.68324089636415986, 0.52618933126277256, -0.85269275292056634, 
-0.10806111014019271, -0.17171007568465901], [0.83656177589621783, -0.15833013767102622, 0.69219282053392539, 
-0.14612311986032722, -0.5569042904577779], [-0.56346269838408225, 0.45194026242467111, 0.058264389149598861, 
0.76994266937276512, -0.1785561307108896], [-0.1678876254876438, -0.58285139923586082, -0.54244343407735407, 
-0.4210730251986925, 0.94886500245540417]]))
      arg1=Data(numpy.array([[-0.61722112040621724, -0.25320231919739533, 0.30400790930944321, 0.055148331687939534, 
-0.75652109819832791], [0.54007098693804756, 0.31105907263661803, -0.48388576637186542, 0.69204595415912551, 
-0.50335121974716412], [-0.32278781945265123, -0.5901729909386304, 0.18387656012102616, 0.27128102614766259, 
0.39016873373812855], [0.024998373971658605, -0.046212595698132075, -0.32649346732550288, -0.53127005870416855, 
-0.11967705502189041]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.15300639798969806, 0.21998595473356142, 0.84946801589096554, 0.34003925506328025, 
0.62546611322835766], [-0.50975518079599724, 0.4260634853576466, 0.3266420091384945, -0.96241591847404218, 
-0.12686209990361186], [-0.36538447347006486, -0.29024523203095676, 0.84864448070847875, 0.68358190413775088, 
0.13966761225455349], [0.11897725756956645, -0.81737951700249223, 0.36624280738599424, 0.67407674126763162, 
0.73573710415460991]]))
      arg1.expand()
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(-0.357783614791)+(1.-msk_ref)*(0.592878316393)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank3_expandedData_rank3(self):
      arg0=Data(numpy.array([[[0.04527592716388873, 0.34114590892642283], [-0.83054331936940384, 0.88153655281857657]], 
[[-0.13272917086387204, -0.71480603910144835], [-0.36607775733150905, 0.81480599636560447]], [[-0.20924221699405821, 
0.6259068731994597], [-0.90102796240691441, 0.83174040570989227]], [[0.70507295639071121, 0.87049430517490101], 
[-0.97020053016372709, -0.28925410812528018]], [[0.43196389756055753, 0.72295081379266812], [-0.28947731243336605, 
-0.72685840948306124]], [[-0.78373848663475099, 0.046307104214429851], [0.54827234142990355, 
0.88675506628355438]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.31644447368303252, -0.85407269091911386], [0.53325611642083071, 
0.9546087439464499]], [[-0.68175983978548538, 0.84967530069626807], [0.18363148543931396, -0.7246707761683977]], 
[[0.96696479484545872, 0.48959920604011176], [-0.7612429871592914, -0.24031010014873466]], [[0.026182145365968967, 
-0.96597809023288228], [0.38450430959240212, -0.88589959379295946]], [[0.57899562041495067, 0.1800154262219722], 
[0.89900074445892364, 0.84428089795492656]], [[0.81166460582694921, -0.54596599158660064], [0.64838049543887233, 
-0.75801153584345138]]]))
      arg1=Data(numpy.array([[[0.83107737053232289, -0.57262746540360876], [0.23672209655988996, -0.65974531141512038]], 
[[0.49733062911711823, 0.95719570405319865], [0.73772840274932494, 0.98101851234263693]], [[0.76022547585316058, 
0.097409290221732947], [-0.69907318728119039, 0.023178103052963195]], [[-0.21937776377214324, 0.020723991796819607], 
[-0.72495879308969391, -0.32417201408052709]], [[-0.92308332596971088, 0.27957346298220309], [0.69348531065060226, 
-0.11821664244865615]], [[0.50253077680478753, 0.31165113944410061], [0.3604186273887604, 
0.65274899907837991]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.29172978614112588, -0.42914637757965024], [0.063644524529885338, 
0.086087423541371555]], [[0.57550431699199112, -0.30604392771451594], [0.58767735617014449, -0.85096865557142465]], 
[[0.84181879782358382, 0.17131229932521164], [-0.16705983065639618, 0.043525372154784359]], [[0.13709589872467887, 
-0.65145122773254927], [-0.94597700784239147, -0.84173516251097169]], [[0.67397514792368463, -0.13865663347316781], 
[0.865377253349773, 0.30106762242922702]], [[0.8107329793189042, -0.083298267334609433], [0.53318109276700598, 
-0.51299279925058561]]]))
      arg1.expand()
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(0.140254474489)+(1.-msk_ref)*(5.32746250104)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_taggedData_rank4_expandedData_rank4(self):
      arg0=Data(numpy.array([[[[-0.17219019122513646, 0.74923535995801038, 0.44855192652037057, 0.22446998606488955], 
[0.42150478907573108, -0.69600362435931684, -0.81509822772369134, -0.32053361903207866], [0.24583278150303212, 
0.66729797894198928, -0.7840034414732493, -0.28743652730633018]], [[0.79161256720759599, 0.93793735933909028, 
-0.62732091535675449, 0.80286361609286194], [0.43887341300565619, 0.34731867557098406, -0.88874607822934681, 
0.80539326716710979], [-0.55223848657954955, -0.80498249389513465, 0.74978160983840558, -0.62486506889592408]]], 
[[[-0.82494456945128447, 0.1675852867282086, 0.67141114272113089, 0.35030758400002604], [-0.11510861238394865, 
0.81624866920338657, -0.37271115026067747, 0.47663545588551903], [-0.79353673977044381, 0.29558103585503193, 
0.29816574112623639, -0.051939520957481999]], [[0.53374928695136226, -0.041045366353953394, -0.58783406731129184, 
0.12582271415553015], [0.35886028239008572, 0.16105479270527412, 0.4252594750066212, -0.62028400596880751], 
[-0.52371456130575211, 0.76380972005845593, -0.22029649010521113, -0.56088486005554228]]], [[[0.1105456325448253, 
0.5563154467611835, 0.35099905470096049, -0.48360534039540704], [0.087924419225235528, 0.82666464333500489, 
-0.32409842144976109, 0.60474469759264005], [0.16307198096912479, -0.18178701604843739, -0.13977445370548058, 
0.94146164916448938]], [[-0.388306180864749, 0.091137342123327469, 0.49181546389088182, 0.066875559531317963], 
[0.35990149086040013, 0.059802926465624573, -0.85688282413133821, -0.62016312390795969], [0.21149473035866007, 
-0.20923197467125121, 0.72731389542375902, -0.96375574939132291]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.62357905207504949, -0.24110643046902624, 0.29557766399251362, 
0.39695143960789081], [-0.042543292322763548, 0.61176154765913937, -0.014188243381858534, -0.69177081888540126], 
[0.17934408391736989, -0.33153743779073541, 0.92435407278191373, -0.9764741359080289]], [[0.17640623596665561, 
0.7550768826616816, -0.45965092079492953, 0.35716078538347618], [-0.93565430474083788, 0.14875055151392536, 
0.82128823897980419, 0.47053065292357199], [0.45489949767011684, 0.39744201553623326, -0.26238127278422119, 
0.5382293212479754]]], [[[-0.69698174048168959, 0.99944459330451552, -0.087249343163490645, 0.17466487869659142], 
[-0.81706275091802727, -0.90657454369626445, -0.034351130983405254, 0.42109769268674224], [0.86139225877878811, 
-0.2900152904196982, 0.91813724474352987, 0.79561378037072328]], [[0.82729261072575566, 0.63851228593111098, 
0.3478062159618478, -0.030396302216106008], [0.63198048549681829, 0.92299575623951613, 0.41497588626123849, 
0.19957783908141313], [-0.53367837591227896, -0.9793534246234139, -0.51841273753799899, 0.4884097440280506]]], 
[[[0.247509530236109, 0.30736862214692873, -0.81698988024644259, -0.58281213669564358], [0.81215595363096327, 
0.050056170515610177, 0.070432786246830803, -0.39931857445719032], [-0.24395275157618279, -0.56401371110809273, 
-0.13923789131922115, 0.80771036953933284]], [[-0.44355131187071195, 0.63000636811961219, 0.040018766730615773, 
-0.65524273137740896], [-0.35072144889062251, 0.81991147433203748, -0.87099808670290857, -0.076152155315448056], 
[-0.076153659397851126, 0.88216448223960309, -0.66497998646877776, -0.54902569044694816]]]]))
      arg1=Data(numpy.array([[[[-0.60789824874002307, -0.51562664877431752, 0.21924446551440835, 0.8703450692197332], 
[0.51184620770479983, 0.26425223736047521, -0.21337514707539706, -0.069977593528601334], [0.93613615457302823, 
0.75287872780637932, 0.36386379031791849, 0.5701902195682087]], [[-0.50223938891886033, 0.73498594836321907, 
0.35380332498527078, -0.78265499395511884], [0.036819094732386315, -0.10011656727314899, -0.84764438215540117, 
0.81335552800575472], [-0.59083404845535559, -0.39873063316752821, -0.68489338312021419, -0.60490738770483565]]], 
[[[0.5753236950903815, 0.25731663378135905, 0.45235386956902013, -0.97082714319410801], [-0.85206290600360446, 
-0.98780592154101576, 0.37624066833805081, 0.66171643403418945], [0.61398004772528258, -0.080252729007424861, 
0.18906395487962935, 0.84126295801149142]], [[-0.97800457387511797, -0.81320925193281846, 0.82592731949981379, 
0.71955870320011073], [0.59737224004683553, 0.51161797334844583, -0.13248103099738207, -0.32931005562438953], 
[-0.65750256079126146, -0.33178166455459812, -0.23781293470048182, 0.89306626552475299]]], [[[0.58433021489119419, 
0.23757868845617747, -0.95524616093622905, -0.87222988474876417], [0.42686927709766653, -0.35521759624874938, 
0.064919829605669443, -0.52924950853283059], [0.15995918164941503, -0.0041293342833377267, 0.90153711754039589, 
0.082167001797124994]], [[0.96435421044386671, -0.7426144108006496, 0.8669703937182891, -0.046732673618262499], 
[-0.121039764882934, 0.72645104809998839, -0.87445452760622722, -0.30003806736680949], [0.33879134768109154, 
-0.10703582791862254, 0.37921488165206618, 0.084292040123948642]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.5973322177732201, -0.32088834326496385, -0.67193384997313688, 
-0.48299214737059315], [-0.31487850978337617, -0.48589729872964127, -0.19301558619195669, 0.1426966178818958], 
[0.60704775566127922, -0.90637345231921351, 0.23958934734156934, 0.63253041366672313]], [[-0.56158012586256012, 
-0.44537522392546891, 0.20449863811989899, -0.68271322535815315], [0.95407293092519874, 0.28684191568381068, 
0.24436974251182186, 0.22529497402326015], [0.072578633281052518, 0.39863646953050336, -0.76454189193586042, 
0.65338678790100491]]], [[[-0.3191285731579534, 0.41505217711876563, 0.62400009529440048, 0.15392724765383337], 
[0.71016822409436475, -0.40967190474257431, -0.43905102253482475, 0.8458810832883954], [0.68358857558344877, 
-0.14476459549402798, -0.94587325731277749, -0.86859611130133252]], [[-0.80498929881911185, -0.2248316936964998, 
-0.81138869604240327, 0.80620314143989824], [0.6797731765549262, -0.75176789378913322, 0.50343888486979838, 
0.65898762310945069], [0.80724306196174944, 0.57053326175887831, -0.17220508145285085, -0.45556191154908832]]], 
[[[0.76023104743407433, -0.11080122606304532, -0.27297296139648286, 0.77520863803503381], [0.99392177894899092, 
-0.55560245098486649, 0.0028916940138923319, -0.67982119277838771], [0.61410843140613136, -0.081387643793096487, 
0.87145881378238621, -0.90544166094346612]], [[-0.011905541882043957, -0.39751189743906035, 0.2131841841697335, 
-0.10764270144202426], [0.3830344175432403, 0.427601367590978, -0.19281425363619054, 0.76675351091177624], 
[0.56460868335711689, 0.32249394605652637, -0.94319147952853788, 0.41054081445047763]]]]))
      arg1.expand()
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(0.440181339453)+(1.-msk_ref)*(-3.12147400145)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=inner(arg0,arg1)      
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank0_taggedData_rank0(self):
      arg0=Data(0.947623884385,self.functionspace)
      arg0.setTaggedValue(1,-0.462852356181)
      arg0.expand()
      arg1=Data(0.755425687204,self.functionspace)
      arg1.setTaggedValue(1,-0.719263469767)
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array(0.7158594240722892)+(1.-msk_ref)*numpy.array(0.33291279169610455)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=inner(arg0,arg1)      
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([-0.28803269141787369, 0.8093167451154808]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.93444246867654313, -0.11189552426228655]))
      arg0.expand()
      arg1=Data(numpy.array([0.16892998241268353, -0.67274394370695889]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.44791939446120788, -0.59261889571098791]))
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(-0.593120296313)+(1.-msk_ref)*(-0.352243502705)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=inner(arg0,arg1)      
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[0.12356938074285129, -0.01920853546063106, 0.99797588924346758, 0.1814521358364265, 
-0.96710490129619142], [0.24615420100213536, 0.78570966815461607, 0.45805035539134975, 0.40295861892577944, 
-0.7654626347054212], [0.23950246235795292, 0.31210794834585487, 0.22275778451495087, -0.62905226860453189, 
0.63832679117724633], [0.80164142837524022, -0.34006151475669655, 0.39347903316006327, -0.62722978075401503, 
0.37051259062934805]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.31139111044816081, 0.41243883762207312, -0.84412309880631886, -0.46229908275624587, 
-0.15822034299642085], [-0.099292473975716389, -0.63230154729246624, -0.76315551850768037, 0.53096541743163228, 
-0.84298130021434936], [-0.15014173680517051, 0.034953258779667706, -0.89299670292643385, -0.26608465750854715, 
0.74625836561061765], [0.87546481899795703, -0.83516731700560309, 0.60269148078892232, -0.057040970150197712, 
-0.14218848069905943]]))
      arg0.expand()
      arg1=Data(numpy.array([[0.44501421718614864, 0.71292554717498646, -0.30385959094627824, 0.98699091601403888, 
-0.90107098306819089], [-0.8453365841560796, 0.93754664347239913, -0.41277677340974606, 0.34312439958775487, 
-0.9046224544010868], [0.11893886787781782, -0.58091610957755258, -0.54290974321439256, -0.87562899765253221, 
-0.93565951335733111], [0.91830731262945808, -0.017258798431958322, 0.66237080960218164, 0.027813429785085386, 
-0.80399945250475913]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.30185639425274813, -0.72520880311568647, -0.72450223051087437, 
-0.012041129145349272, 0.99848200111939067], [-0.96055468361538243, -0.6897296912989217, -0.033108939065968235, 
-0.76005254094112451, 0.15722870764104546], [0.015723189856038822, -0.75275917186204477, -0.97380545980431488, 
0.47134205008990238, -0.15117294499088429], [-0.029051632790475734, -0.036371003881288777, 0.79496823371646652, 
0.37192883298626156, 0.79224516523553623]]))
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(2.32589053931)+(1.-msk_ref)*(1.22760588894)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=inner(arg0,arg1)      
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank3_taggedData_rank3(self):
      arg0=Data(numpy.array([[[-0.72975553555244721, -0.30546473305749067], [-0.56080228686254463, 0.19196525884555093]], 
[[0.16330874959104102, 0.82682868035669044], [0.27102526496232038, -0.24517608521403722]], [[-0.83963475376113794, 
-0.66791774272530535], [-0.37639043923508719, 0.82300098289090817]], [[0.91722211429625577, -0.97918297646753905], 
[0.067107067385226316, 0.88843622243940512]], [[-0.27552289134186125, 0.65791342377921613], [-0.37551267096396979, 
0.85793206764837748]], [[0.26155040462053258, -0.42818822215694752], [-0.83990328379325829, 
0.88309580060131143]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.011360458137832818, -0.93646335053858154], [-0.36443713736367744, 
0.24907142903244806]], [[-0.81458916334694664, -0.47474847029843281], [0.54743876815462422, -0.348883676466061]], 
[[0.50093316712876934, -0.4398043195632535], [0.68508091721063669, 0.59472418355503232]], [[0.88692467780296891, 
-0.50364025455918671], [0.32956321010155509, -0.23251415760310246]], [[-0.75739253707570287, 0.46370814604003407], 
[-0.088561023319100984, 0.66778445028017219]], [[-0.59846571127691983, 0.32548006374158889], [-0.44850244173499854, 
-0.76853835165957074]]]))
      arg0.expand()
      arg1=Data(numpy.array([[[0.65932979334231034, -0.27399540365817776], [0.037000180188222043, -0.39510157224621634]], 
[[-0.51915331006812204, -0.92135429550669912], [0.22128097647462841, -0.24745332734434022]], [[0.089641980651052666, 
0.62427799480017687], [0.79140866831201162, -0.49032008447991382]], [[0.43859468414231784, 0.0055652663596135188], 
[-0.16002111883096459, 0.058452192644826706]], [[-0.24674430968099137, -0.23153624391525907], [-0.54760056166033566, 
-0.76415349509388109]], [[0.063522680563744327, 0.39803355915777372], [-0.88998945484879832, 
0.41881303708055273]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.74580862645566648, 0.57672974773199592], [0.8998859370820409, 
-0.96269851399667927]], [[0.90808913426374782, -0.72717934642920667], [0.11611247866138652, 0.83821916685618558]], 
[[-0.97426941724298688, -0.24005765475720908], [0.88273806186553161, -0.368820129330788]], [[0.43200272334145207, 
-0.89080597605032863], [-0.15785396132005336, -0.76755327631693415]], [[0.61597152516899856, -0.20381125615084161], 
[0.83281013639816326, -0.49443205158207304]], [[0.99858308154502273, 0.63729322460413185], [-0.21441490370741789, 
0.47254648903811192]]]))
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(-1.54637460343)+(1.-msk_ref)*(-2.40064701404)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=inner(arg0,arg1)      
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank4_taggedData_rank4(self):
      arg0=Data(numpy.array([[[[-0.81947004030336634, 0.67058302145475457, -0.7459191324332497, -0.56685991265962676], 
[0.12625168566568967, 0.030640316129083134, -0.80449388144868728, -0.54200006075166196], [0.37775985547602953, 
-0.58498443113456133, -0.17282485220725397, -0.28544889654748795]], [[-0.36450637785497308, -0.56874337767686622, 
0.23117225537130115, 0.75495037184165636], [0.74992076822906606, 0.2870324821934227, -0.59510427587645198, 
-0.40183700613297124], [-0.45935306206812787, -0.12423126734949341, -0.005585510270659011, -0.28793976196914128]]], 
[[[0.92819919528865658, 0.46125809060586631, -0.44644818733618563, 0.10817651477090151], [0.039610465897389924, 
0.88463912849634641, 0.53094392433172799, -0.60775835248092358], [-0.36883877523305575, -0.4881930782326025, 
0.087030421658376822, -0.25222758948775592]], [[0.0026877458846366054, -0.15441076036465473, -0.042951128074558698, 
0.031626744478122415], [-0.91123851640226916, -0.57186463991433922, 0.58154204490668304, -0.50444661480738895], 
[0.78513112280855313, 0.9177559897504306, 0.46148528105097553, -0.031119549525124857]]], [[[-0.83706243187153095, 
-0.70136607266997797, 0.6175299692859868, 0.75859021965132967], [-0.35343073433069128, 0.59196857522832147, 
0.36159706695864502, 0.48992389809307757], [-0.64481325885474128, -0.032372741534755223, 0.15557596853361955, 
-0.72915099530812588]], [[0.20172441264738095, 0.10514721422311157, 0.85542478318553217, 0.20724572965733001], 
[0.6433427982803821, -0.69931369735789684, -0.37169802067700441, 0.64658561555033045], [-0.26070451960146324, 
0.025881750025058325, -0.13025418980636583, 0.1546947923403974]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.64605720721984894, -0.98626487832849263, 0.10152535427338094, 
0.92440732936073178], [-0.9009473786125628, -0.6632565822463008, 0.29433507869379172, -0.84263803545895599], 
[0.92530974890630135, 0.69159562053418711, -0.82219687747500081, 0.87754115294324286]], [[-0.35773552298525213, 
-0.82132454786210563, -0.40762919301021472, 0.27582076375453823], [-0.4479722336819616, 0.37973644829065445, 
0.77302808123184286, -0.22293907876333519], [0.55219125633268673, -0.47116803616196412, 0.48683043453580765, 
-0.92754419214957196]]], [[[-0.18006692590834628, 0.97227675841825945, -0.63651749202640384, -0.41921849139239975], 
[0.7276855458936482, 0.52195096650135198, 0.99536511192686361, 0.57947097632231626], [-0.91302738669032957, 
0.89503790517274484, -0.94895340049321208, -0.78812045603988645]], [[0.071410373876193534, -0.26780532355721864, 
0.62497681913903302, 0.3126820909449517], [0.20395007656176722, -0.071701973664488827, 0.37328219942910312, 
0.046021504655175738], [0.64320814200428367, -0.098661644869028509, -0.83633938821080078, -0.64262333359039969]]], 
[[[0.86500058646653843, -0.27029036986884036, -0.99414894726393643, 0.72691841266018864], [0.67847131914979331, 
-0.026877426027215545, 0.7357347318800751, 0.42889865183659426], [0.21639030857873132, 0.35003493653688378, 
-0.52608516781767989, 0.68304932394382045]], [[-0.51790042448954954, -0.49002499388001097, -0.4261929129502422, 
-0.1479751689299269], [-0.21976416284030553, -0.38837447911267109, 0.2768587166449068, 0.88566207159183197], 
[0.89901331895690029, 0.9513101075593664, 0.88490792821692366, -0.066907535776567606]]]]))
      arg0.expand()
      arg1=Data(numpy.array([[[[-0.81320652563427687, 0.085883992488720251, -0.56182282586392995, 0.098026469908133684], 
[-0.85944703879505435, -0.58655421446612421, 0.082604797948233166, 0.9360870739483298], [0.14420666951064609, 
0.46665845263423589, 0.12564509386997491, -0.26476090845384515]], [[-0.61363302518729501, 0.38807696904498012, 
-0.37734796865857168, 0.84307896756626666], [0.42023635130695292, -0.14153911868266955, -0.22072250942077987, 
-0.61141600747289537], [-0.82872232098466303, -0.8248660915121564, -0.96742618451047013, 0.35691401321500549]]], 
[[[0.11131946679252192, -0.81042203379181621, -0.50103550921701001, -0.27216300561259543], [0.74569190185733669, 
-0.65644473655201874, 0.3826715613541456, 0.18908664033895595], [0.16619726086801401, -0.11816966386278449, 
0.15258125582452342, 0.43588839717124639]], [[-0.67517568975866848, 0.96203361468349557, 0.57152246085306158, 
-0.039767883547068505], [0.50166558466475797, -0.83608235880060278, 0.80517977832576104, -0.30314822128254049], 
[-0.43040765167681383, -0.48543495725866159, 0.96433198011584831, 0.71339694132471432]]], [[[-0.38795524306170481, 
-0.77689177550729993, 0.53450913145705958, 0.97271691497725676], [-0.33048696586776249, -0.99367367729553924, 
-0.19926373850410051, 0.68567184375197066], [0.25235581540383722, -0.16508876097042302, 0.071002736285278756, 
0.98527408412884965]], [[-0.44496258754141671, 0.20764414070614401, -0.45421078466839848, -0.10926523567651425], 
[-0.79856163653683221, 0.56705711384424951, 0.21172579482906229, -0.30481362841012394], [0.25946996108918108, 
-0.97058507882352374, -0.071749416196154492, -0.24108145066049924]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.90160447289175094, -0.59180913174068994, -0.28132325842873107, 
-0.71726721556211515], [-0.7782137611242339, -0.20694868939842825, 0.75927104544848323, -0.45016422772749687], 
[-0.5761210418346967, 0.88003496134895087, -0.58419876136588655, -0.26366101557894006]], [[-0.5268788915130751, 
0.61048942970567688, -0.55578110348084109, 0.29211930415991771], [0.82205545269991798, 0.48404861282324108, 
0.81294003027352946, 0.74542604533258272], [-0.28794793196945956, -0.024676412518504165, 0.70284834650167616, 
0.33656053351445903]]], [[[-0.1650003816398391, -0.80541076697460201, 0.011019828659870612, 0.13881175713585225], 
[0.61430717382059674, -0.60620181826668196, -0.59588214763457281, 0.62500528414733325], [-0.49257636209383837, 
0.20343351200232918, -0.35175015572622259, -0.15894223788426709]], [[0.86656006370664174, 0.87033077110041712, 
0.37280770378914774, 0.89532157858757833], [-0.27553078134143338, -0.83589256562132008, -0.046707659230116105, 
-0.71614111174634565], [0.037431640389452348, 0.46807013453244917, 0.01934442605321296, 0.85101447436985866]]], 
[[[-0.19154482128448946, 0.29386808951224053, -0.20897443911445168, 0.4590938049142268], [0.7982039891093109, 
-0.15224852359600805, 0.85480424889048612, -0.77027973937766103], [0.386441395994366, -0.26415966145170722, 
0.65632489949275441, 0.56457918115387384]], [[0.98166215351674491, 0.99280990187113094, 0.93442735143135791, 
-0.82425261653930604], [-0.56078813969082542, -0.42429815167520135, 0.1078164045366059, 0.18120716020529404], 
[0.76552553397932432, -0.55087389170436873, 0.27712920571932709, 0.42066112856497928]]]]))
      res=inner(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(0.357435097673)+(1.-msk_ref)*(1.87227288471)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_taggedData_rank0(self):
      arg0=-0.023476564344
      arg1=Data(-0.835490373492,self.functionspace)
      arg1.setTaggedValue(1,0.358151040007)
      res=outer(arg0,arg1)
      ref=Data(0.019614443512064931,self.functionspace)
      ref.setTaggedValue(1,-0.0084081559355938442)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_taggedData_rank1(self):
      arg0=0.840924093893
      arg1=Data(numpy.array([0.95729089620662178, -0.84719011818084411]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.11169275196751127, -0.42561661206573587]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.80500897948503769, -0.71242258248674273]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.093925126242749429, -0.3579112638473948]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_taggedData_rank2(self):
      arg0=-0.2887604825
      arg1=Data(numpy.array([[0.79599173434014459, 0.41879737506080095, 0.52972900265768419, -0.8584742457402994, 
-0.22793982405076796], [-0.92049505867550341, 0.62172748210755358, 0.082167410547219433, 0.59577995617280388, 
-0.80084490358879235], [0.42462588636657284, -0.4684433268133823, -0.82595290014283407, -0.73968476605655931, 
-0.17503191832994514], [-0.57972597684596727, 0.22812723867905182, 0.54891646233502644, 0.99028828842144656, 
0.27714349968748841]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.72391217640352634, 0.10391970323314204, 0.3540033780359626, -0.048308558208530528, 
-0.89671717882718438], [0.96196914894104313, -0.24185994027728608, -0.74301018213813719, -0.33547842252260773, 
-0.019729268498721142], [-0.42888477806301095, -0.45836444067559645, -0.5463948090956785, 0.82989552711758741, 
-0.081069197113341707], [0.09763270127468382, 0.97601622522901965, -0.52427261654028889, 0.17546739075337348, 
-0.39797905550434653]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.22985095727375743, -0.12093213209212486, -0.15296480240146734, 0.24789343741345318, 
0.065820013573774788], [0.26580259728164041, -0.1795303277166416, -0.023726701115358205, -0.17203770760805229, 
0.23125236076764921], [-0.12261517582903395, 0.13526792107435234, 0.23850255796719269, 0.21359172994409939, 
0.050542301189786386], [0.16740195279159625, -0.065874131512265524, -0.1585053825158384, -0.28595612397828474, 
-0.080028090691388235]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.20903722934562133, -0.030007903646817838, -0.10222218624815457, 
0.013949602577155521, 0.25893648522382218], [-0.27777867559794983, 0.069839593051794741, 0.21455197869632778, 
0.096872911155834504, 0.0056970330910549715], [0.12384497545021098, 0.13235753705014672, 0.15777722870974761, 
-0.23964103283473842, 0.023409580474304123], [-0.028192465927817486, -0.28183491612457473, 0.15138921371350414, 
-0.050668048416890828, 0.11492062409217212]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_taggedData_rank3(self):
      arg0=-0.776324255672
      arg1=Data(numpy.array([[[-0.70841723318340444, -0.28211567113780767], [0.52390424049298767, -0.31883374308253432]], 
[[0.55390489720629899, -0.75935106867485369], [0.87220517253522223, -0.66921903363972435]], [[-0.057386758732438814, 
-0.048139918400846859], [-0.59494231344414028, 0.083153310888181675]], [[0.5039335543023471, -0.56213524133929793], 
[0.026510246712620145, -0.52605691835092694]], [[-0.64697406038124061, 0.82846094762663336], [-0.44757190472775732, 
-0.56359199577693442]], [[-0.49071963706566502, 0.29181736111438483], [0.56844779991521821, 
-0.29110745294721663]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.79897608620515603, 0.24598629677692085], [-0.51002686750009563, 
-0.88804956515871258]], [[-0.1114159599467861, -0.171444691377866], [-0.48137121591369425, -0.44541949459992747]], 
[[-0.96693365342661486, 0.1146562906433346], [-0.55502396846273472, 0.53006523579873899]], [[-0.98613846602429067, 
0.52453841157930214], [-0.43307574882585609, -0.95042026076495145]], [[-0.18077515697101121, -0.762218130056592], 
[0.47171378941193565, -0.40920580632348202]], [[0.0062301729719855814, 0.87885834591929002], [-0.36298242963992622, 
-0.45308434716000634]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.54996148125623434, 0.21901323840942952], [-0.40671956954405675, 0.24751836828162571]], 
[[-0.43000980703668551, 0.58950265318264727], [-0.67711403136156423, 0.51953096817180933]], [[0.044550732758381936, 
0.037372186320642153], [0.46186814865222453, -0.064553932181919516]], [[-0.39121584145185112, 0.43639922281965932], 
[-0.020580547546852557, 0.40839074557982274]], [[0.50226165586447624, -0.64315432851946086], [0.34746092579741877, 
0.43753013662415413]], [[0.3809575569885742, -0.22654489565925506], [-0.44129981515749572, 
0.2259937767297828]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.62026451542284422, -0.19096512875082358], [0.39594622828466886, 
0.68941441767156819]], [[0.086494912175655969, 0.13309667242281584], [0.37370015089606329, 0.34578995760703063]], 
[[0.75065404878050179, -0.089010459491784696], [0.43087856919688178, -0.41150249963899205]], [[0.76556321062571031, 
-0.40721189194060842], [0.33620720835677187, 0.73783430151381857]], [[0.14034013917948632, 0.59172842247579083], 
[-0.3662028564553797, 0.3176763930106859]], [[-0.0048366343951837289, -0.68227905123680654], [0.2817920645121838, 
0.35174036856556856]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_taggedData_rank4(self):
      arg0=-0.833021296853
      arg1=Data(numpy.array([[[[-0.90855434421590231, -0.7723719863146703, 0.72452648785716645, -0.12247760395825291], 
[-0.77922956246495589, 0.98780212712468085, 0.60797043692928421, 0.34279771003579862], [0.58333174697288559, 
-0.18669064834824378, 0.038625811592457593, 0.7462952669972418]], [[0.82279543277627432, -0.69468389442798406, 
-0.17106121884786618, -0.52208034248189361], [-0.78671020417224735, 0.14748043848641967, -0.21932177320002721, 
0.049811837058228781], [-0.63271712466388341, 0.44615141775229938, -0.14001640059409715, -0.99908113089779893]]], 
[[[-0.53252632348026041, -0.38943828967962224, 0.089860628719042657, -0.43089130288981514], [0.33837711195657638, 
-0.90464306477818623, -0.069778867864798588, -0.062511204849629909], [-0.43506299745272581, -0.47053229885405701, 
0.24608199676000475, -0.30466610687173867]], [[0.25012418501222622, 0.64548943592837138, -0.083426680964325373, 
0.90328575508751952], [-0.27567408237424695, 0.75276653790846426, 0.64586793988014368, -0.64036916647235631], 
[-0.79259725475604137, -0.55365979968476142, -0.2356225716304805, -0.42169261004187009]]], [[[0.028913197845884087, 
0.33050019183055013, 0.75606755812103321, -0.65458983784489355], [0.9966370561747373, -0.63651566152647776, 
-0.38593523367509253, -0.7359031213892413], [0.19393437329974317, 0.019275777009044148, 0.7165775422934102, 
-0.64290086820039871]], [[-0.90843329061350375, 0.052322297520196681, 0.12195889401306181, -0.59477765381256886], 
[-0.92717376181603117, 0.71027598521011415, 0.97685194710925871, -0.4318117150162355], [-0.86786903458205522, 
0.22606469989064593, -0.85281204371690156, 0.42737096105985017]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.33882616595902526, -0.36968245848952686, -0.78797202708069536, 
0.31585745312913938], [-0.92658667826390317, -0.72554711315388287, 0.81439972903102942, 0.32603003013108212], 
[0.5642109828154056, -0.90759915868128727, -0.81842195934021156, 0.4691408416636742]], [[0.18510078344764302, 
-0.34270527486269886, -0.46651003503375699, -0.96535634636413992], [0.32191966811628259, -0.29573924983708344, 
-0.67679295996670596, -0.38538800299229758], [-0.14071356277867375, -0.8213461327307332, 0.19388983531961324, 
-0.56811817286295208]]], [[[-0.90498197923747048, 0.006942996396168466, -0.33034079556955032, 0.52108226477623476], 
[0.38962528862123236, 0.072744507290621874, -0.44124075071020896, -0.88555483209134378], [-0.10558141086212092, 
0.89100179440722593, -0.70166501173136164, -0.87199116641624275]], [[-0.70107408257430737, -0.74645840185196222, 
0.86149140808786928, 0.3936124106603347], [0.7095744615974946, 0.49617920205234078, 0.0057299436413136284, 0.8718734384946174], 
[-0.68848971134095516, -0.11648523655054999, -0.74514849740010836, -0.50320077525644957]]], [[[0.47831932795138177, 
0.47063903817806385, -0.6546331097821172, 0.64919953752100867], [-0.1852875522062023, -0.10546798523549938, 
-0.90895441310898173, -0.043226598698658414], [0.060843433321798868, -0.86629658629967921, 0.26469146778309494, 
0.14226678022175254]], [[-0.9390782992831157, -0.87677919928693426, 0.97697194554398825, 0.12716597602129842], 
[0.87179742343938393, 0.78814812909324994, -0.68970423325908303, -0.88378227959616718], [-0.48107962739859378, 
0.7706674033436911, -0.31728035278840516, 0.14238157976435506]]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.75684511807971622, 0.64340231369239875, -0.603545994518774, 0.10202645248469243], 
[0.64911482067037451, -0.82286020897107348, -0.50645232181882183, -0.28555779297209299], [-0.48592776835859564, 
0.15551728599729067, -0.03217612366472989, -0.62167985114893554]], [[-0.6854061184556175, 0.57868647863895417, 
0.14249763836582119, 0.43490404395547161], [0.65534635452667145, -0.12285434612833472, 0.18269970793907961, 
-0.04149432110485185], [0.52706683972830193, -0.37165363260860812, 0.1166366436035159, 0.8322558593213607]]], 
[[[0.44360576859362788, 0.3244103891129439, -0.074855817471519176, 0.35894163193574319], [-0.28187534062727554, 
0.7535869390101575, 0.058127283001634729, 0.052073164931651865], [0.36241674235061161, 0.39196342580240123, 
-0.20499154407307529, 0.25379335545330234]], [[-0.20835877297306282, -0.5377064470216496, 0.06949620196900326, 
-0.75245627113140778], [0.22964238160802194, -0.62707055763568598, -0.53802174887441878, 0.53344115351916566], 
[0.66025039303861999, 0.46121040434850302, 0.19627862018734724, 0.35127892489020007]]], [[[-0.024085309565731673, 
-0.27531369840868947, -0.6298203777740965, 0.54528727562803003], [-0.83021989302595145, 0.53023110183172228, 
0.32149226885710358, 0.61302297253747873], [-0.16155146315043159, -0.016057132761913829, -0.59692435357664375, 
0.53555011497590321]], [[0.75674427785085752, -0.043585588134577309, -0.10159435605345905, 0.49546245251784166], 
[0.77235548947561417, -0.59167502232292624, -0.81373847581385805, 0.35970835483893265], [0.72295338868568293, 
-0.18831670947548024, 0.71041059462849609, -0.35600911221918163]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.2822494121747523, 0.30795336099457132, 0.65639747988226504, -0.2631159852261678], 
[0.7718664363736597, 0.60439619712704518, -0.67841231843376404, -0.27158995851265821], [-0.4699997646033206, 
0.75604942818693643, 0.68176292194215848, -0.3908043123291538]], [[-0.15419289467597194, 0.28548079250432262, 
0.38861279437853197, 0.80416239557306046], [-0.26816593941655659, 0.24635709342947687, 0.56378294921211691, 
0.32103641404404426], [0.11721739455062843, 0.68419882065215243, -0.16151436206446457, 0.47325453712377702]]], 
[[[0.75386926197255244, -0.0057836638619785863, 0.27518091792863797, -0.43407262397074409], [-0.324566163213794, 
-0.060597723802130986, 0.36756294238079507, 0.73768603466274141], [0.08795156379988206, -0.74222347027502433, 
0.58450189802849328, 0.72638721229199477]], [[0.58400964145573597, 0.62181574595717659, -0.71764068999265518, 
-0.32788752078551631], [-0.59109063821336927, -0.41332784236488646, -0.0047731650829788948, -0.7262891424260467], 
[0.57352659221085545, 0.097034682815511003, 0.6207245676519404, 0.41917696238131802]]], [[[-0.39845018687968292, 
-0.39205234193251054, 0.54532332207329337, -0.5407970406618029], [0.15434847702943852, 0.087857077837297487, 
0.75717838398785964, 0.036008677306479622], [-0.05068387573068435, 0.72164350577826453, -0.22049362975846915, 
-0.11851125775935588]], [[0.78227222271487418, 0.73037574564331076, -0.81383843706557668, -0.10593196626077769], 
[-0.72622582026615579, -0.65654417660914166, 0.57453831483415008, 0.73620946068447024], [0.40074957510490078, 
-0.64198235977532103, 0.26430129094562038, -0.11860688822321271]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_taggedData_rank0(self):
      arg0=numpy.array(-0.888224462909)
      arg1=Data(0.900050292802,self.functionspace)
      arg1.setTaggedValue(1,0.871825604029)
      res=outer(arg0,arg1)
      ref=Data(-0.79944668791524431,self.functionspace)
      ref.setTaggedValue(1,-0.77437682888968606)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_taggedData_rank0(self):
      arg0=numpy.array([0.72173822750820249, -0.45381424956523486])
      arg1=Data(-0.261461866632,self.functionspace)
      arg1.setTaggedValue(1,-0.571685203163)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([-0.18870702418411708, 0.11865512079562178]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.41260706522349899, 0.25943889146094601]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_taggedData_rank0(self):
      arg0=numpy.array([[0.1473211091196287, 0.43262373380570707, -0.057057931636928849, 0.51939061478398774, 
-0.79022098709708466], [-0.0080394628796294221, 0.1687711721732581, -0.59803343605897807, -0.83143652329234907, 
-0.91430940230783486], [0.99620918222588783, 0.6177742919692526, -0.94312173920627917, 0.77339225160336156, 
-0.28870936455533558], [0.43685177364287542, -0.46131718868149951, -0.5474858328210126, 0.31065742398798379, 
-0.11434607498628191]])
      arg1=Data(0.0288312974103,self.functionspace)
      arg1.setTaggedValue(1,0.731784996176)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.0042474587118418675, 0.012473103536102665, -0.001645054196640316, 0.014974705286950749, 
-0.022783096298849337], [-0.00023178814530158592, 0.0048658918592105393, -0.017242079856314312, -0.023971393680819596, 
-0.026360726302962083], [0.028722003215616802, 0.017811234344197128, -0.027191423357166618, 0.022297902020790697, 
-0.0083238655546308462], [0.012595003410110623, -0.013300373067355386, -0.015784726873983169, 0.0089566565837122577, 
-0.0032967456956288668]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.10780737727381472, 0.31658755738884164, -0.041754138284764909, 0.38008225905377341, 
-0.57827186202137382], [-0.0058831583126301564, 0.12350421158349906, -0.43763189571979577, -0.60843277301843479, 
-0.66907790247191168], [0.72901093260609451, 0.45207795788661542, -0.69016233831897378, 0.56595684588244555, 
-0.21127318123722541], [0.31968157350491766, -0.33758499715541196, -0.40064191807757354, 0.22733444182522591, 
-0.083676742046625938]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_taggedData_rank0(self):
      arg0=numpy.array([[[-0.22998941661966765, 0.27990685337714472], [0.055177199192475879, 0.4530358761401585]], 
[[-0.50329573976454434, -0.34267018160207185], [0.95713581137929515, 0.16613940465905341]], [[-0.10489406307193194, 
0.59230199936860517], [-0.16281074906528992, -0.12071493497086361]], [[0.50267329707196828, -0.89492086425710848], 
[-0.32855892752585802, 0.52912075989094687]], [[0.21767057810094603, -0.2084209947877147], [-0.3929867157169229, 
-0.4302326258862994]], [[-0.068602442248387296, -0.31339977711477274], [-0.75011712808319553, 0.75943941199765863]]])
      arg1=Data(-0.224314795568,self.functionspace)
      arg1.setTaggedValue(1,0.632736758892)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.051590028971941322, -0.06278724859349441], [-0.01237706215689831, -0.10162264994154047]], 
[[0.11289668097574133, 0.076865991733462688], [-0.21469972386076203, -0.037267526591954855]], [[0.023529290314321546, 
-0.1328621019031361], [0.036520859892922114, 0.02707814596004459]], [[-0.11275705787040308, 0.20074399071574855], 
[0.073700628660142722, -0.11868961508594569]], [[-0.048826731227973881, 0.046751912837973308], [0.088152734797147012, 
0.096507543522550504]], [[0.015388542808441453, 0.070300206934689177], [0.16826237023835361, 
-0.17035349644885719]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.14552275805138484, 0.17710735519750606], [0.034912642181784137, 
0.28665245193071054]], [[-0.31845371514275711, -0.21682002007581974], [0.60561501111157678, 0.1051225084282119]], 
[[-0.066370329495144709, 0.37477124736572837], [-0.10301634567634634, -0.076380776703319872]], [[0.31805987277086056, 
-0.56624932711484877], [-0.20789131090773502, 0.3347941546758571]], [[0.13772817609373531, -0.13187562472702], 
[-0.24865714079032811, -0.2722239972728811]], [[-0.043407286960318536, -0.19829955920906894], [-0.47462668041271827, 
0.48052523212222648]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1) 
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank4_taggedData_rank0(self):
      arg0=numpy.array([[[[-0.56243952285768684, -0.38055747729341571, 0.26506266794294953, 0.28759030643565464], 
[0.53869559247210419, 0.038458054338470271, 0.18456058086171589, -0.72022970202879066], [0.70380702845467669, 
-0.052760327461741863, 0.082725647492424725, 0.18308427320570964]], [[0.6699910070553261, -0.080476552624917375, 
0.4849508811170129, -0.69076449078778634], [-0.94053959960140743, -0.65504644081841867, -0.89361482549631299, 
-0.561448772960774], [-0.51023825339381657, 0.96060244621206703, 0.44984722169057356, 0.35655778859333065]]], 
[[[-0.5531318630297124, -0.67650955166659976, 0.038878069861318254, -0.049563466219434416], [-0.92777147909327518, 
0.036436810859565849, -0.89914023013227551, 0.48835239325227797], [-0.3198924957319953, 0.90827863784513996, 
-0.26851065142907715, 0.53219627059437591]], [[-0.53139068088798957, -0.47363736615698171, 0.9170621229026017, 
0.64598154152809539], [-0.32352615807098228, 0.71103699225776507, 0.42590554589010199, -0.97048596167757695], 
[0.7755897929291844, 0.48855694469214339, 0.049284017202407382, -0.05031239455679315]]], [[[-0.65561173606876988, 
0.35551653774465453, 0.099134687584160996, 0.72863976547059917], [0.16212842749830858, -0.53586825498873036, 
0.84419447644335133, -0.97244158766475675], [-0.85359197357514094, 0.29199160162483073, -0.44577988392461454, 
0.76164730894513766]], [[-0.91998685969351635, 0.013658352307469146, -0.93935439467434301, -0.88050165229668886], 
[-0.74991650340850113, -0.616200761731581, 0.83526597511329848, -0.57312444878586932], [-0.065982912206490996, 
0.34387825972430131, -0.057559071354035929, -0.3552305352401135]]]])
      arg1=Data(0.748269432855,self.functionspace)
      arg1.setTaggedValue(1,0.00729678277696)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.42085630278413033, -0.28475952770319019, 0.19833829221278518, 0.21519503549129099], 
[0.40308944546075454, 0.028776986508565694, 0.138101041168842, -0.53892587066262931], [0.52663728602135917, 
-0.039478940307057814, 0.061901073331744626, 0.13699636527636228]], [[0.50133379086744423, -0.060218144390797118, 
0.36287392077610825, -0.51687795375836121], [-0.70377703277170178, -0.49015122876508499, -0.66866465866521951, 
-0.42011495492066586], [-0.3817956884880731, 0.71878944762652308, 0.33660692544594062, 0.26680129425087368]]], 
[[[-0.41389166544344175, -0.50621141854676399, 0.029091271285637578, -0.037086826758359316], [-0.69422303848045352, 
0.027264551796943398, -0.67279915005846724, 0.36541916833241345], [-0.23936577635604855, 0.67963714121497287, 
-0.20091831286044434, 0.3982262015653627]], [[-0.3976234034126509, -0.35440836335336567, 0.68620955459741284, 
0.48336824171422432], [-0.24208473481363002, 0.53204724693586058, 0.31869210127311615, -0.72618498013851684], 
[0.58035013448348516, 0.36557222792231131, 0.036877723600876523, -0.037647226940604003]]], [[[-0.49057422192146105, 
0.26602215806887469, 0.074179456454888104, 0.54521886406450848], [0.12131574649388198, -0.40097383524557983, 
0.63168492210784855, -0.72764831528682095], [-0.63871678195691217, 0.21848839014632457, -0.33356346092257549, 
0.56991739990014822]], [[-0.68839804573720154, 0.010220127534847899, -0.70289018015311, -0.65885247199220331], 
[-0.56113959669431335, -0.46108419450589772, 0.62500399748136204, -0.42885150624851237], [-0.049372996294892474, 
0.2573135903751726, -0.043069693677762561, -0.26580815113700657]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.0041039990234716288, -0.0027768452459593354, 0.0019341047102621521, 
0.0020984839948213867], [0.0039307447211766851, 0.00028062006853248251, 0.0013466984677381777, -0.0052553596852213495], 
[0.0051355270035340584, -0.00038498064872980093, 0.00060363107983589189, 0.0013359261714603317]], [[0.0048887788410018427, 
-0.00058721992314290689, 0.0035385812370079721, -0.0050403584393183933], [-0.0068629131514238483, -0.0047797315874751837, 
-0.0065205132679208846, -0.0040967697366875565], [-0.0037230976995120215, 0.0070093073850293732, 0.0032824374594967315, 
0.0026017247308000656]]], [[[-0.0040360830515450289, -0.0049363432450522538, 0.00028368483056565708, -0.00036165384667658927], 
[-0.0067697469496059122, 0.00026587149392756258, -0.0065608309453043315, 0.0035634013321722072], [-0.0023341860533371448, 
0.0066275119213124326, -0.0019592638967789833, 0.0038833205812373347]], [[-0.0038774423681424757, -0.0034560289759006956, 
0.006691603103801437, 0.0047135869864586427], [-0.0023607000981095668, 0.0051882824788905043, 0.0031077402518642026, 
-0.0070814252504539594], [0.0056593102430344856, 0.0035648938995956202, 0.00035961476790210699, -0.00036711861406980856]]], 
[[[-0.0047838564241218457, 0.0025941269495409467, 0.00072336428096377936, 0.0053167260912967097], [0.0011830159174258598, 
-0.0039101142537233393, 0.0061599037161197013, -0.0070956950284753956], [-0.0062284752113375098, 0.0021305992897540996, 
-0.0032527589793379876, 0.005557574966031602]], [[-0.0067129442728445345, 9.9662029878842749e-05, -0.006854264968524872, 
-0.0064248292915665261], [-0.0054719778262319625, -0.0044962831053548891, 0.0060947543813904747, -0.0041819646069575231], 
[-0.00048146297736222885, 0.0025092049629285187, -0.00041999604051415183, -0.0025920400513916434]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_taggedData_rank1(self):
      arg0=numpy.array(-0.490695779658)
      arg1=Data(numpy.array([-0.38101831143130149, -0.96888199323213775]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.89919035726945618, 0.95352392693101518]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.18696407739170537, 0.47542630506550931]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.44122891342116921, -0.46789016674784278]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(arg0*1j)
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)     
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_taggedData_rank1(self):
      arg0=numpy.array([0.92947119363174102, 0.19269891600491529])
      arg1=Data(numpy.array([0.32040652162906014, -0.39372224308492343]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.27311607223449208, 0.82535605058609218]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.29780863210595676, -0.36595348323951027], [0.061741989398825331, 
-0.075869849449488505]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.25385352165980618, 0.76714467350943472], [0.052629171063106765, 
0.15904521626603799]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(arg0*1j)
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)   
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_taggedData_rank1(self):
      arg0=numpy.array([[-0.75993582440916163, 0.027687316844169141, -0.66568939935561344, 0.1141542489676417, 
-0.52554853090396425], [-0.76444851585395335, -0.24278348935372707, -0.66164245062582427, -0.72791122768260519, 
-0.10407864094215391], [-0.6256667379790799, 0.2368394036526229, -0.15274735342330548, 0.019232120181438939, 
-0.17080614493143176], [0.69963978173863239, 0.43954624285616561, 0.51248380637847957, -0.65834079290520253, 
-0.12613540004594825]])
      arg1=Data(numpy.array([0.17903242100362937, -0.6951720673447368]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.93205693098620945, 0.31268570213884206]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.13605315045136118, 0.52828615810384383], [0.0049569273657061683, -0.019247449289789817], 
[-0.11917998480308735, 0.46276867595951787], [0.020437311560527954, -0.079356845251021269], [-0.094090225842637445, 
0.36534665871849814]], [[-0.13686106852596461, 0.53142325514480859], [-0.043466115878706635, 0.16877630021119935], 
[-0.11845544977431563, 0.45995535024459222], [-0.1303197093677409, 0.5060235529915621], [-0.018633451062641276, 
0.072352563990187704]], [[-0.1120146308418381, 0.43494603970975471], [0.042401931824984897, -0.16464413786588847], 
[-0.027346728485271393, 0.10618569346071642], [0.003443173037115773, -0.013369632745953342], [-0.030579837649371024, 
0.11873966088716816]], [[0.12525820395511819, -0.4863700334678655], [0.078693028001588561, -0.30556027033993238], 
[0.091751216581094433, -0.35626442716082746], [-0.1178643459992674, 0.45766013002128286], [-0.02258232604448742, 
0.087685806815297251]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.70830345224527813, -0.2376210668358385], [0.02580615556501907, 
0.0086574281077596172], [-0.62046041855344625, -0.20815155724389403], [0.10639825895181579, 0.035694401490579231], 
[-0.48984115079865997, -0.164331511393743]], [[-0.71250953758379842, -0.23903212092878914], [-0.2262880339811579, 
-0.075914925836288227], [-0.61668843194050038, -0.20688613423880001], [-0.67845470490425297, -0.22760743332268193], 
[-0.097007218657759617, -0.032543902920653831]], [[-0.58315701962093403, -0.19563704326990752], [0.22074780770506774, 
0.074056295225265023], [-0.14236922944799196, -0.047761913455016132], [0.017925430912669918, 0.006013609002551829], 
[-0.15920105123837597, -0.053408639357513561]], [[0.65210410776317118, 0.21876735639721048], [0.40968212214303679, 
0.13743982556997014], [0.47766408375325647, 0.16024635883224128], [-0.61361109897825072, -0.20585375307620526], 
[-0.11756537385554432, -0.039440736127931059]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_taggedData_rank1(self):
      arg0=numpy.array([[[-0.51651482132844717, 0.16838870046166465], [0.030266288621693782, -0.15936743382357843]], 
[[0.95307110799080386, 0.48912409835056403], [0.033818634759858579, 0.33152576519796617]], [[0.22373931465651697, 
0.65165184632789686], [0.36596253834426951, -0.20685984660022738]], [[-0.79631201418088371, 0.037523980552965019], 
[0.012502480984663444, -0.87951823950197539]], [[0.010089749864709452, 0.112488850669201], [-0.23698144740411142, 
0.28901557411917933]], [[0.22892951029030884, -0.94640460942043347], [-0.7005406469439841, 0.42849809538716555]]])
      arg1=Data(numpy.array([0.93877390799400429, 0.64515413554375312]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.81537082954752194, -0.93152156750384663]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.48489063735533122, -0.33323167304969042], [0.15807891839442872, 0.10863666648168124]], 
[[0.028413202049861936, 0.019526421271846582], [-0.14960998865753658, -0.10281655900227703]]], [[[0.8947182886447026, 
0.61487776678753403], [0.45917694130260273, 0.31556043484497581]], [[0.031748051916534313, 0.021818232073766482], 
[0.31122773819559735, 0.21388521845677513]]], [[[0.21004063079199864, 0.14434634413437697], [0.61175375042874802, 
0.42041588359316495]], [[0.34355608230085555, 0.23610224506689478], [-0.19419462659993569, -0.13345648551208308]]], 
[[[-0.74755694153516516, -0.51374398913197294], [0.035226533867197991, 0.024208751235808752]], [[0.011737002933593229, 
0.0080660273118127561], [-0.82566877484927603, -0.56742482950086059]]], [[[0.0094719939111752686, 0.0065094438518193262], 
[0.10560159794847979, 0.072572647211798708]], [[-0.22247199950163327, -0.1528895608399069], [0.27132027998699276, 
0.18645959287954064]]], [[[0.21491305103038685, 0.14769482031179892], [-0.88845995372915954, -0.6105768476652631]], 
[[-0.65764928084025198, -0.45195669549240763], [0.40226283157459702, 0.27644731831165142]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.42115111834016605, 0.48114469600284437], [0.13729923438185671, 
-0.15685770620398556]], [[0.024678248860795185, -0.02819370061940403], [-0.12994355671959096, 0.14845420176440532]]], 
[[[0.77710637994023757, -0.88780629245822151], [0.39881752182378311, -0.45562964679942303]], [[0.027574728278310551, 
-0.031502787662343538], [0.27031643818584267, -0.30882340046512163]]], [[[0.18243051059387827, -0.20841799710107506], 
[0.53133790651655155, -0.6070277493581383]], [[0.29839517847308383, -0.34090199736614052], [-0.16866748472250054, 
0.19269440855864906]]], [[[-0.64928958758132516, 0.74178181567192214], [0.030595959151396169, -0.034954397183681832]], 
[[0.010194158291867152, -0.011646330684520728], [-0.71713351654490176, 0.81929020910910366]]], [[[0.0082268877171151437, 
-0.0093988196096958729], [0.091720127484993738, -0.10478579050208024]], [[-0.19322775935726277, 0.22075332935520825], 
[0.23565486842170857, -0.26922424063652212]]], [[[0.18666244471331708, -0.21325277627351646], [-0.77167071147073729, 
0.88159630526018795]], [[-0.57120040843047404, 0.65256872154141887], [0.34938484749536636, -0.39915521748746524]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_taggedData_rank2(self):
      arg0=numpy.array(0.381023055405)
      arg1=Data(numpy.array([[-0.96374377174039627, 0.28269410150435359, -0.46693824882349722, -0.83571237714671121, 
0.064058933359271819], [-0.87813861155325101, -0.74732919749024562, -0.11473281004766678, 0.96760879638301756, 
-0.58385849784974186], [-0.35243939721348894, -0.72382016954892947, 0.2452968006296361, -0.85789289204818564, 
-0.7916163089717565], [0.78045878538468183, 0.78157511640125077, 0.095751483573530471, -0.079265870680946371, 
0.62423275441157511]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.72717910379825668, -0.66469615433699292, -0.33817718108446582, 
0.95947103901385011, 0.92276765711495479], [-0.65912580097769768, -0.87067394864099312, 0.01312159110347566, 
-0.080334948985110621, 0.78966260421412571], [0.030758256099536085, 0.69508587589403303, -0.056452090936532473, 
0.28707858327623481, -0.68615848490485032], [0.28053988108826866, -0.17449695656411768, 0.90034439998096905, 
0.81080675219235432, 0.57175251504841462]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.36720859653631188, 0.10771297030023252, -0.17791423825230882, -0.31842568338042998, 
0.024407930514551458], [-0.33459105684334939, -0.28474965422129173, -0.043715845839592905, 0.36868126003486007, 
-0.22246354877503202], [-0.13428753597147039, -0.27579217256548388, 0.093463736457037996, -0.32687697093865159, 
-0.3016240647530502], [0.2973727910251468, 0.29779813887992357, 0.036483522830772809, -0.030202124236212127, 
0.23784707136993746]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.2770720039560679, -0.25326455964160499, -0.12885330280513987, 0.36558058685791323, 
0.35159575214309013], [-0.25114212658495938, -0.33174684817295058, 0.0049996287340247272, -0.03060946771813226, 
0.30087965819693796], [0.011719604717982607, 0.26484374420218337, -0.021509548172652989, 0.10938355894132336, 
-0.26144220241068761], [0.10689216265527945, -0.06648736354897844, 0.34305197419776118, 0.30893606606354351, 
0.21785089021938683]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_taggedData_rank2(self):
      arg0=numpy.array([0.21122738013075715, 0.38647702911519244])
      arg1=Data(numpy.array([[0.82448010569624897, 0.78778800925038195, 0.76182135394840622, -0.014743151024537937, 
0.90793287126714861], [0.70339472706759221, 0.20072836124543092, -0.41202920687832423, 0.33619868469705461, 
0.21061116268085422], [-0.94088147074802331, -0.82287905674957873, -0.55671440962084029, 0.066252875605198769, 
0.75007333719112768], [-0.028409464051573297, 0.085903578361213428, -0.6634615974962641, 0.12320652220798589, 
-0.90910588798641823]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.28240737403425742, -0.95907924689419755, 0.82840524365082446, 
-0.11622499258020458, 0.15520913474955944], [-0.32051456889783281, -0.14175594821866255, 0.79045587302605758, 
-0.68131274838171674, -0.68566785701206712], [0.25832614098673279, -0.4087660456168849, -0.067134537232206304, 
-0.7675590141200852, 0.45657394961334696], [-0.19163440717784264, -0.60108356730739931, -0.53337132837456092, 
0.44609710684362591, -0.59166458013109358]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.1741527726961484, 0.16640239729238285, 0.16091752872218809, -0.0031141571657852364, 
0.19178028173235578], [0.14857622539627646, 0.042399325863812579, -0.087031849906262171, 0.071014367371965315, 
0.044486844119369526], [-0.19873992807967858, -0.17381458732168215, -0.11759332622525127, 0.013994421340215087, 
0.1584360259208159], [-0.0060008566625327532, 0.018145187801096314, -0.14014125505650277, 0.026024590901014808, 
-0.19202805498081668]], [[0.31864262181406616, 0.30446196938765935, 0.29442645359049352, -0.0056978892077600273, 
0.35089519872335406], [0.27184590441237466, 0.077576900713295274, -0.15923982378302376, 0.12993306885415298, 
0.081396376451393027], [-0.36362907556422891, -0.31802385317368903, -0.21515733109588067, 0.025605214534235629, 
0.28988611497614503], [-0.010979605265406906, 0.0331997597554059, -0.25641266713237576, 0.047616490670557364, 
-0.35134854274011984]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.059652169746863007, -0.20258379665924095, 0.17498186930294518, 
-0.024549900688433303, 0.032784418905511099], [-0.067701452682028276, -0.029942737560179362, 0.16696592316826456, 
-0.14391190689035577, -0.14483182507652953], [0.054565553979916172, -0.086342580902064162, -0.014180652415849709, 
-0.16212947964833244, 0.096440919212779599], [-0.04047843377108646, -0.12696530716199159, -0.11266262832942027, 
0.094227923162489555, -0.12497575917725533]], [[-0.10914396291698275, -0.3706620980257056, 0.32015959746961775, 
-0.044918289841332747, 0.059984765289549311], [-0.12387151837577108, -0.054785417726955762, 0.30549303745376649, 
-0.26331172689287252, -0.26499487633780427], [0.09983711951134483, -0.1579786869131789, -0.025945956500526366, 
-0.29664392744771656, 0.1764553436179559], [-0.074062296362343738, -0.23230499134292551, -0.20613576640542405, 
0.17240628454980714, -0.22866476916175277]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(arg0*1j)
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1) 
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_taggedData_rank2(self):
      arg0=numpy.array([[-0.26931146648060356, 0.91506700842357525, -0.73706998264364509, -0.28720133727824182, 
0.025800252429858617], [-0.29453834983974625, -0.59285210365103835, -0.040735927784848025, -0.83646988159030711, 
-0.52739983702101423], [-0.87161143206951741, 0.59502029755090935, -0.62679623643588411, -0.36760018250200988, 
-0.55445488830545409], [0.021549255060761885, -0.92761947971335212, -0.85362983147259364, 0.25432476780912405, 
0.089147445262878655]])
      arg1=Data(numpy.array([[-0.70567399039441492, -0.89584695038207984, -0.94634715844997119, -0.70948109275120275, 
-0.1279585411856885], [-0.26638578083208708, 0.77658666721699077, -0.30459674321230046, 0.65580657910950224, 
0.28166598356718131], [-0.71099804361875529, 0.01420756976475257, 0.88834227572250146, -0.40530986441748884, 
-0.78540957743612116], [0.22262751168969541, 0.51962718819174181, -0.45751975587490401, -0.1666299496461936, 
0.90681041528452977]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.54095693743134876, -0.26273807040894326, 0.4877890870465873, -0.76937359878437128, 
-0.48852390946348234], [0.99110914989877874, 0.64594790132996693, -0.92093161527638556, 0.42875435652153526, 
0.98042231520977485], [0.65087641644302385, 0.62865548472018284, -0.52793696081732633, 0.76944980836278853, 
-0.27268121480559637], [-0.2403351288025557, 0.6608784486374617, -0.33909031187408578, -0.025158887018166265, 
0.098034219336275541]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.19004609721033922, 0.24126185594957442, 0.25486214104191385, 0.19107139352908753, 
0.034460702375436476], [0.071740745285470031, -0.20914369419749224, 0.082031395599720461, -0.176616231547608, 
-0.075855879092179182], [0.19147992579180712, -0.0038262614484709983, -0.23924076101154354, 0.10915459396532852, 
0.21151980508723295], [-0.059956141652079582, -0.13994156007511055, 0.12321531639851813, 0.044875356098805524, 
-0.24421444276016183]], [[-0.64573898731254409, -0.8197599888915128, -0.86597106321296624, -0.64922274107693223, 
-0.11709063948503282], [-0.24376083955259609, 0.71062883835188628, -0.27872643058684371, 0.60010696445023093, 
0.2577432489575045], [-0.65061085276922903, 0.013000878361601373, 0.81289270870158026, -0.37088568511707642, 
-0.71870239241169576], [0.20371909111467409, 0.47549369659417134, -0.4186612343031329, -0.15247756953651337, 
0.82979229392175458]], [[0.52013111585208316, 0.66030189616948198, 0.69752408365358309, 0.52293721672012339, 
0.094314399730841575], [0.19634496285442027, -0.57239872132691361, 0.22450911623280112, -0.48337534388182907, 
-0.20760754161916756], [0.52405531566974162, -0.010471973199914554, -0.65477042574840039, 0.29874173473149662, 
0.57890182360899445], [-0.16409205617712169, -0.38300160258165322, 0.33722407852184022, 0.12281793409363137, 
-0.66838273705484497]], [[0.20267051372374914, 0.25728844214636809, 0.27179216943629592, 0.20376391861177376, 
0.036749864144702722], [0.076506352486884047, -0.2230367293371727, 0.08748059198116992, -0.18834852651611814, 
-0.080894847146285753], [0.20419958892952023, -0.0040804330359108538, -0.25513308954829905, 0.11640553507276567, 
0.22557068095079283], [-0.063938919072207931, -0.14923762333480087, 0.13140028571848716, 0.047856344368992901, 
-0.26043716392755478]], [[-0.01820656708536153, -0.023113077458376687, -0.024415995574288668, -0.018304791287192965, 
-0.0033013626631472237], [-0.0068728203891928394, 0.02003613204786097, -0.0078586728641901766, 0.016919975286187203, 
0.0072670534769376871], [-0.018343929002499513, 0.00036655888634544329, 0.022919454957755603, -0.010457096814282982, 
-0.020263765358680515], [0.0057438459994254417, 0.013406512624764588, -0.011804125193219814, -0.0042990947632464251, 
0.023395937620365792]]], [[[0.20784805265559991, 0.26386128247450685, 0.2787355304253874, 0.20896939030143921, 
0.0376886975687339], [0.078460828307055239, -0.22873455546964064, 0.089715422112311913, -0.19316018762496179, 
-0.082961434005866669], [0.20941619050675611, -0.0041846741537432935, -0.26165086798419046, 0.11937929863929844, 
0.23133324088636753], [-0.06557233992201171, -0.15305013454186292, 0.13475711391447778, 0.049078910402669873, 
-0.26709044333540038]], [[0.41836030969715149, 0.53110474908338345, 0.5610439036712479, 0.42061735833818803, 
0.075860490322053456], [0.15792737054902728, -0.46040103932694182, 0.18058081997866746, -0.38879631001325948, 
-0.16698627088474224], [0.42151668585115182, -0.0084229876228024501, -0.52665558692423575, 0.24028880575042538, 
0.46563172021067756], [-0.13198518863583206, -0.3080620716337481, 0.27124154973234638, 0.098786916179012474, 
-0.53760446231410519]], [[0.028746284712352424, 0.036493156677040735, 0.038550329506014155, 0.028901370559028059, 
0.0052125098931947089], [0.010851471930886253, -0.031634978394427138, 0.012408030934996169, -0.026714889447432905, 
-0.011473925166040889], [0.028963164960021843, -0.00057875853593515093, -0.036187446792059375, 0.016510673367397369, 
0.031994387827965834], [-0.0090689382391118418, -0.021167495613222429, 0.018637491735461389, 0.0067878255955802049, 
-0.036939763591578655]], [[0.59027503918657576, 0.749348992509136, 0.79159089557197104, 0.59345956564416025, 
0.1070334657940613], [0.22282368254995738, -0.64959135757160746, 0.25478600172758614, -0.54856245157386974, 
-0.23560511192245753], [0.5947284493567202, -0.011884204198808609, -0.74307155818526471, 0.33902949429668033, 
0.65697145623788544], [-0.18622120834182423, -0.43465249257785049, 0.38270149602190717, 0.13938093424995041, 
-0.7585196006979078]], [[0.3721723475239832, 0.47246953562728156, 0.49910333713181476, 0.37418021268647539, 
0.067485313766778854], [0.14049181739555835, -0.40957168172293357, 0.16064427272729898, -0.34587228293966038, 
-0.14855059382769509], [0.3749802523267915, -0.0074930699783951945, -0.46851157143492417, 0.21376035643679298, 
0.41422488313455397], [-0.1174137133815393, -0.27405129436401254, 0.2412958446823186, 0.087880608286222309, 
-0.47825166523001922]]], [[[0.61507351734188687, 0.78083044333763452, 0.82484700201149785, 0.6183918312791219, 
0.11153012732838428], [0.23218489191401204, -0.67688181713909512, 0.26549000355498426, -0.57160851157824455, 
-0.24550329130226006], [0.61971402299716849, -0.012383480228883564, -0.77428928311038359, 0.35327271135682936, 
0.68457196655021213], [-0.19404468428192864, -0.45291299764206072, 0.39877944961822109, 0.14523656903679039, 
-0.79038632468170278]], [[-0.41989034773842232, -0.53304711897641988, -0.56309576780735937, -0.42215565091556495, 
-0.076137929250488665], [-0.15850494657403977, 0.46208482980152288, -0.18124124477922093, 0.39021822583757998, 
0.16759697735211376], [-0.4230582674721462, 0.0084537923888983769, 0.52858168522745475, -0.24116759612601293, 
-0.46733464046537482], [0.1324678882486211, 0.3091887241333926, -0.27223354127610477, -0.099148202219371151, 
0.5395706031248646]], [[0.44231380132991149, 0.56151349692205188, 0.59316683727823527, 0.4447000787588723, 
0.080203932035015621], [0.16696960486558646, -0.48676160027789622, 0.19092009227609735, -0.4110570956157279, 
-0.17654717843192083], [0.44565089785351236, -0.0089052512574471707, -0.55680959508975236, 0.25404669760722048, 
0.49229176719765882], [-0.13954208645418689, -0.32570036590834467, 0.28677166107745433, 0.10444302531573503, 
-0.56838535546120439]], [[0.2594058876559085, 0.32931350245432156, 0.34787738815646785, 0.26080537917706753, 
0.047037583092550038], [0.097923461649815621, -0.28547340059759341, 0.11196981839435949, -0.24107461816667181, 
-0.10354046696390397], [0.26136301059282641, -0.0052227052384330825, -0.32655478267984234, 0.14899198012973378, 
0.28871670400434457], [-0.081837913927100367, -0.19101504921229051, 0.16818434575788971, 0.06125319990024148, 
-0.3333436741533165]], [[0.39126439352419939, 0.49670672081287776, 0.52470680803656267, 0.3933752600361996, 
0.070947238660839768], [0.14769889835741601, -0.43058227383130143, 0.16888515323598113, -0.36361516357014101, 
-0.15617108145818737], [0.39421634086003332, -0.007877456507007833, -0.49254571726273244, 0.2247260356046975, 
0.43547417953137846], [-0.12343691212763125, -0.28810983458932937, 0.25367406514115853, 0.092388790119423705, 
-0.50278546752080644]]], [[[-0.01520674880875488, -0.019304834429189136, -0.020393076293465671, -0.015288789028483728, 
-0.002757411241213406], [-0.0057404151357108588, 0.016734864168645943, -0.0065638329101592539, 0.014132143243756381, 
0.0060696921218295558], [-0.015321478189643262, 0.00030616254465422186, 0.019143114280801843, -0.0087341256469752847, 
-0.016924991311336186], [0.0047974570329439945, 0.01119757881485036, -0.0098592099146858174, -0.0035907512856877353, 
0.019541088930721741]], [[0.65459693981691225, 0.83100508201621803, 0.87785005874957145, 0.65812848212433117, 
0.11869683539954791], [0.24710463941849567, -0.72037692019615107, 0.28254987246097568, -0.60833895770614976, 
-0.2612788531295383], [0.65953563529884107, -0.013179218473170932, -0.82404359961308205, 0.37597332555364027, 
0.72856122358317843], [-0.20651361656347347, -0.48201630195533562, 0.42440423790325832, 0.15456918719546417, 
-0.84117500562488423]], [[0.60238436949497709, 0.76472168127989182, 0.80783016538221675, 0.6056342256382008, 
0.10922922794781821], [0.22739484919838976, -0.66291754586030294, 0.26001286657541695, -0.55981605960386249, 
-0.2404384860840153], [0.60692914015162192, -0.012128005383920852, -0.75831546711497921, 0.3459845912568808, 
0.67044904522375715], [-0.19004148528483755, -0.44356926908469424, 0.3905525121028765, 0.14224029583476699, 
-0.77408042197692584]], [[-0.17947037375599761, -0.22783606764843434, -0.24067952133961326, -0.18043861417891324, 
-0.032543026276244467], [-0.067748501857772758, 0.19750522382362268, -0.077466495992883702, 0.16678785595972009, 
0.071634635870451935], [-0.1808244123560814, 0.0036133368815526286, 0.22592744300815404, -0.1030803371587254, 
-0.19974910841650376], [0.056619690218404836, 0.13215406398417273, -0.11635860568097209, -0.042378123253814216, 
0.23062434831413339]], [[-0.062909033432123257, -0.07986246697310323, -0.084364431507599563, -0.063248426881085187, 
-0.011407177046268971], [-0.023747611815537674, 0.069230717407608042, -0.027154021492769662, 0.05846348111420005, 
0.025109802852470175], [-0.063383659175516804, 0.0012665685479218095, 0.079193444399672755, -0.036132338952662858, 
-0.07001725731342727], [0.019846673912367999, 0.046323436316426846, -0.040786717393543608, -0.01485463431524027, 
0.080839831860385875]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.14568590612249263, 0.070758375042116586, -0.1313671943657512, 0.20720113216007854, 
0.13156509046844803], [-0.26691705860158443, -0.17396117657724161, 0.24801744383843441, -0.11546846451476218, 
-0.26403897147945299], [-0.17528848220991078, -0.1693041305010671, 0.14217947712702708, -0.20722165627340194, 
0.07343617784100763], [0.064725005984621012, -0.17798214416798105, 0.091320909160175254, 0.0067755767578821756, 
-0.026401739374733509]], [[-0.49501184642128349, -0.24042294008809439, 0.4463597006253876, -0.70402839739969469, 
-0.44703211237613832], [0.90693128481910823, 0.59108561366749957, -0.84271413815365304, 0.39233896637073629, 
0.89715211497072422], [0.59559553524797493, 0.57526189373197034, -0.48309769537134506, 0.70409813427063017, 
-0.24952158348546338], [-0.21992274733244926, 0.60474806492629551, -0.3102903572720368, -0.023022067478980127, 
0.089707879811186275]], [[0.39872312048348363, 0.19365634499614462, -0.3595346939231876, 0.56708218510247532, 
0.36007630946925456], [-0.73051680391385065, -0.47610880842197773, 0.6787910496877495, -0.31602196611971523, 
-0.72263985885511106], [-0.47974146897081749, -0.46336308721153746, 0.38912648654656545, -0.5671383568951166, 
0.20098513826400899], [0.17714380921515793, -0.48711366666677297, 0.24993329028766062, 0.018543860417813238, 
-0.072258080344671904]], [[0.15536355584022554, 0.075458725175353361, -0.14009367810951256, 0.22096512643744493, 
0.14030472009030687], [-0.2846478732396307, -0.18551710107404029, 0.26449279144918925, -0.12313882455685699, 
-0.28157860002567725], [-0.18693257720530626, -0.18055069589893782, 0.15162420114534689, -0.22098701393027975, 
0.078314409542822788], [0.069024570387032483, -0.18980517422704885, 0.097387191028333514, 0.0072256659960495493, 
-0.028155558892406809]], [[-0.013956825539412031, -0.0067787085394847022, 0.012585081578332231, -0.019850033061505543, 
-0.012604040182179241], [0.025570866252931073, 0.016665618910850555, -0.023760268144768188, 0.011061970628657207, 
0.025295143220278603], [0.016792775844871793, 0.016219470197195843, -0.013620906856139398, 0.019851999287866282, 
-0.0070352441748648869], [-0.0062007069908685212, 0.017050830800299864, -0.0087486156428708984, -0.00064910563592298254, 
0.0025293076056400357]]], [[[0.15933256368539234, 0.077386437698329216, -0.14367259276853817, 0.22661003019621573, 
0.14388902615063567], [-0.29191965352226001, -0.19025642894017569, 0.27124967827875868, -0.12628460065645525, 
-0.28877197086795064], [-0.19170806564873571, -0.18516314908718848, 0.15549768125854607, -0.22663247683968471, 
0.080315075041137746], [0.070787912246027623, -0.19465404770632946, 0.099875100906038142, 0.0074102570661353055, 
-0.028874837191134341]], [[0.32070745834079822, 0.15576481775115664, -0.28918678639358875, 0.45612475653288448, 
0.28962242740925492], [-0.58758114446528331, -0.38295157215244424, 0.54597624543535395, -0.25418792221333947, 
-0.5812454320385364], [-0.38587345270509599, -0.37269972658812361, 0.3129885378156877, -0.45616993754176749, 
0.16165963182361848], [0.1424831866918384, -0.39180317853235386, 0.20103040472223843, 0.014915499094238669, 
-0.058119793163298253]], [[0.022036382737915975, 0.010702879062509028, -0.019870541024166726, 0.03134114735964879, 
0.019900474697076051], [-0.040373750757178764, -0.026313287061351671, 0.037515003774682289, -0.017465706504700244, 
-0.039938412631038894], [-0.026514054697083688, -0.025608864427109999, 0.021506001910806746, -0.031344251827531706, 
0.011107922274605405], [0.0097902744510630583, -0.026921496758258036, 0.013813158457044353, 0.0010248706047191714, 
-0.003993514879326472]], [[0.45249418539865549, 0.21977248264423455, -0.40802087988290286, 0.64355784307387154, 
0.408635536702953], [-0.82903295325890136, -0.54031596453898478, 0.77033155918300844, -0.35864010583089689, 
-0.82009373791201512], [-0.54443851899201956, -0.52585137886498845, 0.44160336710201553, -0.64362159009090625, 
0.22808962346033829], [0.20103309673146497, -0.55280491767736339, 0.28363883302173687, 0.021044651245029449, 
-0.0820026718400126]], [[0.28530060063668033, 0.13856801551289244, -0.25725988500899949, 0.40576751060714861, 
0.25764743023190928], [-0.52271080412665183, -0.34067281788549075, 0.48569918380426513, -0.22612497775150753, 
-0.51707456925340067], [-0.34327211595287255, -0.33155280018379113, 0.27843386709242746, -0.40580770352638529, 
0.14381202824716369], [0.12675270776089234, -0.34854718610209801, 0.17883617521779774, 0.013268792913010998, 
-0.051703231300434083]]], [[[0.47150425092247822, 0.22900550580832074, -0.42516254470855847, 0.67059482423292416, 
0.42580302432766509], [-0.8638620654804765, -0.56301557532051183, 0.80269452402914432, -0.37370719869377977, 
-0.85454729819290365], [-0.56731132543617957, -0.54794330731531515, 0.46015589046041849, -0.67066124937270588, 
0.23767206413516154], [0.20947884579220749, -0.57602921104077898, 0.29555499233347121, 0.021928773543179091, 
-0.085447746307508302]], [[-0.32188035787262981, -0.1563344848326812, 0.2902444077165468, -0.45779290767649056, 
-0.29068164196969476], [0.58973006127820016, 0.38435211245174233, -0.5479730037457945, 0.25511754479369259, 
0.58337117772167169], [0.38728467898079766, 0.37406277357521434, -0.31413320751364826, 0.45783825392251659, 
-0.16225085757016938], [-0.14300427985203282, 0.39323609115324581, -0.20176561826794917, -0.014970048439599001, 
0.058332350359641787]], [[0.33906977245585146, 0.16468323370075197, -0.3057443639352968, 0.48224047613117582, 
0.30620494786065533], [-0.62122348505372305, -0.40487771348728108, 0.57723647047005799, -0.26874161702318755, 
-0.61452501729124298], [-0.40796688821136257, -0.39403889183738705, 0.33090890011569896, -0.4822882440081081, 
0.17091555918691267], [0.15064115421677537, -0.41423612434754675, 0.21254053129454714, 0.015769495695902237, 
-0.061447479721907484]], [[0.19885586892549215, 0.096582562632553465, -0.17931135742081428, 0.28282187532536301, 
0.17958147827537146], [-0.36433190438220292, -0.23745056641568613, 0.33853462984747007, -0.15761017970584817, 
-0.36040342200015629], [-0.23926228947070974, -0.23109387091402869, 0.1940697231460056, -0.28284988998029759, 
0.10023766432740698], [0.088347237209463525, -0.24293903833077607, 0.12464966052957738, 0.0092484114594253656, 
-0.036037396919456954]], [[0.29993621832155898, 0.14567640748218116, -0.27045704377503499, 0.42658295277915381, 
0.27086446965611888], [-0.54952531300564089, -0.35814897148304931, 0.51061503588502977, -0.23772494885562467, 
-0.54359994527181044], [-0.36088161077957098, -0.3485611065631401, 0.29271722864229155, -0.42662520755244299, 
0.15118943249803246], [0.13325498699609795, -0.36642728642276562, 0.18801028099560782, 0.013949467891546915, 
-0.054355552132207045]]], [[[-0.011657219021596743, -0.005661809693414733, 0.010511491452623091, -0.016579427917220696, 
-0.010527326328309128], [0.021357663864223667, 0.013919696081723309, -0.019845390271110267, 0.0092393369870953992, 
0.021127370537718123], [0.014025901910965392, 0.013547057385582116, -0.011376648224656019, 0.016581070176864082, 
-0.005876077048124196], [-0.0051790429906273328, 0.014241438253849286, -0.0073071436192079691, -0.00054215527339935589, 
0.0021125643971600765]], [[0.50180219284739613, 0.24372095217363401, -0.45248265913600633, 0.71368593740954778, 
0.45316429472404823], [-0.9193721539682479, -0.59919385615363563, 0.85427410581425778, -0.39772089312133962, 
-0.90945883793425142], [-0.60376564277856892, -0.58315307365508118, 0.48972460891481662, -0.71375663089902841, 
0.2529444066055721], [0.22293954713666819, -0.61304372267884955, 0.31454677867647773, 0.0233378736859584, 
-0.090938451534820566]], [[0.46177697933345263, 0.22428105476462068, -0.41639131616974862, 0.65676025546976569, 
0.417018582505645], [-0.84604033659903999, -0.55140039815237518, 0.78613469954616444, -0.36599750910061846, 
-0.83691773570449013], [-0.55560752567774407, -0.53663907547601131, 0.45066273889064756, -0.65682531023934665, 
0.23276881944024333], [0.2051572354966697, -0.56414555873426553, 0.28945760577906504, 0.021476376485355293, 
-0.083684934130572167]], [[-0.13757874750696261, -0.066820798751371785, 0.12405684630294792, -0.1956707618693054, 
-0.12424372984350569], [0.25206360442150522, 0.1642805500225348, -0.23421571922324833, 0.10904285216948985, 
0.24934567767060983], [0.16553399348430678, 0.15988266018339284, -0.13426744497772114, 0.19569014385264119, 
-0.069349586641343178], [-0.061123175829085899, 0.16807775799977656, -0.086239064833700321, -0.0063985280992311204, 
0.024932530070047018]], [[-0.048224928969235636, -0.023422427750255628, 0.043485150937315191, -0.068587690784333702, 
-0.043550658478503278], [0.08835484869013957, 0.057584605176484567, -0.082098700763706012, 0.038222355529224324, 
0.08740214467966817], [0.05802396970775308, 0.05604303041330095, -0.047064231316713112, 0.068594484673554168, 
-0.024308833671097159], [-0.021425262739672726, 0.058915625325324281, -0.030229035016967516, -0.0022428505033269255, 
0.0087395002021696637]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_taggedData_rank3(self):
      arg0=numpy.array(0.704027105071)
      arg1=Data(numpy.array([[[-0.3782235462989445, -0.49753575665828897], [-0.010510916837058959, 0.52424756111160287]], 
[[-0.52759022930980604, 0.093331993173143779], [0.57416364865148894, -0.73716544963679365]], [[0.41131485065963647, 
0.12482728958110134], [-0.96218077314844686, 0.69311459044768364]], [[0.19590676135975005, 0.073767055570488127], 
[0.47700162226896214, -0.27823587180601472]], [[-0.95243974037525447, -0.52612618077281281], [0.31035877112252375, 
0.25673711331646887]], [[0.7221296787332816, 0.53697125941643087], [0.58139709575186216, 
-0.65415892129479958]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.16917214979085049, 0.68330341495581237], [0.19499838786148582, 
0.87189337191857463]], [[-0.086333929651582597, 0.32667565845120983], [-0.14145632600578262, -0.71764916587928895]], 
[[-0.64476945370576466, -0.4055842208678957], [-0.54731266149788338, 0.57272004049034253]], [[0.5916267990390931, 
-0.93884816733771004], [-0.12474632106651429, 0.80115481137233568]], [[0.95514843830325624, -0.667976777383402], 
[-0.58978403981835581, -0.52737350450940923]], [[-0.46480628289036474, -0.77522320166842351], [0.4038182907972625, 
0.884085768051315]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.26627962837065644, -0.35027865842960682], [-0.007399970352440075, 0.36908449279010475]], 
[[-0.3714378218048997, 0.065708252964225153], [0.40422677139729762, -0.5189844574663941]], [[0.28957680358274857, 
0.087881795317682851], [-0.6774013442749911, 0.48797145859558033]], [[0.13792367006400391, 0.051934006582926379], 
[0.33582207124034347, -0.19588559535458508]], [[-0.67054339317127554, -0.37040709195171645], [0.2185009871668846, 
0.1807498866525625]], [[0.50839886720467875, 0.37804232127345372], [0.40931931421905993, 
-0.46054561161575902]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.11910177887594517, 0.48106412511669144], [0.13728415049969742, 
0.61383656656271091]], [[-0.060781426562035196, 0.22998851811667445], [-0.09958908769187684, -0.5052444647108475]], 
[[-0.45393517193088972, -0.28554228488023381], [-0.38532294864323735, 0.40321043212274837]], [[0.41652130261010778, 
-0.66097455735228772], [-0.087824791288756199, 0.56403470256442956]], [[0.67245038993203932, -0.47027375683611], 
[-0.41522395017058861, -0.37128524167107918]], [[-0.32723622176226719, -0.54577814645474476], [0.28429902224484754, 
0.62242034391592693]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_taggedData_rank3(self):
      arg0=numpy.array([-0.099751811547238667, 0.9673041323588123])
      arg1=Data(numpy.array([[[-0.59209587540687703, 0.31425762767865972], [0.9168162580616952, -0.96510674431262333]], 
[[0.010645605719809437, -0.17466636827806448], [-0.70308232497865153, -0.66961792053243996]], [[0.47061167772961388, 
0.44730410536340481], [-0.23095052674639449, 0.44317594050311526]], [[0.39926882433732946, -0.74222313744253543], 
[0.85656686909661151, 0.40159080034335704]], [[0.5889959758134593, 0.80538045756150178], [0.64374825965196347, 
0.62050829422293563]], [[0.85445885586484516, 0.73631310370537539], [-0.81174994107411358, 
-0.36040549067878391]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.26757775517041149, -0.7032721993970148], [0.56011284925353899, 
-0.996491620256996]], [[0.92824845214919693, -0.13253715690597545], [0.43645736281529479, 0.96477855835738446]], 
[[0.29182000991085899, -0.8221594006250188], [0.024696013487273571, 0.66638812407880499]], [[-0.75026663102583502, 
-0.6107721381511948], [-0.21024074905131274, -0.30769737846779099]], [[-0.28618913958033976, 0.44794811631886722], 
[-0.58489799932720166, 0.52979247222265236]], [[-0.48986848298295782, -0.42693061460914961], [0.024126904658099635, 
-0.10874520793053688]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.0590626361814841, -0.03134776765348396], [-0.091454082597614755, 0.096271146081641862]], 
[[-0.0010619184555686369, 0.017423286652114074], [0.070133735583464862, 0.066795600617605783]], [[-0.046944367388814259, 
-0.044619394822516548], [0.023037733420741845, -0.044207602899337006]], [[-0.039827788521984826, 0.074038102532168013], 
[-0.085444096903733446, -0.040059409834955303]], [[-0.05875341558142614, -0.080338159626503772], [-0.064215055080665523, 
-0.061896826428824797]], [[-0.085233818765099195, -0.073448565960581011], [0.080973527145507068, 0.035951100586780138]]], 
[[[-0.57273678703368058, 0.30398270187884463], [0.88684015503682101, -0.93355174194096024]], [[0.010297538404234276, 
-0.16895549981947794], [-0.68009443834029104, -0.64772418163254397]], [[0.45522462060416913, 0.43267910953908306], 
[-0.22339939889223179, 0.42868591861066652]], [[0.3862143837035435, -0.71795550798048724], [0.82856067211880213, 
0.38846044068941199]], [[0.5697382413470703, 0.77904784472027178], [0.62270035176013794, 0.60022023716476336]], 
[[0.82652158220864747, 0.71223870792415234], [-0.78520907244301241, -0.34862172045839307]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.026691365807992053, 0.07015267590066307], [-0.05587227138392592, 
0.099401844312278387]], [[-0.092594464667802676, 0.013220821498691665], [-0.043537412603956056, -0.096238408938082415]], 
[[-0.029109574634341325, 0.082011889592937565], [-0.0024634720833505777, -0.066473422570426849]], [[0.074840455588270738, 
0.060925627223162006], [0.020971895578916846, 0.030693370910498459]], [[0.028547885118284433, -0.044683636081980185], 
[0.058344635003243946, -0.052847758848299696]], [[0.048865268597447699, 0.042587102212238673], [-0.0024067024466729496, 
0.010847531488152198]]], [[[-0.25882906830363361, -0.68027810464980298], [0.54179947367021675, -0.96391046213552056]], 
[[0.89789856361958942, -0.12820373956623834], [0.42218701064966407, 0.9332342863102755]], [[0.28227870149176348, 
-0.79527818568222497], [0.023888555899028687, 0.64459998617626502]], [[-0.72573601256221454, -0.59080241316327808], 
[-0.20336674534754687, -0.29763694570786764]], [[-0.27683193735227557, 0.4333020639975862], [-0.56577425175760399, 
0.51247044767356287]], [[-0.47385180790175763, -0.41297174774191792], [0.023338054576806853, -0.10518968900542661]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_taggedData_rank4(self):
      arg0=numpy.array(-0.0708515730447)
      arg1=Data(numpy.array([[[[-0.97827869989748129, -0.5363137669536211, -0.54427510467957529, 0.22593676912037153], 
[-0.23713371651922976, -0.35954834839453675, -0.91075392873807126, 0.10286969410010571], [-0.34424044846906554, 
-0.85850694284952689, -0.14494519131033878, 0.66211966482478357]], [[-0.041700412168819811, -0.60783636214129322, 
-0.084341919502126217, 0.27822590088774102], [-0.26981397652701933, 0.33342963211783649, 0.11843242489050576, 
0.70999816999429521], [0.9414292577295893, 0.26604461049427197, 0.23454517685435516, -0.39582462844360222]]], 
[[[-0.95641033877536352, -0.44143263296052426, 0.23721575825434571, 0.43570317795380231], [0.81946644874742591, 
0.48112725864302175, 0.56828100939862702, -0.52760359142695634], [-0.27947807553721016, 0.7224294405657814, 
0.89804869651162789, 0.96505381787309363]], [[0.72420980973730598, -0.86663668014145978, -0.37484799631393662, 
-0.29592339600712259], [-0.056924976147858963, -0.87177739322818626, -0.3722537253722531, 0.20734191878375885], 
[-0.30028822356308416, 0.42772405801095892, -0.58755490665781007, -0.086791102699894873]]], [[[0.14474337489567302, 
0.61556095188938631, -0.088437439158631115, -0.52612631754127048], [-0.40731285419528085, 0.13757484975063883, 
0.62097013714886895, -0.68462432608446178], [-0.75811417408093451, -0.9188428753088953, -0.35131587181041612, 
0.90147708459225706]], [[-0.80978939279462803, -0.15202616059901231, 0.51689588770499739, 0.81806366721492285], 
[0.22563589583665222, -0.64587796516796092, -0.500167084063037, 0.16241148341210332], [-0.41782931970614401, 
0.26833183844255903, 0.75834324552865562, 0.958289893042233]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.82981805382263607, 0.50973384992299176, -0.19902588165663415, 
0.55880949471253372], [0.65120278143864652, 0.39005542512518843, -0.094319616395163841, 0.558260672925043], 
[-0.87067555439476396, 0.73730040630974192, -0.55151379674815937, 0.56636765431211655]], [[0.77924727217736778, 
0.022464511777139151, -0.66398342349787121, 0.00095601543780277076], [-0.91770849372904428, 0.61525027864649529, 
0.53962654307142066, -0.0097274508466951293], [0.34239915683292188, 0.079734817558392956, -0.077349213583399967, 
-0.12297204801843598]]], [[[-0.30795284251753996, -0.64188023920992943, 0.35506454068468063, 0.096504079288820765], 
[-0.58228537542689462, 0.012861580726549526, -0.2156033370124717, -0.6920338805125883], [-0.5349505069395839, 
0.42175843685791214, 0.69320478885711601, -0.35104637849746401]], [[-0.4283120467840853, 0.31690394755050066, 
0.32395286846804305, 0.40366427344827915], [0.80982587237558978, -0.71300714940927623, 0.81148932277682118, 
0.85559856025065528], [-0.78247849977172224, -0.62478202178264608, 0.46740574954895653, 0.045811318251096411]]], 
[[[-0.39721471059287272, -0.84284962959750431, -0.80242522851950571, 0.31034889032502377], [0.64893363521345582, 
0.26516689052541009, -0.30393793155495508, -0.63094047783061069], [0.65088856539406326, -0.0030937550324501295, 
-0.79434923004484359, -0.410914278306165]], [[-0.43704509557339821, -0.86899529281274224, -0.79861039554537228, 
0.33554072829468895], [-0.50025633720531215, -0.14032417411100706, 0.79880096541072088, -0.3591955772553217], 
[0.13704223488208278, -0.47004074139542973, 0.86717083462174749, 0.8041685339827882]]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.069312584763890656, 0.037998674034209201, 0.038562747335633418, -0.016007975500822474], 
[0.016801296837330688, 0.025474566069387829, 0.064528348507760985, -0.0072884796456227502], [0.024389977279656878, 
0.060826567370711765, 0.010269594809606946, -0.046912219796685813]], [[0.002954539798774517, 0.043066162411497265, 
0.0059757576703376985, -0.019712742739683809], [0.019116744666393386, -0.023624013935274604, -0.0083911236029942608, 
-0.050304487202975985], [-0.066701743820474668, -0.018849679153591859, -0.016617894730185642, 0.028044797575075303]]], 
[[[0.067763176978478376, 0.031276196438530413, -0.016807109623318973, -0.030870255538615155], [-0.058060486951134374, 
-0.034088623109557133, -0.04026360344734016, 0.037381544396649292], [0.019801461283325432, -0.051185262277910454, 
-0.063628162818618869, -0.068375581069131802]], [[-0.05131140423431331, 0.061402572046285617, 0.026558570191507851, 
0.020966638107843438], [0.0040332241056095864, 0.061766799655051823, 0.02637476201438534, -0.014690501103942126], 
[0.021275893006252301, -0.030304922339152121, 0.041629189386855801, 0.006149286152574332]]], [[[-0.010255295799161627, 
-0.043613461746274861, 0.0062659316804366859, 0.037276877218030545], [0.028858756441074718, -0.0097473945162252478, 
-0.043996711030799579, 0.048506710447772827], [0.053713581781141072, 0.065101463096578646, 0.024891282153348973, 
-0.06387106950713925]], [[0.057374852314436772, 0.010771292622390892, -0.036622886744251577, -0.057961097672918432], 
[-0.015986658155383828, 0.045761469827079862, 0.035437624691062264, -0.01150710908027572], [0.02960386456539003, 
-0.01901173285163987, -0.053729811853551745, -0.067896346354909015]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.058793914454250834, -0.036115445101190684, 0.014101296791986958, 
-0.039592531732714181], [-0.046138741436032117, -0.027636040444750803, 0.0066826931905729376, -0.039553646845749239], 
[0.061688732640462082, -0.052238893593564359, 0.039075620055479016, -0.040128039229667754]], [[-0.055210895024581978, 
-0.0015916459970921892, 0.047044270030449829, -6.7735197623373288e-05], [0.065021090377213236, -0.043591450058313122, 
-0.038233389433300319, 0.0006892051942036482], [-0.024259518870801994, -0.0056493372504467605, 0.0054803134561567449, 
0.0087127630426383565]]], [[[0.021818943315963957, 0.045478224654351578, -0.025156881239914431, -0.0068374658228463753], 
[0.041255834809937109, -0.00091126322631782531, 0.015275835581026844, 0.049031689034566138], [0.037902084917745688, 
-0.029882248696269834, -0.049114649732667132, 0.024872188128201279]], [[0.030346582268660764, -0.022453143188037827, 
-0.022952570323313614, -0.028600248755768928], [-0.057377436950131921, 0.050517678127786604, -0.057495295027741054, 
-0.06062050388856579], [0.05543983258250753, 0.044266789053367721, -0.03311643260569503, -0.0032458039613429655]]], 
[[[0.028143287082012518, 0.059717222097151856, 0.056853089691384542, -0.021988707072214564], [-0.045977968856508837, 
-0.018787491313105222, 0.021534480558630283, 0.044703125351892851], [-0.046116478734997496, 0.00021919741066414377, 
0.056280892495547886, 0.029113923004532079]], [[0.030965332512859968, 0.061569683464249216, 0.056582802774264279, 
-0.023773588420253317], [0.035443948416591646, 0.0099421884719675356, -0.05659630494899915, 0.025449571679249661], 
[-0.0097096579149610392, 0.033303125922977697, -0.061440417731462894, -0.056976605625755579]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_taggedData_rank0(self):
      arg0=Data(-0.478291111915,self.functionspace)
      arg1=Data(-0.419213192283,self.functionspace)
      arg1.setTaggedValue(1,0.435813324539)
      res=outer(arg0,arg1)
      ref=Data(0.20050594386644371,self.functionspace)
      ref.setTaggedValue(1,-0.20844563958127024)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_taggedData_rank0(self):
      arg0=Data(numpy.array([0.49918682565299277, -0.38388605068826642]),self.functionspace)
      arg1=Data(-0.428347390473,self.functionspace)
      arg1.setTaggedValue(1,-0.684687559657)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([-0.21382537412718941, 0.16443658805148126]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.34178700946923146, 0.2628420032320814]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_taggedData_rank0(self):
      arg0=Data(numpy.array([[0.4990759241891225, -0.71562567842653735, -0.71491441859956928, -0.16977617098779474, 
-0.29276763491100377], [-0.63945796111161357, 0.23224775151724852, 0.92188504234276691, 0.11789528075878786, 
-0.27475427490937632], [-0.18318842961568449, -0.19969730770778238, -0.24351653275592988, 0.28501557927622434, 
-0.14932820637971678], [0.017394628525154499, 0.23504234340026264, 0.17614319978043258, 0.26251721320321453, 
-0.31854269107987609]]),self.functionspace)
      arg1=Data(0.275668291231,self.functionspace)
      arg1.setTaggedValue(1,0.677322567221)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.13757940721556033, -0.19727530793260029, -0.19707923615147899, -0.046801906947883783, 
-0.080706753643547874], [-0.17627828345345795, 0.064023340802914691, 0.2541344743337029, 0.032499990590929825, 
-0.075741041472577153], [-0.050499241365377354, -0.055050215579160684, -0.067129786471233702, 0.078569757713183488, 
-0.041165051485230635], [0.0047951475221208271, 0.064793721171992169, 0.048557094895366443, 0.072367671582355811, 
-0.087812119333994285]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.33803538621019941, -0.48470942168146158, -0.48422766934949807, -0.11499323198648277, 
-0.19829812607727634], [-0.43311930785031894, 0.15730664328907459, 0.62441354356266765, 0.079853134226837041, 
-0.18609727083668967], [-0.12407765743256323, -0.13525949312384911, -0.1649392431271153, 0.19304748385348394, 
-0.10114336410368584], [0.011781774448521291, 0.15919948343761398, 0.11930576427388519, 0.17780883278662474, 
-0.2157561532918543]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_taggedData_rank0(self):
      arg0=Data(numpy.array([[[-0.24140715296006143, -0.64856407812954875], [0.94412318042606169, 0.35234412448467367]], 
[[-0.28664613035923869, 0.49728948539315865], [-0.5753940267241946, -0.11383261804019829]], [[0.24485491221004607, 
0.97481395366069701], [0.66247997805566272, -0.83750295510814632]], [[-0.9389444607352464, -0.52505716150629222], 
[-0.7630733420549487, 0.57829970765001937]], [[-0.79540081390548845, -0.99821052259006793], [-0.38970418087931402, 
-0.14508728623645717]], [[0.75215868638497518, -0.78880463759830888], [-0.61225797093201328, 
0.5446571880752773]]]),self.functionspace)
      arg1=Data(-0.567508857852,self.functionspace)
      arg1.setTaggedValue(1,0.017440077081)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.13700069765363954, 0.36806585922306034], [-0.53579826779508255, -0.19995841165711922]], 
[[0.16267421804783383, -0.28221618787722269], [0.32654120692104416, 0.064601019050282704]], [[-0.13895733156774659, 
-0.55321555346006168], [-0.37596325569611039, 0.47529034550100185]], [[0.5328592984982129, 0.29797459003338816], 
[0.43305088080682402, -0.32819020658454096]], [[0.45139700743396277, 0.56649331357082178], [0.22116057459092356, 
0.082338320100881185]], [[-0.42685671703371075, 0.44765361895168559], [0.34746182179433893, 
-0.30909777872541944]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.0042101593555307305, -0.011311007514553759], [0.01646558104059917, 
0.0061449086900538198]], [[-0.0049991306084384681, 0.0086727669568326636], [-0.010034916178022891, -0.001985249632954269]], 
[[0.0042702885426072288, 0.017000830491487019], [0.011553701881916812, -0.014606116092660033]], [[-0.016375263770010407, 
-0.0091570373686062443], [-0.013308057903902493, 0.010085591477342096]], [[-0.013871851504810098, -0.017408868457046422], 
[-0.0067964709533272416, -0.0025303334554384271]], [[0.013117705467705467, -0.013756813681572952], [-0.010677826206517313, 
0.009498863342759194]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank4_taggedData_rank0(self):
      arg0=Data(numpy.array([[[[-0.51634379244210038, -0.66969872015335108, 0.34993396767937135, 0.28319084630103153], 
[-0.68365477353235571, 0.63545946412083687, -0.43617608234664917, -0.7109611655508612], [-0.46016744918303121, 
0.70827817282568928, 0.28541911241541817, 0.83529560289200422]], [[-0.78106481640937564, -0.83145204890561009, 
-0.93848329845377632, 0.54454036038441522], [0.98310846656333961, -0.20675023492579125, 0.30644455901111223, 
0.66244060995027776], [-0.28502793425906137, 0.85653470228467721, -0.41445306439850449, 0.49127642701739593]]], 
[[[-0.14690736035470442, -0.99733112975063509, -0.2925341546352187, -0.26074071531628618], [-0.37460913740259127, 
-0.085292799979562961, 0.81234887929092658, -0.70925554982527661], [-0.39910855859211414, 0.86154697739012698, 
0.40044112293925282, -0.76663882772607628]], [[-0.77022475708788418, 0.4187103765870952, -0.63307289975741199, 
0.9351420641274204], [0.15291717675255567, -0.10940324256984257, 0.82141430927862236, 0.14419366068272055], 
[0.94921190725272142, -0.12532143331069245, 0.66832866552506442, -0.89194122184669755]]], [[[-0.29346881915714329, 
0.93929818727039693, 0.92645000255827381, -0.31013275549349739], [-0.64480636744624231, -0.58612210631826023, 
-0.56441850870510324, 0.37706876918421672], [-0.75141238332071691, -0.59132095761911341, -0.099450792715001901, 
-0.019789332665212322]], [[-0.74012954004278364, -0.37777856259180043, 0.25002710281506135, 0.22233828409113787], 
[-0.92202693798534097, -0.15003203074229021, -0.4722392385365739, -0.091017181819327275], [0.93339702600405006, 
0.061346729246282594, 0.61494719557210886, 0.45178502259581177]]]]),self.functionspace)
      arg1=Data(0.150529719038,self.functionspace)
      arg1.setTaggedValue(1,0.570809784459)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.07772508600340132, -0.10080956018489143, 0.052675461836680225, 0.042628638527869703], 
[-0.10291036097891439, 0.095655534594241837, -0.065657463126801319, -0.10702078449740555], [-0.069268876836023063, 
0.10661691435630404, 0.042964058800010557, 0.12573681241713405]], [[-0.11757346736468617, -0.12515824331545419, 
-0.1412696272382416, 0.081969507453598031], [0.14798704125580428, -0.031122054774450507, 0.046129013368711974, 
0.099716998895274883], [-0.042905174862040305, 0.12893392808133644, -0.062387503338406453, 0.073951702528993968]]], 
[[[-0.022113923678829656, -0.1501279747493621, -0.044035084106301692, -0.039249226618366363], [-0.056389808202335141, 
-0.012839101216900592, 0.1222826485606178, -0.10676403864144626], [-0.060077699190591277, 0.12968842444470169, 
0.060278289727366319, -0.11540192734134161]], [[-0.1159417162806652, 0.063028355346012699, -0.095296285731149025, 
0.14076667217385463], [0.023018579652669062, -0.016468439365900783, 0.12364726518962565, 0.02170543122965201], 
[0.14288460170641701, -0.018864600145716569, 0.10060332624662852, -0.13426366152312605]]], [[[-0.044175778894181922, 
0.14139229222285482, 0.13945825858798871, -0.046684196548962893], [-0.097062521325691847, -0.08822879598613538, 
-0.084961759535309836, 0.05676005588336052], [-0.11310989494305289, -0.089011377611773926, -0.014970299885510359, 
-0.0029788826860468608]], [[-0.11141149171447413, -0.056866900885579226, 0.037636509538673396, 0.033468519435662973], 
[-0.13879245592053754, -0.022584279434359769, -0.071086039895699546, -0.013700790806907391], [0.14050399207543282, 
0.0092345059173522728, 0.092567828572766755, 0.068007072516991007]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.29473408887078667, -0.3822705821034213, 0.19974573266605897, 
0.16164810593794612], [-0.3902368339245938, 0.36272647974745414, -0.24897357555060437, -0.4058235896670388], 
[-0.26266808248336421, 0.40429211116787744, 0.16292002203841718, 0.47679490304660832]], [[-0.44583943950339933, 
-0.47460096482407688, -0.53569544930907809, 0.31082896574043217], [0.56116793189915914, -0.11801505703490607, 
0.17492155267786641, 0.37812758178282296], [-0.16269673371930196, 0.48891838879305016, -0.2365738643578178, 
0.2804253914157479]]], [[[-0.083856158699557468, -0.56928636720753689, -0.16698135775432055, -0.14883335150945998], 
[-0.21383056097726744, -0.048685964772266804, 0.46369668869382941, -0.4048500075223474], [-0.22781507030583725, 
0.49177944446564281, 0.22857571107360566, -0.43760494401247252]], [[-0.43965182757857246, 0.23900397981056334, 
-0.36136420545756931, 0.53378824006342274], [0.087286620702255069, -0.062448441310443276, 0.46887132483113719, 
0.082307152374705012], [0.54181944418515238, -0.071534700336210247, 0.38148854151635148, -0.50912877659270195]]], 
[[[-0.16751487340862209, 0.53616059581885156, 0.52882672627263094, -0.17702681131702033], [-0.36806178361999098, 
-0.33456423317437273, -0.32217560729881461, 0.21523454286438637], [-0.42891354056336739, -0.33753178836484882, 
-0.056767485553959419, -0.01129594471322376]], [[-0.42247318322380195, -0.21563969988637988, 0.1427179166668551, 
0.12691286801911897], [-0.52630199773710551, -0.085639751130001759, -0.26955877796229832, -0.051953497936385554], 
[0.53279215522834844, 0.035017313298355245, 0.35101787615838292, 0.25788331136986736]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_taggedData_rank1(self):
      arg0=Data(0.46424654058,self.functionspace)
      arg1=Data(numpy.array([0.8444155679336236, 0.10468515662474664]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.82328067854675235, 0.46077250849509999]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.39201700622470165, 0.048599721813067098]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.38220520694131521, 0.21391204306301198]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([0.38779176623433664, -0.86396579699830101]),self.functionspace)
      arg1=Data(numpy.array([0.75047047695349201, -0.70167629614424309]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.45926428861044633, -0.78891093861307615]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.29102627176451967, -0.27210429020654348], [-0.64838082374481876, 
0.60622432043307684]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.17809890964860112, -0.30593316628635314], [0.39678863714218199, 
0.68159206783952409]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_taggedData_rank1(self):
      arg0=Data(numpy.array([[0.46016583671517464, 0.36677732375681638, -0.46638268878369282, 0.67466818327766154, 
0.11772097156627503], [-0.28462916033306684, -0.39384926523711039, -0.95036567347737733, 0.46456024086119485, 
-0.90191384206445746], [-0.50518141833844687, 0.068746627077932665, 0.15907380901193635, -0.60023112577755366, 
0.57914722679372233], [-0.51325715325742838, -0.15852730717510388, 0.66966249997324412, -0.92350677269138104, 
0.38913283945179167]]),self.functionspace)
      arg1=Data(numpy.array([0.51146337700296063, 0.29761212601681741]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.51030826659793327, 0.32600873129986208]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.23535797282773618, 0.13695093298511077], [0.18759316861676953, 0.10915737909802467], 
[-0.23853766498102835, -0.13880114354635453], [0.34506806737564516, 0.20078943238116867], [0.060209965661356536, 
0.035035188624604426]], [[-0.14557739153746749, -0.084709089533105617], [-0.20143947522830724, -0.11721431715737785], 
[-0.48607723674443243, -0.28284034857700679], [0.2376055496121755, 0.13825876094558498], [-0.46129589942800231, 
-0.26842049602079926]], [[-0.25838179422252744, -0.15034811593589642], [0.035161382042842616, 0.020459829841148847], 
[0.081360427549968958, 0.047342294493635549], [-0.30699623857247643, -0.17863606144412547], [0.29621259639781672, 
0.17236123744282361]], [[-0.26251223687597042, -0.15275155257428275], [-0.081080911874964301, -0.047179648920103741], 
[0.34250784368856046, 0.19929968033077411], [-0.47233989264583931, -0.27484681401161165], [0.19902719616876427, 
0.11581065165220861]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.23482643048170837, 0.15001808061505356], [-0.18716950031376994, 
0.11957260998751847], [0.23799894148448966, -0.1520448286705901], [-0.3442887511372002, 0.21994771847873326], 
[-0.060073984942210404, 0.038378064587708462]], [[0.14524861343279258, -0.092791591451128155], [0.20098453584401946, 
-0.12839829928333324], [0.48497945946641785, -0.30982750748129878], [-0.2370689312441947, 0.15145069473551648], 
[0.46025408936459544, -0.29403178739321795]], [[0.25779825390977823, -0.16469355326878193], [-0.035081972098594358, 
0.022412000674821572], [-0.081176679738011931, 0.051859450659017939], [0.3063029053536695, -0.19568058780142822], 
[-0.2955436174101046, 0.18880705264285491]], [[0.26191936819778805, -0.16732631336403309], [0.080897795332965372, 
-0.051681286288539137], [-0.34173430956698475, 0.21831582201537122], [0.47127314036359025, -0.30107127131194722], 
[-0.19857770477697567, 0.12686070329679153]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_taggedData_rank1(self):
      arg0=Data(numpy.array([[[0.90458365695583587, -0.95884760839529637], [-0.28861525659205856, 0.71780550056875736]], 
[[0.30428718510080022, -0.49078563967888611], [0.12149524937110545, 0.97138464759661813]], [[-0.33564140254216657, 
0.94682306651707893], [-0.1092711559647126, 0.48568374193981811]], [[-0.074412694693311465, -0.85801120831705791], 
[-0.24143862300417429, -0.85929677375862035]], [[0.29963688877781491, -0.63988279814283811], [-0.41112741674881814, 
0.41563321653026519]], [[-0.83503215652634899, -0.61040002062265075], [0.056837957213350387, 
0.85992350028919207]]]),self.functionspace)
      arg1=Data(numpy.array([0.26701894195909048, 0.38300198653124418]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.20794850070178561, 0.37053946874032184]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.24154097099383215, 0.34645733759778263], [-0.25603047389371636, -0.367240538796131]], 
[[-0.077065740448462886, -0.11054021661798319], [0.19166766529428492, 0.27492093266088818]]], [[[0.081250442217325594, 
0.11654259636960689], [-0.13104906223577159, -0.18797187495802078]], [[0.032441532940128429, 0.046532921863242281], 
[0.25937810083655294, 0.3720422497154573]]], [[[-0.089622612184474496, -0.12855132393578278], [0.25281969344385197, 
0.36263511536964554]], [[-0.029177468452344316, -0.041851069805050337], [0.12968675889950218, 0.1860178379888785]]], 
[[[-0.019869599005332854, -0.028500209890681263], [-0.22910524503386157, -0.32861999725150637]], [[-0.064468685662634348, 
-0.092471472235966898], [-0.22944851535788674, -0.32911237136944066]]], [[[0.080008725013365811, 0.11476152363994457], 
[-0.17086082773792291, -0.24507638283587813]], [[-0.10977880783064348, -0.15746261733225606], [0.11098194172096497, 
0.15918834759946232]]], [[[-0.22296940293748335, -0.3198189747670605], [-0.16298836767846722, -0.23378442047718764]], 
[[0.015176811198224876, 0.021769050523091058], [0.22961586321297769, 0.32935240887566147]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.18810681522330444, 0.33518394767959298], [-0.19939092256729474, 
-0.3552908834177213]], [[-0.060017109887979724, -0.10694334384797305], [0.1492665776387678, 0.26597526883962813]]], 
[[[0.063276063924478126, 0.11275041191173849], [-0.10205813793719115, -0.18185545019199348]], [[0.02526475494911094, 
0.045018785156442337], [0.20199798107244912, 0.35993635126295564]]], [[[-0.069796126432088032, -0.12436838698523091], 
[0.19689043711209359, 0.35083531605832086]], [[-0.022722773052812961, -0.040489276079805459], [0.10099720595161812, 
0.17996499571419178]]], [[[-0.015474008294653837, -0.027572840359195397], [-0.17842214435485965, -0.31792701730304423]], 
[[-0.050206799665221687, -0.089462539101361593], [-0.17868947576098657, -0.31840337003879171]]], [[[0.062309041776294302, 
0.11102729358273444], [-0.13306266849866652, -0.23710183207991783]], [[-0.085493329910314914, -0.15233893458668796], 
[0.086430304219329263, 0.15400851124395562]]], [[[-0.17364368498743304, -0.30941237166035862], [-0.12693176911681925, 
-0.22617729936059849]], [[0.011819367985468454, 0.021060706470119997], [0.17881980260336899, 0.31863559695447524]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_taggedData_rank2(self):
      arg0=Data(-0.749393110979,self.functionspace)
      arg1=Data(numpy.array([[-0.090635890955572895, 0.76954527301886788, 0.79546776961093957, 0.26437777859790157, 
0.65298242461444556], [0.71414104296785763, -0.76225619609951067, 0.032172151688610207, -0.53478025318993638, 
-0.45166341731561466], [0.78619014812815968, -0.30065263167075829, 0.79324056224607786, 0.91463658833723405, 
0.6545715180832985], [-0.33548386133875074, 0.13669870916776605, -0.71683579746501813, -0.59658133909920918, 
0.77694886721744161]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.9491706637144679, -0.41036941576860775, 0.54486363380302127, 0.20564402853801456, 
0.12934411611822783], [-0.2336051933026857, -0.78166562063195011, 0.86084175296561227, 0.92832490341312801, 
-0.19380400057318892], [0.091819025192839465, -0.72121403994447664, 0.93087734096148789, -0.31720460158329078, 
0.21849588854976165], [-0.35760750774044081, -0.0015917666761569027, 0.91048816458575965, 0.89103581431381729, 
-0.7984823613174139]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.067921912289562722, -0.57669192618689979, -0.59611806655237842, -0.19812288597723529, 
-0.48934053059652], [-0.53517237786756933, 0.57122954215813637, -0.024109588840820342, 0.40076063762821768, 
0.33847345341761725], [-0.58916548092691112, 0.22530701097181455, -0.59444901269642914, -0.68542235834938525, 
-0.49053138629478032], [0.25140929453194027, -0.10244107093006466, 0.53719180832352142, 0.44707394565965658, 
-0.582240128675796]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.71130195653111861, 0.3075280131335284, -0.40831705359504411, -0.15410821830038543, 
-0.096929589564685656], [0.1750621225499826, 0.58577483119081597, -0.64510887931563499, -0.69568028736817211, 
0.14523538290974475], [-0.068808544936333843, 0.54047283307582383, -0.69759306648311747, 0.23771094319740038, 
-0.16373931365645697], [0.26798860273510522, 0.0011928589813981439, -0.68231355816860817, -0.66773610088246138, 
0.59837718080962521]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_taggedData_rank2(self):
      arg0=Data(numpy.array([-0.89874601249969044, 0.87708975082372009]),self.functionspace)
      arg1=Data(numpy.array([[0.52082507766048036, -0.39624889332518132, 0.39662855427161237, 0.57174800166941964, 
-0.82934772286657288], [0.47035610179759102, -0.61489579936594474, -0.84940384374349698, 0.80282561568731414, 
0.95703323706675714], [-0.04437616710486858, 0.0070695494731705111, -0.40193969661362194, -0.86901958171598492, 
0.89906708943354663], [0.49829611593312984, -0.62304557600700172, -0.89824056421645815, -0.28828781125829939, 
0.14372683065624781]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.66656069337991197, -0.09982933862684007, 0.39343646935696697, -0.63777302075842113, 
0.86010820241948327], [0.15966192569601767, 0.6943697879739974, -0.065815891185420083, 0.58569966378007443, 
-0.76657522446037651], [0.1430131208467178, 0.43708332089018298, -0.35663464645020571, -0.63386993943869285, 
-0.44510579623097191], [-0.8670032014721587, 0.3963341627256749, -0.69855849249915658, 0.011551725495012155, 
-0.66938408790877868]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.46808946175719834, 0.35612711283342191, -0.3564683315951287, -0.5138562366550572, 
0.7453729589020307], [-0.42273067094548339, 0.55263514778295253, 0.76339831756637799, -0.72153632083158248, 
-0.8601298056434189], [0.039882903235520566, -0.0063537293991812842, 0.36124169959682806, 0.78102788385139033, 
-0.80803296159810256], [-0.4478416472389839, 0.55995972704186558, 0.80729012535501388, 0.25909752082065995, 
-0.12917391594152097]], [[0.4568103375879754, -0.34754584311075815, 0.34787883983566081, 0.50147431231819117, 
-0.72741238759526206], [0.41254451612406545, -0.53931880344842864, -0.74500340565769385, 0.70415011921808601, 
0.83940404342890029], [-0.038921881348520947, 0.0062006293858590852, -0.35253718834900333, -0.76220816838820671, 
0.78856252944507665], [0.43705041616021639, -0.54646688901180229, -0.78783759264837105, -0.25285428454205744, 
0.1260613300869714]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.5990687652642247, 0.08972122002135384, -0.3535994580065307, 0.57319595928651323, 
-0.7730188172427872], [-0.14349551906731775, -0.62406207814188563, 0.059151769762009823, -0.52639523734475124, 
0.68895642626481857], [-0.12853247209612398, -0.3928268917801746, 0.32052396641635927, 0.56968808051394548, 
0.40003705950308577], [0.77921567014756832, -0.35620374836710378, 0.62782665963141193, -0.010382067226133187, 
0.60160627983875714]], [[0.58463355246547311, -0.087559289741111934, 0.34507909487326632, -0.55938417985909483, 
0.75439208894154242], [0.14003783862475544, 0.60902462431363269, -0.057726443600061177, 0.51371117216240214, 
-0.67235527260958894], [0.12543534252797028, 0.38336130100877469, -0.31280059319011644, -0.55596082723692963, 
-0.39039773190641669], [-0.7604396219425833, 0.34762063202798993, -0.61269849412187882, 0.010131900036004225, 
-0.58710992286927388]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[-0.77863180496942608, 0.49103616298953146, -0.28256978917522102, 0.85141445485928435, 
0.89534100052098475], [-0.019829524657347219, 0.86009618327105652, 0.2231447184293307, 0.53261010401118525, 
0.21973919586386392], [0.40884066766675642, 0.60155082880001731, -0.55282063302010931, 0.60038599894081979, 
-0.16019921169384288], [0.19748372867082775, 0.80209900872838369, 0.95521487864664079, -0.18023053483189555, 
0.24780780289670346]]),self.functionspace)
      arg1=Data(numpy.array([[-0.96878216704166342, -0.032013423311604283, 0.8049217154007291, 0.24139811696083879, 
-0.061131439175322821], [0.85950413875180542, -0.30074978009677067, 0.56924596923283377, -0.0017067914608279722, 
-0.037331484432138851], [0.39312624145320663, -0.26688855821339574, 0.41779360076601679, -0.98114894488416216, 
-0.52930120615507814], [0.79864183978020065, 0.62818244025942915, -0.30434820292422615, 0.97773357032029251, 
0.70680887254158509]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.67477317075489518, 0.11291746664828439, 0.24631750642024763, 0.23615923920376147, 
0.1661900932802316], [-0.19298707326067133, 0.77486579208603401, -0.60597328009063833, -0.99730021883295361, 
-0.75292496597180714], [0.28831681972948831, -0.18183326238673669, 0.12517039650035322, 0.42461861795792943, 
-0.97726043312028699], [-0.13459338070875582, 0.84900844393953112, 0.45871100834215883, -0.73690742232065154, 
0.36576515519971053]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.75432460734584239, 0.024926669576364746, -0.62673764812155641, -0.18796025152543855, 
0.047598882825460292], [-0.66923725893501029, 0.23417334412090651, -0.44323301649533176, 0.0013289621158508874, 
0.029067481105584304], [-0.30610059496355668, 0.20780791980738408, -0.32530738546911947, 0.7639537738990031, 
0.4121307535210228], [-0.62184793723216081, -0.48912282730929801, 0.23697519058209138, -0.76129445463769063, 
-0.55034386819545944]], [[-0.47570707807682172, -0.015719748547089785, 0.39524567063732569, 0.11853520510534842, 
-0.030017747330678444], [0.42204761436630839, -0.14767901803866362, 0.27952035652934759, -0.00083809632994826465, 
-0.018331108874260888], [0.19303920117367868, -0.13105193357091405, 0.20515176664172505, -0.48177961321714624, 
-0.25990603333612056], [0.39216202460856986, 0.30846029512239065, -0.14944597377667132, 0.48010254079613168, 
0.34706871673977674]], [[0.27374857269767661, 0.0090460262759371277, -0.22744655942334127, -0.068211815016919558, 
0.017273897879748815], [-0.24286990328232758, 0.084982801956438564, -0.16085171351496619, 0.00048228770325222758, 
0.01054874968558752], [-0.11108559916667964, 0.075414643627637931, -0.11805584968720982, 0.27724305050540826, 
0.14956453023343064], [-0.22567205629320194, -0.17750537970768276, 0.085999607536155973, -0.27627796883494121, 
-0.19972283410125136]], [[-0.82483514062917407, -0.027256691357029066, 0.68532198352231177, 0.20552984615627032, 
-0.05204819096022098], [0.73179424774466717, -0.25606271007014164, 0.48466424657521812, -0.0014531869211793294, 
-0.031784565466877357], [0.33471336455776129, -0.22723277629943872, 0.35571551083989567, -0.83536439404431095, 
-0.45065469789488755], [0.67997520664427546, 0.53484360992565683, -0.25912645928013284, 0.83245649477187356, 
0.60178729090469907]], [[-0.86739039472597068, -0.028662930457913596, 0.7206794140079561, 0.2161336315635991, 
-0.054733483914521257], [0.76954929554196883, -0.26927360901830882, 0.50966925563546306, -0.0015281603742183898, 
-0.033424408622404762], [0.35198204235376823, -0.23895626873838482, 0.37406774052111036, -0.87846287797269429, 
-0.47390507149585165], [0.71505678388672489, 0.56243749457159098, -0.27249542451294034, 0.87540495309352528, 
0.63283496311849197]]], [[[0.019210489868950936, 0.0006348109669235514, -0.015961215002772977, -0.0047868099125121409, 
0.0012122073804661858], [-0.01704355851247091, 0.005963725180120668, -0.011287876982997994, 3.3844863357437953e-05, 
0.00074026559104247119], [-0.0077955064983465973, 0.0052922732458563796, -0.0082846485080716094, 0.019455717195110703, 
0.010495791318615746], [-0.015836688054310637, -0.012456559188436897, 0.0060350801943052576, -0.019387991940982372, 
-0.014015683966095149]], [[-0.83324584429359783, -0.02753462320375151, 0.69231009524815867, 0.20762559904683753, 
-0.052578917512561905], [0.73925622924610446, -0.25867373798084203, 0.48960628547959362, -0.0014680048210977698, 
-0.032108667275925491], [0.3381263798175988, -0.22954983027805687, 0.35934268141392262, -0.8438824627152921, 
-0.45524994721474937], [0.68690879819552519, 0.54029731926503355, -0.26176872772053189, 0.84094491208846678, 
0.60792361357513602]], [[-0.2161786238838688, -0.0071436263308269094, 0.17961402954074956, 0.053866714838597014, 
-0.013641157781957167], [0.19179380903061602, -0.067110724997377011, 0.12702423152149214, -0.00038086149994404387, 
-0.0083303235821585667], [0.087724044456256867, -0.05955477217453823, 0.093228435404509022, -0.21893820504341127, 
-0.11811076861178003], [0.17821270846363552, 0.14017559375394018, -0.067913694045999243, 0.21817608224802587, 
0.15772066684664471]], [[-0.51598317075224176, -0.01705067271974766, 0.42870943856044397, 0.12857107618261662, 
-0.032559222177522135], [0.4577805887386433, -0.16018237165868213, 0.30318615488104755, -0.00090905437747698908, 
-0.019883125806293415], [0.20938300834991871, -0.14214754274943198, 0.2225210931591958, -0.52256984158521824, 
-0.28191117046350195], [0.42536471335301701, 0.33457631484457473, -0.16209892801508941, 0.52075077858351848, 
0.37645354712040219]], [[-0.21287941435298663, -0.0070346038953414012, 0.17687285047551815, 0.053044628104025687, 
-0.013432973286386145], [0.18886674829098465, -0.066086514834698296, 0.12508565152796872, -0.00037504898310964822, 
-0.0082031903695225453], [0.086385244169910835, -0.058645877167077615, 0.091805629869392738, -0.21559688017152434, 
-0.11630822141029014], [0.17549291565653813, 0.13803630427840671, -0.066877229373181538, 0.21484638851128574, 
0.15531361328173221]]], [[[-0.39607754799696082, -0.0130883893610148, 0.329084731543905, 0.098693367311767088, 
-0.024993018407868692], [0.3514002459496286, -0.12295874089539389, 0.23273090212776165, -0.00069780576041282669, 
-0.015262629020226771], [0.16072599503305149, -0.1091148963325827, 0.17081101468407658, -0.40113358970697444, 
-0.21639985852126162], [0.32651726300234396, 0.25682652829219732, -0.12442992248671809, 0.39973724568994995, 
0.28897221136268897]], [[-0.58277171551058948, -0.019257701325821351, 0.48420132501844027, 0.14521323732855609, 
-0.03677366790165329], [0.51703542702319361, -0.18091627947863534, 0.34243038458308028, -0.001026721817849859, 
-0.02245678540048807], [0.23648541636921216, -0.16054703339050988, 0.25132408680814095, -0.5902109609713303, 
-0.31840157924743606], [0.48042366063415032, 0.37788366757567698, -0.18308091371286409, 0.58815643957177199, 
0.42518146308059629]], [[0.53556277084256565, 0.017697680940261803, -0.44497733223946334, -0.1334498598281533, 
0.033794720902332272], [-0.47515162206817696, 0.16626068381375544, -0.31469091705544083, 0.00094354953580823666, 
0.020637614855355355], [-0.21732829765697803, 0.14754150169735372, -0.23096492284722023, 0.54239938079787486, 
0.29260862784495767], [-0.44150568742363522, -0.34727221427633465, 0.16824996619910337, -0.54051129126947561, 
-0.39073852834266881]], [[-0.58164324911536125, -0.019220411134452865, 0.483263728170025, 0.14493204959396605, 
-0.036702460175966158], [0.51603425093827171, -0.18056595715463153, 0.34176730988089005, -0.0010247336961928632, 
-0.022413300572733347], [0.23602749118473337, -0.16023615362882473, 0.25083742834698702, -0.58906808938400901, 
-0.31778503339799741], [0.47949337877236992, 0.37715194191223922, -0.18272639983850486, 0.58701754631472314, 
0.42435815100111413]], [[0.1551981394631273, 0.0051285251781402995, -0.12894782428245255, -0.038671788041504458, 
0.0097932083655968213], [-0.13769188547563457, 0.048179877688599257, -0.09119275553099751, 0.00027342664655042367, 
0.0059804743773896113], [-0.062978513976967035, 0.042755336635892295, -0.066930205493447997, 0.15717928752468846, 
0.084793635974643733], [-0.12794179315850851, -0.1006343317294751, 0.048756342188898759, -0.15663214721191734, 
-0.1132302241993758]]], [[[-0.19131871461719238, -0.006322130203093212, 0.15895894164545482, 0.047672200231543033, 
-0.012072464547356661], [0.16973808212871502, -0.059393187970441769, 0.1124168165349393, -0.00033706354174783699, 
-0.0073723607424757394], [0.077636036000527375, -0.052706147615562665, 0.082507438094084187, -0.19376095201717281, 
-0.10452837578147131], [0.15771876839232382, 0.12405581058797158, -0.060103817927741901, 0.19308647111349234, 
0.13958325160713606]], [[-0.77705921585785365, -0.025677935104239925, 0.645626910026875, 0.19362519032318723, 
-0.049033466764665919], [0.6894074176907663, -0.24123110049089913, 0.45659162764428396, -0.0013690157388361864, 
-0.029943546657377661], [0.31532616857473228, -0.21407104798391224, 0.33511183302748415, -0.78697859610648602, 
-0.42455197277572604], [0.6405898280167116, 0.50386451263266518, -0.24411739187378675, 0.78423912755437009, 
0.56693069602603186]], [[-0.92539514012573221, -0.030579698263657626, 0.76887319869655335, 0.23058707299827522, 
-0.058393660253350493], [0.82101114159409128, -0.28728066469814068, 0.54375221942083074, -0.0016303525981299141, 
-0.035659589371544474], [0.37552003502253478, -0.25493592174598573, 0.39908266365505379, -0.93720807032180464, 
-0.50559638740494361], [0.76287456806777432, 0.60004921344036133, -0.29071793172258786, 0.93394565372224503, 
0.67515435141117919]], [[0.17460412810152176, 0.0057697964052503131, -0.14507147126448022, -0.04350731172726445, 
0.011017751977611925], [-0.15490889061746566, 0.054204293717415954, -0.10259550548573439, 0.0003076159378315377, 
0.0067282734052729679], [-0.07085335275356433, 0.048101467587313804, -0.075299164115402648, 0.17683299908622255, 
0.095396239472497132], [-0.1439396459227146, -0.1132176571799622, 0.054852839388159554, -0.17621744430192507, 
-0.12738854112209896]], [[-0.24007178030010176, -0.0079331760940507656, 0.19946588179730032, 0.059820336987466909, 
-0.015148847629950214], [0.21299183220470827, -0.074528142227447455, 0.14106359294339299, -0.00042295624191063469, 
-0.009251033136000817], [0.09741975015555808, -0.066137067229130542, 0.10353251427012911, -0.24313636434616301, 
-0.131164968967865], [0.19790967961731259, 0.1556685103389788, -0.07541985948221254, 0.2422900078794212, 
0.17515275377242631]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.52539985188982674, -0.087921130868928651, -0.1917906446195656, 
-0.18388109468143124, -0.12940089229882401], [0.15026587318872339, -0.6033351503010127, 0.47183006884021733, 
0.77652966948630631, 0.58625132526117185], [-0.22449264574901612, 0.14158116129566403, -0.097461651755808759, 
-0.33062156092420575, 0.76092605496565213], [0.10479868695819568, -0.66106497713892087, -0.3571669803848006, 
0.57377955633689603, -0.28479638298807286]], [[0.33133802865576367, 0.055446559557471957, 0.12095080322974767, 
0.11596272667314196, 0.081605345731197237], [-0.094763631960499653, 0.38048712537777019, -0.29755479432988768, 
-0.48971047280435359, -0.3697133863098197], [0.14157398488531237, -0.089286707466251877, 0.061463191217411721, 
0.20850309689597943, -0.47987021332087332], [-0.066090217227016676, 0.41689384865778006, 0.22524369345739262, 
-0.36184819313483896, 0.17960391836453632]], [[-0.19067051260130616, -0.031907064745005771, -0.069601885859335522, 
-0.066731466433587475, -0.046960299621205355], [0.054532316604810831, -0.21895366350884127, 0.17122974200102883, 
0.28180691258002949, 0.21275384889941401], [-0.081469622966631711, 0.051380586617662836, -0.035369372550083635, 
-0.11998439335624583, 0.2761442745560847], [0.038032023211253388, -0.23990413701197577, -0.12961787291959687, 
0.20822777496680206, -0.1033541827924242]], [[0.57451163133194982, 0.096139563310440476, 0.20971828545109342, 
0.20106938990665391, 0.141496647673202], [-0.16431198377512324, 0.65973193595803825, -0.51593440992766326, 
-0.84911582214870418, -0.6410511994528314], [0.24547710789674484, -0.15481546797028864, 0.1065718849008687, 0.3615264291317532, 
-0.8320536589206573], [-0.11459474986381345, 0.72285806146770515, 0.39055318310559178, -0.62741363125689797, 
0.31141774022088309]], [[0.60415208582840518, 0.10109963756516988, 0.22053816264413861, 0.21144304951097037, 
0.14879680439419835], [-0.17278923926082604, 0.69376911355579507, -0.54255272288533507, -0.89292377574969373, 
-0.67412459235042621], [0.25814186984362847, -0.16280277507333557, 0.11207018803823462, 0.3801784582422903, 
-0.87498133395948863], [-0.12050697214727923, 0.76015206964758419, 0.41070277315905829, -0.65978342879191199, 
0.3274845400122221]]], [[[-0.01338043122760056, -0.0022390996891473378, -0.0048843590670965822, -0.0046829254568513485, 
-0.0032954705525071871], [0.0038268419277717562, -0.015365220330304895, 0.012016162099250886, 0.019775989280125833, 
0.014930144177870266], [-0.0057171854859538219, 0.0036056671600236821, -0.0024820694637736822, -0.0084199853547654597, 
0.019378609855208555], [0.0026689227614799952, -0.016835433873394926, -0.0090960212505174446, 0.01461252390108954, 
-0.0072529491638310926]], [[0.58036982873999421, 0.097119882088826229, 0.21185674714489894, 0.20311966028335171, 
0.14293946492778806], [-0.16598744513215519, 0.66645911032050187, -0.52119530537020098, -0.85777411179361285, 
-0.64758788952184143], [0.24798019622218215, -0.15639409497055679, 0.10765858028847862, 0.36521285265144598, 
-0.84053796858857843], [-0.11576325304114912, 0.73022892219728952, 0.39453558749950857, -0.63381126136210497, 
0.31459321396081669]], [[0.15057206919176777, 0.025196936300984761, 0.054964450614361018, 0.052697686936608312, 
0.037084441570761482], [-0.043064046123253118, 0.17290720899555836, -0.13521973696152345, -0.22254227652098932, 
-0.16801122953019229], [0.064336375556976771, -0.040575132136374967, 0.027931112882758999, 0.094751401944073707, 
-0.21807050418075222], [-0.030033802040707026, 0.18945175016701088, 0.1023589387969454, -0.16443699926222569, 
0.081618562568299857]], [[0.35939100865972201, 0.060140983656222294, 0.13119119271426388, 0.12578079695551778, 
0.08851452286761273], [-0.10278686516218039, 0.41270135011765202, -0.32274749173707395, -0.53117217328299726, 
-0.40101544443886233], [0.15356045134429691, -0.096846232792492964, 0.066667017899174424, 0.22615616627565852, 
-0.52049878093021196], [-0.071685794498507496, 0.45219047563300824, 0.24431411786419288, -0.39248433884881662, 
0.19481021735458515]], [[0.14827411393219042, 0.024812393320278688, 0.054125610787977355, 0.051893441318456433, 
0.036518477457938625], [-0.04240682429042051, 0.17026838605540109, -0.13315608128210485, -0.21914594812120874, 
-0.16544712656847202], [0.06335450612138438, -0.039955894858164497, 0.027504842272948624, 0.093305353658900664, 
-0.21474242172342323], [-0.029575441245540898, 0.18656043275290296, 0.10079678810700816, -0.16192744440685275, 
0.080372941078605778]]], [[[0.27587471365504557, 0.046165252455723289, 0.1007046137828646, 0.096551501031739081, 
0.067945268696290417], [-0.078900963882946096, 0.31679664778858418, -0.24774652042047096, -0.40773688733186703, 
-0.30782634579088347], [0.11787564107775984, -0.07434083239821794, 0.051174748477317039, 0.17360135926965523, 
-0.3995438079612017], [-0.055027247632493663, 0.34710917907495187, 0.1875397149166993, -0.30127772255016361, 
0.14953967026108445]], [[0.40591036011962278, 0.067925595648273787, 0.14817250013505354, 0.14206178607180425, 
0.099971788351075508], [-0.11609153386764649, 0.46612115943813565, -0.36452372886918855, -0.59992677320140186, 
-0.45292263730456539], [0.17343722186525887, -0.10938194969215247, 0.07529635575601426, 0.25542968155651036, 
-0.58787182349697253], [-0.080964759716348325, 0.51072173311003799, 0.27593798724791729, -0.44328727064587231, 
0.22002633225655283]], [[-0.37302853140170744, -0.062423105391531661, -0.13616939982317613, -0.13055370011017084, 
-0.091873312568848647], [0.10668723600466254, -0.42836179768662974, 0.33499453229297871, 0.55132813828632699, 
0.41623245630517869], [-0.15938748679320047, 0.10052117921674741, -0.069196777828703343, -0.2347379331716265, 
0.5402497312630633], [0.074405997923730954, -0.46934938541806959, -0.25358491000500488, 0.40737762768451963, 
-0.20220252463420249]], [[0.40512436418214209, 0.067794066011496928, 0.14788558214873218, 0.14178670073845434, 
0.099778205168119874], [-0.11586673676227333, 0.46521857262664312, -0.36381787309866309, -0.59876508812792106, 
-0.45204560782246617], [0.17310138182472909, -0.1091701448787291, 0.075150553540683063, 0.25493507311154179, 
-0.58673348136426173], [-0.080807981327648423, 0.50973278272382638, 0.27540366696865776, -0.44242889887688891, 
0.21960027808232219]], [[-0.10809813002708904, -0.018089289143520954, -0.039459870354916757, -0.03783252395466026, 
-0.026623521934819316], [0.030916377003461452, -0.12413288906070781, 0.097076441778052511, 0.15976670887913616, 
0.12061798601329698], [-0.046188127238739836, 0.02912954529407491, -0.020052198846762334, -0.068023567867389334, 
0.15655635100545343], [0.021561753488751962, -0.13601048344052907, -0.073485141931701636, 0.11805198814711014, 
-0.058595289528069722]]], [[[0.13325672176771383, 0.022299362345767036, 0.048643699604771058, 0.04663760711802474, 
0.032819839289132811], [-0.038111806812787571, 0.15302338584062436, -0.11966986282719112, -0.19695056581936415, 
-0.14869042968946858], [0.056937880598694229, -0.035909110652513739, 0.024719116620095682, 0.083855267937385611, 
-0.19299303421506236], [-0.026580002676777355, 0.16766535318219603, 0.090587960309764695, -0.14552722544509061, 
0.072232666666702833]], [[0.54123489137900982, 0.09057098806670924, 0.19757102773212792, 0.18942309166738633, 
0.13330090908055139], [-0.15479474015977643, 0.62151908372974174, -0.48605056727658819, -0.79993351693051218, 
-0.60392036885283851], [0.23125863530474267, -0.14584827951424958, 0.10039905095507207, 0.34058617255167145, 
-0.78385962467525305], [-0.107957217247895, 0.68098883128592547, 0.36793164508404291, -0.59107271296798303, 
0.29337986841307123]], [[0.64455337241664623, 0.10786044420152709, 0.23528614700375999, 0.22558281901730404, 
0.15874724978495033], [-0.18434412376506254, 0.74016333355489417, -0.57883469320488601, -0.95263600750678812, 
-0.7192051300007859], [0.27540451596968857, -0.17368983766466947, 0.11956462510323682, 0.40560202162378789, 
-0.9334937060291586], [-0.12856559982035531, 0.81098549774767259, 0.43816758016743346, -0.70390493398582998, 
0.34938431833726125]], [[-0.12161472945536873, -0.020351175405883024, -0.044393935920580094, -0.042563105987187488, 
-0.029952529395658751], [0.034782163429413003, -0.13965447613060628, 0.10921488836457378, 0.17974395182822969, 
0.13570006930538556], [-0.051963494620876891, 0.032771906130189953, -0.022559527506379087, -0.076529240614137947, 
0.17613217053131922], [0.024257836989971995, -0.1530172459280171, -0.082673730366785395, 0.13281321884644456, 
-0.06592204954451511]], [[0.16721405689841268, 0.027981829318773146, 0.06103940008099621, 0.05852210220084117, 
0.041183201878972391], [-0.047823702612192112, 0.19201778947665391, -0.15016490715336978, -0.24713877605739579, 
-0.18658068156354873], [0.071447157635329425, -0.045059701245597009, 0.031018200944461751, 0.10522380678518921, 
-0.24217276078941913], [-0.033353289957876328, 0.21039091713340424, 0.1136721671418018, -0.18261140926355382, 
0.090639459486212021]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_taggedData_rank3(self):
      arg0=Data(-0.815684173848,self.functionspace)
      arg1=Data(numpy.array([[[0.5186186715651353, -0.17878803666941812], [-0.28669503476840452, -0.56183717538691691]], 
[[0.50542826737908286, -0.93179610743232599], [-0.72505934893936219, 0.82735382283309677]], [[-0.27442360568362445, 
-0.73442057367741387], [0.039223112481604927, -0.35562775312671979]], [[0.88079417160633566, -0.48903807038661773], 
[-0.53496838967479676, -0.6719434486155027]], [[0.99501951735953709, -0.32145991271566832], [-0.25666408730183576, 
-0.76035265261734564]], [[-0.7811219974282031, -0.79021218561733142], [-0.83958109296364469, 
-0.67545403917546398]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.0037490037003344856, 0.53597278498967538], [0.86925664587023577, 
-0.69164610677631866]], [[0.42597061274731618, -0.27260563443497809], [0.2762552654339101, -0.90810803645440319]], 
[[0.45461931585779869, -0.60883510855469236], [-0.83647002942065285, -0.43613726876411119]], [[0.77651424053378504, 
0.40709976627162736], [-0.0082859153672250052, 0.27452514264385264]], [[-0.3230169718408622, 0.0045812405170664317], 
[-0.066610555656048565, 0.24531116760437843]], [[-0.46936468887416272, -0.61484031430957331], [0.78026194638694957, 
-0.5335850359285228]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.42302904265799479, 0.14583457198468303], [0.23385260258152243, 0.45828169224283133]], 
[[-0.41226983871676726, 0.76005133808615055], [0.59141943603070812, -0.67485941945798222]], [[0.22384299208656358, 
0.59905523889737566], [-0.03199367210032325, 0.29007993000675358]], [[-0.71844986619725526, 0.3989006144237548], 
[0.43636524896692924, 0.54809363675682354]], [[-0.81162167298051047, 0.26220976332887896], [0.20935683400736771, 
0.62020762528366702]], [[0.63714885114708508, 0.6445635737902613], [0.68483301019284015, 
0.5509571699174457]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.0030580029860621657, -0.43718451832956329], [-0.70903888904894929, 
0.56416478320134733]], [[-0.34745748734251819, 0.22236010171053122], [-0.22533704795674683, 0.74072935348045976]], 
[[-0.37082578107102199, 0.49661716253137356], [0.68229536489698484, 0.35575026775637913]], [[-0.63339037677136711, 
-0.33206483652517477], [0.0067586900308932128, -0.22392581417808241]], [[0.26347983181504592, -0.0037368453863644378], 
[0.05433317605989104, -0.20009643708317926]], [[0.38285334847796232, 0.50151551382633364], [-0.63644732112403268, 
0.43523686920925958]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_taggedData_rank3(self):
      arg0=Data(numpy.array([0.27669471521673916, -0.58250617233723312]),self.functionspace)
      arg1=Data(numpy.array([[[0.16186314116445177, -0.55455467264314562], [0.63053843677171084, -0.58891054306059543]], 
[[0.88879036833438829, -0.091610358420473714], [-0.30787278402660778, 0.26297782865212604]], [[0.11994646578035306, 
0.45414234274252108], [0.61398958082577337, -0.84008399595847338]], [[0.8837205760370519, -0.68413124076864862], 
[0.91600875730687648, 0.38844592024244595]], [[-0.054867484444678372, 0.69426025527543289], [0.99276653978330121, 
0.54213885844178455]], [[0.80945681446493967, 0.38793385564382588], [0.47529696475549743, 
0.72067359222248029]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.080378331695805993, 0.6998850955256406], [0.73371545076346112, 
-0.92187226001899059]], [[-0.15070037279070725, 0.76854383931705983], [0.44577559543322653, -0.81312603150992091]], 
[[0.015242070593834089, -0.089179036860166816], [-0.36561499693129251, 0.29464414303154096]], [[0.16783430341927796, 
0.50416587132724988], [0.18410409055350474, 0.026434559952948522]], [[-0.52404210717797417, -0.25098939755140615], 
[-0.84092518777437775, -0.63487489033415923]], [[0.9800349282490759, 0.16298053724950057], [0.12975394494742942, 
0.46446164827756609]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.044786675748584831, -0.15344234721910718], [0.17446665319575641, -0.16294843500028666]], 
[[0.24592359785366427, -0.025348102034056377], [-0.085186772299226887, 0.072764575407216439]], [[0.033188553190349142, 
0.12565878619300461], [0.16988767221263243, -0.23244680201987006]], [[0.24452081311774476, -0.18929549883535565], 
[0.25345478223906531, 0.10748093327858775]], [[-0.015181542983079149, 0.19209814361973654], [0.27469325500204805, 
0.15000695704447764]], [[0.22397242275862536, 0.1073392477103], [0.13151215830640287, 0.1994065743642236]]], 
[[[-0.094286278802186041, 0.32303151971308608], [-0.36729253131539175, 0.34304402628726877]], [[-0.51772587546866411, 
0.053363599229952159], [0.17933779699014696, -0.15318620837770669]], [[-0.069869556667092381, -0.26454071776720978], 
[-0.35765272058176345, 0.48935411292753794]], [[-0.51477269016299787, 0.39851067043646754], [-0.53358075504621416, 
-0.22627214616044133]], [[0.031960648349642275, -0.40441088390636276], [-0.5782926371136502, -0.31579923130620097]], 
[[-0.47151359066626186, -0.22597386537110975], [-0.27686341566322958, -0.41979681571004096]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.022240259598167639, 0.1936545071909074], [0.20301518769911728, 
-0.25507718245216632]], [[-0.041697996732381171, 0.21265201875141321], [0.12334375142896895, -0.22498767572395484]], 
[[0.0042174003822743576, -0.024675368207326943], [-0.10116373745487295, 0.081526477246392393]], [[0.046438864788196907, 
0.13950003218889256], [0.050940628905938749, 0.0073143030380609089]], [[-0.14499968160718946, -0.069447439877907247], 
[-0.23267955534981435, -0.17566652697926868]], [[0.27117048537433547, 0.045095853340121707], [0.035902230845477438, 
0.12851408349925841]]], [[[0.046820874334976455, -0.40768738807051969], [-0.42739377880891138, 0.53699628156753676]], 
[[0.087783897324108989, -0.44768153011394207], [-0.25966703581715977, 0.47365093224260846]], [[-0.0088786002001081925, 
0.051947339414136796], [0.21297299241153644, -0.17163203195888715]], [[-0.097764517671649398, -0.29367973192990232], 
[-0.10724176909994941, -0.015398294335611153]], [[0.30525776199577981, 0.1462028732648977], [0.48984411235242181, 
0.36981854228157174]], [[-0.57087639481116415, -0.094937168918672418], [-0.075582473816983187, -0.27055177693560728]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_taggedData_rank4(self):
      arg0=Data(-0.877835614466,self.functionspace)
      arg1=Data(numpy.array([[[[0.25446175854505104, -0.20354893863031731, -0.17571389889846345, -0.38437941648321261], 
[-0.89986006960559783, -0.5551265572810562, 0.26050217438044521, -0.20452486385144564], [0.069369186982854103, 
-0.58880938638583014, 0.77654937900456456, 0.20372039687038868]], [[0.78783903075766037, 0.75251834446183086, 
0.082763923216648383, -0.9186726679964623], [0.22879532741857544, 0.88873834443256516, -0.6460156917085218, 
0.50707072708753076], [-0.85574843048667759, 0.13048715584708193, 0.091158129172832281, 0.63454010694221274]]], 
[[[0.49798676906348627, -0.41995469659039086, -0.8619692830136152, 0.77881512859705793], [-0.51658319603604896, 
0.18601507832343622, -0.60089917071352095, -0.073424975494905453], [0.7066612058208408, -0.55185334500944783, 
0.98469113369071448, 0.91181487543514006]], [[-0.90526882230635297, 0.41877801499814882, -0.70712825939913482, 
0.52092380106049263], [0.7429134538300437, -0.065130455928156028, -0.81701387372714773, -0.60985897601554462], 
[-0.16865226519561261, 0.80023403426297435, -0.2246960704302674, 0.92877365511449561]]], [[[-0.36169283925788798, 
0.39555153142368527, -0.17368624437806113, -0.40896134626473901], [-0.53743057368219005, 0.021434888874746871, 
0.70077039424781962, -0.20085919793665141], [-0.3811416572736479, -0.78608635139536576, 0.070021560317791165, 
0.88745239518175012]], [[0.0062806481764750011, -0.54939586202548507, -0.75924452358638095, -0.71445035757195274], 
[0.67005253694510314, -0.76155493229923099, -0.2607010373505485, 0.47887806863141891], [-0.72192275627094227, 
0.3241627217708194, 0.95604997446291562, -0.93780488936237871]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.90253670598918534, 0.6622113781747605, -0.40349237230191082, 
-0.49290936492532511], [-0.18314385107746212, 0.29773933930589047, 0.30960250376403842, -0.9172231735221823], 
[0.55443057407782259, 0.19798293037806203, 0.97461218494401169, -0.02773328979130163]], [[-0.86632562241244071, 
0.12526065978564493, 0.24325462771000939, -0.75685876309569977], [0.89291604817504222, -0.10961478423045268, 
-0.77465895490132164, -0.062669217633242846], [0.44840045495959702, -0.36924933560741136, 0.062897985461887407, 
0.72934163371395244]]], [[[-0.24976840161406355, 0.93621824791053077, -0.3155030992891581, -0.57182632505737918], 
[0.12688324888834468, -0.3624987500996808, -0.28319526163987341, 0.47435962262593634], [0.60324520759764799, 
-0.03238412264551993, 0.20967541419065894, 0.39023770571102112]], [[0.87530762247163385, 0.047311175975118358, 
-0.27617932248072607, 0.46420301287553922], [-0.14390700376373045, 0.36681860400354349, -0.099361669795139163, 
-0.91342267142103384], [0.85501076611242066, 0.78163731778232326, 0.26607223537218117, 0.52095227053719051]]], 
[[[-0.36165740979197047, 0.90690159496925093, 0.53149902830955553, -0.98213567204180197], [0.62650851112959649, 
0.18307068921386382, -0.38613766743348132, 0.56260232600471127], [0.14521983523173576, 0.053511254205649061, 
-0.21797268302248018, -0.71201805173745147]], [[-0.19479698265739032, -0.74942067915191224, -0.69473404470941214, 
0.5728683106456538], [0.26827015242668484, -0.58113777431295177, 0.55081420746802112, 0.47424938050346399], 
[-0.58759021932752709, 0.36629494012213359, -0.20824589892713941, 0.73714543634081298]]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.22337559417052838, 0.17868250761647439, 0.15424791840977314, 0.33742194125667568], 
[0.7899292171357698, 0.48730986251728653, -0.22867808631702261, 0.17953920953263658], [-0.060894742880110007, 
0.51687784950143367, -0.68168270128176822, -0.17883301976600272]], [[-0.69159315966555568, -0.66058740330769061, 
-0.07265311939251462, 0.80644358600391897], [-0.20084468683146592, -0.78016617068457716, 0.56709558168571605, 
-0.44512474329067286], [0.75120644930470404, -0.1145462726329616, -0.080021852336016608, -0.55702190468102486]]], 
[[[-0.43715052141685118, 0.36865118912936545, 0.75666733520519147, -0.68367165696752097], [0.45347512731518536, 
-0.16329066058002004, 0.52749069275549521, 0.064455058480731298], [-0.62033237383111828, 0.48443652021156114, 
-0.86439694640274434, -0.8004235714569693]], [[0.79467721288633253, -0.36761825612080862, 0.62074237009600863, 
-0.45728546499397266], [-0.6521558882380557, 0.057173833800152431, 0.71720387587062873, 0.53535592894829409], 
[0.1480489648490963, -0.70247393518395296, 0.19724621305427995, -0.81531059223739222]]], [[[0.31750685579794941, 
-0.34722922164033182, 0.15246797105793072, 0.35900083469120536], [0.47177569788119322, -0.018816308846376761, 
-0.61516120963421106, 0.17632135744189562], [0.3345797209114541, 0.69005459530059376, -0.061467419427445803, 
-0.77903731863381565]], [[-0.0055133766512415481, 0.48227925412629408, 0.66649188289249928, 0.62716996864472563], 
[-0.58819598049379784, 0.66852004194461201, 0.2288526553145778, -0.42037622363141802], [0.63372950634818903, 
-0.28456158205270227, -0.83925471679298702, 0.82323853130277036]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.7922788638802587, -0.58131273206650758, 0.35419997457204677, 
0.43269339523533557], [0.16077019504627843, -0.26136619587032767, -0.2717801041319588, 0.80517116813142409], 
[-0.48669890367441787, -0.17379646734223228, -0.85554928623650983, 0.02434526948511468]], [[0.76049148507818254, 
-0.10995826825136522, -0.21353757558754721, 0.6643975773661932], [-0.78383350781641203, 0.096223761469512323, 
0.68002321967749635, 0.055013271169189734], [-0.39362188890635275, 0.32414021741414439, -0.055214091716618015, 
-0.64024206118702287]]], [[[0.21925559830510608, -0.82184572092894992, 0.27695985703046838, 0.50196951342465679], 
[-0.1113826347533597, 0.31821431303695952, 0.24859888651553641, -0.41640977080576319], [-0.529550127485233, 
0.028427936201476694, -0.18406054605449865, -0.34256455618068932]], [[-0.76837620461927925, -0.041531435233233507, 
0.24244004525270926, -0.40749393704463055], [0.12632669307491484, -0.32200643464306078, 0.087223212458997279, 
0.80183495203418254], [-0.75055890124545843, -0.68614907514510803, -0.23356768423031699, -0.45731045651454327]]], 
[[[0.31747575455096549, -0.79611051888015105, -0.46656877610427283, 0.86215367115592656], [-0.54997148383571326, 
-0.16070597095679115, 0.33896539655999053, -0.49387235854842304], [-0.12747914329332177, -0.046974084716469543, 
0.19134418413787116, 0.62503480395792665]], [[0.17099972896719945, 0.65786816237694778, 0.60986228702803069, 
-0.50288420548380475], [-0.23549709409840283, 0.51014343520349259, -0.48352432826936798, -0.41631299634444258], 
[0.51580762123767132, -0.32154674383794962, 0.18280566664475825, -0.64709251706114546]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_float_rank0(self):
      arg0=Data(0.633355751853,self.functionspace)
      arg0.setTaggedValue(1,-0.152797891828)
      arg1=0.374358979448
      res=outer(arg0,arg1)
      ref=Data(0.2371024128911941,self.functionspace)
      ref.setTaggedValue(1,-0.05720126284658765)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_float_rank0(self):
      arg0=Data(numpy.array([0.63644722113639873, -0.68431120809521695]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.82237587498000297, -0.82718334735978094]))
      arg1=0.884594621769
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.56299778885736107, -0.60533801429757417]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.72746927608024459, -0.73172194029170123]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[0.74001164946818787, 0.9360901483385109, 0.8095118386905078, 0.1792360871576264, 
-0.37371368090282187], [-0.41133210708840995, 0.93687006109657056, -0.62934203257624377, 0.2912861007456633, 
-0.77730023566932838], [0.67831364406596162, 0.75531690073737123, 0.52706593291064108, 0.41234743896167592, 
0.99067176687556446], [0.57596641179834562, -0.28270758180509192, -0.83743456620344237, -0.88615148782239439, 
-0.24244641683446178]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.45368650327869764, -0.82531813814898336, 0.19999740878467831, 0.14514403691181776, 
0.2826360842507305], [0.63997848064942775, 0.50401728156189241, 0.010710001558258631, -0.86172497702587814, 
-0.72338097151438174], [0.052611022475992142, 0.93665531577212424, 0.48274652992628497, -0.21684654240657109, 
-0.39304259734224845], [-0.45992489113885804, -0.73809783999666556, -0.14163459479975815, -0.77901202180262974, 
-0.95924184229460407]]))
      arg1=0.64284665619
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.47571401440263617, 0.60176242175227479, 0.52039197864881115, 0.11522131929794799, 
-0.24024059014102014], [-0.26442346962557434, 0.6022637860609138, -0.40457042124176695, 0.18725229585911687, 
-0.49968485735611035], [0.43605165793619205, 0.48555294400318599, 0.3388225725635286, 0.26507617232522701, 
0.63685003271828478], [0.37025808190260329, -0.18173762364310375, -0.53834201066221943, -0.56965952082485705, 
-0.15585586836740034]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.29165085159146814, -0.53055300540243777, 0.12856766548399368, 0.093305358794751617, 
0.18169166167935671], [0.41140802631935708, 0.32400582411428502, 0.0068848886895215358, -0.5539570200369156, 
-0.46502303868985095], [0.033820819877454451, 0.60212573774716094, 0.31033199255067628, -0.13939907469253426, 
-0.25266611944189099], [-0.29566117836739164, -0.47448372838328234, -0.091049325667919984, -0.50078527334801637, 
-0.61664541079709501]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[-0.60362853463017441, 0.26897113785785609], [0.87598660929647987, -0.4800318763472875]], 
[[-0.18409999891466677, -0.19323328103129134], [-0.84043371555949475, -0.23603874245702294]], [[-0.84596123475340468, 
0.55921725192273697], [0.7151829360345987, 0.38093163549763931]], [[-0.62952125942054971, -0.89321681334745517], 
[0.02942069160828864, -0.42788376418105645]], [[-0.54656251209708762, -0.95305821496554688], [-0.57047844126940084, 
-0.3561363690686743]], [[-0.52122933500507251, 0.89419632178942932], [0.18208312922040437, 
0.039265010085506757]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.88157372500980569, 0.8526319158736928], [-0.93828833533952882, 
0.47470523361401318]], [[0.49929024879343409, -0.53599752054046856], [-0.11754832818267125, 0.53061395418813473]], 
[[-0.038566705661979173, -0.25000026676217524], [-0.72002652652865806, 0.60358740815858325]], [[0.1872569919873035, 
-0.21509754962272587], [0.77510537241581012, -0.038640025246995169]], [[-0.76854479893497096, -0.79818264865015065], 
[0.59848370586099375, -0.65628786729704847]], [[-0.2092043351981494, 0.87193152347247072], [-0.9383569171363253, 
0.49729309923447351]]]))
      arg1=0.220499765032
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.13309995005239261, 0.059308072697970615], [0.19315484152080359, -0.10584691594231074]], 
[[-0.040594006503023229, -0.042607893063707325], [-0.18531543680560081, -0.052046487250155336]], [[-0.18653425348906624, 
0.12330727265064555], [0.15769766935032192, 0.083995336120376823]], [[-0.13880928978470128, -0.19695409746549217], 
[0.0064872555866982299, -0.094348269462809206]], [[-0.12051690549255237, -0.21014911246144996], [-0.12579036225556259, 
-0.078527985698891245]], [[-0.11493094589625617, 0.19717007884679405], [0.040149287209338781, 
0.0086579254978222074]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.19438679922279695, 0.18800513710869127], [-0.20689235747436624, 
0.10467239247121585]], [[0.11009338254157934, -0.1181873273367558], [-0.025919378744150075, 0.11700025222103354]], 
[[-0.0085039495365137659, -0.055125000078926012], [-0.15876567991617188, 0.13309088167507019]], [[0.041290122733746382, 
-0.047428959150708973], [0.170910552492507, -0.0085201164877820156]], [[-0.16946394758150868, -0.1759990864797511], 
[0.13196551651765986, -0.14471132053216532]], [[-0.046129506754808342, 0.19226069604942614], [-0.20690747974444532, 
0.10965301153309534]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[-0.35424636742508286, 0.74292550644012367, -0.53013700582339607, 0.31421154621612013], 
[-0.50957114210754484, -0.015447676316541425, 0.91458628405919895, 0.37155822464168686], [0.31474444285829772, 
-0.513780001703948, -0.75358460263266669, -0.97756498824910643]], [[-0.52843907007903757, 0.65606163418936037, 
-0.15088326947057351, 0.99905677032535856], [0.82631696184191084, -0.37461077589841141, 0.36761229568485887, 
-0.98688608273509226], [0.95807611936938963, -0.93038768457403953, 0.60049297056254902, -0.35182131635195812]]], 
[[[-0.96649160626098518, -0.83222789009046361, -0.5371881854306706, 0.63136431079841948], [0.50776657759979527, 
-0.65044689713366988, 0.8091990275376777, 0.60832306547582093], [0.52813586927080314, 0.74775370804576191, 0.23376812221636989, 
-0.96987534304690559]], [[-0.63541367084332778, 0.82714266255485036, -0.00059373647374671989, 0.64794812425901904], 
[0.0002570183053356434, -0.18564302889467821, 0.68941763320192484, -0.16533142408389789], [-0.43532640483135276, 
0.013442771356942496, -0.80052682469179914, -0.59640292245379611]]], [[[-0.060262282961882407, 0.69351794925355414, 
0.90969953458539354, -0.74299540532401775], [-0.89214959716514541, -0.7851056507426748, 0.0041470257459843562, 
-0.50504068911500388], [-0.2468238250029775, -0.1075112714220765, -0.68181914117983711, -0.7660867098235018]], 
[[0.64264676889660466, 0.84410342355953838, -0.94504796069927566, 0.28366684638748052], [0.54641715719743966, 
-0.17562711038743362, 0.67880361261856303, -0.78294899557740005], [-0.034669329494278367, 0.56581449893635627, 
-0.76009946483667834, 0.60487448184912407]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.55115972344981889, 0.62831237947834784, -0.15864463776482229, 
0.62025317837761995], [0.84779048131554813, -0.56298178702406942, -0.34563314248405397, -0.50825436105924426], 
[0.55714081354509815, -0.80386496697719756, 0.85036045924647707, -0.99278534204422098]], [[-0.60587783354208757, 
0.93872667365077511, -0.91092134233416955, 0.19098068281435143], [-0.44471592726464571, 0.60492272237201017, 
0.59861006438768349, 0.85448185457823378], [-0.10162883840159131, 0.96779245820121451, -0.99639980431215203, 
-0.97575427627587374]]], [[[-0.81573676405938444, -0.088387400077079681, -0.67821932579143018, 0.038533927392983003], 
[0.69365342884288195, -0.63516674803036621, -0.32030575296088659, -0.43829717061350038], [-0.81509416177911054, 
-0.72517086401025255, -0.95678279209236838, 0.66374791656446486]], [[-0.94787300635595129, 0.48183300748437752, 
-0.23668599856891093, 0.93316539356143613], [-0.80371555514731319, 0.80374574792614917, -0.61005365358188701, 
0.34771237813537526], [0.031685479205593614, 0.17452266746968625, 0.87844561762846118, -0.56779606264453775]]], 
[[[-0.80847077176797688, -0.74284837812906024, 0.8989668829785562, 0.51466356513927725], [0.18449815076926335, 
0.81694758116459232, 0.23249827348013197, -0.32237720954163707], [-0.079747440437225148, 0.44702392041129335, 
0.47025911511850871, 0.25971682286853159]], [[-0.33860961124431777, -0.2522719085043974, 0.73760252713697283, 
-0.62287449643849513], [0.54134748799856069, -0.69405010383308796, -0.58084965834516145, 0.47328016763767056], 
[0.69669797725128668, 0.74046406249430152, 0.78299848442112907, 0.28196566333088846]]]]))
      arg1=-0.139453143557
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.049400769531059215, -0.10360329730171661, 0.073929271977943301, -0.043817787861728782], 
[0.071061297632803971, 0.0021542270229919949, -0.12754193236612771, -0.051814962440723762], [-0.043892101973671396, 
0.071648236334312235, 0.10508974177324268, 0.13632451064255374]], [[0.07369248950083479, -0.091489857254818077, 
0.021041146237822318, -0.13932160721372808], [-0.11523249790328528, 0.052240650309342732, -0.051264690243441692, 
0.13762436657001587], [-0.13360672661290798, 0.12974548734052468, -0.083740632428800324, 0.049062588535625805]]], 
[[[0.13478029271450331, 0.11605679542888556, 0.074912581139962423, -0.088045737870508706], [-0.070809645439446983, 
0.09070686452215633, -0.11284534815335856, -0.084832563778805395], [-0.0736502071949975, -0.10427660519334959, 
-0.032599699506478771, 0.1352521654462692]], [[0.088610433858165097, -0.11534764446334192, 8.279841770840041e-05, 
-0.090358402789751435], [-3.5842010630736268e-05, 0.025888503958797143, -0.096141456173602827, 0.023055986817247295], 
[0.060707635627078892, -0.0018746367238429982, 0.11163598220493724, 0.083170262362735559]]], [[[0.0084037647969531122, 
-0.096713258136579563, -0.12686045979023025, 0.10361304492080677], [0.12441306584774885, 0.10948545102039327, 
-0.00057831577668913676, 0.070429511721257124], [0.03442035830141648, 0.014992784767613378, 0.095081822574830263, 
0.10683319992209063]], [[-0.089619112119350239, -0.11771287590256384, 0.13178990893160181, -0.039558233451621468], 
[-0.076199590264636727, 0.024491752637351618, -0.094661297637474814, 0.10918469867802737], [0.0048347469829889073, 
-0.078904610546777151, 0.10599825978743248, -0.084351647951243489]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.076860956037058129, -0.087620136454004793, 0.022123493444758578, 
-0.086496255525950638], [-0.11822704769711545, 0.078509579965817519, 0.048199628236869581, 0.070877668376242237], 
[-0.077695037852742177, 0.11210149664027645, -0.11858543919845549, 0.13844703682533147]], [[0.084491568498920436, 
-0.13090838558136256, 0.12703084472161932, -0.026632856577114665], [0.062017034046900121, -0.08435837524380678, 
-0.083478055243692534, -0.1191601807333099], [0.014172460991143495, -0.13496170060687043, 0.13895108495086245, 
0.13607200116581025]]], [[[0.11375705606305767, 0.012325900791574834, 0.094579817002692218, -0.0053736773085468637], 
[-0.096732151191199128, 0.088575999695681681, 0.044667644149772458, 0.061121918254170807], [0.1136674431550166, 
0.10112735660214142, 0.13342636805847941, -0.092561733494292808]], [[0.13218387042911714, -0.067193127563197327, 
0.033006606536351117, -0.13013284759070354], [0.11208066069091448, -0.11208487116883588, 0.085073899730398583, 
-0.048489584184642026], [-0.0044186396803284969, -0.024337734600592537, -0.12250200282211808, 0.079180945835041427]]], 
[[[0.1127437905969603, 0.10359254151628157, -0.1253637577849552, -0.071771452032900876], [-0.025728847105218448, 
-0.11392590831465144, -0.032422615108368573, 0.044956515281699846], [0.011121031259591921, -0.06233889094650804, 
-0.065579111889587108, -0.036218327383641079]], [[0.047220174726617908, 0.035180110672050256, -0.10286099110480362, 
0.086861806569802302], [-0.075492608958059204, 0.096787468765553775, 0.081001310790214928, -0.066000407160234878], 
[-0.097156723037462492, -0.10326004120578247, -0.10919160005285637, -0.039320998126613875]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_array_rank0(self):
      arg0=Data(0.159978531188,self.functionspace)
      arg0.setTaggedValue(1,-0.00305547553687)
      arg1=numpy.array(0.137376256821)
      res=outer(arg0,arg1)
      ref=Data(0.021977251786369143,self.functionspace)
      ref.setTaggedValue(1,-0.00041974979206237713)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_array_rank0(self):
      arg0=Data(numpy.array([-0.40549299224065383, 0.45915996465376585]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.39193769365165054, -0.3798991034698922]))
      arg1=numpy.array(0.288741310372)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([-0.117082577926256, 0.13257844986451944]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.11316860324918258, -0.1096925649450694]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[0.38496421777624312, -0.024425396136277033, -0.63820129158073424, -0.28050797131720473, 
0.18578608585269363], [-0.38018745794955544, 0.54618400121109434, 0.77082242704352022, 0.95600023376296583, 
-0.3139661638129807], [0.43617068800004088, -0.43657518802492579, 0.61617005863328012, -0.95548632575601555, 
-0.19508013449459227], [0.72385749004930644, 0.15015120570872065, 0.77946897162492879, 0.14081286541788618, 
-0.65639344614537287]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.035898929381477096, 0.095144530120852444, -0.73571907129177583, 
-0.32953802969372825, -0.56923796573757235], [-0.65281377385168593, 0.23833936584819959, -0.909785675101072, 
0.35161924987739779, -0.45695714849144253], [0.42992644775021582, 0.64084714744759719, 0.54432206110034831, 
0.083658185215657488, 0.16541773318665132], [-0.28669711455052149, -0.33244219788119156, 0.65954609570577594, 
-0.48268180619216827, -0.51964963001734721]]))
      arg1=numpy.array(0.0982659814262)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.037828886673756675, -0.0024001855230552779, -0.062713476264657161, -0.027564391099360944, 
0.018256452061649484], [-0.037359493681349901, 0.053671306918303741, 0.075745622298766482, 0.093942301214406262, 
-0.030852193221705505], [0.042860740725670275, -0.042900489317602521, 0.060548555537046078, -0.093891801539740499, 
-0.019169740872868612], [0.071130566672409828, 0.014754755591296532, 0.076595283488004096, 0.013837114417725742, 
-0.06450114618720873]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.0035276435278211371, 0.0093494706296613876, -0.072296156594467836, 
-0.032382377905114515, -0.055936727368263182], [-0.064149386176085357, 0.023420651697574409, -0.089400982251316, 
0.034552210677551091, -0.044903342666235052], [0.042247144329260161, 0.062973473888126755, 0.053488341545964449, 
0.0082207536745524303, 0.016254935896885626], [-0.02817257333337022, -0.032667758842282381, 0.064810944390354647, 
-0.047431201402050233, -0.051063880891422761]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[0.098691326655143463, -0.25493240294095298], [-0.96433322040979363, 0.098742149957928849]], 
[[0.10783850877609513, 0.28916781463797037], [0.0980079791048174, 0.53422199507390822]], [[0.34152209251596943, 
0.7054467305115637], [-0.31740786811527455, 0.79960870701049913]], [[0.23982128084030818, -0.59503282480030872], 
[0.56339236644284996, -0.34554089407565924]], [[0.63859946170200566, -0.54957505062495171], [0.99504744908403264, 
-0.85994119917948741]], [[-0.85651901547178411, 0.90221403424881119], [0.97223007762626912, 
-0.72436049544437253]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.88492515852348763, 0.43389081671405916], [0.068967085713649068, 
-0.2890860615080777]], [[0.53159450547635978, -0.95870493443825233], [-0.66577416732701278, -0.81036758843334833]], 
[[-0.82005561610337829, 0.36441829118494162], [0.079061033823498494, 0.22069602354404783]], [[0.47432644622192366, 
0.0025625709853034362], [0.089445738619604143, -0.32107549400323943]], [[0.18290869469038373, -0.6000155980717814], 
[-0.54880032653837918, -0.13983601525334755]], [[-0.71440155098485425, 0.30095047854014378], [-0.59811671512382603, 
0.26377382980672337]]]))
      arg1=numpy.array(-0.862969480518)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.085167602895193109, 0.21999888333308656], [0.83219013826301313, -0.085211461854396001]], 
[[-0.093061341898311681, -0.24954299878057168], [-0.084577894814675175, -0.46101727757006694]], [[-0.29472314276382855, 
-0.60877899856248341], [0.27391330305967332, -0.69003791050629082]], [[-0.20695844614385336, 0.51349516770890991], 
[-0.48619041779683136, 0.29819124585809786]], [[-0.55109184572387115, 0.47426649594331055], [-0.85869558022652337, 
0.74210300993170164]], [[0.73914976983522873, -0.778583176451487], [-0.83900488503283754, 
0.62510100046118355]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.76366340434806923, -0.37443453270113797], [-0.059516490131128343, 
0.24947244832453752]], [[-0.45874983423700488, 0.82733309924194698], [0.57454278732030528, 0.69932249681871872]], 
[[0.70768296902436567, -0.31448186343502182], [-0.068227259287857847, -0.19045393279013198]], [[-0.4093292468919465, 
-0.0022114205519770708], [-0.077188942591082957, 0.27707835226694372]], [[-0.15784462123913348, 0.51779514897053025], 
[0.47359793270077649, 0.12067421344084836]], [[0.61650673533444833, -0.25971107812734512], [0.51615647093936923, 
-0.22762876488247577]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[0.8780093409875851, 0.27468624305706979, -0.2911475392083458, -0.42464286270525764], 
[-0.37556047486974142, -0.80515461375485953, 0.76431652877965561, 0.34304528724208216], [0.73307829803652402, 
-0.1685158124983559, -0.77137248720628149, -0.98990161051878567]], [[-0.45914017483303837, -0.051839751622165142, 
0.79849913163875641, -0.88098194248805473], [0.49023723077479398, 0.54974853594627904, 0.72205173120977317, 
-0.71517480909525744], [0.53238333070921517, -0.50081579238677243, 0.72678958110318015, 0.50585252993648422]]], 
[[[-0.99240061544727465, -0.87837346981613673, 0.78792447019726208, -0.081056041823533764], [-0.36023862728379519, 
-0.71686333173024486, -0.170993593447887, -0.3807774753357891], [-0.93876177907276981, 0.9720314309815139, 0.90841663283893537, 
0.26900256991028448]], [[-0.29498026125291754, -0.61185410027635334, 0.8402831891056044, 0.48635690199875481], 
[-0.44459986108450655, 0.27690695802102927, -0.8431892376417478, -0.83634231303250317], [-0.50124282419922439, 
0.6160380340546574, -0.32985037604650458, 0.16516162120642774]]], [[[0.49955447737092573, -0.36721151849445688, 
-0.23414502536768178, -0.3469289862308782], [-0.62468896587880707, -0.30739263717406695, 0.079482650963414558, 
-0.22020245172683039], [-0.74654644585830598, -0.85236513768024991, -0.45591552304474892, -0.73743355272176214]], 
[[0.079668847855910441, -0.43171841888892626, -0.17457482587935003, -0.43154285988811747], [-0.03588836402140716, 
-0.3412587843717334, 0.85539585492226822, 0.49801939352968971], [-0.818814177842216, 0.31197621045334478, 0.91519430470675411, 
0.3023843735374947]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.74235237138102583, 0.20066551240529984, -0.79298186709738849, 
-0.2993896905116733], [-0.64593505610264512, -0.71796737833811841, -0.27490652343615052, -0.3814904284826901], 
[0.47676434322176364, -0.71488967204752196, 0.22794978429612689, -0.75926296416679295]], [[-0.43813228934311188, 
0.88471790641972903, -0.37821445105166629, 0.82283024215544032], [-0.86289585414761305, -0.47660911722395749, 
-0.10858805391176052, 0.21796719856277647], [0.29467471056545258, -0.10279935628418424, 0.18634959550501273, 
0.67537645482778341]]], [[[-0.8458114514933881, 0.46152382606223341, 0.39825050482258462, 0.70645135683570781], 
[0.22982191607494218, 0.2331654701312269, 0.95705913854339864, -0.72094212993191675], [0.19109749605440296, 
0.59127584421940282, 0.039333660396231096, -0.20638005491904821]], [[-0.70611572325008276, 0.27788772305359921, 
-0.82987942633453549, -0.0019792859930398699], [-0.098162647690510196, 0.94608597581061282, 0.16985088611524479, 
0.83497322619687875], [-0.48028655940389431, 0.89460449950166354, 0.96320968850035849, 0.32396020770333367]]], 
[[[0.18285578167268302, 0.09943408814469934, -0.041994057146080621, -0.95846335129458504], [0.43088554116964906, 
-0.42802091556279054, -0.80595804649860958, -0.69009626512091504], [-0.50285123512239349, -0.46092422375665443, 
-0.99526432782173258, 0.59974065805215826]], [[0.70441089171472493, -0.3937110956676193, 0.28190305201744459, 
0.46695878813120006], [-0.12211784626701827, 0.23364563250866244, 0.63819883786001252, 0.068274385133155757], 
[0.69748780753014183, 0.77578619606720989, -0.23407878688129857, -0.055774257523006243]]]]))
      arg1=numpy.array(-0.207075436453)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.18181416749489704, -0.056880773668703206, 0.060289503753812312, 0.087933106131382222], 
[0.077769349248182698, 0.16672774305550933, -0.15827117878536046, -0.071036252578830936], [-0.15180250852020405, 
0.034895485422344691, 0.15973229445614889, 0.20498430804379755]], [[0.095076652096701006, 0.010734739192779801, 
-0.16534955619151151, 0.18242972024800791], [-0.10151608852824631, -0.1138394180205248, -0.14951917738197543, 
0.14809513573365815], [-0.11024351056696226, 0.10370664879109268, -0.15050026971650191, -0.10474963341749893]]], 
[[[0.20550179058006285, 0.18188956963099454, -0.16315980355817039, 0.016784715237768442], [0.074596570972055104, 
0.14844478729525909, 0.035408572993904, 0.078849661896665507], [0.1943945051269762, -0.20128383281662188, -0.188110770726372, 
-0.055703824571175911]], [[0.061083166343995396, 0.12669995486034061, -0.17400200812824021, -0.1007125677533665], 
[0.092065710281058866, -0.057340629189103029, 0.17460377939721597, 0.1731859494953952], [0.10379507659003553, 
-0.12756634477357376, 0.068303910584046904, -0.034200914796621504]]], [[[-0.10344546143368138, 0.076040485462842811, 
0.048485683321333316, 0.071840471241988307], [0.12935774025678551, 0.063653464505287277, -0.016458904638697949, 
0.04559851879937462], [0.15459143110861434, 0.17650388290253874, 0.094408905920231712, 0.15270437478501411]], 
[[-0.016497461441477745, 0.089398280016263779, 0.036150158262689185, 0.089361926059548027], [0.0074315986433203798, 
0.070666311717228758, -0.1771314699981956, -0.10312758327726534], [0.16955630325065768, -0.064602609942608469, 
-0.18951426008653641, -0.062616376126871917]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.15372294130571476, -0.041552898562411081, 0.16420706622858058, 
0.06199625083226129], [0.13375728366280856, 0.14867340822844907, 0.056926388324343401, 0.078997296980730644], 
[-0.0987261844579191, 0.14803609085504937, -0.04720280107250896, 0.15722470968750801]], [[0.090726435039917869, 
-0.18320334660973239, 0.078318922524390905, -0.17038793152114229], [0.17868453561118175, 0.098694040966674529, 
0.02248591865736984, -0.045135652774844999], [-0.061019894302030052, 0.021287221569644497, -0.038588423822057911, 
-0.13985387415360612]]], [[[0.17514677547501772, -0.095570247715338522, -0.082467897103801466, -0.1462887230496342], 
[-0.047590473577704843, -0.048282841493214511, -0.19818343882529579, 0.1492894062130744], [-0.039571597400558774, 
-0.12243870350590406, -0.0081450348938473058, 0.042736239947575277]], [[0.14621922157840256, -0.057543721536280387, 
0.17184764441166667, 0.00040986151087422533], [0.020327073113903634, -0.19591116636313335, -0.035171946374258967, 
-0.17290244524136611], [0.099455548911115951, -0.18525061718720812, -0.19945706664205981, -0.067084201403602592]]], 
[[[-0.037864940797842382, -0.02059035720087897, 0.008695937711960788, 0.19847421679362073], [-0.089225811499032401, 
0.088632617901217459, 0.16689411424158207, 0.14290198529456311], [0.1041281389839467, 0.095446084806212472, 
0.20609479506987985, -0.12419155852481609]], [[-0.14586619284413935, 0.081527896971797853, -0.058375197533971412, 
-0.096695694857875797], [0.025287606314454564, -0.048382271327090332, -0.13215530289371905, -0.014137948100014815], 
[-0.14443259216501528, -0.16064626514490254, 0.048471966957855522, 0.011549478719423743]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_constData_rank0(self):
      arg0=Data(0.0320246686947,self.functionspace)
      arg0.setTaggedValue(1,-0.0322179747097)
      arg1=Data(0.232760492341,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(0.0074540776524510011,self.functionspace)
      ref.setTaggedValue(1,-0.0074990716556609226)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_constData_rank0(self):
      arg0=Data(numpy.array([0.73787753264818545, -0.76105249568081179]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.53458444670356053, 0.082544198032926452]))
      arg1=Data(0.523771355839,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.38647911571804855, -0.3986174975271588]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.28000002046020067, 0.043234286520323351]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_constData_rank0(self):
      arg0=Data(numpy.array([[0.68285511936187926, 0.95336003540261305, -0.42121042221384086, -0.24659918725652963, 
-0.26779615095049314], [0.074402775175222757, -0.76201125969000572, -0.13954868340213666, -0.74366210907947217, 
-0.20319844591728131], [-0.62915556717207122, -0.67512226213635262, -0.8610644123188651, 0.2345856340414223, 
-0.41662812336114907], [0.28479513722211069, 0.93230981985582395, -0.0061458454666099005, 0.81256583372427649, 
-0.25872805020813971]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.83608018704422427, -0.87213228048373814, 0.41740719419697636, -0.36389252937888195, 
0.61006498214427585], [0.23419591403286755, -0.69253863876442145, 0.46877352145576756, -0.37394535395578377, 
-0.38975383367986027], [-0.78969417571211542, -0.36426572703615667, 0.10950650536761719, -0.79181681846069374, 
0.56125355243786657], [0.046301625514615807, -0.026699313006583569, -0.96143671694179922, 0.66153115042387944, 
0.44020552859864348]]))
      arg1=Data(0.729365312588,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.49805083758583291, 0.69534774023052881, -0.30721627126340773, -0.17986089329735597, 
-0.19532122334792448], [0.054266803373106264, -0.55578458061953262, -0.10178196909087182, -0.54240134664875284, 
-0.14820589802389528], [-0.45888424691706692, -0.4924107597583367, -0.62803051424952783, 0.17109862430132416, 
-0.30387410142834159], [0.20771969428360543, 0.6799944431881958, -0.0044825664998727312, 0.59265733331280213, 
-0.18870726521539655]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.60980788697231525, -0.63610303337328544, 0.30444232867204313, -0.26541058843894066, 
0.44496023640077725], [0.17081437604546276, -0.50511366074182207, 0.34190714600965938, -0.27274276997886709, 
-0.28427292673436244], [-0.57597553931735168, -0.26568278586489608, 0.079870246517894025, -0.57752372130918161, 
0.40935887271508486], [0.033770799566809749, -0.019473552776937145, -0.70123859158603263, 0.48249787431574748, 
0.32107064296940546]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_constData_rank0(self):
      arg0=Data(numpy.array([[[-0.71301904452735698, 0.56477325063315953], [0.18748041118965642, -0.41541702588215679]], 
[[-0.009408924974988464, -0.70879870624707864], [-0.71485764294836351, 0.62062281147113052]], [[0.6162287835724074, 
0.1675939254874339], [0.22368221214787321, -0.58680922783641321]], [[0.29616104794997566, -0.82638158963606823], 
[0.07203208340962064, -0.2017895252625761]], [[0.88565902506671135, -0.10559732592288618], [-0.94873807738735905, 
0.55392327986841261]], [[-0.39572950566408305, -0.09667490830049319], [0.46186159439181407, 
-0.95367397071007809]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.55305038226431624, 0.15847440775137467], [2.8645079125366735e-05, 
0.48297072843446531]], [[0.96095323914377961, 0.361421059976756], [0.64749491899846756, 0.07098205170252303]], 
[[0.14364872869009515, -0.026517528320679773], [0.14777154123175595, -0.90053415698988393]], [[0.86792067686366536, 
0.40070995708481205], [0.54601886303699043, -0.94338404324001579]], [[-0.63769483682668082, 0.1582295924832311], 
[-0.31638317836958341, -0.74470839979345316]], [[0.51683794637872249, 0.62184071051217038], [-0.033815453412377794, 
0.34273405404343649]]]))
      arg1=Data(-0.65349658148,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.46595550812848807, -0.36907738859989869], [-0.12251780780683273, 0.27147360630241757]], 
[[0.0061487003065531188, 0.4631975314896386], [0.46715702591132963, -0.40557488568465039]], [[-0.40270340347390882, 
-0.10952205738278711], [-0.14617556097643281, 0.38347782437178762]], [[-0.19354023240272897, 0.54003754382485936], 
[-0.047072720265041466, 0.13186876493748756]], [[-0.57877514523766316, 0.069007491503994683], [0.61999709029218009, 
-0.36198696979598283]], [[0.2586078791420956, 0.063176722089227408], [-0.3018249730517743, 
0.62322267970512502]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.36141653419572412, -0.1035624837175297], [-1.8719461284640197e-05, 
-0.31561971998664179]], [[-0.62797965674222145, -0.23618742716954838], [-0.42313571609091777, -0.046386528134007975]], 
[[-0.093873953132869642, 0.017329114106853064], [-0.096568197034926265, 0.58849599309851464]], [[-0.56718319532587758, 
-0.26186258711976729], [-0.35682146041805851, 0.6164982472797661]], [[0.41673139589343583, -0.10340249777670386], 
[0.20675532550217754, 0.48666439346417489]], [[-0.3377518311374389, -0.40637077854455728], [0.022098283206172017, 
-0.22397553267403483]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank4_constData_rank0(self):
      arg0=Data(numpy.array([[[[-0.18409688019144577, 0.78455115060078451, 0.84279147421249423, 0.5393932099082468], 
[0.96903650587366053, 0.70553944513924449, -0.76639273329804602, 0.14310940478860701], [-0.79261332483434344, 
-0.66024730596790704, 0.50664729746942072, 0.25615511518215328]], [[0.69692167057091825, -0.75209364373071197, 
-0.22480041334607259, -0.74368443586106392], [0.31197764529849858, 0.71390868516629302, 0.13968244145215958, 
0.9036885054830921], [0.69209343515387189, -0.92204126152266985, -0.47804002968784753, 0.27735750614406762]]], 
[[[0.43355065967716389, -0.30778405557432609, 0.52869891383649792, 0.36481839519874293], [-0.099068025390791181, 
-0.52144151942174388, 0.33093502745704884, 0.079619903641700462], [-0.2436422200381636, 0.97403275677367862, 
-0.31536321555160085, 0.2571304533028036]], [[-0.56097936318045272, -0.89301545281736749, -0.54408788164042554, 
0.49481698442114586], [-0.25729747827768001, 0.65707333197840034, 0.90387777999765806, -0.015021157497129733], 
[0.88971318740517602, -0.51015558201690103, -0.44319693398142768, -0.90966088850950966]]], [[[0.38608038791821464, 
-0.06826526048235193, 0.78028951179444261, -0.59653685463126971], [0.43817426625036004, -0.74424924860485664, 
0.48560546086411338, -0.59923704401274502], [0.60247285173802267, -0.51300024335769945, -0.73641230511449174, 
-0.04100923427149139]], [[0.35991881852776175, -0.43821819013868613, -0.66131040688011722, 0.5500670200830764], 
[-0.31284003251765924, 0.68386394333049805, -0.79889696449792336, -0.47774092570625903], [-0.82585372594347084, 
-0.64321880126565145, 0.9953048466004224, 0.51671547514654814]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.34919278676042986, 0.96441333659299033, 0.5860970337638689, 
0.41707911174145917], [0.63477511488224581, 0.073261910824590881, 0.39091589613046285, -0.40591434824626127], 
[-0.13305363400647474, 0.48683452294019425, 0.063919930586032736, 0.67687292462194537]], [[0.589238689579896, 
0.9036247264524726, 0.2086708714852108, 0.9222155964862313], [-0.96510716800982377, 0.13611081315090034, -0.95133575898870149, 
-0.87343049274234175], [0.47581408493530164, 0.75293677516983792, -0.87357469224314421, -0.86716458861373114]]], 
[[[-0.16391142956966198, 0.16281507101730042, -0.43243608984658888, 0.93748919533328423], [0.76400837710432667, 
0.34631373218972006, -0.98979680029931316, 0.063731596251182188], [-0.07978594980477105, -0.16699550591922363, 
0.2977390766846344, 0.75893856527683012]], [[-0.10646630765434284, 0.91647070646602713, 0.74010181188636937, 
-0.84735546924526584], [-0.95664705415388607, 0.42763339051204796, 0.29275267779895575, 0.80508402160156667], 
[0.10702599684169423, 0.17325272059972652, -0.75859694150685852, 0.09377386695047929]]], [[[-0.94810131215354776, 
-0.48235410815768143, 0.85672009176240804, -0.031540636961182589], [-0.54836238423802919, 0.60703516238800037, 
0.10394212112579249, -0.68885829062054071], [-0.12618567248793333, -0.3532036579002662, 0.64103950450510361, 
-0.43495587627575216]], [[-0.5251932483988937, 0.18341920825257896, 0.11528498612463434, -0.11375516013736231], 
[-0.52745675922617896, 0.1481534994990783, -0.7619939656995125, -0.44627628499681782], [-0.82771367676593122, 
-0.26484780737509572, -0.70021546723787442, 0.48637008080779109]]]]))
      arg1=Data(-0.249002559601,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.045840594382172577, -0.19535524463726608, -0.20985723428855463, -0.1343102898983887], 
[-0.24129257030905588, -0.17568112773892636, 0.19083375225058663, -0.035634608095295173], [0.19736274665736903, 
0.1644032691554724, -0.12615647388466072, -0.063783279335167142]], [[-0.17353527981335135, 0.18727324234836085, 
0.055975878322466428, 0.18517932806460419], [-0.077683232217524031, -0.17776508992757403, -0.034781285452862028, 
-0.2250207509470171], [-0.17233303683615209, 0.22959063417659908, 0.1190331909838665, -0.069062728954338473]]], 
[[[-0.10795522397618386, 0.076639017642289906, -0.13164738280339555, -0.090840714193905076], [0.024668191896893697, 
0.12984027301809001, -0.082403668898331603, -0.019825559801944153], [0.060667536416298601, -0.24253664957156765, 
0.078526247876254479, -0.064026141023685201]], [[0.13968529731510088, 0.22236313351449863, 0.13547927517618621, 
-0.12321069565476279], [0.064067730669946671, -0.16361294150797923, -0.22506788078561119, 0.0037403066649504806], 
[-0.22154086097438211, 0.12703004571679091, 0.11035717096855596, 0.22650788960751081]]], [[[-0.096135004803264881, 
0.01699822459191384, -0.19429408566639331, 0.14853920369933415], [-0.1091065138474963, 0.18531996788350366, 
-0.1209170027112396, 0.14921155776672815], [-0.15001728217269797, 0.12773837367184684, 0.1833685488949568, 
0.010211404300865919]], [[-0.089620707061870905, 0.10911745100811711, 0.16466798400372659, -0.13696809595261319], 
[0.077897968842461959, -0.170283872307919, 0.19892738901720888, 0.11895871332686421], [0.20563969161569556, 
0.16016312789843828, -0.24783345438648277, -0.12866347589678004]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.08694989769744689, -0.24014138932470169, -0.14593966158157862, 
-0.10385376637960783], [-0.15806062837650475, -0.018242403316561032, -0.097339058725084873, 0.10107371169196715], 
[0.033130695431786351, -0.12122304231409188, -0.015916226325420935, -0.16854309075527302]], [[-0.14672194192115376, 
-0.22500486980514398, -0.051959581113925277, -0.22963404402875376], [0.24031415512342461, -0.033891940863906192, 
0.23688503902785704, 0.21748642832613979], [-0.11847892504295283, -0.18748318423478305, 0.21752233437093274, 
0.21592620215990319]]], [[[0.040814365510654961, -0.040541369424876815, 0.10767769323551696, -0.23343720923598418], 
[-0.19024004145535076, -0.086233005740110072, 0.24646193675910744, -0.015869330593982454], [0.019866905721560604, 
0.041582308415699779, -0.074137792187621732, -0.18897764533361022]], [[0.026510383117166487, -0.22820355170909842, 
-0.18428724552481843, 0.2109936807337196], [0.2382075651187828, -0.10648180880822372, -0.072896166101897683, 
-0.20046798207241173], [-0.026649747157397815, -0.043140370887116052, 0.18889258014046686, -0.02334993289432441]]], 
[[[0.2360796534870114, 0.12010740756517339, -0.21332549571018233, 0.0078536993347707671], [0.13654363726400937, 
-0.15115330920223594, -0.025881854210647856, 0.17152747756667441], [0.03142055543443046, 0.087948614877494696, 
-0.15962047742693239, 0.10830512650602572]], [[0.1307744631363284, -0.045671852334825173, -0.028706256628564616, 
0.028325326041990222], [0.13133808312600634, -0.036890600589070857, 0.1897384478594632, 0.1111239372532971], 
[0.20610282413121961, 0.065947781941030792, 0.17435544361422767, -0.1211073950343371]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_taggedData_rank0(self):
      arg0=Data(0.606963180111,self.functionspace)
      arg0.setTaggedValue(1,-0.105032578026)
      arg1=Data(0.641726111768,self.functionspace)
      arg1.setTaggedValue(1,0.203637657913)
      res=outer(arg0,arg1)
      ref=Data(0.38950412155944253,self.functionspace)
      ref.setTaggedValue(1,-0.021388588193799088)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_taggedData_rank0(self):
      arg0=Data(numpy.array([-0.89668839347075524, 0.6931745538578824]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.064333146340301406, -0.35956760795752674]))
      arg1=Data(-0.793424890874,self.functionspace)
      arg1.setTaggedValue(1,-0.623609340988)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.71145489073729362, -0.54998194475115181]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.040118750992943535, 0.22422971903893318]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_taggedData_rank0(self):
      arg0=Data(numpy.array([[-0.33914850910694949, -0.95877907109519822, 0.25637698875540638, 0.3239385951314937, 
0.73986285084725356], [-0.12400487779353986, 0.11983832857744248, 0.0052254705557972514, 0.94781250844331866, 
-0.5052940688889358], [0.96936200910119141, 0.89714113998681611, 0.049166299169957961, -0.93230325874009612, 
-0.31669549933194374], [0.0092686023875203638, -0.76978676397921331, -0.40118411124144537, 0.54121990351518368, 
0.63144629024264431]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.39413771761982219, -0.68866062578937437, -0.93663343736361115, 0.54800652914746872, 
-0.26548451616588742], [-0.82434835419616515, -0.030912746361342558, 0.6733781857143839, -0.060688717461907737, 
0.13175739107998274], [-0.87866388995216393, 0.56348010106594248, -0.78367184735182316, 0.9136773529687463, 
0.39618827345650454], [0.71146668845288263, 0.83989805783762983, -0.54193380366088051, -0.43629167741746988, 
-0.98342129171225268]]))
      arg1=Data(0.524959005948,self.functionspace)
      arg1.setTaggedValue(1,-0.189857739599)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.17803906420942872, -0.50331970808559445, 0.1345874091649028, 0.17005448288832339, 
0.3883976667184057], [-0.065097477379162716, 0.062910209844448062, 0.0027431578285803007, 0.49756271225720061, 
-0.26525867211520443], [0.50887531670122688, 0.47096232104226543, 0.025810291538388361, -0.48942099195000227, 
-0.16625215451740769], [0.0048656362958771696, -0.40410649441022461, -0.21060521223932069, 0.28411826254844097, 
0.33148341683514143]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.074830096158015127, 0.13074754976321518, 0.17782710725071277, -0.10404328090944129, 
0.050404290137794093], [0.15650891516985382, 0.005869024148962238, -0.12784606023502007, 0.011522222716481199, 
-0.025015160445909221], [0.16682114001359735, -0.10698105829740553, 0.14878616552560311, -0.17346871695745947, 
-0.075219410054089206], [-0.13507745726966242, -0.15946114675465692, 0.1028903269753523, 0.082833351680344364, 
0.18671014351803389]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_taggedData_rank0(self):
      arg0=Data(numpy.array([[[0.57281734438460474, -0.44526159113844965], [0.651028657014038, 0.12278935849380712]], 
[[0.65585152953475401, -0.79530035527043963], [0.15406121541771678, 0.92144697497597305]], [[0.12610596572779031, 
0.97614188822712356], [-0.19590264087200393, 0.4405278989150414]], [[-0.1986401668193476, 0.78853175247174545], 
[-0.62191798221550232, -0.91047971491620761]], [[-0.30250962246894697, 0.089092522352745895], [-0.61698158560904015, 
-0.6663560056585589]], [[-0.19415046998095176, -0.3736693493475276], [-0.82503794413128495, 
0.75206469974806778]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.84537687051522425, 0.20539574947802608], [-0.56822369852885291, 
-0.78558816863154335]], [[-0.69323261396765234, -0.22934901737219682], [0.26504731364221645, 0.8096982400215722]], 
[[0.19422059513021828, 0.16665056489528274], [0.52004015976219331, 0.60361129312247863]], [[0.89510859323167891, 
0.17689319616807397], [0.73594618983621296, -0.35988032075806853]], [[0.35264550443626574, 0.078778556157776514], 
[-0.072604357657335417, -0.69368844049547929]], [[-0.97574456212244276, -0.88793772218261235], [-0.3628395582769135, 
-0.40333053998676038]]]))
      arg1=Data(-0.848644873463,self.functionspace)
      arg1.setTaggedValue(1,-0.99654276544)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.48611850274288509, 0.37786896666977948], [-0.552492132252693, -0.10420455960162286]], 
[[-0.55658503829276595, 0.67492756936383957], [-0.13074326066377817, -0.7819812514816713]], [[-0.10701918132803417, 
-0.82839780921678285], [0.16625177187395787, -0.37385174303183077]], [[0.16857495923514398, -0.66918342929821806], 
[0.52778750732185697, 0.77267394245601206]], [[0.25672324028160531, -0.075607912358576723], [0.5235982596484009, 
0.56549960810365107]], [[0.16476480102983473, 0.3171125776941649], [0.70016422169975678, 
-0.63823585195395127]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.8424542043823231, -0.20468564819447024], [0.56625921592053419, 
0.78287220606508723]], [[0.69083594621658062, 0.22855610402305429], [-0.26413098290947901, -0.80689892328306434]], 
[[-0.19354912897648613, -0.16607441480289681], [-0.51824225894931786, -0.60152446729913844]], [[-0.8920139928682782, 
-0.17628163489686724], [-0.73340185123447077, 0.3586361300757091]], [[-0.35142632621092873, -0.078506200210847352], 
[0.072353347362841783, 0.69129019684518223]], [[0.97237118430062042, 0.88486791320242719], [0.36158513681633292, 
0.40193613170484749]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank4_taggedData_rank0(self):
      arg0=Data(numpy.array([[[[-0.10529784183776658, -0.13662538688498804, -0.54080028147417103, -0.51514156559829649], 
[0.34613635182903324, 0.94178417250878144, 0.2590581462097048, 0.1385738227879898], [0.09738285943524172, 0.44312186170616963, 
-0.23895090092155424, 0.81815940853232272]], [[0.83089753639975195, -0.11147581884524094, 0.70424032355363808, 
0.20391930604746378], [-0.48421266208977709, -0.66891293629795134, -0.39539450992786684, 0.91423639975741389], 
[0.82895779478297293, -0.033826833541758861, 0.95636241032638591, -0.019579073291774884]]], [[[0.12583448875122238, 
0.21198652011653873, 0.75576437157453569, 0.048103664794585166], [-0.62819760048297546, 0.57504881605132363, 
-0.53537823307120225, 0.51250240127644608], [0.3855871589867399, 0.80601610824431846, 0.22904657120400218, 
0.96047344489231667]], [[-0.32227354362265226, -0.80699045176398765, -0.93143285935032072, -0.74988642705263597], 
[-0.19004296597593684, 0.62192293867780224, 0.36463056610169464, 0.32612109759408536], [0.18223902633271027, 
0.91225589621589842, 0.85414964134752847, -0.68029926880728642]]], [[[0.50912703457385189, -0.54217117669600934, 
0.54996588263929791, 0.92925415934258737], [-0.71278051925355301, 0.34718837847410811, -0.95990815196778678, 
0.9626602377832747], [-0.62039478456208275, 0.92382266562784543, -0.52741265777151125, 0.65998599451304152]], 
[[-0.24384583226758405, -0.93432059764897435, 0.87235815513111836, -0.81277744149384756], [0.036740909524032972, 
0.99899387868905598, 0.41564896596707412, 0.70027225063133369], [0.7491808458717264, -0.22408116662102073, 0.48419277847287456, 
-0.78482331171933106]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.17087946661044162, -0.62016103009229506, -0.79997212606881241, 
0.2349293045101768], [0.026090958411256526, -0.55066814679290599, 0.30375589787923252, 0.71206341161109021], 
[-0.298006961251851, -0.6103108457155082, 0.13930379119412817, 0.51730719780169943]], [[-0.1081834717535517, 
0.91498292398909142, 0.31618944477490407, 0.59035039135277212], [-0.51572121624015166, -0.6252179348846798, 
-0.53632854354773141, -0.62010239374435194], [0.57623565164072454, -0.17418528033256631, -0.82038108984672453, 
0.84640669226212117]]], [[[-0.3291803712311725, 0.97503373308624131, 0.17149279483512481, -0.98098606464013516], 
[-0.37417152367663298, 0.37281271178417774, 0.46095849349376206, 0.77171551922940029], [-0.852563769248325, 
0.061082083643657858, -0.69257406612249128, 0.39191824852653978]], [[-0.71789014693209552, 0.42810366806893874, 
0.23288148653145546, 0.66913283750982355], [-0.34433481235129215, 0.12474012833401993, 0.71201919018370785, 
-0.20050613520500149], [0.33398113132311313, -0.92236367646660011, 0.24099143506448217, 0.42716493349309914]]], 
[[[0.69848308593933006, 0.14618452923497749, -0.74488894364817759, 0.040031599174944255], [-0.63714568051557086, 
0.72402883075389513, -0.90674044778988816, 0.74293569434400242], [0.8449464626805212, 0.13926364315114204, 
-0.70474555265395766, -0.36145185127452417]], [[-0.19777885631078784, -0.99480567667506903, 0.87728315715325844, 
-0.24163484938451574], [-0.48024718435008329, 0.96704048507381013, 0.86747878174924975, -0.76009127124359899], 
[-0.61440821159595593, 0.46664129362537032, 0.71672047015908569, 0.087399927359649299]]]]))
      arg1=Data(0.690835681283,self.functionspace)
      arg1.setTaggedValue(1,0.846915159773)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.0727435063036624, -0.094385692229295215, -0.37360413089044914, -0.35587817442748282], 
[0.23912334243275168, 0.65061811043700224, 0.17896661092878921, 0.095731741273782409], [0.067275554043268224, 
0.30612439322333856, -0.16507580843141784, 0.56521371239182916]], [[0.57401366563539935, -0.077011473258573965, 
0.486514343709401, 0.1408747327201324], [-0.33451138430082711, -0.4621089240666571, -0.27315263564172348, 0.63158712608047174], 
[0.5726736229140581, -0.023368783595480256, 0.66068927729163807, -0.013525922436420406]]], [[[0.086930954765395724, 
0.14644785204760061, 0.52210899452639503, 0.033231728040594273], [-0.43398131731023687, 0.39726424060801352, 
-0.36985838638803276, 0.35405494554517852], [0.26637736767272496, 0.55682468726433709, 0.15823354406333759, 
0.66352932665677233]], [[-0.222638063068162, -0.55749779853355186, -0.6434670539589995, -0.51804830071806263], 
[-0.13128846187309931, 0.42964655704723725, 0.25189980554960639, 0.22529609063729153], [0.12589722191297656, 
0.63021892356708475, 0.59007704939826899, -0.46997500884306259]]], [[[0.35172312178961068, -0.37455119422499594, 
0.37993605521573126, 0.64196193025484494], [-0.49241421562404508, 0.23985011997683023, -0.66313880213413012, 
0.66504004121342297], [-0.42859085365759819, 0.63820966059403483, -0.36435548274905671, 0.45594187415690191]], 
[[-0.16845740166268766, -0.64546200661391795, 0.60265614042311311, -0.56149565752615949], [0.025381931262006108, 
0.69014061678207383, 0.28714513657859342, 0.48377305734873877], [0.51756086006224822, -0.15480326540540601, 
0.33449764798879789, -0.54218394723869756]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.14472041076637146, -0.52522377788583763, -0.67750852096381564, 
0.19896518946468453], [0.022096828211509833, -0.46636920152322292, 0.25725547478449556, 0.60305729801338182], 
[-0.25238661320218969, -0.51688150741057393, 0.1179784925762121, 0.43811530807814381]], [[-0.091622222264997763, 
0.77491290926013223, 0.26778563414019296, 0.49997669601480815], [-0.4367721162505479, -0.52950654721603885, 
-0.4542247741497491, -0.52517411787385082], [0.48802270897641975, -0.14752015452302572, -0.69479318178259519, 
0.7168346590104292]]], [[[-0.27878784669550771, 0.82577084984116544, 0.14523984773777254, -0.83081196967015558], 
[-0.31689153575724321, 0.31574073736624253, 0.39039273616616427, 0.65357757226776148], [-0.72204918084993697, 
0.051731342628359225, -0.58655147586502654, 0.33192150606895621]], [[-0.60799204848872812, 0.36256748644217363, 
0.19723086137404897, 0.5666987439892448], [-0.29162237261803009, 0.10564430571815793, 0.60301984621614479, -0.169811685532686], 
[0.28285368319580728, -0.78116378042386925, 0.20409929973165114, 0.36177245789889168]]], [[[0.59155591432730859, 
0.12380589393343674, -0.63085773872321771, 0.033903368211231667], [-0.53960833581276113, 0.61319099287846579, 
-0.76793223121295573, 0.62920350227669497], [0.71559796844102264, 0.1179444905899719, -0.59685969232550284, 
-0.30611905237254622]], [[-0.16750191169224649, -0.84251600860472764, 0.74298440520694375, -0.20464421707330283], 
[-0.40672862086456485, 0.81900124692360909, 0.73468093104517962, -0.64373282042762126], [-0.52035162868986329, 
0.39520558574758546, 0.60700143149763131, 0.074020323443978853]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_expandedData_rank0(self):
      arg0=Data(-0.79756958497,self.functionspace)
      arg0.setTaggedValue(1,-0.968889215103)
      arg1=Data(0.699397625237,self.functionspace)
      arg1.setTaggedValue(1,-0.786857629945)
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array(-0.55781827368974179)+(1.-msk_ref)*numpy.array(0.76237787147567204)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_expandedData_rank0(self):
      arg0=Data(numpy.array([-0.093748755478083812, 0.46462167631749418]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.89704257697357237, 0.69074421066044023]))
      arg1=Data(0.442102213582,self.functionspace)
      arg1.setTaggedValue(1,-0.586651313754)
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-0.041446532317410953, 0.20541027157810626])+(1.-msk_ref)*numpy.array([-0.52625120627456712, 
-0.40522599865172232])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_expandedData_rank0(self):
      arg0=Data(numpy.array([[0.16033381710577488, -0.066427493004955984, -0.88556038210106802, 0.88985782897508758, 
-0.94146679999966199], [-0.32986712529022499, -0.54764367458545804, -0.88604463281314616, -0.12341417429145474, 
-0.11491803882108265], [-0.59250099280449242, -0.37220980908110213, 0.99472806015279169, 0.12679066810731565, 
0.34391724276548685], [-0.75005909166020213, -0.32285038349217143, -0.29866421927549958, -0.66335472967492537, 
-0.21046147175977925]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.6242030092239832, -0.24917721298332252, 0.73567574225321652, 0.27673155632076507, 
0.9888423587949895], [0.65388216869148441, 0.57208164875615708, 0.89285696330263775, -0.38112419687001275, 
0.89423462833468248], [0.4399471506382775, -0.98422507264280523, -0.70724690965955128, 0.54072649538990114, 
-0.65654644530185013], [-0.83127913638398576, 0.27031140828616462, -0.91580636232154178, -0.75753756397810057, 
-0.79390743333931435]]))
      arg1=Data(0.408651843328,self.functionspace)
      arg1.setTaggedValue(1,0.102965318529)
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.065520709908123351, -0.027145717464147189, -0.36188588252403742, 0.36364204211071061, 
-0.3847321432521752], [-0.13480080880322878, -0.22379559710638305, -0.36208377247016116, -0.050433429817032242, 
-0.04696146839589848], [-0.24212662288335166, -0.15210422458583434, 0.4064974553917351, 0.051813240238870115, 
0.14054241520847247], [-0.3065130304120261, -0.13193340433329609, -0.12204968374311428, -0.27108113306214621, 
-0.086005468384202283]])+(1.-msk_ref)*numpy.array([[0.064271261671599952, -0.025656611105033079, 0.075749087135269608, 
0.028493752843634351, 0.10181646844843985], [0.067327185779848586, 0.058904569188857704, 0.091933301627423636, 
-0.039242574329886461, 0.092075153346274585], [0.045299298501461166, -0.10134104810903997, -0.07282190333185104, 
0.055676075834970883, -0.067601513869684646], [-0.085592921064411909, 0.027832700256247435, -0.09429629380745759, 
-0.078000096572799729, -0.081744931756440598]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_expandedData_rank0(self):
      arg0=Data(numpy.array([[[0.067000988686777729, 0.28863467671473031], [-0.82356799701368755, -0.019839912507680912]], 
[[-0.79390342608678366, 0.87577924907225269], [0.06112612567119835, 0.76155722533086156]], [[-0.20684668670365292, 
0.11865307108779355], [0.53785057292637051, -0.71542046897788936]], [[0.91505804504076194, 0.96923923157159919], 
[-0.64020545343184065, 0.69383881649901791]], [[0.22269151873184989, -0.78689341219820763], [-0.44950704033415545, 
0.65671641136170456]], [[0.6443127220306466, -0.63713946566059598], [-0.58895498910611122, 
-0.010847718204218149]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.340176319406303, 0.61054222206075059], [0.73079389495026437, 
-0.016133960019465654]], [[0.84487273169955057, -0.66224988027467724], [-0.3083270778618421, 0.24974814446451377]], 
[[-0.71681959068442813, 0.89783004030504499], [-0.92090021728858185, -0.59527078741654771]], [[-0.98949040935917099, 
-0.29028521932035578], [-0.47113614713834706, -0.52450700841806164]], [[-0.37414742957887492, 0.504282399883901], 
[0.3626512995835105, 0.80366239165404441]], [[-0.42904911628061959, -0.94336362490933379], [0.38917455530590517, 
-0.20944830219930899]]]))
      arg1=Data(-0.995664973412,self.functionspace)
      arg1.setTaggedValue(1,-0.308716489144)
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-0.066710537619416727, -0.28738343771703273], [0.81999780784983423, 0.019753905959461988]], 
[[0.79046183362661193, -0.87198272274254707], [-0.060861142291209119, -0.75825585451097866]], [[0.20595000081720988, 
-0.1181387068699128], [-0.53551897639251189, 0.71231910222346795]], [[-0.91109124408619913, -0.9650375537328707], 
[0.63743014576960744, -0.69083100678189935]], [[-0.22172614507728658, 0.78348220833462323], [0.44755841536293756, 
-0.6538695282578566]], [[-0.64151960924983464, 0.63437744913686878], [0.58640185356936325, 
0.01080069305738672]]])+(1.-msk_ref)*numpy.array([[[-0.1050180390171736, -0.1884844512690079], [-0.22560812553719789, 
0.004980819493205349]], [[-0.26082614350411326, 0.20444745797468927], [0.09518565298565608, -0.077101370329409974]], 
[[0.2212940273860135, -0.27717493789133735], [0.28429708193363451, 0.18376990758145162]], [[0.30547200521940598, 
0.089615833759088742], [0.14544749725356404, 0.16192396217044966]], [[0.1155054808819872, -0.15568029202946396], 
[-0.11195643599107082, -0.24810383200881825]], [[0.13245453684865491, 0.29123190626853229], [-0.12014460237836727, 
0.06466014451222335]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank4_expandedData_rank0(self):
      arg0=Data(numpy.array([[[[0.18797918717000339, 0.73794688990673518, 0.23017202170024698, -0.96422361533901357], 
[-0.86729869401557247, -0.62436327495445232, -0.959970903405029, 0.89160679132806386], [-0.36211459862276763, 
0.68635684659567864, 0.94699460102492949, 0.73560681058376542]], [[0.34088635057878625, -0.70604135225926279, 
0.97848620017366628, -0.74592714339329769], [0.98483569101369084, 0.63428559644232929, 0.89745995932826061, 
-0.29063457497984602], [0.11806742018773697, -0.30641881619920075, -0.039836525225535935, -0.58006481966646195]]], 
[[[-0.1626524556368536, 0.9873892132637303, -0.58470389470145578, -0.070726635698977747], [0.6503099939244219, 
-0.58575793341040971, 0.010516299580759947, -0.86647642418118354], [-0.37103121471868894, 0.75118233931921163, 
0.13313417081855383, -0.4256672345294612]], [[0.64644550534118128, -0.20115376591351275, 0.70162477283556801, 
-0.48740905496757381], [0.90396627677735486, 0.50438416252523544, -0.27214721334250269, -0.62853749685335525], 
[0.15763449363631898, -0.99768709141075917, -0.4695474997628819, 0.25782187068759144]]], [[[-0.95207301488989837, 
0.64540642752526711, -0.81881678208190989, -0.64708985075803316], [0.43422183832153527, -0.1905050452123298, 
0.047487910186180038, 0.23524636531003873], [-0.94976360717188935, -0.69616837448387381, 0.12506215256454323, 
0.4459194954400727]], [[-0.73637682982842589, 0.73161660981725496, 0.26753025307080125, -0.73220479964424845], 
[0.3180944212400123, 0.32721838391782976, 0.53242210199646878, 0.27861954269158851], [0.34943235452048094, 
-0.65279742518504569, -0.98629284146416385, -0.17758594014711226]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.9900494388492207, -0.98563137609318874, 0.097055277471882695, 
-0.18053283270965537], [-0.8391931186777204, 0.046914128472502981, -0.93827271569559301, -0.97358682001622143], 
[-0.24205174690633013, -0.28900899901104715, -0.7989548872371528, 0.8109105402916128]], [[0.57894477533494415, 
0.62187763791225104, 0.57772095165570003, 0.21484987832326041], [0.25784687286011221, 0.84210678357219715, 
0.0094427606313447487, -0.37243763881797354], [0.22257062277608841, 0.40482873874300696, 0.20967429503138391, 
-0.80918714441450779]]], [[[0.59551798691649172, -0.050992575816621644, -0.48400577980214177, 0.74244222228103807], 
[-0.67213417242083873, -0.80153019145329774, -0.7889117790562048, 0.94951095167423571], [0.64126665642041147, 
-0.5357412222263167, 0.84325391958134799, -0.88511244042478965]], [[0.17890703012889642, 0.11307754659658809, 
0.65109866510849734, 0.61001690076050119], [-0.66087529350604735, -0.70175411943418942, -0.30422904361799463, 
-0.96876948981581368], [0.20400898563733882, 0.47310303640370144, -0.62273193400222615, 0.8986782868720653]]], 
[[[-0.32699638389988106, -0.87158328760163561, -0.51763701962370834, -0.77025537499503249], [0.34967706139707277, 
0.25417997086538158, 0.43285308710972803, 0.69117075848268961], [-0.35808512562074712, 0.3805614881494912, 0.63375283664360027, 
0.016394335369582347]], [[0.52349466325015404, 0.85303092434604411, 0.90149701105317259, 0.046007338947993803], 
[-0.49445495495543845, -0.97160540528127504, -0.57796214556016201, 0.90726573450341141], [0.26763539310935203, 
0.51548510512858137, 0.81237166805014804, -0.52767150327849777]]]]))
      arg1=Data(0.106751882141,self.functionspace)
      arg1.setTaggedValue(1,-0.556893190093)
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.020067132033665201, 0.078777219417374422, 0.024571296532617221, -0.10293268574189059], 
[-0.092585767964279961, -0.066651954740880676, -0.10247870073873579, 0.095180703103646208], [-0.038656414953582255, 
0.073269885194202092, 0.10109345603643412, 0.078527411545289014]], [[0.036390259520338919, -0.075371243222797743, 
0.10445524351718029, -0.079629126497024361], [0.10513306361498766, 0.06771118123491604, 0.095805539804152573, 
-0.031025787894243043], [0.012603919324530522, -0.032710785352571006, -0.0042526240457689789, -0.061923011262964789]]], 
[[[-0.017363455774030805, 0.10540565692126737, -0.062418241254342026, -0.0075502014783411378], [0.069421815826299138, 
-0.062530761870371945, 0.0011226347734009303, -0.092497989111831463], [-0.039608280504147365, 0.080190128553133483, 
0.014212323312113874, -0.045440778451620506]], [[0.069009274396527182, -0.021473543110944869, 0.074899765056694712, 
-0.052031833990178362], [0.096500101437647853, 0.053843958671498514, -0.029052227243642009, -0.067097560785061225], 
[0.01682777888596353, -0.10650497479551765, -0.050125079354118585, 0.027522969952920664]]], [[[-0.10163558627480866, 
0.068898350883987769, -0.08741023261558481, -0.069078059482524823], [0.046353998507391858, -0.020336772133703614, 
0.0050694237913003109, 0.02511299226359081], [-0.10138905265428108, -0.074317284262942276, 0.013350620170824652, 
0.047602745421431565]], [[-0.078609612548941027, 0.078101450103345002, 0.028559358044869327, -0.078164240474432531], 
[0.033957178165808365, 0.034931178354246313, 0.056837061481397964, 0.029743160583491014], [0.037302561525896137, 
-0.069687353795066187, -0.10528861716813773, -0.018957633352418984]]]])+(1.-msk_ref)*numpy.array([[[[-0.55135179035012905, 
0.54889140128789304, -0.054049423086639008, 0.1005375051241333], [0.46734093296419188, -0.026126158665466165, 
0.52251768582056402, 0.54218387003094159], [0.13479696950215231, 0.16094714342473054, 0.4449325358935699, 
-0.45159055766270845]], [[-0.32241040282371947, -0.34631942162420304, -0.32172886375087434, -0.119648434130447], 
[-0.1435931675824684, -0.46896353310213784, -0.0052586090912703715, 0.20740798479189609], [-0.12394806413867211, 
-0.22544636775975604, -0.11676618704044395, 0.45063081023491502]]], [[[-0.33163991149144711, 0.028397418217557091, 
0.26953952273727028, -0.41346101762552501], [0.37430694344968918, 0.44636670527395778, 0.4393395973402367, 
-0.5287761829057237], [-0.35711703399397693, 0.29835063830972097, -0.46960236533374394, 0.49291309053880566]], 
[[-0.099632106738473669, -0.062972115652018337, -0.36259241267730319, -0.33971425787491527], [0.36803695045396467, 
0.39080209023232776, 0.16942308261924516, 0.53950113164790703], [-0.11361121481913262, -0.26346785918535182, 
0.34679517329903292, -0.50046781804313534]]], [[[0.1821020593787486, 0.48537879746386914, 0.28826853116827167, 
0.42894997296695397], [-0.19473277422362109, -0.14155109483286596, -0.24105293652196563, -0.38490828859014559], 
[0.19941516793164654, -0.21193210116195657, -0.35293263892868765, -0.0091298937234146234]], [[-0.2915306130138286, 
-0.47504711270670608, -0.50203754634434339, -0.025621173754419701], [0.27535859722222594, 0.54108043365830083, 
0.3218631829937611, -0.50525010914930879], [-0.14904432785035357, -0.28707014464027392, -0.45240424976129201, 
0.2938566667817194]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_array_rank1(self):
      arg0=Data(0.29744190771,self.functionspace)
      arg0.setTaggedValue(1,0.597708645073)
      arg1=numpy.array([0.32546484507071671, -0.1175166654588089])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.096806884410418917, -0.034954381161802356]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.19453315156607734, -0.070240726884869195]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_array_rank1(self):
      arg0=Data(numpy.array([-0.68279508605404571, 0.81377359932211557]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.2387725310549782, 0.51176842117816679]))
      arg1=numpy.array([0.88813591048589813, 0.20070621446611048])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.60641483542790708, -0.13704121697796964], [0.72274155656333361, 
0.16332941855240318]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.21206245926753545, 0.047923130826536479], [0.45451991270100173, 
0.10271510249796789]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_array_rank1(self):
      arg0=Data(numpy.array([[0.045301586501528712, 0.72167299727062861, 0.61374659900743733, 0.43840988246365509, 
0.69993504101864579], [0.044530426699972425, -0.3208458406704684, -0.69253696935764153, 0.8404145327069712, 
0.22345925947711409], [0.40802859105011335, -0.46091681095245041, -0.27040010002007664, 0.90626670460637104, 
0.31309842818842415], [-0.94403688666958341, 0.70602877505964345, -0.076025365433431302, -0.56640917800690871, 
0.97811733697108694]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.54903914639390394, 0.13466566736852692, 0.8781767229193469, -0.8552966536314035, 
-0.40992182955775891], [0.92579449622459853, -0.0089656380003484415, -0.0051856973462300893, -0.55456911954070565, 
-0.22465222873905111], [-0.39153632693981066, 0.10601882529388251, -0.56008523396364751, 0.91796930543729593, 
0.13826924328355994], [-0.93328629441099298, 0.69337568423542328, -0.8111907440192585, 0.10882815168066906, 
0.3538190129422687]]))
      arg1=numpy.array([-0.17719586513095509, 0.99171543244447591])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.0080272538119431774, 0.044926282447784377], [-0.12787747109301842, 0.71569424857174258], 
[-0.10875335958230424, 0.60866197384598708], [-0.077684418405107702, 0.43477784617537557], [-0.12402559512876947, 
0.69413638188684834]], [[-0.0078906074837521954, 0.044161511371700192], [0.056852556311272222, -0.31818777162852496], 
[0.12271468742049703, -0.68679960005030027], [-0.1489179801916391, 0.83345206173611608], [-0.039596056804569804, 
0.22160799614606855]], [[-0.072300979189289516, 0.40464825062297338], [0.081672553070120321, -0.45709831449463811], 
[0.047913779654554269, -0.26815995212443983], [-0.16058671276210565, 0.89875867686873734], [-0.055479746853990035, 
0.31050454310856873]], [[0.16727943284895022, -0.93621594930706264], [-0.125105379604042, 0.70017963197651789], 
[0.013471380399873868, -0.075395528157564631], [0.10036536431504733, -0.56171672290764163], [-0.17331834772417767, 
0.9700140578157207]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.097287466536028905, -0.54449059449497639], [-0.023862199432803554, 
0.13355002054980261], [-0.15560928415556072, 0.87090140853263276], [0.15155503048382737, -0.8482108907243805], 
[0.072636453224551006, -0.40652580446830361]], [[-0.16404695669199448, 0.91812468917809342], [0.0015886739819227084, 
-0.0088913615666561807], [0.0009188841275725386, -0.005142736086242744], [0.098267354911927382, -0.54997475420566311], 
[0.03980744602501337, -0.22279108217356339]], [[0.069378618182296223, -0.38829261778883606], [-0.018786097468117095, 
0.10514050517357804], [0.099244787579261917, -0.55544517000602411], [-0.16266036524062361, 0.91036432671250311], 
[-0.024500738184632904, 0.13712374239672609]], [[0.16537447235301916, -0.92555442105630037], [-0.12286330422886377, 
0.68763136653801715], [0.14373964567271563, -0.8044703795000151], [-0.019283898487658959, 0.10792655750612773], 
[-0.0626952660980859, 0.35088777542711963]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_array_rank1(self):
      arg0=Data(numpy.array([[[-0.95340302908473373, -0.45742866774311075], [0.81444071915788818, 0.43272566342830121]], 
[[0.40783634110489575, -0.84453120876513577], [0.021384721280967023, -0.31211632303820891]], [[-0.26168594229010989, 
0.67675491118993092], [0.31395716132494123, -0.63214478622607428]], [[0.089382408952075387, -0.7923646931705961], 
[0.96762972219816512, -0.42394975596135276]], [[-0.89777999104754036, 0.10836580891419079], [-0.8041458452244703, 
0.76929506916547785]], [[0.4578598563197811, -0.59031970687203161], [-0.22337144300146461, 
0.14327461823462229]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.89925629837741061, 0.96197032221872303], [0.89753819289821091, 
0.68076126290963668]], [[0.044341676689242382, 0.47156667538204489], [-0.067475647188812715, 0.57543223487768858]], 
[[-0.75159940645328382, 0.19241843966215], [-0.5208849404641811, 0.58556171380632849]], [[0.20433439348639326, 
0.54776458289962293], [-0.87066117147534272, 0.61269222109358612]], [[-0.63542853811559019, 0.9023230061029992], 
[0.52165964925784336, -0.72398339714616378]], [[-0.56870745973879688, -0.11186612264632423], [-0.21507628224610342, 
-0.96302448823600906]]]))
      arg1=numpy.array([0.76262204665997291, 0.141646945246372])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.72708616933241732, -0.13504662665849049], [-0.34884518679519583, -0.064793373453929298]], 
[[0.6211104481274089, 0.11536303995297323], [0.33000613108598564, 0.061294268354328579]]], [[[0.31102498515573046, 
0.057768771877965863], [-0.64405811889668874, -0.11962526588680755]], [[0.016308459910544148, 0.0030290804443940619], 
[-0.23802678907138414, -0.044210323719892135]]], [[[-0.19956746889142715, -0.037067014339312455], [0.51610821545885333, 
0.095860265850533491]], [[0.23943065293318197, 0.04447107283990033], [-0.4820875506571598, -0.089541377922344276]]], 
[[[0.06816499565043041, 0.012660745186823454], [-0.60427478400686141, -0.11223603830869378]], [[0.73793575915178566, 
0.13706179427896564], [-0.32331343057224288, -0.060051187869870508]]], [[[-0.68466681422304743, -0.12716779323519931], 
[0.082642154982103713, 0.015349685801847193]], [[-0.61325935029819933, -0.11390480250860809], [0.58668138013240212, 
0.10896829654038641]]], [[[0.34917402071003262, 0.06485444999863979], [-0.45019082303846403, -0.083616983197157033]], 
[[-0.17034798702716841, -0.031639882556431562], [0.10926438259251392, 0.020294412004274395]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.68579267874045213, 0.12737690765872026], [0.73361977595659622, 
0.1362601575599503]], [[0.68448241362352713, 0.12713354326598056], [0.51916354760697503, 0.096427753333212365]]], 
[[[0.033815940229084837, 0.0062808630501334454], [0.35962714311649413, 0.066795979047854198]], [[-0.051458416158838602, 
-0.0095577193028172688], [0.43883730867654508, 0.081508218266717433]]], [[[-0.57318627761782415, -0.10646175997309398], 
[0.14674254427026734, 0.027255484187216895]], [[-0.39723833937115194, -0.073781760641589603], [0.44656227262870357, 
0.082943028013896761]]], [[[0.15582991336361746, 0.028943342646117778], [0.41773734729875683, 0.077589179881884685]], 
[[-0.66398540453789545, -0.12332649528410998], [0.46725259562303528, 0.086785981494121242]]], [[[-0.48459181224386599, 
-0.090006511346441206], [0.68813141766264851, 0.12781129744001332]], [[0.39782914937694014, 0.073891495775667365], 
[-0.55212570007944739, -0.10255003661484506]]], [[[-0.4337088468967954, -0.080555674410824674], [-0.085311571404455333, 
-0.015845494549407823]], [[-0.16402191455454138, -0.030464898375107061], [-0.7344237062022182, -0.13640947695608138]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_constData_rank1(self):
      arg0=Data(-0.204230581137,self.functionspace)
      arg0.setTaggedValue(1,-0.62702589318)
      arg1=Data(numpy.array([-0.5529427686406152, -0.28709977651744789]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.1129278229751914, 0.05863455420257379]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.34670943338403859, 0.18001899380249342]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([0.79257859491204741, 0.98381691047691633]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.11017309942405018, -0.17668228386188356]))
      arg1=Data(numpy.array([-0.39933779599953922, -0.57809222307541352]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.3165065892485886, -0.45818352189469314], [-0.39287527669692773, 
-0.56873690487678563]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.043996282702438307, -0.063690211969157737], [0.070555913829569536, 
0.10213865425575752]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_constData_rank1(self):
      arg0=Data(numpy.array([[-0.66505582861548285, -0.13037499111836648, 0.36803816027010416, -0.40100458173491926, 
0.49681394415954672], [0.48651022381768882, 0.061867931882834348, 0.42766812075813632, -0.62489152706397455, 
-0.81680944741581718], [0.73533417020627878, -0.30934444276395134, 0.28539798770157332, -0.44676951729155556, 
-0.64927479339182015], [0.7549739857940474, -0.85034427958144376, -0.40038587277760485, -0.48990790638405546, 
0.3235954697374801]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.61055653479151495, 0.51709127692454415, -0.63112138679146024, 0.61974322543579374, 
-0.45254667298843088], [-0.39457459516502191, -0.18990373713956865, 0.096728233003309239, -0.71283351263395578, 
-0.72362857768496736], [-0.99811887395821519, -0.37991172478096136, -0.58105703468032921, -0.83132810727864936, 
0.37418297478778584], [-0.8880537608888377, -0.99055451789044935, 0.43771668323415192, -0.38177189742802886, 
-0.3019329884578541]]))
      arg1=Data(numpy.array([-0.96550572973321103, -0.48128192649569446]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.64211521312071707, 0.32007935042324998], [0.12587780093869932, 0.062747126892306479], 
[-0.35534295250125542, -0.1771301147987269], [0.38717222131433426, 0.1929962576309821], [-0.4796767096973979, 
-0.23910757215503101]], [[-0.46972840866976545, -0.23414857777882878], [-0.059733842719620567, -0.029775917444874914], 
[-0.41291602101621544, -0.20582893705926913], [0.60333634984200335, 0.30074899799618604], [0.78863420158018938, 
0.39311562443216813]], [[-0.70996935460277844, -0.35390304605499073], [0.29867383194972236, 0.1488818893641716], 
[-0.27555339238019755, -0.13735689333900772], [0.43135852881513781, 0.21502209398163133], [0.62687853319114917, 
0.31248422338870918]], [[-0.72893170908367266, -0.36335533433709222], [0.8210122741817435, 0.40925533306155065], 
[0.38657485427100996, 0.1926984841920657], [0.47300889065540708, 0.23578382098999054], [-0.31243328014724692, 
-0.15574065108053362]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.58949583266726224, -0.29384982529899584], [-0.49925459066570993, 
-0.24886668593236325], [0.60935131510432494, 0.30374731688762829], [-0.5983656351215999, -0.29827121347039426], 
[0.4369364057420318, 0.21780253460508905]], [[0.3809640324389908, 0.18990162130728047], [0.18335314630600311, 
0.091397236459263564], [-0.093391663191664157, -0.046553550326357086], [0.68824484079393555, 0.34307388623116319], 
[0.69866753795352998, 0.34826935593556041]], [[0.96368949176151741, 0.48037657453032306], [0.36680694706884492, 
0.18284464680088314], [0.56101389628564691, 0.27965224905482439], [0.80265205086580149, 0.40010319302108771], 
[-0.3612758061262249, -0.18008750296775544]], [[0.85742099444929976, 0.4274042248723266], [0.95638606263634729, 
0.47673598666932931], [-0.42261796566239063, -0.21066512856623829], [0.3686029544178816, 0.18373991427607839], 
[0.29151803035152962, 0.14531489035759829]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_constData_rank1(self):
      arg0=Data(numpy.array([[[-0.43561175662946217, 0.84319723561757609], [-0.034924643257009391, 0.61627859740863644]], 
[[0.046987230640658018, 0.78722949166629985], [0.68122999726014699, 0.71183298183667998]], [[-0.43709697960824512, 
-0.63368713627830942], [-0.81276997029498244, -0.58801246032451604]], [[-0.23062051581710685, -0.95270287195549619], 
[0.12293464970030499, -0.50917124613484877]], [[0.61795655996447052, -0.43756085163050007], [0.18691129089082925, 
-0.87609515736642618]], [[0.3356272771665072, 0.85338083173805712], [0.77426223182321485, 
0.46701394611624458]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.96951912102425442, 0.53488854007092268], [0.94090011997294032, 
0.54969105821591491]], [[0.42379849959719351, 0.30397363849289105], [-0.079865649798520844, 0.70203636232176692]], 
[[-0.46215549969260161, 0.88384341735393246], [-0.32375679980758831, -0.11349285843108681]], [[-0.76379409368044593, 
-0.69331059718409738], [0.62888168484873197, 0.22783903051617438]], [[0.8760170251600452, -0.3348774647561461], 
[-0.41352338079792128, 0.12233371813467708]], [[0.097950775803437695, -0.29407931052780412], [-0.48131299736583721, 
0.44473762926641447]]]))
      arg1=Data(numpy.array([0.45452178414226108, 0.89871827112537361]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.19799503281656758, -0.39149224479991723], [0.38325151191672319, 0.75779676181192224]], 
[[-0.015874011163707895, -0.031387415007609915], [0.28011204762286368, 0.55386083559465993]]], [[[0.0213567199026958, 
0.042228282686341358], [0.35781295308157185, 0.70749752772924368]], [[0.30963387376590962, 0.61223384537638226], 
[0.32354359691571349, 0.63973730676628049]]], [[[-0.19867009901473306, -0.39282704182764472], [-0.28802460776921734, 
-0.56950620755043124]], [[-0.36942165699572793, -0.73045122252612793], [-0.26726447256457952, -0.52845754174302639]]], 
[[[-0.10482204830899995, -0.20726287126119214], [-0.43302420911866824, -0.85621147798002173]], [[0.055876476314686507, 
0.11048361584006153], [-0.23142942322714982, -0.45760150203306338]]], [[[0.28087471815746529, 0.5553688512018522], 
[-0.19888093895390208, -0.39324393208950914]], [[0.084955253412032861, 0.16798059220321787], [-0.39820433400458299, 
-0.78736272516966666]]], [[[0.15254990882453001, 0.30163436627759993], [0.38788017819438841, 0.76694894571116001]], 
[[0.35191905100225657, 0.69584361438183284], [0.21226801200807327, 0.41971396624502971]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.44066756064798085, -0.87132454826990979], [0.24311849355028509, 
0.48071410397731473]], [[0.42765960122976832, 0.8456041291237375], [0.24984656050734513, 0.49401739749288415]]], 
[[[0.19262565015372971, 0.38087545486351709], [0.13816264050000354, 0.27318666285402038]], [[-0.036300677638104704, 
-0.071776718709231191], [0.31909081993523231, 0.63093290581296479]]], [[[-0.21005974227143948, -0.41534759167481805], 
[0.40172608695810247, 0.79432622798986818]], [[-0.1471545182767339, -0.29096615138815946], [-0.051584976501502636, 
-0.10199810551426312]]], [[[-0.34716105417695758, -0.68643570736826198], [-0.31512476959685243, -0.62309090125419231]], 
[[0.2858404254118368, 0.56518746054966451], [0.10355780264745465, 0.20476309960037747]]], [[[0.39816882121473973, 
0.78729250632822878], [-0.15220910275000069, -0.30096049616449189]], [[-0.18795538482481081, -0.37164101786062731], 
[0.055603339827329908, 0.10994354766233576]]], [[[0.044520761376297117, 0.088030151885454599], [-0.13366545290042356, 
-0.26429444953129]], [[-0.21876724229357974, -0.4325647848627967], [0.20214294072937017, 0.39969383327870939]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_taggedData_rank1(self):
      arg0=Data(0.0888827520184,self.functionspace)
      arg0.setTaggedValue(1,0.97962819765)
      arg1=Data(numpy.array([0.46549184212655681, 0.17049390147272425]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.70894984962047225, -0.30620658136016643]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.04137419597034523, 0.015153967165257833]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.69450726340786639, -0.29996860140639536]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([0.56674119640885312, -0.30730019781223961]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.59181683541356822, 0.25498399352930146]))
      arg1=Data(numpy.array([-0.98771834119701807, -0.93219702766919399]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.83369000013051275, 0.93554929454152358]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.55978067440496582, -0.5283144587500157], [0.30352604163262081, 
0.28646433100272511]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.49339177759317732, 0.55367382286896072], [-0.21257760559872199, 
0.23855009526571838]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_taggedData_rank1(self):
      arg0=Data(numpy.array([[0.65635138162217666, 0.28715409870231423, -0.60010402997196599, -0.42637869965792796, 
0.097128502801679062], [0.65581649800968522, -0.28636501246117008, 0.34410194476175571, 0.57276894389952115, 
-0.32364004515637612], [0.34451553260270984, -0.8472566173637659, -0.42048485262300939, -0.20490428640632952, 
0.91342545154315236], [-0.1797593767246477, 0.56452304592010671, -0.7301794602743108, 0.76667910017025109, 
0.011520879381168525]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.34133140149021579, 0.37774778528583464, -0.28242691317945368, 0.37749216688162668, 
-0.29484040161208491], [0.49604843963532397, 0.99050370108156716, -0.349659689060414, -0.94312778183937018, 
0.79629878352393968], [-0.73556373567242916, 0.60202463062710687, -0.50352899183298461, -0.18106106760463714, 
-0.49672360351084244], [-0.68149304461286819, -0.62892724168057557, 0.92498236010373391, 0.17153617350064221, 
-0.066155192554459896]]))
      arg1=Data(numpy.array([0.89099279465287751, -0.49777375341211205]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.50664729772083494, 0.18598372245101946]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.58480435178582046, -0.32671449078729642], [0.25585223289880316, -0.14293777351872305], 
[-0.53468836674717612, 0.29871603543688008], [-0.37990034918867716, 0.21224012570370243], [0.086540796151717864, 
-0.04834801940289063]], [[0.58432777434111272, -0.32644823976386794], [-0.25514916274358401, 0.14254498709870286], 
[0.30659235340876678, -0.17128491660046638], [0.51033300201541154, -0.28510934704275609], [-0.28836094829546305, 
0.16109952003195477]], [[0.30696085719501298, -0.17149078977242374], [-0.75489954129308545, 0.42174210652841138], 
[-0.37464897394777846, 0.20930632334309415], [-0.18256824278152917, 0.10199597573470905], [0.81385549577749983, 
-0.45467921547678825]], [[-0.16016430943295329, 0.089479499663249737], [0.50298596633031056, -0.28100475545528963], 
[-0.65058463790793797, 0.36346417060517383], [0.68310555406264539, -0.38163273335436654], [0.010265020516686058, 
-0.0057347913721724677]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.17293463219228319, -0.063482084638573785], [-0.19138489463509831, 
0.070254939255087967], [0.14309083236600706, -0.052526808633465684], [-0.19125538626135863, 0.070207398392746381], 
[0.14938009273568853, -0.054835515420769108]], [[-0.25132160147987359, 0.092256935319397368], [-0.50183602353546164, 
0.18421756542866174], [0.17715413658436616, -0.065031010562521807], [0.47783314207436206, -0.17540641561345907], 
[-0.40344262685079219, 0.14809861194300084]], [[0.37267137897987873, -0.13680288166033611], [-0.30501415226860751, 
0.11196678181122935], [0.255111603036278, -0.093648196263107455], [0.091734100624338816, -0.033674411344066103], 
[0.25166367143292379, -0.092382504810230762]], [[0.34527660946865407, -0.1267466132615799], [0.31864428746048207, 
-0.11697022955860541], [-0.46863981318599701, 0.17203166253362179], [-0.086908338765472673, 0.031902936082653363], 
[0.033517349537918607, -0.012303788970742418]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_taggedData_rank1(self):
      arg0=Data(numpy.array([[[0.93958986544748924, 0.16314460389085861], [-0.38869611333364684, -0.67428270275954638]], 
[[-0.86466889391170509, -0.88634995924272908], [0.76762871328544802, -0.62191463295997096]], [[0.82060355027127163, 
0.6882932647777027], [0.84429472769577951, -0.72409541230293062]], [[0.97746213964027007, -0.65730983654465902], 
[-0.92003318972484061, -0.97911321306788901]], [[-0.062827331105870465, 0.77575748937227629], [-0.066694463841462381, 
0.043335410121515316]], [[-0.44841117208478831, 0.67440175460217033], [0.54575470460683673, 
-0.55613026389412856]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.060410964690499114, 0.27711663174445866], [0.93054965639410603, 
-0.85962140215229033]], [[-0.64911944337497274, -0.22879126169726405], [0.41278511133611939, 0.92899507984650809]], 
[[-0.16708377137615349, -0.47572050498441754], [-0.34080032580962571, 0.59048036190880882]], [[-0.34864891348267579, 
0.10907472403346863], [0.96263703434034475, 0.47038129369165227]], [[-0.25549103518501215, -0.73870561208856445], 
[-0.92177803665151137, 0.020363525206199995]], [[0.7857253455759301, 0.28432899365895525], [0.30845267558619982, 
-0.77199562013426992]]]))
      arg1=Data(numpy.array([-0.13804211978527547, 0.89786740947383326]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.86191282296974525, -0.44198064895510525]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.12970297675513318, 0.84362711845720473], [-0.022520826952623224, 0.14648222286511989]], 
[[0.053656435436874289, -0.34899757235142898], [0.093079413623472593, -0.60541646357972867]]], [[[0.11936072702796124, 
-0.77635801982910746], [0.12235362724545884, -0.79582474179250684]], [[-0.1059650947899667, 0.68922880423533706], 
[0.085850414259275948, -0.55839688040963897]]], [[[-0.11327785358276921, 0.73679318388709714], [-0.095013461303841959, 
0.6179960906042431]], [[-0.11654823393465734, 0.75806471998862501], [0.099955665641089583, -0.65014167205631956]]], 
[[[-0.13493094576579381, 0.8776313991775595], [0.09073644319233766, -0.59017708016002179]], [[0.12700333178242551, 
-0.8260678166881904], [0.13515886344166347, -0.87911384419886685]]], [[[0.0086728179663057344, -0.056410613024182696], 
[-0.10708720827225233, 0.69652736736261045]], [[0.0092066451666178734, -0.059882785475580069], [-0.0059821118749382564, 
0.038909452424291092]]], [[[0.06189962872998412, -0.40261377745889415], [-0.093095847792192751, 0.60552335634925847]], 
[[-0.075337136306714583, 0.49001536283349756], [0.076769400504690152, -0.49933123937262047]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.052068985114713695, -0.026700477377910748], [0.23885037835873368, 
-0.12248018873466876]], [[0.80205268125617035, -0.41128494101801721], [-0.74091870941429117, 0.3799360251789668]]], 
[[[-0.55948437188387246, 0.28689823283224714], [-0.19719812224029862, 0.10112131032021408]], [[0.35578478059159524, 
-0.18244303138734341], [0.80071277179550771, -0.41059784826665946]]], [[[-0.14401164505925196, 0.073847793702698755], 
[-0.41002960339571209, 0.21025925751426325]], [[-0.29374017088758342, 0.15062714916544967], [0.50894259564101829, 
-0.26098089355170073]]], [[[-0.30050496924518755, 0.1540960730385654], [0.094012903306332871, -0.048208917312911478]], 
[[0.82970920376351009, -0.42546694114596351], [0.40542766871793284, -0.20789942944217843]]], [[[-0.22021099937977634, 
0.11292209353328329], [-0.63669983945884812, 0.32649358581768195]], [[-0.79449230972181351, 0.40740805483179776], 
[0.017551583496091403, -0.0090002840856499167]]], [[[0.67722675068422855, -0.34727539813812391], [0.24506680557673691, 
-0.12566791313413705]], [[0.2658593163670725, -0.13633011372752715], [-0.66539292427020769, 0.34120712517744356]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_expandedData_rank1(self):
      arg0=Data(-0.775151292954,self.functionspace)
      arg0.setTaggedValue(1,-0.753081086642)
      arg1=Data(numpy.array([0.57180590124290465, 0.0081309920868766206]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.74898070021909025, 0.070624004542414909]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-0.44323608366688222, -0.006302749029137137])+(1.-msk_ref)*numpy.array([0.56404319959454352, 
-0.053185602083779773])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([0.0074249411735971549, -0.011510527886748667]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.57053702007138352, 0.58147848255543511]))
      arg1=Data(numpy.array([0.6077144893716584, 0.35109125685706388]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.53185942631738259, 0.13079557544879683]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.004512244333927197, 0.0026068319287279882], [-0.0069951145770937003, 
-0.0040412457028468733]])+(1.-msk_ref)*numpy.array([[0.30344549218799505, 0.074623717855078348], [0.30926481214783585, 
0.076054812736931307]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(2,2),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_expandedData_rank1(self):
      arg0=Data(numpy.array([[0.57462282234158835, -0.57043863758709579, 0.65033759305793537, -0.5361596538855089, 
-0.31322792277357947], [0.47515831181550472, -0.70997487710215257, 0.31442873955465211, 0.4472517276519532, 
0.71469591992004133], [0.19745294630689436, -0.44869091822955509, 0.673783266238005, -0.62521536346401785, 
-0.8467586435151071], [0.33183938659892664, 0.40019065137548693, 0.15254126617272346, -0.7659305088680346, 
0.59118453266141047]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.78418034105682666, 0.74824665143523816, -0.52405036376431657, 0.99895486927730981, 
0.26860005575222279], [0.31495219129055085, 0.88038280549977221, 0.66845425591812191, -0.8489377029339471, 
-0.83232450666581625], [0.5000386154942158, -0.27534072672554499, 0.35015229773733925, -0.92222660461360051, 
0.38489004824621365], [-0.042678105561828605, -0.43892210524342778, 0.27786178186096167, -0.19285646807245738, 
-0.55553966917213971]]))
      arg1=Data(numpy.array([-0.91068522105368399, 0.13352921509829274]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.46474540553473864, -0.3365717614832211]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-0.52330051198664118, 0.076728934444838012], [0.51949003676856664, -0.076170223538744378], 
[-0.59225283469348666, 0.086839068349939028], [0.48827267291879139, -0.071592977750704306], [0.2852520400912435, 
-0.041825078674824719]], [[-0.43271965223119824, 0.063447516424154179], [0.64656362789633592, -0.094802388078957286], 
[-0.28634560618695959, 0.04198542279707821], [-0.40730553846336109, 0.059721172144720704], [-0.65086301181854889, 
0.095432785220875407]], [[-0.17981748005519529, 0.026365736939204944], [0.40861618805266281, -0.05991334613292474], 
[-0.61360446275623082, 0.089969750687124816], [0.56937439148238844, -0.083484516750744114], [0.77113058244867294, 
-0.11306701704626732]], [[-0.30220122513916242, 0.044310252831253595], [-0.36444771181150309, 0.053437143567843279], 
[-0.1389170767043155, 0.020368715542143517], [0.69752159478024678, -0.10227409966898461], [-0.53838301681027545, 
0.078940406624529155]]])+(1.-msk_ref)*numpy.array([[[-0.36444421061682458, -0.26393295871000921], [-0.34774419346127999, 
-0.25183869349747984], [0.24354999882827461, 0.17638055403807881], [-0.46425968583318517, -0.33621999999490504], 
[-0.12483064183722019, -0.09040319389901702]], [[-0.14637258386538163, -0.1060040138056611], [-0.40915386396780257, 
-0.29631199162659838], [-0.31066104424808955, -0.22498282638531816], [0.39453989702376674, 0.28572845806599806], 
[0.38681899038690604, 0.28013692533416679]], [[-0.23239064914088856, -0.16829887762651929], [0.12796333770229307, 
0.092671913402086886], [-0.16273167161086027, -0.11785137563685356], [0.42860057735607282, 0.31039543280148962], 
[-0.17887588155847167, -0.12954312151559008]], [[0.019834453476786421, 0.01436424516571151], [0.20398703179951808, 
0.14772878611570425], [-0.12913498649357771, -0.093520429369810407], [0.089629157464331574, 0.064910041172579563], 
[0.25818450884004063, 0.18697896502707295]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank3_expandedData_rank1(self):
      arg0=Data(numpy.array([[[-0.15760858275510836, -0.82167564238141932], [0.55214037985992559, -0.41275909583570902]], 
[[0.56962687559986169, 0.51361613533580619], [0.70978446654860639, -0.66445689489974225]], [[-0.35378570094396844, 
0.11505203145913634], [0.30176345913496161, 0.31264962187589096]], [[0.73565031763187538, -0.79195365228726589], 
[-0.95993794651948217, 0.03064856035108865]], [[-0.36272019889390972, 0.62707574000988453], [-0.091772604666159774, 
-0.46967544621611701]], [[0.59519755857447043, 0.54533002100760974], [-0.20467988901696033, 
-0.57321925583402833]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.79906628595547957, -0.79279337068581435], [-0.0098859489505700449, 
-0.54152653260903594]], [[-0.78798186210077481, 0.92245028784403638], [-0.87345949422250602, -0.50576783493073108]], 
[[-0.81809053102093676, -0.34651233104065948], [-0.37015246519109279, 0.87666984553571115]], [[0.61967171366672824, 
-0.31022324330625173], [-0.38982311817077919, 0.67601000590716498]], [[-0.18950446276071076, -0.36048915721504726], 
[0.33273923587751253, 0.80148695051094876]], [[0.17047294586379036, 0.23988695993883558], [-0.17089551020488924, 
-0.77119868178240814]]]))
      arg1=Data(numpy.array([0.99908292144770883, -0.0070174162857228328]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.1327926244871287, 0.4919444876319452]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-0.15746404330420663, 0.0011060050353953923], [-0.82092210127285126, 0.0057660400344291419]], 
[[0.55163402375970216, -0.0038745988936342329], [-0.41238056332165501, 0.0028965024011977361]]], [[[0.56910448300944039, 
-0.0039973089136198835], [0.51314510899397903, -0.0036042582327155087]], [[0.7091335384375852, -0.0049808530749112841], 
[-0.66384753573250765, 0.0046627706354302762]]], [[[-0.35346125166552544, 0.0024826615394600718], [0.11494651970868763, 
-0.000807367999266839]], [[0.30148671833872376, -0.0021175998125697363], [0.31236289761328662, -0.002193992548276963]]], 
[[[0.73497566850358897, -0.0051623645195470972], [-0.79122736857834453, 0.0055574684570983369]], [[-0.95905760801719875, 
0.0067362841791891483], [0.030620453213732066, -0.00021507370654168861]]], [[[-0.36238755597902134, 0.002545358630878747], 
[0.62650066229805934, -0.0044004515103270607]], [[-0.091688441978732535, 0.00064400657056751286], [-0.46924471693785441, 
0.0032959081252811181]]], [[[0.59465171565912567, -0.0041767490407629585], [0.54482991054142316, -0.003826807770512375]], 
[[-0.20449218148065754, 0.0014363239865475593], [-0.57269356874874267, 
0.0040225181411796332]]]])+(1.-msk_ref)*numpy.array([[[[-0.1061101092512106, 0.39309625462832981], [0.10527711236936636, 
-0.39001032854003576]], [[0.0013127811066919719, -0.004863338091243747], [0.071910729494568559, -0.26640099262345607]]], 
[[[0.10463817951661662, -0.38764333341443175], [-0.1224945946817169, 0.45379433421937487]], [[0.1159889786210066, 
-0.4296935833525487], [0.067162238181624659, -0.24880969841571673]]], [[[0.10863638868233896, -0.40245512711964071], 
[0.046014281856041922, -0.17046483115194822]], [[0.049153517313105764, -0.18209446483413358], [-0.1164152895974128, 
0.43127289798444196]]], [[[-0.082287833178241385, 0.3048440836797881], [0.041195358655546244, -0.15261261447981428]], 
[[0.051765634947653877, -0.1917713341356112], [-0.089769142863971807, 0.33255939599006851]]], [[[0.025164794962018129, 
-0.093225675836784899], [0.04787030128573927, -0.17734065374302818]], [[-0.044185316402016658, 0.16368923290880785], 
[-0.10643155565053432, 0.39428708721279893]]], [[[-0.022637549885304933, 0.083863226008070682], [-0.03185521899051668, 
0.11801106759669543]], [[0.022693663313174128, -0.084071104206344102], [0.10240949695489998, -0.37938694037187831]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(6,2,2,2),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_array_rank2(self):
      arg0=Data(-0.422379565542,self.functionspace)
      arg0.setTaggedValue(1,0.281852135737)
      arg1=numpy.array([[-0.82425421616769867, -0.22861791969420775, 0.19984565262102927, -0.083087555810546165, 
-0.71930050386522715], [0.83403135113822757, -0.54625138929887451, -0.58388985754199596, -0.09336202892329859, 
-0.91872728116376456], [0.34496993319844926, 0.8748926201422933, 0.025490588389900815, 0.52741650606948864, 
0.75289315405348844], [-0.74813305077539849, -0.67788035449855233, -0.42746930085402113, -0.55848770185818486, 
0.40866402699794535]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.34814813772070968, 0.096563537595454191, -0.084410719929439396, 0.035094485725168413, 
0.30381783431641812], [-0.35227779974180284, 0.23072542448853089, 0.24662314435271024, 0.039434213214701197, 
0.38805162986912733], [-0.14570825050926117, -0.36953676479121689, -0.01076670364952498, -0.22276995469307692, 
-0.31800668330832549], [0.31599611295379287, 0.28632280962225559, 0.18055429757707483, 0.2358937928711631, 
-0.17261133417585553]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.23231781121679887, -0.064436448933480897, 0.056326924008920441, 
-0.023418405058339999, -0.20273638325085805], [0.23507351758962788, -0.15396212072299645, -0.16457060338317606, 
-0.026314287248737987, -0.25894524635552679], [0.097230512436910152, 0.2465903535273333, 0.0071845767788772325, 
0.14865346865844375, 0.21220454345147344], [-0.21086289817621678, -0.19106202568932998, -0.12048313540755565, 
-0.15741095155137919, 0.11518282880810822]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_array_rank2(self):
      arg0=Data(numpy.array([-0.31671223523984127, -0.0096385334452044891]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.37747777231496871, 0.7519220303445604]))
      arg1=numpy.array([[-0.025812368900665961, -0.95663766882809309, -0.92085395525416747, 0.71087712206744658, 
-0.20745592260308499], [-0.98071557720437075, -0.080598190505341227, 0.70024984664277756, -0.78405590339381237, 
0.12932139096776929], [0.29810045147408459, -0.47774213322259373, -0.34427103638668388, 0.079324189511316368, 
-0.35695106759128881], [0.88499228488874193, -0.40384672412205491, -0.71846847906690914, -0.60220038705301326, 
-0.82860275285159513]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.0081750930513652814, 0.30297885440917638, 0.29164571449799614, -0.22514348231084649, 
0.065703828961366562], [0.31060462259092736, 0.025526433071233173, -0.22177769415659013, 0.24832009771684735, 
-0.040957666797727631], [-0.094412060312363169, 0.15130677888117769, 0.10903484946236337, -0.025122941368717778, 
0.11305077048808473], [-0.28028788471712784, 0.12790319869098354, 0.2275477579546499, 0.19072423064585739, 
0.26242862998151445]], [[0.00024879338094902507, 0.0092205841659420314, 0.0088756816458661313, -0.0068518129164777978, 
0.0019995708484155886], [0.0094526598911173522, 0.00077684835480869427, -0.0067493815668657258, 0.00755714904777128, 
-0.0012464685520232101], [-0.0028732511715635223, 0.0046047335292493082, 0.0033182678984282643, -0.00076456885361856199, 
0.0034404848032800854], [-0.0085300277366480784, 0.0038924901571866966, 0.0069249824648116047, 0.0058043285713255564, 
0.0079865153461486094]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.0097435955107955646, -0.36110945614181333, -0.34760189965677102, 
0.26834031242769579, -0.078309999517759091], [-0.37019833135769459, -0.03042402540457367, 0.26432875217461416, 
-0.29596367578349658, 0.048815950575186666], [0.11252629434852388, -0.18033703618986568, -0.12995466388781096, 
0.02994311834742211, -0.13474109381980953], [0.33406491621573647, -0.15244316177829104, -0.27120588095670056, 
-0.22731726059198337, -0.3127791212804707]], [[-0.019408888831791538, -0.71931693824930698, -0.69241037568553254, 
0.53452416895045229, -0.15599067853071563], [-0.7374216480020479, -0.060603555046873848, 0.52653328643610431, 
-0.5895489067835139, 0.097239602863467778], [0.22414829671902378, -0.35922483479387413, -0.2588649766687014, 
0.059645605632785682, -0.26839937147690029], [0.66544519569281446, -0.30366124874985506, -0.54023227751855862, 
-0.45280773770718186, -0.62304466427326344]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[0.21768621064740601, 0.69927140646977004, -0.42510025866081413, 0.66052051062874373, 
0.77540660944873907], [0.19297059905140168, 0.39725931731952402, 0.46619567153785901, -0.78118807659372513, 
-0.4004480439259972], [-0.26189720062163002, -0.020271867162172219, -0.69220550122886682, 0.19772136317632061, 
0.56828649517756946], [-0.63128024947423111, -0.48827391732122094, -0.12466502840906313, 0.67043875110512974, 
0.84908181183026721]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.86325521186766263, 0.94598321880962022, 0.92075644177898242, 0.57305133857028867, 
0.92892510026027431], [0.1929172097268228, -0.44980093807671473, -0.27390643756657873, 0.75109664776912344, 
0.31754599564115549], [0.16020203743841854, 0.79838050797951632, -0.0414250502101281, -0.67320903480522465, 
-0.39929237303906606], [-0.63444817444762891, -0.87070478583756117, -0.80518436280115457, -0.67278628072638846, 
-0.87283503748641666]]))
      arg1=numpy.array([[-0.92758889561516034, -0.18949770094841067, 0.66738920265250212, -0.56814063609724097, 
-0.69790029862388714], [0.41615085294798404, -0.039783078402852334, -0.1507534274541058, 0.74412666309525255, 
0.33249498863158378], [0.089337343806613623, -0.86658288092479374, 0.52106795374336068, 0.12847865317157514, 
-0.76528299618936613], [0.64946096816167587, 0.9885499876898769, 0.48096381526556953, 0.7562980322888333, 0.39363809584760445]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.20192331172507649, -0.041251036445854879, 0.14528142655241691, -0.12367638218681525, 
-0.15192327141712705], [0.090590302235932535, -0.0086602275854055812, -0.032816942364592913, 0.16198611353090447, 
0.072379574134461819], [0.019447507842566224, -0.1886431435604306, 0.11342930834019002, 0.027968031158002525, 
-0.16659155551335636], [0.14137869712251083, 0.21519370085574915, 0.10469919040368084, 0.16463565276904563, 
0.085689585451525413]], [[-0.64863639166255382, -0.13251032386498302, 0.46668618640155352, -0.39728450167634749, 
-0.48802172339439809], [0.29100239224453123, -0.027819169188459683, -0.10541756124597101, 0.52034649829427393, 
0.2325042383445578], [0.062471050053924108, -0.60597662996690582, 0.3643679208804449, 0.089841448504629134, 
-0.53514051709273769], [0.45414948465363364, 0.69126474025757412, 0.33632424356182145, 0.52885758874893196, 
0.27525986492343651]], [[0.39431827945690356, 0.080555521688798981, -0.28370732267501314, 0.2415167313606566, 
0.29667759746447386], [-0.17690583523010645, 0.016911796919375974, 0.06408532100474465, -0.31632843695820034, 
-0.14134370567071072], [-0.037977327960261535, 0.36838460683216334, -0.22150612191616376, -0.054616308695629619, 
0.32532199962882236], [-0.27608602555563116, -0.42023285546611128, -0.20445784227588562, -0.32150248915064777, 
-0.16733565636356701]], [[-0.6126914909852782, -0.12516711819341719, 0.44082425692414084, -0.3752685430638889, 
-0.46097746161500269], [0.27487617388778968, -0.026277539261035369, -0.099575730881019245, 0.49151092348013936, 
0.21961975967243205], [0.059009147949360068, -0.5723957670105726, 0.34417607087883922, 0.084862785597782084, 
-0.50548511541849506], [0.42898229032358842, 0.65295754265095585, 0.31768646485316276, 0.49955036247493428, 
0.26000603607218609]], [[-0.71925856051125181, -0.14693776979073822, 0.51749799881147407, -0.44054000432621154, 
-0.5411565042892108], [0.32268612190359708, -0.030848061937789086, -0.11689520404496463, 0.57700073283109388, 
0.2578188117935134], [0.069272766858242574, -0.67195409350421464, 0.40403953530453168, 0.099623196842311565, 
-0.59340549334396864], [0.50359632729154058, 0.76652819422520024, 0.37294252126260496, 0.58643849294983719, 
0.30522958125104876]]], [[[-0.1789973848602856, -0.036567484870878179, 0.12878649423629066, -0.109634438893129, 
-0.13467423870360362], [0.080304879389124256, -0.0076769644715072955, -0.029090979204870821, 0.14359456794761144, 
0.064161757137825709], [0.017239480752023261, -0.16722501765974693, 0.10055079518034438, 0.024792602667836124, 
-0.14767711821851354], [0.12532687208666191, 0.1907610833167713, 0.092811875553844639, 0.14594328435217249, 
0.075960579165165298]], [[-0.36849333142524981, -0.075279727312384942, 0.26512657903215447, -0.22569916123747008, 
-0.27724739618841737], [0.16531980374405378, -0.015804198567186217, -0.059888203673996464, 0.29561125018047546, 
0.13208673219594588], [0.035490092211754934, -0.34425812367696995, 0.20699909958116877, 0.051039342049071838, 
-0.3040158006224275], [0.25800442083758446, 0.39271069324590435, 0.19106735690779381, 0.30044643999716125, 
0.1563764012273767]], [[-0.43243792810237069, -0.088343007948524691, 0.31113395750769951, -0.26486470537329965, 
-0.32535809838343543], [0.19400772635113822, -0.018546698951861038, -0.070280595348600761, 0.34690862941091744, 
0.15500772450807399], [0.04164868298933283, -0.40399718811594676, 0.24291962461224409, 0.059896191993602153, 
-0.35677162032500637], [0.30277589218976053, 0.46085772535982444, 0.22422324884314296, 0.352582869045654, 
0.18351237643655807]], [[0.72462138523530473, 0.14803334452282185, -0.52135648755952801, 0.44382469074753922, 
0.54519139193618082], [-0.32509208438727383, 0.031078066498501582, 0.1177667800327846, -0.5813028766854873, 
-0.25974112064615945], [-0.069789267776280833, 0.67696421395868878, -0.40705207255940407, -0.10036599195445509, 
0.59782995184305399], [-0.50735116454091811, -0.77224346350020556, -0.37572319775849, -0.59081100517533269, 
-0.30750538696920654]], [[0.37145115881656698, 0.075883983673264641, -0.2672547007395254, 0.22751080640001195, 
0.27947280943930491], [-0.16664679504115551, 0.015931055927776803, 0.060368915139136392, -0.29798406666967342, 
-0.1331469678127144], [-0.035774964576902726, 0.34702141956608901, -0.20866064282905078, -0.051449025348803884, 
0.30645607887385801], [-0.26007537430662747, -0.3958629088934798, -0.19260101902228199, -0.30285806765514395, 
-0.1576316054969274]]], [[[0.24293293508931987, 0.049628917402623561, -0.17478736389979205, 0.14879444215325963, 
0.18277813452259567], [-0.10898874342338064, 0.010419076865817854, 0.039481900634346298, -0.19488468997256145, 
-0.087079506743332494], [-0.023397200253924225, 0.22695563062083082, -0.13646623841902716, -0.033648199605272835, 
0.20042547438532854], [-0.17009200947455649, -0.25889847445052555, -0.12596307681835145, -0.1980723374920926, 
-0.10309271536051649]], [[0.018803958873016564, 0.0038414722211632178, -0.013529225261639558, 0.011517271504395295, 
0.014147742146143763], [-0.0084361548103861969, 0.000806477280684905, 0.0030560534555917995, -0.015084836866097441, 
-0.0067402942416274281], [-0.0018110347662689803, 0.017567253047119924, -0.010563020340750306, -0.0026045021902689675, 
0.015513715240219978], [-0.013165786473589253, -0.020039754033616165, -0.0097500345728751651, -0.015331573245571464, 
-0.0079797791889930532]], [[0.64208213642362311, 0.13117135106671252, -0.46197047753680898, 0.39327007377817791, 
0.48309042601672364], [-0.28806190975167972, 0.027538065726273705, 0.10435235181283892, -0.51508856980561335, 
-0.23015486026181181], [-0.06183980084811258, 0.59985343744690223, -0.36068610409522295, -0.088933630515839912, 
0.52973309995918916], [-0.44956045499493796, -0.68427973971866141, -0.33292579881885165, -0.52351365851889753, 
-0.27247845543896776]], [[-0.18340414090824728, -0.037467743750298502, 0.1319571029176104, -0.11233354104500838, 
-0.13798979840507619], [0.082281913931863945, -0.0078659644931624029, -0.029807173179728357, 0.14712973820304001, 
0.065741362401531972], [0.017663901399995269, -0.17134194852171333, 0.10302626612163325, 0.025402974444141545, 
-0.15131279722222044], [0.12841230795473951, 0.19545745113397742, 0.095096821192792447, 0.14953627791171706, 
0.077830660909119498]], [[-0.52713624245477175, -0.10768898431617949, 0.37926827089474308, -0.32286665085565597, 
-0.39660731468834792], [0.23649290968696596, -0.022608186192931409, -0.085671136923899771, 0.42287713333858107, 
0.18895241175354854], [0.050769206000334001, -0.49246734818163207, 0.29611588118216231, 0.073012683516008953, 
-0.43489999172344412], [0.36907989735122981, 0.56177960781210956, 0.2733252408845025, 0.42979395807911336, 
0.2236992138576073]]], [[[0.58556854943346492, 0.11962615592950593, -0.42130962234687974, 0.35865596249191461, 
0.44057067462342786], [-0.26270781426791745, 0.025114271659005517, 0.095167661292323319, -0.46975246551919814, 
-0.20989751937227785], [-0.056396900685604205, 0.54705665726030173, -0.32893990783213584, -0.081106036226265166, 
0.48310804075281011], [-0.40999188200487841, -0.62405208284661362, -0.30362295728892674, -0.47743601050016476, 
-0.24849595534923705]], [[0.45291746372567943, 0.092526784765445722, -0.3258687403570234, 0.2774082539765701, 
0.34076651270873526], [-0.20319560716547952, 0.019425039534857972, 0.07360896657261673, -0.36333764077268738, 
-0.16234863058881824], [-0.043621094823527946, 0.42312981795265819, -0.25442389096482348, -0.062732775276239494, 
0.37366772640870277], [-0.31711485107153425, -0.48268317495718094, -0.23484208616947969, -0.36928060288799985, 
-0.19220321506637605]], [[0.11563789602379546, 0.023623736272185766, -0.083200093908576161, 0.070827268539405741, 
0.087003760554640527], [-0.051879457905216281, 0.0049595585992915723, 0.018793680316329737, -0.092766571594711009, 
-0.041450497203627513], [-0.011137242503641728, 0.10803257946929719, -0.064958951256468445, -0.016016794947592585, 
0.095404026460920277], [-0.080965070046452967, -0.1232376122991375, -0.059959367693813617, -0.094283915681005942, 
-0.049072904401731121]], [[-0.62189154071521469, -0.12704660196114581, 0.44744358352739189, -0.38090349851710825, 
-0.467899404605296], [0.27900365812178091, -0.026672117399525779, -0.10107093962714847, 0.49889135066960877, 
0.22291752492687333], [0.059895217208755631, -0.58099074441630405, 0.34934414814860426, 0.086137067776019954, 
-0.51307537620719035], [0.43542380038584239, 0.66276221915179245, 0.32245677963340674, 0.50705150823099243, 
0.26391023336746927]], [[-0.78759886012255698, -0.16089905125894668, 0.56666803338414384, -0.48239788067184586, 
-0.59257445003245468], [0.35334612021578538, -0.033779088290479431, -0.1280019933223549, 0.63182441533212785, 
0.28231544737178926], [0.075854713743422994, -0.73579976263671654, 0.44242932225110254, 0.10908888761643352, 
-0.64978787296736251], [0.55144549555975519, 0.83935981463250897, 0.40837762769048769, 0.64215890353946847, 
0.33423094762770039]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.80074594861035642, -0.16358487798065524, 0.57612720753397617, 
-0.49045036518475238, -0.60246607015106868], [0.3592443927305205, -0.034342949775402122, -0.13013868195667042, 
0.64237122020666904, 0.28702803185609393], [0.07712092765547246, -0.74808218847362229, 0.44981462680617423, 
0.11090986696410005, -0.66063453501417091], [0.56065056567018479, 0.85337092906500001, 0.41519452024775855, 
0.65287821809859314, 0.33981013783010705]], [[-0.87748352920609018, -0.17926164510020035, 0.63133898612399986, 
-0.53745150767181316, -0.66020197090041988], [0.39367172338210288, -0.037634124561685738, -0.14261021254961759, 
0.70393133595690882, 0.31453467958377368], [0.08451162805408205, -0.81977286306255026, 0.49292154010068662, 
0.12153864987557148, -0.72394487203548685], [0.61437917715279444, 0.93515169930908015, 0.45498369809587902, 0.7154452469639726, 
0.37237503295600666]], [[-0.85408345096031102, -0.17448122885055631, 0.61450290751603009, -0.52311915052294333, 
-0.64259619567741955], [0.38317357860367435, -0.036630525713224601, -0.13880718944862841, 0.68515941854445239, 
0.30614690264176031], [0.082257934801363175, -0.79791176994689272, 0.47977667501379218, 0.1182975475388155, 
-0.70463924852527926], [0.59799537011887771, 0.91021376918598795, 0.44285053116836964, 0.6963662851347121, 
0.36244481248129429]], [[-0.53155605827520336, -0.108591911184479, 0.38244827592737402, -0.32557375201169919, 
-0.3999327003150227], [0.23847580332900961, -0.022797746331201272, -0.08638945339663423, 0.42642278035257669, 
0.19053669830324199], [0.051194884452694024, -0.4965964798960501, 0.29859868837871412, 0.073624864177678995, 
-0.43854644535139736], [0.37217447715420399, 0.56648989368932634, 0.27561695814180764, 0.4333975997611913, 
0.22557483773772932]], [[-0.86166060785962995, -0.17602917085259387, 0.61995458198660003, -0.52776009734856555, 
-0.64829710487086978], [0.38657297279810476, -0.036955500094031958, -0.14003864271238523, 0.69123793512210085, 
0.30886294065063274], [0.08298770105254516, -0.80499058954690139, 0.48403310117346732, 0.11934704577871044, -0.71089058396269], 
[0.60330059496471955, 0.91828889642711187, 0.44677936031713322, 0.70254422547055262, 0.36566030765149943]]], 
[[[-0.17894786151566183, -0.036557367716615288, 0.12875086277752978, -0.10960410624830196, -0.1346369782780367], 
[0.080282661376162426, -0.0076748404798216985, -0.029082930581201098, 0.14355483952766765, 0.064144005454956812], 
[0.017234711091577756, -0.16717875138504273, 0.10052297571423431, 0.024785743279320489, -0.14763626027623528], 
[0.1252921978042314, 0.19070830530061608, 0.092786197220600733, 0.14590290611104825, 0.075939563093099488]], 
[[0.41723035539724296, 0.085236243649975882, -0.30019228941536613, 0.25555019107604038, 0.3139162090050438], 
[-0.18718504403742819, 0.017894465985182469, 0.067809033087136747, -0.33470887110814007, -0.14955655779229299], 
[-0.040184021049496789, 0.38978979276119419, -0.23437685439547784, -0.057789818719407381, 0.3442250095801358], 
[-0.29212815272333315, -0.44465071179863142, -0.21633797528740889, -0.3401835643890907, -0.17705878477498424]], 
[[0.25407256992426563, 0.051904640193836053, -0.18280219896894634, 0.15561737767020525, 0.19115938457272039], 
[-0.11398639762127548, 0.010896841280757179, 0.041292334264905785, -0.20382108338672636, -0.091072517844817208], 
[-0.024470073583730195, 0.23736262977029296, -0.14272386693995076, -0.035191130193578168, 0.20961593921650692], 
[-0.17789154012770586, -0.27077020548461944, -0.13173908523782224, -0.20715489976284765, -0.10782000852410881]], 
[[-0.69670891000441026, -0.14233108794230709, 0.5012737928696025, -0.42672852723405513, -0.52419057477347186], 
[0.31256951061549226, -0.029880936826318601, -0.11323039400048461, 0.55891104216646814, 0.24973587136121539], 
[0.067100979453745158, -0.65088749687672209, 0.39137239331655488, 0.096499885707061944, -0.5748014930325438], 
[0.48780795604312416, 0.74249658190607482, 0.3612503093442172, 0.56805291676652692, 0.29566025422535663]], 
[[-0.29455213940379593, -0.060174236119373004, 0.21192676883644568, -0.18041078395369778, -0.22161544518478199], 
[0.13214703693628368, -0.012632957241103894, -0.047871147217230729, 0.23629444211571266, 0.10558245221071094], 
[0.028368715787007337, -0.27517992372884453, 0.16546304216813501, 0.040797881840002527, -0.24301255097219887], 
[0.20623372976496815, 0.31391009008203397, 0.15272813358587406, 0.24015941166460433, 0.12499820106821614]]], 
[[[-0.14860163098280121, -0.030357917781831528, 0.10691711002933245, -0.091017287454337126, -0.11180504976842745], 
[0.066668214524002747, -0.0063733302157092898, -0.024151006228972573, 0.1192106075401111, 0.053266374616843533], 
[0.014312024497155984, -0.13882834313340642, 0.083476147833554004, 0.020582542005430273, -0.12259989520651395], 
[0.10404497033622835, 0.15836772213764186, 0.077051383139699392, 0.12116048568333795, 0.063061624968165711]], 
[[-0.74056889367739021, -0.15129127074414259, 0.53283053063374897, -0.4535924096511208, -0.55718999493439514], 
[0.33224672937272048, -0.031762034344258172, -0.12035859799046215, 0.59409622328309009, 0.26545751792432737], 
[0.071325193929864875, -0.69186288067908952, 0.41601049760147141, 0.10257485238364626, -0.61098702724575238], 
[0.51851697767378724, 0.78923904133498857, 0.38399213515149166, 0.60381360720266741, 0.31427298292289996]], 
[[0.038425416575215289, 0.007849951776491752, -0.027646631229577258, 0.023535254376742323, 0.02891055491215792], 
[-0.017239069978357873, 0.0016480160203516208, 0.0062449683016352368, -0.030825484381415913, -0.01377362159867933], 
[-0.0037008039528284458, 0.035898239353547044, -0.021585266146707423, -0.005322234658562134, 0.031701886542101765], 
[-0.026903953215615832, -0.040950732875274669, -0.019923950196630996, -0.031329683961386003, -0.016306477885106231]], 
[[0.62446122511312641, 0.12757156435328865, -0.44929244095711945, 0.38247740926064999, 0.4698327864268651], 
[-0.28015651404648328, 0.026782327813164798, 0.10148856938995802, -0.50095279263518755, -0.22383863037424265], 
[-0.06014270699611287, 0.58339142484611128, -0.3507876542075013, -0.086492990094711314, 0.51519542721749356], 
[-0.43722299151978855, -0.66550078306941873, -0.32378918585117245, -0.50914666834225608, -0.26500072256813229]], 
[[0.37037917133486392, 0.075664986697138184, -0.26648341846776774, 0.22685422280719184, 0.27866626638220476], 
[-0.16616586161583197, 0.015885079782274127, 0.060194693791922597, -0.297124101148945, -0.13276271303430237], 
[-0.035671720009549664, 0.34601993495949129, -0.20805845976479678, -0.051300546309741367, 0.30557166359489857], 
[-0.25932481117352491, -0.39472047045243047, -0.19204518314331223, -0.30198403603738444, -0.15717668940956933]]], 
[[[0.58850708146093067, 0.12022647042874186, -0.42342386126893861, 0.36045578940140921, 0.44278157040838023], 
[-0.26402614894767212, 0.025240301466596556, 0.095645236839980483, -0.47210980295858879, -0.21095083855029345], 
[-0.056679914688106202, 0.54980192681030238, -0.33059061201563672, -0.081513046960195915, 0.48553239986815511], 
[-0.41204932562516489, -0.62718373504006841, -0.30514661457060721, -0.47983190592398417, -0.24974297130355341]], 
[[0.80765609070189803, 0.16499655512099612, -0.58109897276584754, 0.49468277087866397, 0.60766513004928169], 
[-0.36234453929219285, 0.034639316760714446, 0.13126173076570552, -0.6479146468263709, -0.28950497786852547], 
[-0.077786452806434092, 0.75453786174611925, -0.45369636107092909, -0.11186697819445464, 0.66633556730218912], 
[-0.56548877319306712, -0.86073520532123804, -0.41877749576642403, -0.65851231623341755, -0.3427425739424938]], 
[[0.74688007385731958, 0.15258058559042981, -0.53737134987812551, 0.45745795605739958, 0.56193840724621003], 
[-0.33507815936007951, 0.032032712634069031, 0.12138430242472427, -0.5991591530677004, -0.26771976555589894], 
[-0.07193303224727586, 0.69775898479181886, -0.41955576831094937, -0.10344900248750527, 0.61619390164929322], 
[-0.52293581581347992, -0.79596499193516268, -0.38726454312501984, -0.60895934921625128, -0.31695123937931319]], 
[[0.62406908312402187, 0.12749145342728263, -0.44901029944952686, 0.38223722548938727, 0.46953774622900085], 
[-0.2799805845759884, 0.026765509354501331, 0.10142483776360327, -0.50063821005319331, -0.22369806676160606], 
[-0.060104939269626234, 0.5830250733985507, -0.35056737060470544, -0.086438675220039646, 0.5148719007093906], 
[-0.43694842924645327, -0.66508286952998941, -0.32358585643649629, -0.50882694026429021, -0.26483431045752742]], 
[[0.80963208847624235, 0.16540023291089581, -0.58252067971522636, 0.49589305340549195, 0.60915183331116196], 
[-0.36323104533285794, 0.034724064729078671, 0.13158287350311024, -0.64949982387738692, -0.29021327586629408], 
[-0.077976763830382489, 0.75638390135707934, -0.45480636693855664, -0.11214067005721612, 0.66796581266666266], 
[-0.56687228849136084, -0.86284106556249041, -0.41980206972693335, -0.66012342136372693, -0.34358112214522551]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_constData_rank2(self):
      arg0=Data(0.225397368462,self.functionspace)
      arg0.setTaggedValue(1,0.0216690720791)
      arg1=Data(numpy.array([[0.76634493336193255, -0.89595771413863501, 0.55782709144223852, 0.37568594650018494, 
0.061233579650666448], [0.047036697634215807, -0.8632881501733054, -0.9030698144293674, -0.23242216754215228, 
0.15151450432281854], [0.30818600139855978, 0.26464667502847483, 0.30440060900476196, 0.80509459503000391, 
-0.088804512131029245], [0.94192922889525943, -0.12147848638351832, 0.75203429485740125, -0.56127253080526107, 
0.81952666370003358]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.17273213131375353, -0.20194651101982838, 0.1257327584677371, 0.084678623709193088, 
0.013801887714751486], [0.010601947867881963, -0.19458287727325119, -0.20354955970959532, -0.052387344936170657, 
0.034150970558145546], [0.069464313711976047, 0.059650664123562826, 0.068611096227818996, 0.1814662030825189, 
-0.020016303341861087], [0.21230836947017076, -0.027380931155558196, 0.1695065510538252, -0.12650935143335681, 
0.18471915338220246]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.016605983598507824, -0.019414572287536701, 0.012087595452162066, 
0.0081407658538345841, 0.0013268748511144008], [0.0010192415914008044, -0.018706653151177713, -0.019568684901370696, 
-0.0050363727012621767, 0.0032831787152071881], [0.0066781046780890899, 0.0057346478766982702, 0.0065960787374600349, 
0.017445652810235773, -0.0019243113743206415], [0.020410732354385508, -0.0026323260775099784, 0.016295885341254538, 
-0.012162254926063728, 0.017758382346497827]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_constData_rank2(self):
      arg0=Data(numpy.array([-0.420405043604823, -0.51788533909262502]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.29847183705600089, 0.81519753475723822]))
      arg1=Data(numpy.array([[-0.12206483254483413, 0.65494064701636567, -0.48430977185652502, 0.45554536203549101, 
-0.27524260680171242], [0.39050802690019681, -0.39837715600681367, 0.53599934512769498, -0.15319346866952466, 
0.58970993316684472], [-0.79872005305498406, 0.71974435353958843, -0.25231870282263702, -0.42133344979722187, 
-0.72979354452764222], [-0.19288013480214028, 0.25527738774271613, -0.89008051794315923, 0.067297699968212665, 
-0.28402397058693918]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.051316671248626412, -0.27534035126748618, 0.20360627075558427, -0.19151356779050549, 
0.11571338011437907], [-0.16417154407701062, 0.16747976564220987, -0.22533682806056518, 0.064403306875985597, 
-0.24791703016720462], [0.33578593873262713, -0.30258415633413582, 0.1060760552624631, 0.17713070733417158, 
0.30680888690966179], [0.081087781481997925, -0.10731990132530189, 0.37419433895769733, -0.028292292489640741, 
0.11940510973941713]], [[0.06321558719376591, -0.33918415906561378, 0.25081693042378833, -0.23592026428982288, 
0.1425441107562429], [-0.20223838192960036, 0.20631368852534426, -0.27758620260488126, 0.079336651468692204, 
-0.30540212870440064], [0.41364540551645984, -0.37274504859285196, 0.13067215697071266, 0.21820241651929975, 
0.37794937727530675], [0.09988979401623764, -0.13220441651381606, 0.46095965085473234, -0.034852492168191557, 
0.14709185031785074]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.036432914809589767, -0.1954813380776205, 0.1445528273101897, -0.13596746106907401, 
0.082152166488189626], [-0.11655564817401595, 0.11890436159449874, -0.15998070920107657, 0.04572393601877394, 
-0.17601180708247965], [0.23839544152878758, -0.21482341941164473, 0.075310026755059628, 0.12575616877411913, 
0.21782281990677577], [0.057569288166003901, -0.076193110878425521, 0.26566396731825148, -0.020086468139156006, 
0.084773156269023306]], [[-0.099506950571103883, 0.5339060008600518, -0.39480813207627968, 0.37135945610142584, 
-0.22437709452491181], [0.31834118083195373, -0.32475607548035418, 0.43694534477959107, -0.12488293800030671, 
0.48073008373946752], [-0.65111461821159355, 0.58673382266091456, -0.20568958451415789, -0.34346998958545788, 
-0.59492589838068066], [-0.15723541039434855, 0.20810149716712981, -0.72559144396270914, 0.054860919108919236, 
-0.23153564063443516]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[-0.10911550803518799, -0.58492445287555062, 0.078162430202799449, 0.0018071103240271391, 
0.55105439933911038], [0.80214629737797472, 0.1625286610433514, -0.67139562704171207, 0.31911802847334125, 
0.97492426824503897], [-0.95707788280475481, 0.76813103185057163, -0.65430035969129241, -0.17316352762927045, 
0.64421876273006484], [-0.90719832292528757, -0.1649570449938127, 0.16986383978424224, 0.33195144142893329, 
0.19265533267355117]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.84749935283404754, 0.26848289875054632, -0.13272972895251733, 0.40619605694701488, 
0.9659301801147977], [-0.32547175976318443, 0.76810258246697671, 0.04342699846518383, -0.076610162106647506, 
-0.53441321116498908], [0.91737603412556257, 0.33861644850525674, -0.54587963474813139, -0.52891806794545593, 
0.047745226608445268], [0.079375503223253263, -0.3753499528523101, 0.78351497398548631, -0.72708272993510481, 
0.034669027430718202]]))
      arg1=Data(numpy.array([[-0.36559153355718021, -0.43250581132968224, 0.5972138355338461, -0.48948698119653167, 
0.71647512525894097], [-0.4846906895366847, -0.98356804961651645, -0.67439700014266579, -0.66968689500835832, 
-0.80637602655417684], [0.99388470778043159, -0.89375506817150696, 0.78949275865264124, 0.75718180128294543, 
-0.50560531829848099], [0.63015164563814263, -0.2740411316487994, -0.33031299293891969, 0.79575198902682187, 
-0.29144407265203753]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.039891705917455195, 0.047193091331409441, -0.065165291069918821, 0.053410620629870065, 
-0.078178547287204289], [0.052887270828720925, 0.10732252742108518, 0.073587171287973721, 0.073073225773344616, 
0.08798812980485525], [-0.10844823481786615, 0.097522538322558056, -0.086145903450485006, -0.082620276921987354, 
0.055169381171431685], [-0.068759316953015695, 0.029902137302396582, 0.036042270035153687, -0.086828882552673015, 
0.031801068051271315]], [[0.21384342774186715, 0.25298322505751053, -0.34932497599934398, 0.28631290466608622, 
-0.41908382064102762], [0.28350743639111864, 0.57531300328781332, 0.3944712963293614, 0.39171624066069027, 
0.47166905614416238], [-0.58134746891984546, 0.5227791942549691, -0.46179361990410528, -0.44289415084275074, 
0.29574091417670761], [-0.36859110655351845, 0.16029335899507072, 0.19320814667248321, -0.46545479680614493, 
0.17047276473981529]], [[-0.02857552272439751, -0.033805705290361435, 0.046679684736060392, -0.038259492002952915, 
0.056001436970093965], [-0.037884602190857854, -0.076878069027854545, -0.052712508452628447, -0.052344355188820288, 
-0.063028309892751599], [0.077684444101517716, -0.069858068134353676, 0.061708672643802663, 0.059183169693608186, 
-0.039519340401669219], [0.049254184019370532, -0.021419720825195457, -0.025818066255666097, 0.0621979093010478, 
-0.022779976986684495]], [[-0.00066066423466809468, -0.00078158571685560277, 0.0010792312878450591, -0.0008845569771971305, 
0.00129474959576407], [-0.00087588954902157574, -0.0017774159768452443, -0.0012187097814507434, -0.001210198101835283, 
-0.0014572104426340353], [0.0017960593163227141, -0.0016151140108443098, 0.0014267005149058544, 0.0013683110502638765, 
-0.00091368459058021277], [0.001138753544535379, -0.00049522255821062575, -0.00059691201970022529, 0.0014380116347355006, 
-0.00052667159256601257]], [[-0.20146082292781617, -0.23833423007295265, 0.32909731141710979, -0.2697339544075692, 
0.39481676979097957], [-0.26709093678789703, -0.5419995007905698, -0.37162943382971464, -0.36903390967410477, 
-0.44435705695427041], [0.54768454065827299, -0.49250766224753539, 0.43505345790190847, 0.41724836269647914, 
-0.27861603497762916], [0.34724783657967861, -0.15101157119493921, -0.18202042791786019, 0.43850263433607767, 
-0.16060153839621258]]], [[[-0.29325789499562771, -0.34693293515256152, 0.47905286691637339, -0.3926401695815202, 
0.57471786888988019], [-0.38879284198542913, -0.78896546921916477, -0.5409650566272528, -0.53718686323350717, 
-0.64683154399479637], [0.79724093836666354, -0.71692231869657363, 0.63328869315993919, 0.60737057834110009, 
-0.40556943400773887], [0.50547380933527375, -0.21982107908135456, -0.26495934426179152, 0.63830951162902394, 
-0.23378078377058936]], [[-0.059419102437833972, -0.070294590408881613, 0.097064365045880285, -0.079555663652024419, 
0.11644774277920315], [-0.078776128790576092, -0.15985799814919302, -0.10960884144484033, -0.10884331436398792, 
-0.13105921589330832], [0.16153475078701612, -0.14526081453062428, 0.12831520096723556, 0.1230637443289101, 
-0.082175355399449604], [0.10241770321983176, -0.044539538197684157, -0.0536853284675846, 0.129332505299113, 
-0.047368014897156883]], [[0.24545655691376414, 0.2903825103968764, -0.40096675758623251, 0.3286394186692001, 
-0.48103826598301586], [0.3254192094227622, 0.66036328741047479, 0.45278719678583468, 0.44962485279575393, 
0.54139733797974587], [-0.66728984658741164, 0.60006324441671699, -0.53006198574048113, -0.50836855025693617, 
0.33946119971463307], [-0.4230810592545875, 0.18399001741856605, 0.22177069901425059, -0.53426440564235267, 
0.19567427590580502]], [[-0.11666684941531273, -0.13802040181479111, 0.19058170177256323, -0.15620412040280465, 
0.22864012942282347], [-0.15467353726433117, -0.31387429686299223, -0.21521224109386314, -0.21370916162950079, 
-0.25732912780213557], [0.31716652847669424, -0.285213355292948, 0.25194137263521033, 0.2416303636213068, 
-0.16134777236104741], [0.20109275079527564, -0.087451465652368221, -0.10540883108579674, 0.25393880589197926, 
-0.093005057874959454]], [[-0.35642405832981555, -0.42166041162231732, 0.582238261593648, -0.4772127369585018, 
0.69850898720884569], [-0.47253671582173568, -0.9589043610415825, -0.65748600187073791, -0.65289400606931602, 
-0.78615555761867295], [0.96896232145277161, -0.87134350582740139, 0.76969565001418339, 0.73819491354423605, 
-0.49292689496294662], [0.61435013200717326, -0.26716934974174816, -0.32203015293280501, 0.77579792560650862, 
-0.28413589926464167]]], [[[0.34989957090824952, 0.41394174620816504, -0.57158015329444045, 0.46847716362406738, 
-0.68572249596509871], [0.46388673895694693, 0.94135122652137759, 0.64545045306642046, 0.64094251561668969, 
0.77176466023898227], [-0.95122507187451788, 0.85539320839160515, -0.75560605794095514, -0.72468195527017198, 
0.48390366757193432], [-0.6031042028532857, 0.26227870607985199, 0.31613525994488317, -0.7615966288954632, 
0.2789346760098072]], [[-0.28082220190710971, -0.33222113513803747, 0.4587384797240508, -0.37599013994391323, 
0.55034677726041792], [-0.37230595948217871, -0.75550914084718901, -0.51802526359651602, -0.51440728567957572, 
-0.61940244933662381], [0.76343368612788676, -0.6865210027362576, 0.60643388734240766, 0.58161483831794336, 
-0.38837113485374891], [0.48403903378636226, -0.21049949722289063, -0.25372366009982295, 0.61124179642831744, 
-0.22386723625294255]], [[0.2392066719065542, 0.28298870792158537, -0.39075722740241187, 0.32027150786109554, 
-0.46878993216678888], [0.31713329250287337, 0.64354892864494961, 0.44125819976807479, 0.43817637628451361, 
0.52761212422083303], [-0.65029912179241145, 0.58478426258053262, -0.51656539596009388, -0.49542432493113187, 
0.33081774162452648], [-0.41230844840109654, 0.17930521100801827, 0.2161239100906425, -0.52066081264531094, 
0.19069196156612331]], [[0.063307119622156127, 0.074894232010007458, -0.10341565451004774, 0.084761292392593793, 
-0.12406736014846163], [0.083930749609235855, 0.17031811313503731, 0.11678096356730162, 0.1159653451467402, 
0.13963491735379552], [-0.17210458205604615, 0.15476578044111725, -0.13671135112605559, -0.1311162717668401, 
0.087552400504685088], [-0.10911928190009075, 0.047453929071823414, 0.057198163079085637, -0.13779522153789298, 
0.050467483727068205]], [[-0.23552092541279362, -0.27862835864837077, 0.38473635821289076, -0.31533669739890413, 
0.46156671872118316], [-0.31224683632010503, -0.6336329919847753, -0.43445920102077551, -0.43142486291882343, 
-0.51948256612191768], [0.64027917674264168, -0.57577378420117298, 0.50860604816355026, 0.48779072318422095, 
-0.32572043258398803], [0.4059555134853185, -0.17654243876793635, -0.2127938276247755, 0.51263836181084732, 
-0.18775373988890676]]], [[[0.33166402611875789, 0.39236854669372856, -0.54179139002408372, 0.4440617684352553, 
-0.64998503205259661], [0.4397105806851816, 0.89229128509499978, 0.6118118275152713, 0.60753882803662584, 0.7315429789371064], 
[-0.90165054007949708, 0.81081309895116715, -0.71622650661133491, -0.68691406027343649, 0.45868429682248818], 
[-0.57167251611153314, 0.24860965504433877, 0.29965939323462032, -0.72190486990959468, 0.26439757393644414]], 
[[0.060306899050348764, 0.071344880579595857, -0.098514629539084103, 0.080744325981121828, -0.11818761947428671], 
[0.079953143881985003, 0.16224647901506831, 0.11124653629622602, 0.11046957127166049, 0.13301740649422925], 
[-0.16394828445999904, 0.14743119499381541, -0.13023239251135305, -0.12490247246272697, 0.083403159239673522], 
[-0.10394795336245621, 0.045205015283546347, 0.054487455238266314, -0.13126489665781341, 0.048075753005642173]], 
[[-0.062100781682632285, -0.073467097841458842, 0.10144503527605403, -0.083146138150440044, 0.12170321588637964], 
[-0.082331421632373308, -0.16707264559695958, -0.11455566398320738, -0.11375558743930643, -0.13697412818045257], 
[0.16882507276642364, -0.15181666770623936, 0.13410627146659168, 0.1286178081806702, -0.085884060781513977], 
[0.10703997817445406, -0.046549678880684099, -0.056108233311230195, 0.13516948837204415, -0.04950580926303276]], 
[[-0.12135863653852021, -0.14357092749727829, 0.19824599354676212, -0.16248590896888584, 0.23783495057768098], 
[-0.16089377303888608, -0.32649683181364719, -0.22386705629270645, -0.22230353010409126, -0.26767768434839478], 
[0.32992146136188844, -0.29668328316394627, 0.26207325923244923, 0.25134759035962989, -0.16783641420331538], 
[0.20917974708839582, -0.090968348661635029, -0.10964787412877945, 0.26415101977739425, -0.096745280012762613]], 
[[-0.070433158520092298, -0.083324550964964092, 0.11505643016202059, -0.094302277201790102, 0.13803275360908551], 
[-0.0933782460364629, -0.18948962980594586, -0.1299261784165302, -0.12901875154495282, -0.15535264165577128], 
[0.19147718901659425, -0.17218667998725409, 0.15209999006158426, 0.14587511182052457, -0.097407560798310588], 
[0.12140207492520209, -0.052795485384035883, -0.063636559541043927, 0.15330586417160241, 
-0.056148254772512909]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.3098385880913172, -0.36654839519887039, 0.50613833911847383, 
-0.41483989978475216, 0.60721220497864559], [-0.41077504570702855, -0.83357328551824406, -0.57155102117413237, 
-0.56755921012102639, -0.68340316064555562], [0.84231664663557226, -0.75745684186750217, 0.66909460202528037, 
0.64171108656501463, -0.42850018004741525], [0.53405311186563598, -0.23224968172226751, -0.27994004774841175, 
0.67439929571663759, -0.24699866295992093]], [[-0.098155074688089372, -0.11612041395224997, 0.16034170173805903, 
-0.13141888361229898, 0.19236131851218125], [-0.13013116132421021, -0.26407120107946352, -0.18106406150697552, 
-0.17979947882709682, -0.21649817309221292], [0.26684104736872993, -0.23995795147567833, 0.21196530438562658, 
0.20329036488960531, -0.1357463814804688], [0.16918494047335558, -0.073575357401949745, -0.088683389839209897, 
0.21364580070043407, -0.078247749449283852]], [[0.048524865156379673, 0.057406379108177322, -0.079268030517100657, 
0.064969474340001596, -0.095097549176840143], [0.064332863848012886, 0.13054872063195636, 0.089512531035326831, 
0.088887360057512327, 0.10703007143834381], [-0.13191804787374858, 0.11862786794834278, -0.10478915986594026, 
-0.10050053525206419, 0.067108856854708632], [-0.083639857124533423, 0.036373405125586261, 0.043842354022277583, 
-0.10561994581695662, 0.038683292767922713]], [[-0.14850183938413888, -0.17568215516878649, 0.24258590514805134, 
-0.19882768168892878, 0.29102937078080043], [-0.19687944692873108, -0.39952146349329487, -0.27393740227484625, 
-0.27202417614148472, -0.32754676240290798], [0.40371204936034744, -0.3630397845676766, 0.32068884555292415, 
0.30756426207317061, -0.20537488666428336], [0.25596511373688613, -0.11131442711704012, -0.13417183529015636, 
0.32323132025043932, -0.11838343313183698]], [[-0.35313589585733218, -0.41777041623837669, 0.5768668677242571, 
-0.47281024791101445, 0.6920649467891411], [-0.46817736504413537, -0.95005806332124199, -0.65142041581668442, 
-0.64687078311594315, -0.77890294056973097], [0.96002323479969531, -0.863304993977417, 0.76259488256467423, 
0.73138475369288247, -0.48837943617105134], [0.60868249257088725, -0.2647045996523878, -0.31905928876374862, 
0.76864086208738658, -0.28151462559017282]]], [[[0.11898971978137673, 0.14076842752127552, -0.19437623800612189, 
0.15931418915120393, -0.2331924198445755], [0.15775313166433605, 0.32012362395553073, 0.21949717841544597, 0.21796417220871331, 
0.26245262439343225], [-0.32348140484301541, 0.29089203483504522, -0.25695759747896618, -0.24644129332421807, 
0.1645602526922316], [-0.20509656502351289, 0.089192649365229235, 0.1075075510844745, -0.25899480020361393, 
0.09485681519860803]], [[-0.28081180105333248, -0.33220883061430395, 0.4587214893585555, -0.37597621434102046, 
0.55032639398474315], [-0.37229217033082718, -0.75548115894245382, -0.51800607741756366, -0.51438823350021112, 
-0.61937950843572265], [0.76340541072058599, -0.68649557595548327, 0.60641142676007131, 0.58159329696242756, 
-0.38835675069410103], [0.48402110636047252, -0.21049170092161557, -0.25371426289878046, 0.61121915777473523, 
-0.22385894484872321]], [[-0.015876542966671868, -0.0187824292047972, 0.025935204319114884, -0.021256950381149248, 
0.031114364164962421], [-0.021048661830598499, -0.042713408181100317, -0.029287037490120127, -0.029082491761681702, 
-0.035018490467529272], [0.04316142967935048, -0.038813099973734305, 0.034285300818282002, 0.032882132922179598, 
-0.021956921381736917], [0.027365594547960684, -0.011900783803509651, -0.014344501837388743, 0.034557120406134775, 
-0.012656541295746958]], [[0.028008026650633438, 0.033134340318034056, -0.045752648752580671, 0.037499676978559812, 
-0.054889275491468048], [0.037132232296988171, 0.075351307724040445, 0.051665663505166409, 0.051304821587287761, 
0.061776598113229782], [-0.076141668578376856, 0.068470720656256936, -0.060483168222403183, -0.058007820540489807, 
0.038734505396829738], [-0.048276019724108804, 0.020994335519503651, 0.02530533193498255, -0.060962688876032012, 
0.022327577650894149]], [[0.19537694542302553, 0.23113681948021439, -0.31915896359980234, 0.26158830944469513, 
-0.38289377240946842], [0.25902510781707244, 0.52563175979484789, 0.36040666644627761, 0.35788952403652768, 
0.43093800175728214], [-0.53114511821271726, 0.47763451597651874, -0.42191536034306371, -0.40464795785930951, 
0.27020216173398764], [-0.33676136446638211, 0.14645120115572241, 0.17652362724600643, -0.42526037574675102, 
0.15575156274097776]]], [[[-0.33538491116456853, -0.39677046593388271, 0.54786965996695569, -0.44904362556616806, 
0.65727710895966296], [-0.4446436225447481, -0.90230175664981438, -0.61867564541705522, -0.61435470784862978, 
-0.73975004125420007], [0.91176601160165593, -0.81990947991879892, 0.72426173590360998, 0.69462043797299833, 
-0.46383020173345324], [0.57808601757321609, -0.2513987665392568, -0.30302122348245109, 0.73000380384095398, 
-0.26736380753892852]], [[-0.12379530669672276, -0.14645358179034162, 0.20222642798667348, -0.16574834316232895, 
0.24261026235754155], [-0.16412423991447617, -0.33305231982438693, -0.22836191707090861, -0.22676699799824304, 
-0.27305218627155597], [0.33654570997229466, -0.30264016701780933, 0.26733523405557519, 0.25639421242324406, 
-0.17120627722760154], [0.21337971226573091, -0.09279483474327796, -0.11184941256411893, 0.26945471241525648, 
-0.098687756819340972]], [[0.19956897280520275, 0.23609611431509117, -0.32600687040774651, 0.26720097450952818, 
-0.39110917968247239], [0.26458277657010537, 0.53690976767459597, 0.36813958811311392, 0.36556843764277286, 
0.44018425084504353], [-0.54254142126493532, 0.48788269016775349, -0.43096801872959845, -0.41333012512226647, 
0.27599964647948749], [-0.34398695015688324, 0.14959347285041119, 0.18031113593805959, -0.43438480512006056, 
0.15909338392880212]], [[0.19336796758628008, 0.2287601381036774, -0.31587718804085718, 0.25889850837892325, 
-0.37895663898293758], [0.25636166306089408, 0.52022691249604824, 0.35670075834367015, 0.35420949863621226, 
0.42650685000256888], [-0.52568357939975996, 0.47272320387373273, -0.41757698456348313, -0.40048713541803566, 
0.26742378809737982], [-0.333298590923576, 0.14494530588926932, 0.17470851004253443, -0.42088760459982028, 
0.15415003582127079]], [[-0.017455250615816591, -0.020650087971405155, 0.028514109911262246, -0.023370666839112191, 
0.034208267214802343], [-0.023141666806932602, -0.046960679413767116, -0.032199237595867271, -0.031974352558880162, 
-0.038500606119446855], [0.047453250595645112, -0.04267253826229505, 0.037694510667596944, 0.036151816686045002, 
-0.024140240496596074], [0.030086733118677822, -0.01308415593060671, -0.015770868699582503, 0.037993359040206662, 
-0.013915063292459718]]], [[[-0.029019011950262061, -0.03433036642127496, 0.047404148727388247, -0.038853275453705809, 
0.056870573614371853], [-0.038472567389599963, -0.078071208892624727, -0.053530601258576498, -0.053156734293306418, 
-0.064006502894905226], [0.078890098825967772, -0.070942258294446392, 0.062666385009167838, 0.060101686508323149, 
-0.040132676572295072], [0.050018603979488742, -0.021752152728493249, -0.026218760035705651, 0.063163214569908699, 
-0.023133519928189863]], [[0.13722476488389135, 0.16234103589094637, -0.22416418501037641, 0.18372891531393776, 
-0.26892890448579648], [0.18192862746554828, 0.36918222105059806, 0.25313488220728897, 0.25136694446719726, 
0.30267320354834343], [-0.37305457820601701, 0.33547092269968831, -0.29633606973750914, -0.28420815341218081, 
0.18977893238521207], [-0.23652739048008245, 0.10286132584397056, 0.12398296632612894, -0.29868547156334957, 
0.10939351892902753]], [[-0.2864464409043681, -0.33887477951254763, 0.46792598281207393, -0.38352037933843475, 
0.56136898912850719], [-0.37976241300334296, -0.77064029480824037, -0.52840014802267077, -0.52470971012089496, 
-0.63180769146811566], [0.77872355096115753, -0.70027047898779482, 0.61857939825745401, 0.59326327933449063, 
-0.39614933781355788], [0.49373325023908071, -0.21471533013476229, -0.25880517606960579, 0.62348359898124928, 
-0.22835079500218536]], [[0.26581529025991607, 0.31446750601438272, -0.43422386589496353, 0.35589753055606754, 
-0.520936690003867], [0.35241022972246105, 0.71513534259212341, 0.4903424119237747, 0.48691777582444107, 0.58630208274123341], 
[-0.72263640657375006, 0.64983387485947497, -0.57402655022515925, -0.55053381113398403, 0.36761689509816714], 
[-0.4581723787836795, 0.19925057411371452, 0.24016487263906472, -0.57857752853291122, 0.21190395196724848]], 
[[-0.012674702905332215, -0.014994555836933786, 0.02070482284612734, -0.016970037578082003, 0.024839495771029483], 
[-0.016803754810961042, -0.034099347692133013, -0.023380688097140147, -0.023217393333037274, -0.027956272584080306], 
[0.03445701619701113, -0.030985618974781391, 0.027370946106081803, 0.026250756638719053, -0.017528844649207044], 
[0.021846744688140983, -0.0095007395102772851, -0.011451630212922035, 0.027587947535619457, -0.010104082549293718]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_taggedData_rank2(self):
      arg0=Data(-0.529182564701,self.functionspace)
      arg0.setTaggedValue(1,0.505263550184)
      arg1=Data(numpy.array([[0.42884373860760117, -0.068232018320336429, -0.72086021317208449, 0.61673057172625256, 
-0.25343088331633368], [-0.42115584199478984, -0.45782821471451496, 0.89441452271385002, -0.38106766062023834, 
0.48753906781464851], [0.077379961060325186, 0.59339673892114719, -0.68196652339579877, -0.90043093145735642, 
-0.88059979073462347], [-0.14209365154539189, 0.78766008225188555, -0.032405291587796814, 0.13652901171293585, 
-0.24408701154776602]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.71192732027475625, 0.97480147407534923, 0.33045577685618999, -0.72858811265773404, 
0.52950938502611322], [0.48365917575093342, 0.63811692441682299, 0.92153749509836369, 0.39296655163513883, 
-0.1441144123146576], [-0.59589390319785274, -0.77269337971048424, -0.16921667450785649, -0.02184256561968545, 
0.97308315273670942], [-0.36761708017574901, 0.00018879234760871277, 0.024060525442881842, -0.63203088672620056, 
-0.025545414181100679]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.22693662945245047, 0.036107194449499523, 0.38146665639750627, -0.3263630656757775, 
0.13411120480784519], [0.22286832860572478, 0.24227470885522973, -0.4733085710357755, 0.20165436197173003, -0.257997174298221], 
[-0.040948126250387115, -0.31401520818766127, 0.36088479389099593, 0.47649234964495535, 0.46599805573634778], 
[0.075193482952558746, -0.41681598243886431, 0.017148315312322738, -0.072248772574380823, 
0.12916659078111478]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.35971092531494386, 0.49253165351582379, 0.16696725899314191, -0.36812901642324436, 
0.26754179173399512], [0.24437535221894233, 0.32241722266328443, 0.4656193064009907, 0.19855167498270068, 
-0.072815759598772303], [-0.30108346906269678, -0.39041380023612654, -0.085499017712155428, -0.011036252250127379, 
0.49166344837590586], [-0.18574351103784359, 9.538989180033382e-05, 0.012156906504560874, -0.31934216965316742, 
-0.012907166660061437]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_taggedData_rank2(self):
      arg0=Data(numpy.array([0.48320818533568022, 0.021138959966740645]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.77572627931908156, 0.79053816903827934]))
      arg1=Data(numpy.array([[-0.79924535774142424, 0.23747002905816617, -0.41964508102496145, 0.11037869360897012, 
0.64387686311190473], [-0.51950722275519223, 0.88451201061098272, -0.193933370623117, 0.21701971817055821, 
0.60508430216249254], [0.33302029079136886, -0.35224166521252065, -0.84793030448704587, -0.58220347819015661, 
0.013652909153885506], [0.20221680679882148, -0.84754850904434487, 0.48655986012028385, -0.18417267540557414, 
-0.19281930449595275]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.8625259293690104, 0.87533076307492763, -0.48463181966183888, 0.59839268331547402, 
-0.17375801410267844], [-0.50554705901607555, -0.51594269167766105, 0.82091654472323183, -0.17008286215992063, 
-0.87988438920757095], [-0.28509541427022422, -0.80970035921218364, 0.30510419926439702, 0.96253476854290287, 
-0.1051045255528813], [-0.6877827323718817, 0.70659456979078494, 0.60718770550057433, -0.90364803647626291, 
-0.59126546554669135]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.38620189895220014, 0.11474746181280772, -0.20277593808711611, 0.053335888238513499, 
0.31112657060393367], [-0.25103014237631543, 0.42740344355494686, -0.093710192094828282, 0.10486570419925617, 
0.29238168762304445], [0.16091813039325789, -0.1702060558469603, -0.4097268637223162, -0.28132548619238684, 
0.0065971974568019131], [0.097712816257634366, -0.40954237703927926, 0.23510970706590475, -0.088993744271144745, 
-0.093171866223177288]], [[-0.016895215620899271, 0.0050198694375613124, -0.0088708605680262939, 0.0023332907853811508, 
0.0136108872328331], [-0.010981842384254623, 0.01869766398240684, -0.0040995497578171468, 0.0045875711344007669, 
0.01279085283991613], [0.0070397025951510744, -0.0074460224595455345, -0.01792436476113787, -0.01230717601795888, 
0.00028860830003353262], [0.0042746529835224149, -0.017916294002559126, 0.010285369404505609, -0.0038932188123659509, 
-0.0040759995585547193]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.66908403000565542, 0.67901707601364614, -0.37594163830591437, 0.46418892980007409, 
-0.13478865778174326], [-0.39216613911124443, -0.40023030455698405, 0.63680653686962907, -0.13193774583925544, 
-0.6825494434709316], [-0.22115600496277324, -0.62810584701499106, 0.23667734529999837, 0.7466635147170394, 
-0.081532342546733944], [-0.53353113996275137, 0.54812397661087275, 0.47101145963625074, -0.70098352914972517, 
-0.45866015967839946]], [[0.6818596689514177, 0.69198237874413315, -0.38311995137315968, 0.47305225623411773, 
-0.13736234232445893], [-0.39965424639725533, -0.40787239080753962, 0.64896586219873442, -0.13445699443669371, 
-0.69558219400951793], [-0.22537880679839278, -0.64009903944123669, 0.24119651505236675, 0.76092047355959047, 
-0.083089139188211827], [-0.54371850194541227, 0.55858997745479777, 0.48000505696897799, -0.7143682642109811, 
-0.46741791854884723]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[0.86175393352299823, 0.23957413082465506, 0.41141632995944821, 0.79156287481707488, 
0.30696954655573916], [-0.040430119873048342, -0.9587596943025678, -0.2109072508208949, -0.97799047967991837, 
-0.63745015003207817], [-0.56656595927783293, -0.55914262954433624, 0.384184976323799, 0.71035901197763818, 
0.58290551759881071], [-0.48413237917059559, 0.92862300784878826, -0.45927107074108187, 0.17008503557304389, 
-0.85814566225308253]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.0080394578689553242, -0.58167803677557761, 0.70290875802421904, 
0.28002327896666634, 0.75931179393502135], [-0.42598554496756424, -0.024800019092734216, 0.86297740987682348, 
-0.79067675290410655, -0.22598628180359692], [0.52580316670819038, 0.45594366540153097, -0.92285609770294075, 
0.94421446978099977, 0.27330160781988599], [0.93578626496954942, 0.57655297538788064, 0.34212179057288572, 0.27624775584389449, 
0.047633988714469]]))
      arg1=Data(numpy.array([[0.93059186365930069, -0.12048183719956307, 0.56360629673973794, 0.69109000453071401, 
0.80761188853414145], [-0.57316087041503194, -0.79165514975819984, 0.25958211161227829, -0.49210601356258232, 
-0.22146966712563199], [-0.2537459053975617, 0.34664495200968393, 0.8973616487178695, 0.58485629419059459, 
0.72514438049619723], [-0.16250651511487568, -0.54464415669592059, 0.57274929220302351, -0.7549587105677853, 
-0.27920860244179124]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.30920078193483103, -0.81730472364112527, -0.04065000636178695, 0.58314195253955048, 
0.77957527883548305], [0.11803466794636774, 0.42166139994840823, 0.1440393024152582, -0.61098046016547491, 
0.65364121412798148], [0.55429339933174537, 0.84004848923525466, -0.32854656427279716, -0.7035722888285556, 
0.68849185217122355], [-0.94712788362227096, -0.71837210769902793, 0.74603793890866221, -0.96075687668654042, 
-0.14791522607261087]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.80194119901290006, -0.10382569712480097, 0.48568994317379932, 0.59554952982276943, 
0.69596272170423357], [-0.49392363462161926, -0.68221193929786694, 0.22369590575408677, -0.42407429289787724, 
-0.19085235680154242], [-0.21866653209170339, 0.29872265093023609, 0.77330493057530703, 0.50400221206442875, 
0.62489602226469565], [-0.1400406286233587, -0.46934924440302578, 0.49356895547846863, -0.65058863847923976, 
-0.24060911142767261]], [[0.22294573688867286, -0.028864331427242915, 0.13502548866872535, 0.16556728715705274, 
0.19348291623922514], [-0.13731451735238404, -0.18966009441618287, 0.062189158767140172, -0.11789587047284157, 
-0.053058403005648964], [-0.060790954735935994, 0.083047163082474298, 0.21498463702696302, 0.14011643833804047, 
0.17372583467975941], [-0.038932357111990015, -0.13048265044915242, 0.13721591385997575, -0.18086857689277949, 
-0.066891158248758797]], [[0.3828606892368327, -0.049568195287415961, 0.23187683414669871, 0.28432571333568479, 
0.33226471921233547], [-0.23580774178251532, -0.32569985630701592, 0.1067963196826474, -0.20246045005089205, 
-0.091116237646168163], [-0.10439520914090218, 0.14261539395479322, 0.36918923616186544, 0.24061943010957779, 
0.29833623971446316], [-0.066857834043061753, -0.22407550008169427, 0.23563841178503953, -0.31060234197271552, 
-0.11487097850970839]], [[0.7366219708795354, -0.09536894941692893, 0.44612982051231237, 0.5470411907436773, 
0.63927558822453201], [-0.45369286631837963, -0.62664482620634265, 0.20547556251890181, -0.38953285081036809, 
-0.17530716639474589], [-0.20085583834955545, 0.27439127475361236, 0.71031816640970691, 0.46295052958436794, 
0.57399737048301669], [-0.12863412428083543, -0.43112009442654431, 0.45336707628567013, -0.5975972873052281, 
-0.22101116402248203]], [[0.28566336241595575, -0.036984254933352265, 0.17300996934615673, 0.212143585319997, 
0.24791225521634955], [-0.17594293249479512, -0.24301402234979039, 0.079683803095602343, -0.15106155984065833, 
-0.067984443293405736], [-0.077892265520265, 0.10640944373424864, 0.2754626984034349, 0.1795330714279568, 0.22259724166836004], 
[-0.049884551257166759, -0.1671891698151797, 0.17581659051768267, -0.23174933305129858, -0.085708538086018304]]], 
[[[-0.037623940600628986, 0.0048710951205034299, -0.022786670138392459, -0.027940851726242288, -0.032651845464334292], 
[0.023172962697420467, 0.032006712602840055, -0.010494935889383425, 0.019895905118583156, 0.008954045190133415], 
[0.010258977372518603, -0.014014896963138611, -0.03628043902713976, -0.023645810082632567, -0.029317674228728632], 
[0.0065701578862457661, 0.022020028543371394, -0.023156322540971833, 0.030523071167457567, 0.011288437266307919]], 
[[-0.89221397072244801, 0.11551312940246483, -0.54036300076919341, -0.66258924147942755, -0.77430572736611292], 
[0.54952354090530975, 0.75900704937522523, -0.24887686597580297, 0.47181141112771668, 0.21233619035066237], 
[0.24328134668949455, -0.33234920822033287, -0.86035418000359276, -0.5607366418291071, -0.69523920456975896], 
[0.15580469675371383, 0.52218286517746071, -0.54912893630458293, 0.72382398255503055, 0.26769395432373894]], 
[[-0.19626857160067615, 0.025410493057610475, -0.11886865459072363, -0.14575589292537269, -0.17033120314100678], 
[0.12088378345734559, 0.16696581123370577, -0.054747749522428314, 0.10378872643291426, 0.046709558633685769], 
[0.053516851314458616, -0.07310993383930349, -0.18926007832319147, -0.12335043313303483, -0.15293820773867392], 
[0.034273802343362636, 0.1148694017644013, -0.1207969786281531, 0.15922626612913929, 0.058887118746542393]], 
[[-0.91010998312638869, 0.11783008975551852, -0.55120159249911871, -0.67587944503298991, -0.78983673826270973], 
[0.56054587459095662, 0.7742311996530995, -0.25386883385201814, 0.4812749962574423, 0.21659522598674869], [0.24816107973657656, 
-0.33901546289457307, -0.87761114927595163, -0.5719838876992791, -0.70918430051867321], [0.15892982466830918, 
0.53265680006190796, -0.56014335501796864, 0.73834243148672096, 0.27306335503280704]], [[-0.59320592310825249, 
0.076801165199001889, -0.35927091841576991, -0.44053542707377324, -0.51481231951377848], [0.36536148283857861, 
0.50464069398703182, -0.16547065599289046, 0.31369305217715598, 0.14117587253678851], [0.16175036546570121, 
-0.22096887666643555, -0.57202331760823899, -0.37281673247899977, -0.46224339414221932], [0.10358980244116768, 
0.34718349939990928, -0.36509912224558394, 0.48124854331945899, 0.17798156551676669]]], [[[-0.52724167193027804, 
0.068260907668526147, -0.31932014216737659, -0.39154807136426589, -0.45756540435152815], [0.32473343836721025, 
0.44852485933999098, -0.14707038807697595, 0.27881051564047471, 0.12547717440597603], [0.14376379230439179, 
-0.19639722976418494, -0.50841456332497748, -0.33135966735777267, -0.41084212155075789], [0.092070659624957196, 
0.3085768391034906, -0.32450025216270584, 0.42773390606799311, 0.15819008968105655]], [[-0.52033358167902588, 
0.067366531264096322, -0.31513630678680254, -0.38641788238511066, -0.45157023500624721], [0.32047867623578152, 
0.44264814212811515, -0.14514342446956066, 0.27515745043796308, 0.12383313204093471], [0.14188015278010102, 
-0.19382396998496493, -0.5017531519163505, -0.32701808623928497, -0.40545913570994241], [0.090864320179418007, 
0.30453376594091452, -0.32024854531205599, 0.42212959862427296, 0.15611743216070234]], [[0.35751941310706842, 
-0.046287311771961943, 0.21652907176890027, 0.26550639702824652, 0.31027235427530775], [-0.22019979543012708, 
-0.30414201496646753, 0.099727547403844888, -0.1890597371693398, -0.08508531882110057], [-0.097485364657423176, 
0.13317578268060487, 0.34475286376655995, 0.2246930015364384, 0.27858957665226741], [-0.062432561661871597, 
-0.20924410244511774, 0.22004167326449123, -0.29004379434493044, -0.10726775031850058]], [[0.66105431682344984, 
-0.08558535883433227, 0.40036281209641578, 0.49092201280605952, 0.57369438320050714], [-0.40714998961226517, 
-0.56235937000924408, 0.184396492331967, -0.34957194158257021, -0.15732297392238034], [-0.18025069065158317, 
0.24624236561663487, 0.63744893416985016, 0.41545793929013364, 0.51511284567041515], [-0.11543796751693221, 
-0.3868928850299081, 0.40685762132023134, -0.53629172372284362, -0.198338346966208]], [[0.54244713195956651, 
-0.070229527674066955, 0.32852922012302582, 0.40284017679834028, 0.47076142590494674], [-0.33409863383665905, 
-0.46146015482956748, 0.15131184512874732, -0.28685131054918439, -0.12909589094830282], [-0.14790988832434457, 
0.20206125517421972, 0.52307705631921186, 0.34091596088609083, 0.42269066044700476], [-0.094725944306215559, 
-0.31747608406600336, 0.33385872262595589, -0.44006959794924561, -0.16275223492437288]]], [[[-0.45052965299017578, 
0.058329158490268838, -0.27286005735613805, -0.33457904811447226, -0.39099106504249176], [0.27748573584151887, 
0.38326589113509146, -0.12567210528497938, 0.23824445515021037, 0.10722063685965205], [0.12284660888491843, 
-0.16782204534392522, -0.4344418299702304, -0.28314786917939033, -0.35106587417181162], [0.078674665793287121, 
0.26367987138255872, -0.27728647750252444, 0.36549995672274699, 0.13517392498504135]], [[0.8641690155109093, 
-0.11188220605140628, 0.52337777452097212, 0.64176207870154434, 0.7499669811050147], [-0.53225037146603649, 
-0.73514918634744242, 0.24105392126913372, -0.45698096649496178, -0.20566182843347428], [-0.23563428589959781, 
0.32190247799083155, 0.83331067336053566, 0.5431110110705657, 0.67338575574102488], [-0.15090728886100044, 
-0.50576909499823253, 0.53186817046883617, -0.70107202860909956, -0.25927953221675271]], [[-0.42739392164574591, 
0.055333822375496042, -0.25884806738007538, -0.31739764635928014, -0.37091277679030238], [0.26323620666240222, 
0.36358430828713995, -0.11921855434540207, 0.22601005576701255, 0.10171461115745999], [0.11653815366810343, 
-0.15920399827647846, -0.41213224524863851, -0.26860757646257555, -0.33303783607236698], [0.074634541199210758, 
0.25013930501860904, -0.26304718069627936, 0.34673069536777329, 0.1282324338035625]], [[0.15827975023447735, 
-0.020492157565993367, 0.095860997030169856, 0.11754406800478155, 0.13736269679054261], [-0.097486087033617511, 
-0.13464869430820681, 0.044151032687700202, -0.083699868822500634, -0.037668676211413302], [-0.043158381346058511, 
0.058959118993783179, 0.15262778794406415, 0.099475303602525905, 0.12333620775228858], [-0.027639926404165023, 
-0.092635820766276147, 0.097416083738786963, -0.1284071791431011, -0.047489205078611929]], [[-0.79858337112724087, 
0.10339096597308713, -0.48365629876572974, -0.5930558896144954, -0.69304863892959345], [0.49185551471986083, 
0.67935543276531363, -0.22275926307857213, 0.42229864090738661, 0.19005323416449521], [0.21775094803139861, 
-0.29747186190903824, -0.77006700631951408, -0.50189189190107142, -0.62227950463001047], [0.13945426103369557, 
0.46738402054009243, -0.4915023206625479, 0.64786454265392535, 0.23960165104916858]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.0024858066594131167, -0.006570686891811001, -0.00032680401351835211, 
0.0046881451590620616, 0.0062673626098769652], [0.00094893474003095494, 0.0033899290598499486, 0.0011579979032411833, 
-0.0049119516682552726, 0.005254921002394713], [0.0044562184309675963, 0.0067535344370864003, -0.0026413362614611754, 
-0.0056563397738016398, 0.0055351012386495686], [-0.0076143947168940684, -0.0057753222940789715, 0.0059977405784984563, 
-0.0077239644324305478, -0.0011891582281877572]], [[-0.17985530380532599, 0.47540820709497578, 0.023645215895438974, 
-0.33920086611468275, -0.45346181771179728], [-0.068658173922500385, -0.24527117530603174, -0.083784498647431108, 
0.35539391457729247, -0.38020873818956924], [-0.32242029632095093, -0.48863775601465287, 0.19110832049556178, 
0.40925254769549391, -0.40048058890693849], [0.55092348792081036, 0.41786127728070438, -0.43395388366448895, 
0.55885117384966254, 0.086039038311132021]], [[0.21733993760992945, -0.57449064822191098, -0.028573245485440268, 
0.4098955856113935, 0.54797029103263362], [0.082967601849982447, 0.29638949094448913, 0.10124648716738403, 
-0.42946351643197983, 0.4594501340161421], [0.38961768490529963, 0.59047744024847437, -0.2309382574461161, 
-0.49454712372073711, 0.48394695271946897], [-0.66574448436703759, -0.50495004602196425, 0.52439660107723596, 
-0.67532442295496387, -0.1039709078515705]], [[0.086583416816448558, -0.22886434862893296, -0.011382948071443428, 
0.16329332165314905, 0.21829922578086516], [0.033052454750083567, 0.11807500782722818, 0.040334357762391866, 
-0.17108875184009895, 0.18303475604787026], [0.15521505519045511, 0.23523313244665031, -0.092000686220901254, 
-0.19701661930785458, 0.19279374598681934], [-0.26521785557266747, -0.20116091311607698, 0.20890798988673712, 
-0.26903429089953818, -0.041419706613948233]], [[0.23477980041704788, -0.62058911589950971, -0.030866029254038481, 
0.44278656210157713, 0.59194070347996508], [0.089625115464881056, 0.32017247402797838, 0.10937074111407875, 
-0.4639246692674916, 0.49631748288938304], [0.42088151541292873, 0.63785872535362564, -0.24946928110916541, 
-0.53423073679337951, 0.52277998338177722], [-0.71916537239910672, -0.54546841380983124, 0.56647540573632216, 
-0.72951402757226513, -0.1123137756594984]]], [[[-0.13171506359690599, 0.34815999810482923, 0.017316315112960768, 
-0.24841004244600989, -0.33208779999797411], [-0.050281062350198948, -0.17962166124880874, -0.061358660736111556, 
0.26026884428812302, -0.27844170881356856], [-0.23612097578625726, -0.35784851348605901, 0.13995608722896838, 
0.29971162488070879, -0.29328757685288637], [0.40346278765880889, 0.30601613378766818, -0.31780137797248487, 
0.40926854169665083, 0.063009748187541612]], [[-0.0076681852954721581, 0.020269172750881768, 0.0010081209338920836, 
-0.014461931556755162, -0.01933348179934358], [-0.0029272620186744634, -0.010457210769389562, -0.0035721774500025211, 
0.015152327077391314, -0.016210314590171915], [-0.013746486886403836, -0.020833218571856847, 0.0081479610668175988, 
0.01744860619606689, -0.017074611079038286], [0.02348878959709327, 0.017815641986623613, -0.018501755128838906, 
0.023826788885301896, 0.0036683004307068476]], [[0.26683328992600897, -0.70531551348791133, -0.035080037201571296, 
0.50323833179309485, 0.67275585493344758], [0.10186125202002735, 0.36388426277251268, 0.124302664118784, -0.52726233499895125, 
0.56407760195690759], [0.47834268206712943, 0.72494286941117869, -0.28352826306006779, -0.60716699147437525, 
0.59415291530801939], [-0.81734996783046487, -0.61993890082986158, 0.64381388818924123, -0.82911148096429732, 
-0.12764749867748651]], [[-0.24447787025564294, 0.64622384502175312, 0.032141015035668977, -0.4610767855161324, 
-0.61639205011395315], [-0.093327267981948472, -0.33339786653620723, -0.11388852792426898, 0.48308804633149455, 
-0.51681891275101022], [-0.43826690513980371, -0.66420681175053142, 0.2597741306170156, 0.55629825276427258, 
-0.54437450207567717], [0.7488719996073957, 0.56800012549234657, -0.58987485507957327, 0.75964812758880484, 
0.1169531306561688]], [[-0.069875135040217251, 0.18469965559617424, 0.0091863437929927926, -0.13178208161810259, 
-0.17617331865003313], [-0.026674215733121849, -0.095289691954440175, -0.032550906386408056, 0.13807320244744634, 
-0.14771394761437126], [-0.12526270434325748, -0.18983943461700412, 0.074247016459355913, 0.15899768553241164, 
-0.15558971372424651], [0.21403790881230689, 0.1623422415703164, -0.16859433989838757, 0.21711787427962814, 
0.033426811962287785]]], [[[0.16257875028998278, -0.42974141185606607, -0.021373902071735663, 0.30661788528569289, 
0.4099031502991175], [0.062063002187549891, 0.22171089937148183, 0.075736321340381454, -0.32125546075183409, 
0.34368662027947899], [0.29144922465407924, 0.44170015582832806, -0.17275082390573276, -0.36994053747418409, 
0.36201119612441662], [-0.49800284048621646, -0.37772232910298609, 0.39226911076262605, -0.50516900819845334, 
-0.07777429427333668]], [[0.14097813786038635, -0.37264491144691997, -0.018534112899188694, 0.26587987929028828, 
0.35544241008867072], [0.053817159147919506, 0.19225384425081818, 0.065673807505092413, -0.2785726704965607, 
0.29802357102701887], [0.25272656419919048, 0.38301478729694055, -0.14979872476961881, -0.32078932824343626, 
0.3139134986780367], [-0.43183695886273288, -0.32753721190651819, 0.3401512723946189, -0.43805101191618795, 
-0.067441010344242297]], [[-0.28534782702307609, 0.75425464789362928, 0.037514106242638422, -0.53815610672752301, 
-0.71943579969179583], [-0.10892901305464732, -0.389132794108347, -0.13292754854279895, 0.56384704324105717, 
-0.60321678016796132], [-0.51153304348979234, -0.77524387065689793, 0.303201200218502, 0.64929597692024721, 
-0.6353789039950053], [0.87406274270529394, 0.66295408000976164, -0.68848566103959297, 0.88664034206020614, 
0.13650446832421795]], [[0.291951852370467, -0.77171094628231163, -0.038382324203488929, 0.55061106952418859, 
0.73608625856002063], [0.11145004141075598, 0.39813879517940037, 0.13600399355764808, -0.57689659124169512, 
0.61717749242486097], [0.52337184815313198, 0.7931859388535959, -0.31021841998320832, -0.66432313564885903, 
0.65008396914639033], [-0.89429185244920306, -0.67829733877649689, 0.70441981692315236, -0.90716054490903109, 
-0.13966369675868698]], [[0.084505070841955285, -0.22337069504990711, -0.011109712096564969, 0.15937363321628681, 
0.21305917712237346], [0.032259064528228666, 0.11524073856148397, 0.03936617293934485, -0.16698194210975811, 
0.17864119475851972], [0.15148927724131614, 0.22958660275466128, -0.089792304259454975, -0.19228743775436147, 
0.18816593016928665], [-0.25885157340501252, -0.19633225204710464, 0.20389336819837126, -0.26257639912244346, 
-0.040425469106686471]]], [[[0.28934584485245968, -0.76482253467809846, -0.038039717624285034, 0.5456962297140362, 
0.72951583844405177], [0.11045522105445246, 0.39458494653955228, 0.13479000081599388, -0.57174712278762629, 
0.61166847039898531], [0.51870014985792889, 0.78610583813477175, -0.30744936224941888, -0.65839328429895105, 
0.64428121880527645], [-0.88630926466339899, -0.67224275152197621, 0.69813205637691789, -0.89906308917830768, 
-0.13841703693861504]], [[0.17827063081678607, -0.47121947021386029, -0.023436882117424541, 0.33621222781017612, 
0.44946644655143447], [0.068053239003398824, 0.24311013474647392, 0.083046288380311861, -0.35226260221226102, 
0.37685878684163454], [0.3195795086225805, 0.48433245593868007, -0.18942449918494678, -0.40564669652456509, 
0.3969520258996318], [-0.54606939937524668, -0.4141795761295376, 0.43013039343003112, -0.55392723587799197, 
-0.085280963697334816]], [[0.10578432516208076, -0.27961775549577927, -0.013907252963293747, 0.19950556896099975, 
0.26670969028155211], [0.040382231947487329, 0.14425955316581912, 0.04927898405517752, -0.20902972903685796, 
0.22362490256970002], [0.1896358502821083, 0.28739889330521284, -0.11240293885557905, -0.24070741125148895, 
0.23554806525966154], [-0.32403308744635906, -0.24577075178360933, 0.25523583549473666, -0.32869586295721237, 
-0.050605021996954824]], [[0.085416022114674459, -0.22577859574647524, -0.011229473032483679, 0.16109165572747766, 
0.21535592128968048], [0.032606812131963356, 0.11648301546174263, 0.039790534045535127, -0.16878198098518241, 
0.1805669185299334], [0.15312230764447832, 0.23206150995129307, -0.090760251070582054, -0.19436026586284086, 
0.19019432907910686], [-0.26164195234782961, -0.19844868261270496, 0.20609130639792239, -0.26540693109624608, 
-0.040861249257701064]], [[0.014728466557188732, -0.038931483982203564, -0.0019363219442804526, 0.027777377186202365, 
0.037134280034128421], [0.0056224620408733767, 0.020085414366469677, 0.0068611665056883967, -0.02910343634428331, 
0.031135538217084084], [0.026403205528273018, 0.040014860255838856, -0.015649983334747986, -0.033513954465872541, 
0.032795613116327921], [-0.045115478919622164, -0.034218928870924807, 0.035536762762540926, -0.045764682221435149, 
-0.0070457922094408773]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_expandedData_rank2(self):
      arg0=Data(-0.130038950848,self.functionspace)
      arg0.setTaggedValue(1,-0.906693275297)
      arg1=Data(numpy.array([[0.022374821877257034, -0.60605242550184912, -0.48742171694874914, -0.28868359562505197, 
-0.82271094521870869], [0.068395405208333759, 0.61904244594026259, 0.48734824780679675, -0.72988810654146219, 
0.78182100473729133], [-0.29462200663292948, 0.59832259824002976, 0.1603893108966532, 0.0041925770183708355, 
0.076013192166932031], [-0.33152309541417058, 0.65472666270098845, 0.877056979949419, -0.96727395798858073, 
-0.49534600574983867]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.45301118255465966, -0.47998866642196614, -0.36359892622993595, 0.33278305895960836, 
-0.017084259481602304], [0.20096504329829723, -0.76864722291314647, -0.86861941419218214, -0.61672802905076818, 
-0.67786284713486222], [0.93111566437119175, 0.75434564305343543, 0.084162839182626126, -0.08909161940491428, 
0.29323871890935416], [-0.3221783813743091, -0.097239913766794528, 0.70499159583568871, -0.98228268329435275, 
-0.73084577485551194]]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[-0.0029095983623184472, 0.078810421570849942, 0.063383808692307947, 0.037540111901968952, 
0.1069844681670052], [-0.0088940667360821301, -0.080499630200148956, -0.063374254842168784, 0.094913883610728264, 
-0.10166718320658451], [0.038312336639134657, -0.077805242943490463, -0.020856857716156089, -0.00054519831681633043, 
-0.0098846757599581024], [0.043110915509378116, -0.085139968309528927, -0.11405156950610917, 0.12578329067895472, 
0.064414274894214307]])+(1.-msk_ref)*numpy.array([[-0.41074219285654767, 0.43520249606346156, 0.32967270131780951, 
-0.30173216169136563, 0.015490183185393903], [-0.18221365332829045, 0.69692726809088779, 0.78757138164027185, 
0.55918315662736295, 0.61461368507070246], [-0.84423631140884448, -0.68396012180596799, -0.076309880316770662, 
0.080778772199735061, -0.26587757449175142], [0.29211697183808444, 0.08816677590279047, -0.63921113908495775, 
0.89062910338345724, 0.66265294934055041]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_expandedData_rank2(self):
      arg0=Data(numpy.array([-0.77128426750148482, -0.89459264959940055]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.021742976732915809, -0.65865783162823477]))
      arg1=Data(numpy.array([[0.28339678091606735, 0.092845846770463814, -0.87290568140841818, -0.58754553991083402, 
0.0030464068944782152], [0.32657399996738001, -0.74315703551129864, 0.2874382536626825, 0.72458979477562324, 
0.19175986244274812], [-0.029187139254410743, -0.79583224252594587, 0.54395557110574244, -0.019955831921485645, 
0.62904608533069539], [0.57938222950911311, -0.61683981360204743, 0.026245201381084771, 0.60227097362972781, 
-0.52465384196776044]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.62226490145678115, -0.79117336830835949, 0.42172217172812987, -0.86938681872649259, 
-0.509401849789477], [0.90154542318208164, 0.44345627049739278, -0.99040486628991831, 0.3543428663184629, 
-0.29945566205118723], [0.69305791716194798, 0.074065627831958025, 0.0090008501177756273, -0.95932073791795847, 
0.99156178998935562], [-0.40280396103883365, -0.69740882620881295, 0.2917419650930011, 0.48569642586191408, 
0.39522665028354043]]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-0.21857947858112778, -0.071610540916912283, 0.67325841908297634, 0.45316463137389201, 
-0.0023496457101191033], [-0.25188138834987062, 0.57318532977290693, -0.22169660292812807, -0.5588647091025678, 
-0.14790136504034046], [0.022511581320302024, 0.61381288823068814, -0.41954437421364438, 0.015391619205945804, 
-0.48517334914896193], [-0.44686839849031346, 0.4757588437998076, -0.020242510922638925, -0.4645221267334107, 
0.40465725419394388]], [[-0.25352467712764554, -0.083059212066689175, 0.78089500638152698, 0.52561392130914331, 
-0.002725293215489148], [-0.29215069992109305, 0.66482282146648841, -0.25714014894032372, -0.64821270438101064, 
-0.17154696342947462], [0.02611060023982998, 0.71194567447791868, -0.48661865561984124, 0.017852340553602141, 
-0.56274000419611736], [-0.51831108382736546, 0.55182036322865591, -0.02347876424277447, -0.53878718607622889, 
0.46935147060844395]]])+(1.-msk_ref)*numpy.array([[[0.013529891274084941, -0.017202464138831291, 0.0091694953676394533, 
-0.018903057371473821, -0.011075912567676873], [0.019602281159914737, 0.0096420593714904301, -0.021534349963908288, 
0.0077044686978370355, -0.0065110574925188631], [0.015069142167415327, 0.0016104072226590649, 0.00019570527468725798, 
-0.020858488483953795, 0.021559504928986912], [-0.0087581571527936865, -0.015163743881588344, 0.0063433387590322585, 
0.010560486086775966, 0.0085934038613432722]], [[-0.40985965069188068, 0.52111253521199086, -0.27777061118000007, 
0.5726284368685608, 0.3355215178097487], [-0.59380995354746924, -0.29208594554775669, 0.65233792166456939, 
-0.23339070398225223, 0.19723881703543245], [-0.45648802491066942, -0.048783905825981307, -0.0059284804213848366, 
0.63186411707304058, -0.65309993851980008], [0.26530998354910212, 0.45935378522908915, -0.19215813012311625, 
-0.31990775468779203, -0.26031912847744737]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[-0.90906177782164099, -0.5426776042724335, -0.13367714682544229, 0.59442066266604376, 
-0.55645786502537664], [-0.71570104348454677, 0.34319663780258014, -0.41696518261653681, -0.69745673309431622, 
0.25816809713530287], [-0.15946958516735199, 0.65045105055393115, -0.26476696746193507, -0.014940695246608016, 
0.85399535455069531], [0.76799850213596965, 0.25006868841222651, -0.21277995670364014, -0.14837370367805591, 
0.61269671411955295]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.19342040355078871, -0.081587606517716793, 0.51170193817260023, 
0.18473469854397639, 0.50707495832284288], [-0.17542922349159151, 0.19444584173195767, 0.82765017451859402, 
-0.83107296010709142, -0.96137479629248412], [0.68847824466732965, -0.48929319493369539, -0.28508611231283942, 
0.8042768226800785, 0.45905940915577115], [-0.84412592352281535, 0.4122546237289757, 0.19428975991426212, 0.30138868818248299, 
0.071745200749156579]]))
      arg1=Data(numpy.array([[0.56586052000967468, 0.35839047020323167, 0.68998092288277912, 0.034606151906608762, 
-0.55269610344951436], [-0.86346170278906564, -0.99941754318371889, 0.53078950198553243, 0.95896171993778179, 
-0.39627813271875079], [0.57338062780332177, -0.25784239778380202, -0.71303635506900997, -0.94026141585758483, 
-0.79061514575599423], [-0.62061859271996522, -0.8929642853092663, 0.92514148028855336, -0.50647231874686716, 
-0.21412950833823863]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.63306187944472558, -0.53558755943315561, -0.30410319751286607, 
-0.033764008257253675, -0.33515077053523901], [0.12440285337869894, 0.64917898207685254, 0.67983214832133632, 
0.43004050848670761, 0.98116831205090405], [0.37263380068740304, -0.63287182787835805, -0.42079428547821585, 
0.92342936849098978, -0.83361889634780395], [0.10838603513958289, 0.52530896437113483, 0.46261653669324843, 
-0.70505825439479963, -0.34878706002835402]]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-0.51440217031907309, -0.32579907799728364, -0.6272352844188358, -0.03145912997578753, 
0.50243490239690913], [0.7849400306183294, 0.90853228859272817, -0.48252044832403157, -0.87175544598953847, 
0.36024130384114778], [-0.52123841287937633, 0.23439466852713781, 0.64819409659049709, 0.85475571431658937, 
0.71871800997365998], [0.56418064124717648, 0.81175970073447268, -0.84101075880765697, 0.46041462649747589, 
0.19465695153403312]], [[-0.3070798313512037, -0.19449048176396072, -0.37443719422370925, -0.018779983609766349, 
0.29993579731069153], [0.46858132825056614, 0.54236151800278187, -0.28804757531046682, -0.5204070487648077, 
0.21505126768936511], [-0.31116082543253054, 0.1399252947091735, 0.38694886092799857, 0.51025881254740046, 
0.42904913320036375], [0.33679581106419987, 0.48459171905247844, -0.50205356213604491, 0.27485118456785418, 
0.11620328858902941]], [[-0.075642619816054413, -0.047908615506196704, -0.092234681134955449, -0.0046260516494832996, 
0.073882838170670581], [0.11542509682188035, 0.1335992856600928, -0.070954426190323411, -0.12819126663610153, 
0.052973330131156557], [-0.076647886369728926, 0.034467636066369399, 0.095316665528438246, 0.12569146334189263, 
0.10568717692164251], [0.08296252274162616, 0.11936891787716293, -0.12367027349483997, 0.067703774516147175, 
0.028624221725790495]], [[0.33635918528070297, 0.21303470079139999, 0.41013891740691, 0.020570611748648153, 
-0.32853398406540063], [-0.51325947755862689, -0.59407443829933559, 0.31551224750641954, 0.57002666103678534, 
-0.23555591025074227], [0.3408292927387227, -0.15326684895404924, -0.42384354268510138, -0.55891081389337816, 
-0.46995797885408885], [-0.36890851514746925, -0.53079642221064427, 0.54992321177296655, -0.30105761133152054, 
-0.12728300424276995]], [[-0.31487753686673298, -0.19942919589473115, -0.38394531125559034, -0.01925686540669538, 
0.30755209373336145], [0.48048005566517976, 0.55613375234891937, -0.29536199305275229, -0.5336217913176412, 
0.22051208368891892], [-0.31906215999434656, 0.14347843018379838, 0.39677468782717767, 0.52321586003384946, 
0.43994401606410749], [0.34534809710000558, 0.49689699974710561, -0.51480225296778492, 0.28183050518433372, 
0.11915404904882985]]], [[[-0.40498696463763245, -0.2565004334993703, -0.49382006649163562, -0.02476765903054463, 
0.39556517796866042], [0.61798044169507782, 0.71528417853334969, -0.37988660044168848, -0.68632990362120616, 
0.28361667309691763], [-0.41036911363266193, 0.18453807314842469, 0.51032086336530824, 0.6729460764775308, 
0.56584408481225212], [0.44417737441559008, 0.6390954707902744, -0.66212472281335588, 0.36248276702317078, 
0.15325271255851033]], [[0.19420142793253997, 0.12299840439423489, 0.23679913288129112, 0.011876714981633476, 
-0.18968344443046034], [-0.29633715326849808, -0.34299674058156726, 0.18216517246234065, 0.32911243806402618, 
-0.1360013227837599], [0.19678230364323263, -0.088490644002356289, -0.24471167969069094, -0.3226945565778167, 
-0.27133645981925408], [-0.21299421437926092, -0.30646234039592413, 0.31750544552673349, -0.17381959693400148, 
-0.073488527316003041]], [[-0.23594413506132247, -0.14943634785631699, -0.28769802151174462, -0.014429560449394736, 
0.23045503170627507], [0.36003346658582858, 0.41672231840376989, -0.22132074162633816, -0.39985364867612544, 
0.16523418397601414], [-0.23907975818079658, 0.10751130247820874, 0.29731133400357951, 0.39205627297034129, 
0.32965898862954801], [0.25877634484869838, 0.37233501629402349, -0.38575178627464984, 0.21118132287650832, 
0.089284549547842915]], [[-0.39466322967299866, -0.24996184652008183, -0.48123184037122446, -0.024136293653748989, 
0.38548161870585651], [0.60222717837931716, 0.69705049466606428, -0.37020271201558852, -0.66883430835031188, 
0.27638685184273576], [-0.39990817948727286, 0.17983391641149571, 0.49731200678391058, 0.65579165535866746, 
0.5514198566938624], [0.43285461617605892, 0.62280395320170179, -0.64524615449209421, 0.35324252883589319, 
0.14934606734468006]], [[0.14608713369489068, 0.092524985723794778, 0.17813106192030723, 0.0089342043869044176, 
-0.14268850132165761], [-0.22291826475826151, -0.2580177253673801, 0.13703291570699996, 0.24757332246193436, 
-0.10230637146033089], [0.14802858561422891, -0.066566681196648006, -0.18408323897645848, -0.24274550054169836, 
-0.20411160774617515], [-0.16022392112930295, -0.23053489034807897, 0.23884201554703313, -0.13075499478258326, 
-0.055281407708201037]]], [[[-0.090237542388524897, -0.057152379611241581, -0.11003097154550347, -0.0055186286887852661, 
0.088138218340705912], [0.13769587955166768, 0.15937670102048174, -0.084644781692818208, -0.15292522766984845, 
0.063194309435552043], [-0.09143677085879158, 0.041118020213138269, 0.11370761175209572, 0.14994309793567609, 
0.12607906932073393], [0.098969789528198626, 0.14240064414752965, -0.14753192808272556, 0.080766930569309781, 
0.034147143866787952]], [[0.36806456970728668, 0.23311545785220941, 0.44879881615127465, 0.022509607863282596, 
-0.35950176112580084], [-0.561639571692234, -0.65007219090587876, 0.34525258918948748, 0.62375765817453488, 
-0.25775952773846161], [0.37295603172194325, -0.16771385851581863, -0.4637952462377834, -0.61159402573989274, 
-0.51425645214083582], [-0.40368201562800371, -0.58082955748655263, 0.60175924776470857, -0.32943545180538525, 
-0.13928076365320408]], [[-0.1498211738893952, -0.094889957962966648, -0.1826841565582607, -0.0091625658958398633, 
0.14633567123835586], [0.2286161365669796, 0.2646127521370108, -0.14053552680134018, -0.25390138650000793, 
0.10492135947142188], [-0.151812250024906, 0.068268149744331222, 0.18878847342173335, 0.24895016369807815, 
0.20932877457129037], [0.16431930274495896, 0.23642744587314862, -0.24494690420924584, 0.13409713993802258, 
0.056694420566830578]], [[-0.0084543495815516869, -0.0053546027945950347, -0.01030879469476495, -0.00051703996929446442, 
0.0082576640456269307], [0.012900718158488656, 0.01493199293682165, -0.0079303641892646808, -0.014327554810753464, 
0.0059206708138457402], [-0.008566705220318209, 0.003852344686942464, 0.010653258880838262, 0.01404815926647234, 
0.011812339950092886], [0.0092724732582077397, 0.013341507252910879, -0.013822256916987092, 0.0075670485652398582, 
0.0031992437273876333]], [[0.48324225541190297, 0.30606379666879924, 0.5892405028704949, 0.029553492967119569, 
-0.47199990482415577], [-0.73739228301429516, -0.85349793913536487, 0.45329176893992174, 0.81894885401881057, 
-0.33841968445183707], [0.48966439253339805, -0.22019620991357941, -0.60892973485469459, -0.80297888120563687, 
-0.67518166171303995], [-0.53000539513064027, -0.76258735143379508, 0.79006652646857822, -0.4325250074183436, 
-0.18286560539308017]]], [[[0.43458003178531102, 0.27524334429588776, 0.52990431527636839, 0.026577472828965359, 
-0.42446977958561394], [-0.66313729439377622, -0.76755117617350688, 0.40764554247438617, 0.73648116451794965, 
-0.30434101235723959], [0.44035546330673303, -0.19802257528510681, -0.54761085266149112, -0.72211935899487123, 
-0.60719124770661492], [-0.47663414960666667, -0.68579523357843319, 0.71050727112546264, -0.38896998217092538, 
-0.16445114166687888]], [[0.14150399806307981, 0.089622234823163296, 0.17254262441475418, 0.0086539150182799243, 
-0.13821198968016832], [-0.21592473551064939, -0.24992303420012232, 0.132733834584501, 0.23980629954237398, 
-0.099096752895424231], [0.1433845415557557, -0.064478310230858954, -0.17830806610234196, -0.23512993902812931, 
-0.19770809253804278], [-0.15519727748572348, -0.22330240762624945, 0.23134891657150425, -0.12665286846612819, 
-0.053547085300498252]], [[-0.12040377694795787, -0.076258308732840868, -0.1468141108973354, -0.0073634955043678056, 
0.11760265296225828], [0.18372734373470878, 0.21265602156749011, -0.1129413672512283, -0.20404783324880948, 
0.084320043922495147], [-0.1220039051586968, 0.054863694236800153, 0.15171984475970532, 0.20006878335628028, 
0.1682270564832026], [0.13205519728842827, 0.19000490196600264, -0.19685156412053992, 0.10776715805455062, 
0.045562467513182166]], [[-0.083958821119026097, -0.053175721426973424, -0.10237502499532102, -0.0051346429284289579, 
0.082005567877234384], [0.12811501082697441, 0.14828728240299174, -0.078755204283024266, -0.14228470207264729, 
0.058797254238105244], [-0.085074607364427735, 0.03825703152441326, 0.10579584485869034, 0.13951006869636259, 
0.11730649735978287], [0.092083479173324184, 0.13249241826356406, -0.13726666785661182, 0.075147173742885556, 
0.031771188218905622]], [[0.34670088125990922, 0.2195846634652816, 0.42274904425545545, 0.021203075561501292, 
-0.33863508649019797], [-0.52904014806693456, -0.61233984474210101, 0.32521298375568963, 0.58755269477231387, 
-0.24279830979421071], [0.35130842659490163, -0.15797918988284221, -0.43687503179856524, -0.57609507990934072, 
-0.48440730193784909], [-0.3802509724810238, -0.54711628343510244, 0.56683114506849586, -0.31031392548871639, 
-0.13119644615487422]]]])+(1.-msk_ref)*numpy.array([[[[-0.12244708419481957, 0.10359356188234299, 0.058819763184023759, 
0.0065306481026101678, 0.06482499728728372], [-0.024062050103377548, -0.12556446068999505, -0.13149340847511251, 
-0.083178608694685366, -0.18977797086813206], [-0.072074980105621661, 0.1224103243441573, 0.081390200509062302, 
-0.17861008110417723, 0.16123890333915536], [-0.02096407065596809, -0.10160547187751179, -0.089479477216476366, 
0.13637265209185678, 0.067462533903977398]], [[-0.051650003521502535, 0.043697307054816559, 0.024811052019459229, 
0.0027547246201537535, 0.02734414919053867], [-0.010149731051142504, -0.05296495934925817, -0.055465877815335268, 
-0.035085975793092347, -0.080051174171261522], [-0.030402299905685146, 0.051634497669087669, 0.034331598588500462, 
-0.075340391963346576, 0.068012970500957975], [-0.0088429571869837152, -0.042858701085341462, -0.037743775964317647, 
0.05752401543163118, 0.028456701412064616]], [[0.32393899069505505, -0.27406119222307845, -0.15561019557181863, 
-0.017277108465712385, -0.1714972988629222], [0.063657181188082057, 0.33218614334964114, 0.34787142792807041, 
0.22005256168537882, 0.50206572694998619], [0.19067743804036655, -0.32384174094019208, -0.21532125145115749, 
0.47252059762233972, -0.4265644049584752], [0.055461344251768123, 0.2688016152081511, 0.23672177845663106, 
-0.36077967529840921, -0.17847501462603182]], [[0.11694849545890451, -0.098941606335788038, -0.056178412518798626, 
-0.0062373838870400872, -0.061913976561608783], [0.022981523616924446, 0.11992588355505281, 0.1255885869806459, 
0.079443403696990253, 0.18125583234762593], [0.068838392837283588, -0.11691338634008379, -0.07773530547684615, 
0.1705894460148375, -0.15399833551737388], [0.020022661527887679, 0.097042793175550035, 0.085461326447485636, 
-0.13024872408156554, -0.06443307239037778]], [[0.32100982613521484, -0.27158303937780054, -0.15420311620467983, 
-0.017120883079859029, -0.16994656300102501], [0.063081571692246499, 0.32918240528068554, 0.34472585827657037, 
0.21806277291803142, 0.49752588094090627], [0.18895326895324743, -0.32091345574511981, -0.21337424477135675, 
0.46824790854165776, -0.42270726712269696], [0.05495984425118218, 0.26637102121510897, 0.23458126106318686, 
-0.35751738496241936, -0.1768611839274245]]], [[[-0.11105755393311573, 0.093957709663115105, 0.053348587800992178, 
0.0059231937505336962, 0.058795239427605547], [-0.021823895968363465, -0.11388496473280404, -0.1192624258846325, 
-0.075441672473752289, -0.17212559509764566], [-0.065370858301311591, 0.11102421333440451, 0.073819614751142487, 
-0.16199649706370506, 0.14624111567421275], [-0.019014077981869378, -0.092154543712800296, -0.081156459806465922, 
0.12368782208481668, 0.061187443104689265]], [[0.12309625001704477, -0.10414277381514485, -0.059131602213769022, 
-0.0065652710058264601, -0.065168673683838746], [0.024189617539078431, 0.12623015360462905, 0.13219053431678734, 
0.083619588651536944, 0.19078409831746215], [0.072457093032440634, -0.12305929528024996, -0.081821699035809367, 
0.17955700083624063, -0.16209372798401431], [0.021075213834705737, 0.10214414374648828, 0.089953861876441776, 
-0.13709564574586156, -0.067820193472428139]], [[0.52395377500349627, -0.44327913703483907, -0.25169106449318607, 
-0.027944787326563254, -0.27738759372353183], [0.10296204330949124, 0.53729309780971024, 0.56266319620150473, 
0.35592310189908849, 0.81206412470104505], [0.30841043017045611, -0.52379647879142466, -0.3482704638124724, 
0.76427647798716281, -0.68994482504425769], [0.089705720898654251, 0.43477205603795166, 0.38288465732935462, 
-0.58354158729563121, -0.28867367110229453]], [[-0.52612061008108679, 0.44511233841464537, 0.2527319445350491, 
0.028060354287436088, 0.27853474295089364], [-0.10338784760320381, -0.5395150982739183, -0.56499011588137615, 
-0.35739503835400688, -0.81542245345942321], [-0.30968587577323597, 0.5259626633632527, 0.34971075242852934, 
-0.76743717872162898, 0.69279812378897609], [-0.090076703057724389, -0.43657007599070963, -0.38446809454414882, 
0.58595485052782481, 0.28986749442481397]], [[-0.60860973539171015, 0.51490038084683876, 0.2923571495608247, 
0.032459866560335005, 0.32220550375058449], [-0.11959776782515047, -0.62410431165149627, -0.65357349310550661, 
-0.41343010624392479, -0.94327048612657838], [-0.3582407442275462, 0.60842702460580855, 0.40454102048266116, 
-0.88776172102352258, 0.80142019666193542], [-0.10419960245326652, -0.50501879861291554, -0.44474787872500621, 
0.67782523569313491, 0.33531508878421329]]], [[[0.43584933152590533, -0.36874038278419807, -0.20936843562138027, 
-0.023245785137887234, -0.23074401419700433], [0.085648658125773824, 0.44694560605519529, 0.4680496441446933, 
0.29607353441877432, 0.67551303720401312], [0.25655026500097883, -0.43571848515709632, -0.28970771103208526, 
0.63576103069293732, -0.57392847447905271], [0.074621427219351541, 0.36166379369825175, 0.31850142113664698, 
-0.48541726937394319, -0.24013230285099971]], [[-0.3097528695842397, 0.26205934812178916, 0.14879562510062286, 
0.016520499473959324, 0.16398699129967692], [-0.060869469588531663, -0.31763885822418736, -0.33263724387078453, 
-0.2104158943483721, -0.48007897817108786], [-0.18232718287862529, 0.30965987864612954, 0.20589178035147773, 
-0.45182770600456107, 0.40788405315111803], [-0.053032549419642291, -0.25703010150446332, -0.22635512326780063, 
0.34498020590720568, 0.17065913495280394]], [[-0.18047715006435625, 0.15268857512192016, 0.086695598320846529, 
0.0096256498501590595, 0.095546830210543818], [-0.035465525830357461, -0.18507191219549635, -0.1938107041902154, 
-0.1225985767015121, -0.27971745960714312], [-0.10623272155432921, 0.18042296900216157, 0.11996260693044365, 
-0.2632568886585967, 0.23765317031031527], [-0.03089935338694649, -0.1497582904156507, -0.13188554993750823, 
0.20100231669949037, 0.099434346968508405]], [[0.50915699695968275, -0.4307606605678761, -0.24458315346250029, 
-0.02715560928208792, -0.26955399684486209], [0.10005433164775565, 0.52211960905545862, 0.54677324020765627, 
0.3458716137894145, 0.7891309325306769], [0.29970072924006619, -0.50900414288973939, -0.33843509092635332, 0.7426928384594047, 
-0.67046035728068543], [0.087172375964955071, 0.42249382478977887, 0.37207175825090777, -0.5670620126490119, 
-0.28052134843153037]], [[0.29061301233693776, -0.24586650858456588, -0.13960143417263707, -0.01549968568130545, 
-0.15385411470000826], [0.057108300369317563, 0.29801171994854486, 0.31208334433349122, 0.19741414173895538, 
0.45041454561245331], [0.17106105237502864, -0.29052576737717195, -0.19316957606775464, 0.42390894029656062, 
-0.3826805980185089], [0.049755629251913569, 0.24114802280844319, 0.21236847400009173, -0.3236636256828761, 
-0.16011398169779467]]], [[[-0.53438394363336816, 0.45210334323384321, 0.25670139244678919, 0.02850107465198622, 
0.2829094536974418], [-0.10501167349716763, -0.54798880777722436, -0.57386394004224761, -0.36300834137856319, 
-0.82822960754129127], [-0.31454985114107081, 0.53422351617939123, 0.35520336484242215, -0.77949066848554693, 
0.70367932074566009], [-0.091491462009176724, -0.44342691468459788, -0.3905066112731147, 0.59515795012839434, 
0.29442019915924195]], [[0.26098268690764354, -0.22079844778803598, -0.12536794926544498, -0.013919368519676142, 
-0.13816745479948123], [0.051285651510446464, 0.26762703698885232, 0.28026394650507369, 0.17728618801440502, 
0.40449117329933965], [0.15362000729108347, -0.26090433727066159, -0.17347438982712507, 0.38068802684753866, 
-0.3436632444472279], [0.044682644133944287, 0.21656104944828009, 0.19071580626527701, -0.29066352537253654, 
-0.14378907819352474]], [[0.12299744056818727, -0.1040591783353334, -0.059084137233934179, -0.0065600010580449805, 
-0.065116362742371536], [0.024170200515596568, 0.12612882856909674, 0.13208442487934946, 0.083552467147289616, 
0.19063095578385197], [0.07239893167149454, -0.12296051549498641, -0.081756020698856027, 0.1794128703018931, 
-0.16196361523140698], [0.021058296745328338, 0.10206215256847746, 0.089881655846498668, -0.13698559897193435, 
-0.067765754154110236]], [[0.19079768938418304, -0.16142003194441643, -0.091653263770501231, -0.010176090156436208, 
-0.10101065107496406], [0.037493612785963845, 0.19565520180378224, 0.20489371936684675, 0.12960934471813676, 
0.29571303045524311], [0.11230761236162923, -0.1907404099919085, -0.12682263769496471, 0.27831116599867811, 
-0.2512433056143939], [0.032666324948019396, 0.15832217966231502, 0.13942739112550165, -0.21249658238428004, 
-0.10512047447697057]], [[0.045419151627400199, -0.038425836970282581, -0.02181794495402099, -0.0024224055505128453, 
-0.024045459313285233], [0.0089253076894226478, 0.046575476391236909, 0.048774693957044667, 0.03085334261164821, 
0.070394117516803217], [0.026734686836238933, -0.045405516339618465, -0.030189970485732499, 0.066251625420052945, 
-0.059808155066763548], [0.0077761778494945142, 0.037688397104138613, 0.033190516294936669, -0.050584546001404808, 
-0.025023797640442387]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_array_rank3(self):
      arg0=Data(0.739212190281,self.functionspace)
      arg0.setTaggedValue(1,-0.248694802076)
      arg1=numpy.array([[[0.82610915066483726, -0.51851578144622912], [-0.7926999614241439, -0.69595626245951725]], 
[[-0.83087415903136996, 0.93817006045798945], [0.83398112637562294, -0.60638321731104905]], [[0.094132825899504224, 
-0.47259964739342442], [-0.64201721936055844, -0.3583489910168598]], [[-0.77787201937831685, -0.27420465077379674], 
[0.6442447044819084, -0.74779985040375441]], [[-0.99233987243452115, 0.56277024770101192], [0.39329501158454616, 
-0.5319975739670586]], [[0.10746824347752759, -0.4957403909088296], [0.28326380155894837, -0.12698576101650305]]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.61066995467426766, -0.38329318649821714], [-0.58597347472013683, -0.51445935311259339]], 
[[-0.61419230694560034, 0.69350674524736378], [0.61648901508127774, -0.44824586621824053]], [[0.069584132410528132, 
-0.34935142047579976], [-0.47458695492174185, -0.26489594253461862]], [[-0.57501247920307874, -0.20269542048378036], 
[0.47623353907711369, -0.55278276530888715]], [[-0.73354973060565476, 0.41600662742813904], [0.29072846694006871, 
-0.39325909187645575]], [[0.079441835646692743, -0.36645734017455711], [0.20939205517775963, 
-0.093869422535529864]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.2054490517177319, 0.12895217964003217], [0.19714036001199936, 
0.17308070494589517]], [[0.20663408453033705, -0.23331801749919201], [-0.20740677115907252, 0.15080435421135577]], 
[[-0.023410344505928094, 0.11753307576967661], [0.1596663452982329, 0.089119531395058252]], [[0.19345272789971857, 
0.068193271352497378], [-0.16022030926961417, 0.18597393578859492]], [[0.24678976816718759, -0.13995803536624268], 
[-0.097810425063481476, 0.13230503136262903]], [[-0.026726793541094907, 0.1232880583981309], [-0.070446235063986951, 
0.031580698702464507]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6,2,2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_array_rank3(self):
      arg0=Data(numpy.array([-0.57447826335126084, -0.54768328937179001]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.41568641547238916, -0.66053156051837991]))
      arg1=numpy.array([[[-0.23457102430861054, 0.68835792537299167], [0.79787400490407046, -0.61464083168702355]], 
[[0.8824726965101215, -0.49490519712468539], [-0.51425798605194939, 0.8923506085570474]], [[0.19332538091745355, 
0.83145053584653783], [0.38098794460317231, -0.6960132934843597]], [[0.88291458519428412, 0.91798711182658943], 
[-0.80553662838505979, -0.64366564828710504]], [[-0.73109847735851163, -0.57190386021582462], [0.71399613358746916, 
0.75825927955958572]], [[0.71797421733059985, -0.94619680840056475], [-0.33476176918371503, -0.4009056466422698]]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.13475595467733698, -0.39544666553235308], [-0.45836127271040578, 0.35309779757233589]], 
[[-0.50696138214603892, 0.28431227816770266], [0.29543003474164081, -0.5126360279042933]], [[-0.1110612290911797, 
-0.47765025989559429], [-0.21886929277339681, 0.39984450811028638]], [[-0.50721523758991116, -0.52736364178097883], 
[0.46276328334047911, 0.36977192380683954]], [[0.42000018361166885, 0.32854633642066916], [-0.41017525886284412, 
-0.43560347409136901]], [[-0.41246058150306375, 0.5435694992784621], [0.19231335979705624, 0.2303115796507654]]], 
[[[0.12847063018464994, -0.37700213283342121], [-0.436982259510105, 0.3366285124805618]], [[-0.48331554920545672, 
0.27105130628844182], [0.28165050538664377, -0.48872551656744229]], [[-0.10588108053992525, -0.45537156442236926], 
[-0.20866073071126273, 0.38119485002200715]], [[-0.48355756425353502, -0.5027662010060957], [0.44117895034339077, 
0.35252491950950737]], [[0.40041041893441681, 0.31322218736742724], [-0.39104375104192513, -0.41528593642587758]], 
[[-0.39322248103175939, 0.51821618041791062], [0.18334342690245697, 0.21956932328076284]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.097507988268532977, -0.28614103856030926], [-0.33166538509717247, 
0.25549784412694687]], [[-0.36683191196454595, 0.20572536739141664], [0.21377005884998473, -0.37093802581568414]], 
[[-0.080362734613410494, -0.3456226928886445], [-0.15837151303028588, 0.28932327108964545]], [[-0.36701559908770331, 
-0.38159477196504621], [0.33485063358509953, 0.26756306609917824]], [[0.30390770541048134, 0.23773266564793843], 
[-0.29679849343212017, -0.31519808191880044]], [[-0.29845212880375116, 0.39332115961544578], [0.13915591986917381, 
0.1666510311953654]]], [[[0.15494156473896134, -0.45468213464181667], [-0.52702096155633515, 0.40598966771254458]], 
[[-0.58290106734069325, 0.32690050216542488], [0.33968363003593338, -0.58942573999971248]], [[-0.12769751554521583, 
-0.5491993199365568], [-0.2516545615874235, 0.45973874688676125]], [[-0.58319294876281857, -0.60635945951057768], 
[0.53208236620189786, 0.42516147511515617]], [[0.48291361814222911, 0.37776054925484404], [-0.47161698032262067, 
-0.50085418520503566]], [[-0.47424463018534357, 0.62499285441033559], [0.221120713800813, 0.26481083239724867]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_constData_rank3(self):
      arg0=Data(0.775833186917,self.functionspace)
      arg0.setTaggedValue(1,-0.77164208597)
      arg1=Data(numpy.array([[[0.71002297254341307, 0.026421061697135206], [-0.54250319611533526, 0.74255828851969152]], 
[[-0.73848045341790081, -0.03263329644610824], [0.43262024861195192, -0.90189829446902015]], [[-0.32158701787813282, 
0.031457573356408197], [0.51049423421189544, 0.085141710823527506]], [[0.66124830745675767, 0.92383146642207614], 
[0.02474471465440975, -0.74652933215433115]], [[0.90399282306957351, -0.98173917658228493], [-0.11910893201742767, 
-0.45121298288374923]], [[0.20317681750733629, -0.35157062228109059], [0.21849796596450388, 
0.82180955111947895]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.55085938557282255, 0.020498336498225966], [-0.42089198355496005, 0.57610136345405871]], 
[[-0.57293764365131339, -0.025317994381399863], [0.33564114620554814, -0.69972262807314156]], [[-0.24949788095160977, 
0.024405829389785669], [0.39605836863150112, 0.066055764847805148]], [[0.51301838171782066, 0.71673911076868535], 
[0.019197770829688949, -0.57918223089250875]], [[0.70134763287239832, -0.76166583408936106], [-0.092408662317392212, 
-0.35006600648914238]], [[0.15763131783442333, -0.2727601563108229], [0.16951797326918014, 
0.63758712308406851]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.54788360762019483, -0.020387603161526012], [0.41861829789596344, 
-0.57298922670783148]], [[0.56984259752364297, 0.025181224941760422], [-0.33382799107189354, 0.69594268127708303]], 
[[0.24815007729643362, -0.024273987524300895], [-0.39391883576305098, -0.065698927342942345]], [[-0.51024702331022798, 
-0.71286723983488332], [-0.019094063232667286, 0.57605345110154371]], [[-0.69755890769553841, 0.75755126609666734], 
[0.091909464759616269, 0.34817492732927374]], [[-0.15677978328215725, 0.27128668834283864], [-0.16860222623710588, 
-0.63414283629610735]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_constData_rank3(self):
      arg0=Data(numpy.array([0.48352531735516435, 0.55668210958892583]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.35027806854298271, -0.74586440683354338]))
      arg1=Data(numpy.array([[[-0.7642484206888922, -0.29484253145121975], [-0.53803447191211662, -0.21060506070262885]], 
[[-0.73741784575038261, 0.95548775872787473], [0.92546130948738581, -0.39939687200978002]], [[0.68015855316427132, 
-0.78145128893835203], [-0.1407821507604603, -0.89128339174273097]], [[0.055238130232139637, -0.77025380058032122], 
[-0.238675937096678, 0.9003502618777286]], [[0.6659686773774689, 0.012219701973260833], [0.48513991195586503, 
-0.98957210127028672]], [[0.25565193260412822, 0.35159186889487248], [-0.76320440501199038, 
-0.078927460055780108]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.36953346015177974, -0.14256382858975106], [-0.26015328877932448, -0.10183287881284227]], 
[[-0.35656019788981541, 0.46200252176787032], [0.44748397336981421, -0.19311849928918884]], [[0.32887388027058373, 
-0.3778514824815189], [-0.068071734124394156, -0.43095808484579129]], [[0.026709034450601216, -0.37243721336962132], 
[-0.11540585822971247, 0.43534214610523403]], [[0.32201271607753973, 0.0059085352746064725], [0.23457742989011612, 
-0.47848316431253224]], [[0.12361418184487219, 0.17000356998688856], [-0.36902865214028202, -0.038163425171508135]]], 
[[[-0.42544342307909738, -0.16413356240480423], [-0.29951416485560073, -0.11724006948204321]], [[-0.41050732202084411, 
0.5319029412150279], [0.5151877541083677, -0.22233709327362255]], [[0.37863209823043814, -0.43501995206718702], 
[-0.078370904677799239, -0.4961615187569165]], [[0.030750078867375314, -0.42878651062594103], [-0.13286662417109246, 
0.50120888315103573]], [[0.37073284824263614, 0.0068024894730227999], [0.2700687096333767, -0.55087708492548937]], 
[[0.14231685716255199, 0.19572490329071066], [-0.42486223822963576, -0.043937504968347349]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.26769946068593009, -0.10327687244105689], [-0.18846167563091989, 
-0.073770333888294459]], [[-0.25830129871857116, 0.33468640664366345], [0.32416879999850107, -0.13989996490969461]], 
[[0.23824462430537058, -0.27372524814975024], [-0.049312899853701041, -0.31219702498408242]], [[0.019348705567639613, 
-0.26980301355516667], [-0.083602946253910806, 0.31537295074269944]], [[0.23327422202190459, 0.0042802936053646786], 
[0.1699338713330131, -0.34662540431697692]], [[0.089549265171854825, 0.12315492075191353], [-0.26733376489109628, 
-0.027646558263342075]]], [[[0.57002569497059286, 0.21991254983016437], [0.40130076224872963, 0.15708281867710866]], 
[[0.55001372410907856, -0.71266431040027811], [-0.69026865064820331, 0.29789591103274721]], [[-0.50730605580863031, 
0.58285670209331186], [0.1050043953697012, 0.66477655830278071]], [[-0.04120015524018885, 0.57450489408112371], 
[0.17801988624805384, -0.67153921401785743]], [[-0.49672233252186526, -0.00911424076396887], [-0.36184859266223873, 
0.73808660833298556]], [[-0.19068167706762709, -0.26223986074077099], [0.56924700083701563, 0.058869183177382621]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_taggedData_rank3(self):
      arg0=Data(-0.605264779164,self.functionspace)
      arg0.setTaggedValue(1,0.109645622226)
      arg1=Data(numpy.array([[[0.058154633230961927, 0.34706300715958505], [0.77388850219250882, -0.90670996169394003]], 
[[-0.93515939714673313, 0.87481009917288155], [0.63074491141538358, -0.95984242260628694]], [[-0.55689918647700365, 
-0.14908352777073408], [-0.19157238900021945, -0.29700047935331075]], [[0.4107237573042275, 0.29356504540276429], 
[0.96618374230499215, 0.15269892034253241]], [[-0.85542250392337538, 0.83644790322275431], [-0.79069232324877214, 
-0.50034682736849501]], [[-0.77767605509064164, 0.20630916958582413], [0.60779403519359687, 
-0.0039258984857535317]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.60935873042030209, 0.71159974334140075], [-0.88165137276696659, 
0.57938544494805999]], [[0.92633739546061555, 0.53684267472307234], [0.10301065492151484, 0.23676061898921841]], 
[[0.4953182011889834, 0.28156233774434547], [0.89823412730976715, -0.82888539007788142]], [[0.16182445206503848, 
-0.28625251382852346], [-0.81475721745450347, 0.14717523871250626]], [[-0.74229264437310705, -0.70266174304981854], 
[-0.53841555742629077, -0.2864358327579275]], [[-0.88078561972820224, 0.36225749767892745], [-0.55502892082679711, 
0.060873490775430916]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.035198951239930512, -0.21006501438461261], [-0.46840745337749251, 0.54879960473093248]], 
[[0.56601904599762198, -0.52949174148674616], [-0.38176767951696261, 0.58095881195151045]], [[0.33707146311989189, 
0.090235008513217571], [0.11595201972223301, 0.17976392954753151]], [[-0.24859662426235588, -0.17768458237611978], 
[-0.58479698941855873, -0.092423278299780059]], [[0.5177571129295232, -0.50627245542672727], [0.47857821441823145, 
0.30284231197284905]], [[0.47069992574595471, -0.12487167396897468], [-0.36787632248895119, 
0.0023762080800018462]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.066813517155952226, 0.078023796634731593], [-0.096669213353691164, 
0.063527077620182226]], [[0.10156884011675413, 0.058862449107633635], [0.011294667354806479, 0.025959765387752909]], 
[[0.054309472369369609, 0.030872077717465776], [0.09848743979376226, -0.090883654349365114]], [[0.01774334273809855, 
-0.031386334992566309], [-0.089334562071154491, 0.016137120624934571]], [[-0.081389138866282781, -0.077043784031302368], 
[-0.059034908810316984, -0.031406435110646669]], [[-0.096574287323061545, 0.039719948739142488], [-0.060856491377636865, 
0.0066745117731580577]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_taggedData_rank3(self):
      arg0=Data(numpy.array([0.34413038280819586, -0.2103307435848194]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.57264369878268195, 0.25133984049887559]))
      arg1=Data(numpy.array([[[-0.80072481908312798, 0.30972585838775668], [0.42782144039236636, 0.064157260250357506]], 
[[-0.94950197355359167, -0.56343713996190936], [0.11674958272923885, -0.38765169072310646]], [[-0.0051384010835482385, 
0.57561526938661656], [-0.16435826616193117, -0.9206278396339449]], [[0.81080680605349742, 0.052199521735473597], 
[0.094036884043960489, -0.89791174996920708]], [[0.53407650775646021, 0.30962384903028894], [-0.59334298059280854, 
-0.84829097388582397]], [[0.9543639666631889, -0.99524530375103182], [0.53959900666741678, 
0.32618382906487065]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.047375451688465908, -0.83237826291213124], [0.2696364287194819, 
-0.10045452347110118]], [[-0.96137531149564914, 0.088875452300248847], [0.52128245024069164, 0.60350178261658671]], 
[[-0.90017675992292379, -0.1505072553090967], [-0.76675429692688923, 0.39316834270842094]], [[-0.93741425123834432, 
0.71088800864366464], [0.93201208641404087, -0.45873350782226829]], [[-0.41993354856752418, 0.12453231803689691], 
[-0.72714276130766953, 0.25843848699568106]], [[-0.62259226918020127, -0.028593066514580157], [-0.38045129152727597, 
-0.90831977226819594]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.27555373851510018, 0.10658607821257576], [0.14722635605577877, 0.022078462529880577]], 
[[-0.32675247763613496, -0.1938958386634469], [0.040177078597310095, -0.13340272472478698]], [[-0.0017682799319035038, 
0.19808670300425912], [-0.056560673051996718, -0.31681601087711181]], [[0.27902325655068067, 0.017963441397233271], 
[0.03236094890413805, -0.30899871424488029]], [[0.18379195306309504, 0.10655097369334038], [-0.20418734704795913, 
-0.29192269757606587]], [[0.32842563718615148, -0.34249414736790174], [0.18569241272738035, 0.11224976596193706]]], 
[[[0.16841704660457429, -0.065144870102143329], [-0.08998400167925491, -0.013494244254822471]], [[0.19970945613278046, 
0.11850815261149236], [-0.024556026548658197, 0.081535068361703419]], [[0.0010807637207397428, -0.12106958762886319], 
[0.034569596336150643, 0.19363633807509351]], [[-0.17053759842086455, -0.010979164221394104], [-0.01977884774536565, 
0.18885844604456978]], [[-0.11233270900759984, -0.065123414398134544], [0.12479827030891849, 0.17842167131369596]], 
[[-0.20073208275882631, 0.20933068478725397], [-0.11349426030998769, -0.068606487312557876]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.027129253886383372, -0.47665616726030652], [0.1544056018684771, 
-0.057524649879943116]], [[-0.5505255142932215, 0.050893967736198316], [0.29850911041632899, 0.34559149301950426]], 
[[-0.51548054936047338, -0.086187031373830575], [-0.43907701664972865, 0.22514537401280726]], [[-0.53680436412072374, 
0.40708553868996328], [0.53371084847430095, -0.26269085267489806]], [[-0.24047230049464405, 0.07131264721862994], 
[-0.41639372037827671, 0.14799317110100685]], [[-0.35652353985685364, -0.01637363936844843], [-0.21786303478682775, 
-0.52014359406910304]]], [[[0.011907338470941208, -0.2092098198350662], [0.067770376987041012, -0.025248223906617125]], 
[[-0.24163191745087328, 0.022337942005409971], [0.1310190478983585, 0.15168404178364001]], [[-0.2262502832598223, 
-0.037828469543311913], [-0.19271590269143182, 0.098818868545541769]], [[-0.23560954838761836, 0.17867447870506195], 
[0.23425176914232929, -0.1152980066875386]], [[-0.10554603111708835, 0.031299932952348918], [-0.18275994564698161, 
0.064955888100265216]], [[-0.15648224163158481, -0.0071865767771483171], [-0.09562256693005676, -0.22829694668386336]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_expandedData_rank3(self):
      arg0=Data(0.228203902148,self.functionspace)
      arg0.setTaggedValue(1,-0.582025825707)
      arg1=Data(numpy.array([[[0.18219833718655032, 0.09136415777484852], [-0.24714399648573737, -0.95411352806058458]], 
[[-0.56003002687251469, 0.89612106752070941], [-0.29032325966866424, 0.31800131617471616]], [[0.88123818901080964, 
0.75282018595038314], [0.70070139443145152, -0.53605679804063655]], [[0.0143035045347637, -0.49142057567515529], 
[-0.46322981296177734, 0.27908663175767523]], [[0.74055138211147242, -0.34317931368204602], [-0.7127155916050063, 
0.52901566573979442]], [[-0.54130729606745565, 0.39044532610577121], [0.072082576382971331, 
0.66247939707550252]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.54239104278730776, 0.10351077665486841], [-0.95296615235066229, 
-0.98163950169608327]], [[0.40648539129402916, 0.27806399373350499], [-0.080979163228941164, 0.47490521963341736]], 
[[0.95830625070081488, 0.032194268064548393], [-0.45370281574708571, 0.16531319438516401]], [[-0.81073703757490767, 
0.34083104255702379], [-0.80833745661848044, 0.34807418507556243]], [[0.52908241308030357, -0.81402055860073763], 
[0.84821288215003898, 0.3188831666358789]], [[0.066686002982491832, 0.13715276057925752], [0.40382088973916597, 
0.8388046290152007]]]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.041578371510860343, 0.020849657320692234], [-0.056399224390513827, -0.21773243019568617]], 
[[-0.12780103745239557, 0.20449832440531876], [-0.06625290074073617, 0.072569141239291965]], [[0.20110199345416402, 
0.17179650404971206], [0.1599027924498502, -0.12233025308587241]], [[0.0032641155492256712, -0.11214409296492069], 
[-0.10571085090919757, 0.063688658404462584]], [[0.16899671513898343, -0.078314858518738983], [-0.16264447912603169, 
0.12072343921927943]], [[-0.12352843722381326, 0.089101146992812158], [0.016449525207480272, 
0.15118038350532947]]])+(1.-msk_ref)*numpy.array([[[0.31568559453442024, -0.060245945252133462], [0.55465091169281655, 
0.57133954152137345]], [[-0.2365849955057828, -0.16184042555215836], [0.047131964343394873, -0.27640710258975354]], 
[[-0.55775898684442127, -0.018737895453304641], [0.26406675596083584, -0.096216548462304152]], [[0.47186989372586741, 
-0.19837246897086505], [0.47047327563835184, -0.20258816497593171]], [[-0.30793962834017102, 0.47378098776215277], 
[-0.49368180310877935, -0.18559823836534362]], [[-0.038812975948991239, -0.079826448724151164], [-0.23503418678821567, 
-0.48820595680951362]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(6,2,2),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank1_expandedData_rank3(self):
      arg0=Data(numpy.array([-0.33790913306894765, -0.8715614151541422]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.42667656693497014, 0.2339000130632285]))
      arg1=Data(numpy.array([[[0.83466720680097461, 0.21865310359296974], [0.9612985125997453, -0.0029043359986478112]], 
[[0.56744088009819427, 0.42010691254376753], [-0.56185411880630753, 0.59329373441306688]], [[0.55944183842739692, 
0.38750723523638619], [-0.82156673278560133, 0.27827027376317837]], [[0.24133620412657319, 0.19405126867850209], 
[-0.98399383195314738, -0.031986753324890982]], [[0.24439285972277469, 0.81757797446968694], [-0.25842977913832188, 
-0.42345542441585549]], [[-0.93935417405340171, 0.36645987361650967], [-0.3934811269453935, 
-0.17466683694827356]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.26621717213635665, 0.083182748324255318], [-0.53526844777547833, 
0.8817700186299704]], [[0.2658880326961679, -0.78728401462668729], [0.65311556852202735, -0.23209419809332399]], 
[[-0.41772630372396757, 0.19103258298752479], [0.060534256904470896, -0.200613168763772]], [[0.41442178247181616, 
-0.46007291860026633], [-0.19891997755859125, 0.3462263171465354]], [[0.84453992120160515, 0.050165814595225289], 
[0.78835500121560909, -0.001257249583341169]], [[-0.99046355050122581, -0.93606609054838064], [0.2257625278268558, 
0.70199834581605347]]]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-0.28204167225119736, -0.073884880677935208], [-0.32483154701304878, 
0.00098140165944401829]], [[-0.19174345586186151, -0.1419579626139367], [0.18985563819705689, -0.20047937145075789]], 
[[-0.18904050662549998, -0.13094223391667204], [0.27761490243387033, -0.09403006696617433]], [[-0.081549707514560935, 
-0.06557169597008207], [0.33250050270047982, 0.01080861608570419]], [[-0.082582579357163732, -0.27626706456931815], 
[0.087325782627829959, 0.14308945535770501]], [[0.31741635459908235, -0.1238301381983109], [0.13296086648511046, 
0.059021519449086353]]], [[[-0.72746373194221248, -0.19056960839533396], [-0.837830692027006, 0.0025313071930646051]], 
[[-0.49455957647469412, -0.36614897521268347], [0.48969037089700895, -0.51709192676713833]], [[-0.48758792039621701, 
-0.33773635432509386], [0.71604586427018369, -0.24252963359636631]], [[-0.21033932359648508, -0.16912759834189195], 
[0.85761105668003235, 0.027878419994028448]], [[-0.21300338667354926, -0.71256941642765748], [0.22523742402376823, 
0.36906740895858092]], [[0.81870485326893316, -0.31939228604641329], [0.34294296783697381, 
0.15223287559113513]]]])+(1.-msk_ref)*numpy.array([[[[0.11358862906627665, 0.035492129483208901], [-0.22838650368545144, 
0.37623060437522043]], [[0.11344819297989402, -0.33591564056369577], [0.27866910858875987, -0.099029155647984368]], 
[[-0.17823402519137713, 0.081509126681836852], [0.025828548917959153, -0.08559693813007202]], [[0.17682406340814549, 
-0.1963023334481136], [-0.084874493119481018, 0.14772665638262192]], [[0.36034539421783107, 0.021404577548986944], 
[0.33637260544469028, -0.00053643893600043165]], [[-0.42260758740208443, -0.39939746593942194], [0.096327580315723499, 
0.29952624418682167]]], [[[0.062268200040349567, 0.019456445919678569], [-0.12519929692701842, 0.2062460188763133]], 
[[0.062191214320989799, -0.18414574130565314], [0.15276374000910009, -0.054286835965928024]], [[-0.097706187897890165, 
0.044682523656284327], [0.014158963480728572, -0.046923422794501936]], [[0.096933260333844237, -0.10761106167063995], 
[-0.046527385349491614, 0.080982340103408129]], [[0.19753789860147342, 0.011733784689150694], [0.18439624508279248, 
-0.00029407069396723801]], [[-0.23166943740088838, -0.21894587080731145], [0.052805858207889057, 0.16419742225673969]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_array_rank4(self):
      arg0=Data(-0.150904683599,self.functionspace)
      arg0.setTaggedValue(1,-0.490629899716)
      arg1=numpy.array([[[[-0.40547153441452943, 0.47816613420462928, -0.94009175264256739, -0.90046551099169259], 
[0.93555584566755456, -0.54343526353781479, 0.64093923688564436, 0.19778230300137967], [0.7794205807565564, 
-0.045240031958294402, -0.69970388359085511, -0.14389694224200977]], [[-0.90371721492009094, 0.071067869328625521, 
0.58081761320409675, 0.4209148529081832], [-0.28750376736648042, 0.73358720392325938, -0.045303059870043239, 
0.11972950793243475], [-0.84007622503811508, 0.25954173715149298, -0.95223237075217448, 0.43865599806451461]]], 
[[[0.90981678335073757, 0.79963713327190455, -0.93272136040383224, 0.17562447633144629], [0.056902674555987742, 
0.10459808364243539, -0.41767842984949266, -0.20949581363284975], [0.99958274070488495, 0.37918073075702563, 
0.9577726013612371, -0.21051343650634879]], [[-0.55664901190900862, -0.040542431977547588, -0.64911057515293558, 
0.50590024580173432], [0.66284138948504889, 0.87111000714174125, -0.90983669695297742, -0.85176956945222559], 
[-0.35709778315040275, 0.58413606334190571, -0.34398179625555469, -0.52597581928819381]]], [[[0.77555991963163917, 
0.79899943871380574, 0.42040395771615091, 0.70824853740381233], [-0.71905420320894042, -0.6651259143527819, 
0.33678549899273036, -0.090179759606354049], [0.096941868869977021, -0.15708081427854315, 0.1888821635464577, 
-0.25840593671508238]], [[0.82447195320414557, -0.76578778984736573, 0.94065161723731028, 0.37773504832109661], 
[-0.62751600514088479, 0.85063471727306283, -0.70896252516937008, -0.96623923587159077], [0.62173798307776496, 
0.36954037883823654, 0.10897413977723325, -0.060678505495639357]]]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.061187553609288896, -0.072157509189981192, 0.14186424848670276, 0.1358844630281538], 
[-0.14117975887980325, 0.082006926500797953, -0.096720732748512728, -0.029846275855935619], [-0.11761821612973863, 
0.0068269327086821268, 0.10558859316637874, 0.021714722539916523]], [[0.13637516038062689, -0.010724474335102404, 
-0.087648098149381251, -0.063518022700294718], [0.043385665048004111, -0.11070174490042906, 0.0068364439157625008, 
-0.018067743512026717], [0.12677143693855022, -0.039166063725621383, 0.14369632462123119, -0.066195244596797612]]], 
[[[-0.13729561382474501, -0.12066898859053297, 0.14075202177791474, -0.026502556033064381], [-0.0085868800998170798, 
-0.015784340717139779, 0.063029631302630007, 0.031613899471612965], [-0.15084171721724796, -0.057220148201785746, 
-0.14453237136835814, 0.031767463529361599]], [[0.084000943017911819, 0.0061180428699121273, 0.097953825964319979, 
-0.076342716525446011], [-0.10002587015666629, -0.13145458000778307, 0.13729861888059033, 0.12853601737757753], 
[0.053887727980271583, -0.088148867817466892, 0.051908464127813858, 0.079372214590491791]]], [[[-0.11703562428419945, 
-0.12057275749501017, -0.063440926222988747, -0.10687802144648738], [0.10850864702588844, 0.10037061565900594, 
-0.050822509166281879, 0.013608548090444815], [-0.014628982049334762, 0.023704230578201355, -0.028503203127502257, 
0.03899466612013306]], [[-0.12441667923464982, 0.11556096413101377, -0.14194873467623079, -0.057001987951207027], 
[0.094695104209191625, -0.12836476286854914, 0.10698576554434254, 0.14581002617029276], [-0.093823173617927594, 
-0.055765373945696362, -0.016444708083573595, 0.0091566706730891109]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.19893645826761824, -0.2346026024725521, 0.46123712232315139, 0.44179530335584849], 
[-0.45901167073887833, 0.26662558885186366, -0.31446395351745088, -0.097037911487227416], [-0.38240704137341608, 
0.022196112342860567, 0.3432956462372927, 0.070600142341680661]], [[0.44339068652814567, -0.034868021601755242, 
-0.28496648731979157, -0.20651341207144661], [0.14105794455107712, -0.3599198162940358, 0.022227035720881225, 
-0.058742876469973407], [0.41216651404450527, -0.12733893647083347, 0.46719367256876188, -0.21521774834034993]]], 
[[[-0.4463833171755866, -0.39232588650663081, 0.45762098751819091, -0.086166619210226655], [-0.027918153510993995, 
-0.051318947288006109, 0.20492552615072171, 0.10278491003367149], [-0.49042517983019035, -0.186037403905676, 
-0.4699118753568915, 0.10328418624204536]], [[0.273108648890099, 0.019891329335399426, 0.31847305639208001, 
-0.24820978686416065], [-0.325209804450868, -0.42739261544582513, 0.44639308738425654, 0.41790361844174856], 
[0.17520284953599813, -0.28659461817811832, 0.16876775420109841, 0.25805946347056963]]], [[[-0.38051288559285928, 
-0.39201301448954118, -0.20626275161461424, -0.34748790888065451], [0.35278949161099243, 0.32633066065762328, 
-0.16523703559671016, 0.044244886412106277], [-0.047562579401988339, 0.077068544156837671, -0.092671236958997885, 
0.12678167881661959]], [[-0.4045105917194588, 0.37571838653678646, -0.4615118086331248, -0.18532810887711451], 
[0.30787811467265075, -0.41734682603089313, 0.34783821262646875, 0.47406585939764095], [-0.30504324428726365, 
-0.18130755901053058, -0.053465971270574918, 0.029770689066261]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_constData_rank4(self):
      arg0=Data(0.1656822013,self.functionspace)
      arg0.setTaggedValue(1,-0.722496327291)
      arg1=Data(numpy.array([[[[0.16479743329774732, 0.20364181085661115, 0.14689095497798643, 0.54842743202365707], 
[-0.84772087110675276, 0.24586213698101811, 0.37626212505508061, -0.13784188722616419], [0.22296403616016414, 
-0.37599821840943615, -0.37672491569592537, -0.017435872398913199]], [[0.92769812786488792, -0.72698070400348969, 
0.086307179609877771, 0.056694352792610259], [-0.84593935959177946, 0.82077654526064281, -0.93093649245778054, 
0.41155320446911126], [0.12296662680314197, 0.93823062706098237, -0.024658451344554466, 0.77545792592095353]]], 
[[[0.74717067419793226, 0.83184264177982614, 0.030393992532188552, -0.8125650185169504], [-0.25154414553174709, 
-0.17862932847253399, -0.39079718431105248, -0.85623877307317597], [-0.90528908372269234, 0.52442488287472977, 
-0.69806528835106318, 0.18988439924066181]], [[-0.70923327843906692, 0.89944860568119611, 0.31217063853242211, 
-0.27813039240991611], [0.19210084172640696, -0.10523048797222256, -0.61921780229864876, -0.6095460358064495], 
[0.2942674029385639, -0.23827405242981303, -0.9042910830330948, -0.28059372795645654]]], [[[0.73123714313297161, 
0.084900190645528451, 0.38366440616728226, -0.259482991140771], [-0.05098245704126203, 0.91080544303900202, 
0.81172674486061847, -0.36784265028159568], [0.32248671926272254, 0.17029937059226641, -0.75316208356132219, 
0.69643002508002083]], [[0.25378352385636482, 0.49489738349937618, 0.68090000594740152, 0.67820363349592183], 
[-0.67532286496036575, 0.21796109767502148, -0.60259266243764142, 0.72926100669006555], [-0.85189433626075117, 
-0.48610431804692844, 0.45892986586131346, -0.31779612561597048]]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.02730400151737802, 0.033739823499462987, 0.02433721677182743, 0.090864664191043276], 
[-0.14045226001300948, 0.040734980071363065, 0.062339937144981199, -0.022837947306991728], [0.036941172321772235, 
-0.062296212510993099, -0.062416613317097441, -0.0028888137206396838]], [[0.15370306796664104, -0.12044776334199833, 
0.014299563505768099, 0.0093932451719644303], [-0.14015709526356723, 0.13598806479427866, -0.15423960734100381, 
0.068187040868554621], [0.02037338141519307, 0.15544811561864155, -0.0040854664994173211, 0.12847957618219744]]], 
[[[0.12379288204799709, 0.1378215200253764, 0.0050357435890319557, -0.13462756096734904], [-0.041676387755853861, 
-0.029595700358088978, -0.064748137758538091, -0.14186352476126499], [-0.14999028820413085, 0.086887869011235033, 
-0.11565699362519678, 0.031460465258740862]], [[-0.117507330807075, 0.14902262494557084, 0.051721118573311117, 
-0.046081255662936957], [0.031827690328834195, -0.01743481889112207, -0.10259336856905343, -0.10099092900616526], 
[0.048754871089726294, -0.03947776951926809, -0.1498249372529793, -0.046489386518828571]]], [[[0.1211529795466708, 
0.014066450476949748, 0.063566363374292958, -0.042991713172138617], [-0.0084468857102843378, 0.15090425075881939, 
0.13448867394267608, -0.060944980030719513], [0.053430309537496889, 0.028215574599749087, -0.1247855519402136, 
0.11538605960674528]], [[0.04204741288622027, 0.081995687915838972, 0.11281301185062016, 0.11236627092733405], 
[-0.11188897885492703, 0.036112274460584785, -0.099838878799959604, 0.12082556891074075], [-0.14114372890677324, 
-0.080538833475501531, 0.076036510418264383, -0.05265316165669872]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.11906554030459199, -0.14713046042677275, -0.10612817548385035, 
-0.39623680542268019], [0.61247521594248411, -0.17763449098868145, -0.27184800345097071, 0.099590257267752128], 
[-0.16109069724367736, 0.27165733186874497, 0.27218236798928569, 0.012597353771327823]], [[-0.6702584902170392, 
0.52524088865388519, -0.062356620286974043, -0.040961461670796347], [0.61118808041588935, -0.59300803947733993, 
0.67259819674183308, -0.2973456787137399], [-0.088842936244622667, -0.67786818220341194, 0.017815640533122328, 
-0.56026550344651938]]], [[[-0.53982806796748295, -0.6010032535698967, -0.021959547976213727, 0.58707524156357105], 
[0.18173972129821872, 0.1290590337678483, 0.28234953038036625, 0.61862936882944886], [0.65406803812620296, 
-0.37889505181696054, 0.50434960704291676, -0.13719078106121999]], [[0.51241843886472083, -0.64984831419159872, 
-0.22554213982773477, 0.20094818702414538], [-0.1387921526168224, 0.076028641078961609, 0.44738258795392566, 
0.44039477218489637], [-0.21260711786454825, 0.17215212776926286, 0.65334698629333487, 0.20272793790940602]]], 
[[[-0.52831615029227319, -0.061340075927692805, -0.27719612436811131, 0.18747550809366809], [0.036834637968578672, 
-0.65805358747225373, -0.58646959192560655, 0.26576496384940934], [-0.23299547026741343, -0.12304066979286694, 
0.54415683922782854, -0.50316813533543492]], [[-0.1833576639131699, -0.35756154196418277, -0.49194775354935982, 
-0.48999963435615734], [0.48791828966944284, -0.15747609256249942, 0.43537098546365005, -0.5268883989700478], 
[0.61549052918832459, 0.3512085844691607, -0.3315751425689113, 0.22960653358482097]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_taggedData_rank4(self):
      arg0=Data(0.875302450355,self.functionspace)
      arg0.setTaggedValue(1,0.0173419771558)
      arg1=Data(numpy.array([[[[0.40774352673890957, 0.78540916772531211, -0.94921701601466757, 0.85157754528384433], 
[0.70535719066060198, 0.5454322142403194, 0.19678959692359865, 0.041128660770753056], [0.77220194968532629, 
-0.99875008759449191, 0.77524055632096056, -0.84084226687199548]], [[0.90507986174988897, 0.87800144279965209, 
0.79150565872113687, -0.70251281336415095], [-0.85847604128143518, -0.46963203392901454, 0.19427526837140441, 
0.1151039379150729], [-0.94327440895614134, -0.16600980415135314, -0.84246339825238326, -0.56296463030040655]]], 
[[[-0.55434605322612573, 0.43148289791869665, 0.99472924240968896, 0.6518618731597885], [-0.19843421328931155, 
-0.93970871111286125, 0.36203790401978964, 0.58438435142930434], [-0.41765214039157073, -0.70498316197551447, 
0.9229135459169977, 0.41331032823998171]], [[0.62144546299886616, 0.94010567934718381, -0.41001995195693008, 
0.66815845359993187], [-0.49429732395738224, 0.99229245113648168, -0.4616008583957889, 0.25354570522147823], 
[0.91258722097544975, 0.25296689732501587, -0.36260867298224442, -0.34389243775459732]]], [[[-0.83717745861378079, 
0.41353092085105203, 0.37102507172503607, -0.61277014465294455], [0.13041113656027448, 0.66753543265137427, 
-0.49530516459652585, 0.056776470263388834], [0.087960829222912151, -0.34173819525950599, -0.8351285948547218, 
0.00054254868734493478]], [[-0.3671187156506539, -0.47791148269381156, 0.61421529479450609, 0.60814411737633334], 
[-0.54742326309712186, -0.53039527899661221, -0.4217917283009569, -0.85420150453686028], [-0.95356634205296542, 
0.44691759847551027, 0.70770936016894437, -0.3140155802606921]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.1979460594313549, -0.64441112295244074, 0.56514237463907424, 
0.87978662458686063], [-0.084441509236192092, -0.85775135823168824, -0.93056222622293583, -0.4808611172223578], 
[0.62786912714201981, -0.39298138680483952, 0.068114934130811156, -0.73690991475681211]], [[-0.063927200909925208, 
0.31470995059084572, 0.54996122757464883, 0.70103979859923626], [0.70417032219518338, -0.38473841616371773, 
-0.67298590501300826, 0.45386867818305454], [0.18520670265220907, -0.21091421323027504, -0.90352076928300207, 
0.64716490295083795]]], [[[-0.97804582982873312, 0.81448082444734959, 0.25308540593893158, -0.80044668771742433], 
[0.51836481200072626, 0.27080171312794143, 0.50928333449477781, -0.16728136097540292], [-0.80790265614558088, 
-0.28872194465656587, -0.63032963344273685, -0.96089757904857365]], [[0.92158223235927039, 0.9405319971081445, 
-0.50589055646231396, -0.42829640296501914], [-0.75830485398051684, 0.95797120552606918, -0.083758826387434704, 
0.12217984371418367], [-0.50929236813459955, 0.67035027952467829, -0.82621555577809458, 0.30605787776453308]]], 
[[[-0.70402395146151653, 0.20731030136487405, -0.084351586120795297, 0.8355380193202302], [-0.049462326220570896, 
-0.92989518388499826, 0.069813063820197918, 0.79408379432997922], [0.054900846362986311, 0.78960565402830718, 
-0.38358056833949794, -0.53218602462702602]], [[0.080388391934530512, 0.47541951027583118, 0.68684199219215181, 
-0.25116904607587176], [0.096588395712015718, 0.53812973276623643, 0.2082890562903148, -0.15289159834586141], 
[-0.45988400027424925, -0.40868311022684067, 0.82143964554308502, -0.089579679543674517]]]]))
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.35689890807095093, 0.68747056904123516, -0.83085198003628569, 0.74538791205423227], 
[0.61740087736073335, 0.47741815362709678, 0.17225041639159575, 0.036000017552459102], [0.675910258728463, 
-0.87420839896371483, 0.67856995856229863, -0.73599129655509798]], [[0.79221862075662897, 0.76851681429774776, 
0.69280684254844771, -0.61491118694341562], [-0.75142618250468751, -0.41107007006326191, 0.17004961844886263, 
0.10075075890257137], [-0.82565040151646085, -0.14530878835663058, -0.73741027682469873, -0.49276432036513418]]], 
[[[-0.48522045873344283, 0.3776780378345051, 0.8706889433209587, 0.57057629486975336], [-0.17368995312639815, 
-0.82252933745701229, 0.31689266450990478, 0.51151305475517883], [-0.3655719418807461, -0.61707348913617321, 
0.80782848820695619, 0.36177154306547921]], [[0.54395273652489573, 0.82287680472522795, -0.35889146864233418, 
0.58484073166141803], [-0.43265965886380858, 0.86855601394851678, -0.40404036243979852, 0.22192917705734264], 
[0.79878983068245746, 0.22142254508728432, -0.31739225998132803, -0.30100989342514822]]], [[[-0.7327834809066015, 
0.36196462831847925, 0.32475915442405817, -0.53635920911910107], [0.11414918738478683, 0.58429539989852308, 
-0.4335418242448183, 0.049696583544051195], [0.076992329354071382, -0.29912427969053595, -0.73099010543785359, 
0.0004748941954699021]], [[-0.32133991138019236, -0.4183170918546773, 0.53762415257914076, 0.53231003610847427], 
[-0.47916092357023243, -0.46425628736245061, -0.36919533332129167, -0.74768467001802874], [-0.83465895577500049, 
0.39118806905237952, 0.61945973709503566, -0.27485860685182628]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.0034327760407412283, -0.011175362973191301, 0.0098006861507712213, 
0.015257239545572805], [-0.0014643827241761893, -0.014875104459819209, -0.016137788869218146, -0.0083390825099875568], 
[0.010888492059735471, -0.0068150742326281849, 0.0011812476316660516, -0.012779474907602765]], [[-0.0011086240578148194, 
0.0054576927738526401, 0.0095374150451809584, 0.01215741617262182], [0.012211705641308461, -0.0066721248240738358, 
-0.011670906190917922, 0.0078709802487883559], [0.0032118504064975639, -0.0036576694676751361, -0.015668836540705985, 
0.011223118963015633]]], [[[-0.016961248438225433, 0.01412470785141149, 0.0043890013282619324, -0.013881328172839607], 
[0.0089894707280925041, 0.0046962371228190594, 0.0088319799526433412, -0.002900989540628297], [-0.014010629406995141, 
-0.005007009368615296, -0.010931162103794233, -0.016663863864933803]], [[0.015982058020775142, 0.016310684408158097, 
-0.0087731424735096181, -0.007427506436135092], [-0.013150505454870202, 0.016613114762157162, -0.001452543653808375, 
0.0021188400585918478], [-0.008832136613818764, 0.011625199233908031, -0.014328211294078842, 0.0053076487245483197]]], 
[[[-0.012209167283388932, 0.0035951705104337981, -0.0014628232795631986, 0.014489881243862432], [-0.00085777453139037816, 
-0.016126221036231674, 0.0012106965579470001, 0.013770983021069678], [0.0009520892234615606, 0.013693323214257569, 
-0.006652045453556312, -0.0092291578817233922]], [[0.0013940936565209545, 0.0082447142866299925, 0.011911198138247545, 
-0.0043557678592944368], [0.0016750337519541437, 0.0093322335324943825, 0.0036121440559919284, -0.0026514426058292552], 
[-0.0079752978270786964, -0.0070873731615193813, 0.014245387567885105, -0.0015534887562712123]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_taggedData_rank0_expandedData_rank4(self):
      arg0=Data(-0.757537859603,self.functionspace)
      arg0.setTaggedValue(1,0.547692889363)
      arg1=Data(numpy.array([[[[-0.56408634628692611, 0.029807270678304576, 0.377478257286195, -0.53461924813103434], 
[0.26126192935302028, -0.12054661601412242, -0.43189026326189017, 0.9631160954556135], [-0.79983541463719998, 
0.48522903103027115, -0.15629905840859748, 0.35684949738752514]], [[0.39842990750906804, 0.97474771916254843, 
-0.53519461494946174, 0.46541867586621488], [-0.24409710885046643, 0.31105539856735454, 0.84680755699225374, 
-0.36334045402324322], [0.73718177614253633, -0.82489130946900757, 0.25524969268104014, -0.64845544618726381]]], 
[[[0.94718388659948216, -0.36384299945446341, 0.36969395542744876, -0.24618346217230647], [0.34393326903008758, 
-0.83204265782503883, -0.19531737415996453, -0.54209679703616254], [-0.7630305624308642, -0.46545868864364626, 
0.59704584293218455, 0.26704120894998074]], [[-0.071450963412232449, -0.4393070347495549, 0.96179961482779741, 
0.8893917547583241], [0.37113022552394881, -0.46171278683058925, -0.62977257975584244, -0.19202229532482384], 
[-0.65754036740767008, 0.24480921031549796, -0.32015841166155834, -0.49984183975619367]]], [[[-0.065697143580287554, 
0.013107186858187569, -0.99819615856279187, 0.1950751152128638], [0.24072009777272596, -0.28047899218338812, 
0.76394170579651455, 0.22347349677158257], [-0.13861468108085351, 0.91403094049647104, -0.25849510432670764, 
0.95722937569499145]], [[-0.95882044575611736, -0.20262232814634618, 0.025744306595197441, -0.26271268910005796], 
[0.09022262197540587, 0.71629078282395531, 0.98678984146210502, 0.33330018533248906], [0.97572830737141136, 
0.35990401360257374, 0.19154547749906992, -0.85957071262274365]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.089017207806223686, -0.09307222171496643, 0.459447141534574, 
-0.93357175621330901], [-0.9469238944351337, -0.71745190908327938, -0.81106923887518922, 0.42850141198949832], 
[0.056627642692557378, 0.54869029575846628, -0.3171797380484298, -0.56778322148822635]], [[0.12394492102547994, 
-0.56436796708244796, -0.97010481726578623, 0.82106968001365543], [-0.1838592146085396, -0.77894923133649763, 
0.67911164315293004, -0.77651007021409524], [0.043118773767471641, -0.26295952076906559, -0.74465406336128037, 
0.9648752905114677]]], [[[-0.7862683303471687, -0.15919970145157669, -0.13678944987474395, -0.087139024185715508], 
[-0.85688882220584217, -0.5325737602867191, -0.017668549244263421, 0.93808560643613759], [-0.49683066436708923, 
0.73919297136937656, 0.57659493462042777, -0.97540095534226423]], [[-0.85436082466862251, -0.17276731025832825, 
0.22089845274487363, 0.38475616825389491], [-0.63741843835599421, -0.16334993539745857, 0.51862533289993462, 
0.00080820580835694322], [0.39575998613708241, 0.3740692933801586, 0.59535153346047731, -0.58640249765603714]]], 
[[[0.83018766189817872, -0.29826868855962352, 0.28098909605236977, -0.82742330761843741], [0.40097357606145989, 
-0.87733004969324835, -0.43432281018797347, -0.337247139594826], [0.31457069372758495, 0.49999004978577144, 
-0.70808411254762915, -0.48716938972954238]], [[-0.48185328773843406, -0.95335657100159321, -0.69436527312485663, 
0.67067713197238388], [0.19743991135068062, 0.79057772960368822, 0.58234659914237774, 0.57459369456994525], 
[0.65411570497305016, -0.43763464161197985, -0.87966682118151129, -0.98004678495557962]]]]))
      arg1.expand()
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.42731676339723756, -0.02258013603023758, -0.28595407107109605, 0.40499432093152421], 
[-0.19791580275772738, 0.09131862547767236, 0.32717322561460693, -0.72959690550023926], [0.60590560803860627, 
-0.36757936158370536, 0.11840245416474753, -0.27032700445120217]], [[-0.30182573933607321, -0.73840830082689335, 
0.40543018307964207, -0.35257226753475923], [0.18491280137376026, -0.2356362408485411, -0.64148878421920141, 
0.27524414984779716], [-0.55844310483704507, 0.62488639697992121, -0.19336130585781139, 0.4912295507523356]]], 
[[[-0.71752765410462427, 0.27562484703811679, -0.28005716770252104, 0.18649329300356168], [-0.26054247246717077, 
0.63030381410682157, 0.14796030556433573, 0.41065884732418861], [0.57802453907522944, 0.35260257872853135, 
-0.45228482993946506, -0.20229382585365366]], [[0.0541268098898448, 0.33279171081253389, -0.72859962158323521, 
-0.67374792624780333], [-0.28114519667723487, 0.34976491628678674, 0.47707657210463578, 0.14546415859634149], 
[0.49811172252830022, -0.18545224519340009, 0.24253211790385848, 0.37864911742872259]]], [[[0.04976807352981439, 
-0.0099291902779624734, 0.75617138142117457, -0.14777678524007948], [-0.18235458763007453, 0.2124734554020925, 
-0.5787147646702353, -0.16928963442224867], [0.10500586881548396, -0.69241304227422951, 0.19581982804939965, 
-0.72513749241269743]], [[0.72634278822128051, 0.15349408477167462, -0.019502286915078444, 0.19901480819129588], 
[-0.068347051938981634, -0.54261738647351532, -0.74753066437877203, -0.25248750900191691], [-0.7391511335197869, 
-0.27264091612687141, -0.14510295104119952, 0.65115735781729733]]]])+(1.-msk_ref)*numpy.array([[[[0.048754091746449367, 
-0.050974994030537298, 0.25163593245680782, -0.51131061258849431], [-0.51862348375038447, -0.39294330906508051, 
-0.44421685491329427, 0.23468717642880818], [0.031014557244122755, 0.3005137734495914, -0.17371708717925843, 
-0.31097083310892376]], [[0.067883751918358673, -0.30910032255551201, -0.53131951035361358, 0.44969402541532905], 
[-0.1006983844850293, -0.42662495517805532, 0.37194461803872786, -0.42528904397530398], [0.023615845790511628, 
-0.14402105971561421, -0.40784173553850678, 0.52845533573553793]]], [[[-0.43063357366274629, -0.08719254447379847, 
-0.074918609036323122, -0.047725423932578302], [-0.46931191489708485, -0.29168686157054308, -0.0096769387864494572, 
0.51378281625918876], [-0.27211062209152925, 0.40485073428638191, 0.3157969457345402, -0.53422016751918711]], 
[[-0.46792734862162155, -0.094623427342920008, 0.12098451183973566, 0.21072821749135109], [-0.34910954623667573, 
-0.089465598095152407, 0.28404740707300008, 0.00044264857437926488], [0.21675493030182227, 0.20487509211348975, 
0.3260698015478764, -0.32116847827112638]]], [[[0.45468787925882637, -0.16335963984384061, 0.15389572989652131, 
-0.45317386207613092], [0.21961037643146028, -0.48050742984179612, -0.23787551482826572, -0.18470786031421893], 
[0.17228813215669783, 0.27384099502009979, -0.38781263351350215, -0.26681921067035824]], [[-0.26390761941069785, 
-0.5221466149654086, -0.38029892271133192, 0.36732509623988696], [0.10813643552330009, 0.4329938009929698, 0.31894709149521572, 
0.31470088078898173], [0.35825452043464151, -0.23968938134996412, -0.48178726296998475, -0.53676465536359375]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(3,2,3,4),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_taggedData_rank0(self):
      arg0=Data(0.321320071636,self.functionspace)
      arg0.setTaggedValue(1,-0.417719679857)
      arg0.expand()
      arg1=Data(0.315101255367,self.functionspace)
      arg1.setTaggedValue(1,-0.516168618373)
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array(0.10124835794732541)+(1.-msk_ref)*numpy.array(0.21561379001913808)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_taggedData_rank0(self):
      arg0=Data(numpy.array([-0.53389867378993516, -0.36871938142193272]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.18619702824976248, 0.71163793281689314]))
      arg0.expand()
      arg1=Data(-0.197202833488,self.functionspace)
      arg1.setTaggedValue(1,0.132256256872)
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.10528633126705458, 0.07271250677848172])+(1.-msk_ref)*numpy.array([0.024625721996932731, 
0.094118569242219133])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_taggedData_rank0(self):
      arg0=Data(numpy.array([[0.39543185767634026, -0.9790234836580971, -0.92093385964809538, -0.1205125543184995, 
-0.45935601999058973], [0.14483140686234575, -0.93632924171739651, 0.63648438037303445, -0.4841478366787515, 
-0.55450043856701869], [-0.67805397499649533, 0.7001941932400626, -0.58976639378391993, 0.4625604606600815, 
0.76670388049907023], [-0.85249349969631449, -0.071133152844088166, -0.55350924442860405, -0.59600086828610244, 
-0.53992646848239767]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.25407123069462245, 0.12579367753123694, 0.26022142360246381, 0.46580824252824682, 
-0.56459911818343222], [-0.028745685228212192, 0.9181939139226869, 0.72632053373589822, 0.53794255967610294, 
0.66620887983307031], [0.078016920134768242, -0.86344653195479504, -0.81543313785292182, 0.93409031113167162, 
0.12882034804263109], [0.99689622801818456, 0.55519649218337852, -0.8660049093461355, 0.32659400502451974, 
0.6057362031977469]]))
      arg0.expand()
      arg1=Data(-0.306696445732,self.functionspace)
      arg1.setTaggedValue(1,0.484427826186)
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[-0.12127754527868202, 0.3002630227264616, 0.28244714150866435, 0.036960772075612998, 
0.14088285865688147], [-0.04441927771510025, 0.28716885047001084, -0.19520749722457964, 0.14848642071838913, 
0.17006331366554514], [0.20795674414613052, -0.2147470703891716, 0.18087925678592567, -0.14186564922077485, 
-0.23514535507828074], [0.26145672636680906, 0.021816285151019282, 0.16975931794626259, 0.18279134795675414, 
0.16559352884038198]])+(1.-msk_ref)*numpy.array([[0.12307917398168873, 0.060937957754345844, 0.12605849856265625, 
0.22565047434727997, -0.27350752348789048], [-0.013925209807317518, 0.44479868173838871, 0.35184987727162503, 
0.26059434479659649, 0.32273011944305868], [0.0377935670265791, -0.41827752650233174, -0.39501850236977015, 
0.45249933888251959, 0.062404161170760489], [0.48292427267144472, 0.26895262981424856, -0.4195168757005811, 
0.15821122389926762, 0.29343547215698601]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_taggedData_rank0(self):
      arg0=Data(numpy.array([[[0.2967967474409392, -0.97613994635216805], [-0.81410908737832455, -0.47391218065637286]], 
[[0.085252316370088232, -0.55888502726843403], [-0.16108448504463957, 0.20938387677583248]], [[-0.35962928958746865, 
0.92383998703841019], [-0.53859366936613706, -0.84558009952390023]], [[0.049913112241493307, 0.54575498336140438], 
[0.90634308192577873, -0.67131420049002322]], [[-0.72146267925981511, 0.79406250857577132], [0.87830519228680703, 
0.55856855725805676]], [[0.78509858138861244, 0.90816579869808178], [-0.68852081816358579, 
0.76553985300414062]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.38134398850326634, -0.0397675047221282], [0.020677886762163533, 
-0.55701656946760258]], [[-0.77443543754581068, -0.87104025632003812], [0.073595342292162691, 0.10136678202394522]], 
[[-0.98145888789111013, -0.48153206410368732], [-0.84521107414445162, 0.19029871093188988]], [[0.6614065420185713, 
0.60831117208293795], [-0.12002267620296347, 0.23769411285079922]], [[-0.93635256270168199, -0.70316976204862547], 
[0.75157405006459954, 0.24071485643479718]], [[0.6357668213686738, -0.97338289161755709], [-0.098058097492607077, 
-0.21754866345396606]]]))
      arg0.expand()
      arg1=Data(0.976102453671,self.functionspace)
      arg1.setTaggedValue(1,-0.99016298095)
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.289704033418555, -0.95281259676024188], [-0.79465387774551788, -0.46258684236307157]], 
[[0.083214995189945629, -0.54552904643648026], [-0.15723496110033822, 0.20438011587995325]], [[-0.35103503197814395, 
0.90176247814721011], [-0.52572260219973976, -0.82537280992031159]], [[0.048720311329257825, 0.53271277836202591], 
[0.88468370613512881, -0.65527143828223067]], [[-0.70422149145727275, 0.77508636298864453], [0.85731585326278303, 
0.54522013928283786]], [[0.76633665166673415, 0.88646286444892042], [-0.67206686001276705, 
0.7472453288999743]]])+(1.-msk_ref)*numpy.array([[[-0.37759270042373516, 0.039376311020603412], [-0.020474477996169218, 
0.5515371868625526]], [[0.76681730139363369, 0.86247181672525175], [-0.072871383508039261, -0.10036963505813275]], 
[[0.971804258114078, 0.47679522401588631], [0.8368967167067739, -0.18842673888725167]], [[-0.65490027326490263, 
-0.60232720349479585], [0.11884201085071616, -0.23535591133459963]], [[0.9271416447048163, 0.69625266770392946], 
[-0.74418080181658597, -0.23834693980641647]], [[-0.62951277103547632, 0.96380770556971607], [0.097093498119560001, 
0.21540863310725505]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank4_taggedData_rank0(self):
      arg0=Data(numpy.array([[[[-0.56440249581330715, 0.59361505603581, -0.26406547264844149, -0.37286001343395681], 
[0.73471085391204993, 0.057491028457305138, 0.6402249393668924, 0.43933394158397387], [-0.30977226708062022, 
-0.30692561334294322, 0.80669456703281606, -0.34220328266962441]], [[-0.9221324049293993, 0.16918005537092018, 
0.55287742626263636, -0.9536763584486172], [-0.68148216613245705, -0.014870212106626424, 0.52040390563053496, 
-0.93394929024005702], [-0.61987357076692051, -0.399890930896309, -0.76335949187483609, -0.75020828332500633]]], 
[[[0.040717789282378458, -0.23922802361222151, 0.56767375175797996, -0.8324940127027789], [-0.8372055057709209, 
0.17033853598374504, 0.80557993378312087, 0.62648853179944086], [0.64073946033165008, 0.752506764341486, -0.48986290908254704, 
-0.90890379385855646]], [[0.0017813869929013126, -0.84780868070728665, -0.25935381445012617, -0.83767504064447906], 
[0.77875053886859535, 0.080347802735174501, -0.87799952110197843, -0.97158903472818814], [0.55793095465609066, 
-0.36747242528032609, 0.42175884937550245, 0.49179136944881541]]], [[[-0.91802659346620596, 0.78664793633460772, 
0.73383421990250031, -0.94678901943663107], [-0.93975291751946077, -0.16842339422527708, -0.70903417521901901, 
0.34940609239000175], [0.94355970684309698, 0.064997741184742441, -0.089486091737490447, -0.32247306588217284]], 
[[0.83239404342330636, -0.86166244624981947, -0.12452120578776582, -0.29175704932344559], [0.33859082136694174, 
0.42960527550258099, 0.3438522345203523, -0.1786864512535018], [0.71954443033069482, -0.97635906745092971, 
0.056243731231058858, 0.86296404115750103]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.045548635440329388, -0.52266407109324731, 0.26321998645037792, 
-0.45158498882187836], [-0.91774573458094966, -0.17202161504414115, 0.25237796435059989, -0.052805350645545257], 
[-0.47469582395425713, 0.79213592769423702, 0.92929141396053216, 0.47284494234292529]], [[-0.81531841529233762, 
0.22928198620036278, 0.19740114685932819, -0.37849326977369535], [-0.60539378698685065, -0.2837163185911038, 
-0.0096605480392844001, 0.48640021812525824], [0.70466383384988385, 0.93303299015543817, 0.85624904275624947, 
0.76613458044744731]]], [[[0.35548070682858013, -0.010316714245208525, 0.6806685818545295, -0.46197498226074796], 
[-0.81530692969634355, -0.77115888417856726, -0.96777577827783623, -0.7337027307495021], [0.77760901986322351, 
0.54081025136802841, 0.86698574982625587, 0.57245717823728226]], [[-0.46840643379120661, -0.60125760344985646, 
-0.5392223774892444, -0.44527432559834335], [0.32573725827031375, -0.42143577050656633, -0.14906324879316002, 
-0.63909807619657455], [0.30252513821811022, 0.64074278171588306, 0.22283989330748843, -0.85655374718033195]]], 
[[[-0.77748716108603011, 0.84411677333364454, 0.25831603386477431, 0.99733501126710222], [0.091234794204767544, 
0.53201415680322017, -0.34334707821815735, 0.3691237009756505], [0.62480668476248358, 0.71467943891780195, 0.23460983080779729, 
0.9759675457699728]], [[0.76584042825128162, -0.18498865556307997, 0.53368040615137491, -0.53440868150446197], 
[-0.056834819149981675, -0.72446265751426742, 0.4194464602624719, 0.40951083698503199], [-0.75230446387658967, 
0.1410139443279983, -0.92127442695342854, 0.36063691669044795]]]]))
      arg0.expand()
      arg1=Data(-0.637578110843,self.functionspace)
      arg1.setTaggedValue(1,0.789204807795)
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.35985067703596413, -0.37847596599552735, 0.16836236519017023, 0.23772738297427734], 
[-0.4684355582534066, -0.036655021314254368, -0.40819340735639148, -0.28010970450450701], [0.19750401683694765, 
0.19568905272465417, -0.51433079807643989, 0.21818132248891869]], [[0.58793143668239334, -0.10786550009577783, 
-0.35250254496450834, 0.60804317097570926], [0.43449811205621908, 0.009480921742783947, -0.33179813902745803, 
0.59546562409481596], [0.39521782021134316, 0.25496170426428849, 0.48670130272395712, 0.47831638002144872]]], 
[[[-0.025960771168379637, 0.15252655135548718, -0.36193635822125397, 0.53077995990750249], [0.53378390475714033, 
-0.10860412197635144, -0.51362013231481596, -0.39943537456976036], [-0.40852145466109102, -0.4797818412057453, 
0.31232586814511626, 0.57949716382676297]], [[-0.0011357733536150739, 0.54054425700201081, 0.1653583150571484, 
0.53408326991479871], [-0.49651429739014052, -0.051228000278312973, 0.55979327598563389, 0.61946390127818796], 
[-0.35572456405070058, 0.23429237469728303, -0.26890421041633072, -0.31355541226227807]]], [[[0.58531366116621131, 
-0.50154950514710039, -0.46787663559769682, 0.60365195437970887], [0.59916588981165764, 0.10738306951199007, 
0.45206466995957023, -0.22277367630320158], [-0.60159301535700049, -0.041441137033658162, 0.057054373316750846, 
0.20560176814304404]], [[-0.53071622168315402, 0.54937711466468664, 0.079391995146109465, 0.18601790833289528], 
[-0.2158780962360593, -0.27390691996330602, -0.21923265809477757, 0.11392657002352397], [-0.45876577855815498, 
0.62250516973021486, -0.035859771905084013, -0.55020698308700977]]]])+(1.-msk_ref)*numpy.array([[[[0.035947202078015314, 
-0.41248899776856313, 0.20773447881440557, -0.35639304430633362], [-0.72428934606475304, -0.13576028563751819, 
0.19917790284703976, -0.041674236606771657], [-0.37463222650496736, 0.62515748256354231, 0.73340125174038051, 
0.37317150183864473]], [[-0.64345321323261417, 0.1809504458501415, 0.15578993416565309, -0.29870870822349715], 
[-0.47777968729931952, -0.22391028268203214, -0.0076241509585390039, 0.38386939065705067], [0.55612408555367254, 
0.7363541216621321, 0.67575586121320452, 0.60463709430722534]]], [[[0.28054708290752428, -0.0081420004829670083, 
0.53718691731468338, -0.36459287708124927], [-0.64344414874503519, -0.60860229896764828, -0.76377329708453601, 
-0.57904172259991815], [0.61369277706090986, 0.42681005048453741, 0.68422932205274145, 0.45178595732169319]], 
[[-0.36966860955018854, -0.47451539136600068, -0.42555689278522874, -0.35141263854994387], [0.25707341030493353, 
-0.33259913626062454, -0.11764143261312256, -0.50437927438695096], [0.23875429356061692, 0.50567728389019639, 
0.17586631516682227, -0.67599633540964688]]], [[[-0.61359660552807682, 0.66618101585541889, 0.20386425585664789, 
0.78710158587440093], [0.072002938224601229, 0.4198681303641702, -0.27097116487217815, 0.29131419948111298], 
[0.49310043955708427, 0.56402844922625073, 0.18515520642951416, 0.77023827937366995]], [[0.60440494797978794, 
-0.14599393635793886, 0.42118314236071941, -0.42175790077077441], [-0.044854312523331906, -0.57174941237829191, 
0.33102916305178914, 0.3231879213927924], [-0.59372229981713753, 0.11128888282981017, -0.72707420705034331, 
0.28461638852051102]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_taggedData_rank1(self):
      arg0=Data(-0.406007736236,self.functionspace)
      arg0.setTaggedValue(1,0.541416506049)
      arg0.expand()
      arg1=Data(numpy.array([0.19901533882497957, -0.21088428820377492]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.55641998509236879, 0.38849978938277641]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-0.080801767192660259, 0.085620652461449995])+(1.-msk_ref)*numpy.array([-0.30125496422455794, 
0.21034019856840283])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([-0.25184069451725555, 0.072658893585815232]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.19027690549287435, -0.81604650565348069]))
      arg0.expand()
      arg1=Data(numpy.array([0.91552659559027161, 0.40574786290714315]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.43926211868276965, -0.72112127257367686]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[-0.23056685368247254, -0.10218382359342712], [0.066521149483977235, 
0.029481190793642063]])+(1.-msk_ref)*numpy.array([[0.083581436643201112, -0.13721272423040279], [-0.35845831701701869, 
0.58846849463614015]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_taggedData_rank1(self):
      arg0=Data(numpy.array([[-0.96750895893843469, 0.63595825573268239, 0.40607489247447615, 0.45984449438550623, 
-0.87406809284057152], [-0.63107539262611789, 0.9518138143519963, -0.76747373136780328, -0.27648775526763769, 
0.17092439701817042], [0.20566312091533256, 0.71073668490023656, -0.39524708168557332, 0.84814189650855032, 
0.44942984085197879], [0.49524467834037833, 0.60294235105664562, 0.27930286291817086, -0.85721838659860072, 
0.50525831417036238]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.70548782871616766, 0.10320633175144311, -0.34818177368673964, -0.10012486240426544, 
0.024860576397676581], [-0.24946991164149734, 0.95419831509027264, 0.085178033587823609, -0.83309865205509093, 
-0.35455930864871532], [-0.7398637408556179, 0.96027512839361306, 0.75310834339201471, -0.12991733024061536, 
0.79813551546053896], [0.7645937641240621, 0.00019025808572403768, 0.2763500544233104, 0.21603829153843468, 
0.7517932867289272]]))
      arg0.expand()
      arg1=Data(numpy.array([0.77155885869570273, 0.97135906170787178]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.1329710793868224, 0.85381625179109721]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-0.74649010813640615, -0.93979859454839776], [0.49067922597121827, 0.61774381457387317], 
[0.31331068058258704, 0.39444452653713208], [0.35479709326558367, 0.44667411659783607], [-0.67439498013440091, 
-0.84903396253040653]], [[-0.48691180968555003, -0.61300080124823264], [0.73438038029222974, 0.92455297362954558], 
[-0.59215115625307468, -0.74549256358686866], [-0.21332657689763532, -0.26856888653048822], [0.13187823268659074, 
0.16602896191055377]], [[0.15868120284923029, 0.19977273616023003], [0.5483751854347938, 0.69038051936605715], 
[-0.30495638724812812, -0.38392683440887304], [0.65439139368214594, 0.82385031678768028], [0.34676157507154409, 
0.43655774851349627]], [[0.38211041879542273, 0.48106040606852668], [0.46520551224056922, 0.58567351638632148], 
[0.21549859814358621, 0.27130336685651679], [-0.66139444001698811, -0.83266684768515253], [0.38983652822779963, 
0.4907872419726243]]])+(1.-msk_ref)*numpy.array([[[-0.093809478078654487, 0.6023569735986779], [-0.013723457332543872, 
0.088119243337125661], [0.046298106269944087, -0.29728325695118812], [0.013313711027352248, -0.085488234729109255], 
[-0.0033057376777776161, 0.021226364157230438]], [[0.033172283425505114, -0.21300146489239946], [-0.12688077990664082, 
0.81471002885575694], [-0.011326215066219917, 0.072726389372891734], [0.11077802699947223, -0.71131316846989323], 
[0.047146133977665192, -0.30272849994808887]], [[0.098380480220743763, -0.63170768605348337], [-0.12768882033081819, 
0.81989851081324938], [-0.1001416293160579, 0.64301614294757259], [0.017275247633148888, -0.11092552794874837], 
[-0.10612894098774574, 0.68146107423187274]], [[-0.10166885810801002, 0.65282258182725295], [-2.5298823020795877e-05, 
0.00016244544562584712], [-0.036746565025274701, 0.23595216764997662], [-0.028726844814750678, 0.18445700432469861], 
[-0.099966764812112308, 0.64189332619660222]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_taggedData_rank1(self):
      arg0=Data(numpy.array([[[-0.34838792046976197, -0.34282193528755522], [-0.26044230436261095, 0.98479784489330791]], 
[[-0.20277493335186136, 0.44804863460952338], [-0.66796628450293238, 0.19247654736068864]], [[-0.17431132009849781, 
-0.37901406562046258], [-0.14850869602200234, 0.73124915753936492]], [[-0.081095403904902996, -0.2322148241916473], 
[0.21488767099231643, 0.65096842451524251]], [[-0.54590513332153656, -0.51554889404233539], [-0.090340168175748969, 
-0.096560548096329857]], [[-0.54303389386544643, -0.12366275726630804], [0.11115383958574476, 
0.41519246419755551]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.87504947191118054, 0.13426761755966798], [0.33807326713064678, 
-0.90944708891369475]], [[-0.95641602687052285, 0.081586862644031921], [-0.57541998957135854, -0.81457382939511391]], 
[[0.28913367556112424, -0.070382091277428449], [0.54812699120038744, -0.66345737344045919]], [[-0.017364919522525391, 
0.72258983242581976], [-0.86281935146061373, 0.76290911465187961]], [[0.49104820353263268, 0.79883423018334554], 
[-0.13788309529101284, 0.56579915514368539]], [[0.25170593703395494, 0.71035895498589374], [-0.78196345209837137, 
0.35076984001043376]]]))
      arg0.expand()
      arg1=Data(numpy.array([-0.96659910970715113, -0.47303458064649773]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.86405103766694302, 0.90890630188408106]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.33675145375879767, 0.16479953386171925], [0.33137137743703343, 0.16216663039516946]], 
[[0.2517432995269786, 0.1231982162267752], [-0.95190472011539251, -0.46584343558068064]]], [[[0.1960020700488361, 
0.095919555563719269], [-0.43308341131906997, -0.21194249798175177]], [[0.64565561591492804, 0.3159711512758438], 
[-0.18604765931834796, -0.091048062865049115]]], [[[0.16848916681908621, 0.082455282204730349], [0.36635465839522691, 
0.17928675958989967]], [[0.1435483733586374, 0.070249748745126087], [-0.70682478465165444, -0.34590613858473823]]], 
[[[0.078386745215821063, 0.038360930378514145], [0.2244586423244489, 0.10984564198139607]], [[-0.20771023146821627, 
-0.10164929933395297], [-0.62922549958390017, -0.30793057570467902]]], [[[0.52767141585316091, 0.25823200581352346], 
[0.49832910199182778, 0.24387245489608181]], [[0.087322726129473255, 0.042734023568549495], [0.093335339822746985, 
0.045676478375743367]]], [[[0.52489607835114815, 0.25687381026147621], [0.11953231107754489, 0.058496760525057666]], 
[[-0.10744120238411238, -0.052579609895690854], [-0.40132466625047536, 
-0.19640039318927668]]]])+(1.-msk_ref)*numpy.array([[[[0.75608740421476606, 0.79533797948040919], [0.11601407427749938, 
0.12203668373894393]], [[0.292112557271689, 0.30727692299358522], [-0.78580870087905852, -0.82660219034378934]]], 
[[[-0.82639226045877012, -0.86929255404555283], [0.070495213327566128, 0.074154813608111533]], [[-0.49719223908343385, 
-0.52300285475148001], [-0.70383336254518358, -0.74037128688706733]]], [[[0.24982625239304665, 0.26279541980441312], 
[-0.060813719001431552, -0.063970726301835326]], [[0.47360969551995413, 0.49819607653479236], [-0.57326103196901323, 
-0.60302058775149348]]], [[[-0.01500417673244102, -0.015783084785733235], [0.62435449451511205, 0.65676645236918962]], 
[[-0.74551995594866205, -0.78422194593008765], [0.65919241216052538, 0.69341290207189832]]], [[[0.42429070980685951, 
0.44631680671966673], [0.69023354551379335, 0.72606546597436139]], [[-0.11913803156292962, -0.12532281423328484], 
[0.48887934711298103, 0.51425841771078451]]], [[[0.21748677608111899, 0.22877711239179935], [0.61378639217156672, 
0.64564973078646903]], [[-0.67565633220322263, -0.71073150945524044], [0.30308304424328286, 0.31881691809635415]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_taggedData_rank2(self):
      arg0=Data(-0.624250642381,self.functionspace)
      arg0.setTaggedValue(1,-0.64725921499)
      arg0.expand()
      arg1=Data(numpy.array([[-0.88904245368897516, -0.73666550638605943, 0.34303327476281864, -0.77024368699478285, 
0.10702276496452101], [-0.74544785860218377, 0.71475983802464027, 0.23731002830074588, -0.47744416459578143, 
0.17433110303665589], [-0.93981462909968205, 0.1233610570121173, -0.9758980848375518, -0.62154673202092425, 
-0.14328483003387538], [-0.96049376638593653, 0.25289910682801331, 0.16420356292509508, -0.30589734924260137, 
-0.83887826027547541]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.38532341634013156, -0.069793396919272555, 0.83512484634491124, 
-0.39040632048761115, 0.58038283656229295], [0.44341644407043801, -0.21124262572279795, 0.41026812410576596, 
-0.02758829249276995, 0.41063816656694296], [-0.6945862172457351, 0.066241259164652844, -0.16246117767968871, 
-0.17880863537623792, -0.1499077811520928], [-0.74066271420455632, 0.018463444397112339, 0.18722276645878178, 
0.66311714214995066, 0.080623148530377664]]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.55498532281923207, 0.45986391558134682, -0.21413874212871248, 0.48082511639632419, 
-0.066809029778482062], [0.46534630459387771, -0.44618928803494795, -0.14814093761016958, 0.29804482644992752, 
-0.10882630305760288], [0.58667988593444154, -0.07700821908460076, 0.60920500635812935, 0.38800094673380958, 
0.089445647192084421], [0.59958885066926859, -0.15787242989494255, -0.10250417963722273, 0.19095661676730766, 
0.52367029285633526]])+(1.-msk_ref)*numpy.array([[0.24940413197739003, 0.045174419301419702, -0.54054225246344312, 
0.25269408852575459, -0.37565813918669533], [-0.28700537950247207, 0.13672873609766126, -0.2655498239439173, 
0.017856776541771296, -0.26578933733685151], [0.44957732971700837, -0.04287526540682994, 0.10515449433122676, 
0.11573553696696945, 0.097029192749322979], [0.47940076696804157, -0.011950634526477415, -0.12118166084627562, 
-0.42920868087406494, -0.052184075827754986]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_taggedData_rank2(self):
      arg0=Data(numpy.array([-0.038797074451863978, -0.041894838192979345]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.68210482130290107, -0.17411820856094695]))
      arg0.expand()
      arg1=Data(numpy.array([[-0.4561153142549883, 0.85342621849792399, 0.60718994051335229, 0.27674980193836829, 
-0.63644344490528737], [0.40481028073282421, 0.11630771806468476, -0.60346126277932322, -0.25296785929946886, 
0.50151586091388278], [0.25426472809468725, -0.82616340686010536, 0.083486142497740889, 0.27986276143702127, 
-0.4541864374287885], [-0.4347704256219993, 0.40105489252397408, 0.44697311819010155, 0.99021709715771133, 
-0.25535139434006404]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.73972177549918428, 0.11505004284707909, 0.87601891306551627, 0.63351125241249773, 
0.31850486312185233], [-0.15371510251020615, -0.055161665331776355, -0.13904651389325351, -0.28809668374954822, 
0.71414169686372975], [0.022587269591964443, -0.58005792020390401, -0.81804413538647136, -0.020056052647248857, 
0.31628882376473832], [0.51570671136502333, -0.68989720546033007, -0.8884104872841736, 0.33371316933670303, 
-0.38782662563571146]]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.017695939805786116, -0.033110440538236689, -0.023557193328519389, -0.010737082670341484, 
0.024692143716391224], [-0.01570545460047134, -0.0045123991970819795, 0.023412531540865256, 0.0098144128711701445, 
-0.01945734819466657], [-0.0098647275863725323, 0.032052723205357098, -0.0032390180861837786, -0.01085785639177636, 
0.017621105027951566], [0.016867820572325297, -0.015559756524536929, -0.017341249344403165, -0.038417526441936356, 
0.0099068870575987431]], [[0.019108877288052661, -0.035754153333616763, -0.025438124310211647, -0.011594388172147021, 
0.026663695143289378], [-0.016959461210156214, -0.0048726930289146285, 0.025281911959870736, 0.010598047533375612, 
-0.021010925844199852], [-0.010652379641708811, 0.034611982251364679, -0.0034976384312988711, -0.011724805106644386, 
0.01902806730552483], [0.018214636632526423, -0.016802129828794615, -0.018725866463185737, -0.041484985061344022, 
0.010697905348228644]]])+(1.-msk_ref)*numpy.array([[[-0.50456778949073577, -0.078476188917097997, -0.59753672415451564, 
-0.4321210796202038, -0.21725370274383604], [0.10484981252928129, 0.037626037873901744, 0.094844297511949033, 
0.196512136986944, -0.4871194945241849], [-0.015406885488747357, 0.39566030400601637, 0.5579918487856752, 0.013680330206993257, 
-0.21574213161415159], [-0.35176603420034602, 0.47058221004788925, 0.6059890766725945, -0.22762736173683659, 
0.26453841117575405]], [[-0.12879903038344095, -0.020032307355393599, -0.15253084380847562, -0.11030584437326599, 
-0.055457496184726549], [0.026764598277839415, 0.009604650348807392, 0.024210529905738121, 0.050162878466821013, 
-0.12434507291658745], [-0.0039328549176359997, 0.10099864592749248, 0.14243637937728115, 0.0034921239577430088, 
-0.055071643381765303], [-0.089793928725735203, 0.12012366550595623, 0.15468844251267824, -0.058105539218102666, 
0.067527677287927107]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[-0.54411798718058924, 0.78539703813148121, -0.076792090912178423, -0.68244583960681493, 
0.47476506351236347], [-0.87732703346721896, 0.67267403988342789, -0.79817867718719615, -0.63797289066597962, 
0.67366877265140368], [-0.20152644386739604, -0.41129959167878627, -0.054078501978257698, 0.5169896409595196, 
0.079752425650144509], [0.66023787880068263, -0.044906491353615463, 0.22655095753782839, 0.18627434712372049, 
0.73716849589931632]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.10098653545926317, -0.94282388393543926, 0.32525603743121279, 0.3624832150884485, 
0.88651068707151159], [-0.69445136023806575, -0.096201506029467865, 0.84070809125685053, -0.025878610196192309, 
-0.26624121301602632], [-0.060859104863993041, -0.95478043235456833, 0.71327743155813716, -0.51728396052491576, 
-0.811352259303604], [-0.40573550704327443, 0.28160415449871512, 0.95401452602899584, 0.61858029046980634, 
-0.9559952892639576]]))
      arg0.expand()
      arg1=Data(numpy.array([[-0.8653356669508061, 0.17999164902653608, 0.20829974854944777, 0.50074680171470587, 
0.4547159383673367], [-0.20216263534058698, 0.96497885287416874, -0.42248221004317665, -0.3844675884914861, 
0.41152097123174047], [0.92437882176188446, -0.4670201227751738, 0.78734602231258433, 0.1524644887015727, 
-0.35134237891812226], [0.24034128439796532, -0.19913690473844237, -0.66694476422097004, 0.44943760033129854, 
-0.80210244516378504]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.96335452455346204, -0.83797028493831505, -0.34702045640630996, 0.34916893996467024, 
0.13830791614213656], [-0.20697762229357308, -0.12329627848171887, -0.3624601805529104, 0.48225813480968038, 
0.49014749700380511], [-0.17115640458950754, -0.46431510005350507, 0.45780863162704089, -0.63407853875737974, 
-0.09404280432444323], [-0.82178275814769308, -0.40209299340984317, -0.19450733845978241, 0.0049325651031069118, 
-0.23052582657927023]]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.47084470133684536, -0.097936693777633879, -0.11333963991094838, -0.27246534183612336, 
-0.24741912112336811], [0.11000032622464365, -0.5250623510977267, 0.22988016974830019, 0.20919573038616249, 
-0.22391596254921581], [-0.5029711438894412, 0.25411404917725922, -0.42840913287536669, -0.08295867070881742, 
0.19117170802816857], [-0.13077401590301843, 0.108353971779654, 0.36289664266854688, -0.2445470824555403, 
0.43643836797514768]], [[-0.67963206981269297, 0.14136490803384255, 0.16359800555426857, 0.39328505492054211, 
0.35713255118488341], [-0.15877793501735171, 0.75789153290688649, -0.33181627643115325, -0.30195970525876631, 
0.32320735193439948], [0.72600438872325246, -0.36679622117542221, 0.61837923390890681, 0.11974515784644588, 
-0.27594326377236178], [0.18876333290687794, -0.15640153516424357, -0.52381644241644898, 0.35298696012512226, 
-0.62996888470965562]], [[0.066450935206036857, -0.013821935075478671, -0.015995773227593098, -0.038453393921258273, 
-0.034918587678321039], [0.01552449147211993, -0.074102743798242818, 0.032443292282413681, 0.029524070008224202, 
-0.031601555835095774], [-0.070984982518031009, 0.035863451725967874, -0.060461947324770039, -0.01170806687724997, 
0.026980315903181486], [-0.018456309761438281, 0.01529213929264428, 0.051216082967458137, -0.034513253063992388, 
0.06159512388989799]], [[0.59054472577396599, -0.12283455204212956, -0.14215329678871633, -0.34173257152661973, 
-0.31031900034169779], [0.13796504941213325, -0.65854580345253322, 0.28832122655185843, 0.26237830622967967, 
-0.28084077472805707], [-0.63083848113204755, 0.31871593980058127, -0.5373210172581977, -0.10404875600216854, 
0.23977214477023365], [-0.16401990962314972, 0.13590015215092863, 0.45515367959014913, -0.30671682050896515, 
0.54739147664047849]], [[-0.41083114287941286, 0.085453746681778436, 0.098893443349687915, 0.23773708711969521, 
0.21588324135905257], [-0.095979756407300554, 0.45813824637289235, -0.20057979328399245, -0.18253177906860563, 
0.19537578004350678], [0.43886277002326479, -0.22172483825090716, 0.37380438428944085, 0.072384812661782191, 
-0.16680508684164719], [0.11410564515184302, -0.094543245225802069, -0.31664207334460714, 0.21337727086613317, 
-0.38081021832160644]]], [[[0.75918237363932806, -0.15791153948932374, -0.18274700046685466, -0.43931870606656059, 
-0.39893458527807829], [0.17736274514127234, -0.84660203435069437, 0.37065506402985465, 0.33730380887553102, 
-0.36103847290029162], [-0.81098252949627725, 0.40972937888383959, -0.6907599500677144, -0.13376121758164711, 
0.30824216702755175], [-0.21085790606056812, 0.17470818988802181, 0.58512867148047742, -0.39430375662728373, 
0.70370615875234621]], [[-0.58208883894301922, 0.12107570969596008, 0.14011783336345923, 0.33683937406813702, 
0.30587560726094021], [-0.13598955662803289, 0.64911622336494312, -0.28419281500862259, -0.25862136595480728, 
0.27681947421520675], [0.62180563641724995, -0.31415231269403066, 0.52962722961515363, 0.10255890355364815, 
-0.23633889740910741], [0.16167134272675121, -0.13395422620028935, -0.44863642892762023, 0.30232500629036801, 
-0.53955349218869897]], [[0.69069247796969457, -0.14366549632474265, -0.16626041775562381, -0.39968541979836314, 
-0.3629445661819754], [0.16136190485282723, -0.77022554430072199, 0.33721629154738592, 0.30687383120348566, 
-0.32846726445254082], [-0.73781946517375996, 0.37276550381649015, -0.62844280657805918, -0.12169390390984351, 
0.28043399524466944], [-0.19183528845423967, 0.15894683120328262, 0.53234108966282034, -0.35873150931062364, 
0.6402210686494455]], [[0.55206069684097914, -0.11482979262519569, -0.13288959270708789, -0.31946288458167504, 
-0.29009644163210324], [0.12897428085288662, -0.61563034819967444, 0.269532196796197, 0.2452798987972917, 
-0.26253922358638493], [-0.5897286289898418, 0.2979461777260583, -0.5023054178091203, -0.097268210580852929, 
0.22414691309185639], [-0.15333122395374424, 0.1270439467542599, 0.42549267914459249, -0.28672900505733978, 
0.51171961555139034]], [[-0.58294961668623335, 0.12125475328720878, 0.14032503594890247, 0.3373374833202617, 
0.30632792810495502], [-0.13619065442586653, 0.65007611945030075, -0.28461307190683938, -0.2590038084633044, 
0.27722882761000023], [0.6227251463212794, -0.31461687291345919, 0.53041042850328335, 0.10271056497651228, 
-0.23668838918619584], [0.16191041807783926, -0.13415231420474596, -0.44929986073902067, 0.30277207659857797, 
-0.54035136977417686]]], [[[0.17438801971221735, -0.03627307695414627, -0.041977907583642997, -0.10091372222753676, 
-0.091637286028995402], [0.040741116983049656, -0.19446875662697038, 0.085141337387239668, 0.077480385890962594, 
-0.082932357909189652], [-0.1862867767360061, 0.094116904557395464, -0.15867104396979459, -0.030725626224088729, 
0.070804780203280362], [-0.048435124359244429, 0.040131352254698702, 0.134407006589431, -0.090573561335062613, 
0.16164485339120063]], [[0.35591220648195676, -0.074030491750205693, -0.08567360152518172, -0.20595695507971667, 
-0.18702447978032172], [0.083149409368290794, -0.39689540816580915, 0.17376676048230977, 0.15813136216027587, 
-0.16925840743487242], [-0.38019663194718062, 0.1920851858032056, -0.3238350974870825, -0.062708581948471773, 
0.14450697698847709], [-0.098852272136438185, 0.081904927607098707, 0.27431410919638938, -0.18485350150135663, 
0.32990440818042083]], [[0.046796056577056114, -0.0097336787479513961, -0.011264538364001892, -0.027079636907134937, 
-0.024590356772543322], [0.010932652475195722, -0.052184610804132579, 0.022847205031598612, 0.020791431244812798, 
-0.022254437656850208], [-0.049989021941309587, 0.025255748633383389, -0.042578493425204422, -0.0082450511538620479, 
0.01900006953336944], [-0.012997296623772364, 0.01076902549684197, 0.036067373751312345, -0.024304912158619522, 
0.043376498667555087]], [[-0.44736957576636366, 0.093053818005940758, 0.10768881221453722, 0.25888090923011353, 
0.23508342971510043], [-0.10451598826016036, 0.49888407068094559, -0.21841892608200625, -0.19876576053478573, 
0.2127520791644103], [0.47789427517326044, -0.24144456559440786, 0.40704973738628886, 0.078822561272902797, 
-0.18164037033074351], [0.12425395432865388, -0.10295171688251738, -0.34480353419443077, 0.23235458362898609, 
-0.41467865513797797]], [[-0.069012618440912374, 0.014354770606635724, 0.016612410209133632, 0.039935772073299736, 
0.036264699066576714], [-0.016122960544237439, 0.076959404217808877, -0.03369398104497718, -0.030662222766057599, 
0.032819795661634642], [0.073721453255132879, -0.037245987618748408, 0.062792755105421399, 0.012159412799459474, 
-0.028020406952412445], [0.019167800414608964, -0.015881651189352534, -0.053190462721286073, 0.035843738804801246, 
-0.063969615621723874]]], [[[-0.57132738519817416, 0.11883730455511714, 0.13752738413700297, 0.33061200618034342, 
0.30022068660451234], [-0.13347542953002506, 0.63711559090915715, -0.27893875818993141, -0.25384006509323254, 
0.27170173312804108], [0.61030991248834088, -0.30834437521831509, 0.51983566765381561, 0.10066283061275701, 
-0.23196954698968672], [0.15868241979914421, -0.13147772757544279, -0.44034219640647465, 0.2967357278960055, 
-0.52957841697577834]], [[0.038859188645901446, -0.008082793430733131, -0.0093540108571960514, -0.022486781921552037, 
-0.020419697354643947], [0.0090784146359461845, -0.043333814513015628, 0.018972193712360263, 0.017265090438338307, 
-0.018479962936449591], [-0.041510609566915317, 0.020972235105367776, -0.035356947343283596, -0.0068466452436105772, 
0.015777553501045344], [-0.010792883809734064, 0.0089425396908226085, 0.029950149287828094, -0.020182665713267139, 
0.036019606518461332]], [[-0.19604262393934049, 0.040777280435774485, 0.04719050748876627, 0.11344466741247171, 
0.10301633124483228], [-0.045800138614780807, 0.21861688312239816, -0.095713749227979608, -0.08710150031500595, 
0.09323047007944793], [0.20941890719784453, -0.10580385600414979, 0.17837399526851638, 0.034540975905856711, 
-0.079596952367519125], [0.054449548116230574, -0.045114656449613433, -0.15109697495910196, 0.1018205187085595, 
-0.181717076995289]], [[-0.16118983640413065, 0.033527826910139849, 0.038800899667083527, 0.093276283563697951, 
0.084701914546125565], [-0.037657712910878625, 0.17975080580733252, -0.078697597847179279, -0.071616449036482802, 
0.076655800243911817], [0.17218806151868901, -0.086993868463585283, 0.1466623662667349, 0.028400223092437313, 
-0.065446072249868037], [0.044769415838107417, -0.03709409691839187, -0.12423470052284488, 0.083718695574564253, 
-0.14941110929922388]], [[-0.63789819205415743, 0.13268417318732925, 0.15355201233440222, 0.36913476664642292, 
0.33520226434769584], [-0.14902792582106247, 0.71135200954789868, -0.31144057532174757, -0.28341739393030613, 
0.30336029539392795], [0.68142294567939055, -0.34427252146088894, 0.58040668302047738, 0.11239201781419665, 
-0.25899853301275982], [0.17717202312215791, -0.146797452544083, -0.49165066868869667, 0.33131123983682143, 
-0.59128465305855127]]]])+(1.-msk_ref)*numpy.array([[[[0.097285835853659811, -0.084623715893732021, -0.035044393625965509, 
0.035261361537015506, 0.013967237277784671], [-0.020901952993023897, -0.01245126399888929, -0.036603597875977417, 
0.04870157823147591, 0.049498297586443855], [-0.017284492321158296, -0.04688957331582462, 0.046232507611360912, 
-0.064033394838179908, -0.0094970569935989343], [-0.0829889936454931, -0.040605978336904401, -0.01964262223245572, 
0.00049812266069003029, -0.023280004560123427]], [[-0.90827365444627362, 0.79005839866802896, 0.32717917451404593, 
-0.32920481612711061, -0.13040000667614624], [0.19514344573854894, 0.11624667615291971, 0.34173611520083558, 
-0.45468448772072351, -0.46212276682636161], [0.16137034613550494, 0.43776736600231775, -0.43163291216977545, 
0.59782439063134074, 0.088665802029352092], [0.77479641178798575, 0.37910287774989532, 0.18338616430059709, 
-0.0046505401882756692, 0.21734525516289507]], [[0.31333687529768905, -0.27255489436414065, -0.11287049855828729, 
0.11356930580696567, 0.044985484749759809], [-0.067320721264141833, -0.040102858968979187, -0.11789236205324156, 
0.15685736994716426, 0.15942343263228489], [-0.055669653937756663, -0.15102128956288016, 0.14890502142481712, 
-0.206237872936399, -0.030587989883487328], [-0.26728980354441134, -0.13078317371534035, -0.063264686158720565, 
0.0016043465798080356, -0.074979916878728389]], [[0.34919984533014264, -0.30375016303302371, -0.12578909073962002, 
0.12656787996741911, 0.050134298115385181], [-0.075025913980336909, -0.044692831432494147, -0.13138573158835851, 
0.17481047920837137, 0.17767024058149494], [-0.062041323818583974, -0.16830643028150916, 0.16594794468741295, 
-0.22984282734736042, -0.034088938067458031], [-0.29788245627762866, -0.14575196101573828, -0.070505645403198952, 
0.0017879720572072778, -0.083561742779375989]], [[0.85402408145533903, -0.74286961304617605, -0.30763734323662739, 
0.3095419968721112, 0.12261144576659448], [-0.1834878741479033, -0.10930346855018902, -0.32132482369802473, 
0.42752699043595543, 0.43452099433522495], [-0.15173198182933395, -0.41162029836611042, 0.40585224457095659, 
-0.56211740105110486, -0.083369951075793883], [-0.72851919754903327, -0.35645973585440083, -0.17243283425843275, 
0.0043727716785802694, -0.20436360890851699]]], [[[-0.66900285996764686, 0.58192960421449247, 0.24098882798179636, 
-0.24248084531134878, -0.096048120496599068], [0.14373589134061243, 0.08562326830392103, 0.25171096541710353, 
-0.33490481770445502, -0.3403835960115757], [0.11885979798064022, 0.32244425281123018, -0.31792582696212612, 
0.44033670373782746, 0.065308153383711848], [0.57068815421585484, 0.27923402621566118, 0.13507588576968174, 
-0.0034254265453154101, 0.16008897383797865]], [[-0.092676156102345031, 0.080614003419008221, 0.033383890529320316, 
-0.033590577883314127, -0.013305429828670886], [0.019911558979040095, 0.011861287677770027, 0.034869215244902822, 
-0.046393958863653391, -0.047152927388340142], [0.016465503888099552, 0.044667811897370242, -0.044041879835811204, 
0.060999310369424242, 0.0090470594072459918], [0.079056738962858031, 0.038681951529923812, 0.018711898893614503, 
-0.00047452019150728234, 0.02217693169561373]], [[0.80989994354099182, -0.70448839878045, -0.29174290553242993, 
0.29354915304387574, 0.11627658418556817], [-0.17400776177131119, -0.10365617894143896, -0.30472320654925072, 
0.40543831600893532, 0.41207096664039183], [-0.14389257420883014, -0.39035346150771583, 0.38488342085608018, 
-0.53307495802564964, -0.079062546520044158], [-0.69087941403013708, -0.33804283299734261, -0.16352389325197386, 
0.0041468473928331622, -0.193804927648866]], [[-0.024930276221657218, 0.021685506359910855, 0.0089804071214436409, 
-0.0090360068899633748, -0.003579216648890006], [0.005356293206670101, 0.0031907363294695762, 0.0093799657241702521, 
-0.012480170284682479, -0.012684336013600811], [0.0044292898769536454, 0.012015829482490688, -0.011847451122328389, 
0.016409071338273448, 0.0024336970748690548], [0.021266595664055928, 0.010405607839073454, 0.0050335795922995539, 
-0.0001276479295706449, 0.0059656880061999627]], [[-0.25648467718159107, 0.2231022251333622, 0.092391147254991049, 
-0.092963162123713874, -0.036823267363401281], [0.055105973226613827, 0.032826550743334619, 0.096501838140414783, 
-0.12839699079857567, -0.13049746415906219], [0.045568888773372264, 0.12361981545990282, -0.12188752541359052, 
0.16881783930619423, 0.025038070298768571], [0.21879243836489759, 0.10705372631068172, 0.051785869732051258, 
-0.0013132521163317051, 0.061375475699987025]]], [[[-0.058628894031001307, 0.050998121443971041, 0.021119354346382343, 
-0.021250109132559158, -0.0084172959720146441], [0.012596472819664507, 0.007503701141459017, 0.022059002137291426, 
-0.029349798397896032, -0.02982993791897829], [0.01041642557505686, 0.028257801364091686, -0.027861823519831241, 
0.03858945228224285, 0.0057233608900852688], [0.050012963053531888, 0.024471019651006509, 0.011837542508140085, 
-0.00030019149685845618, 0.014029595453646482]], [[-0.91979204946388415, 0.80007763095368523, 0.33132834140349626, 
-0.33337967146425418, -0.13205369197224853], [0.19761818370117826, 0.11772087407648481, 0.34606988789962267, 
-0.46045063046009432, -0.46798323910680251], [0.16341678597422343, 0.4433189719778402, -0.43710672324051941, 
0.6054057813815239, 0.089790229372727973], [0.78462209712578401, 0.38391052209459264, 0.1857118007107674, 
-0.0047095166417614734, 0.22010154837024987]], [[0.68713904095340383, -0.59770529256284166, -0.24752185984362526, 
0.24905432467787739, 0.09865191519002138], [-0.14763246681957004, -0.087944452836117251, -0.25853466662687857, 
0.34398384374506669, 0.34961114774752383], [-0.1220820006603293, -0.33118548199982356, 0.32654456491208106, 
-0.45227391153100055, -0.067078609925063343], [-0.58615909503034835, -0.28680385758689592, -0.13873769479580286, 
0.0035182873677373962, -0.16442886949027841]], [[-0.49832784385061207, 0.43346858779508379, 0.17950811607301989, 
-0.18061949215721115, -0.071544466633952322], [0.10706620420004959, 0.063779187251006481, 0.18749483772898554, 
-0.24946439796971021, -0.25354543849150257], [0.088536462835265325, 0.24018275388719962, -0.23681706213052792, 
0.32799865781226867, 0.048646834279817669], [0.42509503982572766, 0.20799625613036252, 0.10061552638963651, 
-0.002551536812082133, 0.1192473125762048]], [[-0.78161987000680067, 0.67988908391398672, 0.28155583132982742, 
-0.28329900831897969, -0.11221644024149589], [0.16793176147317851, 0.10003671410986893, 0.29408288639919611, 
-0.39128122724537623, -0.39768227908604376], [0.13886813555797867, 0.37672310545719029, -0.37144406759929099, 
0.5144610549967279, 0.076301841759883748], [0.66675529747987794, 0.32623905865322539, 0.15781396851047524, 
-0.0040020478405679068, 0.18703765022292171]]], [[[-0.39086713648213151, 0.33999429844664442, 0.1407985208344027, 
-0.14167023690032812, -0.056116432484028457], [0.083978170527894219, 0.050025678066328974, 0.14706296513963191, 
-0.19566924885274947, -0.1988702432228307], [0.069444230599827669, 0.18838912254805759, -0.18574921728198507, 
0.25726817742798402, 0.038156504896349412], [0.33342644405647481, 0.16314340455969073, 0.078918533593617612, 
-0.002001316803133044, 0.093532513133710152]], [[0.27128463636938938, -0.23597591358510162, -0.097722402220057139, 
0.098327424115963583, 0.03894808378568556], [-0.058285758326136056, -0.034720744254682565, -0.10207029268405396, 
0.13580589430320741, 0.13802757147341804], [-0.048198354601468277, -0.13075306117155361, 0.12892081263154659, 
-0.17855915079255269, -0.026482844398472944], [-0.2314174387898032, -0.11323105743903632, -0.054774074590762438, 
0.0013890308253702896, -0.064917030483972821]], [[0.91905421013975974, -0.79943582421180925, -0.33106255624083158, 
0.33311224076444179, 0.13194776106439851], [-0.19745965823101164, -0.11762644067687611, -0.34579227735456908, 
0.46008126590408482, 0.4676078320383838], [-0.16328569620128608, -0.44296335010565041, 0.43675608471365457, 
-0.60492013661777988, -0.089718201394021313], [-0.78399268851307236, -0.38360255652747166, -0.18556282630987078, 
0.0047057387589477051, -0.21992498718146497]], [[0.59591212162368268, -0.51835190226220929, -0.21466001472277998, 
0.21598902430638009, 0.085554550941476445], [-0.1280322777191083, -0.076268647757067792, -0.22421072377015777, 
0.29831537711199912, 0.30319558106966232], [-0.10587397846674526, -0.28721616946061435, 0.28319139633143953, 
-0.39222848668521032, -0.058173025215609256], [-0.5083386172380786, -0.24872680065933472, -0.12031840592296114, 
0.0030511875542411039, -0.14259873276619719]], [[-0.92096238736422931, 0.80109564494420549, 0.33174992160266087, 
-0.3338038617635144, -0.13222171629979704], [0.19786963189571055, 0.11787066141230031, 0.34651022515434587, 
-0.46103650508727706, -0.46857869818015746], [0.16362471651492522, 0.44388304838527398, -0.43766289521982954, 
0.60617609607542877, 0.089904477923339873], [0.78562044558753674, 0.3843990075458536, 0.1859480992948222, 
-0.0047155090025579952, 0.22038160426346237]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_taggedData_rank3(self):
      arg0=Data(0.469364707777,self.functionspace)
      arg0.setTaggedValue(1,0.703530101597)
      arg0.expand()
      arg1=Data(numpy.array([[[0.27156038466300214, 0.90923620326573751], [-0.7703042854947959, 0.5509668079621699]], 
[[-0.43516727007889933, 0.83768282177669273], [0.94335181796930145, -0.01928605569671582]], [[0.35777445354397841, 
-0.2513972484878968], [-0.35837872689554051, 0.17854403413566011]], [[-0.34573491894994102, 0.57740777268969756], 
[0.021658895631055275, 0.49566894734811329]], [[-0.27141022718799235, 0.13476011301633339], [0.54045242044536201, 
0.65721223329364209]], [[-0.36191557633651783, 0.66217326482908501], [-0.60449183654943295, 
0.045917715894456146]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.60517407908845255, 0.26585360179474971], [-0.37142793424603693, 
-0.80231921440483855]], [[-0.78808539509049225, 0.53183962164176868], [-0.69055930833302681, -0.60110494133225845]], 
[[0.27297368247410669, -0.43011599574901482], [0.91867857847971979, -0.27812134947823108]], [[-0.89441057171053306, 
-0.18183020220843016], [-0.17252699913478642, -0.61397362451656012]], [[0.3741759449930977, 0.22025992749228207], 
[-0.55679922710391927, 0.58395033015884623]], [[-0.73984185693861781, -0.071550562847045418], [-0.55179746264340346, 
0.14444952290202351]]]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.12746086059114892, 0.42676338484605575], [-0.36155364586060507, 0.25860437481396847]], 
[[-0.20425215855468012, 0.39317875285299686], [0.44277605037202539, -0.0090521938962592006]], [[0.16792670183773106, 
-0.11799699607245355], [-0.16821032642280442, 0.083802268407403721]], [[-0.1622757692012301, 0.27101483049664538], 
[0.010165921218641941, 0.23264951062616071]], [[-0.12739038197177041, 0.063251641065901462], [0.25366929238968822, 
0.30847222782731376]], [[-0.16987039872711984, 0.31080076094421921], [-0.2837271342155826, 
0.021552155302586893]]])+(1.-msk_ref)*numpy.array([[[-0.42575818134507132, 0.18703601148063317], [-0.26131073231614044, 
-0.56445571842359576]], [[-0.55444179807525795, 0.37416518304703267], [-0.48582926035040414, -0.42289542044604322]], 
[[0.19204520256436125, -0.30259955018787127], [0.64631803365297869, -0.19566674125476125]], [[-0.62924476038509203, 
-0.12792302063313038], [-0.12137793722955073, -0.43194892643411675]], [[0.2632440405962102, 0.15495948916642996], 
[-0.39172501681364469, 0.41082663510435263]], [[-0.52050101677786287, -0.050337974749116388], [-0.38820612495457291, 
0.10162458752292299]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_taggedData_rank3(self):
      arg0=Data(numpy.array([0.031778298448050135, 0.16960568803667719]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.63445706697954707, -0.83292144601314999]))
      arg0.expand()
      arg1=Data(numpy.array([[[-0.48512442486872853, 0.75604336762743518], [0.92128271104607751, -0.2247384614186867]], 
[[0.6181281690820386, -0.33743913620027888], [-0.16881876686658037, -0.51574203611038372]], [[-0.78370822903806214, 
-0.99435576069490428], [-0.77818383505917921, 0.27203764168961997]], [[0.59917935754162577, -0.81661690669992248], 
[0.71416738114277312, -0.081985527439166983]], [[0.93800438152969412, 0.72501077275078196], [0.027427340110406151, 
-0.92472538561007256]], [[-0.95923164159982388, -0.67134072131568812], [-0.78261689772649734, 
-0.93513665442476213]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.15411072364017842, 0.32835878187937007], [-0.23602267728594128, 
-0.27039985978140169]], [[-0.90587643176845267, 0.17687057959430952], [0.33507256115354678, 0.87364686894440768]], 
[[-0.19800899512028081, 0.75046581132567769], [0.86592287250563027, 0.019205119393878922]], [[-0.70496425764287896, 
0.95154267476642551], [0.50263365183843156, -0.52711075384753325]], [[0.26460127781398968, -0.84645788507146946], 
[-0.21309508989448056, 0.90847875937234956]], [[-0.50211985046252372, -0.52108642131180316], [0.24885495065421281, 
-0.62635152044504872]]]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-0.015416428757917129, 0.02402577177613352], [0.029276796946650988, -0.0071418058997186271]], 
[[0.019643061436235818, -0.0107232415782247], [-0.0053647731571179885, -0.016389404345720823]], [[-0.024904913998564371, 
-0.031598934126900589], [-0.024729358157958816, 0.0086448933667164687]], [[0.019040900447868724, -0.02595069577883365], 
[0.022695024179817416, -0.0026053605593826519]], [[0.029808183181829307, 0.023039608714525803], [0.00087159419966466301, 
-0.029386199286405131]], [[-0.030482749387572266, -0.021334065802299192], [-0.024870233346439761, -0.029717051694021213]]], 
[[[-0.082279861863258016, 0.12822925555201761], [0.15625478808326523, -0.038116921377220586]], [[0.10483805341201069, 
-0.057231596865750328], [-0.028632623107909767, -0.087472782883938449]], [[-0.13292137340600632, -0.16864839294589279], 
[-0.1319844047642322, 0.046139131390643053]], [[0.10162422719322164, -0.13850287232322339], [0.12112685005207191, 
-0.013905211790369793]], [[0.15909087851076162, 0.12296595094639939], [0.0046518328904413889, -0.15683868527137798]], 
[[-0.16269114256008949, -0.11386320494578644], [-0.1327362774080324, 
-0.15860449568202822]]]])+(1.-msk_ref)*numpy.array([[[[0.097776637710843148, -0.20832954966816197], [0.14974625557149848, 
0.17155710194858889]], [[0.57473970394571028, -0.11221678916437815], [-0.21258915437480422, -0.55429143004633363]], 
[[0.1256282062795808, -0.4761383375221156], [-0.54939088592042651, -0.012184823721632437]], [[0.44726955522951473, 
-0.6037129745381794], [-0.31889947251063011, 0.33442914285948394]], [[-0.16787815064090419, 0.53704118708415505], 
[0.13519968572219507, -0.57639076908459863]], [[0.3185734875966616, 0.3306069625083552], [-0.15788778209541179, 
0.39739314855974539]]], [[[0.12836212678051034, -0.27349707141408142], [0.19658834965690128, 0.2252218422108781]], 
[[0.75452390745781217, -0.14731929891287623], [-0.27908912215534182, -0.72767921338603703]], [[0.16492593853919504, 
-0.62507906875281527], [-0.72124573110325008, -0.015996355816404823]], [[0.58717984886349361, -0.79256030060967164], 
[-0.4186543481041366, 0.43904185130376894]], [[-0.22039207893375551, 0.70503292562296105], [0.17749147041321292, 
-0.7566914419286499]], [[0.4182263919191519, 0.43402405553684459], [-0.20727662534643804, 0.52170161412162508]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(2,6,2,2),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_taggedData_rank4(self):
      arg0=Data(-0.7253342588,self.functionspace)
      arg0.setTaggedValue(1,-0.462587611348)
      arg0.expand()
      arg1=Data(numpy.array([[[[0.43772063448462406, -0.49671650839169157, 0.91196011939063948, 0.61523214447528085], 
[-0.54249666994400725, -0.64387013072952426, -0.42587693629363921, -0.32525112033964354], [-0.32947321634172066, 
-0.74442526061845959, 0.53173519790910984, 0.97770475306402593]], [[0.56404838605243302, 0.08669045715493362, 
-0.56106007633015809, -0.0046657403875491887], [-0.79600006472409901, 0.24368196324932745, -0.93949854148128176, 
0.69427108060660947], [0.025120273208480315, 0.28831281256218522, 0.74561765120572576, -0.49630122873532301]]], 
[[[0.82974033575777129, 0.26282454130732114, -0.48184378498397051, 0.72545008929058996], [-0.416297585829428, 
-0.13039490497798445, 0.93256800180857757, 0.21392674964611325], [0.2915494940098835, 0.30387154279393225, 0.54418892926884888, 
0.6853194676787977]], [[-0.65850448757046487, 0.96389340191765616, 0.75282104550310058, 0.55017843981280823], 
[0.26375562223679339, -0.53673795425173121, -0.45324515088409334, 0.88480927453368463], [0.63043944073072433, 
-0.80316226594481166, -0.15776749806406776, 0.45329597203192584]]], [[[0.024990344044336421, -0.43780571674507707, 
-0.12911144658316087, -0.42973923140461201], [-0.97975027576664164, -0.31290502647004281, 0.53474605448971624, 
-0.36635144178718004], [-0.83403790506215469, 0.71363695931217919, 0.1713710838897573, 0.5594944150070047]], 
[[-0.78225600419531527, -0.69006666611045819, -0.82052692632754121, -0.52942526668076306], [-0.21847162414653254, 
0.87999416009247189, 0.82594634249676324, -0.79979798200198715], [-0.049001508207088484, 0.35855338695267802, 
0.23904219502739954, -0.92930682776816753]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.44729772760503694, 0.59950563622932718, -0.23150850482133878, 
0.57695986712793434], [-0.76863110765311338, -0.06219258238487213, 0.18914928562440836, 0.63516509075853333], 
[-0.72486528388148819, 0.46750990061818132, -0.617135386617923, -0.75375159572628592]], [[0.13314600710095381, 
-0.58436400515818354, -0.33201466069376373, -0.77053742977647466], [0.66303945805071196, 0.16348430423896887, 
-0.80719279071691852, 0.95887161373568963], [-0.56791351560663639, -0.59410364169793795, 0.64584851230674034, 
-0.0091260216901081215]]], [[[-0.31610041572915115, -0.58536109928280644, 0.69184728371096105, 0.16903186952028282], 
[-0.86404856349688086, 0.25836851100086866, 0.13799680048340179, -0.051695358243066591], [-0.119291874175671, 
0.17424335671077462, -0.40417053966762451, -0.49177388552669798]], [[0.18325228305450847, -0.8222595718096648, 
-0.78632194875044492, 0.3838788766856307], [-0.69463224336093066, -0.59833810122109199, 0.58945715407459187, 
0.69077295833678454], [0.054068763748110982, -0.29354453628315524, -0.53787926328683455, -0.33717483039158247]]], 
[[[0.25173721986879571, 0.77719986302990662, 0.95965914714441669, 0.67963899304005304], [-0.82880656850840961, 
0.77644142920704029, -0.20550957818075966, -0.26870420405821394], [0.48065607213209471, 0.27092635315563118, 
0.075039498182671194, -0.54615588761313694]], [[0.16050484090523964, -0.85982065723083689, 0.69159775628780129, 
-0.32848888524985331], [-0.089559446727719427, -0.16235278676414389, -0.89382032954770474, 0.11115761508627142], 
[0.10629586453397666, 0.8639040144821537, 0.83779647080305208, -0.53547234476697447]]]]))
      res=outer(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-0.31749377197523726, 0.3602855004478604, -0.66147591725309141, -0.44624895150272509], 
[0.39349141999513959, 0.46702106403596261, 0.30890313192643198, 0.23591578029532595], [0.23897821116957371, 
0.53995714464246058, -0.38568575565311364, -0.70916275238863469]], [[-0.40912361802450609, -0.062879558485480547, 
0.40695609460703586, 0.0033842213457547952], [0.57736611695116413, -0.17675087619630558, 0.68145047822872062, 
-0.50357859965785878], [-0.01822059474851892, -0.20912316020224819, -0.5408220263852751, 0.3599842838861137]]], 
[[[-0.60183909143307357, -0.19063564386351575, 0.34949780463858815, -0.52619380281176309], [0.30195490085769083, 
0.094579891753463083, -0.67642352037213782, -0.15516840039199159], [-0.21147083614108514, -0.22040844026275683, 
-0.39471887365822045, -0.49708568812980269]], [[0.4776358644081965, -0.69914490624186021, -0.54604689504880333, 
-0.39906327084919618], [-0.19131098875937705, 0.38931442621684437, 0.32875423557107003, -0.64178247932298649], 
[-0.45727932446051456, 0.58256110686496398, 0.114434171270983, -0.32879109789066446]]], [[[-0.018126352674548145, 
0.31755548505355996, 0.09364895540995348, 0.31170458688801511], [0.7106464400819944, 0.22696073544934764, -0.38786963307935995, 
0.26572725148890408], [0.60495626567910887, -0.51762533493476803, -0.12430131811287756, -0.40582046681167505]], 
[[0.56739707899462055, 0.50052899378560622, 0.59515628993297953, 0.38401028339772247], [0.15846495356909085, 
-0.63828991185873374, -0.59908717814320933, 0.58012087646490362], [0.035542472635455728, -0.26007105516544116, 
-0.17338549335205111, 0.67405807911672022]]]])+(1.-msk_ref)*numpy.array([[[[-0.20691438737438628, -0.2773238802532339, 
0.10709296625214522, -0.26689448677860778], [0.35555922809733714, 0.028769518129007268, -0.087498116225253414, 
-0.29381950214588665], [0.33531370022012552, -0.21626428820869756, 0.2854791843741632, 0.34867615021707582]], 
[[-0.061591693385408805, 0.27031954930411473, 0.15358586882298136, 0.35644106909484347], [-0.30671383912942385, 
-0.075625813790861529, 0.37339738495539726, -0.44356212938778894], [0.26270975663695251, 0.27482498450644138, 
-0.29876152060089989, 0.0042215845747409055]]], [[[0.14622413625838773, 0.27078079269351352, -0.32003998238973996, 
-0.078192048763143923], [0.39969816107704803, -0.11951807235153751, -0.063835610309340096, 0.023913632287460566], 
[0.055182943128198853, -0.080602818174166369, 0.18696428452224464, 0.22748850702932216]], [[-0.084770235892327989, 
0.38036709123180013, 0.36374279202329601, -0.17757761261311716], [0.32132827022192112, 0.27678379302260875, 
-0.27267557689559779, -0.31954301278108865], [-0.025011540270800221, 0.13579006586360179, 0.24881628359769989, 
0.15597289939764777]]], [[[-0.11645051922659598, -0.35952302817931675, -0.44392643258618908, -0.31439257836963752], 
[0.38339565079617843, -0.35917218608883117, 0.095066184879857157, 0.12429923591456521], [-0.22234554428769526, 
-0.12532717455759948, -0.034712342221105301, 0.25264494747483229]], [[-0.074247550964211553, 0.39774238401643375, 
-0.31992455409509457, 0.15195488878223196], [0.041429090535461004, 0.075102387824983255, 0.41347021122012234, 
-0.051420135645944401], [-0.049171150070986547, -0.39963129449360263, -0.38755426822491268, 0.24770287290888757]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not ref.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(3,2,3,4),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_taggedData_rank0(self):
      arg0=0.749376761194
      arg1=Data(0.0550571660662,self.functionspace)
      arg1.setTaggedValue(1,0.574082400833)
      res=maximum(arg0,arg1)
      ref=Data(0.749376761194,self.functionspace)
      ref.setTaggedValue(1,0.749376761194)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_taggedData_rank1(self):
      arg0=-0.680855544333
      arg1=Data(numpy.array([0.67376030731870862, -0.80778821721518979]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.037848835866975516, 0.87213440903157147]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.67376030731870862, -0.68085554433347961]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.037848835866975516, 0.87213440903157147]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_taggedData_rank2(self):
      arg0=-0.909239747813
      arg1=Data(numpy.array([[-0.64454356812956481, -0.86276994251191574, -0.65475003484092542, 0.49772756092649728, 
-0.62301147321250361], [-0.25438398978405408, 0.36710307398687525, -0.34346378503873143, 0.25592269950575552, 
-0.12026356085185963], [0.54205772884656955, -0.43073745100231031, 0.44379518197302503, -0.85039873610355188, 
-0.10277968817858407], [0.98988818757648045, 0.0012025089604543737, 0.52680589505035313, -0.0097011550684382719, 
-0.48338662818135369]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.7668418281811995, 0.58088223900421432, -0.7437776408065091, 0.86419864086959608, 
-0.4723956335556756], [0.53697589117635824, 0.2711376690963625, -0.98131055778752607, -0.49188560271449933, 
-0.80938040678345691], [0.52105010002175089, -0.30817215251878882, -0.44885501512912174, 0.55707100193269343, 
0.0088579617731630034], [-0.76920443360085389, 0.84064312551950282, -0.68937920661052132, 0.32843690937592629, 
-0.32067573229901547]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.64454356812956481, -0.86276994251191574, -0.65475003484092542, 0.49772756092649728, 
-0.62301147321250361], [-0.25438398978405408, 0.36710307398687525, -0.34346378503873143, 0.25592269950575552, 
-0.12026356085185963], [0.54205772884656955, -0.43073745100231031, 0.44379518197302503, -0.85039873610355188, 
-0.10277968817858407], [0.98988818757648045, 0.0012025089604543737, 0.52680589505035313, -0.0097011550684382719, 
-0.48338662818135369]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.7668418281811995, 0.58088223900421432, -0.7437776408065091, 0.86419864086959608, 
-0.4723956335556756], [0.53697589117635824, 0.2711376690963625, -0.9092397478133023, -0.49188560271449933, 
-0.80938040678345691], [0.52105010002175089, -0.30817215251878882, -0.44885501512912174, 0.55707100193269343, 
0.0088579617731630034], [-0.76920443360085389, 0.84064312551950282, -0.68937920661052132, 0.32843690937592629, 
-0.32067573229901547]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_taggedData_rank3(self):
      arg0=0.249424174179
      arg1=Data(numpy.array([[[0.46754350221048768, -0.0073365002244627497], [0.8317337297696763, -0.10363370033103259]], 
[[0.51665498962708267, -0.71530066676331061], [0.63627740305765257, 0.41496430034083032]], [[-0.79504585189406951, 
-0.026351466709852778], [0.67263108755904, -0.51771009361603881]], [[-0.44083971000942612, 0.41952031191027528], 
[-0.34095817640025805, 0.26541265647867629]], [[-0.7345904848224849, -0.51133348832345904], [-0.0079815304116552799, 
-0.83616722215923511]], [[-0.63542863975090969, -0.026519165942767264], [0.79422556870657468, 
-0.21553699106415625]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.77925049701239635, 0.050615725840031045], [0.99903673790220382, 
-0.52104306227264607]], [[0.5655662287691956, -0.79521425247232047], [-0.9255830796185045, 0.44873538443219596]], 
[[0.14311466150804653, -0.060513820338312874], [-0.037702316714060879, -0.9214627922508023]], [[0.7249495132205459, 
0.71780977359732856], [0.081651919414779339, -0.47637153816118616]], [[0.57853885909055536, 0.68400129002489174], 
[-0.23444447635966981, 0.92630085486080049]], [[0.49991894177161522, 0.2545702276318178], [-0.93871928511281189, 
-0.12168727526156964]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.46754350221048768, 0.24942417417927998], [0.8317337297696763, 0.24942417417927998]], 
[[0.51665498962708267, 0.24942417417927998], [0.63627740305765257, 0.41496430034083032]], [[0.24942417417927998, 
0.24942417417927998], [0.67263108755904, 0.24942417417927998]], [[0.24942417417927998, 0.41952031191027528], 
[0.24942417417927998, 0.26541265647867629]], [[0.24942417417927998, 0.24942417417927998], [0.24942417417927998, 
0.24942417417927998]], [[0.24942417417927998, 0.24942417417927998], [0.79422556870657468, 
0.24942417417927998]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.77925049701239635, 0.24942417417927998], [0.99903673790220382, 
0.24942417417927998]], [[0.5655662287691956, 0.24942417417927998], [0.24942417417927998, 0.44873538443219596]], 
[[0.24942417417927998, 0.24942417417927998], [0.24942417417927998, 0.24942417417927998]], [[0.7249495132205459, 
0.71780977359732856], [0.24942417417927998, 0.24942417417927998]], [[0.57853885909055536, 0.68400129002489174], 
[0.24942417417927998, 0.92630085486080049]], [[0.49991894177161522, 0.2545702276318178], [0.24942417417927998, 
0.24942417417927998]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_taggedData_rank4(self):
      arg0=0.480003299289
      arg1=Data(numpy.array([[[[-0.36477176076547235, 0.6785411283629923, -0.6550000926385362, 0.61482412056032998], 
[-0.66613998431222687, 0.69991594974851945, -0.57939029525197538, 0.56994720938299581], [-0.39451657149433483, 
-0.20256475913692462, -0.61009320061752392, -0.35350439670959632]], [[-0.37742676281496412, 0.45279923826324686, 
-0.037031221267454617, 0.67735513734586728], [-0.021014665692789203, 0.47035321194035218, 0.55088403325705326, 
-0.34775231887863445], [0.20870317515573, -0.36665613391867447, 0.095218795896232944, -0.54483047001137863]]], 
[[[-0.46639778146125033, 0.39471356375867428, -0.19056338855266453, -0.36746350047000242], [0.30833043988102937, 
-0.085558178128796847, 0.97499474706442957, 0.73081769699119969], [0.11433855209802335, 0.13402598839960378, 
0.031760127929625126, 0.78129823610024118]], [[0.14654407753049825, -0.2634255554407301, -0.44538459691394205, 
-0.16782739716365436], [0.21400945173068164, 0.58638523605963311, -0.55089817298428345, 0.27921003357182284], 
[-0.84255501661397769, 0.351422823005475, -0.88008992497756089, 0.2324122529746071]]], [[[-0.94380389642852602, 
0.87809214287099713, -0.74741279742314837, 0.78230888486504302], [0.40306143996895361, 0.68965156317024512, 
-0.24266261187782501, 0.13202562682568852], [-0.45564220471240868, 0.69736990248179698, 0.49809031191600606, 
-0.96164413094941792]], [[-0.48082818059383925, 0.21316275824960762, -0.99622086107244501, -0.014050265407334672], 
[-0.37310441558193275, 0.71984262885649453, -0.079540037845887035, -0.57283688478967587], [-0.95972352401364658, 
-0.51998802141851974, -0.75860324851962946, -0.018953701074107654]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.9956445974414001, -0.53948138473896767, 0.5178416086515607, 0.32328684791328821], 
[0.028268824462836628, 0.33772417131459154, -0.005317112328966811, -0.78109210501383752], [0.2612130974710416, 
-0.42205687106802836, -0.83270340528142173, -0.54599030528474102]], [[0.82336183430657828, -0.082255277137609273, 
-0.08063835783906681, 0.024319678422843882], [-0.78846515048027266, -0.54877385857319627, -0.80885038971713841, 
0.9289263563726422], [-0.59482932749842954, 0.80857551447868814, 0.68226134967843088, 0.20193842581342025]]], 
[[[-0.86912586595496433, -0.83743169899808945, 0.67841274474353441, 0.10871334160379598], [0.43412972413801998, 
-0.56900823017571067, 0.62510220326897636, 0.61870141786141875], [0.66216626903724252, -0.87860985912038769, 
-0.43002308505447195, -0.64199947923652223]], [[0.37900293554450704, 0.50010977600274975, 0.90287560363233577, 
-0.89973230366231483], [-0.60816386276352352, -0.16007761683149324, 0.72344042832643463, -0.84043194935396892], 
[0.99989070044986206, 0.29711802572966928, -0.23578637726200546, -0.40313944331686358]]], [[[-0.89175114250177034, 
-0.90809237437177504, 0.86341145611875847, -0.41145936468723887], [-0.46213665195301945, 0.71218535134846528, 
-0.81488501212875564, 0.99559963582160527], [-0.79663136613755769, -0.95748096116658155, 0.11084198853771476, 
-0.33412677702670135]], [[0.97375804609899808, -0.71971848129559124, 0.39481171644203195, 0.30735387830906324], 
[0.72125978200611796, -0.97760329079510155, -0.74732273823629258, 0.22421778333310938], [-0.073938924611220314, 
0.24482734141624141, -0.94752876215258675, -0.58098484783514137]]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.48000329928907992, 0.6785411283629923, 0.48000329928907992, 0.61482412056032998], 
[0.48000329928907992, 0.69991594974851945, 0.48000329928907992, 0.56994720938299581], [0.48000329928907992, 
0.48000329928907992, 0.48000329928907992, 0.48000329928907992]], [[0.48000329928907992, 0.48000329928907992, 
0.48000329928907992, 0.67735513734586728], [0.48000329928907992, 0.48000329928907992, 0.55088403325705326, 
0.48000329928907992], [0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 0.48000329928907992]]], 
[[[0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 0.48000329928907992], [0.48000329928907992, 
0.48000329928907992, 0.97499474706442957, 0.73081769699119969], [0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 
0.78129823610024118]], [[0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 0.48000329928907992], 
[0.48000329928907992, 0.58638523605963311, 0.48000329928907992, 0.48000329928907992], [0.48000329928907992, 
0.48000329928907992, 0.48000329928907992, 0.48000329928907992]]], [[[0.48000329928907992, 0.87809214287099713, 
0.48000329928907992, 0.78230888486504302], [0.48000329928907992, 0.68965156317024512, 0.48000329928907992, 
0.48000329928907992], [0.48000329928907992, 0.69736990248179698, 0.49809031191600606, 0.48000329928907992]], 
[[0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 0.48000329928907992], [0.48000329928907992, 
0.71984262885649453, 0.48000329928907992, 0.48000329928907992], [0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 
0.48000329928907992]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.9956445974414001, 0.48000329928907992, 0.5178416086515607, 0.48000329928907992], 
[0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 0.48000329928907992], [0.48000329928907992, 
0.48000329928907992, 0.48000329928907992, 0.48000329928907992]], [[0.82336183430657828, 0.48000329928907992, 
0.48000329928907992, 0.48000329928907992], [0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 0.9289263563726422], 
[0.48000329928907992, 0.80857551447868814, 0.68226134967843088, 0.48000329928907992]]], [[[0.48000329928907992, 
0.48000329928907992, 0.67841274474353441, 0.48000329928907992], [0.48000329928907992, 0.48000329928907992, 0.62510220326897636, 
0.61870141786141875], [0.66216626903724252, 0.48000329928907992, 0.48000329928907992, 0.48000329928907992]], 
[[0.48000329928907992, 0.50010977600274975, 0.90287560363233577, 0.48000329928907992], [0.48000329928907992, 
0.48000329928907992, 0.72344042832643463, 0.48000329928907992], [0.99989070044986206, 0.48000329928907992, 0.48000329928907992, 
0.48000329928907992]]], [[[0.48000329928907992, 0.48000329928907992, 0.86341145611875847, 0.48000329928907992], 
[0.48000329928907992, 0.71218535134846528, 0.48000329928907992, 0.99559963582160527], [0.48000329928907992, 
0.48000329928907992, 0.48000329928907992, 0.48000329928907992]], [[0.97375804609899808, 0.48000329928907992, 
0.48000329928907992, 0.48000329928907992], [0.72125978200611796, 0.48000329928907992, 0.48000329928907992, 
0.48000329928907992], [0.48000329928907992, 0.48000329928907992, 0.48000329928907992, 0.48000329928907992]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_taggedData_rank0(self):
      arg0=numpy.array(0.186020397663)
      arg1=Data(-0.574686137847,self.functionspace)
      arg1.setTaggedValue(1,0.450323924159)
      res=maximum(arg0,arg1)
      ref=Data(0.186020397663,self.functionspace)
      ref.setTaggedValue(1,0.450323924159)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_taggedData_rank0(self):
      arg0=numpy.array([0.43536293548300886, 0.43485056921826382])
      arg1=Data(-0.25240823978,self.functionspace)
      arg1.setTaggedValue(1,0.838820447873)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.43536293548300886, 0.43485056921826382]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.83882044787271215, 0.83882044787271215]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_taggedData_rank0(self):
      arg0=numpy.array([[0.84128402854693007, 0.60846433427406565, 0.63674444749652159, -0.39358411784212821, 
-0.035045771880696464], [0.49646587830780975, -0.5793829200595304, -0.75593103545321827, -0.18809037247616267, 
0.32299151346286203], [0.69090048049995123, -0.32570079187366696, 0.092799231188626985, -0.92753532762223045, 
0.31230629257754283], [-0.96393819072551068, -0.35261580509587387, -0.91486020864761053, 0.64462555941700117, 
-0.30810563430751992]])
      arg1=Data(0.224016817107,self.functionspace)
      arg1.setTaggedValue(1,0.493288543973)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.84128402854693007, 0.60846433427406565, 0.63674444749652159, 0.22401681710740107, 
0.22401681710740107], [0.49646587830780975, 0.22401681710740107, 0.22401681710740107, 0.22401681710740107, 
0.32299151346286203], [0.69090048049995123, 0.22401681710740107, 0.22401681710740107, 0.22401681710740107, 
0.31230629257754283], [0.22401681710740107, 0.22401681710740107, 0.22401681710740107, 0.64462555941700117, 
0.22401681710740107]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.84128402854693007, 0.60846433427406565, 0.63674444749652159, 0.49328854397322552, 
0.49328854397322552], [0.49646587830780975, 0.49328854397322552, 0.49328854397322552, 0.49328854397322552, 
0.49328854397322552], [0.69090048049995123, 0.49328854397322552, 0.49328854397322552, 0.49328854397322552, 
0.49328854397322552], [0.49328854397322552, 0.49328854397322552, 0.49328854397322552, 0.64462555941700117, 
0.49328854397322552]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_taggedData_rank0(self):
      arg0=numpy.array([[[-0.73832420641101382, -0.89614124503956227], [0.61128733125805956, -0.51119420370932089]], 
[[-0.71475844944675471, -0.36292614146602653], [-0.73503702643924229, -0.42609282696835105]], [[-0.79290979403430151, 
-0.47089790577335133], [-0.1888329603711032, -0.69933780891735631]], [[0.34499154911187024, 0.26950694303532163], 
[-0.84191598788203903, -0.51180786897021413]], [[0.10538186305270592, 0.80248814771570998], [-0.84528294280476457, 
-0.065054457314116076]], [[0.0981128982664492, -0.22710165145076022], [0.54031990338685731, 0.58491168497345214]]])
      arg1=Data(0.70818217473,self.functionspace)
      arg1.setTaggedValue(1,-0.555450471204)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.70818217473011602, 0.70818217473011602], [0.70818217473011602, 0.70818217473011602]], 
[[0.70818217473011602, 0.70818217473011602], [0.70818217473011602, 0.70818217473011602]], [[0.70818217473011602, 
0.70818217473011602], [0.70818217473011602, 0.70818217473011602]], [[0.70818217473011602, 0.70818217473011602], 
[0.70818217473011602, 0.70818217473011602]], [[0.70818217473011602, 0.80248814771570998], [0.70818217473011602, 
0.70818217473011602]], [[0.70818217473011602, 0.70818217473011602], [0.70818217473011602, 
0.70818217473011602]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.55545047120405466, -0.55545047120405466], [0.61128733125805956, 
-0.51119420370932089]], [[-0.55545047120405466, -0.36292614146602653], [-0.55545047120405466, -0.42609282696835105]], 
[[-0.55545047120405466, -0.47089790577335133], [-0.1888329603711032, -0.55545047120405466]], [[0.34499154911187024, 
0.26950694303532163], [-0.55545047120405466, -0.51180786897021413]], [[0.10538186305270592, 0.80248814771570998], 
[-0.55545047120405466, -0.065054457314116076]], [[0.0981128982664492, -0.22710165145076022], [0.54031990338685731, 
0.58491168497345214]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_taggedData_rank0(self):
      arg0=numpy.array([[[[0.11153448068840621, -0.56848274975111002, -0.94410595745938508, 0.23561250754911889], 
[0.83704769356272468, 0.38582533180195089, -0.57371119755119149, -0.3424949955212413], [-0.32286701476399804, 
-0.45696971352908311, 0.37912500175617692, 0.53133712592956095]], [[-0.93608807105993885, 0.52830038109686384, 
-0.73259305645056405, 0.83274969889087758], [0.33777563978509728, 0.95199275442344833, -0.87048456555626674, 
0.067889619896001285], [0.51373785681450501, 0.048828551321738356, -0.74924306720872602, -0.90632749194292117]]], 
[[[0.584571816639891, -0.29867811322419247, -0.64912283552775341, -0.37864480895502917], [-0.37633294188661459, 
-0.53883681914658177, 0.9682466618160066, 0.1958715359898997], [0.53348045673584843, -0.94802817619600055, 
-0.18144428807382185, -0.7671663000623048]], [[-0.3287191463483623, 0.70058602837689055, 0.5368351345405431, 
-0.26118657384722122], [-0.36234845428234919, -0.45189798440432738, -0.55574342617233796, -0.34637027849253021], 
[0.67898209443156876, 0.94690687672788543, 0.49000742427290844, 0.25547087835907645]]], [[[0.16510906047036444, 
0.67631930672069496, -0.84828623051608787, 0.0018374299653354242], [0.13125191332528208, -0.27403094341367984, 
-0.26003413883030069, -0.29371169796284158], [-0.086582930823249749, -0.33379290778756432, 0.59204208885925458, 
0.26032937226889308]], [[0.74325122791607612, -0.11136111817292083, 0.57497847778268119, -0.12766528515340192], 
[-0.48518691912389311, -0.91547170505064601, 0.22013513700087417, -0.20573158170436878], [0.15002537658554727, 
0.69803769385007075, -0.76683761137817141, 0.53718304990791577]]]])
      arg1=Data(-0.559564751892,self.functionspace)
      arg1.setTaggedValue(1,0.177106951604)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.11153448068840621, -0.55956475189212518, -0.55956475189212518, 0.23561250754911889], 
[0.83704769356272468, 0.38582533180195089, -0.55956475189212518, -0.3424949955212413], [-0.32286701476399804, 
-0.45696971352908311, 0.37912500175617692, 0.53133712592956095]], [[-0.55956475189212518, 0.52830038109686384, 
-0.55956475189212518, 0.83274969889087758], [0.33777563978509728, 0.95199275442344833, -0.55956475189212518, 
0.067889619896001285], [0.51373785681450501, 0.048828551321738356, -0.55956475189212518, -0.55956475189212518]]], 
[[[0.584571816639891, -0.29867811322419247, -0.55956475189212518, -0.37864480895502917], [-0.37633294188661459, 
-0.53883681914658177, 0.9682466618160066, 0.1958715359898997], [0.53348045673584843, -0.55956475189212518, 
-0.18144428807382185, -0.55956475189212518]], [[-0.3287191463483623, 0.70058602837689055, 0.5368351345405431, 
-0.26118657384722122], [-0.36234845428234919, -0.45189798440432738, -0.55574342617233796, -0.34637027849253021], 
[0.67898209443156876, 0.94690687672788543, 0.49000742427290844, 0.25547087835907645]]], [[[0.16510906047036444, 
0.67631930672069496, -0.55956475189212518, 0.0018374299653354242], [0.13125191332528208, -0.27403094341367984, 
-0.26003413883030069, -0.29371169796284158], [-0.086582930823249749, -0.33379290778756432, 0.59204208885925458, 
0.26032937226889308]], [[0.74325122791607612, -0.11136111817292083, 0.57497847778268119, -0.12766528515340192], 
[-0.48518691912389311, -0.55956475189212518, 0.22013513700087417, -0.20573158170436878], [0.15002537658554727, 
0.69803769385007075, -0.55956475189212518, 0.53718304990791577]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.17710695160418588, 0.17710695160418588, 0.17710695160418588, 0.23561250754911889], 
[0.83704769356272468, 0.38582533180195089, 0.17710695160418588, 0.17710695160418588], [0.17710695160418588, 
0.17710695160418588, 0.37912500175617692, 0.53133712592956095]], [[0.17710695160418588, 0.52830038109686384, 
0.17710695160418588, 0.83274969889087758], [0.33777563978509728, 0.95199275442344833, 0.17710695160418588, 
0.17710695160418588], [0.51373785681450501, 0.17710695160418588, 0.17710695160418588, 0.17710695160418588]]], 
[[[0.584571816639891, 0.17710695160418588, 0.17710695160418588, 0.17710695160418588], [0.17710695160418588, 
0.17710695160418588, 0.9682466618160066, 0.1958715359898997], [0.53348045673584843, 0.17710695160418588, 0.17710695160418588, 
0.17710695160418588]], [[0.17710695160418588, 0.70058602837689055, 0.5368351345405431, 0.17710695160418588], 
[0.17710695160418588, 0.17710695160418588, 0.17710695160418588, 0.17710695160418588], [0.67898209443156876, 
0.94690687672788543, 0.49000742427290844, 0.25547087835907645]]], [[[0.17710695160418588, 0.67631930672069496, 
0.17710695160418588, 0.17710695160418588], [0.17710695160418588, 0.17710695160418588, 0.17710695160418588, 
0.17710695160418588], [0.17710695160418588, 0.17710695160418588, 0.59204208885925458, 0.26032937226889308]], 
[[0.74325122791607612, 0.17710695160418588, 0.57497847778268119, 0.17710695160418588], [0.17710695160418588, 
0.17710695160418588, 0.22013513700087417, 0.17710695160418588], [0.17710695160418588, 0.69803769385007075, 0.17710695160418588, 
0.53718304990791577]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_taggedData_rank1(self):
      arg0=numpy.array(-0.796368420403)
      arg1=Data(numpy.array([0.12072095401364846, 0.23516273087246264]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.36057840054987578, -0.63546172502583143]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.12072095401364846, 0.23516273087246264]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.36057840054987578, -0.63546172502583143]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_taggedData_rank1(self):
      arg0=numpy.array([0.47256141691521192, -0.55089126102487307])
      arg1=Data(numpy.array([-0.97331785689805672, 0.49899386181139893]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.30337503223254658, 0.86872548623120083]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.47256141691521192, 0.49899386181139893]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.47256141691521192, 0.86872548623120083]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_taggedData_rank2(self):
      arg0=numpy.array(-0.376582891747)
      arg1=Data(numpy.array([[-0.06174903893753414, -0.92016879375268323, -0.27007458489560476, -0.19244852024769976, 
0.888209024082524], [0.98249722533714756, -0.52250379927822754, -0.76353277595214974, 0.36158823840457521, 
-0.26901014579527827], [-0.69708891177632948, 0.044279174139640887, 0.88583224269563821, -0.044382811075480033, 
0.90970378806785535], [0.57477947973781496, -0.35937609437864504, -0.35316552129869594, 0.54728029845119486, 
-0.23308921281994333]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.42737094802805919, -0.12900070570946287, -0.21540646839749877, 0.80575283763774186, 
0.71043096806397044], [-0.65628221246896223, 0.039178585521630227, -0.53870332811126009, 0.28918473289278279, 
0.38824704060470849], [0.44794932984344982, -0.39920729763965057, -0.33862905380276187, -0.46598074851492099, 
0.52981562043331465], [0.045044532113411195, -0.46767008450238334, -0.86198767388285402, 0.2897022314053268, 
-0.36626590872147324]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.06174903893753414, -0.37658289174742876, -0.27007458489560476, -0.19244852024769976, 
0.888209024082524], [0.98249722533714756, -0.37658289174742876, -0.37658289174742876, 0.36158823840457521, 
-0.26901014579527827], [-0.37658289174742876, 0.044279174139640887, 0.88583224269563821, -0.044382811075480033, 
0.90970378806785535], [0.57477947973781496, -0.35937609437864504, -0.35316552129869594, 0.54728029845119486, 
-0.23308921281994333]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.42737094802805919, -0.12900070570946287, -0.21540646839749877, 0.80575283763774186, 
0.71043096806397044], [-0.37658289174742876, 0.039178585521630227, -0.37658289174742876, 0.28918473289278279, 
0.38824704060470849], [0.44794932984344982, -0.37658289174742876, -0.33862905380276187, -0.37658289174742876, 
0.52981562043331465], [0.045044532113411195, -0.37658289174742876, -0.37658289174742876, 0.2897022314053268, 
-0.36626590872147324]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_taggedData_rank2(self):
      arg0=numpy.array([[0.16007677616632399, -0.24093574180704302, 0.97122223227264759, 0.57952946654876425, 
-0.29555507311215012], [-0.51591090261269823, 0.85158609912100869, 0.13118192616012547, -0.72318685403466954, 
0.0015537171664505856], [-0.0046800859495694791, -0.68948711226486581, 0.41927431327567155, 0.81022506490223312, 
-0.83452299576297073], [-0.54213499820282918, -0.058122838204531346, 0.86174319866742888, -0.6582800347578075, 
-0.19690324064813902]])
      arg1=Data(numpy.array([[-0.21804427548199468, -0.5864764312678763, -0.94915833269956917, -0.17281941846598614, 
-0.13679738347556603], [-0.66948509054351835, 0.57073476819336699, -0.79688344499053021, -0.56584458860304054, 
0.69312654964886722], [0.4609932477085219, 0.62494960993282711, 0.090223484562575784, -0.37654323175965732, 
0.6186313301960964], [0.34893476475996565, -0.99038635805361386, -0.94992136203826849, -0.41189587896053292, 
0.97347600797930545]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.40788844279118819, -0.598482499891414, 0.64149530686047451, -0.85467158756555683, 
0.45904264183891841], [-0.36422260646062043, 0.12069256732618694, 0.076601332168098279, 0.71333906642589984, 
0.1794213765386079], [-0.2035205862884899, -0.47037011549699215, 0.51669193270339453, 0.43180534317612906, 
0.77229176619101914], [0.55915035273612346, -0.64818211782070634, 0.084357768324000437, -0.70545826604604112, 
0.72414456159416063]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.16007677616632399, -0.24093574180704302, 0.97122223227264759, 0.57952946654876425, 
-0.13679738347556603], [-0.51591090261269823, 0.85158609912100869, 0.13118192616012547, -0.56584458860304054, 
0.69312654964886722], [0.4609932477085219, 0.62494960993282711, 0.41927431327567155, 0.81022506490223312, 0.6186313301960964], 
[0.34893476475996565, -0.058122838204531346, 0.86174319866742888, -0.41189587896053292, 
0.97347600797930545]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.40788844279118819, -0.24093574180704302, 0.97122223227264759, 0.57952946654876425, 
0.45904264183891841], [-0.36422260646062043, 0.85158609912100869, 0.13118192616012547, 0.71333906642589984, 
0.1794213765386079], [-0.0046800859495694791, -0.47037011549699215, 0.51669193270339453, 0.81022506490223312, 
0.77229176619101914], [0.55915035273612346, -0.058122838204531346, 0.86174319866742888, -0.6582800347578075, 
0.72414456159416063]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_taggedData_rank3(self):
      arg0=numpy.array(0.676688967795)
      arg1=Data(numpy.array([[[0.17710255673529685, -0.98616482925104432], [-0.18593698135976156, -0.08830113270196116]], 
[[-0.19176188484489076, 0.46447992323848686], [-0.21055292006775717, -0.40155718736176915]], [[-0.48016199982715557, 
-0.25416306925664744], [0.89792509419334232, -0.60096716111156745]], [[-0.16453085934336853, 0.75811874187950168], 
[-0.41550358559945688, 0.41734405318183398]], [[-0.72700659300295278, 0.24800660476870395], [0.58588385527690479, 
0.26746861802302058]], [[0.47753613866069733, 0.53586790882863977], [0.43531878513051114, 
0.80308541825369395]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.33858768282152618, -0.26527226732627351], [0.012672891555068899, 
-0.59953251167370758]], [[0.80630804067299899, 0.10904683081217437], [0.15983888023859572, -0.19949121957080629]], 
[[0.76785224435282684, 0.83500596348083422], [-0.81451399601488061, -0.10276250571170853]], [[-0.83989589905521633, 
-0.55260259622606944], [-0.66349003992417432, -0.49522313511192384]], [[-0.99231611892617533, -0.12589759739925777], 
[-0.056740787000463255, -0.41209606223364093]], [[-0.026226263289273355, 0.15985474500396335], [-0.90226140217503148, 
-0.72634292634655884]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.67668896779475474, 0.67668896779475474], [0.67668896779475474, 0.67668896779475474]], 
[[0.67668896779475474, 0.67668896779475474], [0.67668896779475474, 0.67668896779475474]], [[0.67668896779475474, 
0.67668896779475474], [0.89792509419334232, 0.67668896779475474]], [[0.67668896779475474, 0.75811874187950168], 
[0.67668896779475474, 0.67668896779475474]], [[0.67668896779475474, 0.67668896779475474], [0.67668896779475474, 
0.67668896779475474]], [[0.67668896779475474, 0.67668896779475474], [0.67668896779475474, 
0.80308541825369395]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.67668896779475474, 0.67668896779475474], [0.67668896779475474, 
0.67668896779475474]], [[0.80630804067299899, 0.67668896779475474], [0.67668896779475474, 0.67668896779475474]], 
[[0.76785224435282684, 0.83500596348083422], [0.67668896779475474, 0.67668896779475474]], [[0.67668896779475474, 
0.67668896779475474], [0.67668896779475474, 0.67668896779475474]], [[0.67668896779475474, 0.67668896779475474], 
[0.67668896779475474, 0.67668896779475474]], [[0.67668896779475474, 0.67668896779475474], [0.67668896779475474, 
0.67668896779475474]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_taggedData_rank3(self):
      arg0=numpy.array([[[-0.72250606016448571, 0.23069996923721736], [-0.78415077938324207, -0.014398005342852294]], 
[[0.38352996951071283, 0.83490613125671564], [0.22182384561493884, -0.65282433022556163]], [[0.75149356822193147, 
-0.048573144954658476], [0.86584999266726381, -0.95055825354290202]], [[0.55268803292578972, -0.5966705009047184], 
[0.75560638165257443, 0.23607758626660447]], [[-0.080006856532859105, -0.73895343922309187], [0.41731964754099571, 
0.65120844155216662]], [[0.038885915750427547, 0.35848348981943512], [-0.85218229982139904, -0.7069783497491402]]])
      arg1=Data(numpy.array([[[0.68858614315334288, 0.067352986292107619], [-0.1437896689695235, -0.94233503796062013]], 
[[-0.40528326755315214, 0.051702683918499526], [-0.88806070543240145, -0.26755995424444134]], [[-0.36726739185307333, 
0.16178182217143311], [-0.96632821655816148, -0.56880773054872225]], [[-0.87709133395299954, -0.86568602160756192], 
[-0.75104128460213149, 0.92447900339143807]], [[-0.21458964468869746, -0.039532819097020111], [-0.22085446558200239, 
-0.67458439739974918]], [[-0.17517753607455511, 0.88509489531508612], [-0.25705056958719386, 
0.42323212474330463]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.24403334335130689, -0.35900007026410075], [0.53303706391801642, 
0.91122670565195252]], [[0.42964558654152296, -0.44911303070632891], [-0.084439178201057397, -0.8674668869576474]], 
[[0.14938488781701675, -0.20554837404478676], [0.67803799258250663, -0.44935931546653274]], [[-0.4549169190064708, 
0.12935188987382884], [0.37977906528542915, -0.072264303249091277]], [[0.040287317784536647, 0.3522774793759107], 
[-0.094167146838618043, 0.64515938462470945]], [[-0.79759199686000182, 0.37624723918955261], [-0.55920172900805998, 
-0.53910364948531231]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.68858614315334288, 0.23069996923721736], [-0.1437896689695235, -0.014398005342852294]], 
[[0.38352996951071283, 0.83490613125671564], [0.22182384561493884, -0.26755995424444134]], [[0.75149356822193147, 
0.16178182217143311], [0.86584999266726381, -0.56880773054872225]], [[0.55268803292578972, -0.5966705009047184], 
[0.75560638165257443, 0.92447900339143807]], [[-0.080006856532859105, -0.039532819097020111], [0.41731964754099571, 
0.65120844155216662]], [[0.038885915750427547, 0.88509489531508612], [-0.25705056958719386, 
0.42323212474330463]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.24403334335130689, 0.23069996923721736], [0.53303706391801642, 
0.91122670565195252]], [[0.42964558654152296, 0.83490613125671564], [0.22182384561493884, -0.65282433022556163]], 
[[0.75149356822193147, -0.048573144954658476], [0.86584999266726381, -0.44935931546653274]], [[0.55268803292578972, 
0.12935188987382884], [0.75560638165257443, 0.23607758626660447]], [[0.040287317784536647, 0.3522774793759107], 
[0.41731964754099571, 0.65120844155216662]], [[0.038885915750427547, 0.37624723918955261], [-0.55920172900805998, 
-0.53910364948531231]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_taggedData_rank4(self):
      arg0=numpy.array(0.311125833111)
      arg1=Data(numpy.array([[[[-0.11380560841847664, -0.52603636376884189, 0.56792514441060882, 0.83798273462233097], 
[0.16448356485497495, -0.12756765129413217, 0.84669010854045879, 0.76857705418208111], [0.96873131593785455, 
-0.068275868587937216, 0.33504489971259521, -0.3062753491142467]], [[0.51778162970808062, -0.21982301832546169, 
0.58686443390997112, -0.4764270683744094], [0.87944180739516353, -0.75934004745918182, -0.17494584440049921, 
-0.56406799050041179], [0.86972467769939765, 0.13420815149912624, 0.12785705162181449, 0.83326098752044331]]], 
[[[-0.74362564092020667, 0.94578463545413527, -0.99206707209481237, -0.083099762753271023], [0.87457351803343708, 
0.84279130281285841, -0.19255518585809384, 0.54815325971303763], [0.22134480896891096, 0.87105051352679386, 
-0.83038672103499067, 0.82749351075958266]], [[0.55067474361207625, 0.42561700790653823, -0.34449583562975827, 
0.078778405587881739], [0.29673278902895861, 0.18796065134032758, 0.18987802833749678, -0.32829112669317517], 
[0.21388502146738264, 0.8655016867366514, 0.55542070746423766, 0.13757537381085716]]], [[[-0.11106998893840725, 
0.64837734307043626, -0.37988514433104181, 0.91325189424969855], [-0.79152603146630796, 0.31919286120759405, 
0.69160091209088659, -0.53149393164900793], [0.3518630180454152, 0.45380620428057838, -0.012464662928450387, 
-0.58246822596373837]], [[-0.83184081891475703, -0.8235501848477671, 0.978781141858452, 0.8247533415828423], 
[-0.29355985995534928, -0.77322976392008003, -0.3754791142169922, 0.006609878020302995], [0.92460099408469554, 
-0.90625803330699584, -0.036477348393576525, 0.37519366885496086]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.67826584193345374, -0.025190054251609917, -0.7091047366392873, 
-0.71616804590919259], [0.17979962583543152, -0.57645605110264619, -0.95784121866427507, 0.78593377264682118], 
[-0.038695086837311043, -0.10549446655881156, -0.15355887266597446, 0.8718942192331538]], [[-0.93470293387540271, 
-0.31361167207603624, 0.154712619988989, -0.53151969823029677], [0.19071572545118776, -0.027541714059774902, 
0.15838782243184935, 0.30035678815598121], [0.8528199737608082, 0.12058296861598028, -0.17278398285675256, 
-0.081901229252531627]]], [[[0.89452608218484864, 0.20712800374880014, -0.38376458150624404, -0.81871180360787088], 
[0.1315509650511888, 0.83404404867284221, 0.77620626150004779, 0.50876641413890278], [-0.95999267833153579, 
0.10805183121243567, -0.6599667442957875, 0.59839211753680988]], [[0.19649778209528579, 0.925081766433828, 0.72071861805547011, 
0.58974748494836793], [0.86712395085325977, -0.80528666590519804, -0.14346705146844552, -0.58772903356309159], 
[0.029226946755463779, 0.59447037646577416, 0.027553622540348321, -0.052019429289529739]]], [[[-0.96069378908671088, 
-0.38743165706094929, 0.81794356716184624, 0.2392864331125355], [-0.83678472342781207, -0.2768479741775407, 
0.037672868643620738, -0.46937001254067989], [-0.22827748875320553, 0.74670501782312049, -0.43256664653545651, 
0.69447898637295591]], [[-0.52935095169456226, 0.83607651457566701, -0.71785096129520598, -0.3166494507066886], 
[-0.7588076609032508, -0.33107609805735194, -0.060247210980912236, 0.60193694229655037], [0.43042160676338992, 
-0.36237752256376288, -0.32144160911552455, 0.23255100548759855]]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.31112583311064923, 0.31112583311064923, 0.56792514441060882, 0.83798273462233097], 
[0.31112583311064923, 0.31112583311064923, 0.84669010854045879, 0.76857705418208111], [0.96873131593785455, 
0.31112583311064923, 0.33504489971259521, 0.31112583311064923]], [[0.51778162970808062, 0.31112583311064923, 
0.58686443390997112, 0.31112583311064923], [0.87944180739516353, 0.31112583311064923, 0.31112583311064923, 
0.31112583311064923], [0.86972467769939765, 0.31112583311064923, 0.31112583311064923, 0.83326098752044331]]], 
[[[0.31112583311064923, 0.94578463545413527, 0.31112583311064923, 0.31112583311064923], [0.87457351803343708, 
0.84279130281285841, 0.31112583311064923, 0.54815325971303763], [0.31112583311064923, 0.87105051352679386, 0.31112583311064923, 
0.82749351075958266]], [[0.55067474361207625, 0.42561700790653823, 0.31112583311064923, 0.31112583311064923], 
[0.31112583311064923, 0.31112583311064923, 0.31112583311064923, 0.31112583311064923], [0.31112583311064923, 0.8655016867366514, 
0.55542070746423766, 0.31112583311064923]]], [[[0.31112583311064923, 0.64837734307043626, 0.31112583311064923, 
0.91325189424969855], [0.31112583311064923, 0.31919286120759405, 0.69160091209088659, 0.31112583311064923], 
[0.3518630180454152, 0.45380620428057838, 0.31112583311064923, 0.31112583311064923]], [[0.31112583311064923, 
0.31112583311064923, 0.978781141858452, 0.8247533415828423], [0.31112583311064923, 0.31112583311064923, 0.31112583311064923, 
0.31112583311064923], [0.92460099408469554, 0.31112583311064923, 0.31112583311064923, 
0.37519366885496086]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.31112583311064923, 0.31112583311064923, 0.31112583311064923, 0.31112583311064923], 
[0.31112583311064923, 0.31112583311064923, 0.31112583311064923, 0.78593377264682118], [0.31112583311064923, 
0.31112583311064923, 0.31112583311064923, 0.8718942192331538]], [[0.31112583311064923, 0.31112583311064923, 
0.31112583311064923, 0.31112583311064923], [0.31112583311064923, 0.31112583311064923, 0.31112583311064923, 
0.31112583311064923], [0.8528199737608082, 0.31112583311064923, 0.31112583311064923, 0.31112583311064923]]], 
[[[0.89452608218484864, 0.31112583311064923, 0.31112583311064923, 0.31112583311064923], [0.31112583311064923, 
0.83404404867284221, 0.77620626150004779, 0.50876641413890278], [0.31112583311064923, 0.31112583311064923, 0.31112583311064923, 
0.59839211753680988]], [[0.31112583311064923, 0.925081766433828, 0.72071861805547011, 0.58974748494836793], 
[0.86712395085325977, 0.31112583311064923, 0.31112583311064923, 0.31112583311064923], [0.31112583311064923, 
0.59447037646577416, 0.31112583311064923, 0.31112583311064923]]], [[[0.31112583311064923, 0.31112583311064923, 
0.81794356716184624, 0.31112583311064923], [0.31112583311064923, 0.31112583311064923, 0.31112583311064923, 
0.31112583311064923], [0.31112583311064923, 0.74670501782312049, 0.31112583311064923, 0.69447898637295591]], 
[[0.31112583311064923, 0.83607651457566701, 0.31112583311064923, 0.31112583311064923], [0.31112583311064923, 
0.31112583311064923, 0.31112583311064923, 0.60193694229655037], [0.43042160676338992, 0.31112583311064923, 0.31112583311064923, 
0.31112583311064923]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_taggedData_rank4(self):
      arg0=numpy.array([[[[0.002562308848454542, -0.15183306579631028, 0.98594863414905065, 0.38876845683487726], 
[0.4007939057073524, -0.61845117501699121, 0.076166936848131028, -0.015591623626320272], [-0.79273446558231964, 
-0.55473573225302553, -0.41571297736559165, 0.65095326433517031]], [[0.44947637187348088, -0.43824637463790883, 
0.59212025347785335, 0.83122656474479695], [0.16815843376338746, 0.86135881182639951, -0.58001385281386542, 
0.5731565021744669], [-0.29602946815563258, -0.65654195858044173, -0.75908993222687005, 0.67153262857459728]]], 
[[[0.53270105359196052, 0.41681501901552975, 0.65216139994222622, -0.29509500900305841], [0.4969220496535327, 
0.74009856566202825, 0.57487999521149313, -0.57620655762793738], [-0.97940230399260941, 0.42870575165153824, 
0.26530113757928153, -0.64739611694503529]], [[-0.55993660619351382, -0.91610169686772269, -0.62434184180422236, 
-0.85044543351701396], [0.93294937179581128, -0.47490075605930882, -0.21622691984481901, -0.70242417207338548], 
[-0.70054800697348041, 0.029700481712530369, 0.95248756894557896, -0.58098770763767149]]], [[[0.08563725603405703, 
-0.95819851282617585, -0.93246587740383946, -0.14391009022025902], [0.08332034850987835, 0.81564051822547845, 
-0.011370710776029913, 0.79167198558767571], [0.8409260857443146, 0.81010388649627574, -0.40679677438693718, 
-0.44367130166338664]], [[0.46975488663284981, 0.05538088997511692, 0.28181116603516299, 0.90447077748810889], 
[-0.70642408692746739, 0.4992217777101422, -0.99634717981272058, -0.52884200664848424], [0.77333208330556524, 
0.56890042181004885, -0.95244727861145217, -0.80028684319724563]]]])
      arg1=Data(numpy.array([[[[0.58054499777805324, -0.47598878781671305, 0.37745487115494103, 0.86346386289982702], 
[-0.4626909081905235, 0.014254063410625184, -0.25123319602905769, -0.67848671461907406], [-0.67458112863104391, 
0.59481188249674188, 0.1855903882800658, -0.57524218933803528]], [[0.97075290656984614, 0.16518812230083513, 
0.4661549238413023, -0.005879778596060925], [0.51026944743156943, -0.74147633490268872, 0.86642643627434413, 
-0.79228655786465674], [-0.10016904611104449, 0.10634179921641063, 0.76286295106416513, 0.98293391977057043]]], 
[[[0.17839996211775389, 0.21298275629933117, 0.69149224363506945, -0.55428543693712617], [0.48913167331658802, 
0.018632955265404982, 0.4387537222003548, 0.14073064911063438], [-0.47073257735566743, 0.1929703590779297, 0.85828115031564711, 
-0.75925450525096139]], [[0.11898796887580865, 0.76748540038864843, 0.9448214457725328, -0.68273364410474335], 
[0.94935905895202488, -0.16029114174550685, 0.13228352829997148, -0.02652730913932988], [0.89412412499107297, 
0.53209616105427227, -0.068207385979760948, -0.61863336376472322]]], [[[0.95730262587366943, -0.23248363870347188, 
0.2037665282475738, 0.10077571000905672], [0.083645782907106447, 0.94184834016552865, -0.88401938037107586, 
0.40619867638887386], [-0.50209859466214346, -0.18628343640311118, -0.19749860763562999, -0.33479540161838761]], 
[[0.6849636240266086, -0.0035934923854694656, -0.4464547301912174, 0.37996296014871089], [-0.39077173143902422, 
0.9696037978571086, 0.051629517904298039, -0.55717317908534847], [-0.92030671214234427, 0.81374302920050434, 
-0.88471842433117409, 0.66673826387402713]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.46744102703542678, 0.11426019985327063, -0.95818837190184869, 
-0.41338790981571139], [-0.89389222605241336, 0.065334570872593867, 0.41229717310738812, -0.44525868755226172], 
[0.56601940107973991, -0.20044192033574881, 0.18850452913052274, -0.88982983958745554]], [[-0.17359765631314295, 
0.65832785319463571, 0.67597955803042709, 0.9231325328848774], [-0.7498911311405847, -0.28526315575627637, 
-0.77146496854541513, 0.25772973280054523], [-0.11882481479487961, -0.43717818215821191, -0.25437509001197478, 
0.53925719265170136]]], [[[0.99853762861398798, -0.47224890856616453, -0.52303989226703473, 0.54500976485963482], 
[-0.58664549540218558, 0.18249016687510844, -0.65523027591653316, 0.86864416931978261], [-0.43631906460503922, 
0.92153041582750728, 0.36650842247477966, 0.049124199930399293]], [[-0.32913150545945613, -0.43089855138150179, 
-0.033867704396374432, -0.48656039512906957], [-0.73571516323257558, -0.53026275153578872, 0.36903012401534752, 
-0.38320259378266464], [-0.7783722490247873, 0.52033011271475549, -0.58536901341864911, -0.96541528120210707]]], 
[[[-0.66345604741287589, -0.72975738364800025, 0.4078409262878937, 0.30955776485595177], [0.03006948878656468, 
-0.016997796482028038, 0.82835547548977528, -0.41609286006222757], [-0.85260987990426695, 0.17531606659629317, 
-0.010257024635147882, -0.012074135891210647]], [[0.62981312625639996, 0.82335516087640093, 0.059496654950929706, 
0.35375705041227312], [0.73934844890291229, 0.97585840642752975, -0.84191929190275361, -0.018577171923371605], 
[0.69038774040910478, -0.026902631136289434, 0.94024116864915763, -0.51150591535662704]]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.58054499777805324, -0.15183306579631028, 0.98594863414905065, 0.86346386289982702], 
[0.4007939057073524, 0.014254063410625184, 0.076166936848131028, -0.015591623626320272], [-0.67458112863104391, 
0.59481188249674188, 0.1855903882800658, 0.65095326433517031]], [[0.97075290656984614, 0.16518812230083513, 
0.59212025347785335, 0.83122656474479695], [0.51026944743156943, 0.86135881182639951, 0.86642643627434413, 0.5731565021744669], 
[-0.10016904611104449, 0.10634179921641063, 0.76286295106416513, 0.98293391977057043]]], [[[0.53270105359196052, 
0.41681501901552975, 0.69149224363506945, -0.29509500900305841], [0.4969220496535327, 0.74009856566202825, 0.57487999521149313, 
0.14073064911063438], [-0.47073257735566743, 0.42870575165153824, 0.85828115031564711, -0.64739611694503529]], 
[[0.11898796887580865, 0.76748540038864843, 0.9448214457725328, -0.68273364410474335], [0.94935905895202488, 
-0.16029114174550685, 0.13228352829997148, -0.02652730913932988], [0.89412412499107297, 0.53209616105427227, 
0.95248756894557896, -0.58098770763767149]]], [[[0.95730262587366943, -0.23248363870347188, 0.2037665282475738, 
0.10077571000905672], [0.083645782907106447, 0.94184834016552865, -0.011370710776029913, 0.79167198558767571], 
[0.8409260857443146, 0.81010388649627574, -0.19749860763562999, -0.33479540161838761]], [[0.6849636240266086, 
0.05538088997511692, 0.28181116603516299, 0.90447077748810889], [-0.39077173143902422, 0.9696037978571086, 
0.051629517904298039, -0.52884200664848424], [0.77333208330556524, 0.81374302920050434, -0.88471842433117409, 
0.66673826387402713]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.46744102703542678, 0.11426019985327063, 0.98594863414905065, 0.38876845683487726], 
[0.4007939057073524, 0.065334570872593867, 0.41229717310738812, -0.015591623626320272], [0.56601940107973991, 
-0.20044192033574881, 0.18850452913052274, 0.65095326433517031]], [[0.44947637187348088, 0.65832785319463571, 
0.67597955803042709, 0.9231325328848774], [0.16815843376338746, 0.86135881182639951, -0.58001385281386542, 0.5731565021744669], 
[-0.11882481479487961, -0.43717818215821191, -0.25437509001197478, 0.67153262857459728]]], [[[0.99853762861398798, 
0.41681501901552975, 0.65216139994222622, 0.54500976485963482], [0.4969220496535327, 0.74009856566202825, 0.57487999521149313, 
0.86864416931978261], [-0.43631906460503922, 0.92153041582750728, 0.36650842247477966, 0.049124199930399293]], 
[[-0.32913150545945613, -0.43089855138150179, -0.033867704396374432, -0.48656039512906957], [0.93294937179581128, 
-0.47490075605930882, 0.36903012401534752, -0.38320259378266464], [-0.70054800697348041, 0.52033011271475549, 
0.95248756894557896, -0.58098770763767149]]], [[[0.08563725603405703, -0.72975738364800025, 0.4078409262878937, 
0.30955776485595177], [0.08332034850987835, 0.81564051822547845, 0.82835547548977528, 0.79167198558767571], 
[0.8409260857443146, 0.81010388649627574, -0.010257024635147882, -0.012074135891210647]], [[0.62981312625639996, 
0.82335516087640093, 0.28181116603516299, 0.90447077748810889], [0.73934844890291229, 0.97585840642752975, 
-0.84191929190275361, -0.018577171923371605], [0.77333208330556524, 0.56890042181004885, 0.94024116864915763, 
-0.51150591535662704]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_taggedData_rank0(self):
      arg0=Data(0.121391948718,self.functionspace)
      arg1=Data(0.802207186761,self.functionspace)
      arg1.setTaggedValue(1,0.87499740935)
      res=maximum(arg0,arg1)
      ref=Data(0.802207186761,self.functionspace)
      ref.setTaggedValue(1,0.87499740935)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_taggedData_rank0(self):
      arg0=Data(numpy.array([-0.74290610332604712, -0.96489954045805582]),self.functionspace)
      arg1=Data(-0.302661500287,self.functionspace)
      arg1.setTaggedValue(1,-0.487219158279)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([-0.30266150028726524, -0.30266150028726524]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.48721915827901019, -0.48721915827901019]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_taggedData_rank0(self):
      arg0=Data(numpy.array([[0.70540052615388604, -0.31803862075235023, -0.18102391800155959, 0.37733634517100678, 
-0.26391151583282557], [0.40052661807828494, -0.31762469744963373, -0.0098686953021549506, -0.64867309239665616, 
0.48749355773112679], [0.67878884736345357, -0.79014871612611692, -0.26584000949039766, 0.22341585594305524, 
-0.57916757553408282], [0.16699615231275899, 0.77798872359195337, 0.18718392506650772, 0.23898465447919093, 
0.16157925161549902]]),self.functionspace)
      arg1=Data(-0.716757633534,self.functionspace)
      arg1.setTaggedValue(1,0.502833902765)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.70540052615388604, -0.31803862075235023, -0.18102391800155959, 0.37733634517100678, 
-0.26391151583282557], [0.40052661807828494, -0.31762469744963373, -0.0098686953021549506, -0.64867309239665616, 
0.48749355773112679], [0.67878884736345357, -0.71675763353441502, -0.26584000949039766, 0.22341585594305524, 
-0.57916757553408282], [0.16699615231275899, 0.77798872359195337, 0.18718392506650772, 0.23898465447919093, 
0.16157925161549902]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.70540052615388604, 0.50283390276450302, 0.50283390276450302, 0.50283390276450302, 
0.50283390276450302], [0.50283390276450302, 0.50283390276450302, 0.50283390276450302, 0.50283390276450302, 
0.50283390276450302], [0.67878884736345357, 0.50283390276450302, 0.50283390276450302, 0.50283390276450302, 
0.50283390276450302], [0.50283390276450302, 0.77798872359195337, 0.50283390276450302, 0.50283390276450302, 
0.50283390276450302]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_taggedData_rank0(self):
      arg0=Data(numpy.array([[[-0.19723925395471054, -0.31633169320843635], [-0.92752364914230578, -0.78685424054218611]], 
[[-0.69265508023018763, 0.46171880943015298], [-0.55140022193617022, -0.35369538247259791]], [[-0.36099395700596526, 
0.95747838116230954], [0.89891384455772738, 0.046475775151562981]], [[0.91738526026264267, -0.57601253734342617], 
[0.42856864753374091, -0.78730292423599235]], [[-0.27189694644867357, -0.5468765883753437], [0.084656296613520476, 
-0.31592810273635852]], [[0.10676619450615843, -0.4928679685399866], [0.51063583544931235, 
0.42265548609189341]]]),self.functionspace)
      arg1=Data(0.135414789959,self.functionspace)
      arg1.setTaggedValue(1,0.290722128995)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.13541478995921596, 0.13541478995921596], [0.13541478995921596, 0.13541478995921596]], 
[[0.13541478995921596, 0.46171880943015298], [0.13541478995921596, 0.13541478995921596]], [[0.13541478995921596, 
0.95747838116230954], [0.89891384455772738, 0.13541478995921596]], [[0.91738526026264267, 0.13541478995921596], 
[0.42856864753374091, 0.13541478995921596]], [[0.13541478995921596, 0.13541478995921596], [0.13541478995921596, 
0.13541478995921596]], [[0.13541478995921596, 0.13541478995921596], [0.51063583544931235, 
0.42265548609189341]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.29072212899522709, 0.29072212899522709], [0.29072212899522709, 
0.29072212899522709]], [[0.29072212899522709, 0.46171880943015298], [0.29072212899522709, 0.29072212899522709]], 
[[0.29072212899522709, 0.95747838116230954], [0.89891384455772738, 0.29072212899522709]], [[0.91738526026264267, 
0.29072212899522709], [0.42856864753374091, 0.29072212899522709]], [[0.29072212899522709, 0.29072212899522709], 
[0.29072212899522709, 0.29072212899522709]], [[0.29072212899522709, 0.29072212899522709], [0.51063583544931235, 
0.42265548609189341]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_taggedData_rank0(self):
      arg0=Data(numpy.array([[[[-0.12458163237947906, -0.59822478664598178, 0.17341920563200719, 0.41623144103464194], 
[0.20050395598082882, 0.84492937133851398, -0.68660745369533061, 0.81960666534564464], [0.51577768489615017, 
0.88086492036408615, 0.78963735691956938, -0.89338404328780441]], [[0.68118201371776976, -0.73850013986141483, 
0.95757040152737649, -0.63229876199900747], [-0.70331888242174334, -0.79691061758469872, -0.80500031627132662, 
-0.79787492241976965], [-0.85968481737238811, -0.39647041135589745, 0.44659800463404631, -0.88510772756112122]]], 
[[[-0.64416374836090906, 0.62062562363899221, 0.26964189871320698, 0.59423645220954824], [0.035374754592919944, 
0.95516521280588274, -0.27026094526942934, -0.81057656724939275], [0.26244861645691842, 0.090637778830348292, 
0.4229195698217818, 0.65319208554143326]], [[-0.29987259338263628, 0.16275142756663219, 0.9703193425076746, 
-0.33882052600541757], [-0.8335242661793445, -0.46420881700311178, -0.37027238936411666, 0.2880813478071087], 
[-0.15573435137820657, -0.46446688238283662, 0.70397475205135707, -0.95345788469841186]]], [[[-0.63784594818254337, 
-0.90342686399656413, -0.80175838335550531, 0.34994329009839276], [0.84545583011894587, -0.76575410641336306, 
0.66599420476227866, 0.21586993966488643], [-0.29456711132643121, 0.37986828857951127, 0.15820321508940305, 
0.80383769939257022]], [[-0.60579491765781501, 0.22779643602248201, -0.3115357278626818, 0.48011732837019672], 
[-0.18951607090714551, -0.1288740871321179, -0.38397337979444468, 0.11947949282813775], [0.8422139735678178, 
0.5801082073730095, 0.53540520324088714, 0.69172984512057445]]]]),self.functionspace)
      arg1=Data(-0.853304402567,self.functionspace)
      arg1.setTaggedValue(1,-0.598547923213)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.12458163237947906, -0.59822478664598178, 0.17341920563200719, 0.41623144103464194], 
[0.20050395598082882, 0.84492937133851398, -0.68660745369533061, 0.81960666534564464], [0.51577768489615017, 
0.88086492036408615, 0.78963735691956938, -0.85330440256659013]], [[0.68118201371776976, -0.73850013986141483, 
0.95757040152737649, -0.63229876199900747], [-0.70331888242174334, -0.79691061758469872, -0.80500031627132662, 
-0.79787492241976965], [-0.85330440256659013, -0.39647041135589745, 0.44659800463404631, -0.85330440256659013]]], 
[[[-0.64416374836090906, 0.62062562363899221, 0.26964189871320698, 0.59423645220954824], [0.035374754592919944, 
0.95516521280588274, -0.27026094526942934, -0.81057656724939275], [0.26244861645691842, 0.090637778830348292, 
0.4229195698217818, 0.65319208554143326]], [[-0.29987259338263628, 0.16275142756663219, 0.9703193425076746, 
-0.33882052600541757], [-0.8335242661793445, -0.46420881700311178, -0.37027238936411666, 0.2880813478071087], 
[-0.15573435137820657, -0.46446688238283662, 0.70397475205135707, -0.85330440256659013]]], [[[-0.63784594818254337, 
-0.85330440256659013, -0.80175838335550531, 0.34994329009839276], [0.84545583011894587, -0.76575410641336306, 
0.66599420476227866, 0.21586993966488643], [-0.29456711132643121, 0.37986828857951127, 0.15820321508940305, 
0.80383769939257022]], [[-0.60579491765781501, 0.22779643602248201, -0.3115357278626818, 0.48011732837019672], 
[-0.18951607090714551, -0.1288740871321179, -0.38397337979444468, 0.11947949282813775], [0.8422139735678178, 
0.5801082073730095, 0.53540520324088714, 0.69172984512057445]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.12458163237947906, -0.59822478664598178, 0.17341920563200719, 
0.41623144103464194], [0.20050395598082882, 0.84492937133851398, -0.59854792321265093, 0.81960666534564464], 
[0.51577768489615017, 0.88086492036408615, 0.78963735691956938, -0.59854792321265093]], [[0.68118201371776976, 
-0.59854792321265093, 0.95757040152737649, -0.59854792321265093], [-0.59854792321265093, -0.59854792321265093, 
-0.59854792321265093, -0.59854792321265093], [-0.59854792321265093, -0.39647041135589745, 0.44659800463404631, 
-0.59854792321265093]]], [[[-0.59854792321265093, 0.62062562363899221, 0.26964189871320698, 0.59423645220954824], 
[0.035374754592919944, 0.95516521280588274, -0.27026094526942934, -0.59854792321265093], [0.26244861645691842, 
0.090637778830348292, 0.4229195698217818, 0.65319208554143326]], [[-0.29987259338263628, 0.16275142756663219, 
0.9703193425076746, -0.33882052600541757], [-0.59854792321265093, -0.46420881700311178, -0.37027238936411666, 
0.2880813478071087], [-0.15573435137820657, -0.46446688238283662, 0.70397475205135707, -0.59854792321265093]]], 
[[[-0.59854792321265093, -0.59854792321265093, -0.59854792321265093, 0.34994329009839276], [0.84545583011894587, 
-0.59854792321265093, 0.66599420476227866, 0.21586993966488643], [-0.29456711132643121, 0.37986828857951127, 
0.15820321508940305, 0.80383769939257022]], [[-0.59854792321265093, 0.22779643602248201, -0.3115357278626818, 
0.48011732837019672], [-0.18951607090714551, -0.1288740871321179, -0.38397337979444468, 0.11947949282813775], 
[0.8422139735678178, 0.5801082073730095, 0.53540520324088714, 0.69172984512057445]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_taggedData_rank1(self):
      arg0=Data(0.191044066235,self.functionspace)
      arg1=Data(numpy.array([0.18878917078137847, -0.19256801725704031]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.11227902493879283, -0.54746427961671329]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.19104406623487868, 0.19104406623487868]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.19104406623487868, 0.19104406623487868]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([0.69515429789349747, 0.065493511797723558]),self.functionspace)
      arg1=Data(numpy.array([0.80145161365432394, 0.82508591090347627]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.12199277424988386, -0.10766451613470585]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.80145161365432394, 0.82508591090347627]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.69515429789349747, 0.065493511797723558]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_taggedData_rank2(self):
      arg0=Data(-0.132887771896,self.functionspace)
      arg1=Data(numpy.array([[0.94555830246720896, -0.25406975197643122, -0.15450801717376339, 0.0091071314536093073, 
-0.73612340527463815], [0.7733172160011812, 0.78435965271730601, -0.38234841008560627, 0.97891965593680164, 
0.54502107251024157], [-0.13665535676381957, 0.74368983883655582, 0.72546122427200532, 0.84697528289999302, 
-0.11539695822929419], [0.29314031525253448, -0.61479198533787383, 0.74438108018186555, 0.55349157033049501, 
-0.55604746140957273]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.74056719902765433, 0.10100448304414678, -0.19394562368214108, -0.18626986494349529, 
0.11110944753665963], [-0.594418590688228, -0.2254513164133487, 0.27304611299607129, -0.31216213030533524, 
0.70300024422937102], [0.74644548826249402, -0.33558121285136999, -0.87468352477362421, 0.70047147996322812, 
0.41768429824086595], [0.56691405278146334, -0.75456571237180148, 0.29466306993921654, 0.42022226418575559, 
-0.5033789428584412]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.94555830246720896, -0.13288777189649514, -0.13288777189649514, 0.0091071314536093073, 
-0.13288777189649514], [0.7733172160011812, 0.78435965271730601, -0.13288777189649514, 0.97891965593680164, 
0.54502107251024157], [-0.13288777189649514, 0.74368983883655582, 0.72546122427200532, 0.84697528289999302, 
-0.11539695822929419], [0.29314031525253448, -0.13288777189649514, 0.74438108018186555, 0.55349157033049501, 
-0.13288777189649514]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.74056719902765433, 0.10100448304414678, -0.13288777189649514, -0.13288777189649514, 
0.11110944753665963], [-0.13288777189649514, -0.13288777189649514, 0.27304611299607129, -0.13288777189649514, 
0.70300024422937102], [0.74644548826249402, -0.13288777189649514, -0.13288777189649514, 0.70047147996322812, 
0.41768429824086595], [0.56691405278146334, -0.13288777189649514, 0.29466306993921654, 0.42022226418575559, 
-0.13288777189649514]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[-0.044580857181160782, 0.85604091545943461, -0.25709984937220609, 0.9812378409675091, 
-0.087945933024101475], [-0.66027194325369076, -0.76198389185626691, 0.94940653328555658, -0.92814666543983804, 
0.64500125175748702], [-0.29690521970236672, -0.95189473676409109, 0.50233936436037152, -0.23074502514572726, 
0.6299701171022043], [-0.12871549934614768, -0.14273943770437558, -0.1234699264157606, -0.56884730200250133, 
-0.55806124082082254]]),self.functionspace)
      arg1=Data(numpy.array([[-0.70032557439728671, -0.51331046006049696, 0.61189140749091431, 0.17391303359656041, 
-0.6049103609964932], [0.3175485088685106, 0.39873872898640772, 0.15168398697589436, -0.24244996997361667, 
-0.8342778525661827], [0.43225443660508511, 0.92587936399107118, -0.28137933610758825, -0.081573946800039154, 
0.23032873474094195], [0.14015253512761872, -0.90542987774290928, -0.35481530938682071, -0.31148353141201368, 
-0.055086011367168286]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.80220668203000312, 0.47476245365311898, -0.5593555139154982, -0.5480764064702115, 
0.47892069431381912], [0.80551116244117149, 0.26173910494717467, 0.91103482741194464, -0.37911700884105604, 
0.058182537467474704], [-0.76331248508594918, 0.48471594396268713, 0.34686292858310797, 0.72855868759979892, 
-0.36091715997720097], [-0.87533748179733251, -0.030174504166970539, -0.5690356911726111, 0.23981567354560474, 
-0.63561344893261262]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.044580857181160782, 0.85604091545943461, 0.61189140749091431, 0.9812378409675091, 
-0.087945933024101475], [0.3175485088685106, 0.39873872898640772, 0.94940653328555658, -0.24244996997361667, 
0.64500125175748702], [0.43225443660508511, 0.92587936399107118, 0.50233936436037152, -0.081573946800039154, 
0.6299701171022043], [0.14015253512761872, -0.14273943770437558, -0.1234699264157606, -0.31148353141201368, 
-0.055086011367168286]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.80220668203000312, 0.85604091545943461, -0.25709984937220609, 0.9812378409675091, 
0.47892069431381912], [0.80551116244117149, 0.26173910494717467, 0.94940653328555658, -0.37911700884105604, 
0.64500125175748702], [-0.29690521970236672, 0.48471594396268713, 0.50233936436037152, 0.72855868759979892, 
0.6299701171022043], [-0.12871549934614768, -0.030174504166970539, -0.1234699264157606, 0.23981567354560474, 
-0.55806124082082254]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_taggedData_rank3(self):
      arg0=Data(-0.891016896475,self.functionspace)
      arg1=Data(numpy.array([[[0.9325086027176237, 0.70747828597483564], [-0.51072449936485809, -0.70720876741336847]], 
[[-0.82083221813359342, 0.37684277951746115], [0.96335665861017694, -0.52166699275939132]], [[-0.64161247297755519, 
-0.4371207835847859], [-0.0063867228458982517, -0.26244767945153713]], [[-0.702581461681131, 0.76375137652412373], 
[-0.99445351839562957, -0.83814683135694534]], [[-0.47080166721252348, 0.5731151770230607], [0.64565814437219338, 
-0.44496251299419121]], [[-0.92182224772813992, -0.35129925337038759], [-0.46113685731971121, 
0.70053287717877222]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.41863886213543844, -0.63035383492945862], [0.46000590292650845, 
0.29345092547600005]], [[-0.032355440081924813, -0.82086331932436418], [-0.023719548418603242, -0.7577034515440948]], 
[[-0.02669531824519833, 0.021735114531001853], [0.27458583187952823, -0.23737890103910586]], [[-0.10580988121991775, 
-0.6948329873640402], [-0.42026719884892927, 0.91929411913159975]], [[-0.20617136146521964, 0.67749564787621885], 
[0.58893421052210759, -0.026475337782694286]], [[-0.22401474946665689, -0.94843838439336947], [0.10656878935770209, 
-0.13495839759094852]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.9325086027176237, 0.70747828597483564], [-0.51072449936485809, -0.70720876741336847]], 
[[-0.82083221813359342, 0.37684277951746115], [0.96335665861017694, -0.52166699275939132]], [[-0.64161247297755519, 
-0.4371207835847859], [-0.0063867228458982517, -0.26244767945153713]], [[-0.702581461681131, 0.76375137652412373], 
[-0.89101689647519033, -0.83814683135694534]], [[-0.47080166721252348, 0.5731151770230607], [0.64565814437219338, 
-0.44496251299419121]], [[-0.89101689647519033, -0.35129925337038759], [-0.46113685731971121, 
0.70053287717877222]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.41863886213543844, -0.63035383492945862], [0.46000590292650845, 
0.29345092547600005]], [[-0.032355440081924813, -0.82086331932436418], [-0.023719548418603242, -0.7577034515440948]], 
[[-0.02669531824519833, 0.021735114531001853], [0.27458583187952823, -0.23737890103910586]], [[-0.10580988121991775, 
-0.6948329873640402], [-0.42026719884892927, 0.91929411913159975]], [[-0.20617136146521964, 0.67749564787621885], 
[0.58893421052210759, -0.026475337782694286]], [[-0.22401474946665689, -0.89101689647519033], [0.10656878935770209, 
-0.13495839759094852]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_taggedData_rank3(self):
      arg0=Data(numpy.array([[[0.37365671971655501, -0.95866990670637753], [0.58908423455136782, -0.44089893492224141]], 
[[-0.64212573215482505, -0.96914971933032756], [-0.17049222222538374, 0.15111761979145166]], [[-0.26812991045753631, 
-0.1882821241490733], [-0.91960941919432271, -0.24676784714716571]], [[0.8224396831862526, -0.4329310126257977], 
[-0.79463781374347087, 0.57140395354648432]], [[0.24605578055524258, 0.35728530397077574], [-0.97321913541911909, 
0.48252745798292751]], [[-0.69818806483253271, -0.61856055423257073], [0.8746971566869024, 
-0.92535601399166767]]]),self.functionspace)
      arg1=Data(numpy.array([[[-0.16070586595581537, -0.26725510149956344], [0.22495113831112179, -0.47410602257703705]], 
[[-0.55027629698141123, 0.53682890174660725], [-0.71491913027401721, 0.4534849072439433]], [[0.015344558303559852, 
0.98796774669442033], [0.43890160098747288, 0.45820587320625106]], [[0.45030491171400988, 0.22168600024660701], 
[0.27116664173706395, -0.87107827200332744]], [[0.48799286894713179, -0.15687952682290374], [0.044389175568014583, 
-0.77951047019617503]], [[0.97306453726871589, -0.5587268814786317], [-0.20320541829666805, 
0.3678772498851004]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.0012886532899734071, 0.3150147231932463], [-0.099170559617714149, 
-0.50900431355155629]], [[-0.82228359777233817, -0.78291417837981081], [-0.0076132178370915682, 0.94214643314267588]], 
[[-0.13570465826971922, 0.65946407437584154], [0.63329347372632627, 0.16635825714364061]], [[-0.077682415840976571, 
-0.8750012715662161], [0.039341287290900473, 0.2199617117163728]], [[-0.57709355664844986, -0.25251599432482319], 
[-0.66217700675770219, -0.71459546673531249]], [[-0.51352836218382314, -0.80480127503459808], [-0.82200032084280461, 
-0.64848389450301513]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.37365671971655501, -0.26725510149956344], [0.58908423455136782, -0.44089893492224141]], 
[[-0.55027629698141123, 0.53682890174660725], [-0.17049222222538374, 0.4534849072439433]], [[0.015344558303559852, 
0.98796774669442033], [0.43890160098747288, 0.45820587320625106]], [[0.8224396831862526, 0.22168600024660701], 
[0.27116664173706395, 0.57140395354648432]], [[0.48799286894713179, 0.35728530397077574], [0.044389175568014583, 
0.48252745798292751]], [[0.97306453726871589, -0.5587268814786317], [0.8746971566869024, 
0.3678772498851004]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.37365671971655501, 0.3150147231932463], [0.58908423455136782, 
-0.44089893492224141]], [[-0.64212573215482505, -0.78291417837981081], [-0.0076132178370915682, 0.94214643314267588]], 
[[-0.13570465826971922, 0.65946407437584154], [0.63329347372632627, 0.16635825714364061]], [[0.8224396831862526, 
-0.4329310126257977], [0.039341287290900473, 0.57140395354648432]], [[0.24605578055524258, 0.35728530397077574], 
[-0.66217700675770219, 0.48252745798292751]], [[-0.51352836218382314, -0.61856055423257073], [0.8746971566869024, 
-0.64848389450301513]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_taggedData_rank4(self):
      arg0=Data(-0.935809053062,self.functionspace)
      arg1=Data(numpy.array([[[[0.94323367504932754, 0.75043469724469358, 0.23889440516936578, 0.95420440815335872], 
[-0.56204824740798398, 0.75944243135960021, 0.30627342617954212, 0.63408115161677991], [-0.53382195077039052, 
-0.07330185156963176, -0.45909932029795142, 0.24646544731995279]], [[-0.18508046873734219, 0.80982790094001689, 
-0.33239328318495387, 0.13143779990210236], [0.5473229325959208, 0.94720944694356923, -0.715613069945924, 
-0.31084358313547411], [-0.3634542445783544, 0.36559825061099449, -0.83461079195174914, -0.46670894854169376]]], 
[[[-0.58310807256385022, 0.50593022547925171, -0.43058191591512407, -0.95895779399621417], [-0.57057964070667411, 
0.047984045734246195, -0.2725165401098586, -0.40839044936075131], [-0.70089248436306861, -0.75308454340162045, 
-0.62801846775423287, 0.57487689894384619]], [[0.076655772437350223, 0.98135221037614118, -0.055723435045593517, 
-0.097547133361197913], [-0.59731206113295743, 0.11625604348180207, -0.79274660073366432, 0.70998706113619292], 
[-0.63183678425277212, 0.66979190024534185, -0.16900332658838191, 0.10957819941236413]]], [[[-0.85864820040908763, 
0.36129280166431066, -0.16432595438853292, -0.33936584368341682], [-0.092573706577926851, 0.63223829113682672, 
0.89427450221604698, -0.33450818273232774], [0.9516554907810888, 0.78565535271713971, 0.65371076370893011, 
0.90366038286745054]], [[-0.78574722247805817, -0.88516230192953849, -0.89433620070831998, 0.066066240762615358], 
[-0.6347596849537207, -0.058613355587173066, 0.48717276237338503, -0.43467656120662568], [0.82033146654420763, 
0.10448737437829325, 0.55801623370482756, 0.98628372428516053]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.37319690653678461, 0.96816837763651842, 0.54197892432948125, 0.4145444570793424], 
[0.1589679670934272, -0.89173172290665637, 0.12882352434855981, -0.27667628201950234], [-0.66731682515298973, 
-0.90062475869131431, -0.11483469751570285, -0.0085920205034029085]], [[0.26748499040337692, -0.070632119990836628, 
-0.38461852821074127, 0.10665093473312393], [-0.95597103461713395, -0.77004750380558407, 0.23245288316351531, 
0.83705565419364381], [0.39650287844856225, -0.2523975731230057, -0.86800679915997825, -0.99749116519001246]]], 
[[[0.96528920223332881, 0.62880977093699686, 0.053406648810849777, -0.80119531024482682], [-0.1960029338003213, 
0.87681792954741256, -0.6176115768758923, 0.51568487957685161], [0.5196706962976283, 0.1138525778114674, 0.41147841116292083, 
0.56800974807485516]], [[0.82711178202336999, 0.16326171975404491, 0.92334045640680551, 0.11099885042960533], 
[0.63059371599109704, -0.12235440421726174, -0.39929056523279227, -0.47288045445702998], [-0.26298695836117414, 
0.61457691131432179, -0.8229866040010505, 0.6011109969184083]]], [[[0.25698041723699405, 0.22458408564835253, 
0.74325854221196308, 0.43803105051154367], [-0.035606838330939139, -0.93134062390725103, 0.44709064523174158, 
0.39054509518098324], [0.66234331262164847, 0.0035485492730196189, 0.39545246249979416, -0.9441071213140193]], 
[[0.032325364478498431, -0.70262802392873036, -0.98671212083451332, 0.44413714967574358], [0.96414294595738004, 
-0.57313127103141381, -0.73421868717713368, -0.59117477480893288], [0.10669155590334123, 0.7008361803327039, 
-0.80578500370812578, -0.20859126604789102]]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.94323367504932754, 0.75043469724469358, 0.23889440516936578, 0.95420440815335872], 
[-0.56204824740798398, 0.75944243135960021, 0.30627342617954212, 0.63408115161677991], [-0.53382195077039052, 
-0.07330185156963176, -0.45909932029795142, 0.24646544731995279]], [[-0.18508046873734219, 0.80982790094001689, 
-0.33239328318495387, 0.13143779990210236], [0.5473229325959208, 0.94720944694356923, -0.715613069945924, 
-0.31084358313547411], [-0.3634542445783544, 0.36559825061099449, -0.83461079195174914, -0.46670894854169376]]], 
[[[-0.58310807256385022, 0.50593022547925171, -0.43058191591512407, -0.93580905306181816], [-0.57057964070667411, 
0.047984045734246195, -0.2725165401098586, -0.40839044936075131], [-0.70089248436306861, -0.75308454340162045, 
-0.62801846775423287, 0.57487689894384619]], [[0.076655772437350223, 0.98135221037614118, -0.055723435045593517, 
-0.097547133361197913], [-0.59731206113295743, 0.11625604348180207, -0.79274660073366432, 0.70998706113619292], 
[-0.63183678425277212, 0.66979190024534185, -0.16900332658838191, 0.10957819941236413]]], [[[-0.85864820040908763, 
0.36129280166431066, -0.16432595438853292, -0.33936584368341682], [-0.092573706577926851, 0.63223829113682672, 
0.89427450221604698, -0.33450818273232774], [0.9516554907810888, 0.78565535271713971, 0.65371076370893011, 
0.90366038286745054]], [[-0.78574722247805817, -0.88516230192953849, -0.89433620070831998, 0.066066240762615358], 
[-0.6347596849537207, -0.058613355587173066, 0.48717276237338503, -0.43467656120662568], [0.82033146654420763, 
0.10448737437829325, 0.55801623370482756, 0.98628372428516053]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.37319690653678461, 0.96816837763651842, 0.54197892432948125, 0.4145444570793424], 
[0.1589679670934272, -0.89173172290665637, 0.12882352434855981, -0.27667628201950234], [-0.66731682515298973, 
-0.90062475869131431, -0.11483469751570285, -0.0085920205034029085]], [[0.26748499040337692, -0.070632119990836628, 
-0.38461852821074127, 0.10665093473312393], [-0.93580905306181816, -0.77004750380558407, 0.23245288316351531, 
0.83705565419364381], [0.39650287844856225, -0.2523975731230057, -0.86800679915997825, -0.93580905306181816]]], 
[[[0.96528920223332881, 0.62880977093699686, 0.053406648810849777, -0.80119531024482682], [-0.1960029338003213, 
0.87681792954741256, -0.6176115768758923, 0.51568487957685161], [0.5196706962976283, 0.1138525778114674, 0.41147841116292083, 
0.56800974807485516]], [[0.82711178202336999, 0.16326171975404491, 0.92334045640680551, 0.11099885042960533], 
[0.63059371599109704, -0.12235440421726174, -0.39929056523279227, -0.47288045445702998], [-0.26298695836117414, 
0.61457691131432179, -0.8229866040010505, 0.6011109969184083]]], [[[0.25698041723699405, 0.22458408564835253, 
0.74325854221196308, 0.43803105051154367], [-0.035606838330939139, -0.93134062390725103, 0.44709064523174158, 
0.39054509518098324], [0.66234331262164847, 0.0035485492730196189, 0.39545246249979416, -0.93580905306181816]], 
[[0.032325364478498431, -0.70262802392873036, -0.93580905306181816, 0.44413714967574358], [0.96414294595738004, 
-0.57313127103141381, -0.73421868717713368, -0.59117477480893288], [0.10669155590334123, 0.7008361803327039, 
-0.80578500370812578, -0.20859126604789102]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_taggedData_rank4(self):
      arg0=Data(numpy.array([[[[0.10845550544873728, -0.48673778143663937, -0.22758575972764006, -0.043144699169465284], 
[-0.056497115103625317, 0.021753318714725411, -0.14895150982487415, -0.29090658723806451], [-0.33108625396830771, 
0.52984859700828979, -0.86606702812901659, -0.39974495069158755]], [[0.80153882125099485, -0.32093090408244285, 
-0.9985471625693203, 0.31923356832215943], [-0.69030935671099458, 0.53561118602122959, 0.8491139388367257, 
0.72195365404201461], [-0.93786399905986362, 0.026041312211865009, -0.22890559505367936, -0.24610389122722509]]], 
[[[0.80325402914774902, 0.8459559099146523, 0.082082059186315615, -0.29051068321559259], [-0.42962178746605062, 
0.56712780159219078, -0.76002013730504325, -0.20170316329475191], [-0.7586296269896029, 0.050879585698642149, 
0.76795059503363072, -0.8101716505978549]], [[-0.78838632304939882, -0.041097735820624592, 0.21104208198163876, 
-0.66879037611420689], [0.032451538091870624, 0.29638695402919923, -0.65196181789373875, 0.55402305292580056], 
[0.82936728409111815, -0.98651990875459172, -0.29595117058779441, 0.77358656668899006]]], [[[-0.14969692922247835, 
0.21060062729011664, 0.65486061934273643, -0.825536583685887], [0.51870803147561095, 0.5798651899249514, 0.95414506891871076, 
-0.0013759504034529524], [0.25097599259097336, -0.30583012400339493, -0.63244303720603168, -0.60453712200633714]], 
[[-0.92063746830516835, -0.042632226113337612, -0.085570251662872243, 0.70054554919919632], [0.30531042027844113, 
0.56237981644588997, 0.8144988455107125, -0.78022543604342975], [0.68820007664728355, 0.21737283229892546, 0.25311229803675039, 
-0.77186399794036076]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[-0.00032994280755604954, 0.62099392559096311, 0.42932150083049514, 0.89339418677260141], 
[-0.57326478847840634, -0.29289689789152806, -0.075582392955726885, 0.035008291874085273], [0.14996607225052139, 
-0.65116700704462005, 0.82891503021350332, 0.74528593031096513]], [[0.38387895484736623, 0.50218316647087469, 
-0.79899919471296754, 0.49765989217580064], [-0.84098635909127561, 0.96116887286245634, -0.23892215641571712, 
0.65530711875098269], [-0.80612470265511726, 0.09632538549297176, 0.25700143742656278, -0.65226607965150918]]], 
[[[0.26757684029672602, 0.15455891205706895, -0.44351174410198868, 0.064343841740669871], [-0.78833832442092677, 
-0.30886745216602751, 0.48078898247343438, 0.7056167149071586], [-0.4614997399646148, 0.20421412301056607, 0.71373033075787862, 
0.092014366665211789]], [[-0.71920997096988626, 0.44199047128023383, 0.49165293041182823, -0.49907169374155069], 
[-0.72955652024367712, -0.27453869614110893, 0.5528484672394387, 0.26391574937519646], [0.75149155665165868, 
-0.24656152810147525, 0.47833844674272386, -0.69972346734578461]]], [[[0.36506781182422143, 0.80433027945096769, 
0.50656906442458971, 0.32269863125009213], [-0.35019131237572254, 0.57294140218016687, 0.82012396838963997, 
-0.030923328277502549], [0.015545642373764323, 0.78888038734087385, -0.13610586472262276, 0.91805607416477586]], 
[[-0.20041983847796052, 0.24682761853703061, 0.77572566143556698, 0.51880485408180999], [-0.58049021614141516, 
-0.55968368721395678, 0.49609616802278622, -0.54320817069727534], [0.65654306131902729, -0.25813279442687742, 
0.83808785233029348, -0.99580052453682599]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.60332564414176448, 0.78174497779889562, 0.48250144789091598, 
0.43748547827674167], [-0.031532065425390776, -0.060724048674144049, 0.46186848668264968, 0.32331345297088254], 
[-0.21150653577402712, -0.66338034072353014, 0.63187368411638767, 0.77341309022114824]], [[-0.30139820535668682, 
-0.58697963987269186, 0.41762718688763267, -0.94465454786256009], [0.18221871160438585, -0.06931834132824144, 
0.25972462357259429, 0.00076619605568706284], [-0.55876663198288146, -0.61658481734923298, 0.83700655360084864, 
-0.23851034966360252]]], [[[0.11107390161348674, -0.20575151010029957, -0.98012804616264848, 0.11552308626828789], 
[-0.67293485644248463, 0.96140918537691578, 0.63915137065856364, 0.59999045151632768], [0.48837931622591269, 
-0.14644653183032941, -0.95633965036259783, 0.99743471563239328]], [[0.30476436915068472, -0.815728714648283, 
0.11152573207011862, -0.56404563768211613], [-0.63410846910627905, -0.60126860075120447, -0.38781022401393317, 
0.23095581010629385], [0.062775807585675247, -0.4161575449102326, 0.22837399156699001, -0.1421547161185801]]], 
[[[0.35746303768748322, -0.85176228652149777, -0.71180490978585342, -0.9941058045304374], [0.1724377444362255, 
0.28809532060771836, -0.77983142056051991, 0.1715823996990371], [-0.2178550296158559, -0.73984877147432582, 
-0.17367542863839414, -0.62707583964322633]], [[-0.77991193081480215, 0.62806871587386204, -0.88082940145954658, 
0.22588341520381183], [-0.66267033913540918, -0.20069383663120055, -0.57508164261899619, 0.33871256324488663], 
[-0.074254009640646546, 0.71115919570890851, 0.15639013755074571, -0.92671870543888657]]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.10845550544873728, 0.62099392559096311, 0.42932150083049514, 0.89339418677260141], 
[-0.056497115103625317, 0.021753318714725411, -0.075582392955726885, 0.035008291874085273], [0.14996607225052139, 
0.52984859700828979, 0.82891503021350332, 0.74528593031096513]], [[0.80153882125099485, 0.50218316647087469, 
-0.79899919471296754, 0.49765989217580064], [-0.69030935671099458, 0.96116887286245634, 0.8491139388367257, 
0.72195365404201461], [-0.80612470265511726, 0.09632538549297176, 0.25700143742656278, -0.24610389122722509]]], 
[[[0.80325402914774902, 0.8459559099146523, 0.082082059186315615, 0.064343841740669871], [-0.42962178746605062, 
0.56712780159219078, 0.48078898247343438, 0.7056167149071586], [-0.4614997399646148, 0.20421412301056607, 0.76795059503363072, 
0.092014366665211789]], [[-0.71920997096988626, 0.44199047128023383, 0.49165293041182823, -0.49907169374155069], 
[0.032451538091870624, 0.29638695402919923, 0.5528484672394387, 0.55402305292580056], [0.82936728409111815, 
-0.24656152810147525, 0.47833844674272386, 0.77358656668899006]]], [[[0.36506781182422143, 0.80433027945096769, 
0.65486061934273643, 0.32269863125009213], [0.51870803147561095, 0.5798651899249514, 0.95414506891871076, 
-0.0013759504034529524], [0.25097599259097336, 0.78888038734087385, -0.13610586472262276, 0.91805607416477586]], 
[[-0.20041983847796052, 0.24682761853703061, 0.77572566143556698, 0.70054554919919632], [0.30531042027844113, 
0.56237981644588997, 0.8144988455107125, -0.54320817069727534], [0.68820007664728355, 0.21737283229892546, 0.83808785233029348, 
-0.77186399794036076]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.10845550544873728, 0.78174497779889562, 0.48250144789091598, 0.43748547827674167], 
[-0.031532065425390776, 0.021753318714725411, 0.46186848668264968, 0.32331345297088254], [-0.21150653577402712, 
0.52984859700828979, 0.63187368411638767, 0.77341309022114824]], [[0.80153882125099485, -0.32093090408244285, 
0.41762718688763267, 0.31923356832215943], [0.18221871160438585, 0.53561118602122959, 0.8491139388367257, 0.72195365404201461], 
[-0.55876663198288146, 0.026041312211865009, 0.83700655360084864, -0.23851034966360252]]], [[[0.80325402914774902, 
0.8459559099146523, 0.082082059186315615, 0.11552308626828789], [-0.42962178746605062, 0.96140918537691578, 
0.63915137065856364, 0.59999045151632768], [0.48837931622591269, 0.050879585698642149, 0.76795059503363072, 
0.99743471563239328]], [[0.30476436915068472, -0.041097735820624592, 0.21104208198163876, -0.56404563768211613], 
[0.032451538091870624, 0.29638695402919923, -0.38781022401393317, 0.55402305292580056], [0.82936728409111815, 
-0.4161575449102326, 0.22837399156699001, 0.77358656668899006]]], [[[0.35746303768748322, 0.21060062729011664, 
0.65486061934273643, -0.825536583685887], [0.51870803147561095, 0.5798651899249514, 0.95414506891871076, 0.1715823996990371], 
[0.25097599259097336, -0.30583012400339493, -0.17367542863839414, -0.60453712200633714]], [[-0.77991193081480215, 
0.62806871587386204, -0.085570251662872243, 0.70054554919919632], [0.30531042027844113, 0.56237981644588997, 
0.8144988455107125, 0.33871256324488663], [0.68820007664728355, 0.71115919570890851, 0.25311229803675039, 
-0.77186399794036076]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_float_rank0(self):
      arg0=Data(0.781189370724,self.functionspace)
      arg0.setTaggedValue(1,0.218741881483)
      arg1=0.0125402541995
      res=maximum(arg0,arg1)
      ref=Data(0.781189370724,self.functionspace)
      ref.setTaggedValue(1,0.218741881483)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_float_rank0(self):
      arg0=Data(numpy.array([-0.51812545546824396, 0.55330867489301871]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.5640202814852131, 0.83729192015196108]))
      arg1=-0.92649530698
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([-0.51812545546824396, 0.55330867489301871]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.5640202814852131, 0.83729192015196108]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[0.93480005656464904, 0.3326044609673382, -0.97347037521913871, -0.2720821447149282, 
0.25288249094993254], [0.078016162222240615, 0.14026928157959651, 0.47798638498447099, -0.79465964907511299, 
0.0023462007272536667], [-0.20523579180798057, 0.38896152402621231, -0.65481776935648583, 0.68305255859923619, 
0.11792006499793128], [0.3279746094109548, 0.61188140404289282, -0.71100229248176428, 0.2750960084779599, 
-0.24274334248074125]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.82939215060104843, -0.078441374528316121, -0.18319692474211235, 
-0.37190277953776363, -0.51652031544431543], [0.97796440308165078, -0.9342307921555113, -0.011057605869359444, 
0.3714235134523578, 0.45968138041850715], [0.62062344020178251, -0.41787907428574389, -0.35888807800463973, 
0.16694796164766323, -0.3977989533021975], [0.16518429534248313, -0.36314720068962214, -0.32058997573789605, 
-0.89515177291997983, 0.86881834023441074]]))
      arg1=0.813915837636
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.93480005656464904, 0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 
0.81391583763603204], [0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 
0.81391583763603204], [0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 
0.81391583763603204], [0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 
0.81391583763603204]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 
0.81391583763603204], [0.97796440308165078, 0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 
0.81391583763603204], [0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 
0.81391583763603204], [0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 0.81391583763603204, 
0.86881834023441074]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[-0.17424627648952273, -0.91369326722302535], [0.86981571454702555, -0.30056599212949142]], 
[[-0.83744079680925254, -0.57489835866873551], [-0.78652432997037325, -0.87217131091755329]], [[0.78772663404459142, 
0.76745643012266873], [-0.15790300634831911, -0.72995023028235062]], [[0.53178881570910019, 0.45410682131640656], 
[-0.29067462771405683, 0.23144581811663034]], [[0.30882533991055627, -0.93338200589874143], [0.83298933643490325, 
0.09998482062762859]], [[0.4933134750811039, -0.57402507256739521], [-0.070196917219758959, 
-0.41939337563063406]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.68404622654997804, -0.28910163710006809], [-0.042657623096307828, 
-0.26884108057848488]], [[0.40378076310142719, 0.9488325876337047], [0.64388831611326669, -0.93389908990622739]], 
[[0.055513256105979458, -0.30472263073409334], [0.30910180910249663, 0.25282720917999901]], [[0.57464800398805105, 
-0.3898457041435035], [0.21617061001627613, -0.096073139627144144]], [[-0.84774839950760605, 0.47590289745014824], 
[0.10840624480471028, 0.58323317319559931]], [[0.64177338938010386, -0.45949466159584973], [-0.35326311112160758, 
-0.75653917946776028]]]))
      arg1=0.692344403817
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.69234440381681428, 0.69234440381681428], [0.86981571454702555, 0.69234440381681428]], 
[[0.69234440381681428, 0.69234440381681428], [0.69234440381681428, 0.69234440381681428]], [[0.78772663404459142, 
0.76745643012266873], [0.69234440381681428, 0.69234440381681428]], [[0.69234440381681428, 0.69234440381681428], 
[0.69234440381681428, 0.69234440381681428]], [[0.69234440381681428, 0.69234440381681428], [0.83298933643490325, 
0.69234440381681428]], [[0.69234440381681428, 0.69234440381681428], [0.69234440381681428, 
0.69234440381681428]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.69234440381681428, 0.69234440381681428], [0.69234440381681428, 
0.69234440381681428]], [[0.69234440381681428, 0.9488325876337047], [0.69234440381681428, 0.69234440381681428]], 
[[0.69234440381681428, 0.69234440381681428], [0.69234440381681428, 0.69234440381681428]], [[0.69234440381681428, 
0.69234440381681428], [0.69234440381681428, 0.69234440381681428]], [[0.69234440381681428, 0.69234440381681428], 
[0.69234440381681428, 0.69234440381681428]], [[0.69234440381681428, 0.69234440381681428], [0.69234440381681428, 
0.69234440381681428]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[0.51344591455293354, -0.018441243493981574, 0.47110899708526577, -0.30918788301297573], 
[0.85620732591987236, -0.92034786714028871, -0.1787563715249243, 0.37325499319884581], [0.64137645798178844, 
-0.11026990150999616, 0.40744602113487649, -0.71946344206892543]], [[-0.45308066414525183, 0.34467574075542839, 
-0.20685979682482358, -0.84023591171266832], [-0.2413762095156653, -0.91891040235221277, -0.80845566209432351, 
-0.31587878081559695], [-0.36984878552281675, 0.26542087635947453, -0.51283180013117491, 0.021928982379654238]]], 
[[[-0.28169913804355806, -0.6928525662673799, 0.28473453732940901, 0.21602101787601313], [-0.94153361534981572, 
0.002357903028225028, 0.52489496154840332, -0.45071579437499931], [0.9284361704710038, -0.54922204319660017, 
0.64352796527590139, -0.3525208246382594]], [[0.47301405457397849, 0.64667241992778934, -0.46304504193527074, 
0.65269583573367029], [-0.29297019443524719, -0.11534881519162021, -0.72645113340053835, -0.29977943515792416], 
[0.84968671403456253, -0.84568246280364834, 0.68490422443498633, 0.13896446057648792]]], [[[-0.56879280814763655, 
0.53856592287852112, 0.418821466814012, -0.43811331694679412], [-0.22492197875356079, 0.75887167374914366, 0.35113680031718042, 
-0.94922861243802137], [0.33212813085962134, -0.20964539197588761, -0.54739474055381265, -0.6291521297883047]], 
[[0.78509272302164446, -0.76857760144230114, -0.43901690547351158, 0.28516276164445875], [0.9618269447301222, 
0.74527960751937194, -0.30261256871456244, 0.3679638015873179], [-0.6484191631051166, 0.1515932441327219, 0.031335332542854921, 
-0.90309000988094934]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.013607793411813507, -0.36282289887139152, 0.23518482902590265, 
-0.062041477811495938], [0.53034041017110134, -0.22579490630284704, -0.56096600949707542, -0.084929405097973643], 
[-0.37689004997752718, 0.22013122980568434, -0.9892706290233193, 0.37750003517931563]], [[-0.0099952159502807447, 
-0.11748521316287031, 0.12538769343171863, 0.84147604475294391], [-0.95742578443259996, -0.3597036800353024, 
-0.74424854966202592, -0.63698557655901777], [-0.28969074114670423, 0.068563439647513214, 0.84802425316138219, 
0.51835240644519254]]], [[[-0.98943454862130964, -0.7173162606299579, 0.45785971364767342, 0.093125386056379655], 
[-0.15795404609652985, -0.4504805523576616, 0.7479447304899578, -0.84401467910570482], [-0.75996652977744694, 
-0.96785677143680338, -0.9702777509071383, -0.055804832316411535]], [[-0.97758869960742256, 0.75139305795081235, 
-0.13817173730790855, -0.79901047615130305], [-0.68100845604105698, 0.95695171738302931, 0.96497242206391109, 
-0.88712737188715507], [0.83168687571710365, 0.87919645382961997, 0.85154438270907651, -0.83960628186079078]]], 
[[[-0.9878193348256965, 0.62345573910354957, 0.49292493494861311, -0.45680410336998456], [-0.92855605330786117, 
0.75968358062165331, -0.43343521882198455, -0.58998586398908448], [0.1296403096372043, -0.32944102432200295, 
-0.15849230325234309, -0.070143155505777655]], [[0.56400549150394719, -0.55809590289568756, -0.3446672399996622, 
-0.457237638930996], [0.35059915076627979, -0.38205004947025278, 0.19401029435141304, 0.061196810205449426], 
[0.90655743719059334, -0.92343663975887713, 0.45939741323556738, 0.84478155007424616]]]]))
      arg1=-0.65879782083
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.51344591455293354, -0.018441243493981574, 0.47110899708526577, -0.30918788301297573], 
[0.85620732591987236, -0.65879782083030158, -0.1787563715249243, 0.37325499319884581], [0.64137645798178844, 
-0.11026990150999616, 0.40744602113487649, -0.65879782083030158]], [[-0.45308066414525183, 0.34467574075542839, 
-0.20685979682482358, -0.65879782083030158], [-0.2413762095156653, -0.65879782083030158, -0.65879782083030158, 
-0.31587878081559695], [-0.36984878552281675, 0.26542087635947453, -0.51283180013117491, 0.021928982379654238]]], 
[[[-0.28169913804355806, -0.65879782083030158, 0.28473453732940901, 0.21602101787601313], [-0.65879782083030158, 
0.002357903028225028, 0.52489496154840332, -0.45071579437499931], [0.9284361704710038, -0.54922204319660017, 
0.64352796527590139, -0.3525208246382594]], [[0.47301405457397849, 0.64667241992778934, -0.46304504193527074, 
0.65269583573367029], [-0.29297019443524719, -0.11534881519162021, -0.65879782083030158, -0.29977943515792416], 
[0.84968671403456253, -0.65879782083030158, 0.68490422443498633, 0.13896446057648792]]], [[[-0.56879280814763655, 
0.53856592287852112, 0.418821466814012, -0.43811331694679412], [-0.22492197875356079, 0.75887167374914366, 0.35113680031718042, 
-0.65879782083030158], [0.33212813085962134, -0.20964539197588761, -0.54739474055381265, -0.6291521297883047]], 
[[0.78509272302164446, -0.65879782083030158, -0.43901690547351158, 0.28516276164445875], [0.9618269447301222, 
0.74527960751937194, -0.30261256871456244, 0.3679638015873179], [-0.6484191631051166, 0.1515932441327219, 0.031335332542854921, 
-0.65879782083030158]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.013607793411813507, -0.36282289887139152, 0.23518482902590265, 
-0.062041477811495938], [0.53034041017110134, -0.22579490630284704, -0.56096600949707542, -0.084929405097973643], 
[-0.37689004997752718, 0.22013122980568434, -0.65879782083030158, 0.37750003517931563]], [[-0.0099952159502807447, 
-0.11748521316287031, 0.12538769343171863, 0.84147604475294391], [-0.65879782083030158, -0.3597036800353024, 
-0.65879782083030158, -0.63698557655901777], [-0.28969074114670423, 0.068563439647513214, 0.84802425316138219, 
0.51835240644519254]]], [[[-0.65879782083030158, -0.65879782083030158, 0.45785971364767342, 0.093125386056379655], 
[-0.15795404609652985, -0.4504805523576616, 0.7479447304899578, -0.65879782083030158], [-0.65879782083030158, 
-0.65879782083030158, -0.65879782083030158, -0.055804832316411535]], [[-0.65879782083030158, 0.75139305795081235, 
-0.13817173730790855, -0.65879782083030158], [-0.65879782083030158, 0.95695171738302931, 0.96497242206391109, 
-0.65879782083030158], [0.83168687571710365, 0.87919645382961997, 0.85154438270907651, -0.65879782083030158]]], 
[[[-0.65879782083030158, 0.62345573910354957, 0.49292493494861311, -0.45680410336998456], [-0.65879782083030158, 
0.75968358062165331, -0.43343521882198455, -0.58998586398908448], [0.1296403096372043, -0.32944102432200295, 
-0.15849230325234309, -0.070143155505777655]], [[0.56400549150394719, -0.55809590289568756, -0.3446672399996622, 
-0.457237638930996], [0.35059915076627979, -0.38205004947025278, 0.19401029435141304, 0.061196810205449426], 
[0.90655743719059334, -0.65879782083030158, 0.45939741323556738, 0.84478155007424616]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_array_rank0(self):
      arg0=Data(-0.925174714018,self.functionspace)
      arg0.setTaggedValue(1,-0.638792248656)
      arg1=numpy.array(-0.234557032758)
      res=maximum(arg0,arg1)
      ref=Data(-0.234557032758,self.functionspace)
      ref.setTaggedValue(1,-0.234557032758)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_array_rank0(self):
      arg0=Data(numpy.array([-0.69849163386416246, -0.12612598296766264]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.29944056207788727, 0.60762433355628587]))
      arg1=numpy.array(0.798087045957)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.79808704595692603, 0.79808704595692603]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.79808704595692603, 0.79808704595692603]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[-0.52662965183815769, 0.57345406187402537, 0.90414404746228882, 0.42262577005483792, 
0.31989683726049378], [0.0059418039362177932, 0.19227754679174658, 0.58599614019070523, -0.24443187900901697, 
0.54375558535038571], [0.70261483573563899, -0.56558643939253539, -0.51690173380451476, -0.59465240855557822, 
-0.73198211064054508], [-0.21546932091661608, 0.59279865135361454, 0.85977065169379085, 0.7711017105670348, 
-0.31646152944503925]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.60532457963913266, -0.7034719638067779, -0.28360614769376324, 0.80306839482586345, 
0.42156217240843952], [0.48704494279914323, -0.86406044419690198, -0.54485210339812462, -0.17272514998453703, 
-0.37657680303876084], [0.60230010764391606, -0.63121348322279713, -0.74999412961581124, 0.67569953926345527, 
0.27593314823678194], [0.58695545383872427, 0.7253535427899156, 0.44388220925172672, 0.36340364324674068, 
0.0096800520668269829]]))
      arg1=numpy.array(0.419361707219)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.41936170721941157, 0.57345406187402537, 0.90414404746228882, 0.42262577005483792, 
0.41936170721941157], [0.41936170721941157, 0.41936170721941157, 0.58599614019070523, 0.41936170721941157, 
0.54375558535038571], [0.70261483573563899, 0.41936170721941157, 0.41936170721941157, 0.41936170721941157, 
0.41936170721941157], [0.41936170721941157, 0.59279865135361454, 0.85977065169379085, 0.7711017105670348, 
0.41936170721941157]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.41936170721941157, 0.41936170721941157, 0.41936170721941157, 0.80306839482586345, 
0.42156217240843952], [0.48704494279914323, 0.41936170721941157, 0.41936170721941157, 0.41936170721941157, 
0.41936170721941157], [0.60230010764391606, 0.41936170721941157, 0.41936170721941157, 0.67569953926345527, 
0.41936170721941157], [0.58695545383872427, 0.7253535427899156, 0.44388220925172672, 0.41936170721941157, 
0.41936170721941157]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[0.11996937338660096, -0.2381971872247417], [-0.63441963782454036, -0.66779750742805222]], 
[[0.92995457126434156, 0.15192851071559521], [0.084605389733761305, 0.35634515769340558]], [[0.84443922176028074, 
-0.88541263149824601], [-0.27094771060174816, 0.70139934037785689]], [[0.68994436428437256, 0.83186483072251027], 
[-0.14858063057687732, -0.18307204728965343]], [[0.78969080055932905, -0.93127882201058809], [-0.59100323574875491, 
0.8780312990028496]], [[-0.51860050073948694, 0.19882970552560275], [-0.81585601658525864, 
-0.5773314991681564]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.4077119693670197, -0.15089092327581843], [0.32678651591661101, 
0.45399585483056915]], [[-0.27171005023639183, 0.91576378268016767], [0.31142147182595403, -0.47936869062670207]], 
[[-0.23275353497070261, -0.069095101997814545], [0.060908158283257352, 0.23933590824512607]], [[0.60103860701300182, 
-0.44741778363728923], [0.88541668276824037, 0.95921374279165583]], [[-0.81126871572017278, -0.33059594874974341], 
[0.29995171427272238, 0.11876151588425388]], [[0.96038505930864182, -0.61891418238199214], [-0.95906798793793335, 
0.51506426474349243]]]))
      arg1=numpy.array(-0.557855377146)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.11996937338660096, -0.2381971872247417], [-0.5578553771462389, -0.5578553771462389]], 
[[0.92995457126434156, 0.15192851071559521], [0.084605389733761305, 0.35634515769340558]], [[0.84443922176028074, 
-0.5578553771462389], [-0.27094771060174816, 0.70139934037785689]], [[0.68994436428437256, 0.83186483072251027], 
[-0.14858063057687732, -0.18307204728965343]], [[0.78969080055932905, -0.5578553771462389], [-0.5578553771462389, 
0.8780312990028496]], [[-0.51860050073948694, 0.19882970552560275], [-0.5578553771462389, 
-0.5578553771462389]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.4077119693670197, -0.15089092327581843], [0.32678651591661101, 
0.45399585483056915]], [[-0.27171005023639183, 0.91576378268016767], [0.31142147182595403, -0.47936869062670207]], 
[[-0.23275353497070261, -0.069095101997814545], [0.060908158283257352, 0.23933590824512607]], [[0.60103860701300182, 
-0.44741778363728923], [0.88541668276824037, 0.95921374279165583]], [[-0.5578553771462389, -0.33059594874974341], 
[0.29995171427272238, 0.11876151588425388]], [[0.96038505930864182, -0.5578553771462389], [-0.5578553771462389, 
0.51506426474349243]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[0.89275404109845424, -0.35146473741579665, -0.17568117509227243, -0.7578795999650203], 
[-0.33534647809383666, 0.59800996974635745, 0.31503435476011932, -0.46176882323818425], [-0.13146855290018755, 
0.53080183078481635, 0.097739104079239469, 0.489397617772376]], [[0.66328825052473595, -0.016493169414107278, 
-0.19296570640621535, 0.53259751456448678], [0.14885726258574916, 0.20963457673525387, 0.33964874008570622, 
-0.60286867327138904], [-0.13878881815609945, 0.035625457403632765, -0.50046005552156148, -0.27555633863831641]]], 
[[[-0.89651648497925707, 0.29016557054006253, -0.55815574108322674, 0.21708312729185297], [0.14841730972031653, 
-0.37015999891426477, -0.43021572931663354, 0.86739175996362983], [-0.74377091514797833, 0.95901439254019638, 
-0.51386281665723521, -0.64879136103815416]], [[-0.15491801368832081, 0.16841099093477885, 0.65528197412357048, 
-0.064834886303676065], [-0.83402936883282486, 0.38346467957551589, -0.72018607765832909, -0.60954704574371155], 
[-0.34601567013589429, -0.35717627025481558, 0.42362282917531946, 0.23394075297738337]]], [[[-0.05834418636912031, 
-0.017425540900098957, -0.82371395647776446, 0.66954716651459911], [0.12689604172544566, 0.18110160335397429, 
0.85959643269502739, 0.89393140565626927], [-0.74705661684005475, -0.36546182942180527, -0.71591627299616456, 
0.50218215364204299]], [[0.60111164141963269, -0.92941174077167998, -0.98760106313591756, 0.53313765111509226], 
[-0.46946584016440918, 0.74282731709564231, 0.12617367148196079, 0.72282401835048349], [0.87975310489161518, 
-0.46260381145371832, -0.73799641157718865, 0.40842362602733084]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.65832055711473147, 0.81977989475922763, 0.079060551117023703, 
-0.26083516362499615], [0.9011359200617044, -0.0020651577687489375, -0.98780037380796992, 0.318512556404142], 
[-0.098970777121136111, 0.93218583512683151, -0.75252613988451889, 0.50835112543510741]], [[0.27761772736531509, 
0.22919045310778796, 0.18232481657294253, 0.1425134075457013], [0.47206045286859166, -0.73923765056264301, 0.27436450445356342, 
-0.56751723211924832], [0.62408752154214509, 0.71696546109438009, 0.96933745291446982, -0.023259873422613264]]], 
[[[-0.73384524032662513, -0.81564987888035945, -0.26761806526241916, 0.48013584699356304], [-0.084512492829282859, 
0.2250774673643523, 0.96305055306276599, -0.68561304642276055], [0.66893632640183154, 0.50261954844894197, 
-0.81443979088134077, 0.10366661510514263]], [[-0.28870400357374915, 0.58838031677115343, -0.45793492733324936, 
0.36403731953361351], [0.56168483329450147, -0.18113163365732321, -0.89656668226069214, -0.1938204315453429], 
[0.77411369722988854, 0.11257709116035186, 0.70170686853277231, -0.6693926170189155]]], [[[-0.37636393999209816, 
0.63801870058103005, 0.3653488860837073, -0.66928500592201678], [-0.59259165330789743, 0.78617092236936137, 
-0.20604168110115295, 0.955633421887778], [-0.81696514728783787, 0.61154474690435578, -0.14237359445393083, 
-0.74314451147533411]], [[-0.94117651794335067, 0.35173957026195968, 0.17124685404860052, 0.42927640273794432], 
[0.5822566172546888, -0.70242468758245735, 0.5672314332554671, 0.17717092789636069], [0.85532249600047172, 
-0.41529472301095383, 0.47186050130451185, -0.74030950630891934]]]]))
      arg1=numpy.array(-0.0617969524811)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.89275404109845424, -0.06179695248106376, -0.06179695248106376, -0.06179695248106376], 
[-0.06179695248106376, 0.59800996974635745, 0.31503435476011932, -0.06179695248106376], [-0.06179695248106376, 
0.53080183078481635, 0.097739104079239469, 0.489397617772376]], [[0.66328825052473595, -0.016493169414107278, 
-0.06179695248106376, 0.53259751456448678], [0.14885726258574916, 0.20963457673525387, 0.33964874008570622, 
-0.06179695248106376], [-0.06179695248106376, 0.035625457403632765, -0.06179695248106376, -0.06179695248106376]]], 
[[[-0.06179695248106376, 0.29016557054006253, -0.06179695248106376, 0.21708312729185297], [0.14841730972031653, 
-0.06179695248106376, -0.06179695248106376, 0.86739175996362983], [-0.06179695248106376, 0.95901439254019638, 
-0.06179695248106376, -0.06179695248106376]], [[-0.06179695248106376, 0.16841099093477885, 0.65528197412357048, 
-0.06179695248106376], [-0.06179695248106376, 0.38346467957551589, -0.06179695248106376, -0.06179695248106376], 
[-0.06179695248106376, -0.06179695248106376, 0.42362282917531946, 0.23394075297738337]]], [[[-0.05834418636912031, 
-0.017425540900098957, -0.06179695248106376, 0.66954716651459911], [0.12689604172544566, 0.18110160335397429, 
0.85959643269502739, 0.89393140565626927], [-0.06179695248106376, -0.06179695248106376, -0.06179695248106376, 
0.50218215364204299]], [[0.60111164141963269, -0.06179695248106376, -0.06179695248106376, 0.53313765111509226], 
[-0.06179695248106376, 0.74282731709564231, 0.12617367148196079, 0.72282401835048349], [0.87975310489161518, 
-0.06179695248106376, -0.06179695248106376, 0.40842362602733084]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.06179695248106376, 0.81977989475922763, 0.079060551117023703, 
-0.06179695248106376], [0.9011359200617044, -0.0020651577687489375, -0.06179695248106376, 0.318512556404142], 
[-0.06179695248106376, 0.93218583512683151, -0.06179695248106376, 0.50835112543510741]], [[0.27761772736531509, 
0.22919045310778796, 0.18232481657294253, 0.1425134075457013], [0.47206045286859166, -0.06179695248106376, 0.27436450445356342, 
-0.06179695248106376], [0.62408752154214509, 0.71696546109438009, 0.96933745291446982, -0.023259873422613264]]], 
[[[-0.06179695248106376, -0.06179695248106376, -0.06179695248106376, 0.48013584699356304], [-0.06179695248106376, 
0.2250774673643523, 0.96305055306276599, -0.06179695248106376], [0.66893632640183154, 0.50261954844894197, 
-0.06179695248106376, 0.10366661510514263]], [[-0.06179695248106376, 0.58838031677115343, -0.06179695248106376, 
0.36403731953361351], [0.56168483329450147, -0.06179695248106376, -0.06179695248106376, -0.06179695248106376], 
[0.77411369722988854, 0.11257709116035186, 0.70170686853277231, -0.06179695248106376]]], [[[-0.06179695248106376, 
0.63801870058103005, 0.3653488860837073, -0.06179695248106376], [-0.06179695248106376, 0.78617092236936137, 
-0.06179695248106376, 0.955633421887778], [-0.06179695248106376, 0.61154474690435578, -0.06179695248106376, 
-0.06179695248106376]], [[-0.06179695248106376, 0.35173957026195968, 0.17124685404860052, 0.42927640273794432], 
[0.5822566172546888, -0.06179695248106376, 0.5672314332554671, 0.17717092789636069], [0.85532249600047172, 
-0.06179695248106376, 0.47186050130451185, -0.06179695248106376]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_constData_rank0(self):
      arg0=Data(-0.849363709294,self.functionspace)
      arg0.setTaggedValue(1,-0.382967103025)
      arg1=Data(0.974014450724,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(0.974014450724,self.functionspace)
      ref.setTaggedValue(1,0.974014450724)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_constData_rank0(self):
      arg0=Data(numpy.array([0.15359534220057491, 0.81025586040974384]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.45081025408796638, 0.82764996189716289]))
      arg1=Data(-0.995343707269,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.15359534220057491, 0.81025586040974384]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.45081025408796638, 0.82764996189716289]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_constData_rank0(self):
      arg0=Data(numpy.array([[0.13708122850182125, 0.99008715870007546, -0.70545154117672348, 0.99916050781045551, 
0.25514058627455105], [-0.12561395088999538, -0.86438506107979385, 0.13034668941012062, 0.35224690602370012, 
-0.017162563999681746], [-0.91569579082587493, -0.67743146551876521, -0.95589806519073139, 0.1743108864158176, 
-0.33355994781829934], [-0.55523563941647414, 0.70472497252224087, 0.11627919923214836, -0.37172189001972078, 
0.18550702772614969]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.073798446664826489, 0.3006244069574715, -0.89279180141507752, -0.87506777897706445, 
-0.62920651002059946], [0.038644010795130024, 0.40897497262022364, 0.15238521249533399, 0.55244990810882744, 
0.99449051852373382], [-0.62128735737764407, -0.35315318920526884, 0.3995909771915227, 0.014927585301315327, 
0.85133795538885293], [0.018435201197165352, -0.030162727474547957, -0.13451624890192559, -0.42008480913114044, 
0.84075589075060631]]))
      arg1=Data(-0.921146847276,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.13708122850182125, 0.99008715870007546, -0.70545154117672348, 0.99916050781045551, 
0.25514058627455105], [-0.12561395088999538, -0.86438506107979385, 0.13034668941012062, 0.35224690602370012, 
-0.017162563999681746], [-0.91569579082587493, -0.67743146551876521, -0.92114684727582952, 0.1743108864158176, 
-0.33355994781829934], [-0.55523563941647414, 0.70472497252224087, 0.11627919923214836, -0.37172189001972078, 
0.18550702772614969]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.073798446664826489, 0.3006244069574715, -0.89279180141507752, -0.87506777897706445, 
-0.62920651002059946], [0.038644010795130024, 0.40897497262022364, 0.15238521249533399, 0.55244990810882744, 
0.99449051852373382], [-0.62128735737764407, -0.35315318920526884, 0.3995909771915227, 0.014927585301315327, 
0.85133795538885293], [0.018435201197165352, -0.030162727474547957, -0.13451624890192559, -0.42008480913114044, 
0.84075589075060631]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_constData_rank0(self):
      arg0=Data(numpy.array([[[-0.41383177356797463, -0.0079349927271721299], [-0.90947471917084233, -0.79468119935536907]], 
[[-0.19064212513470236, -0.82820544671402252], [0.31168520087678786, 0.57404206055363316]], [[0.98197346242601569, 
0.80253966596195125], [-0.33540931513018069, 0.44691416089131786]], [[0.44594410245513605, 0.85517197254951238], 
[-0.066556018640275161, 0.62804259202268797]], [[0.543593854735404, -0.16270199888648107], [0.38740947895305733, 
0.95732655994382188]], [[0.74719294730162322, 0.18724540092319875], [0.97336861552944964, 
0.88152257094513686]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.80870088556382247, -0.57012139600601452], [0.31992431989323777, 
0.72728320881432018]], [[-0.98129612402840216, -0.30068237958185762], [-0.34254372207608919, 0.81760308616373334]], 
[[-0.58300511224222129, -0.62990156406393161], [0.66577278756599267, -0.89483189164245869]], [[-0.23555132532767087, 
0.21281660069146269], [-0.77768605222075915, -0.57698070336982643]], [[0.054578977408722418, 0.10953806304154989], 
[-0.37891307232139115, -0.14558540370308215]], [[0.35566828339792633, 0.63078660214558968], [-0.34058348881531586, 
-0.69302067106712517]]]))
      arg1=Data(0.117893688047,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.11789368804668054, 0.11789368804668054], [0.11789368804668054, 0.11789368804668054]], 
[[0.11789368804668054, 0.11789368804668054], [0.31168520087678786, 0.57404206055363316]], [[0.98197346242601569, 
0.80253966596195125], [0.11789368804668054, 0.44691416089131786]], [[0.44594410245513605, 0.85517197254951238], 
[0.11789368804668054, 0.62804259202268797]], [[0.543593854735404, 0.11789368804668054], [0.38740947895305733, 
0.95732655994382188]], [[0.74719294730162322, 0.18724540092319875], [0.97336861552944964, 
0.88152257094513686]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.11789368804668054, 0.11789368804668054], [0.31992431989323777, 
0.72728320881432018]], [[0.11789368804668054, 0.11789368804668054], [0.11789368804668054, 0.81760308616373334]], 
[[0.11789368804668054, 0.11789368804668054], [0.66577278756599267, 0.11789368804668054]], [[0.11789368804668054, 
0.21281660069146269], [0.11789368804668054, 0.11789368804668054]], [[0.11789368804668054, 0.11789368804668054], 
[0.11789368804668054, 0.11789368804668054]], [[0.35566828339792633, 0.63078660214558968], [0.11789368804668054, 
0.11789368804668054]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_constData_rank0(self):
      arg0=Data(numpy.array([[[[0.058530631584147841, 0.14750170330075418, 0.083554821050923156, 0.12739051526091272], 
[-0.96524804058847291, 0.208606108995401, -0.93204062175176716, -0.064016818762620442], [-0.16874685746413665, 
0.12806083598258522, 0.41076238998523329, -0.48148824169789872]], [[-0.15831879740579691, 0.28831400768868387, 
0.3982174652620778, -0.16248028079896515], [-0.27532326311687805, -0.021862137691255779, 0.09350394863730882, 
0.78286608006869374], [-0.63089464456191502, 0.28554127685291042, -0.62526764012057812, 0.72594337680791199]]], 
[[[-0.6424127600311349, -0.21762021092041217, 0.2611370537310389, -0.52434944778859016], [0.35997637929992288, 
0.76173275479140012, -0.6563115461338449, -0.90626014629268803], [0.50176242567542029, 0.16915711622775853, 
0.87403536456826458, 0.61308431638548]], [[0.033212229761786283, 0.63077470361800581, -0.05709057602998735, 
-0.27186221772582786], [0.84770710741426436, 0.028921811923150686, -0.25629745341448174, 0.70688266388332743], 
[-0.97553041516851535, -0.79794387583154935, 0.62806989822101023, 0.40672359067803687]]], [[[-0.74111207066215212, 
-0.31687926245252651, -0.78612721063127888, 0.58083076665099287], [-0.30743221260109577, 0.57338542003085302, 
-0.14911240473892806, 0.98569614719602749], [0.24596223558102115, -0.5443517484938829, -0.79259488248175103, 
-0.69342114529009313]], [[-0.42259849098299629, -0.034397453248862897, -0.099822524038713389, -0.8468836448326178], 
[0.97527226747919893, 0.61972856955552458, -0.74471183260979834, -0.40408785286516768], [-0.87658138339376124, 
0.84390281173153592, 0.13925306092932255, -0.33452693973067404]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.12144409800449729, -0.33417039576434138, -0.61085065582742182, 
0.8677662783183886], [0.8932843691663277, 0.084340565721825911, 0.40982244051084171, 0.034826116046352107], 
[-0.52327027500949352, 0.28231558158830583, 0.96316694749696996, 0.65530290034532812]], [[-0.48599557420440109, 
0.94733329294485413, -0.54789959265104793, 0.9728641970472145], [0.19946614339353386, -0.46233735344589122, 
0.38061862918115064, 0.60582971644701722], [0.39088733929715191, -0.27125261669514367, 0.64896347670959309, 
0.20518579171289808]]], [[[-0.42708438888246159, 0.58991510815264414, -0.8979168816067753, 0.39660460366783656], 
[0.89047849986284233, 0.79571279345534807, 0.48161316017611755, -0.16618845208499899], [-0.37900902267404524, 
0.37429071509740552, 0.15566009975997486, 0.36021174077288243]], [[0.60577858865639511, -0.907251230609458, 
0.72699394638914705, 0.4580268717701188], [-0.60268237753401377, -0.012975109597635148, -0.24614971362638083, 
0.53144328109785066], [0.34904906643582678, -0.29029548244600134, 0.82432163597412145, 0.64993980699502396]]], 
[[[0.38651003856272448, 0.91544426713746319, -0.28499396863792059, 0.84403596898915811], [0.044799630121824308, 
-0.5595319739528597, 0.98476752736083739, -0.055205325073725664], [0.56308155093763035, -0.22612614540599085, 
0.84487723147154825, -0.32251627516945569]], [[-0.83935271937690636, 0.11596577500942273, 0.026618480268968625, 
-0.65760920822899149], [0.28333140444788762, 0.78566755623194018, -0.79801040291438397, 0.49856405176807894], 
[-0.039629093631027734, 0.29220360739139317, 0.51911055345014745, 0.17497093994658242]]]]))
      arg1=Data(0.0537820413535,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.058530631584147841, 0.14750170330075418, 0.083554821050923156, 0.12739051526091272], 
[0.053782041353522159, 0.208606108995401, 0.053782041353522159, 0.053782041353522159], [0.053782041353522159, 
0.12806083598258522, 0.41076238998523329, 0.053782041353522159]], [[0.053782041353522159, 0.28831400768868387, 
0.3982174652620778, 0.053782041353522159], [0.053782041353522159, 0.053782041353522159, 0.09350394863730882, 
0.78286608006869374], [0.053782041353522159, 0.28554127685291042, 0.053782041353522159, 0.72594337680791199]]], 
[[[0.053782041353522159, 0.053782041353522159, 0.2611370537310389, 0.053782041353522159], [0.35997637929992288, 
0.76173275479140012, 0.053782041353522159, 0.053782041353522159], [0.50176242567542029, 0.16915711622775853, 
0.87403536456826458, 0.61308431638548]], [[0.053782041353522159, 0.63077470361800581, 0.053782041353522159, 
0.053782041353522159], [0.84770710741426436, 0.053782041353522159, 0.053782041353522159, 0.70688266388332743], 
[0.053782041353522159, 0.053782041353522159, 0.62806989822101023, 0.40672359067803687]]], [[[0.053782041353522159, 
0.053782041353522159, 0.053782041353522159, 0.58083076665099287], [0.053782041353522159, 0.57338542003085302, 
0.053782041353522159, 0.98569614719602749], [0.24596223558102115, 0.053782041353522159, 0.053782041353522159, 
0.053782041353522159]], [[0.053782041353522159, 0.053782041353522159, 0.053782041353522159, 0.053782041353522159], 
[0.97527226747919893, 0.61972856955552458, 0.053782041353522159, 0.053782041353522159], [0.053782041353522159, 
0.84390281173153592, 0.13925306092932255, 0.053782041353522159]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.053782041353522159, 0.053782041353522159, 0.053782041353522159, 
0.8677662783183886], [0.8932843691663277, 0.084340565721825911, 0.40982244051084171, 0.053782041353522159], 
[0.053782041353522159, 0.28231558158830583, 0.96316694749696996, 0.65530290034532812]], [[0.053782041353522159, 
0.94733329294485413, 0.053782041353522159, 0.9728641970472145], [0.19946614339353386, 0.053782041353522159, 
0.38061862918115064, 0.60582971644701722], [0.39088733929715191, 0.053782041353522159, 0.64896347670959309, 
0.20518579171289808]]], [[[0.053782041353522159, 0.58991510815264414, 0.053782041353522159, 0.39660460366783656], 
[0.89047849986284233, 0.79571279345534807, 0.48161316017611755, 0.053782041353522159], [0.053782041353522159, 
0.37429071509740552, 0.15566009975997486, 0.36021174077288243]], [[0.60577858865639511, 0.053782041353522159, 
0.72699394638914705, 0.4580268717701188], [0.053782041353522159, 0.053782041353522159, 0.053782041353522159, 
0.53144328109785066], [0.34904906643582678, 0.053782041353522159, 0.82432163597412145, 0.64993980699502396]]], 
[[[0.38651003856272448, 0.91544426713746319, 0.053782041353522159, 0.84403596898915811], [0.053782041353522159, 
0.053782041353522159, 0.98476752736083739, 0.053782041353522159], [0.56308155093763035, 0.053782041353522159, 
0.84487723147154825, 0.053782041353522159]], [[0.053782041353522159, 0.11596577500942273, 0.053782041353522159, 
0.053782041353522159], [0.28333140444788762, 0.78566755623194018, 0.053782041353522159, 0.49856405176807894], 
[0.053782041353522159, 0.29220360739139317, 0.51911055345014745, 0.17497093994658242]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_taggedData_rank0(self):
      arg0=Data(-0.0773050286218,self.functionspace)
      arg0.setTaggedValue(1,0.795622660396)
      arg1=Data(0.687408080924,self.functionspace)
      arg1.setTaggedValue(1,-0.893012441951)
      res=maximum(arg0,arg1)
      ref=Data(0.687408080924,self.functionspace)
      ref.setTaggedValue(1,0.795622660396)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_taggedData_rank0(self):
      arg0=Data(numpy.array([-0.97940722383606604, 0.20831974957133181]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.1414990755156631, 0.25244721263550951]))
      arg1=Data(-0.932634599616,self.functionspace)
      arg1.setTaggedValue(1,-0.0406184999095)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([-0.93263459961646955, 0.20831974957133181]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.040618499909469197, 0.25244721263550951]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_taggedData_rank0(self):
      arg0=Data(numpy.array([[0.81865421202670863, -0.48309271412923671, -0.2721938821807186, 0.46431024750046324, 
0.80266295409265509], [-0.92189300694573761, -0.047361568571882007, 0.67496605976373658, 0.63397415606415652, 
0.12548268283303088], [-0.2720386995977282, -0.99921605658341295, -0.97570019181320355, -0.87761388633010884, 
0.43244382500846212], [0.31147069716597664, 0.68650179884169527, 0.21568489539421165, 0.71188054668904255, 
-0.074654471444180803]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.64092321491154935, 0.75352868437484521, 0.70581906492627078, -0.59626477374775932, 
-0.49808340791716743], [-0.16077276729815138, 0.98644961425401534, 0.65421989255235213, 0.011816660133845858, 
-0.61492545805462773], [0.64343690927522323, -0.78078133450788467, -0.90704878580122839, -0.03273812861031411, 
0.88026881938891033], [-0.75705605703252088, 0.65109494974282045, 0.33485057620634295, -0.43886483245682828, 
0.53205333363042961]]))
      arg1=Data(-0.73132633447,self.functionspace)
      arg1.setTaggedValue(1,0.5627616956)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.81865421202670863, -0.48309271412923671, -0.2721938821807186, 0.46431024750046324, 
0.80266295409265509], [-0.7313263344703429, -0.047361568571882007, 0.67496605976373658, 0.63397415606415652, 
0.12548268283303088], [-0.2720386995977282, -0.7313263344703429, -0.7313263344703429, -0.7313263344703429, 
0.43244382500846212], [0.31147069716597664, 0.68650179884169527, 0.21568489539421165, 0.71188054668904255, 
-0.074654471444180803]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.5627616955998842, 0.75352868437484521, 0.70581906492627078, 0.5627616955998842, 
0.5627616955998842], [0.5627616955998842, 0.98644961425401534, 0.65421989255235213, 0.5627616955998842, 0.5627616955998842], 
[0.64343690927522323, 0.5627616955998842, 0.5627616955998842, 0.5627616955998842, 0.88026881938891033], [0.5627616955998842, 
0.65109494974282045, 0.5627616955998842, 0.5627616955998842, 0.5627616955998842]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_taggedData_rank0(self):
      arg0=Data(numpy.array([[[-0.98539672133847422, -0.066559654209057628], [0.73258799215294079, -0.96930561391189896]], 
[[0.91281479696554624, 0.88681589442296715], [-0.29821008712592056, 0.86769445558487956]], [[0.54549310803389695, 
-0.94714668467079655], [-0.88406559478922886, -0.94384448275619626]], [[0.088804525243413313, 0.55716033028391743], 
[0.84248101886488769, -0.80351993484400186]], [[0.63959700723873314, 0.38407245797650402], [0.72260249029460488, 
-0.52723602697267569]], [[-0.63819007263327387, 0.79618949712425513], [-0.38164845782148848, 
-0.34440124307925601]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.068822580807178957, -0.056368405194844096], [0.097742974763102008, 
-0.70483015098791957]], [[-0.66067964199224471, 0.34217386326535326], [-0.28722013448256978, 0.26926934492791066]], 
[[-0.24311752434158374, -0.80417487986334546], [0.38207143293187751, 0.038754968094078857]], [[-0.39836327466705601, 
0.74975919273299629], [0.55814389365676575, -0.33827474179200778]], [[0.33785974041791622, 0.45821033877180106], 
[-0.35472232065460507, 0.88151115955192849]], [[0.28123899673688557, 0.53127396097933866], [-0.53894090929285499, 
-0.24499886092776557]]]))
      arg1=Data(-0.343424592439,self.functionspace)
      arg1.setTaggedValue(1,-0.181401039795)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[-0.34342459243887902, -0.066559654209057628], [0.73258799215294079, -0.34342459243887902]], 
[[0.91281479696554624, 0.88681589442296715], [-0.29821008712592056, 0.86769445558487956]], [[0.54549310803389695, 
-0.34342459243887902], [-0.34342459243887902, -0.34342459243887902]], [[0.088804525243413313, 0.55716033028391743], 
[0.84248101886488769, -0.34342459243887902]], [[0.63959700723873314, 0.38407245797650402], [0.72260249029460488, 
-0.34342459243887902]], [[-0.34342459243887902, 0.79618949712425513], [-0.34342459243887902, 
-0.34342459243887902]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.068822580807178957, -0.056368405194844096], [0.097742974763102008, 
-0.18140103979546862]], [[-0.18140103979546862, 0.34217386326535326], [-0.18140103979546862, 0.26926934492791066]], 
[[-0.18140103979546862, -0.18140103979546862], [0.38207143293187751, 0.038754968094078857]], [[-0.18140103979546862, 
0.74975919273299629], [0.55814389365676575, -0.18140103979546862]], [[0.33785974041791622, 0.45821033877180106], 
[-0.18140103979546862, 0.88151115955192849]], [[0.28123899673688557, 0.53127396097933866], [-0.18140103979546862, 
-0.18140103979546862]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_taggedData_rank0(self):
      arg0=Data(numpy.array([[[[0.82223168576566574, -0.39034148514526379, -0.94711419906947292, 0.86806035007101068], 
[-0.2548080146990308, -0.076426927175190285, -0.93645365020155502, -0.88057796359480744], [-0.62632340504906492, 
-0.91564753892682949, 0.20897202558403905, 0.42521530117093609]], [[0.85706286544748389, 0.18445700944364263, 
-0.41235539256262443, -0.86354278504183912], [-0.068520321389177186, 0.92680817821087369, -0.21753025675661553, 
0.11504290657947713], [0.67612716594324618, 0.72911408811086109, 0.31613685556875448, -0.28280101200466023]]], 
[[[-0.5287979150830171, -0.43314984241026488, -0.423060681824589, -0.28680119480062327], [0.46829230775079789, 
-0.73927400726354553, -0.57885260942280059, -0.86597688869894807], [0.85219176461254653, -0.83165235642055091, 
-0.82090980966195892, 0.96552320190135776]], [[-0.53754032866502199, 0.10361563310991806, 0.04536386739623155, 
-0.083126518077624612], [-0.64065353373786227, 0.070242300568236882, -0.05511910627669514, 0.010746148569006131], 
[-0.29282928335317449, -0.28973787162126263, 0.8937649861634509, -0.34212368796139314]]], [[[-0.084883236922512761, 
-0.24802788469888815, -0.83603257677797727, -0.16733188815661548], [-0.96751180804381454, -0.9324284016154305, 
-0.71291710547447074, 0.51460065373168051], [-0.52079118499827715, 0.93730765691831586, 0.51010620006895446, 
-0.39915618582193901]], [[0.19812793189184785, -0.46876475960914865, 0.35505566574804037, -0.077963408541119827], 
[0.42135760706020275, 0.18151614891811407, -0.27859677619875689, -0.5706629366910998], [-0.28630661981927763, 
-0.6382594098998402, 0.69441943434339248, 0.17952645963348113]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.54952366530408225, -0.71759607590503838, -0.050516796200858138, 
-0.1675098663390413], [-0.74431641125859538, 0.17133554793892669, 0.44986317513980922, -0.13041645263811108], 
[0.31865080791602152, 0.16192661121360175, -0.16960116270526027, -0.29754093334032317]], [[0.37651600855466572, 
-0.41242332065767218, -0.97670073815869229, 0.88382920458145176], [-0.18910784762409105, 0.32326820817394575, 
-0.32785764767200365, -0.21038790837238541], [-0.31899961237167096, -0.21533510249156174, 0.47173709175828171, 
0.67947703219883659]]], [[[0.53639964146440167, 0.92849226037664412, 0.8338730454241956, -0.85536478844296182], 
[-0.22830297017919721, 0.34686252987501409, -0.86055709472397024, -0.59270840703730898], [-0.42298476784023098, 
0.26829644328408486, -0.23256227464061485, 0.22715556948205951]], [[0.1240651761997138, -0.29784061059828915, 
0.1756006268404986, 0.95332632944863516], [0.85500918190924424, -0.41143222047842953, -0.43059802994177043, 
-0.66360443585995088], [-0.2927336423613196, 0.25230657561304648, -0.22409949967070686, -0.59779458927362494]]], 
[[[-0.77446415717404093, -0.42828041921756022, -0.95092214224900284, 0.51114636793983603], [-0.94679606661444993, 
-0.45504604302435681, 0.79244715667059351, 0.78853431082389247], [-0.020581155417678243, 0.84373370013654525, 
-0.62638304380153209, -0.20526441124213579]], [[0.84529422818669242, 0.052482399563541726, 0.47434350715509299, 
-0.87457049333209924], [0.82830876346993176, 0.44340216476794092, -0.70171255874245575, -0.37606319845637515], 
[-0.86393232160446631, -0.044187973750994924, -0.63919765525092975, 0.55678681387162343]]]]))
      arg1=Data(0.131061463012,self.functionspace)
      arg1.setTaggedValue(1,-0.154171401658)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.82223168576566574, 0.13106146301229837, 0.13106146301229837, 0.86806035007101068], 
[0.13106146301229837, 0.13106146301229837, 0.13106146301229837, 0.13106146301229837], [0.13106146301229837, 
0.13106146301229837, 0.20897202558403905, 0.42521530117093609]], [[0.85706286544748389, 0.18445700944364263, 
0.13106146301229837, 0.13106146301229837], [0.13106146301229837, 0.92680817821087369, 0.13106146301229837, 
0.13106146301229837], [0.67612716594324618, 0.72911408811086109, 0.31613685556875448, 0.13106146301229837]]], 
[[[0.13106146301229837, 0.13106146301229837, 0.13106146301229837, 0.13106146301229837], [0.46829230775079789, 
0.13106146301229837, 0.13106146301229837, 0.13106146301229837], [0.85219176461254653, 0.13106146301229837, 0.13106146301229837, 
0.96552320190135776]], [[0.13106146301229837, 0.13106146301229837, 0.13106146301229837, 0.13106146301229837], 
[0.13106146301229837, 0.13106146301229837, 0.13106146301229837, 0.13106146301229837], [0.13106146301229837, 
0.13106146301229837, 0.8937649861634509, 0.13106146301229837]]], [[[0.13106146301229837, 0.13106146301229837, 
0.13106146301229837, 0.13106146301229837], [0.13106146301229837, 0.13106146301229837, 0.13106146301229837, 
0.51460065373168051], [0.13106146301229837, 0.93730765691831586, 0.51010620006895446, 0.13106146301229837]], 
[[0.19812793189184785, 0.13106146301229837, 0.35505566574804037, 0.13106146301229837], [0.42135760706020275, 
0.18151614891811407, 0.13106146301229837, 0.13106146301229837], [0.13106146301229837, 0.13106146301229837, 0.69441943434339248, 
0.17952645963348113]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.15417140165809018, -0.15417140165809018, -0.050516796200858138, 
-0.15417140165809018], [-0.15417140165809018, 0.17133554793892669, 0.44986317513980922, -0.13041645263811108], 
[0.31865080791602152, 0.16192661121360175, -0.15417140165809018, -0.15417140165809018]], [[0.37651600855466572, 
-0.15417140165809018, -0.15417140165809018, 0.88382920458145176], [-0.15417140165809018, 0.32326820817394575, 
-0.15417140165809018, -0.15417140165809018], [-0.15417140165809018, -0.15417140165809018, 0.47173709175828171, 
0.67947703219883659]]], [[[0.53639964146440167, 0.92849226037664412, 0.8338730454241956, -0.15417140165809018], 
[-0.15417140165809018, 0.34686252987501409, -0.15417140165809018, -0.15417140165809018], [-0.15417140165809018, 
0.26829644328408486, -0.15417140165809018, 0.22715556948205951]], [[0.1240651761997138, -0.15417140165809018, 
0.1756006268404986, 0.95332632944863516], [0.85500918190924424, -0.15417140165809018, -0.15417140165809018, 
-0.15417140165809018], [-0.15417140165809018, 0.25230657561304648, -0.15417140165809018, -0.15417140165809018]]], 
[[[-0.15417140165809018, -0.15417140165809018, -0.15417140165809018, 0.51114636793983603], [-0.15417140165809018, 
-0.15417140165809018, 0.79244715667059351, 0.78853431082389247], [-0.020581155417678243, 0.84373370013654525, 
-0.15417140165809018, -0.15417140165809018]], [[0.84529422818669242, 0.052482399563541726, 0.47434350715509299, 
-0.15417140165809018], [0.82830876346993176, 0.44340216476794092, -0.15417140165809018, -0.15417140165809018], 
[-0.15417140165809018, -0.044187973750994924, -0.15417140165809018, 0.55678681387162343]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_expandedData_rank0(self):
      arg0=Data(0.404613896023,self.functionspace)
      arg0.setTaggedValue(1,-0.605787471295)
      arg1=Data(-0.953147829874,self.functionspace)
      arg1.setTaggedValue(1,0.871023339507)
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(0.404613896023)+(1.-msk_ref)*(0.871023339507)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_expandedData_rank0(self):
      arg0=Data(numpy.array([-0.53734286036823153, -0.40147255817066241]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.37406391752627832, -0.98975771429517279]))
      arg1=Data(-0.673204948535,self.functionspace)
      arg1.setTaggedValue(1,0.865550829826)
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-0.53734286036823153, -0.40147255817066241])+(1.-msk_ref)*numpy.array([0.86555082982552345, 
0.86555082982552345])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_expandedData_rank0(self):
      arg0=Data(numpy.array([[0.59718433144947225, 0.2185118451777206, 0.68106689437209922, 0.8330950201681262, 
0.92251529616059336], [0.35915436627683461, 0.82050354603191278, -0.48346987667009711, 0.95798739323304094, 
-0.45212823813081804], [-0.6143397531819943, -0.59662090135653489, -0.17527607094482889, -0.51980046717121464, 
-0.69654082146478391], [0.80804619319490834, -0.64871787794038505, -0.61174035027023321, 0.61921189194019122, 
0.57947500303286836]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.7121781219674157, 0.87711958760290343, 0.17124948184868671, 0.0092844154019224856, 
0.59174550729124076], [-0.79171061609947269, -0.7378641467324718, 0.55914098178797467, -0.64741556788173793, 
-0.53631396719845847], [0.26075613237228623, 0.53213127069458754, -0.8726256530309553, 0.27111087452724303, 
0.11739286222190493], [0.29166396957322704, 0.75671926964971514, -0.076399138446586301, -0.69802601769574379, 
0.95997927785343595]]))
      arg1=Data(0.281821267134,self.functionspace)
      arg1.setTaggedValue(1,-0.267044396323)
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.59718433144947225, 0.28182126713414779, 0.68106689437209922, 0.8330950201681262, 
0.92251529616059336], [0.35915436627683461, 0.82050354603191278, 0.28182126713414779, 0.95798739323304094, 
0.28182126713414779], [0.28182126713414779, 0.28182126713414779, 0.28182126713414779, 0.28182126713414779, 
0.28182126713414779], [0.80804619319490834, 0.28182126713414779, 0.28182126713414779, 0.61921189194019122, 
0.57947500303286836]])+(1.-msk_ref)*numpy.array([[-0.26704439632263077, 0.87711958760290343, 0.17124948184868671, 
0.0092844154019224856, 0.59174550729124076], [-0.26704439632263077, -0.26704439632263077, 0.55914098178797467, 
-0.26704439632263077, -0.26704439632263077], [0.26075613237228623, 0.53213127069458754, -0.26704439632263077, 
0.27111087452724303, 0.11739286222190493], [0.29166396957322704, 0.75671926964971514, -0.076399138446586301, 
-0.26704439632263077, 0.95997927785343595]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_expandedData_rank0(self):
      arg0=Data(numpy.array([[[-0.8533458041194999, 0.045970167640473347], [-0.30892003571780102, 0.68404306174037122]], 
[[-0.2233635724050711, -0.1972699426853608], [0.55480158886511521, -0.081679837610963313]], [[0.46092131939377401, 
0.61841845376278903], [-0.20135629761268814, 0.64568662909420782]], [[0.42220843169453914, 0.48788430349364686], 
[-0.21026274889184049, 0.92207885217687857]], [[0.19255327953788837, -0.98249593407651603], [-0.77974761806396331, 
0.80531737845613161]], [[-0.076443695327172545, 0.0038557847104050413], [0.24935171772121745, 
-0.25051807392206449]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.27025548477652839, 0.86031576930313269], [-0.67371780949017235, 
0.82251684778702772]], [[0.30087515206645254, 0.7888837003038216], [0.32479659296094598, -0.65301409296449964]], 
[[-0.15740090470976709, -0.99101728027566072], [0.57171440467946577, -0.71735225967556637]], [[0.5187975482154179, 
0.12427821264169658], [-0.066862130192447466, 0.072664756311500156]], [[0.47887099987031401, -0.45696493285976181], 
[0.19388688089778983, -0.77558040097252556]], [[0.12149530302926204, 0.88141576474629457], [-0.21574845515507768, 
0.11591491519891162]]]))
      arg1=Data(0.764990544123,self.functionspace)
      arg1.setTaggedValue(1,0.652885364686)
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.76499054412263034, 0.76499054412263034], [0.76499054412263034, 0.76499054412263034]], 
[[0.76499054412263034, 0.76499054412263034], [0.76499054412263034, 0.76499054412263034]], [[0.76499054412263034, 
0.76499054412263034], [0.76499054412263034, 0.76499054412263034]], [[0.76499054412263034, 0.76499054412263034], 
[0.76499054412263034, 0.92207885217687857]], [[0.76499054412263034, 0.76499054412263034], [0.76499054412263034, 
0.80531737845613161]], [[0.76499054412263034, 0.76499054412263034], [0.76499054412263034, 
0.76499054412263034]]])+(1.-msk_ref)*numpy.array([[[0.65288536468628089, 0.86031576930313269], [0.65288536468628089, 
0.82251684778702772]], [[0.65288536468628089, 0.7888837003038216], [0.65288536468628089, 0.65288536468628089]], 
[[0.65288536468628089, 0.65288536468628089], [0.65288536468628089, 0.65288536468628089]], [[0.65288536468628089, 
0.65288536468628089], [0.65288536468628089, 0.65288536468628089]], [[0.65288536468628089, 0.65288536468628089], 
[0.65288536468628089, 0.65288536468628089]], [[0.65288536468628089, 0.88141576474629457], [0.65288536468628089, 
0.65288536468628089]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_expandedData_rank0(self):
      arg0=Data(numpy.array([[[[0.020321372851521025, -0.93303289613231555, -0.95544220026675974, -0.74796287954883267], 
[-0.55901645715191917, -0.36217214791370034, 0.53668281200168733, -0.62072014134910858], [0.83805495909946126, 
0.77801126114371688, 0.29009932155835116, -0.11888480953765868]], [[0.59529921332903846, 0.18339193991589875, 
-0.53504376500242934, -0.44771907747332174], [-0.84892514285551224, 0.094393593853048241, -0.30451187753969244, 
0.48476622589639429], [0.31356243394435968, -0.057441534965831709, 0.21499351301171732, -0.15075804141589444]]], 
[[[-0.94703424018204241, -0.92104669477115619, 0.99823597603366054, -0.32846952396898588], [0.27321519973915231, 
0.57318554411784661, -0.30698818125973681, 0.36977603750084009], [-0.92416980500478063, -0.824412204771928, 
-0.5059174628188523, -0.85789760717090657]], [[0.94487600518674952, -0.70850736941014691, 0.99934299337146282, 
-0.27228751751816205], [0.98790892089967119, -0.86273585977927381, 0.58791085686482214, 0.47962584847037482], 
[0.45615710497316209, 0.98114105471982738, 0.035620550773440396, -0.26053229372544595]]], [[[-0.1202291869497587, 
0.52289135145881516, 0.37036877069633567, -0.73751673095228876], [-0.7786897073727419, -0.40538194892533808, 
0.13045362303694263, -0.080414453370386108], [0.71746193044629014, 0.74301083690226499, 0.73084215912703909, 
0.80085666126038157]], [[0.95037071748664692, -0.91198227675294419, -0.65964614505382402, 0.084204985381424624], 
[-0.36948676097327637, 0.028114901134742132, -0.72803456102656239, -0.44262625387838828], [0.69591152795978251, 
0.19324513934386278, -0.84347572196959231, 0.32397343231861542]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.24422409034649251, 0.45246529593531504, 0.45562364834245339, 
0.4808189145316053], [-0.16981439839788326, -0.30658463816074488, -0.080968711430397855, 0.97248637580601649], 
[0.24869889786329136, 0.47964155030130162, -0.40834468120180056, -0.68979933049205955]], [[0.76755525804469693, 
-0.69809845994799447, -0.20296863341718785, -0.82781540863709657], [-0.60734832748063483, 0.47561020582127189, 
-0.1236043806840661, 0.83631467315572583], [0.12443889610964276, 0.0003527545890085193, 0.077149308433344999, 
0.38046623627369702]]], [[[0.91461817871600304, -0.57475023451064855, -0.90841472637097342, -0.62829270542168669], 
[0.29480893596414326, -0.97192517793673971, -0.47991983665599292, 0.51390104336966513], [-0.75775548373292678, 
-0.18052549363005266, -0.56327405379176243, 0.96964582135188926]], [[-0.22650781069521786, -0.10512952061089442, 
0.84466780553372245, 0.84278325439355717], [0.074018912142888027, 0.10121224724730116, -0.35621293935333775, 
0.92522669299227434], [-0.16210447824173713, -0.77087007456783763, 0.66780917153393071, -0.90668068535906987]]], 
[[[-0.56005535597335143, 0.31356129488603779, -0.15316971454601092, 0.40223673586421604], [-0.050377244061466486, 
-0.73004306346445169, 0.21636888803187748, -0.53958091647537199], [0.46247110370429745, 0.42897768398448388, 
-0.17689648225954002, -0.10259309906145675]], [[-0.26689251428607474, 0.85792406448711533, 0.7690855531164642, 
-0.86751842029893322], [-0.94075022205947945, -0.48534849005619107, 0.22360137438091021, -0.56367049064610764], 
[-0.63452523314118614, -0.30384914180664246, -0.60704022107059785, -0.3528338703453564]]]]))
      arg1=Data(-0.608331286804,self.functionspace)
      arg1.setTaggedValue(1,-0.229410045543)
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.020321372851521025, -0.60833128680443482, -0.60833128680443482, -0.60833128680443482], 
[-0.55901645715191917, -0.36217214791370034, 0.53668281200168733, -0.60833128680443482], [0.83805495909946126, 
0.77801126114371688, 0.29009932155835116, -0.11888480953765868]], [[0.59529921332903846, 0.18339193991589875, 
-0.53504376500242934, -0.44771907747332174], [-0.60833128680443482, 0.094393593853048241, -0.30451187753969244, 
0.48476622589639429], [0.31356243394435968, -0.057441534965831709, 0.21499351301171732, -0.15075804141589444]]], 
[[[-0.60833128680443482, -0.60833128680443482, 0.99823597603366054, -0.32846952396898588], [0.27321519973915231, 
0.57318554411784661, -0.30698818125973681, 0.36977603750084009], [-0.60833128680443482, -0.60833128680443482, 
-0.5059174628188523, -0.60833128680443482]], [[0.94487600518674952, -0.60833128680443482, 0.99934299337146282, 
-0.27228751751816205], [0.98790892089967119, -0.60833128680443482, 0.58791085686482214, 0.47962584847037482], 
[0.45615710497316209, 0.98114105471982738, 0.035620550773440396, -0.26053229372544595]]], [[[-0.1202291869497587, 
0.52289135145881516, 0.37036877069633567, -0.60833128680443482], [-0.60833128680443482, -0.40538194892533808, 
0.13045362303694263, -0.080414453370386108], [0.71746193044629014, 0.74301083690226499, 0.73084215912703909, 
0.80085666126038157]], [[0.95037071748664692, -0.60833128680443482, -0.60833128680443482, 0.084204985381424624], 
[-0.36948676097327637, 0.028114901134742132, -0.60833128680443482, -0.44262625387838828], [0.69591152795978251, 
0.19324513934386278, -0.60833128680443482, 0.32397343231861542]]]])+(1.-msk_ref)*numpy.array([[[[-0.22941004554296418, 
0.45246529593531504, 0.45562364834245339, 0.4808189145316053], [-0.16981439839788326, -0.22941004554296418, 
-0.080968711430397855, 0.97248637580601649], [0.24869889786329136, 0.47964155030130162, -0.22941004554296418, 
-0.22941004554296418]], [[0.76755525804469693, -0.22941004554296418, -0.20296863341718785, -0.22941004554296418], 
[-0.22941004554296418, 0.47561020582127189, -0.1236043806840661, 0.83631467315572583], [0.12443889610964276, 
0.0003527545890085193, 0.077149308433344999, 0.38046623627369702]]], [[[0.91461817871600304, -0.22941004554296418, 
-0.22941004554296418, -0.22941004554296418], [0.29480893596414326, -0.22941004554296418, -0.22941004554296418, 
0.51390104336966513], [-0.22941004554296418, -0.18052549363005266, -0.22941004554296418, 0.96964582135188926]], 
[[-0.22650781069521786, -0.10512952061089442, 0.84466780553372245, 0.84278325439355717], [0.074018912142888027, 
0.10121224724730116, -0.22941004554296418, 0.92522669299227434], [-0.16210447824173713, -0.22941004554296418, 
0.66780917153393071, -0.22941004554296418]]], [[[-0.22941004554296418, 0.31356129488603779, -0.15316971454601092, 
0.40223673586421604], [-0.050377244061466486, -0.22941004554296418, 0.21636888803187748, -0.22941004554296418], 
[0.46247110370429745, 0.42897768398448388, -0.17689648225954002, -0.10259309906145675]], [[-0.22941004554296418, 
0.85792406448711533, 0.7690855531164642, -0.22941004554296418], [-0.22941004554296418, -0.22941004554296418, 
0.22360137438091021, -0.22941004554296418], [-0.22941004554296418, -0.22941004554296418, -0.22941004554296418, 
-0.22941004554296418]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_array_rank1(self):
      arg0=Data(0.157645231269,self.functionspace)
      arg0.setTaggedValue(1,0.716081649316)
      arg1=numpy.array([-0.0288762834591898, 0.48103408390263014])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.15764523126859631, 0.48103408390263014]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.71608164931572404, 0.71608164931572404]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_array_rank1(self):
      arg0=Data(numpy.array([-0.47594729252226253, 0.69591812782109863]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.047244738730578772, 0.15536580029588487]))
      arg1=numpy.array([0.0012221931857514523, 0.2807431886449292])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.0012221931857514523, 0.69591812782109863]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.0012221931857514523, 0.2807431886449292]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_constData_rank1(self):
      arg0=Data(0.586837179464,self.functionspace)
      arg0.setTaggedValue(1,0.546120003623)
      arg1=Data(numpy.array([-0.7842602045051339, 0.016535060489875297]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.58683717946353786, 0.58683717946353786]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.54612000362283886, 0.54612000362283886]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([-0.67359413520434108, -0.09712658701640331]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.84882422147895475, -0.10311432585644575]))
      arg1=Data(numpy.array([0.35883918452068664, -0.25504388543838896]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.35883918452068664, -0.09712658701640331]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.35883918452068664, -0.10311432585644575]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_taggedData_rank1(self):
      arg0=Data(0.708247927721,self.functionspace)
      arg0.setTaggedValue(1,-0.544069251896)
      arg1=Data(numpy.array([0.1322099809790529, 0.0011452422922577643]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.8180746389225646, 0.1358975439783694]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.70824792772120904, 0.70824792772120904]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.8180746389225646, 0.1358975439783694]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([-0.51503313886621216, 0.85281181698410458]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.94713786377098041, -0.61862964694821243]))
      arg1=Data(numpy.array([-0.84325579274513851, -0.478261048265473]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.16213555285616055, 0.57048289256596396]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([-0.51503313886621216, 0.85281181698410458]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.16213555285616055, 0.57048289256596396]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_expandedData_rank1(self):
      arg0=Data(0.369528177665,self.functionspace)
      arg0.setTaggedValue(1,0.0822421658421)
      arg1=Data(numpy.array([-0.9845991986964342, -0.46923308913203976]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.98799743233626836, -0.6799905201091454]))
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.36952817766485468, 0.36952817766485468])+(1.-msk_ref)*numpy.array([0.082242165842115966, 
0.082242165842115966])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([-0.79270890908270375, 0.45978510662593486]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.83309549385730208, 0.15503831909823051]))
      arg1=Data(numpy.array([0.18269716440918637, -0.64345121320159193]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.74718380549682117, -0.18492910420451336]))
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.18269716440918637, 0.45978510662593486])+(1.-msk_ref)*numpy.array([-0.74718380549682117, 
0.15503831909823051])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_array_rank2(self):
      arg0=Data(-0.566144580073,self.functionspace)
      arg0.setTaggedValue(1,0.528503409453)
      arg1=numpy.array([[-0.87296213805104572, 0.78171285196302387, -0.75206465126245625, -0.37760325752482382, 
0.56140852913273864], [-0.40408379117250814, -0.24086342170557162, 0.8196150565102196, 0.12732037183922351, 
-0.69490245846358101], [0.86331196812568689, -0.14995362196070072, -0.21427445909629594, 0.2824984342989636, 
0.21435006695361514], [-0.086206831999868561, 0.62128214949299521, -0.028443307644340665, 0.17196722697690414, 
0.52156533272349992]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.56614458007328183, 0.78171285196302387, -0.56614458007328183, -0.37760325752482382, 
0.56140852913273864], [-0.40408379117250814, -0.24086342170557162, 0.8196150565102196, 0.12732037183922351, 
-0.56614458007328183], [0.86331196812568689, -0.14995362196070072, -0.21427445909629594, 0.2824984342989636, 
0.21435006695361514], [-0.086206831999868561, 0.62128214949299521, -0.028443307644340665, 0.17196722697690414, 
0.52156533272349992]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.52850340945270546, 0.78171285196302387, 0.52850340945270546, 0.52850340945270546, 
0.56140852913273864], [0.52850340945270546, 0.52850340945270546, 0.8196150565102196, 0.52850340945270546, 0.52850340945270546], 
[0.86331196812568689, 0.52850340945270546, 0.52850340945270546, 0.52850340945270546, 0.52850340945270546], 
[0.52850340945270546, 0.62128214949299521, 0.52850340945270546, 0.52850340945270546, 0.52850340945270546]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[-0.81265839421787689, 0.26758136299963464, -0.039939932546667389, 0.082787369558762425, 
0.96314292603725948], [-0.20035649646977394, 0.30575758837077016, -0.014967614657659389, -0.34454970578264077, 
0.16326060363713779], [0.99730052382471346, -0.20752367257200888, -0.28054897850668659, -0.32346909583472461, 
0.83256462668205899], [0.82903872612833673, -0.027773951243003969, -0.94960232582589588, 0.62054418211137197, 
-0.70848122510744749]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.96080046835659583, -0.85008526456100042, 0.41518071014390467, 0.23122321813314395, 
0.84922894101851676], [0.49724369844222727, 0.092773358522958427, 0.70959454786472476, -0.58860469047485164, 
-0.53449686276170838], [0.26904004689599925, -0.35567861386234068, 0.14923076840486571, 0.32405907812410284, 
0.99459882904575658], [0.9585300557126224, 0.8794254635764116, 0.29544460705476605, -0.57828890766257723, 
-0.53844059216552131]]))
      arg1=numpy.array([[-0.42759689305620907, 0.5288377513195337, -0.36520782027363063, -0.096353962936305493, 
0.52597520435345646], [0.22481642713756611, -0.056850043728278177, -0.93738809829177772, 0.38229119722314686, 
-0.45318890370183262], [0.75133292650894035, 0.33349120206980376, -0.35069851925638473, -0.063532053787333176, 
-0.98384425698514311], [-0.44244992103700653, -0.8725983072010739, -0.93880217759007722, -0.342313301291431, 
-0.91388239718385478]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.42759689305620907, 0.5288377513195337, -0.039939932546667389, 0.082787369558762425, 
0.96314292603725948], [0.22481642713756611, 0.30575758837077016, -0.014967614657659389, 0.38229119722314686, 
0.16326060363713779], [0.99730052382471346, 0.33349120206980376, -0.28054897850668659, -0.063532053787333176, 
0.83256462668205899], [0.82903872612833673, -0.027773951243003969, -0.93880217759007722, 0.62054418211137197, 
-0.70848122510744749]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.42759689305620907, 0.5288377513195337, 0.41518071014390467, 0.23122321813314395, 
0.84922894101851676], [0.49724369844222727, 0.092773358522958427, 0.70959454786472476, 0.38229119722314686, 
-0.45318890370183262], [0.75133292650894035, 0.33349120206980376, 0.14923076840486571, 0.32405907812410284, 
0.99459882904575658], [0.9585300557126224, 0.8794254635764116, 0.29544460705476605, -0.342313301291431, -0.53844059216552131]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_constData_rank2(self):
      arg0=Data(-0.233453242662,self.functionspace)
      arg0.setTaggedValue(1,-0.456279104054)
      arg1=Data(numpy.array([[0.66295701883303426, 0.29837779390150132, -0.82144022260100957, -0.77655984622382679, 
-0.66922622684304289], [-0.03256402393406832, -0.42799728354943078, 0.84506060005715655, 0.58217244657439315, 
0.87359667279386199], [0.23466185627816616, 0.6848241906434076, 0.43997066196917545, -0.23893662021692919, 
0.21875346774664495], [-0.77373251780068508, 0.70651723473975481, 0.0010900801234461355, 0.61280029446432605, 
0.13102123501431007]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.66295701883303426, 0.29837779390150132, -0.23345324266177125, -0.23345324266177125, 
-0.23345324266177125], [-0.03256402393406832, -0.23345324266177125, 0.84506060005715655, 0.58217244657439315, 
0.87359667279386199], [0.23466185627816616, 0.6848241906434076, 0.43997066196917545, -0.23345324266177125, 
0.21875346774664495], [-0.23345324266177125, 0.70651723473975481, 0.0010900801234461355, 0.61280029446432605, 
0.13102123501431007]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.66295701883303426, 0.29837779390150132, -0.45627910405357452, -0.45627910405357452, 
-0.45627910405357452], [-0.03256402393406832, -0.42799728354943078, 0.84506060005715655, 0.58217244657439315, 
0.87359667279386199], [0.23466185627816616, 0.6848241906434076, 0.43997066196917545, -0.23893662021692919, 
0.21875346774664495], [-0.45627910405357452, 0.70651723473975481, 0.0010900801234461355, 0.61280029446432605, 
0.13102123501431007]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[-0.65215706962235931, 0.90295000568745021, -0.74612755909172068, 0.37953410665950926, 
0.78357573816776882], [-0.78423218701280173, 0.75555214479296429, -0.59550852179237301, -0.24260494628579798, 
-0.2932238102951934], [-0.055686626024025632, 0.27514500129879305, -0.01600793024888203, 0.36687585617384455, 
-0.22472615152530251], [-0.16564252376898891, -0.50691065174515448, 0.5496939242032004, 0.87263539425927727, 
-0.33391848438760219]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.12386188640925844, -0.25058730239632698, -0.9689271379594171, -0.5124211412670221, 
0.74240065030212299], [0.45117376601281056, -0.83930628760299553, -0.86873208426025794, 0.68322721489185634, 
-0.77441137178524722], [0.022366441843409968, 0.59963352898563937, 0.90647843959010643, -0.45676736088574121, 
0.86628078310093226], [-0.87044361286207428, 0.81055443077557054, -0.28150703377487196, -0.97466590847188317, 
0.35149769824613997]]))
      arg1=Data(numpy.array([[0.1020745016007365, 0.21200816967965475, -0.0048679588261282269, -0.48528770101328256, 
0.37724018457132891], [-0.40651544887620483, -0.43859554346733276, 0.99662000840930154, -0.61717493207815277, 
-0.059381870626120881], [-0.98663711200355841, -0.23691413704363962, 0.029172336036756841, 0.47465699025438401, 
-0.43916394214769139], [-0.0086894600203351047, 0.78036589414306445, 0.58198638128365721, 0.71269158150109702, 
0.41115923503302954]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.1020745016007365, 0.90295000568745021, -0.0048679588261282269, 0.37953410665950926, 
0.78357573816776882], [-0.40651544887620483, 0.75555214479296429, 0.99662000840930154, -0.24260494628579798, 
-0.059381870626120881], [-0.055686626024025632, 0.27514500129879305, 0.029172336036756841, 0.47465699025438401, 
-0.22472615152530251], [-0.0086894600203351047, 0.78036589414306445, 0.58198638128365721, 0.87263539425927727, 
0.41115923503302954]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.1020745016007365, 0.21200816967965475, -0.0048679588261282269, -0.48528770101328256, 
0.74240065030212299], [0.45117376601281056, -0.43859554346733276, 0.99662000840930154, 0.68322721489185634, 
-0.059381870626120881], [0.022366441843409968, 0.59963352898563937, 0.90647843959010643, 0.47465699025438401, 
0.86628078310093226], [-0.0086894600203351047, 0.81055443077557054, 0.58198638128365721, 0.71269158150109702, 
0.41115923503302954]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_taggedData_rank2(self):
      arg0=Data(-0.585208864451,self.functionspace)
      arg0.setTaggedValue(1,0.0506898621422)
      arg1=Data(numpy.array([[-0.80997906792202423, -0.075924759840865441, 0.98928518123637499, 0.86706250171141597, 
-0.20578992343413982], [0.34054971591980054, -0.8731692484525917, -0.31849377908136511, -0.64068207469152583, 
0.50897802680347048], [-0.29260185280664297, -0.69905302942108505, -0.10391526485862457, -0.93109086526597662, 
0.29696330488998668], [0.67915180353525195, 0.71820396998563285, -0.76723938174828388, -0.37773962131976435, 
-0.13767357299522898]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.099303007344046401, 0.49937050384309001, 0.091044772694302223, 
0.83837186928029905, -0.017212533407631625], [-0.12266788468962431, 0.98722623400467557, 0.78431773233335655, 
0.80737492716491621, 0.22088529892352859], [0.71858828029752075, 0.79590416091199878, 0.76064250837914393, 0.81458140120815936, 
-0.72761294910228314], [0.70188024720370601, 0.15650206134474565, 0.69772536269288676, 0.12717822173663551, 
0.40879517285709865]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.58520886445123543, -0.075924759840865441, 0.98928518123637499, 0.86706250171141597, 
-0.20578992343413982], [0.34054971591980054, -0.58520886445123543, -0.31849377908136511, -0.58520886445123543, 
0.50897802680347048], [-0.29260185280664297, -0.58520886445123543, -0.10391526485862457, -0.58520886445123543, 
0.29696330488998668], [0.67915180353525195, 0.71820396998563285, -0.58520886445123543, -0.37773962131976435, 
-0.13767357299522898]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.050689862142220488, 0.49937050384309001, 0.091044772694302223, 0.83837186928029905, 
0.050689862142220488], [0.050689862142220488, 0.98722623400467557, 0.78431773233335655, 0.80737492716491621, 
0.22088529892352859], [0.71858828029752075, 0.79590416091199878, 0.76064250837914393, 0.81458140120815936, 
0.050689862142220488], [0.70188024720370601, 0.15650206134474565, 0.69772536269288676, 0.12717822173663551, 
0.40879517285709865]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[-0.30850678432326584, -0.14671440717841144, -0.62505292697061354, -0.48788175882104245, 
-0.88033622977949011], [-0.20413849077041912, 0.39729468723019079, -0.65937111106987834, 0.50049182101978684, 
0.65618274263823384], [-0.1379286800364965, 0.94149976500306054, 0.90332903869476699, 0.16809094104236655, 
-0.59432802779299387], [-0.55196115682150837, -0.47973942612365184, 0.79574098934383808, 0.33896290658985317, 
0.2343806675438127]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.37368847761905766, -0.95593326659472955, 0.39754064835368874, 
-0.45246762809233432, -0.42425605634194241], [-0.99798056188674145, -0.52507689732559304, 0.0015822699455976252, 
0.18946957061682079, 0.27764304571427934], [-0.89045114004022086, 0.9222661634403877, -0.75320893801876476, 
-0.58466222910172183, -0.9324897007904267], [-0.77738407415352162, 0.33277681575170703, 0.44868550432487542, 
0.80841032194704487, 0.50893188843742299]]))
      arg1=Data(numpy.array([[-0.76289796142918442, 0.39345737042716311, 0.3596731444573853, 0.66182356019299826, 
-0.54510163887049989], [-0.26177332627229943, -0.78574343883971465, -0.75984224877101436, 0.028640779973349906, 
0.46101551650890005], [-0.12687242083637007, -0.76264019315585818, -0.96340043133788567, 0.99172307905538948, 
0.93811391467075067], [0.10831278342078354, -0.90173438972021147, -0.71795722450200561, 0.24798855275616649, 
-0.91077721563078629]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.75864681094935826, -0.37420478894040654, 0.94829061011275528, 
0.092053105818372449, 0.88394212203325573], [0.50298175228281328, -0.37248434444266132, -0.97564238193308217, 
0.069365016354938858, -0.68888590754687073], [0.42821414550127868, -0.81605567791083389, -0.39448602367127994, 
0.85091949395493804, 0.3780204265086502], [0.80158313848467411, -0.49235019033769323, -0.010926781924918805, 
-0.63494508446086328, 0.66241645779934299]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.30850678432326584, 0.39345737042716311, 0.3596731444573853, 0.66182356019299826, 
-0.54510163887049989], [-0.20413849077041912, 0.39729468723019079, -0.65937111106987834, 0.50049182101978684, 
0.65618274263823384], [-0.12687242083637007, 0.94149976500306054, 0.90332903869476699, 0.99172307905538948, 
0.93811391467075067], [0.10831278342078354, -0.47973942612365184, 0.79574098934383808, 0.33896290658985317, 
0.2343806675438127]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.37368847761905766, -0.37420478894040654, 0.94829061011275528, 0.092053105818372449, 
0.88394212203325573], [0.50298175228281328, -0.37248434444266132, 0.0015822699455976252, 0.18946957061682079, 
0.27764304571427934], [0.42821414550127868, 0.9222661634403877, -0.39448602367127994, 0.85091949395493804, 0.3780204265086502], 
[0.80158313848467411, 0.33277681575170703, 0.44868550432487542, 0.80841032194704487, 0.66241645779934299]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_expandedData_rank2(self):
      arg0=Data(-0.8367060972,self.functionspace)
      arg0.setTaggedValue(1,0.0350605925437)
      arg1=Data(numpy.array([[-0.85989987056335004, 0.95618415745286112, 0.77463129739257752, -0.70067281119395219, 
-0.066047724905169858], [0.86264523702229678, -0.034293428705689299, 0.94341208480632655, -0.97918888349110356, 
-0.079698215200129541], [-0.0074088951964081939, -0.32211469546282068, 0.28914186744481163, -0.76611669248815795, 
-0.80112036211678572], [-0.19454051292116281, -0.85192083234455263, -0.17204001110029332, -0.16718805755325405, 
0.94370688593350516]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.79462672331170969, 0.10786244880092033, 0.92368483185879247, -0.52964637915988444, 
0.89011451705386446], [0.9919205122299033, 0.95607270137793043, 0.063625713506444859, -0.82052222689628329, 
0.86050943954591586], [-0.89960127327577322, 0.50723964196251692, -0.02283813441781124, 0.94861758985346567, 
0.61719197613764143], [-0.39088026638104756, 0.57775086029719569, 0.62360626786169004, 0.60219522446603913, 
-0.095231346974827158]]))
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[-0.83670609720026534, 0.95618415745286112, 0.77463129739257752, -0.70067281119395219, 
-0.066047724905169858], [0.86264523702229678, -0.034293428705689299, 0.94341208480632655, -0.83670609720026534, 
-0.079698215200129541], [-0.0074088951964081939, -0.32211469546282068, 0.28914186744481163, -0.76611669248815795, 
-0.80112036211678572], [-0.19454051292116281, -0.83670609720026534, -0.17204001110029332, -0.16718805755325405, 
0.94370688593350516]])+(1.-msk_ref)*numpy.array([[0.03506059254370375, 0.10786244880092033, 0.92368483185879247, 
0.03506059254370375, 0.89011451705386446], [0.9919205122299033, 0.95607270137793043, 0.063625713506444859, 0.03506059254370375, 
0.86050943954591586], [0.03506059254370375, 0.50723964196251692, 0.03506059254370375, 0.94861758985346567, 
0.61719197613764143], [0.03506059254370375, 0.57775086029719569, 0.62360626786169004, 0.60219522446603913, 
0.03506059254370375]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[-0.34373928503159124, -0.20095674682530928, 0.95586896214465544, -0.80855219520628352, 
0.99396751891132795], [0.40162251877028354, -0.19260684688049534, 0.4379110629309686, 0.38692750630684891, 
0.63370086700747796], [0.37464522230985753, 0.2472286384517759, 0.19957447822987961, 0.30034489040810564, 0.20417583681583196], 
[0.43170346131406179, -0.40595521447218741, 0.38328741641588371, -0.64164601736722027, 
0.44536105747043542]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.78167198056366582, -0.69901804178307247, 0.32279307641069166, 
0.090874473790751864, -0.24649499080861936], [-0.72101815532963576, 0.019336192244026273, -0.85743081322410064, 
-0.79792604048493221, -0.41024219136817819], [0.95637313861325035, -0.52993977638935363, -0.10838678348774478, 
0.24616595678724185, -0.46684632231387457], [-0.91688210200431519, -0.92995326912189413, -0.15935492948777386, 
-0.46493458970034784, 0.92308361870309619]]))
      arg1=Data(numpy.array([[0.34999144895161294, 0.2867861514795973, -0.24803374798797839, -0.83053123485943026, 
0.71679812088305694], [0.87160151772896022, 0.15880702138094605, -0.70250327340032048, 0.88803288312847806, 
0.4017949060043422], [-0.67803992778620725, 0.65822511725017097, 0.28320404906394159, 0.70990960617397381, 
-0.4670691811622889], [0.41680006736135655, 0.49938710728969693, -0.14441005731430834, -0.073652978098713406, 
0.048693430751279188]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.040544367585485741, -0.054138641181803671, -0.18487061883388378, 
-0.18530692129090642, -0.25801541779005044], [-0.79216945806755601, 0.51987045565101075, 0.4253962162954632, 
0.89908331742690306, -0.58343361278128958], [-0.32286527453853275, -0.48630932124048942, 0.39634533656084936, 
0.033506780092776856, -0.32207988533792431], [0.69982448441067535, -0.73033174432454162, 0.94246537337208935, 
0.23627320563011223, 0.36489433134884752]]))
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.34999144895161294, 0.2867861514795973, 0.95586896214465544, -0.80855219520628352, 
0.99396751891132795], [0.87160151772896022, 0.15880702138094605, 0.4379110629309686, 0.88803288312847806, 0.63370086700747796], 
[0.37464522230985753, 0.65822511725017097, 0.28320404906394159, 0.70990960617397381, 0.20417583681583196], 
[0.43170346131406179, 0.49938710728969693, 0.38328741641588371, -0.073652978098713406, 
0.44536105747043542]])+(1.-msk_ref)*numpy.array([[0.040544367585485741, -0.054138641181803671, 0.32279307641069166, 
0.090874473790751864, -0.24649499080861936], [-0.72101815532963576, 0.51987045565101075, 0.4253962162954632, 
0.89908331742690306, -0.41024219136817819], [0.95637313861325035, -0.48630932124048942, 0.39634533656084936, 
0.24616595678724185, -0.32207988533792431], [0.69982448441067535, -0.73033174432454162, 0.94246537337208935, 
0.23627320563011223, 0.92308361870309619]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_array_rank3(self):
      arg0=Data(0.0581603033552,self.functionspace)
      arg0.setTaggedValue(1,-0.349121797934)
      arg1=numpy.array([[[0.11212022647318243, 0.92431326100322941], [0.86294049550861751, -0.54577029057950299]], 
[[0.30162042594634375, 0.65153183276311566], [0.80784816169055063, -0.49647676963675558]], [[0.0016770050239853784, 
-0.76312931795236461], [-0.38377799913819266, -0.12288106017343825]], [[0.86225188976602163, -0.73263296462566041], 
[0.59637972478647505, -0.49278052730666388]], [[-0.27638452396632274, -0.75982151285804878], [0.32525341084371373, 
-0.93082850394127825]], [[-0.021108338065601862, 0.044094406189113311], [-0.33098268109813711, -0.45115291322275364]]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.11212022647318243, 0.92431326100322941], [0.86294049550861751, 0.05816030335518696]], 
[[0.30162042594634375, 0.65153183276311566], [0.80784816169055063, 0.05816030335518696]], [[0.05816030335518696, 
0.05816030335518696], [0.05816030335518696, 0.05816030335518696]], [[0.86225188976602163, 0.05816030335518696], 
[0.59637972478647505, 0.05816030335518696]], [[0.05816030335518696, 0.05816030335518696], [0.32525341084371373, 
0.05816030335518696]], [[0.05816030335518696, 0.05816030335518696], [0.05816030335518696, 
0.05816030335518696]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.11212022647318243, 0.92431326100322941], [0.86294049550861751, 
-0.34912179793353193]], [[0.30162042594634375, 0.65153183276311566], [0.80784816169055063, -0.34912179793353193]], 
[[0.0016770050239853784, -0.34912179793353193], [-0.34912179793353193, -0.12288106017343825]], [[0.86225188976602163, 
-0.34912179793353193], [0.59637972478647505, -0.34912179793353193]], [[-0.27638452396632274, -0.34912179793353193], 
[0.32525341084371373, -0.34912179793353193]], [[-0.021108338065601862, 0.044094406189113311], [-0.33098268109813711, 
-0.34912179793353193]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_array_rank3(self):
      arg0=Data(numpy.array([[[0.51745400487065463, -0.046018812730892478], [0.93448372998822093, -0.15624548432249896]], 
[[-0.19172879200641457, -0.79651544933935736], [-0.15176950730662009, -0.13079058253123965]], [[0.037505168169944758, 
-0.46778341016285641], [-0.27839242608504011, -0.21031804391015574]], [[0.18704468648397143, -0.92178872447095439], 
[0.099081878288656222, 0.35867382961234551]], [[-0.45608361794547347, -0.26142429345451146], [-0.54796118279600425, 
-0.20833025068917332]], [[-0.17579136952010566, 0.55497568775611072], [-0.67560881772233694, 
-0.21997428049707923]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.60566560883647336, -0.29630658336264948], [-0.87510153393660994, 
0.69526723875992325]], [[0.37148881382237198, -0.51171679135224224], [-0.27107595939647466, 0.35755098694698262]], 
[[-0.14043640634308741, 0.13036455401271119], [-0.53395960656650399, -0.023442547016014226]], [[-0.51288684167259024, 
0.31930186247779213], [0.4647898134236943, 0.49107334166705852]], [[-0.55324165237516754, -0.014599188008774799], 
[0.96183327522495943, -0.14508297231814549]], [[-0.27185406040237092, 0.95658917212938332], [-0.87508686477361231, 
-0.42737850356341411]]]))
      arg1=numpy.array([[[0.9872501315717821, -0.61258669757993767], [0.79290816451735524, 0.075272751954812422]], 
[[0.72410541969379305, -0.83079825068869595], [-0.92718658420664535, 0.029072395647118832]], [[-0.98386594926830795, 
0.2420005355888577], [0.75731198088670593, -0.65028424942459195]], [[0.11481268877581563, -0.66970518060714834], 
[0.95079934440937763, 0.0036926128296570493]], [[-0.83413942275382147, 0.45274370051699075], [0.28400088867799789, 
0.8662157653352538]], [[-0.33098582440372182, 0.21458367461821037], [0.44429092297623662, -0.94171297612495297]]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.9872501315717821, -0.046018812730892478], [0.93448372998822093, 0.075272751954812422]], 
[[0.72410541969379305, -0.79651544933935736], [-0.15176950730662009, 0.029072395647118832]], [[0.037505168169944758, 
0.2420005355888577], [0.75731198088670593, -0.21031804391015574]], [[0.18704468648397143, -0.66970518060714834], 
[0.95079934440937763, 0.35867382961234551]], [[-0.45608361794547347, 0.45274370051699075], [0.28400088867799789, 
0.8662157653352538]], [[-0.17579136952010566, 0.55497568775611072], [0.44429092297623662, 
-0.21997428049707923]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.9872501315717821, -0.29630658336264948], [0.79290816451735524, 
0.69526723875992325]], [[0.72410541969379305, -0.51171679135224224], [-0.27107595939647466, 0.35755098694698262]], 
[[-0.14043640634308741, 0.2420005355888577], [0.75731198088670593, -0.023442547016014226]], [[0.11481268877581563, 
0.31930186247779213], [0.95079934440937763, 0.49107334166705852]], [[-0.55324165237516754, 0.45274370051699075], 
[0.96183327522495943, 0.8662157653352538]], [[-0.27185406040237092, 0.95658917212938332], [0.44429092297623662, 
-0.42737850356341411]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_constData_rank3(self):
      arg0=Data(0.257971702048,self.functionspace)
      arg0.setTaggedValue(1,0.646644382588)
      arg1=Data(numpy.array([[[0.49904929085854621, -0.079761501970988746], [0.075034915444887496, -0.94716344759297177]], 
[[-0.37980260593056059, 0.85797450155684918], [0.041844423739421988, 0.62190282388567164]], [[-0.60519295629858183, 
-0.30044611114194431], [-0.88830803768029365, 0.05880187491888389]], [[0.91664888909366882, -0.66140129841949724], 
[-0.83569339239858054, -0.74103720634562897]], [[-0.20530841406562872, -0.79656147644353958], [-0.021618611122280029, 
0.85238657512193949]], [[0.97795389625070084, 0.22832269349992895], [-0.60788049508889452, 
0.30726030792880121]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.49904929085854621, 0.25797170204809405], [0.25797170204809405, 0.25797170204809405]], 
[[0.25797170204809405, 0.85797450155684918], [0.25797170204809405, 0.62190282388567164]], [[0.25797170204809405, 
0.25797170204809405], [0.25797170204809405, 0.25797170204809405]], [[0.91664888909366882, 0.25797170204809405], 
[0.25797170204809405, 0.25797170204809405]], [[0.25797170204809405, 0.25797170204809405], [0.25797170204809405, 
0.85238657512193949]], [[0.97795389625070084, 0.25797170204809405], [0.25797170204809405, 
0.30726030792880121]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.64664438258815893, 0.64664438258815893], [0.64664438258815893, 
0.64664438258815893]], [[0.64664438258815893, 0.85797450155684918], [0.64664438258815893, 0.64664438258815893]], 
[[0.64664438258815893, 0.64664438258815893], [0.64664438258815893, 0.64664438258815893]], [[0.91664888909366882, 
0.64664438258815893], [0.64664438258815893, 0.64664438258815893]], [[0.64664438258815893, 0.64664438258815893], 
[0.64664438258815893, 0.85238657512193949]], [[0.97795389625070084, 0.64664438258815893], [0.64664438258815893, 
0.64664438258815893]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_constData_rank3(self):
      arg0=Data(numpy.array([[[-0.56751632774592187, -0.73873819050204514], [-0.70675214085599025, 0.075273297867217703]], 
[[-0.82619054246682722, -0.43841409013876054], [-0.81593285813249583, 0.57072143683876275]], [[0.75097389353446231, 
0.8059142329234954], [0.75469092644153113, -0.590953918982426]], [[0.99708694453613278, -0.47598366396407665], 
[-0.15581235865014786, -0.81818514340188542]], [[0.80447282591491498, 0.7575818859809631], [0.43866527734468796, 
-0.77086721558887961]], [[-0.095833961867939577, 0.64854342025417644], [0.13004699443460566, 
0.80163635085132712]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.99316257628238525, 0.9198743871338837], [0.051959190663095622, 
-0.87982546560302755]], [[0.61856474049176824, 0.48035555547172004], [-0.77183438688142281, -0.47372766104737107]], 
[[0.9235928849032522, -0.10938503521583876], [0.55613100507586699, -0.5588780048717723]], [[0.11274959464476875, 
-0.81729613150631075], [0.92870934279327022, -0.5586894895370409]], [[0.90668972584954544, 0.67362436327774144], 
[0.57843369758161201, -0.91218232013865941]], [[0.18244027677393304, 0.90502484761620838], [-0.37599484305367947, 
-0.085246056017418637]]]))
      arg1=Data(numpy.array([[[-0.66932848598996109, -0.58207476065266528], [-0.1493405950290152, 0.12445960583277182]], 
[[0.45155948881913743, -0.49232825711577721], [-0.56890294923615747, 0.046142203525306957]], [[0.96275426269804543, 
-0.25600355922921492], [-0.9213170293556654, -0.95218629371696761]], [[-0.79758049150801025, 0.93439075617400502], 
[0.14110111669783043, -0.0053009431666664053]], [[0.4887902663007464, 0.82106629975476064], [-0.5139818225807713, 
-0.23995011242420916]], [[0.45373257160847591, -0.87255921993470342], [-0.13467963047673903, 
0.22706913569205978]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[-0.56751632774592187, -0.58207476065266528], [-0.1493405950290152, 0.12445960583277182]], 
[[0.45155948881913743, -0.43841409013876054], [-0.56890294923615747, 0.57072143683876275]], [[0.96275426269804543, 
0.8059142329234954], [0.75469092644153113, -0.590953918982426]], [[0.99708694453613278, 0.93439075617400502], 
[0.14110111669783043, -0.0053009431666664053]], [[0.80447282591491498, 0.82106629975476064], [0.43866527734468796, 
-0.23995011242420916]], [[0.45373257160847591, 0.64854342025417644], [0.13004699443460566, 
0.80163635085132712]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.66932848598996109, 0.9198743871338837], [0.051959190663095622, 
0.12445960583277182]], [[0.61856474049176824, 0.48035555547172004], [-0.56890294923615747, 0.046142203525306957]], 
[[0.96275426269804543, -0.10938503521583876], [0.55613100507586699, -0.5588780048717723]], [[0.11274959464476875, 
0.93439075617400502], [0.92870934279327022, -0.0053009431666664053]], [[0.90668972584954544, 0.82106629975476064], 
[0.57843369758161201, -0.23995011242420916]], [[0.45373257160847591, 0.90502484761620838], [-0.13467963047673903, 
0.22706913569205978]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_taggedData_rank3(self):
      arg0=Data(0.408669382137,self.functionspace)
      arg0.setTaggedValue(1,-0.703473386386)
      arg1=Data(numpy.array([[[-0.72265172414617895, 0.36842589688480998], [0.82800897405276208, -0.31630373506981457]], 
[[-0.24366919257554343, -0.23889030631113917], [-0.52800896225048288, -0.21248414052650766]], [[0.24773819543004527, 
-0.26540119896144665], [-0.83808606494769955, -0.67309354394148579]], [[0.04492619479074178, -0.5273146784313123], 
[-0.4614640788557387, -0.26351672856909425]], [[0.60540692576582789, 0.076964383439644291], [-0.18122633890826956, 
-0.10989714114659166]], [[-0.91176828299242252, -0.4863357690158705], [-0.028353540080707784, 
-0.67873198803390133]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.75234456096077973, 0.47310059411707095], [0.97485531260531388, 
0.036623209278044522]], [[0.80159772524220596, 0.0039440509463060103], [0.47162965967450066, 0.063875645716604534]], 
[[0.67962813041533177, 0.70506608156224315], [0.31622514170725724, 0.95725919878846244]], [[0.34190799146654549, 
0.33607946212592732], [-0.54969741489444357, -0.67229942405314369]], [[-0.75877915279959729, -0.25502321313553011], 
[-0.11246023107344127, 0.70044262849547567]], [[0.8818668030507637, -0.64953062543270512], [0.042723626292767714, 
0.89103423021136363]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.4086693821370766, 0.4086693821370766], [0.82800897405276208, 0.4086693821370766]], 
[[0.4086693821370766, 0.4086693821370766], [0.4086693821370766, 0.4086693821370766]], [[0.4086693821370766, 
0.4086693821370766], [0.4086693821370766, 0.4086693821370766]], [[0.4086693821370766, 0.4086693821370766], [0.4086693821370766, 
0.4086693821370766]], [[0.60540692576582789, 0.4086693821370766], [0.4086693821370766, 0.4086693821370766]], 
[[0.4086693821370766, 0.4086693821370766], [0.4086693821370766, 0.4086693821370766]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.75234456096077973, 0.47310059411707095], [0.97485531260531388, 
0.036623209278044522]], [[0.80159772524220596, 0.0039440509463060103], [0.47162965967450066, 0.063875645716604534]], 
[[0.67962813041533177, 0.70506608156224315], [0.31622514170725724, 0.95725919878846244]], [[0.34190799146654549, 
0.33607946212592732], [-0.54969741489444357, -0.67229942405314369]], [[-0.70347338638573675, -0.25502321313553011], 
[-0.11246023107344127, 0.70044262849547567]], [[0.8818668030507637, -0.64953062543270512], [0.042723626292767714, 
0.89103423021136363]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_taggedData_rank3(self):
      arg0=Data(numpy.array([[[0.34361794289520842, -0.79214603437560327], [-0.67469557682196202, -0.79554951867898738]], 
[[0.96474076774081019, 0.86760220798227095], [-0.65602182052786651, -0.090618351473820358]], [[-0.50410919250595509, 
-0.39449742983708891], [-0.61047476471374895, -0.13986711763859283]], [[0.40614535872757696, 0.49363459177702729], 
[-0.99096492684145732, -0.48164033297073883]], [[0.28262904021112401, -0.36774714969436406], [-0.43020413774246302, 
0.33125918175575886]], [[0.41135236616191628, 0.24704948263970783], [0.17671215385410055, 
-0.36880284585106216]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.23270826050508076, 0.55031406934725324], [0.092799947294044838, 
0.70142274746084854]], [[-0.25806542137818278, -0.87043773253800327], [-0.20561418588308666, 0.60804147150696619]], 
[[0.27875168984588394, 0.046223650669499516], [0.21294073592740315, 0.17510990250879055]], [[0.5360111538711636, 
0.35942691987175945], [0.17741284066963869, 0.93652114853145196]], [[-0.78753623491747038, -0.21960161615292195], 
[-0.18155304297147001, -0.69524778068455761]], [[0.83868620490676649, 0.0061992069149379159], [-0.20834087789014211, 
0.11313571214536511]]]))
      arg1=Data(numpy.array([[[-0.95777503985776336, 0.56386012454524304], [0.76573667300539716, -0.89229386480914141]], 
[[-0.31035742914627096, -0.5228172508939628], [0.096398948411295926, 0.91370766889932464]], [[0.85033658242781307, 
0.85268637772571898], [-0.13271122394598978, -0.62815262953593365]], [[-0.82915300066758801, 0.16335568135696987], 
[-0.15570534038852646, -0.72875100039206209]], [[-0.210459568078049, -0.29221352220263896], [0.80332611313097635, 
-0.51027457536803156]], [[0.55922997294548638, -0.55841452089231103], [0.70573807495946217, 
-0.14111887714839244]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.044900424380813986, -0.35514225753300876], [-0.43271652761471247, 
-0.094909517366706275]], [[0.46485774988607798, -0.25534078780382652], [-0.23160409661392989, -0.68419287018217623]], 
[[0.19410334639787785, 0.36164243112719752], [0.922804486894377, 0.17831645481510328]], [[-0.30227034917847595, 
0.25943111330765767], [0.050140014847715575, -0.64471232473927387]], [[0.4661054807291336, 0.47678010231455681], 
[0.35863653458299161, 0.016778825939667508]], [[-0.22201071392861627, 0.87879019799689506], [-0.028293896975291943, 
0.15393400831946891]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.34361794289520842, 0.56386012454524304], [0.76573667300539716, -0.79554951867898738]], 
[[0.96474076774081019, 0.86760220798227095], [0.096398948411295926, 0.91370766889932464]], [[0.85033658242781307, 
0.85268637772571898], [-0.13271122394598978, -0.13986711763859283]], [[0.40614535872757696, 0.49363459177702729], 
[-0.15570534038852646, -0.48164033297073883]], [[0.28262904021112401, -0.29221352220263896], [0.80332611313097635, 
0.33125918175575886]], [[0.55922997294548638, 0.24704948263970783], [0.70573807495946217, 
-0.14111887714839244]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.23270826050508076, 0.55031406934725324], [0.092799947294044838, 
0.70142274746084854]], [[0.46485774988607798, -0.25534078780382652], [-0.20561418588308666, 0.60804147150696619]], 
[[0.27875168984588394, 0.36164243112719752], [0.922804486894377, 0.17831645481510328]], [[0.5360111538711636, 
0.35942691987175945], [0.17741284066963869, 0.93652114853145196]], [[0.4661054807291336, 0.47678010231455681], 
[0.35863653458299161, 0.016778825939667508]], [[0.83868620490676649, 0.87879019799689506], [-0.028293896975291943, 
0.15393400831946891]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_expandedData_rank3(self):
      arg0=Data(-0.226197098806,self.functionspace)
      arg0.setTaggedValue(1,0.522192103891)
      arg1=Data(numpy.array([[[-0.476801185411313, 0.44963895943426313], [0.60716647491303011, 0.39648169172310621]], 
[[-0.61188355620687029, -0.8200235603683792], [0.7148583348047397, -0.91767242519128334]], [[0.39767620370403223, 
0.62277713712673211], [-0.93601734373066692, 0.21708579495611535]], [[0.39840239441288894, 0.83592894519047656], 
[-0.77696316832477064, 0.38217695509729332]], [[-0.51941026427864934, 0.63602057176447357], [0.13171585832349253, 
0.58992892878117908]], [[-0.49900668044267915, 0.71167296228538834], [0.33167107311847377, 
-0.55775348136514102]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.71337553566700285, 0.59202111257777457], [-0.21456982223917076, 
-0.71002583837425925]], [[0.24531735150782907, 0.39937968963766823], [-0.21484993112655526, -0.86378912365397964]], 
[[0.69918125290293864, -0.94461214810425709], [0.8242591161308539, -0.19657343614514389]], [[-0.64708651457614685, 
-0.4732042074857854], [0.8064938979298597, -0.11741313018408772]], [[0.66272112582001386, -0.54392741431807989], 
[0.499942470885264, -0.42093049581276465]], [[-0.34841932486190474, -0.48760940304902634], [0.49404808300120928, 
-0.45202140599415497]]]))
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-0.22619709880625916, 0.44963895943426313], [0.60716647491303011, 0.39648169172310621]], 
[[-0.22619709880625916, -0.22619709880625916], [0.7148583348047397, -0.22619709880625916]], [[0.39767620370403223, 
0.62277713712673211], [-0.22619709880625916, 0.21708579495611535]], [[0.39840239441288894, 0.83592894519047656], 
[-0.22619709880625916, 0.38217695509729332]], [[-0.22619709880625916, 0.63602057176447357], [0.13171585832349253, 
0.58992892878117908]], [[-0.22619709880625916, 0.71167296228538834], [0.33167107311847377, 
-0.22619709880625916]]])+(1.-msk_ref)*numpy.array([[[0.71337553566700285, 0.59202111257777457], [0.52219210389146631, 
0.52219210389146631]], [[0.52219210389146631, 0.52219210389146631], [0.52219210389146631, 0.52219210389146631]], 
[[0.69918125290293864, 0.52219210389146631], [0.8242591161308539, 0.52219210389146631]], [[0.52219210389146631, 
0.52219210389146631], [0.8064938979298597, 0.52219210389146631]], [[0.66272112582001386, 0.52219210389146631], 
[0.52219210389146631, 0.52219210389146631]], [[0.52219210389146631, 0.52219210389146631], [0.52219210389146631, 
0.52219210389146631]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank3_expandedData_rank3(self):
      arg0=Data(numpy.array([[[-0.15254677491326296, -0.73792843638706129], [-0.8860393636164603, 0.17141576138477377]], 
[[-0.58390457441332733, 0.58893441480122144], [0.63803303568095848, -0.9880496752401795]], [[-0.34820091961137156, 
0.49777371182112007], [0.86080228247319424, 0.42868250082943504]], [[0.67542965261369536, 0.6121873787033163], 
[0.2652931974973356, 0.16497221834953968]], [[-0.74059011075673564, 0.59969423838435021], [-0.53251293900871088, 
0.99180907488808523]], [[-0.50592776611259538, 0.29029544745011449], [-0.19869794858667844, 
-0.98656336507129083]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.42486946532406078, -0.091090334688143093], [-0.13379915635758977, 
-0.59823779897950735]], [[0.08619882870789497, 0.5604474445370613], [0.12656809755099241, -0.68357805878068012]], 
[[-0.6719040453202263, -0.43482705708570735], [-0.26808535349731311, 0.90186927101211256]], [[-0.9897041292722466, 
0.30520100226807645], [-0.3055416703957925, -0.048221750186620893]], [[0.21951523032010689, -0.36215940449847883], 
[-0.56081915812168681, 0.66782741691403369]], [[-0.57022064974883424, -0.53594351926414108], [-0.748573617352565, 
-0.077799281835090994]]]))
      arg1=Data(numpy.array([[[0.70447088573450056, -0.71278717909498845], [-0.86132753808998697, 0.76366555387101842]], 
[[-0.46138212438459769, -0.3489329333342146], [0.54606158476413524, -0.83731465773119829]], [[0.60904460269062644, 
-0.27504282756581611], [0.59442412655021282, 0.63394121077583687]], [[-0.093599642360094437, 0.28524722572151129], 
[0.57215715523487187, 0.32043802017988243]], [[-0.46978319490960008, 0.8320899984237411], [-0.81959981195018106, 
0.3503904055563718]], [[-0.67782474338099807, 0.6277948712480681], [0.5904568698600825, 
0.19793043441246461]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.72594789754747269, 0.87349402665851694], [-0.39536817653526857, 
-0.43834992550681506]], [[-0.066702830682781489, 0.90125757338912393], [0.24153649435848101, -0.61076430522901282]], 
[[-0.44706750826588104, -0.32475521625981529], [-0.75805998693509413, 0.5609814816056331]], [[-0.048254096644559263, 
-0.1019965676559671], [0.94934345435851686, 0.49525773976526155]], [[-0.17137160131178897, -0.59131541634931173], 
[0.3355019114914144, 0.81136679086153363]], [[0.4361355584873543, -0.37189941405839422], [-0.97448706411876906, 
0.13546560223519699]]]))
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.70447088573450056, -0.71278717909498845], [-0.86132753808998697, 0.76366555387101842]], 
[[-0.46138212438459769, 0.58893441480122144], [0.63803303568095848, -0.83731465773119829]], [[0.60904460269062644, 
0.49777371182112007], [0.86080228247319424, 0.63394121077583687]], [[0.67542965261369536, 0.6121873787033163], 
[0.57215715523487187, 0.32043802017988243]], [[-0.46978319490960008, 0.8320899984237411], [-0.53251293900871088, 
0.99180907488808523]], [[-0.50592776611259538, 0.6277948712480681], [0.5904568698600825, 
0.19793043441246461]]])+(1.-msk_ref)*numpy.array([[[-0.42486946532406078, 0.87349402665851694], [-0.13379915635758977, 
-0.43834992550681506]], [[0.08619882870789497, 0.90125757338912393], [0.24153649435848101, -0.61076430522901282]], 
[[-0.44706750826588104, -0.32475521625981529], [-0.26808535349731311, 0.90186927101211256]], [[-0.048254096644559263, 
0.30520100226807645], [0.94934345435851686, 0.49525773976526155]], [[0.21951523032010689, -0.36215940449847883], 
[0.3355019114914144, 0.81136679086153363]], [[0.4361355584873543, -0.37189941405839422], [-0.748573617352565, 
0.13546560223519699]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_array_rank4(self):
      arg0=Data(0.140204506116,self.functionspace)
      arg0.setTaggedValue(1,0.776411556396)
      arg1=numpy.array([[[[-0.99730381064015416, -0.24646978869934588, -0.8483851020872577, -0.7636563070313116], 
[-0.37243951612773407, 0.94238036382120205, 0.10234422233575913, -0.63481799846807152], [-0.67656654171039943, 
0.61957842577897493, -0.75412115238345123, -0.14535307618694615]], [[-0.60794681550683238, -0.050666392748995426, 
0.52975855520730453, 0.56525643887581056], [-0.68144300499306509, -0.22034933599205075, -0.019087784600194002, 
-0.5766101077680903], [0.99924951249979999, 0.66069703297235249, -0.40435693431731012, 0.12691208330117165]]], 
[[[0.11258502364035783, -0.64800200326456747, 0.23414988213600085, -0.71024538014446348], [-0.02764389408377399, 
0.38832760452106951, -0.41253157132059348, 0.46027425110814213], [-0.014904176332623642, -0.87466358698261537, 
0.28557945253459471, 0.47967954202610219]], [[-0.68820256981476446, -0.2638698803868027, 0.66531286373165166, 
-0.4022134780781832], [0.52543063559744185, -0.3687195790800637, 0.089403779258439231, 0.42119045530723787], 
[-0.69780787995295634, -0.018028374192776475, 0.8458285634538667, 0.73504128402816327]]], [[[-0.55258302630417377, 
-0.98693739100258671, 0.92802184049228553, -0.67964316167207195], [0.11568531485433886, -0.03064436749490107, 
-0.62766216298735977, 0.71464348176870485], [-0.32604829252105594, 0.65983171661889051, -0.16689514632621183, 
0.83420411995502253]], [[0.99089692365004511, -0.7366512690612772, 0.80586227471226946, 0.35265029390335689], 
[0.47413984995904923, 0.96956365223556262, -0.89326421583911775, -0.044234821495733545], [0.76312554985775916, 
-0.48494934965578018, -0.76505524948669201, 0.10779366614869068]]]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.14020450611621582, 0.14020450611621582, 0.14020450611621582, 0.14020450611621582], 
[0.14020450611621582, 0.94238036382120205, 0.14020450611621582, 0.14020450611621582], [0.14020450611621582, 
0.61957842577897493, 0.14020450611621582, 0.14020450611621582]], [[0.14020450611621582, 0.14020450611621582, 
0.52975855520730453, 0.56525643887581056], [0.14020450611621582, 0.14020450611621582, 0.14020450611621582, 
0.14020450611621582], [0.99924951249979999, 0.66069703297235249, 0.14020450611621582, 0.14020450611621582]]], 
[[[0.14020450611621582, 0.14020450611621582, 0.23414988213600085, 0.14020450611621582], [0.14020450611621582, 
0.38832760452106951, 0.14020450611621582, 0.46027425110814213], [0.14020450611621582, 0.14020450611621582, 0.28557945253459471, 
0.47967954202610219]], [[0.14020450611621582, 0.14020450611621582, 0.66531286373165166, 0.14020450611621582], 
[0.52543063559744185, 0.14020450611621582, 0.14020450611621582, 0.42119045530723787], [0.14020450611621582, 
0.14020450611621582, 0.8458285634538667, 0.73504128402816327]]], [[[0.14020450611621582, 0.14020450611621582, 
0.92802184049228553, 0.14020450611621582], [0.14020450611621582, 0.14020450611621582, 0.14020450611621582, 
0.71464348176870485], [0.14020450611621582, 0.65983171661889051, 0.14020450611621582, 0.83420411995502253]], 
[[0.99089692365004511, 0.14020450611621582, 0.80586227471226946, 0.35265029390335689], [0.47413984995904923, 
0.96956365223556262, 0.14020450611621582, 0.14020450611621582], [0.76312554985775916, 0.14020450611621582, 0.14020450611621582, 
0.14020450611621582]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 0.77641155639600079], 
[0.77641155639600079, 0.94238036382120205, 0.77641155639600079, 0.77641155639600079], [0.77641155639600079, 
0.77641155639600079, 0.77641155639600079, 0.77641155639600079]], [[0.77641155639600079, 0.77641155639600079, 
0.77641155639600079, 0.77641155639600079], [0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 
0.77641155639600079], [0.99924951249979999, 0.77641155639600079, 0.77641155639600079, 0.77641155639600079]]], 
[[[0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 0.77641155639600079], [0.77641155639600079, 
0.77641155639600079, 0.77641155639600079, 0.77641155639600079], [0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 
0.77641155639600079]], [[0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 0.77641155639600079], 
[0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 0.77641155639600079], [0.77641155639600079, 
0.77641155639600079, 0.8458285634538667, 0.77641155639600079]]], [[[0.77641155639600079, 0.77641155639600079, 
0.92802184049228553, 0.77641155639600079], [0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 
0.77641155639600079], [0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 0.83420411995502253]], 
[[0.99089692365004511, 0.77641155639600079, 0.80586227471226946, 0.77641155639600079], [0.77641155639600079, 
0.96956365223556262, 0.77641155639600079, 0.77641155639600079], [0.77641155639600079, 0.77641155639600079, 0.77641155639600079, 
0.77641155639600079]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_array_rank4(self):
      arg0=Data(numpy.array([[[[0.89689625041254883, 0.52797242797125987, 0.012935829969779045, 0.49739496562383811], 
[0.34654145869541342, 0.4942154511824135, 0.94728577054973773, -0.10647593987831683], [0.42098816875060185, 
-0.60112678726905733, 0.74841354845432084, 0.44557512382722142]], [[-0.61998183370410276, 0.85144500853434146, 
-0.28365889739397487, 0.57815050434017645], [-0.073398626336939499, -0.035793593325857964, -0.93049258850187866, 
-0.69393875636990177], [0.32501660176603653, -0.87489337072275752, -0.46854428856990138, 0.86399974360707255]]], 
[[[-0.92438693157358687, -0.75297959172395323, -0.33823121248451793, 0.014123356193518788], [0.98598606579865589, 
0.54835313244053441, 0.80251720308499097, 0.98875411533401869], [-0.17381509958185415, 0.44756510667464799, 
-0.95865420581901217, -0.63813924760819751]], [[-0.51554994901347917, 0.42325971045002753, -0.19806502163747419, 
0.74110302895556757], [0.88399089280465937, -0.30018223381520892, -0.4050697214624428, 0.2465561846189932], 
[-0.98658498763042668, -0.92954507208238568, 0.34050996236947095, -0.30283411802274318]]], [[[-0.40279303616480244, 
0.36341107069788103, 0.3228965744327823, 0.86919171803320583], [0.10338485362839478, -0.78546891952979836, 
-0.87519266310857957, -0.32557356812767146], [0.30051447520272312, -0.54311085576072449, -0.014839064605417507, 
-0.078365146479810965]], [[-0.040636503187079898, -0.70404312747433018, 0.86303384584731657, -0.25462623792025108], 
[-0.18808201385852819, -0.44580495135621301, 0.34898203854670484, 0.78465268140519728], [-0.0025280295686012089, 
0.36433072033888747, -0.64542201771473051, 0.66645257859304641]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.85559785511179776, -0.42936125949288684, -0.33648916313606492, 
-0.77929994043028983], [0.020366478246076181, -0.8363148142078094, 0.7529516721432552, 0.6965240356486615], 
[-0.89540620718949748, 0.62380401139707753, -0.69521784993527147, -0.45892632490273066]], [[0.69034651039417039, 
-0.33652641970660402, -0.9789857874991823, -0.049177230513813619], [-0.74603529283905035, -0.32823769853088014, 
-0.45275283441605074, -0.26007166682012217], [0.31275465231469335, 0.49437754374087128, 0.34806959467048393, 
-0.31195372954442813]]], [[[0.2271524360393844, 0.30619559786492001, 0.87568312184360875, 0.8032341525540927], 
[-0.83008411611779698, -0.57617978140716897, -0.77894079072162548, 0.25906244606102025], [0.68367611193427469, 
0.75744030915131488, 0.83491116847936997, 0.54938827549770952]], [[0.79484821673100248, 0.70712266760039011, 
0.89243778698890619, -0.573749082381108], [0.54924472046747308, -0.0095753891050591733, 0.74675761108210903, 
-0.08251004626096603], [0.068950333437353351, -0.76377814757397666, -0.1142338675364134, -0.030399841045661402]]], 
[[[0.2186100506517219, -0.055932476862738278, 0.12997017593997096, 0.13555431069824042], [0.82266213810484645, 
-0.57343346541492091, 0.97736407173466588, 0.98176763854796856], [0.60260684810778309, 0.996967155568111, -0.15882057312466502, 
-0.792335270074354]], [[0.96602989206010337, -0.5106269598547557, 0.91047056823779182, 0.3353665072357912], 
[-0.0026088601145151102, -0.58109645451299929, -0.031488711946613401, -0.76919028629929831], [-0.55700524280657149, 
0.066918152406544751, -0.13042055185497259, -0.30333596282980113]]]]))
      arg1=numpy.array([[[[0.69669428736757877, 0.7573662080876189, -0.28698915009281389, -0.26263218314437808], 
[-0.15335012146772953, -0.62535631572943706, -0.53623194531254237, -0.50555271878711228], [0.55592278359816816, 
0.28333179997930236, 0.65537435793536547, -0.31404112777228099]], [[0.56940127981552546, 0.71165007047354578, 
0.4325715566033419, -0.42514367104530604], [-0.27930914984964161, 0.5034942187155893, -0.25361414135973326, 
0.35097037311346213], [0.45914696727200144, 0.46275307374310293, -0.4998987424366792, 0.84722848964391684]]], 
[[[0.46778209786126901, -0.11143967069851946, -0.41618899067705462, -0.36463067467562715], [0.76855607801782044, 
0.36795344917547901, -0.024865778266954131, 0.22488717628373256], [-0.058634999803149901, -0.65314137012873452, 
0.21396523701043657, -0.87909034263150931]], [[0.64964244910871694, 0.0074080822454571926, -0.77322071351678745, 
-0.9513615698801865], [0.88590983808550816, -0.96921308694915442, -0.86220659803169108, 0.97756109908032229], 
[-0.34837342682409878, 0.16535564442036033, 0.81256597584619628, -0.83271776398797193]]], [[[0.15476061942959141, 
0.05462973166284546, 0.75392817594319128, -0.46973878944029845], [-0.76850410382242185, 0.86756778812159374, 
-0.77845519937521712, -0.47092759508271609], [-0.23798857255206429, -0.84954729015921648, 0.18724388871121578, 
0.94132468469936303]], [[-0.43665169072688226, -0.01535133018101309, 0.55649733815446201, 0.34191191590172498], 
[0.48863937044966188, 0.81250399692378195, 0.80263147202960305, 0.51097322438609094], [0.93752210265262925, 
-0.12745960614585439, 0.54530101717511803, -0.47122533892842222]]]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.89689625041254883, 0.7573662080876189, 0.012935829969779045, 0.49739496562383811], 
[0.34654145869541342, 0.4942154511824135, 0.94728577054973773, -0.10647593987831683], [0.55592278359816816, 
0.28333179997930236, 0.74841354845432084, 0.44557512382722142]], [[0.56940127981552546, 0.85144500853434146, 
0.4325715566033419, 0.57815050434017645], [-0.073398626336939499, 0.5034942187155893, -0.25361414135973326, 
0.35097037311346213], [0.45914696727200144, 0.46275307374310293, -0.46854428856990138, 0.86399974360707255]]], 
[[[0.46778209786126901, -0.11143967069851946, -0.33823121248451793, 0.014123356193518788], [0.98598606579865589, 
0.54835313244053441, 0.80251720308499097, 0.98875411533401869], [-0.058634999803149901, 0.44756510667464799, 
0.21396523701043657, -0.63813924760819751]], [[0.64964244910871694, 0.42325971045002753, -0.19806502163747419, 
0.74110302895556757], [0.88590983808550816, -0.30018223381520892, -0.4050697214624428, 0.97756109908032229], 
[-0.34837342682409878, 0.16535564442036033, 0.81256597584619628, -0.30283411802274318]]], [[[0.15476061942959141, 
0.36341107069788103, 0.75392817594319128, 0.86919171803320583], [0.10338485362839478, 0.86756778812159374, 
-0.77845519937521712, -0.32557356812767146], [0.30051447520272312, -0.54311085576072449, 0.18724388871121578, 
0.94132468469936303]], [[-0.040636503187079898, -0.01535133018101309, 0.86303384584731657, 0.34191191590172498], 
[0.48863937044966188, 0.81250399692378195, 0.80263147202960305, 0.78465268140519728], [0.93752210265262925, 
0.36433072033888747, 0.54530101717511803, 0.66645257859304641]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.69669428736757877, 0.7573662080876189, -0.28698915009281389, 
-0.26263218314437808], [0.020366478246076181, -0.62535631572943706, 0.7529516721432552, 0.6965240356486615], 
[0.55592278359816816, 0.62380401139707753, 0.65537435793536547, -0.31404112777228099]], [[0.69034651039417039, 
0.71165007047354578, 0.4325715566033419, -0.049177230513813619], [-0.27930914984964161, 0.5034942187155893, 
-0.25361414135973326, 0.35097037311346213], [0.45914696727200144, 0.49437754374087128, 0.34806959467048393, 
0.84722848964391684]]], [[[0.46778209786126901, 0.30619559786492001, 0.87568312184360875, 0.8032341525540927], 
[0.76855607801782044, 0.36795344917547901, -0.024865778266954131, 0.25906244606102025], [0.68367611193427469, 
0.75744030915131488, 0.83491116847936997, 0.54938827549770952]], [[0.79484821673100248, 0.70712266760039011, 
0.89243778698890619, -0.573749082381108], [0.88590983808550816, -0.0095753891050591733, 0.74675761108210903, 
0.97756109908032229], [0.068950333437353351, 0.16535564442036033, 0.81256597584619628, -0.030399841045661402]]], 
[[[0.2186100506517219, 0.05462973166284546, 0.75392817594319128, 0.13555431069824042], [0.82266213810484645, 
0.86756778812159374, 0.97736407173466588, 0.98176763854796856], [0.60260684810778309, 0.996967155568111, 0.18724388871121578, 
0.94132468469936303]], [[0.96602989206010337, -0.01535133018101309, 0.91047056823779182, 0.34191191590172498], 
[0.48863937044966188, 0.81250399692378195, 0.80263147202960305, 0.51097322438609094], [0.93752210265262925, 
0.066918152406544751, 0.54530101717511803, -0.30333596282980113]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_constData_rank4(self):
      arg0=Data(0.175075982346,self.functionspace)
      arg0.setTaggedValue(1,-0.753273109847)
      arg1=Data(numpy.array([[[[-0.90520491931836822, -0.93468993950409329, -0.69817984330991467, -0.19472712451275731], 
[0.024217147917435344, -0.051917085489648462, 0.95832186406934072, -0.15829046726386564], [-0.48306067231914596, 
-0.41568967685574232, 0.14434666355953474, -0.77262231860205799]], [[0.68747264268070341, 0.078098731827530354, 
-0.13676776472237995, 0.39580091142465523], [0.085784643596960031, 0.51757857880302582, -0.22620668285994228, 
-0.97695853326632109], [0.33320282360444509, -0.46906331447209726, 0.84298134633045896, 0.65728527461243669]]], 
[[[0.22868537840705616, -0.42917598979637384, 0.98021942369056747, 0.036486378920451923], [0.33526830672431229, 
0.29845278576530587, 0.73318498063457294, 0.96455250980131879], [0.70157227491685203, -0.11966179656269249, 
-0.35144444326452851, 0.92978237965210186]], [[0.49662231192561057, -0.44191431253928704, 0.93392477626295167, 
-0.049858840167848584], [-0.92454951849267708, 0.35236313087098647, 0.060998074836456917, 0.57844628283558808], 
[0.76405432987726818, 0.67022116939456367, -0.41218118864500397, -0.069592294330405702]]], [[[-0.9489924452956886, 
0.0023052791008568718, 0.91174990075813356, -0.045011693952122345], [-0.98325017436556439, -0.27232336410089664, 
0.56200175967572896, 0.071945996926542355], [-0.86371744481279089, 0.49082955373961301, -0.3804849735134217, 
-0.6430043583685896]], [[0.654115340545951, -0.38797201046510255, -0.88752480607181661, 0.28133634556572007], 
[-0.66333324462780863, -0.98636701436618246, -0.39400019159584487, 0.031940367391527724], [0.29437766730392068, 
-0.14066757728058876, 0.7102223684414517, 0.51992231433698177]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.17507598234624222, 0.17507598234624222, 0.17507598234624222, 0.17507598234624222], 
[0.17507598234624222, 0.17507598234624222, 0.95832186406934072, 0.17507598234624222], [0.17507598234624222, 
0.17507598234624222, 0.17507598234624222, 0.17507598234624222]], [[0.68747264268070341, 0.17507598234624222, 
0.17507598234624222, 0.39580091142465523], [0.17507598234624222, 0.51757857880302582, 0.17507598234624222, 
0.17507598234624222], [0.33320282360444509, 0.17507598234624222, 0.84298134633045896, 0.65728527461243669]]], 
[[[0.22868537840705616, 0.17507598234624222, 0.98021942369056747, 0.17507598234624222], [0.33526830672431229, 
0.29845278576530587, 0.73318498063457294, 0.96455250980131879], [0.70157227491685203, 0.17507598234624222, 0.17507598234624222, 
0.92978237965210186]], [[0.49662231192561057, 0.17507598234624222, 0.93392477626295167, 0.17507598234624222], 
[0.17507598234624222, 0.35236313087098647, 0.17507598234624222, 0.57844628283558808], [0.76405432987726818, 
0.67022116939456367, 0.17507598234624222, 0.17507598234624222]]], [[[0.17507598234624222, 0.17507598234624222, 
0.91174990075813356, 0.17507598234624222], [0.17507598234624222, 0.17507598234624222, 0.56200175967572896, 
0.17507598234624222], [0.17507598234624222, 0.49082955373961301, 0.17507598234624222, 0.17507598234624222]], 
[[0.654115340545951, 0.17507598234624222, 0.17507598234624222, 0.28133634556572007], [0.17507598234624222, 0.17507598234624222, 
0.17507598234624222, 0.17507598234624222], [0.29437766730392068, 0.17507598234624222, 0.7102223684414517, 
0.51992231433698177]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.75327310984690454, -0.75327310984690454, -0.69817984330991467, 
-0.19472712451275731], [0.024217147917435344, -0.051917085489648462, 0.95832186406934072, -0.15829046726386564], 
[-0.48306067231914596, -0.41568967685574232, 0.14434666355953474, -0.75327310984690454]], [[0.68747264268070341, 
0.078098731827530354, -0.13676776472237995, 0.39580091142465523], [0.085784643596960031, 0.51757857880302582, 
-0.22620668285994228, -0.75327310984690454], [0.33320282360444509, -0.46906331447209726, 0.84298134633045896, 
0.65728527461243669]]], [[[0.22868537840705616, -0.42917598979637384, 0.98021942369056747, 0.036486378920451923], 
[0.33526830672431229, 0.29845278576530587, 0.73318498063457294, 0.96455250980131879], [0.70157227491685203, 
-0.11966179656269249, -0.35144444326452851, 0.92978237965210186]], [[0.49662231192561057, -0.44191431253928704, 
0.93392477626295167, -0.049858840167848584], [-0.75327310984690454, 0.35236313087098647, 0.060998074836456917, 
0.57844628283558808], [0.76405432987726818, 0.67022116939456367, -0.41218118864500397, -0.069592294330405702]]], 
[[[-0.75327310984690454, 0.0023052791008568718, 0.91174990075813356, -0.045011693952122345], [-0.75327310984690454, 
-0.27232336410089664, 0.56200175967572896, 0.071945996926542355], [-0.75327310984690454, 0.49082955373961301, 
-0.3804849735134217, -0.6430043583685896]], [[0.654115340545951, -0.38797201046510255, -0.75327310984690454, 
0.28133634556572007], [-0.66333324462780863, -0.75327310984690454, -0.39400019159584487, 0.031940367391527724], 
[0.29437766730392068, -0.14066757728058876, 0.7102223684414517, 0.51992231433698177]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_constData_rank4(self):
      arg0=Data(numpy.array([[[[-0.83554041071651497, 0.71408854090245599, 0.60208475969377528, -0.48074567274281188], 
[-0.52759893410924819, -0.71615915803340324, 0.53333465163432692, -0.66988543484474605], [0.77637769090560016, 
-0.63913469863001549, 0.29301783500609346, -0.96046850219501256]], [[-0.2934774353287235, -0.83491727033450602, 
0.068674887022271092, 0.15208898687960493], [-0.15482824781008264, -0.33821742546013978, -0.72427728138855074, 
0.8070363392853257], [-0.29904185760156743, -0.2258288530271948, -0.037614687529824442, 0.83542534634450893]]], 
[[[0.19062537975270066, -0.23547948354521986, -0.41417602748868476, 0.72248428650742569], [-0.48296945282438974, 
0.42128745145501134, 0.41652452158546938, -0.49944580063054222], [0.92772652297265146, 0.7270672572706014, 
-0.81760023566538487, 0.50095443130798856]], [[-0.74454869833090132, -0.70129367956832489, 0.36418680176033158, 
0.38020322489950353], [0.20567532340417127, 0.15899857920753857, -0.41608677444765596, -0.20142186667572437], 
[-0.59223036711040766, -0.72216663548334203, -0.89066959971018722, -0.18384858463777798]]], [[[-0.86601762107365099, 
-0.39565676563683971, -0.3572559636912318, -0.75130471010020128], [0.61425518475424612, -0.047414537516563104, 
-0.46216310325753218, -0.79662509756346322], [-0.15391711914956874, 0.44577576348654691, -0.072098272539505626, 
-0.8810102747315931]], [[-0.18482310072006736, 0.53784496328328535, -0.24060782705170269, -0.13964203947011744], 
[0.4607198252775071, -0.28876214166544623, -0.64125936246661297, -0.45000998399530601], [-0.36469590037570154, 
-0.76346190967180227, 0.18143150646312955, 0.71074540892672355]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.38410988074270191, 0.48698261126925724, -0.71546872240537973, 
0.24481247675581641], [0.99885682895314476, -0.25511670532604369, -0.71070355861906731, -0.72696546965933506], 
[-0.94912342150579154, -0.46411231113604301, -0.41021969012408888, 0.28731035829800411]], [[0.025228750258471155, 
0.052193465179895204, 0.9667489767755737, -0.75023922525523989], [0.42541075459313893, -0.33499016293652595, 
0.53921113379042773, 0.56807827977061542], [-0.85656916434076646, -0.50708798843058855, 0.095729837523134798, 
-0.93853871926968524]]], [[[0.72256037457316435, 0.87951699212687973, -0.25700948888996167, -0.36895326307397425], 
[-0.10383347850044355, 0.055990596622860211, 0.8505383875796928, -0.060493735214111277], [-0.15474973198303821, 
0.084300406749669099, 0.18359703378401737, -0.39026465982377245]], [[-0.57679037452493054, -0.79939222661951659, 
0.96534755723804411, -0.5570215917480128], [0.90049139166436554, 0.97719408860871826, 0.50274901942217842, 
0.32543237814304438], [-0.49678670997716967, -0.564748676453384, 0.53223489483479858, 0.084495376513626264]]], 
[[[0.99677248621279202, -0.72739982709913842, -0.13880590557865724, 0.15133583376283433], [0.76244310510429059, 
-0.14755688136107215, 0.16518884190374328, 0.083060985420594369], [-0.038160642105724119, -0.24133253446363812, 
0.58857369505920176, -0.024863384107122855]], [[0.6765400209954735, -0.26957688268590263, 0.26961637512199643, 
-0.49644190164725588], [-0.74754865637610313, -0.27081415764153416, 0.34883570273480924, -0.14892014181377533], 
[0.39562098011519908, -0.55360659958531344, 0.42315840716945852, -0.92845920852039332]]]]))
      arg1=Data(numpy.array([[[[-0.96631869879503518, -0.28641755911727418, 0.12373979419442604, 0.48542674969519384], 
[-0.76157324731506382, 0.47654843860186769, -0.52233692676173993, -0.3924418091656916], [0.091788811985614283, 
-0.65599501387602555, -0.73156215199864261, -0.61436310586467435]], [[-0.090856928024656947, 0.94405650117949436, 
0.26960360201567446, 0.86164324014188276], [0.051142373301623412, -0.34635650648076655, -0.63840458037438319, 
-0.67326079913537829], [0.091684265168592072, 0.4926229620141751, -0.23501283462010925, -0.57864913724747113]]], 
[[[-0.91848196770480595, 0.47177138324142032, -0.20804568417267366, 0.74766249038054111], [0.13648841759432262, 
-0.86260941082421949, 0.83438299358276091, -0.44680910343388991], [0.48124568087048081, 0.49492688617119374, 
-0.87250408657471179, 0.42135484555644287]], [[-0.58724283171517877, -0.81287498457771457, -0.48121017671547528, 
-0.30112540340537608], [0.52688704971498135, -0.27495826164856152, 0.62176459312402876, -0.70535467900420312], 
[-0.93424398659195496, 0.43798922141764662, -0.14965859673809367, -0.97767246532594232]]], [[[-0.4630811897807674, 
-0.36376768653981673, -0.72017037184924604, -0.49842765407284229], [0.64937429811150449, -0.15823487622512489, 
0.65967401184976682, 0.41715402488099307], [-0.97143026036379743, -0.3672551538027351, -0.73381298765192149, 
0.72454560789134392]], [[-0.52279746407536987, 0.74263228500102474, -0.53788299341072765, -0.29063337196731642], 
[-0.30189250558381775, -0.30671529171740897, -0.99527483620813095, 0.22945727156053297], [0.66460628434796365, 
0.78890813192725351, 0.048234854061694143, 0.6439469411562937]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.83554041071651497, 0.71408854090245599, 0.60208475969377528, 0.48542674969519384], 
[-0.52759893410924819, 0.47654843860186769, 0.53333465163432692, -0.3924418091656916], [0.77637769090560016, 
-0.63913469863001549, 0.29301783500609346, -0.61436310586467435]], [[-0.090856928024656947, 0.94405650117949436, 
0.26960360201567446, 0.86164324014188276], [0.051142373301623412, -0.33821742546013978, -0.63840458037438319, 
0.8070363392853257], [0.091684265168592072, 0.4926229620141751, -0.037614687529824442, 0.83542534634450893]]], 
[[[0.19062537975270066, 0.47177138324142032, -0.20804568417267366, 0.74766249038054111], [0.13648841759432262, 
0.42128745145501134, 0.83438299358276091, -0.44680910343388991], [0.92772652297265146, 0.7270672572706014, 
-0.81760023566538487, 0.50095443130798856]], [[-0.58724283171517877, -0.70129367956832489, 0.36418680176033158, 
0.38020322489950353], [0.52688704971498135, 0.15899857920753857, 0.62176459312402876, -0.20142186667572437], 
[-0.59223036711040766, 0.43798922141764662, -0.14965859673809367, -0.18384858463777798]]], [[[-0.4630811897807674, 
-0.36376768653981673, -0.3572559636912318, -0.49842765407284229], [0.64937429811150449, -0.047414537516563104, 
0.65967401184976682, 0.41715402488099307], [-0.15391711914956874, 0.44577576348654691, -0.072098272539505626, 
0.72454560789134392]], [[-0.18482310072006736, 0.74263228500102474, -0.24060782705170269, -0.13964203947011744], 
[0.4607198252775071, -0.28876214166544623, -0.64125936246661297, 0.22945727156053297], [0.66460628434796365, 
0.78890813192725351, 0.18143150646312955, 0.71074540892672355]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.38410988074270191, 0.48698261126925724, 0.12373979419442604, 0.48542674969519384], 
[0.99885682895314476, 0.47654843860186769, -0.52233692676173993, -0.3924418091656916], [0.091788811985614283, 
-0.46411231113604301, -0.41021969012408888, 0.28731035829800411]], [[0.025228750258471155, 0.94405650117949436, 
0.9667489767755737, 0.86164324014188276], [0.42541075459313893, -0.33499016293652595, 0.53921113379042773, 
0.56807827977061542], [0.091684265168592072, 0.4926229620141751, 0.095729837523134798, -0.57864913724747113]]], 
[[[0.72256037457316435, 0.87951699212687973, -0.20804568417267366, 0.74766249038054111], [0.13648841759432262, 
0.055990596622860211, 0.8505383875796928, -0.060493735214111277], [0.48124568087048081, 0.49492688617119374, 
0.18359703378401737, 0.42135484555644287]], [[-0.57679037452493054, -0.79939222661951659, 0.96534755723804411, 
-0.30112540340537608], [0.90049139166436554, 0.97719408860871826, 0.62176459312402876, 0.32543237814304438], 
[-0.49678670997716967, 0.43798922141764662, 0.53223489483479858, 0.084495376513626264]]], [[[0.99677248621279202, 
-0.36376768653981673, -0.13880590557865724, 0.15133583376283433], [0.76244310510429059, -0.14755688136107215, 
0.65967401184976682, 0.41715402488099307], [-0.038160642105724119, -0.24133253446363812, 0.58857369505920176, 
0.72454560789134392]], [[0.6765400209954735, 0.74263228500102474, 0.26961637512199643, -0.29063337196731642], 
[-0.30189250558381775, -0.27081415764153416, 0.34883570273480924, 0.22945727156053297], [0.66460628434796365, 
0.78890813192725351, 0.42315840716945852, 0.6439469411562937]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_taggedData_rank4(self):
      arg0=Data(0.359848099791,self.functionspace)
      arg0.setTaggedValue(1,-0.32301879346)
      arg1=Data(numpy.array([[[[-0.16702432092235409, -0.06866876640451669, -0.92785635272040379, -0.92005334963800678], 
[-0.9738980183757846, -0.76396923416949791, -0.1456849962483111, 0.4545699448145355], [-0.75613105524163693, 
-0.48664003507384024, 0.17761051693218599, 0.88396540985438654]], [[-0.12740955092566053, -0.66428545503968239, 
0.92830953756719303, 0.90914274494830827], [-0.57723852475234083, 0.070878461299128448, -0.55777398440813886, 
-0.49042966779867481], [-0.19776456885718008, -0.39523659475973272, 0.65506918257425273, -0.7493940072567562]]], 
[[[0.72888194265778572, -0.041168654770973312, -0.059569991301786862, -0.07323948419301729], [0.82838219301280835, 
-0.060765027842354602, 0.17723089681084914, 0.66279184149014925], [-0.6249853567294843, -0.36892972584898809, 
-0.05891880437023822, 0.99230971020254266]], [[-0.91915653784996665, 0.44538349382517528, -0.098012471289684822, 
-0.39235609068909483], [-0.095784275082066328, 0.92889885409591244, -0.33940090860481908, -0.56520432599462844], 
[-0.84518516880775918, 0.2096567118449062, 0.73364142442547964, -0.80439567355077379]]], [[[0.98097911106432778, 
0.37425430053692543, -0.95421426335162574, -0.9711798879942215], [0.20793947160953397, 0.18980820452364755, 
0.62427193579423057, 0.97936933611501353], [-0.84744217037696457, -0.92800447234171024, -0.67435896077327473, 
-0.40465279589978764]], [[0.17651512016349158, -0.075802975431532138, 0.80479273628853232, 0.84252390054322657], 
[-0.29535938280495855, -0.70019289558460041, 0.50743658132201852, -0.23494470011704305], [0.89991130167365241, 
0.37038846620431487, -0.020752227049099403, 0.10638454882043757]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.97578498738753083, -0.78566072302491374, 0.78180749249393844, 
-0.59478729639483663], [0.848606560968721, 0.58370925867603796, 0.63526222841440694, -0.099861890757521055], 
[0.28848242860671069, 0.13789155213119297, 0.61435463081039376, -0.58646995769074328]], [[-0.37096038653019647, 
0.69581697388381025, -0.88679025736391726, 0.7749694667466025], [0.63817615633948832, -0.72722528507215278, 
-0.4425160299955575, 0.66445379947171856], [-0.93345379909845216, 0.67998510495666631, 0.53825423228386082, 
0.50356993307639808]]], [[[-0.96614128056737081, -0.35729090986965129, 0.6037760685289093, 0.10230903877022546], 
[-0.2386724832095497, 0.19052031803304703, -0.88325401184715369, 0.22532508246496374], [0.3318869974006442, 
0.19915372276857712, -0.95932746578114902, 0.79477805730513129]], [[0.9443821536396948, 0.73570756597820552, 
-0.2792714255865858, 0.83906303842776286], [0.97073740444357415, -0.41792084568097976, -0.28021605438811803, 
0.6832261432593556], [0.99821460704228837, -0.81852541485186525, -0.24587890850209848, -0.95760062960296066]]], 
[[[0.47497950241118625, 0.6268272063294047, -0.36180249908656426, 0.19800693306026362], [0.59402477011371491, 
0.72905899355312687, -0.50047002505571792, -0.2858694836640483], [0.92021086832490906, 0.24454277677303526, 
0.40377936020168037, 0.15756138553056842]], [[0.37256951712130082, -0.85651180048010866, -0.62348844997233877, 
0.94297969025886741], [-0.28818150118089991, 0.86231107270803986, -0.51024752748623681, 0.6549079906323394], 
[-0.88536217536802075, 0.098109147666445073, 0.51905538404276341, 0.25916674167511156]]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.35984809979125787, 0.35984809979125787, 0.35984809979125787, 0.35984809979125787], 
[0.35984809979125787, 0.35984809979125787, 0.35984809979125787, 0.4545699448145355], [0.35984809979125787, 0.35984809979125787, 
0.35984809979125787, 0.88396540985438654]], [[0.35984809979125787, 0.35984809979125787, 0.92830953756719303, 
0.90914274494830827], [0.35984809979125787, 0.35984809979125787, 0.35984809979125787, 0.35984809979125787], 
[0.35984809979125787, 0.35984809979125787, 0.65506918257425273, 0.35984809979125787]]], [[[0.72888194265778572, 
0.35984809979125787, 0.35984809979125787, 0.35984809979125787], [0.82838219301280835, 0.35984809979125787, 0.35984809979125787, 
0.66279184149014925], [0.35984809979125787, 0.35984809979125787, 0.35984809979125787, 0.99230971020254266]], 
[[0.35984809979125787, 0.44538349382517528, 0.35984809979125787, 0.35984809979125787], [0.35984809979125787, 
0.92889885409591244, 0.35984809979125787, 0.35984809979125787], [0.35984809979125787, 0.35984809979125787, 0.73364142442547964, 
0.35984809979125787]]], [[[0.98097911106432778, 0.37425430053692543, 0.35984809979125787, 0.35984809979125787], 
[0.35984809979125787, 0.35984809979125787, 0.62427193579423057, 0.97936933611501353], [0.35984809979125787, 
0.35984809979125787, 0.35984809979125787, 0.35984809979125787]], [[0.35984809979125787, 0.35984809979125787, 
0.80479273628853232, 0.84252390054322657], [0.35984809979125787, 0.35984809979125787, 0.50743658132201852, 
0.35984809979125787], [0.89991130167365241, 0.37038846620431487, 0.35984809979125787, 
0.35984809979125787]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.97578498738753083, -0.32301879346006701, 0.78180749249393844, 
-0.32301879346006701], [0.848606560968721, 0.58370925867603796, 0.63526222841440694, -0.099861890757521055], 
[0.28848242860671069, 0.13789155213119297, 0.61435463081039376, -0.32301879346006701]], [[-0.32301879346006701, 
0.69581697388381025, -0.32301879346006701, 0.7749694667466025], [0.63817615633948832, -0.32301879346006701, 
-0.32301879346006701, 0.66445379947171856], [-0.32301879346006701, 0.67998510495666631, 0.53825423228386082, 
0.50356993307639808]]], [[[-0.32301879346006701, -0.32301879346006701, 0.6037760685289093, 0.10230903877022546], 
[-0.2386724832095497, 0.19052031803304703, -0.32301879346006701, 0.22532508246496374], [0.3318869974006442, 
0.19915372276857712, -0.32301879346006701, 0.79477805730513129]], [[0.9443821536396948, 0.73570756597820552, 
-0.2792714255865858, 0.83906303842776286], [0.97073740444357415, -0.32301879346006701, -0.28021605438811803, 
0.6832261432593556], [0.99821460704228837, -0.32301879346006701, -0.24587890850209848, -0.32301879346006701]]], 
[[[0.47497950241118625, 0.6268272063294047, -0.32301879346006701, 0.19800693306026362], [0.59402477011371491, 
0.72905899355312687, -0.32301879346006701, -0.2858694836640483], [0.92021086832490906, 0.24454277677303526, 
0.40377936020168037, 0.15756138553056842]], [[0.37256951712130082, -0.32301879346006701, -0.32301879346006701, 
0.94297969025886741], [-0.28818150118089991, 0.86231107270803986, -0.32301879346006701, 0.6549079906323394], 
[-0.32301879346006701, 0.098109147666445073, 0.51905538404276341, 0.25916674167511156]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_taggedData_rank4(self):
      arg0=Data(numpy.array([[[[0.26563127580812163, -0.34457797630255338, 0.2346903853285327, -0.8454883899910306], 
[0.2163845841669394, 0.053180577149721575, 0.31193541105564782, -0.50332171144939508], [-0.63201079937206139, 
-0.21150738299206129, 0.36186929378192878, 0.67650322506866201]], [[-0.54480747048081724, 0.12266763421497173, 
0.80210944561378761, 0.0084698895634938154], [-0.055181153809809924, 0.8900392186506163, 0.0069214374987596194, 
-0.70161401569131709], [0.62173917883517626, 0.87861441240175409, -0.064371810173868838, 0.5082145071910984]]], 
[[[-0.63133302440615457, -0.047684747320537957, 0.12413281764706618, -0.98961720351875915], [0.1866616419093623, 
0.043198711151814839, 0.62670028456093441, 0.0016937528271721014], [-0.99986630288429335, -0.41687463572411132, 
0.73185666060689925, 0.92659394364832615]], [[-0.22157046318387597, -0.020675529999787612, -0.35218831231208192, 
0.63483071623135645], [0.94607227720114495, 0.62615232251281494, -0.044533114877439273, -0.37003516015413451], 
[0.82063080133814892, 0.54096160623847278, -0.22246914607905643, -0.83172892127578213]]], [[[0.86754739333404673, 
0.30595625521104064, -0.70612129879028118, 0.15718748076070055], [-0.71268367667725374, -0.29730161390926568, 
-0.32629361912768418, -0.7579722067667729], [0.41934454460428694, -0.95593762763858381, -0.23225038340368021, 
0.37944713138969899]], [[0.77047144164271053, -0.50683589201042145, 0.21800816850393812, 0.21164251772657794], 
[0.25414763143208341, 0.80815173698637732, -0.46136883329991663, 0.68175673965981565], [0.6893702060110618, 
-0.16899889812522995, 0.68975555805828126, -0.1530772898537438]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.75401054459557981, 0.22974491712268508, -0.9122286508760824, 
-0.71704655166790099], [-0.25772543935471504, 0.64733600330531482, -0.13380189816965116, 0.6837271101692175], 
[0.55016408391760385, 0.59860890181370552, 0.1678663011437469, 0.84544454148220383]], [[-0.5390322424098386, 
0.38471591463575439, 0.14116680406396132, -0.71500462019902677], [-0.94603615037127642, 0.11116321983180111, 
-0.83037248593137369, -0.49206844633311175], [0.72800264379158741, 0.57317609607871511, 0.81428936715884426, 
-0.6451313611386611]]], [[[-0.1054601921648366, -0.71417564446292769, -0.63899885864164752, -0.78045985178766419], 
[-0.55826576249476534, 0.34383219474527271, -0.66865907842784855, -0.55330184587791731], [-0.10602478556251405, 
-0.52284096714332962, -0.42695030789151867, -0.23131983333227413]], [[0.28730778310526506, -0.085300698105907014, 
0.5669155701973807, -0.20487499776860418], [-0.020350594546123979, 0.49563110676180955, 0.66872971083776456, 
0.58659063108834153], [-0.40323833026885181, -0.13804956957639525, -0.13780754419072094, -0.28226957781803019]]], 
[[[-0.60381667373682957, -0.058075783190817187, -0.10532115562777933, -0.26715167856932887], [-0.19840431185134255, 
0.69200725798237595, 0.38388056071569476, -0.64213175790884369], [-0.61074301744663906, 0.68826224833107874, 
0.25197297202419233, -0.27118324277121952]], [[0.71196898811688136, 0.75110748472376376, -0.22232922168177227, 
-0.95967398083135169], [-0.78464593947887984, 0.75900640150763565, 0.83029869697090319, 0.29517865933303722], 
[0.29715263603632724, 0.67665285355536331, 0.33062147124950947, 0.91780337254142141]]]]))
      arg1=Data(numpy.array([[[[0.10834925808760554, 0.55593666772031813, -0.22422168329127934, 0.63184497781252724], 
[0.43452302333321979, 0.47315661964460642, -0.61443737662358711, -0.81886754397548867], [0.83411846958641367, 
0.4207149140946822, 0.41805764797970157, -0.36104035745344842]], [[-0.63615594549860166, -0.052706169424120652, 
-0.15506592979579281, -0.046722162261591871], [0.36734157985950233, -0.43910412475291039, -0.061632693839534403, 
0.18387311643630877], [0.45207152039801857, -0.53174209014251117, -0.56930417812185241, 0.069623692061308251]]], 
[[[-0.80553258796465199, -0.9476361259810826, -0.67678809193888267, 0.50047211585197537], [0.57132412591369652, 
-0.89036491249248639, 0.1463541471029719, 0.1310204313578176], [-0.52678159360532018, 0.86253046002929401, 
-0.13259801867844456, -0.73430184522051367]], [[-0.58156092425108663, -0.77775450829480652, 0.0078060289706782804, 
0.28714248943423071], [0.98988944133317314, 0.54181301559287998, -0.61462537129798167, 0.14488621669222557], 
[0.87738560858575676, -0.014532853678190349, 0.17835610721597273, 0.13696312060367277]]], [[[0.47219801825391206, 
0.27692443894787355, 0.82782943299942358, -0.045268759625685995], [0.74104720022727033, 0.60944473904656138, 
0.96323454865477021, -0.37858728270454667], [0.066889023852941376, 0.79432558840660694, -0.99667963232663292, 
-0.94808878863656876]], [[-0.98117591436714591, -0.40309070095698152, 0.9066629270881108, 0.20871508818579465], 
[0.68208565955272249, 0.66093198766169658, -0.37112210438141235, -0.4100888357839263], [-0.51930559641087237, 
0.29025264374472637, -0.20291928312909602, 0.78965727017388976]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.7535796556325729, 0.16409966774352025, -0.88163609330597614, 
0.56611570203052342], [0.99867923676244463, -0.23230312747792015, 0.62170084861892039, -0.22474648980293477], 
[-0.33076740528642024, -0.96091332651139116, 0.49387004370177379, 0.087929575219180212]], [[0.056606714059171725, 
-0.46722548152987597, -0.4149824551994532, -0.52836187405201196], [-0.057637558987633009, 0.32534381846242311, 
-0.95764919152402617, 0.50425020136131793], [0.41881030044180179, -0.55193233174518119, -0.078461126391198288, 
0.64283212596462569]]], [[[-0.31961433638862391, 0.48336634285543356, 0.59868359034545504, 0.77807690526146023], 
[0.65047394099235656, 0.32490505228075128, 0.42586694990221696, -0.20898406086196863], [-0.78063775101477328, 
-0.60017480640193521, -0.19528339985812826, -0.4290661099696067]], [[-0.6281113768473745, 0.88512376401871418, 
0.51250293267364899, -0.12808536932751546], [-0.68566879547771453, 0.15835536317350218, 0.56336708108297739, 
0.97844620592141629], [0.85685030465286061, -0.2685316814753469, 0.42370516851863238, -0.10127727176306944]]], 
[[[-0.15757344636460613, -0.52528087961252456, 0.13534155518702806, 0.045330858581224653], [0.53022236877841289, 
-0.47026331333427995, -0.3252704688762782, -0.068276962336971003], [-0.58499778641657674, -0.23213972270253658, 
0.76509335187050698, -0.80540742615170369]], [[-0.47529529033301121, 0.68570197963703494, -0.57953510489505522, 
-0.21311529731625289], [0.411965909939493, 0.90315382735644922, -0.82763488040329602, 0.35884447681529119], 
[-0.65609615546301359, 0.11773605632689677, 0.39005319527180782, 0.0077075450799588552]]]]))
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.26563127580812163, 0.55593666772031813, 0.2346903853285327, 0.63184497781252724], 
[0.43452302333321979, 0.47315661964460642, 0.31193541105564782, -0.50332171144939508], [0.83411846958641367, 
0.4207149140946822, 0.41805764797970157, 0.67650322506866201]], [[-0.54480747048081724, 0.12266763421497173, 
0.80210944561378761, 0.0084698895634938154], [0.36734157985950233, 0.8900392186506163, 0.0069214374987596194, 
0.18387311643630877], [0.62173917883517626, 0.87861441240175409, -0.064371810173868838, 0.5082145071910984]]], 
[[[-0.63133302440615457, -0.047684747320537957, 0.12413281764706618, 0.50047211585197537], [0.57132412591369652, 
0.043198711151814839, 0.62670028456093441, 0.1310204313578176], [-0.52678159360532018, 0.86253046002929401, 
0.73185666060689925, 0.92659394364832615]], [[-0.22157046318387597, -0.020675529999787612, 0.0078060289706782804, 
0.63483071623135645], [0.98988944133317314, 0.62615232251281494, -0.044533114877439273, 0.14488621669222557], 
[0.87738560858575676, 0.54096160623847278, 0.17835610721597273, 0.13696312060367277]]], [[[0.86754739333404673, 
0.30595625521104064, 0.82782943299942358, 0.15718748076070055], [0.74104720022727033, 0.60944473904656138, 0.96323454865477021, 
-0.37858728270454667], [0.41934454460428694, 0.79432558840660694, -0.23225038340368021, 0.37944713138969899]], 
[[0.77047144164271053, -0.40309070095698152, 0.9066629270881108, 0.21164251772657794], [0.68208565955272249, 
0.80815173698637732, -0.37112210438141235, 0.68175673965981565], [0.6893702060110618, 0.29025264374472637, 0.68975555805828126, 
0.78965727017388976]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.75401054459557981, 0.22974491712268508, -0.88163609330597614, 
0.56611570203052342], [0.99867923676244463, 0.64733600330531482, 0.62170084861892039, 0.6837271101692175], 
[0.55016408391760385, 0.59860890181370552, 0.49387004370177379, 0.84544454148220383]], [[0.056606714059171725, 
0.38471591463575439, 0.14116680406396132, -0.52836187405201196], [-0.057637558987633009, 0.32534381846242311, 
-0.83037248593137369, 0.50425020136131793], [0.72800264379158741, 0.57317609607871511, 0.81428936715884426, 
0.64283212596462569]]], [[[-0.1054601921648366, 0.48336634285543356, 0.59868359034545504, 0.77807690526146023], 
[0.65047394099235656, 0.34383219474527271, 0.42586694990221696, -0.20898406086196863], [-0.10602478556251405, 
-0.52284096714332962, -0.19528339985812826, -0.23131983333227413]], [[0.28730778310526506, 0.88512376401871418, 
0.5669155701973807, -0.12808536932751546], [-0.020350594546123979, 0.49563110676180955, 0.66872971083776456, 
0.97844620592141629], [0.85685030465286061, -0.13804956957639525, 0.42370516851863238, -0.10127727176306944]]], 
[[[-0.15757344636460613, -0.058075783190817187, 0.13534155518702806, 0.045330858581224653], [0.53022236877841289, 
0.69200725798237595, 0.38388056071569476, -0.068276962336971003], [-0.58499778641657674, 0.68826224833107874, 
0.76509335187050698, -0.27118324277121952]], [[0.71196898811688136, 0.75110748472376376, -0.22232922168177227, 
-0.21311529731625289], [0.411965909939493, 0.90315382735644922, 0.83029869697090319, 0.35884447681529119], 
[0.29715263603632724, 0.67665285355536331, 0.39005319527180782, 0.91780337254142141]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank0_expandedData_rank4(self):
      arg0=Data(0.469968604774,self.functionspace)
      arg0.setTaggedValue(1,-0.841933836654)
      arg1=Data(numpy.array([[[[0.96799917063261409, -0.86661811201272831, 0.47201141981822858, 0.049952270268001753], 
[-0.22338695322375646, 0.96872332483282375, 0.19877209616296909, 0.49772168968845887], [-0.73591764400471771, 
0.62012726879516444, -0.5521797985767154, 0.72050763180743038]], [[-0.39231636254539071, -0.074539493325833428, 
0.92293794375703264, -0.20157806333761341], [-0.11248886972963335, 0.97843635480846136, -0.14603677189557196, 
-0.42671608085961998], [-0.49876635566180272, -0.43130758530937729, -0.4584513671641568, 0.9525109700965102]]], 
[[[0.52539281253864512, 0.76087490949891379, -0.49103094194895758, -0.13973163539196176], [-0.12037225493441173, 
0.47148631628813731, -0.37094299665033681, -0.19863435762328141], [-0.29172470012921492, 0.40403163351028182, 
-0.49021494890655504, 0.3017376697760985]], [[0.37259332003286083, -0.32301598375868767, 0.21334657588342654, 
0.91568668490748339], [0.38348229129047495, -0.40656229355948459, -0.20679538025187538, 0.4883167426693038], 
[0.74528188999593414, 0.12326480339528434, 0.19371601127829052, 0.20745454690184806]]], [[[-0.18933480162300764, 
0.65853853119832761, -0.56680098625762776, -0.96099802784436772], [-0.84061607066227451, 0.62349684395724125, 
0.52607653495550122, -0.90964589662813178], [0.2637452143217951, 0.39860784868507193, -0.15461854461685509, 
-0.80144007937114847]], [[0.43037498830435017, -0.20517355213181587, -0.40137631349231695, 0.35128258168015569], 
[-0.91803534480975713, 0.78874943586980617, 0.6970466108744684, -0.51093512430996157], [0.2678122718491025, 
0.84711792647927608, -0.087433537131151207, -0.42251221731093414]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.79902570155778041, -0.96231459941099207, -0.23753030474741288, 
0.32015814046069635], [-0.69519556701166341, 0.56633323595909246, 0.75751301528775272, -0.06669352527047101], 
[0.82836725287295332, -0.027963371325754327, 0.087921933670011221, 0.86536517681111547]], [[0.4325375591348628, 
-0.34896586965417353, 0.70354538947504319, -0.20056808943825621], [-0.52988367698250194, -0.59423561520650758, 
-0.97968384708944667, 0.026349425862099096], [-0.50171628506273414, -0.29951170726168153, 0.2288122009424558, 
0.39674411303152968]]], [[[-0.90269088072410208, -0.16278684381394903, -0.15269927499094682, 0.18509624754432652], 
[0.64036370121272612, -0.31610867439020707, 0.95209701283440196, -0.35041334363144094], [0.96111584894830626, 
-0.29793105576122403, 0.97110776227797535, 0.25247132709610609]], [[0.27743693873052067, -0.38424998978589087, 
0.61128936845407544, -0.94025011164981076], [0.88629254641622657, 0.41485617895265148, -0.63255160284551004, 
0.17669809948946202], [-0.56608338692442239, -0.17214950023195152, 0.52140620940859761, 0.32567292143787885]]], 
[[[-0.12136715353558047, -0.98637919281311182, 0.98268049823328707, 0.79774302895185389], [0.045567737996956126, 
-0.55633965988549927, 0.73487999272018034, -0.17918250878589093], [0.97769835593420829, 0.75944281097916932, 
0.016724295704277692, 0.54206523256588879]], [[-0.097591940836256885, 0.16199513255394415, -0.70561669003548055, 
-0.073562737251948951], [-0.16227839596318816, 0.19795405960405921, -0.76257514431383444, -0.91606555039067894], 
[0.75840636984124221, 0.83500880212967443, 0.42673148597153787, -0.27665081857084672]]]]))
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.96799917063261409, 0.46996860477429592, 0.47201141981822858, 0.46996860477429592], 
[0.46996860477429592, 0.96872332483282375, 0.46996860477429592, 0.49772168968845887], [0.46996860477429592, 
0.62012726879516444, 0.46996860477429592, 0.72050763180743038]], [[0.46996860477429592, 0.46996860477429592, 
0.92293794375703264, 0.46996860477429592], [0.46996860477429592, 0.97843635480846136, 0.46996860477429592, 
0.46996860477429592], [0.46996860477429592, 0.46996860477429592, 0.46996860477429592, 0.9525109700965102]]], 
[[[0.52539281253864512, 0.76087490949891379, 0.46996860477429592, 0.46996860477429592], [0.46996860477429592, 
0.47148631628813731, 0.46996860477429592, 0.46996860477429592], [0.46996860477429592, 0.46996860477429592, 0.46996860477429592, 
0.46996860477429592]], [[0.46996860477429592, 0.46996860477429592, 0.46996860477429592, 0.91568668490748339], 
[0.46996860477429592, 0.46996860477429592, 0.46996860477429592, 0.4883167426693038], [0.74528188999593414, 0.46996860477429592, 
0.46996860477429592, 0.46996860477429592]]], [[[0.46996860477429592, 0.65853853119832761, 0.46996860477429592, 
0.46996860477429592], [0.46996860477429592, 0.62349684395724125, 0.52607653495550122, 0.46996860477429592], 
[0.46996860477429592, 0.46996860477429592, 0.46996860477429592, 0.46996860477429592]], [[0.46996860477429592, 
0.46996860477429592, 0.46996860477429592, 0.46996860477429592], [0.46996860477429592, 0.78874943586980617, 0.6970466108744684, 
0.46996860477429592], [0.46996860477429592, 0.84711792647927608, 0.46996860477429592, 
0.46996860477429592]]]])+(1.-msk_ref)*numpy.array([[[[-0.79902570155778041, -0.84193383665442045, -0.23753030474741288, 
0.32015814046069635], [-0.69519556701166341, 0.56633323595909246, 0.75751301528775272, -0.06669352527047101], 
[0.82836725287295332, -0.027963371325754327, 0.087921933670011221, 0.86536517681111547]], [[0.4325375591348628, 
-0.34896586965417353, 0.70354538947504319, -0.20056808943825621], [-0.52988367698250194, -0.59423561520650758, 
-0.84193383665442045, 0.026349425862099096], [-0.50171628506273414, -0.29951170726168153, 0.2288122009424558, 
0.39674411303152968]]], [[[-0.84193383665442045, -0.16278684381394903, -0.15269927499094682, 0.18509624754432652], 
[0.64036370121272612, -0.31610867439020707, 0.95209701283440196, -0.35041334363144094], [0.96111584894830626, 
-0.29793105576122403, 0.97110776227797535, 0.25247132709610609]], [[0.27743693873052067, -0.38424998978589087, 
0.61128936845407544, -0.84193383665442045], [0.88629254641622657, 0.41485617895265148, -0.63255160284551004, 
0.17669809948946202], [-0.56608338692442239, -0.17214950023195152, 0.52140620940859761, 0.32567292143787885]]], 
[[[-0.12136715353558047, -0.84193383665442045, 0.98268049823328707, 0.79774302895185389], [0.045567737996956126, 
-0.55633965988549927, 0.73487999272018034, -0.17918250878589093], [0.97769835593420829, 0.75944281097916932, 
0.016724295704277692, 0.54206523256588879]], [[-0.097591940836256885, 0.16199513255394415, -0.70561669003548055, 
-0.073562737251948951], [-0.16227839596318816, 0.19795405960405921, -0.76257514431383444, -0.84193383665442045], 
[0.75840636984124221, 0.83500880212967443, 0.42673148597153787, -0.27665081857084672]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_taggedData_rank4_expandedData_rank4(self):
      arg0=Data(numpy.array([[[[0.17011890199565505, 0.012491731554322483, 0.68256748742414319, -0.81719958495420797], 
[0.74406930512472269, -0.14723025410749746, 0.44075186805280797, 0.26350411120931172], [-0.6561756812946733, 
-0.0057499514698438059, -0.42610418167990249, 0.62037645373934813]], [[0.04214322089694944, -0.51678689718464721, 
-0.47284322793307076, 0.2362074239254115], [0.094616672018278836, -0.97784214369059996, 0.62203411436109834, 
-0.84914557611328068], [-0.73480465431865083, 0.32701771835981663, 0.29173732358266657, -0.53414340612908906]]], 
[[[0.87760671365571685, 0.23702139444517556, 0.96537866890262891, 0.016090149009011068], [0.99366467643201961, 
0.71372417442915892, 0.85225018627477067, -0.64500940039217913], [0.48197861012437127, -0.73532690596378769, 
-0.22143168285654791, 0.29176529211391244]], [[-0.13555889099911278, 0.8267135744566636, 0.87899259525077778, 
0.69133559278711698], [0.07268574366965419, -0.76271205666442321, 0.57022619167259458, 0.31546663582717716], 
[-0.31537561003821302, -0.78782689871054901, 0.1113608900572931, -0.076429146299475281]]], [[[0.51958114573630576, 
-0.74017735928116934, -0.12370880033659826, -0.16262742669583541], [0.40996949833873697, -0.44090572970717146, 
-0.18625749746739984, -0.70066367152560827], [0.84295948903907658, -0.62143739657913155, 0.67911533223602527, 
-0.64691833789742681]], [[0.10078587849351872, 0.33589231968040756, 0.1162008658852578, 0.82764502498062087], 
[0.14890247007952917, -0.72628235215206072, -0.93575242911087453, -0.84236507788984083], [-0.60229223033356982, 
0.53131604811058852, -0.94053633532047853, 0.16539484858314313]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.018976007218767244, 0.099322777123676653, 0.71366707156868481, 
-0.22878348385769076], [0.78578474236157958, 0.84630287446581098, 0.72929454627823076, -0.88830712257451983], 
[0.64301481937440785, 0.52116322320377417, 0.1970425485822751, -0.72517228523071497]], [[-0.020402857553825315, 
0.91427721251920091, -0.50768591256964068, -0.66545474524420634], [0.50189121168846618, 0.84533523042586189, 
-0.85709751948982982, -0.59235917428014084], [0.69544846189320775, -0.99232447839305404, -0.41196226863364238, 
-0.88439740854067783]]], [[[0.79656532563074944, 0.56896777821890021, 0.33138835863853866, -0.66292590878912305], 
[0.30452117957021851, 0.065858998465211149, 0.81465880143475711, -0.4882852026900204], [-0.31574407895226431, 
0.63352382252195083, -0.27839534078752504, -0.63546346809565302]], [[-0.12760679020742605, -0.86882714833599151, 
-0.18047061675034959, -0.96996723969541998], [-0.10781508772821979, -0.35569215942943799, -0.15828371288345755, 
0.90563308858060365], [-0.81685869695187474, 0.24029199408039514, 0.20419548781326236, -0.81764921130223067]]], 
[[[0.2850054984772834, 0.7627457219523146, 0.60192919646893839, 0.92860863577549324], [0.38272295399764844, 
0.63922506079556873, 0.91012536728959792, 0.23840788997679008], [-0.66417151972746624, 0.41012468409180891, 
0.69551622234874477, 0.69385347288012644]], [[0.3510118799331543, 0.036668719298536656, -0.30489782793779985, 
-0.77390172511630761], [-0.28600675097554262, -0.26913515312460157, 0.9282219697537224, -0.12188253822477479], 
[0.66024739821145428, -0.16637608341532695, 0.76041699763549997, -0.45709713833511723]]]]))
      arg1=Data(numpy.array([[[[0.61136503290314326, -0.9583623189339503, -0.071111339698633813, 0.50823450262690639], 
[-0.91370014958462753, -0.57444315283280867, 0.88871572476171035, -0.72648904747012533], [0.91718509774010948, 
0.74029234483051298, 0.81359933629774561, 0.17231905669887015]], [[-0.39889780284662257, 0.24678695785705496, 
0.62462078938126075, 0.46766672497251749], [-0.96150034521586658, -0.11514014559100438, 0.65390919502033795, 
-0.22254683439345824], [0.968826953341269, 0.4548040265697908, 0.63841722597965811, 0.27597672152017383]]], 
[[[-0.67355264176575647, -0.1578078828847953, 0.53605000621095833, 0.66735108383917829], [0.97674770476592254, 
-0.15475778131628037, 0.46725942432288603, 0.27546459379283661], [-0.165986261182546, 0.54327085078645654, 
-0.29409779812674008, 0.91303065524149729]], [[-0.20407223332156255, 0.50784612751752678, 0.50504834054454339, 
0.35806688218678717], [0.6368167147413637, 0.13094643656949279, 0.42013937012956948, -0.1532152807402456], 
[0.61624494910961691, -0.62288004538551411, 0.88353811932694071, -0.78330402028144697]]], [[[0.84136347714033266, 
0.2991007327978259, -0.19113648177205667, -0.78642546935555457], [-0.28913146194132966, -0.34256583676388908, 
0.82091750950874065, -0.97792363430747864], [-0.028382417989220032, -0.75992603329343278, -0.51749656663179566, 
0.076851139682463376]], [[0.54359012126046236, 0.26967236287800045, 0.8998349269257937, -0.19140928789968226], 
[0.11792876188493961, -0.72723528139193472, -0.72989452527701926, -0.07985159697537858], [-0.58211421995747359, 
0.99248594105873855, -0.84701415247397738, 0.69680114105695368]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.63571254524701759, 0.20370592255074182, -0.64548309985704999, 
0.77652367903402131], [0.36230291102741763, -0.77716092138963599, 0.37503342523027161, 0.50580548237875722], 
[-0.80476176999744298, 0.43811320566049328, 0.9523892099430129, -0.080610444991192187]], [[-0.29860215012414315, 
0.92120981653648437, 0.92326725495046702, 0.79082000429192978], [-0.30546740941720318, 0.056650128257929699, 
-0.16008152867712999, -0.62196364825925499], [0.61818701505402518, -0.12151838977341978, 0.28908127563919428, 
-0.87769634840198751]]], [[[0.54636794023437063, -0.56427286960541045, -0.8046880441135964, -0.86286817830595197], 
[-0.89930195996257845, -0.65201723944472989, -0.54635454097129665, -0.78225496543791539], [0.64326740231955726, 
0.53428131623295072, -0.60124494225447478, -0.19047720636463028]], [[0.64480428134100465, -0.9604075252803197, 
-0.018342425030688725, -0.902529630428063], [-0.61508139044329324, -0.76582597096513005, -0.91068255111115981, 
0.54021795903014769], [-0.78398977334913633, 0.19354315441019265, 0.75161914561978693, -0.1784273766678417]]], 
[[[0.70651195164291747, -0.38013254980625022, -0.42892733702099628, 0.76218105791882018], [0.87008144981991875, 
0.8415007055668402, -0.93553838665506706, 0.36596867894300966], [0.37708148652459728, 0.73706705651814941, 
-0.91814349937326178, -0.12915340952471444]], [[-0.57945334441687035, 0.068897411819411891, 0.13444813818129719, 
0.012465407268180906], [-0.16485376132646112, 0.51736826348586362, -0.32716335023341148, 0.52188670759134292], 
[-0.71241361375043133, -0.87684700004061544, 0.92751538707310321, -0.3163835738640477]]]]))
      arg1.expand()
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.61136503290314326, 0.012491731554322483, 0.68256748742414319, 0.50823450262690639], 
[0.74406930512472269, -0.14723025410749746, 0.88871572476171035, 0.26350411120931172], [0.91718509774010948, 
0.74029234483051298, 0.81359933629774561, 0.62037645373934813]], [[0.04214322089694944, 0.24678695785705496, 
0.62462078938126075, 0.46766672497251749], [0.094616672018278836, -0.11514014559100438, 0.65390919502033795, 
-0.22254683439345824], [0.968826953341269, 0.4548040265697908, 0.63841722597965811, 0.27597672152017383]]], 
[[[0.87760671365571685, 0.23702139444517556, 0.96537866890262891, 0.66735108383917829], [0.99366467643201961, 
0.71372417442915892, 0.85225018627477067, 0.27546459379283661], [0.48197861012437127, 0.54327085078645654, 
-0.22143168285654791, 0.91303065524149729]], [[-0.13555889099911278, 0.8267135744566636, 0.87899259525077778, 
0.69133559278711698], [0.6368167147413637, 0.13094643656949279, 0.57022619167259458, 0.31546663582717716], 
[0.61624494910961691, -0.62288004538551411, 0.88353811932694071, -0.076429146299475281]]], [[[0.84136347714033266, 
0.2991007327978259, -0.12370880033659826, -0.16262742669583541], [0.40996949833873697, -0.34256583676388908, 
0.82091750950874065, -0.70066367152560827], [0.84295948903907658, -0.62143739657913155, 0.67911533223602527, 
0.076851139682463376]], [[0.54359012126046236, 0.33589231968040756, 0.8998349269257937, 0.82764502498062087], 
[0.14890247007952917, -0.72628235215206072, -0.72989452527701926, -0.07985159697537858], [-0.58211421995747359, 
0.99248594105873855, -0.84701415247397738, 0.69680114105695368]]]])+(1.-msk_ref)*numpy.array([[[[0.018976007218767244, 
0.20370592255074182, 0.71366707156868481, 0.77652367903402131], [0.78578474236157958, 0.84630287446581098, 0.72929454627823076, 
0.50580548237875722], [0.64301481937440785, 0.52116322320377417, 0.9523892099430129, -0.080610444991192187]], 
[[-0.020402857553825315, 0.92120981653648437, 0.92326725495046702, 0.79082000429192978], [0.50189121168846618, 
0.84533523042586189, -0.16008152867712999, -0.59235917428014084], [0.69544846189320775, -0.12151838977341978, 
0.28908127563919428, -0.87769634840198751]]], [[[0.79656532563074944, 0.56896777821890021, 0.33138835863853866, 
-0.66292590878912305], [0.30452117957021851, 0.065858998465211149, 0.81465880143475711, -0.4882852026900204], 
[0.64326740231955726, 0.63352382252195083, -0.27839534078752504, -0.19047720636463028]], [[0.64480428134100465, 
-0.86882714833599151, -0.018342425030688725, -0.902529630428063], [-0.10781508772821979, -0.35569215942943799, 
-0.15828371288345755, 0.90563308858060365], [-0.78398977334913633, 0.24029199408039514, 0.75161914561978693, 
-0.1784273766678417]]], [[[0.70651195164291747, 0.7627457219523146, 0.60192919646893839, 0.92860863577549324], 
[0.87008144981991875, 0.8415007055668402, 0.91012536728959792, 0.36596867894300966], [0.37708148652459728, 0.73706705651814941, 
0.69551622234874477, 0.69385347288012644]], [[0.3510118799331543, 0.068897411819411891, 0.13444813818129719, 
0.012465407268180906], [-0.16485376132646112, 0.51736826348586362, 0.9282219697537224, 0.52188670759134292], 
[0.66024739821145428, -0.16637608341532695, 0.92751538707310321, -0.3163835738640477]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_taggedData_rank0(self):
      arg0=Data(-0.770416527778,self.functionspace)
      arg0.setTaggedValue(1,-0.119957544315)
      arg0.expand()
      arg1=Data(0.474753756871,self.functionspace)
      arg1.setTaggedValue(1,0.810578501236)
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(0.474753756871)+(1.-msk_ref)*(0.810578501236)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_taggedData_rank0(self):
      arg0=Data(numpy.array([-0.90345143142679607, -0.13416885186088923]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.33205401475987184, 0.19290593142804124]))
      arg0.expand()
      arg1=Data(0.439515912286,self.functionspace)
      arg1.setTaggedValue(1,-0.32959484933)
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.4395159122859198, 0.4395159122859198])+(1.-msk_ref)*numpy.array([-0.32959484933029048, 
0.19290593142804124])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_taggedData_rank0(self):
      arg0=Data(numpy.array([[0.43922074256473276, 0.0079120810391168916, 0.1259514035506768, -0.81976263905332281, 
0.22765318928035638], [-0.88565682969644199, 0.58003358429582219, -0.46586840067079094, -0.18975265061192093, 
-0.35961893325441308], [-0.97387322162339141, 0.91203728971074072, 0.58315914470271135, -0.96277527234543614, 
0.54600207266375422], [0.967552839442682, 0.14676060927846857, -0.039638732927746112, -0.11956428889953585, 
0.74642774018193192]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.62075677261147288, -0.45667940372405669, 0.30063606656178843, -0.55559974833961401, 
-0.2755921860001902], [-0.98316388377705821, -0.54118758166498004, 0.2947378884171552, 0.20623349090806964, 
-0.20563354685188306], [0.99570732573031462, 0.42428346604349709, -0.67254094797032282, -0.72385152307713096, 
0.12652687164985221], [0.86027605858930056, 0.69983693229105892, -0.54255499543667995, -0.34652000344244804, 
0.29258488270051397]]))
      arg0.expand()
      arg1=Data(-0.683436160364,self.functionspace)
      arg1.setTaggedValue(1,0.237423149367)
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.43922074256473276, 0.0079120810391168916, 0.1259514035506768, -0.68343616036418209, 
0.22765318928035638], [-0.68343616036418209, 0.58003358429582219, -0.46586840067079094, -0.18975265061192093, 
-0.35961893325441308], [-0.68343616036418209, 0.91203728971074072, 0.58315914470271135, -0.68343616036418209, 
0.54600207266375422], [0.967552839442682, 0.14676060927846857, -0.039638732927746112, -0.11956428889953585, 
0.74642774018193192]])+(1.-msk_ref)*numpy.array([[0.62075677261147288, 0.23742314936707154, 0.30063606656178843, 
0.23742314936707154, 0.23742314936707154], [0.23742314936707154, 0.23742314936707154, 0.2947378884171552, 0.23742314936707154, 
0.23742314936707154], [0.99570732573031462, 0.42428346604349709, 0.23742314936707154, 0.23742314936707154, 
0.23742314936707154], [0.86027605858930056, 0.69983693229105892, 0.23742314936707154, 0.23742314936707154, 
0.29258488270051397]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_taggedData_rank0(self):
      arg0=Data(numpy.array([[[-0.21366601469826718, 0.99527040818565293], [0.22215618408937154, 0.31701252929306478]], 
[[0.40623060792399568, 0.21372472646539054], [-0.47540828991388162, -0.46252911439545796]], [[-0.72333391349122134, 
0.098643583537729285], [0.52420807960470261, -0.50471492899503079]], [[-0.40644921673789036, -0.45254174080439036], 
[0.78953583656759241, -0.94694089619305699]], [[0.10109303594579955, 0.71996063617220285], [0.99742338935639263, 
-0.70472498807905137]], [[0.081799072345769552, 0.18293416803446694], [0.94055435086427286, 
0.20985468194654966]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.85922501286389963, -0.95314495089101836], [-0.41714431819315956, 
-0.98482774191370592]], [[-0.36714035024248104, -0.68192801230381117], [0.050175258818712987, -0.71275657603909282]], 
[[-0.067382389619428773, 0.89129921856134686], [0.36619893079335464, -0.5296889938497984]], [[0.37130647053619814, 
0.93695230545987362], [0.3719018436922632, 0.60940463384310828]], [[-0.034716030432518696, -0.21726170531723676], 
[0.13410420999221606, 0.45218157437810746]], [[-0.60130757007073377, -0.55224098558489643], [0.35071707364717586, 
0.73918188881062408]]]))
      arg0.expand()
      arg1=Data(0.589930215921,self.functionspace)
      arg1.setTaggedValue(1,-0.791358930914)
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.58993021592108552, 0.99527040818565293], [0.58993021592108552, 0.58993021592108552]], 
[[0.58993021592108552, 0.58993021592108552], [0.58993021592108552, 0.58993021592108552]], [[0.58993021592108552, 
0.58993021592108552], [0.58993021592108552, 0.58993021592108552]], [[0.58993021592108552, 0.58993021592108552], 
[0.78953583656759241, 0.58993021592108552]], [[0.58993021592108552, 0.71996063617220285], [0.99742338935639263, 
0.58993021592108552]], [[0.58993021592108552, 0.58993021592108552], [0.94055435086427286, 
0.58993021592108552]]])+(1.-msk_ref)*numpy.array([[[0.85922501286389963, -0.79135893091398235], [-0.41714431819315956, 
-0.79135893091398235]], [[-0.36714035024248104, -0.68192801230381117], [0.050175258818712987, -0.71275657603909282]], 
[[-0.067382389619428773, 0.89129921856134686], [0.36619893079335464, -0.5296889938497984]], [[0.37130647053619814, 
0.93695230545987362], [0.3719018436922632, 0.60940463384310828]], [[-0.034716030432518696, -0.21726170531723676], 
[0.13410420999221606, 0.45218157437810746]], [[-0.60130757007073377, -0.55224098558489643], [0.35071707364717586, 
0.73918188881062408]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_taggedData_rank0(self):
      arg0=Data(numpy.array([[[[0.058452530027124805, 0.20115061735928141, -0.013407311335628558, -0.62733037718045215], 
[0.30578317601612204, 0.18587813517772922, -0.46083224983819182, 0.38420623366127171], [-0.35041386682037245, 
0.28613237541338643, 0.84287606761862599, -0.37452032983582861]], [[-0.1571001411973949, 0.48404434987845502, 
-0.82445608306831009, -0.5087589407859181], [0.080024058489493344, 0.60071403089801323, 0.76226797059719953, 
0.52457829108635523], [-0.47422782685361131, 0.68442557385398506, 0.66334103264465472, 0.64005155311144235]]], 
[[[0.65185373935916635, -0.052178567808071508, -0.97872127915831397, 0.75346659417896045], [-0.7930913742759933, 
0.035965062214120769, -0.048637444759485504, -0.57671265123293947], [-0.77598367520914446, -0.63886273463026555, 
0.77179543911420989, -0.72328815552507297]], [[0.6245388094023514, 0.21638017300875068, 0.98319402644101062, 
-0.90884610419532641], [0.063053421739729831, 0.13831930984516361, -0.35843200982016654, 0.50589485921428756], 
[0.17854873853074138, 0.14205857566189661, 0.27873814750700032, 0.044143543347939129]]], [[[-0.98235903467690555, 
-0.3779883610410002, 0.20182363674090387, -0.99195045062105591], [-0.84378162546138302, 0.13009634759743482, 
-0.021923717778070007, -0.21732600979488104], [0.64072354550435562, -0.29472251730728827, -0.85957706692075986, 
-0.97194980141864828]], [[-0.20303445856569358, -0.32820645194490372, -0.54738924943535761, 0.63120829931168476], 
[0.72062853854960118, 0.9314323308822694, -0.76951439295743085, 0.69074779508974804], [0.029387956008169347, 
-0.17961113301791842, -0.28654263317059403, 0.93712330786747544]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.23539178002763772, -0.73120437151627637, 0.28316833446181899, 
0.86284986085143967], [-0.24353257383874816, -0.893212124404694, 0.59742195037052448, -0.40530874064915312], 
[-0.57175026142836627, 0.25022701748997545, 0.17567464768761232, -0.071490198224184232]], [[-0.48372399998626525, 
-0.46003016110634976, 0.162227597393092, -0.10421637975460807], [-0.84250355339179328, -0.45635672728478416, 
-0.41133900862963402, 0.15232657179157161], [-0.78484435415388187, 0.91063328248964459, 0.94267309796018428, 
0.6465806608067326]]], [[[0.96940080044077348, 0.92260535563446022, 0.14530523054519273, -0.059208623621642698], 
[0.65642237359346445, -0.74588495205964689, 0.20182736082108588, -0.31654806017892678], [0.53577721101623577, 
0.75646492881119287, 0.7657224840563881, 0.50502072922800201]], [[-0.51642103766997005, 0.62533968630116266, 
-0.98595880165403349, 0.69718594133432021], [0.94920592257949976, -0.40699371519169691, -0.72576193045697934, 
0.13718863337454934], [-0.95814498747220611, 0.09240739415652377, -0.58862477478085462, 0.40331711277464044]]], 
[[[0.11456697769797963, -0.59435563966968585, -0.50208215421818791, 0.55936719433132809], [0.89532220784394423, 
-0.4195336101704854, -0.72937425429419789, 0.74317410016178731], [-0.19327200728151439, -0.14918712139341905, 
0.52793569011349706, -0.20821618856608448]], [[-0.46777868535808764, 0.20593248679186105, -0.87810200969679886, 
-0.98507550795727039], [-0.42479747623997333, -0.21395671462352706, 0.49812776303723116, 0.77515813391518718], 
[0.033556016565504887, 0.72639708232512157, -0.67862416607200315, 0.31556674450753253]]]]))
      arg0.expand()
      arg1=Data(-0.872258244985,self.functionspace)
      arg1.setTaggedValue(1,-0.599062516076)
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.058452530027124805, 0.20115061735928141, -0.013407311335628558, -0.62733037718045215], 
[0.30578317601612204, 0.18587813517772922, -0.46083224983819182, 0.38420623366127171], [-0.35041386682037245, 
0.28613237541338643, 0.84287606761862599, -0.37452032983582861]], [[-0.1571001411973949, 0.48404434987845502, 
-0.82445608306831009, -0.5087589407859181], [0.080024058489493344, 0.60071403089801323, 0.76226797059719953, 
0.52457829108635523], [-0.47422782685361131, 0.68442557385398506, 0.66334103264465472, 0.64005155311144235]]], 
[[[0.65185373935916635, -0.052178567808071508, -0.87225824498472293, 0.75346659417896045], [-0.7930913742759933, 
0.035965062214120769, -0.048637444759485504, -0.57671265123293947], [-0.77598367520914446, -0.63886273463026555, 
0.77179543911420989, -0.72328815552507297]], [[0.6245388094023514, 0.21638017300875068, 0.98319402644101062, 
-0.87225824498472293], [0.063053421739729831, 0.13831930984516361, -0.35843200982016654, 0.50589485921428756], 
[0.17854873853074138, 0.14205857566189661, 0.27873814750700032, 0.044143543347939129]]], [[[-0.87225824498472293, 
-0.3779883610410002, 0.20182363674090387, -0.87225824498472293], [-0.84378162546138302, 0.13009634759743482, 
-0.021923717778070007, -0.21732600979488104], [0.64072354550435562, -0.29472251730728827, -0.85957706692075986, 
-0.87225824498472293]], [[-0.20303445856569358, -0.32820645194490372, -0.54738924943535761, 0.63120829931168476], 
[0.72062853854960118, 0.9314323308822694, -0.76951439295743085, 0.69074779508974804], [0.029387956008169347, 
-0.17961113301791842, -0.28654263317059403, 0.93712330786747544]]]])+(1.-msk_ref)*numpy.array([[[[-0.23539178002763772, 
-0.59906251607554895, 0.28316833446181899, 0.86284986085143967], [-0.24353257383874816, -0.59906251607554895, 
0.59742195037052448, -0.40530874064915312], [-0.57175026142836627, 0.25022701748997545, 0.17567464768761232, 
-0.071490198224184232]], [[-0.48372399998626525, -0.46003016110634976, 0.162227597393092, -0.10421637975460807], 
[-0.59906251607554895, -0.45635672728478416, -0.41133900862963402, 0.15232657179157161], [-0.59906251607554895, 
0.91063328248964459, 0.94267309796018428, 0.6465806608067326]]], [[[0.96940080044077348, 0.92260535563446022, 
0.14530523054519273, -0.059208623621642698], [0.65642237359346445, -0.59906251607554895, 0.20182736082108588, 
-0.31654806017892678], [0.53577721101623577, 0.75646492881119287, 0.7657224840563881, 0.50502072922800201]], 
[[-0.51642103766997005, 0.62533968630116266, -0.59906251607554895, 0.69718594133432021], [0.94920592257949976, 
-0.40699371519169691, -0.59906251607554895, 0.13718863337454934], [-0.59906251607554895, 0.09240739415652377, 
-0.58862477478085462, 0.40331711277464044]]], [[[0.11456697769797963, -0.59435563966968585, -0.50208215421818791, 
0.55936719433132809], [0.89532220784394423, -0.4195336101704854, -0.59906251607554895, 0.74317410016178731], 
[-0.19327200728151439, -0.14918712139341905, 0.52793569011349706, -0.20821618856608448]], [[-0.46777868535808764, 
0.20593248679186105, -0.59906251607554895, -0.59906251607554895], [-0.42479747623997333, -0.21395671462352706, 
0.49812776303723116, 0.77515813391518718], [0.033556016565504887, 0.72639708232512157, -0.59906251607554895, 
0.31556674450753253]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_taggedData_rank1(self):
      arg0=Data(-0.093529730868,self.functionspace)
      arg0.setTaggedValue(1,0.110676465592)
      arg0.expand()
      arg1=Data(numpy.array([-0.24077231783613207, 0.021968113006201273]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.69246963874543788, -0.11448974195720685]))
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-0.09352973086798122, 0.021968113006201273])+(1.-msk_ref)*numpy.array([0.69246963874543788, 
0.11067646559234223])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([-0.043611999909903609, -0.75469634662679885]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.97796930150784322, 0.27203588315418514]))
      arg0.expand()
      arg1=Data(numpy.array([0.99115736338403471, -0.27063973980015166]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.66221422553399734, -0.4445440806254044]))
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.99115736338403471, -0.27063973980015166])+(1.-msk_ref)*numpy.array([0.97796930150784322, 
0.27203588315418514])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_taggedData_rank2(self):
      arg0=Data(0.562200963296,self.functionspace)
      arg0.setTaggedValue(1,-0.548717725726)
      arg0.expand()
      arg1=Data(numpy.array([[0.46302058489008435, 0.39744900050570653, -0.31760126560331359, -0.74752181980080645, 
0.66145632559247969], [0.23022554234537362, 0.045049075349218093, 0.47633449866518918, 0.92590614489792178, 
0.83235604073104486], [-0.064032426645444174, -0.95919048878578828, 0.12438083700652847, 0.76795909320208078, 
-0.79733667826256194], [0.94008936801365239, 0.18171575357678926, 0.13755218013794224, -0.41510128527708789, 
0.98167626068722313]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.085533977491544011, -0.93060326532816573, 0.56871857978607299, 
0.64816224420242974, -0.31867781829316999], [-0.037176782985713919, 0.047131059915608065, -0.10164449238423523, 
-0.34884037578484639, 0.88404765826084875], [0.7918226937102506, -0.2261614400671339, 0.6482237690064454, 0.2079147708003577, 
-0.53283060407360239], [0.76435770832885708, 0.3881379874070694, 0.6014042528428114, 0.5028235019961087, 
-0.055323669447446155]]))
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.56220096329648528, 0.56220096329648528, 0.56220096329648528, 0.56220096329648528, 
0.66145632559247969], [0.56220096329648528, 0.56220096329648528, 0.56220096329648528, 0.92590614489792178, 
0.83235604073104486], [0.56220096329648528, 0.56220096329648528, 0.56220096329648528, 0.76795909320208078, 
0.56220096329648528], [0.94008936801365239, 0.56220096329648528, 0.56220096329648528, 0.56220096329648528, 
0.98167626068722313]])+(1.-msk_ref)*numpy.array([[-0.085533977491544011, -0.54871772572627697, 0.56871857978607299, 
0.64816224420242974, -0.31867781829316999], [-0.037176782985713919, 0.047131059915608065, -0.10164449238423523, 
-0.34884037578484639, 0.88404765826084875], [0.7918226937102506, -0.2261614400671339, 0.6482237690064454, 0.2079147708003577, 
-0.53283060407360239], [0.76435770832885708, 0.3881379874070694, 0.6014042528428114, 0.5028235019961087, 
-0.055323669447446155]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[0.95250070334252501, 0.64179594117717476, 0.7167453892355482, -0.061767675759885954, 
0.2083685039484644], [0.48813056886651807, -0.1606681211293568, 0.20439892123657244, 0.28289621453136227, 0.70024296852753021], 
[0.76591981855093239, 0.3046465331639463, -0.62959399544720429, 0.063923416563474333, -0.86871278195397172], 
[0.58170260659522488, -0.39794844598673951, -0.65152408954750385, -0.47787725382817348, 
-0.71387975157040096]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.90856041306199531, -0.23227397078715706, 0.27336542418286824, 
-0.80781812954858623, 0.867873827511638], [-0.087352806605938582, -0.86269266300680103, -0.11942072102346013, 
0.86401044819248507, -0.11498573224937791], [-0.7024042295787194, 0.71001251168456547, -0.50926611775019337, 
0.64503564060050755, -0.72376495202954239], [0.33615193865620419, -0.77494971698915838, 0.72401984261341212, 
0.0012922102318457807, -0.37428786801575198]]))
      arg0.expand()
      arg1=Data(numpy.array([[0.20367156607055192, -0.82704445252224801, -0.87142392198822427, -0.31031405049634375, 
0.58087729006045619], [0.51057143457714127, -0.27207610740823251, 0.42578927315563231, -0.093080378558869636, 
0.015547166871238582], [0.79676548326720398, -0.035900358788261144, 0.30551798071437042, -0.40888810953186705, 
-0.67286861845528234], [0.16156799341392203, 0.35652355717772588, -0.60846707575455095, -0.40466620246153551, 
-0.85212577402488598]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.22540342674499247, -0.31253350564886428, 0.66349648680317075, -0.72430451779761729, 
-0.64802942349008652], [0.46990597522394184, 0.17192636352441948, -0.81927146771373249, 0.33673292484183692, 
0.097905227205821621], [0.11962243052664623, 0.1852073417887059, -0.37523746529768909, -0.90949222260581686, 
-0.82374706876749482], [0.24737438344353802, 0.17879293284034126, -0.89391109198951679, 0.26302854741774051, 
0.91163389900291247]]))
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.95250070334252501, 0.64179594117717476, 0.7167453892355482, -0.061767675759885954, 
0.58087729006045619], [0.51057143457714127, -0.1606681211293568, 0.42578927315563231, 0.28289621453136227, 
0.70024296852753021], [0.79676548326720398, 0.3046465331639463, 0.30551798071437042, 0.063923416563474333, 
-0.67286861845528234], [0.58170260659522488, 0.35652355717772588, -0.60846707575455095, -0.40466620246153551, 
-0.71387975157040096]])+(1.-msk_ref)*numpy.array([[0.22540342674499247, -0.23227397078715706, 0.66349648680317075, 
-0.72430451779761729, 0.867873827511638], [0.46990597522394184, 0.17192636352441948, -0.11942072102346013, 0.86401044819248507, 
0.097905227205821621], [0.11962243052664623, 0.71001251168456547, -0.37523746529768909, 0.64503564060050755, 
-0.72376495202954239], [0.33615193865620419, 0.17879293284034126, 0.72401984261341212, 0.26302854741774051, 
0.91163389900291247]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_taggedData_rank3(self):
      arg0=Data(0.518888053272,self.functionspace)
      arg0.setTaggedValue(1,-0.526416582974)
      arg0.expand()
      arg1=Data(numpy.array([[[0.29196426529678976, 0.32043612632866503], [-0.89707110782253463, 0.038228958797583923]], 
[[0.030493489554798137, -0.723108722645039], [0.98623722967674055, 0.66252274455661753]], [[0.39916522486562722, 
-0.45876533710564305], [0.75447948886732941, 0.15340589898476398]], [[0.81578682948478276, 0.20522022333925705], 
[0.47335990871775424, -0.6355595726654879]], [[-0.93239359678287848, 0.217306855880812], [-0.88354692825016867, 
0.56677014722507546]], [[0.11742167565485251, -0.12080999397647729], [0.79569009080530706, 
0.22001425648306694]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.24172936662080913, 0.62410363287595794], [-0.94649908269893368, 
0.80957984698971019]], [[-0.95851870834110486, -0.49011739229006857], [-0.049499510664444246, -0.37934214612873873]], 
[[-0.5325422920695364, -0.87275009910709511], [0.67744619631994052, 0.35293782191134171]], [[-0.93486756159181361, 
-0.27050536347040222], [0.24337856921346934, 0.9450297004409578]], [[0.54054330962315134, -0.39032751650920527], 
[0.98009854283693931, 0.40832854762680859]], [[-0.34552393114212099, -0.39712782701241256], [-0.44647745946589668, 
-0.81843160678995419]]]))
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.51888805327181631, 0.51888805327181631], [0.51888805327181631, 0.51888805327181631]], 
[[0.51888805327181631, 0.51888805327181631], [0.98623722967674055, 0.66252274455661753]], [[0.51888805327181631, 
0.51888805327181631], [0.75447948886732941, 0.51888805327181631]], [[0.81578682948478276, 0.51888805327181631], 
[0.51888805327181631, 0.51888805327181631]], [[0.51888805327181631, 0.51888805327181631], [0.51888805327181631, 
0.56677014722507546]], [[0.51888805327181631, 0.51888805327181631], [0.79569009080530706, 
0.51888805327181631]]])+(1.-msk_ref)*numpy.array([[[-0.24172936662080913, 0.62410363287595794], [-0.52641658297364669, 
0.80957984698971019]], [[-0.52641658297364669, -0.49011739229006857], [-0.049499510664444246, -0.37934214612873873]], 
[[-0.52641658297364669, -0.52641658297364669], [0.67744619631994052, 0.35293782191134171]], [[-0.52641658297364669, 
-0.27050536347040222], [0.24337856921346934, 0.9450297004409578]], [[0.54054330962315134, -0.39032751650920527], 
[0.98009854283693931, 0.40832854762680859]], [[-0.34552393114212099, -0.39712782701241256], [-0.44647745946589668, 
-0.52641658297364669]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_taggedData_rank3(self):
      arg0=Data(numpy.array([[[-0.37358195248163262, -0.27336564520633688], [0.31185513666213693, -0.97253484086889852]], 
[[-0.23168627929530583, -0.03616192464774648], [-0.32948193730210429, 0.78686549068518663]], [[-0.61449118133391689, 
0.90998031960320436], [0.99605491014410519, -0.19027309086387212]], [[0.94604471442750215, -0.72799516092415306], 
[0.86948766116606113, -0.81240404435014413]], [[0.99030218260245695, 0.56350567065263735], [-0.039620325891805752, 
-0.81732668479461523]], [[-0.39750688951062196, 0.40169693961536113], [0.94787126923095011, 
0.57812481686625183]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.32594941395302701, 0.087905777161087384], [-0.52219886862144738, 
-0.9731314947029206]], [[0.58502802758929651, -0.41142097068201489], [-0.88364691152976493, -0.10414245222136698]], 
[[-0.576353412686192, -0.95369860222969849], [0.2099372100186514, 0.50421743886742365]], [[-0.51847992419818412, 
-0.24059284345629162], [0.84039522274325185, 0.11037716688528287]], [[-0.1507436522189145, -0.27490435756027432], 
[0.28670037025118655, 0.33083817728664044]], [[-0.20086532148474423, 0.37400644898968993], [0.50401082589377033, 
-0.71131837198550185]]]))
      arg0.expand()
      arg1=Data(numpy.array([[[-0.65818845288483652, -0.86621168561924344], [-0.67077872357267854, 0.49138367131164107]], 
[[-0.5281433157210047, -0.72346242749506806], [0.61631433617349707, 0.12025122366548602]], [[0.41212118683148069, 
-0.46248287281405998], [0.30957103073079462, 0.15147402020930234]], [[-0.646681250682696, -0.32759770784581188], 
[0.92304065918314526, -0.71108271334269801]], [[-0.58354307425432239, 0.37676623449431679], [-0.52285211273345134, 
0.21083981696926224]], [[0.23564951404346224, 0.88899136001079682], [-0.12150868659653757, 
0.43961348936939371]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.98583186182546845, 0.93635507276650998], [-0.80192348917001577, 
-0.17450779277321748]], [[0.15122760577460825, -0.54809589011071025], [-0.72163021964022223, -0.34435970073118782]], 
[[-0.1075348665208602, 0.82723530422236191], [-0.026284279321377113, 0.99655978862994155]], [[0.88591020241017526, 
0.4033730619812268], [0.58710034420101342, 0.45449118143951517]], [[-0.75632762213219995, -0.18778964902536099], 
[-0.86748293055806136, 0.037624977312142649]], [[0.91517418897127945, -0.51665633539711497], [-0.32873185902105706, 
-0.27720413216145157]]]))
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-0.37358195248163262, -0.27336564520633688], [0.31185513666213693, 0.49138367131164107]], 
[[-0.23168627929530583, -0.03616192464774648], [0.61631433617349707, 0.78686549068518663]], [[0.41212118683148069, 
0.90998031960320436], [0.99605491014410519, 0.15147402020930234]], [[0.94604471442750215, -0.32759770784581188], 
[0.92304065918314526, -0.71108271334269801]], [[0.99030218260245695, 0.56350567065263735], [-0.039620325891805752, 
0.21083981696926224]], [[0.23564951404346224, 0.88899136001079682], [0.94787126923095011, 
0.57812481686625183]]])+(1.-msk_ref)*numpy.array([[[0.32594941395302701, 0.93635507276650998], [-0.52219886862144738, 
-0.17450779277321748]], [[0.58502802758929651, -0.41142097068201489], [-0.72163021964022223, -0.10414245222136698]], 
[[-0.1075348665208602, 0.82723530422236191], [0.2099372100186514, 0.99655978862994155]], [[0.88591020241017526, 
0.4033730619812268], [0.84039522274325185, 0.45449118143951517]], [[-0.1507436522189145, -0.18778964902536099], 
[0.28670037025118655, 0.33083817728664044]], [[0.91517418897127945, 0.37400644898968993], [0.50401082589377033, 
-0.27720413216145157]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_taggedData_rank4(self):
      arg0=Data(0.450629877736,self.functionspace)
      arg0.setTaggedValue(1,0.579460820819)
      arg0.expand()
      arg1=Data(numpy.array([[[[-0.93458352340359307, -0.41190067471865022, -0.23664358363068883, -0.40370986429176203], 
[0.41247850740268155, 0.44897475734401082, 0.79900860948329, -0.80980394525129284], [0.5108826808220337, 0.013981347297780555, 
0.17465414053669592, -0.1669664733120535]], [[-0.12378328511678194, -0.17844456261491004, -0.44505368193438311, 
-0.46383709809413642], [0.89281480055911766, -0.31338677696404593, 0.93180996621646917, -0.71015409425700105], 
[0.31137743484605229, -0.009819696375075182, 0.85505937142450295, 0.5378881600091503]]], [[[0.34441745593066697, 
-0.075449850187962575, 0.26642985235916572, 0.5423072309997008], [-0.61883754887568521, 0.98860123211230855, 
0.90741603461179854, -0.15597073824641106], [0.54808223548024415, -0.22522989614967615, 0.68978257581627056, 
-0.72646580012977768]], [[-0.32116504709223048, 0.59210693779982515, 0.87444883555895481, -0.44800630836857747], 
[0.45053215713839712, 0.56398716173861096, -0.23978471018735403, -0.6354746422998645], [-0.6487212918194265, 
0.95086139306366735, 0.08394378260375901, 0.38780478286457187]]], [[[0.57335107011599828, -0.37310359482598487, 
0.97431386356392546, 0.40055848390252358], [-0.13201356350120763, 0.85412964511631384, 0.19285445907386611, 
-0.56721463144240136], [0.8075580531915918, 0.87134037440501655, -0.60961756368144604, -0.18929027481643401]], 
[[-0.27613225648092476, 0.7593111093249223, -0.39704005972987222, -0.020132271921616374], [0.58468253560648864, 
0.82466000389884875, 0.3965353792647317, 0.26845317296980276], [-0.95534621342452231, 0.013290004356439233, 
0.76350095350066249, -0.20133960448758703]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.10792932086568485, 0.76021978616093389, -0.70032561973317597, 
0.47528252889807487], [0.58819854593101017, 0.10720165909492785, 0.40461033718299455, 0.96470130871039728], 
[0.2733654977507971, 0.22627953190591543, -0.15219119226102595, -0.7598912157762121]], [[-0.90956332777302396, 
-0.10955833859077013, 0.6649135519190148, -0.57746657860324735], [-0.019921200392668315, -0.70777890883173944, 
0.47962326795551902, 0.58757555444487441], [-0.8665727165391135, -0.87611823532523214, -0.38793769719568871, 
0.85610589809831183]]], [[[0.77200272262876735, 0.0037098363709966442, 0.99387194888657837, 0.3494515184202891], 
[-0.72958616039152324, 0.96944670676713685, -0.006253374601870032, -0.018096547195453283], [0.41754132743125272, 
0.33086367724606069, -0.27529259918889237, -0.30633432878374722]], [[0.66334426462768281, -0.94726367994136851, 
0.10911913469154411, 0.89104989766953868], [-0.1063190734973869, 0.078911289428541309, 0.49957092103283962, 
-0.73778001984841701], [-0.86299384831882842, 0.58737042896179137, 0.21143673144036601, -0.62130701205123984]]], 
[[[0.35506351814086479, 0.45092232689175282, 0.16925970871320795, 0.47406940636561812], [0.53818751206031901, 
0.93889883620185999, -0.79558165482196253, 0.62792107870495362], [0.11989219421739072, -0.71732760316640487, 
0.3239610675465181, -0.99104959828236661]], [[0.18035311137897203, -0.63337421992458909, -0.21054837143515481, 
0.60920122284201339], [-0.85183723100083641, 0.75783166690834491, 0.44873865199630925, 0.69285091955764977], 
[-0.87699217725458767, -0.64176974664742015, 0.32529932761770453, 0.5328938499677689]]]]))
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.45062987773571628, 0.45062987773571628, 0.45062987773571628, 0.45062987773571628], 
[0.45062987773571628, 0.45062987773571628, 0.79900860948329, 0.45062987773571628], [0.5108826808220337, 0.45062987773571628, 
0.45062987773571628, 0.45062987773571628]], [[0.45062987773571628, 0.45062987773571628, 0.45062987773571628, 
0.45062987773571628], [0.89281480055911766, 0.45062987773571628, 0.93180996621646917, 0.45062987773571628], 
[0.45062987773571628, 0.45062987773571628, 0.85505937142450295, 0.5378881600091503]]], [[[0.45062987773571628, 
0.45062987773571628, 0.45062987773571628, 0.5423072309997008], [0.45062987773571628, 0.98860123211230855, 0.90741603461179854, 
0.45062987773571628], [0.54808223548024415, 0.45062987773571628, 0.68978257581627056, 0.45062987773571628]], 
[[0.45062987773571628, 0.59210693779982515, 0.87444883555895481, 0.45062987773571628], [0.45062987773571628, 
0.56398716173861096, 0.45062987773571628, 0.45062987773571628], [0.45062987773571628, 0.95086139306366735, 0.45062987773571628, 
0.45062987773571628]]], [[[0.57335107011599828, 0.45062987773571628, 0.97431386356392546, 0.45062987773571628], 
[0.45062987773571628, 0.85412964511631384, 0.45062987773571628, 0.45062987773571628], [0.8075580531915918, 0.87134037440501655, 
0.45062987773571628, 0.45062987773571628]], [[0.45062987773571628, 0.7593111093249223, 0.45062987773571628, 
0.45062987773571628], [0.58468253560648864, 0.82466000389884875, 0.45062987773571628, 0.45062987773571628], 
[0.45062987773571628, 0.45062987773571628, 0.76350095350066249, 
0.45062987773571628]]]])+(1.-msk_ref)*numpy.array([[[[0.5794608208193015, 0.76021978616093389, 0.5794608208193015, 
0.5794608208193015], [0.58819854593101017, 0.5794608208193015, 0.5794608208193015, 0.96470130871039728], [0.5794608208193015, 
0.5794608208193015, 0.5794608208193015, 0.5794608208193015]], [[0.5794608208193015, 0.5794608208193015, 0.6649135519190148, 
0.5794608208193015], [0.5794608208193015, 0.5794608208193015, 0.5794608208193015, 0.58757555444487441], [0.5794608208193015, 
0.5794608208193015, 0.5794608208193015, 0.85610589809831183]]], [[[0.77200272262876735, 0.5794608208193015, 
0.99387194888657837, 0.5794608208193015], [0.5794608208193015, 0.96944670676713685, 0.5794608208193015, 0.5794608208193015], 
[0.5794608208193015, 0.5794608208193015, 0.5794608208193015, 0.5794608208193015]], [[0.66334426462768281, 0.5794608208193015, 
0.5794608208193015, 0.89104989766953868], [0.5794608208193015, 0.5794608208193015, 0.5794608208193015, 0.5794608208193015], 
[0.5794608208193015, 0.58737042896179137, 0.5794608208193015, 0.5794608208193015]]], [[[0.5794608208193015, 0.5794608208193015, 
0.5794608208193015, 0.5794608208193015], [0.5794608208193015, 0.93889883620185999, 0.5794608208193015, 0.62792107870495362], 
[0.5794608208193015, 0.5794608208193015, 0.5794608208193015, 0.5794608208193015]], [[0.5794608208193015, 0.5794608208193015, 
0.5794608208193015, 0.60920122284201339], [0.5794608208193015, 0.75783166690834491, 0.5794608208193015, 0.69285091955764977], 
[0.5794608208193015, 0.5794608208193015, 0.5794608208193015, 0.5794608208193015]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_taggedData_rank4(self):
      arg0=Data(numpy.array([[[[0.76955091692252009, -0.53121537020924503, 0.92331536295401873, -0.34367627453467331], 
[0.78723109623384668, 0.39458953116365492, 0.59502877122173681, 0.050372844530587102], [-0.19899713844924194, 
0.41152644602298483, 0.94677471664022517, 0.4876799155785223]], [[-0.24703339834170768, -0.94331939863901515, 
-0.88611729975454079, -0.40581722478832827], [0.029730838876508292, -0.38460076098005458, -0.17591045709031206, 
-0.71320405508614582], [-0.78628883956651152, -0.99007191820143547, 0.21418622927908659, -0.40818445968490713]]], 
[[[0.0042569312712867902, -0.25976403772271817, -0.92926712763264585, -0.79485232395125816], [0.86639872074193747, 
0.65318357533211779, -0.89911442494314642, -0.94473972318104615], [0.39754126543806034, -0.4010815317447014, 
0.56256299492880091, -0.23949565583407417]], [[0.82370891735547125, -0.85852746636505017, 0.29845825367811174, 
-0.28040293335963939], [-0.61552789972771471, 0.14297290291382159, 0.33854689551282213, -0.68819791199573888], 
[-0.037077917233494517, 0.42902512706506779, 0.45111029338748332, -0.27704393207309663]]], [[[-0.369691271262375, 
-0.074492575493751767, 0.20962012924176232, -0.43068635164595492], [-0.74226235220622327, -0.8736015539788955, 
0.11358838335880383, 0.004443406808822381], [-0.20248253955628082, 0.53718374549305725, -0.36808790294556393, 
-0.19406034651763182]], [[-0.16352139419534062, 0.078533157655853225, -0.53797014473836535, 0.60321275736829927], 
[-0.66057616988324241, -0.2118944680944983, 0.80243206808067136, -0.23643742948344371], [-0.28823401957596984, 
-0.32859411841026498, -0.52803065603921806, -0.53227609313995283]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.22893873837133638, -0.49032344510039905, -0.55698272642024715, 
0.71618665253896974], [0.60529046560589306, 0.26565627287768057, 0.92207632299530951, -0.40949122741896682], 
[-0.41264242762927905, -0.15102882820247232, 0.61915382485641035, -0.43730975896852309]], [[-0.85059990370747496, 
-0.34077663679356318, 0.23348722429510049, 0.31777151583548213], [0.41588343854033138, -0.27144967369935746, 
0.21249667702630326, 0.87427855430994206], [0.34943255691829012, 0.69192103905430069, 0.063709530598745179, 
0.20567366439114698]]], [[[0.96246207288049179, 0.24475471346016198, -0.22633498783383055, 0.146473569651703], 
[0.76997197890428226, 0.39220687810199539, 0.31182239497041775, -0.13463352690629304], [-0.067475434723136107, 
-0.57799434312011666, 0.69886245428531701, -0.069116225520234886]], [[0.092685825210803374, 0.021846384689953702, 
-0.50376649709604227, -0.62298523066650313], [0.92087793061697765, 0.86901049387336493, 0.68251311608741205, 
0.18148555381187048], [0.60720079482140465, -0.44678148870224432, -0.48119781664038652, -0.054439679635572169]]], 
[[[0.27848461453784745, -0.15652525408976947, -0.55869430457320268, -0.73858941451889715], [-0.39210312683432447, 
0.60836444957177926, -0.74841957373166634, 0.70011599175139705], [-0.092769041470482927, 0.33156247684724027, 
-0.68495653893819952, 0.35319609484609216]], [[-0.22495933585518757, -0.096475024360099582, 0.2355610004718216, 
0.17889993988415509], [-0.86491942178254, 0.54177935518757092, -0.63422377027012633, 0.86326923436841296], 
[0.14631001107532704, 0.44478026064068987, 0.75750928188365085, -0.73386122551657351]]]]))
      arg0.expand()
      arg1=Data(numpy.array([[[[0.91290552097700961, 0.96789553150377028, 0.5661915230390262, 0.89714680715674189], 
[0.29992278520617788, 0.28755083362727496, 0.83096466656338142, 0.82333208753058562], [-0.078770147403909752, 
0.62503017271004491, -0.62979893293063482, -0.99508655286186998]], [[0.31757643511433842, -0.45348384093288541, 
0.72962857906377487, 0.53449962172469667], [-0.46890067225838106, 0.83138237158744688, -0.69273616988371511, 
-0.99605462794789057], [-0.49834757578929834, -0.1134992107172359, -0.50010087765904365, 0.98861392243258051]]], 
[[[-0.74185912566710188, 0.36515059841721165, -0.075699256695932471, -0.23203876779367794], [-0.1740338386390734, 
-0.76410429808447922, 0.21767257898811554, 0.59540241770907931], [-0.85459236163275376, 0.6537143985414422, 
-0.37696767012777999, 0.47291717166063463]], [[0.71337854110853649, -0.85661217912548882, -0.014994334384107244, 
-0.45117484346745385], [-0.17315566873478483, 0.39028491927421438, 0.51966032402965601, -0.76330002225642324], 
[-0.79767631128999539, -0.56314787584153136, 0.37540611842940774, -0.24913563575056741]]], [[[-0.45432811231748649, 
-0.26720600757130697, -0.79997604759698082, -0.23031243943826785], [-0.76477573937415166, 0.43777080081593645, 
-0.13968903747072337, -0.59451523956293006], [0.026345511656530318, -0.67031213472108453, 0.51762859567908182, 
0.76037417901906479]], [[-0.97241171971550711, 0.99517960221360946, 0.85174195373738004, -0.57165876215219691], 
[-0.81856970336308632, -0.16628617968552795, -0.69999943836543976, 0.50540871726773329], [0.17351700489486732, 
0.21749504868276825, 0.41234604437155831, 0.75119729004736091]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.20221150611649064, 0.73543647417815916, -0.70506602619258829, 
0.97888830472663924], [-0.042473056413596666, -0.24031177393964387, 0.61387827889649693, 0.16788650420211271], 
[0.081750874778339755, -0.17016812549515126, -0.23293652933024123, -0.42891126898530652]], [[-0.069632045523091213, 
-0.50946504220329247, 0.49041692338883713, 0.92752077966828472], [-0.14923501242296999, 0.15792377649587719, 
0.5750229888370284, -0.55892720436307552], [0.691679244310895, 0.87540802150655361, 0.35034306590253417, 
0.89165513560989673]]], [[[0.98075125166370025, 0.43447886286465809, 0.043444723860348367, 0.53661282977717994], 
[-0.91869101586427027, -0.04378102201555234, 0.016213763078045229, 0.051842716078229545], [0.19528495664367918, 
0.27828591208281717, -0.26927916824350495, 0.77252153037147786]], [[-0.85061031838009438, -0.8017408311567813, 
0.72728115638694568, -0.62571931677148807], [-0.20249642479494723, -0.95831749554600032, 0.72112511685081904, 
0.175876101828081], [-0.11501949288432667, 0.88291274097133776, 0.23600578977840847, 0.20437019632383424]]], 
[[[-0.86143674311986329, 0.56267400173528315, 0.12179348205187379, 0.55380088835123287], [0.05963080459609138, 
-0.98314564757026535, 0.92094209135081018, -0.5441422501105595], [-0.43506249239884576, -0.14306290639549979, 
0.015644877962105008, 0.91126736181798007]], [[0.034198797224265887, -0.23845077679811655, -0.18418503551626553, 
-0.76966425306439756], [-0.17138782453655388, -0.51812286796946161, 0.81256262258343193, -0.47681509006956491], 
[-0.41117855881499032, 0.047955510684507763, -0.070536658862883517, 0.63977421617673258]]]]))
      res=maximum(arg0,arg1)
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.91290552097700961, 0.96789553150377028, 0.92331536295401873, 0.89714680715674189], 
[0.78723109623384668, 0.39458953116365492, 0.83096466656338142, 0.82333208753058562], [-0.078770147403909752, 
0.62503017271004491, 0.94677471664022517, 0.4876799155785223]], [[0.31757643511433842, -0.45348384093288541, 
0.72962857906377487, 0.53449962172469667], [0.029730838876508292, 0.83138237158744688, -0.17591045709031206, 
-0.71320405508614582], [-0.49834757578929834, -0.1134992107172359, 0.21418622927908659, 0.98861392243258051]]], 
[[[0.0042569312712867902, 0.36515059841721165, -0.075699256695932471, -0.23203876779367794], [0.86639872074193747, 
0.65318357533211779, 0.21767257898811554, 0.59540241770907931], [0.39754126543806034, 0.6537143985414422, 0.56256299492880091, 
0.47291717166063463]], [[0.82370891735547125, -0.85661217912548882, 0.29845825367811174, -0.28040293335963939], 
[-0.17315566873478483, 0.39028491927421438, 0.51966032402965601, -0.68819791199573888], [-0.037077917233494517, 
0.42902512706506779, 0.45111029338748332, -0.24913563575056741]]], [[[-0.369691271262375, -0.074492575493751767, 
0.20962012924176232, -0.23031243943826785], [-0.74226235220622327, 0.43777080081593645, 0.11358838335880383, 
0.004443406808822381], [0.026345511656530318, 0.53718374549305725, 0.51762859567908182, 0.76037417901906479]], 
[[-0.16352139419534062, 0.99517960221360946, 0.85174195373738004, 0.60321275736829927], [-0.66057616988324241, 
-0.16628617968552795, 0.80243206808067136, 0.50540871726773329], [0.17351700489486732, 0.21749504868276825, 
0.41234604437155831, 0.75119729004736091]]]])+(1.-msk_ref)*numpy.array([[[[0.22893873837133638, 0.73543647417815916, 
-0.55698272642024715, 0.97888830472663924], [0.60529046560589306, 0.26565627287768057, 0.92207632299530951, 
0.16788650420211271], [0.081750874778339755, -0.15102882820247232, 0.61915382485641035, -0.42891126898530652]], 
[[-0.069632045523091213, -0.34077663679356318, 0.49041692338883713, 0.92752077966828472], [0.41588343854033138, 
0.15792377649587719, 0.5750229888370284, 0.87427855430994206], [0.691679244310895, 0.87540802150655361, 0.35034306590253417, 
0.89165513560989673]]], [[[0.98075125166370025, 0.43447886286465809, 0.043444723860348367, 0.53661282977717994], 
[0.76997197890428226, 0.39220687810199539, 0.31182239497041775, 0.051842716078229545], [0.19528495664367918, 
0.27828591208281717, 0.69886245428531701, 0.77252153037147786]], [[0.092685825210803374, 0.021846384689953702, 
0.72728115638694568, -0.62298523066650313], [0.92087793061697765, 0.86901049387336493, 0.72112511685081904, 
0.18148555381187048], [0.60720079482140465, 0.88291274097133776, 0.23600578977840847, 0.20437019632383424]]], 
[[[0.27848461453784745, 0.56267400173528315, 0.12179348205187379, 0.55380088835123287], [0.05963080459609138, 
0.60836444957177926, 0.92094209135081018, 0.70011599175139705], [-0.092769041470482927, 0.33156247684724027, 
0.015644877962105008, 0.91126736181798007]], [[0.034198797224265887, -0.096475024360099582, 0.2355610004718216, 
0.17889993988415509], [-0.17138782453655388, 0.54177935518757092, 0.81256262258343193, 0.86326923436841296], 
[0.14631001107532704, 0.44478026064068987, 0.75750928188365085, 0.63977421617673258]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_taggedData_rank0(self):
      arg0=0.314410800242
      arg1=Data(-0.577024414806,self.functionspace)
      arg1.setTaggedValue(1,0.964775347448)
      res=minimum(arg0,arg1)
      ref=Data(-0.577024414806,self.functionspace)
      ref.setTaggedValue(1,0.314410800242)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_taggedData_rank1(self):
      arg0=-0.515997945622
      arg1=Data(numpy.array([-0.64331923939785285, -0.83496733964576708]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.9282790514923196, -0.15112956150860923]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([-0.64331923939785285, -0.83496733964576708]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.51599794562235357, -0.51599794562235357]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_taggedData_rank2(self):
      arg0=0.959434014116
      arg1=Data(numpy.array([[-0.82865643130235433, -0.86989432976922765, 0.78539270692837082, -0.3296068834342365, 
-0.59752452029779235], [0.45399470911674844, 0.89720495650475707, -0.27922321552613094, -0.95581005927706841, 
-0.025799264642782038], [0.60866190386723429, 0.6393901864959548, 0.018062705447958294, 0.17843032126338887, 
-0.80827623740659926], [-0.42832263621583189, 0.019812411062545721, -0.13572887746655038, -0.83792574167706202, 
0.005010313213395845]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.98861548397887522, -0.73569853435701704, 0.80664186007766214, 0.078577026539418782, 
-0.17440451686191838], [0.32816757792655271, -0.88534578669733222, -0.48904232538334602, -0.94845276964437808, 
-0.57891853329910492], [-0.30705592020091399, 0.84474071634453729, 0.5699908556937392, 0.41989954790660988, 
-0.29492857264065675], [-0.24205776724643568, -0.97213289794262114, 0.89060261029711829, -0.19066848137499925, 
0.95440668495914882]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.82865643130235433, -0.86989432976922765, 0.78539270692837082, -0.3296068834342365, 
-0.59752452029779235], [0.45399470911674844, 0.89720495650475707, -0.27922321552613094, -0.95581005927706841, 
-0.025799264642782038], [0.60866190386723429, 0.6393901864959548, 0.018062705447958294, 0.17843032126338887, 
-0.80827623740659926], [-0.42832263621583189, 0.019812411062545721, -0.13572887746655038, -0.83792574167706202, 
0.005010313213395845]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.95943401411573781, -0.73569853435701704, 0.80664186007766214, 0.078577026539418782, 
-0.17440451686191838], [0.32816757792655271, -0.88534578669733222, -0.48904232538334602, -0.94845276964437808, 
-0.57891853329910492], [-0.30705592020091399, 0.84474071634453729, 0.5699908556937392, 0.41989954790660988, 
-0.29492857264065675], [-0.24205776724643568, -0.97213289794262114, 0.89060261029711829, -0.19066848137499925, 
0.95440668495914882]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_taggedData_rank3(self):
      arg0=0.343560195642
      arg1=Data(numpy.array([[[-0.73752299737078775, 0.39826638960056715], [0.25261602074959999, 0.50844838788569269]], 
[[-0.55422873481124335, -0.62208683127921471], [0.44032548263239413, 0.64427653780141836]], [[-0.48004968311606921, 
0.97684469371737292], [0.21464445505671104, -0.45535623764597899]], [[-0.28118124517849785, -0.042127054721157187], 
[-0.85380673186879164, -0.18520435409349867]], [[-0.52645407926751542, -0.91188357179212498], [0.41719397724901097, 
0.19383508951655326]], [[-0.74986133429957991, -0.52359829593459484], [-0.12817264546573459, 
0.63978256050128035]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.081728623698809733, 0.17906755292231713], [0.047824950970851265, 
-0.57962966450962194]], [[-0.013175077410390479, -0.38844365501858635], [-0.89612886718668916, -0.76039587634523853]], 
[[0.039392437174874839, 0.5988089324356789], [-0.17168318204191513, 0.036963051615617193]], [[0.80259229240606844, 
0.2685081745207798], [0.67724833404577156, -0.63384185480486144]], [[0.92126110395884275, 0.4947193346772103], 
[0.85131094557460307, 0.44452093214705934]], [[-0.59417652826563017, 0.90088781157777698], [-0.7949098617674546, 
-0.27667140575417437]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[-0.73752299737078775, 0.34356019564188056], [0.25261602074959999, 0.34356019564188056]], 
[[-0.55422873481124335, -0.62208683127921471], [0.34356019564188056, 0.34356019564188056]], [[-0.48004968311606921, 
0.34356019564188056], [0.21464445505671104, -0.45535623764597899]], [[-0.28118124517849785, -0.042127054721157187], 
[-0.85380673186879164, -0.18520435409349867]], [[-0.52645407926751542, -0.91188357179212498], [0.34356019564188056, 
0.19383508951655326]], [[-0.74986133429957991, -0.52359829593459484], [-0.12817264546573459, 
0.34356019564188056]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.081728623698809733, 0.17906755292231713], [0.047824950970851265, 
-0.57962966450962194]], [[-0.013175077410390479, -0.38844365501858635], [-0.89612886718668916, -0.76039587634523853]], 
[[0.039392437174874839, 0.34356019564188056], [-0.17168318204191513, 0.036963051615617193]], [[0.34356019564188056, 
0.2685081745207798], [0.34356019564188056, -0.63384185480486144]], [[0.34356019564188056, 0.34356019564188056], 
[0.34356019564188056, 0.34356019564188056]], [[-0.59417652826563017, 0.34356019564188056], [-0.7949098617674546, 
-0.27667140575417437]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_taggedData_rank4(self):
      arg0=0.0632747947091
      arg1=Data(numpy.array([[[[0.87238226406626684, -0.45846347040032809, 0.0023464448190655407, -0.37591518354903264], 
[0.55817899856936481, -0.45194091104291845, 0.30775271128008574, 0.15505497086672482], [0.89696571776698653, 
-0.55294720136226161, 0.99421147223002526, -0.41148243178855859]], [[-0.39646904563533836, -0.93352763924776627, 
0.61438604341774927, 0.51661611163655774], [0.14200882041878593, 0.49107306933069172, -0.81440598856715307, 
0.9959896368711707], [0.43157857160005064, 0.090814591976436709, 0.74883972448246316, 0.80580641609047521]]], 
[[[-0.39913744414275398, -0.47515014279264212, -0.51219666797955554, -0.47081690458262626], [0.72936156620842185, 
-0.94807839964718066, -0.96958941305753199, -0.38643871103607341], [0.83195795924518201, 0.80082248782250898, 
0.69661215359228423, -0.54283844005609172]], [[0.42856329004324856, 0.72663668404545834, 0.12183478498444322, 
0.94807111171478575], [-0.60035358939179129, 0.77651141624611975, 0.042261687417343596, -0.61442516969369865], 
[0.67595484862207078, 0.25088974744681791, -0.76714966778968829, -0.59288410608547348]]], [[[0.69698091451590694, 
0.71169947419023383, 0.57848500306922412, -0.43511881151463072], [0.46396502461574118, -0.99365569043960345, -0.67597116460633, 
0.99120564943885592], [0.43121277811008163, -0.20451945415268025, 0.20630020117137304, 0.32413195254649829]], 
[[-0.66050181102977823, 0.48358288676150396, -0.50558621946335136, 0.011381847457725414], [-0.98805498927891167, 
-0.66499533378745235, -0.1434243355116922, -0.48380644484363788], [-0.12553880722863431, 0.14159374452497153, 
0.79150568900005203, -0.90283886547542691]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.71475588746426078, 0.99143332186761679, -0.51739944603141752, 
0.67913058075763555], [0.35164679998427495, -0.51524338404955872, -0.32136539713493439, 0.33096253681463117], 
[-0.65394572409524043, -0.6195278956843242, -0.21886260195976992, 0.76373424534136114]], [[-0.42884215689516414, 
-0.83124535255341181, 0.19483645496670787, 0.76930158892093914], [0.41854727595618457, 0.60924937597464623, 
-0.44574269238184905, -0.62493158805228077], [-0.96456141069981371, 0.78140099292967657, -0.94500425468556148, 
0.4790767667719078]]], [[[0.49939391649850551, -0.17343560513503564, -0.39444193063498822, -0.9637293728948686], 
[-0.73950545804405854, -0.18319256411268414, -0.90160189420914794, 0.89314960346073513], [-0.93177705110266507, 
-0.96358452422353014, 0.67858177722377011, 0.67894941054537772]], [[0.71580112488656344, -0.73345408801105383, 
-0.58361518632461862, -0.72516644734241997], [-0.65189216870623845, 0.15779867217473842, -0.4709865711774579, 
-0.81135557168013439], [-0.51223145159549111, 0.69504366056104239, -0.31692557885043482, -0.898628820656729]]], 
[[[-0.32366051426516185, -0.75792137329597931, 0.98488216633372572, 0.36852846523016991], [-0.3678142333385892, 
0.43865958144671935, 0.65914988652001871, 0.61233533499616688], [-0.32361378959661469, 0.076780425063808, -0.20490340108832195, 
0.7081414422184793]], [[-0.31045314804213242, -0.73456413272403998, 0.42435052137265927, -0.6717759746480747], 
[-0.043444555765702164, 0.44329835902102932, -0.86102564369737866, -0.2529972549806716], [-0.61105551262077018, 
-0.31856318802178119, 0.9670324004382822, 0.89913580753121702]]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[0.063274794709083482, -0.45846347040032809, 0.0023464448190655407, -0.37591518354903264], 
[0.063274794709083482, -0.45194091104291845, 0.063274794709083482, 0.063274794709083482], [0.063274794709083482, 
-0.55294720136226161, 0.063274794709083482, -0.41148243178855859]], [[-0.39646904563533836, -0.93352763924776627, 
0.063274794709083482, 0.063274794709083482], [0.063274794709083482, 0.063274794709083482, -0.81440598856715307, 
0.063274794709083482], [0.063274794709083482, 0.063274794709083482, 0.063274794709083482, 0.063274794709083482]]], 
[[[-0.39913744414275398, -0.47515014279264212, -0.51219666797955554, -0.47081690458262626], [0.063274794709083482, 
-0.94807839964718066, -0.96958941305753199, -0.38643871103607341], [0.063274794709083482, 0.063274794709083482, 
0.063274794709083482, -0.54283844005609172]], [[0.063274794709083482, 0.063274794709083482, 0.063274794709083482, 
0.063274794709083482], [-0.60035358939179129, 0.063274794709083482, 0.042261687417343596, -0.61442516969369865], 
[0.063274794709083482, 0.063274794709083482, -0.76714966778968829, -0.59288410608547348]]], [[[0.063274794709083482, 
0.063274794709083482, 0.063274794709083482, -0.43511881151463072], [0.063274794709083482, -0.99365569043960345, 
-0.67597116460633, 0.063274794709083482], [0.063274794709083482, -0.20451945415268025, 0.063274794709083482, 
0.063274794709083482]], [[-0.66050181102977823, 0.063274794709083482, -0.50558621946335136, 0.011381847457725414], 
[-0.98805498927891167, -0.66499533378745235, -0.1434243355116922, -0.48380644484363788], [-0.12553880722863431, 
0.063274794709083482, 0.063274794709083482, -0.90283886547542691]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.71475588746426078, 0.063274794709083482, -0.51739944603141752, 
0.063274794709083482], [0.063274794709083482, -0.51524338404955872, -0.32136539713493439, 0.063274794709083482], 
[-0.65394572409524043, -0.6195278956843242, -0.21886260195976992, 0.063274794709083482]], [[-0.42884215689516414, 
-0.83124535255341181, 0.063274794709083482, 0.063274794709083482], [0.063274794709083482, 0.063274794709083482, 
-0.44574269238184905, -0.62493158805228077], [-0.96456141069981371, 0.063274794709083482, -0.94500425468556148, 
0.063274794709083482]]], [[[0.063274794709083482, -0.17343560513503564, -0.39444193063498822, -0.9637293728948686], 
[-0.73950545804405854, -0.18319256411268414, -0.90160189420914794, 0.063274794709083482], [-0.93177705110266507, 
-0.96358452422353014, 0.063274794709083482, 0.063274794709083482]], [[0.063274794709083482, -0.73345408801105383, 
-0.58361518632461862, -0.72516644734241997], [-0.65189216870623845, 0.063274794709083482, -0.4709865711774579, 
-0.81135557168013439], [-0.51223145159549111, 0.063274794709083482, -0.31692557885043482, -0.898628820656729]]], 
[[[-0.32366051426516185, -0.75792137329597931, 0.063274794709083482, 0.063274794709083482], [-0.3678142333385892, 
0.063274794709083482, 0.063274794709083482, 0.063274794709083482], [-0.32361378959661469, 0.063274794709083482, 
-0.20490340108832195, 0.063274794709083482]], [[-0.31045314804213242, -0.73456413272403998, 0.063274794709083482, 
-0.6717759746480747], [-0.043444555765702164, 0.063274794709083482, -0.86102564369737866, -0.2529972549806716], 
[-0.61105551262077018, -0.31856318802178119, 0.063274794709083482, 0.063274794709083482]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank0_taggedData_rank0(self):
      arg0=numpy.array(-0.20252478412)
      arg1=Data(-0.46028111512,self.functionspace)
      arg1.setTaggedValue(1,-0.455126378888)
      res=minimum(arg0,arg1)
      ref=Data(-0.46028111512,self.functionspace)
      ref.setTaggedValue(1,-0.455126378888)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank1_taggedData_rank0(self):
      arg0=numpy.array([-0.72037285165823328, 0.95503431920973725])
      arg1=Data(0.970658979843,self.functionspace)
      arg1.setTaggedValue(1,0.839322765094)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([-0.72037285165823328, 0.95503431920973725]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.72037285165823328, 0.83932276509393922]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank2_taggedData_rank0(self):
      arg0=numpy.array([[0.68790811289178388, 0.17678941910803614, -0.41927624429074006, 0.60551635925208736, 
-0.0073344075085466365], [0.046816913771521529, 0.24513811064348356, 0.017612150222259926, -0.17115776083565448, 
0.37475881757124885], [0.35035503835119086, -0.89972054188734929, -0.38758518050033808, -0.22163646159226169, 
-0.22785697903304913], [-0.30412878337396854, -0.63297420372891988, 0.26105300074372151, -0.7459870933271191, 
-0.78211306238154688]])
      arg1=Data(0.605901594887,self.functionspace)
      arg1.setTaggedValue(1,-0.23244763727)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[0.60590159488701123, 0.17678941910803614, -0.41927624429074006, 0.60551635925208736, 
-0.0073344075085466365], [0.046816913771521529, 0.24513811064348356, 0.017612150222259926, -0.17115776083565448, 
0.37475881757124885], [0.35035503835119086, -0.89972054188734929, -0.38758518050033808, -0.22163646159226169, 
-0.22785697903304913], [-0.30412878337396854, -0.63297420372891988, 0.26105300074372151, -0.7459870933271191, 
-0.78211306238154688]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.23244763726974238, -0.23244763726974238, -0.41927624429074006, 
-0.23244763726974238, -0.23244763726974238], [-0.23244763726974238, -0.23244763726974238, -0.23244763726974238, 
-0.23244763726974238, -0.23244763726974238], [-0.23244763726974238, -0.89972054188734929, -0.38758518050033808, 
-0.23244763726974238, -0.23244763726974238], [-0.30412878337396854, -0.63297420372891988, -0.23244763726974238, 
-0.7459870933271191, -0.78211306238154688]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank3_taggedData_rank0(self):
      arg0=numpy.array([[[-0.0028994602095799848, 0.6529286086211612], [-0.28098282832212296, 0.94733416730102871]], 
[[-0.088819657346991931, -0.58143097558931278], [-0.23961579693744972, 0.20032045218946992]], [[-0.91749971994417612, 
0.89170403190697067], [0.17872742332625124, -0.36455219743560319]], [[0.28285275438832458, 0.10929072006405671], 
[-0.65666535622897149, 0.32116199707133908]], [[0.31762421552959674, -0.44102896645837419], [0.6319923089971482, 
0.56009029939432109]], [[0.027818654688745559, -0.12753390561676259], [0.01688503944756703, -0.47393981980602917]]])
      arg1=Data(0.714420224537,self.functionspace)
      arg1.setTaggedValue(1,0.0644888589294)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[-0.0028994602095799848, 0.6529286086211612], [-0.28098282832212296, 0.71442022453675591]], 
[[-0.088819657346991931, -0.58143097558931278], [-0.23961579693744972, 0.20032045218946992]], [[-0.91749971994417612, 
0.71442022453675591], [0.17872742332625124, -0.36455219743560319]], [[0.28285275438832458, 0.10929072006405671], 
[-0.65666535622897149, 0.32116199707133908]], [[0.31762421552959674, -0.44102896645837419], [0.6319923089971482, 
0.56009029939432109]], [[0.027818654688745559, -0.12753390561676259], [0.01688503944756703, 
-0.47393981980602917]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.0028994602095799848, 0.064488858929440562], [-0.28098282832212296, 
0.064488858929440562]], [[-0.088819657346991931, -0.58143097558931278], [-0.23961579693744972, 0.064488858929440562]], 
[[-0.91749971994417612, 0.064488858929440562], [0.064488858929440562, -0.36455219743560319]], [[0.064488858929440562, 
0.064488858929440562], [-0.65666535622897149, 0.064488858929440562]], [[0.064488858929440562, -0.44102896645837419], 
[0.064488858929440562, 0.064488858929440562]], [[0.027818654688745559, -0.12753390561676259], [0.01688503944756703, 
-0.47393981980602917]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank4_taggedData_rank0(self):
      arg0=numpy.array([[[[-0.95003081240586251, -0.086367597986508837, 0.82525936987273618, 0.85953101241538588], 
[0.31948730544154857, 0.7753946500866411, 0.75007381356868352, 0.75396760694678022], [-0.47112620993376253, 
-0.047074523000005142, -0.52266906246601752, -0.32494414840371055]], [[-0.58220951111906261, -0.055595353413821602, 
-0.20550610088269461, 0.4277756650464084], [-0.17473866838743324, 0.87514098564269993, -0.30559217253475146, 
0.96916407322340348], [0.74679708120675059, 0.4529194492289077, 0.92187158067767849, -0.42419702603924714]]], 
[[[0.7407660740777009, -0.05636688555057523, -0.86533489319575874, 0.1769581309189161], [-0.62803639730053695, 
-0.91314434220294505, 0.60968780325061922, -0.87458044450145089], [-0.34980834403363659, 0.65053250718375599, 
-0.22817275302420326, 0.77728394303886161]], [[-0.042866731292045435, -0.35996284640160581, 0.72258986785769319, 
0.59548688486958112], [-0.2267629396389681, 0.84450284822613297, -0.24615898484893872, 0.3340570574936288], 
[0.5658345572110437, 0.73397872443131695, 0.59418815345828513, 0.0081730536508146656]]], [[[0.55974514608672554, 
-0.67409551749498653, 0.098084779245508846, 0.077505494645091932], [0.93681287662757717, 0.10180072085412006, 
-0.62046563506403785, 0.92024870461246766], [0.19292450384570858, -0.76644825385427584, -0.97099453984172857, 
0.64621148000496342]], [[-0.33700688604334994, -0.49452500927743048, -0.62948775841611537, -0.36593801966908845], 
[0.4380851249823845, 0.56966895305049303, 0.34213507066098336, -0.094678658488436485], [-0.96255362531891753, 
0.030110984082165171, -0.41425734220559174, -0.95520666128723719]]]])
      arg1=Data(-0.929563528633,self.functionspace)
      arg1.setTaggedValue(1,0.79107831908)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.95003081240586251, -0.92956352863349068, -0.92956352863349068, -0.92956352863349068], 
[-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, -0.92956352863349068], [-0.92956352863349068, 
-0.92956352863349068, -0.92956352863349068, -0.92956352863349068]], [[-0.92956352863349068, -0.92956352863349068, 
-0.92956352863349068, -0.92956352863349068], [-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, 
-0.92956352863349068], [-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, -0.92956352863349068]]], 
[[[-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, -0.92956352863349068], [-0.92956352863349068, 
-0.92956352863349068, -0.92956352863349068, -0.92956352863349068], [-0.92956352863349068, -0.92956352863349068, 
-0.92956352863349068, -0.92956352863349068]], [[-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, 
-0.92956352863349068], [-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, -0.92956352863349068], 
[-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, -0.92956352863349068]]], [[[-0.92956352863349068, 
-0.92956352863349068, -0.92956352863349068, -0.92956352863349068], [-0.92956352863349068, -0.92956352863349068, 
-0.92956352863349068, -0.92956352863349068], [-0.92956352863349068, -0.92956352863349068, -0.97099453984172857, 
-0.92956352863349068]], [[-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, -0.92956352863349068], 
[-0.92956352863349068, -0.92956352863349068, -0.92956352863349068, -0.92956352863349068], [-0.96255362531891753, 
-0.92956352863349068, -0.92956352863349068, -0.95520666128723719]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.95003081240586251, -0.086367597986508837, 0.79107831908040183, 
0.79107831908040183], [0.31948730544154857, 0.7753946500866411, 0.75007381356868352, 0.75396760694678022], 
[-0.47112620993376253, -0.047074523000005142, -0.52266906246601752, -0.32494414840371055]], [[-0.58220951111906261, 
-0.055595353413821602, -0.20550610088269461, 0.4277756650464084], [-0.17473866838743324, 0.79107831908040183, 
-0.30559217253475146, 0.79107831908040183], [0.74679708120675059, 0.4529194492289077, 0.79107831908040183, 
-0.42419702603924714]]], [[[0.7407660740777009, -0.05636688555057523, -0.86533489319575874, 0.1769581309189161], 
[-0.62803639730053695, -0.91314434220294505, 0.60968780325061922, -0.87458044450145089], [-0.34980834403363659, 
0.65053250718375599, -0.22817275302420326, 0.77728394303886161]], [[-0.042866731292045435, -0.35996284640160581, 
0.72258986785769319, 0.59548688486958112], [-0.2267629396389681, 0.79107831908040183, -0.24615898484893872, 
0.3340570574936288], [0.5658345572110437, 0.73397872443131695, 0.59418815345828513, 0.0081730536508146656]]], 
[[[0.55974514608672554, -0.67409551749498653, 0.098084779245508846, 0.077505494645091932], [0.79107831908040183, 
0.10180072085412006, -0.62046563506403785, 0.79107831908040183], [0.19292450384570858, -0.76644825385427584, 
-0.97099453984172857, 0.64621148000496342]], [[-0.33700688604334994, -0.49452500927743048, -0.62948775841611537, 
-0.36593801966908845], [0.4380851249823845, 0.56966895305049303, 0.34213507066098336, -0.094678658488436485], 
[-0.96255362531891753, 0.030110984082165171, -0.41425734220559174, -0.95520666128723719]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank0_taggedData_rank1(self):
      arg0=numpy.array(-0.828880570789)
      arg1=Data(numpy.array([0.64177570395415229, -0.31114428674347105]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.97776757662637648, -0.71646498675377002]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([-0.828880570789158, -0.828880570789158]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.828880570789158, -0.828880570789158]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank1_taggedData_rank1(self):
      arg0=numpy.array([0.24938067058240132, -0.48026019808279363])
      arg1=Data(numpy.array([-0.65307384532077273, -0.42882409479737471]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([0.70847337578671321, -0.41800412668037157]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([-0.65307384532077273, -0.48026019808279363]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.24938067058240132, -0.48026019808279363]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank0_taggedData_rank2(self):
      arg0=numpy.array(-0.193118221984)
      arg1=Data(numpy.array([[-0.94675529095071509, 0.92530161828688451, 0.40386674715870008, -0.85353337318869071, 
-0.79232014116742322], [-0.25706735123401492, -0.11902896463516655, 0.52301264414040993, 0.077834454814631249, 
0.68733853376048692], [0.79287392172825699, 0.40566275274315577, -0.49565085745319926, -0.49250460867430013, 
0.25864322079351165], [-0.14615002893947571, -0.89456694320425045, -0.31822856058640814, -0.45289419927664909, 
-0.3188699472009453]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.88277144694232446, 0.49261305471843309, -0.62932625759247451, 0.21494504956811067, 
-0.21359697146048195], [0.056170266828586168, 0.31179416765764256, 0.31775300511305193, -0.42263188092404702, 
0.22102900619820609], [-0.89723443521624646, 0.68550705628858011, -0.58301359456940371, 0.93550407187430373, 
-0.81092289770563442], [-0.23476401118657142, 0.28159797090006133, -0.11458369458827811, -0.7253636969757904, 
0.35024271780680216]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.94675529095071509, -0.19311822198370843, -0.19311822198370843, -0.85353337318869071, 
-0.79232014116742322], [-0.25706735123401492, -0.19311822198370843, -0.19311822198370843, -0.19311822198370843, 
-0.19311822198370843], [-0.19311822198370843, -0.19311822198370843, -0.49565085745319926, -0.49250460867430013, 
-0.19311822198370843], [-0.19311822198370843, -0.89456694320425045, -0.31822856058640814, -0.45289419927664909, 
-0.3188699472009453]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.19311822198370843, -0.19311822198370843, -0.62932625759247451, 
-0.19311822198370843, -0.21359697146048195], [-0.19311822198370843, -0.19311822198370843, -0.19311822198370843, 
-0.42263188092404702, -0.19311822198370843], [-0.89723443521624646, -0.19311822198370843, -0.58301359456940371, 
-0.19311822198370843, -0.81092289770563442], [-0.23476401118657142, -0.19311822198370843, -0.19311822198370843, 
-0.7253636969757904, -0.19311822198370843]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank2_taggedData_rank2(self):
      arg0=numpy.array([[-0.37133591867434013, -0.10105924023572754, 0.54566918734681269, -0.85184601828884565, 
0.25915795839039246], [-0.099825368868847519, 0.81746594046825583, -0.80649876256962649, -0.86577960197851289, 
-0.93845071691824167], [0.89510061440327271, 0.83081746482459851, 0.95976889324839076, 0.40310677493764668, 
-0.87561111394252622], [0.060210106298779653, 0.44955126888006758, 0.20775677293258998, 0.8456927311886433, 
-0.56480847644495213]])
      arg1=Data(numpy.array([[0.09871480861370352, -0.87906967067563269, -0.62821615377818785, -0.80847007660798376, 
0.8979279544626344], [-0.70867100567303964, 0.17851759437367876, 0.76036093479947353, 0.82917823517526279, 
-0.80406762019974209], [0.78325247098989093, -0.8066813170823568, -0.80869245397111356, -0.04733038011938473, 
-0.71370138472671085], [0.23969373758081458, 0.5943770398313093, -0.87481887020553395, -0.75938564526286179, 
0.033762430835217794]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.36175842694602234, -0.097988748663490499, 0.43988114574383275, 
-0.44877558014360708, 0.5289963239492308], [0.14474139217167026, -0.51848179046143716, 0.82758834602040321, 
-0.14062538411358427, 0.55082843312039165], [0.35420526378711403, -0.61578423648467506, 0.43849373792402058, 
-0.31408245301632021, 0.28477663837511979], [0.83391327884325128, 0.97408210265889883, 0.79723217034214588, 
0.54307327569348329, 0.57553765901544995]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.37133591867434013, -0.87906967067563269, -0.62821615377818785, -0.85184601828884565, 
0.25915795839039246], [-0.70867100567303964, 0.17851759437367876, -0.80649876256962649, -0.86577960197851289, 
-0.93845071691824167], [0.78325247098989093, -0.8066813170823568, -0.80869245397111356, -0.04733038011938473, 
-0.87561111394252622], [0.060210106298779653, 0.44955126888006758, -0.87481887020553395, -0.75938564526286179, 
-0.56480847644495213]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.37133591867434013, -0.10105924023572754, 0.43988114574383275, -0.85184601828884565, 
0.25915795839039246], [-0.099825368868847519, -0.51848179046143716, -0.80649876256962649, -0.86577960197851289, 
-0.93845071691824167], [0.35420526378711403, -0.61578423648467506, 0.43849373792402058, -0.31408245301632021, 
-0.87561111394252622], [0.060210106298779653, 0.44955126888006758, 0.20775677293258998, 0.54307327569348329, 
-0.56480847644495213]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank0_taggedData_rank3(self):
      arg0=numpy.array(0.203029524701)
      arg1=Data(numpy.array([[[0.17094001119275259, 0.067140826306332446], [0.44532990064370725, 0.51789389752579273]], 
[[0.82114346947936911, 0.10282232389889834], [0.04866550530690894, -0.20870869506574818]], [[0.19441974115863947, 
-0.28344758490329602], [0.7571429611708258, -0.50812577666110004]], [[-0.069146843255929236, 0.062357934842570906], 
[0.05471633956068267, -0.34430968979544341]], [[-0.20409525100284021, 0.35148533762486656], [0.7592971485610629, 
-0.76273753864705096]], [[-0.80960417963744202, 0.62113007843213186], [0.34357986875124658, 
0.17567196898373627]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.013736771881341703, 0.73068164404186953], [0.32093216542863856, 
0.087462380141183349]], [[-0.28973481635429255, 0.40987092598089125], [0.79279816248875612, -0.17486873131021241]], 
[[-0.95107631719717256, 0.55203732857972532], [0.0029821337466202635, 0.78351724940563394]], [[0.27025999112705001, 
-0.32649927600308448], [0.22667864106355951, -0.5051158980483037]], [[-0.58727814097138942, -0.71676326712662219], 
[0.56692487594097396, 0.17759101180604486]], [[-0.13796018760230688, 0.1467627194153307], [0.60618280867324836, 
0.45343725498505982]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[0.17094001119275259, 0.067140826306332446], [0.20302952470092794, 0.20302952470092794]], 
[[0.20302952470092794, 0.10282232389889834], [0.04866550530690894, -0.20870869506574818]], [[0.19441974115863947, 
-0.28344758490329602], [0.20302952470092794, -0.50812577666110004]], [[-0.069146843255929236, 0.062357934842570906], 
[0.05471633956068267, -0.34430968979544341]], [[-0.20409525100284021, 0.20302952470092794], [0.20302952470092794, 
-0.76273753864705096]], [[-0.80960417963744202, 0.20302952470092794], [0.20302952470092794, 
0.17567196898373627]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.013736771881341703, 0.20302952470092794], [0.20302952470092794, 
0.087462380141183349]], [[-0.28973481635429255, 0.20302952470092794], [0.20302952470092794, -0.17486873131021241]], 
[[-0.95107631719717256, 0.20302952470092794], [0.0029821337466202635, 0.20302952470092794]], [[0.20302952470092794, 
-0.32649927600308448], [0.20302952470092794, -0.5051158980483037]], [[-0.58727814097138942, -0.71676326712662219], 
[0.20302952470092794, 0.17759101180604486]], [[-0.13796018760230688, 0.1467627194153307], [0.20302952470092794, 
0.20302952470092794]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank3_taggedData_rank3(self):
      arg0=numpy.array([[[0.88502592369187472, 0.85742372196903149], [0.95976586879998771, -0.56036430367132684]], 
[[-0.97715398812929877, 0.59208471472826507], [-0.30813298039883286, -0.22034279818830593]], [[0.55591160845086618, 
-0.32695352913627795], [0.52710589636781058, 0.33036935066400797]], [[0.46784168392430514, -0.34446586394572476], 
[0.94834975280704481, -0.2447525595338107]], [[-0.32356541060801969, 0.81579809801884529], [-0.80159170475370045, 
0.89565274725994448]], [[0.0599796427701631, -0.13999217181597801], [0.038354321743574671, -0.064582972070557521]]])
      arg1=Data(numpy.array([[[0.30167181991238623, -0.22424642575008313], [0.22362997834625409, -0.12801473865196655]], 
[[-0.58489150633202502, -0.27725740964875811], [-0.37223469869999937, -0.1936587136276573]], [[0.86810938230082701, 
-0.34324889592637953], [0.64321995850606184, -0.95689563918660414]], [[-0.35168289005407094, -0.37669604385487077], 
[0.20958004500538463, -0.96259570122393878]], [[-0.95970777368582993, 0.29930529415460616], [0.022094003885609004, 
0.44257345067806053]], [[-0.70842609919647059, 0.2100050587389406], [0.11274820332002222, 
-0.16029281241528892]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.095707658665982187, -0.87800332198596709], [-0.38791154579167308, 
0.51988358408873681]], [[-0.45562391332623675, -0.12809397165941694], [0.54509076256290201, -0.1331291518931883]], 
[[-0.13276652586452298, -0.28650584169764359], [0.71920567169905825, -0.76794400899403348]], [[0.52184174625259416, 
0.017901062113908184], [0.53553620170355121, -0.85637309040467691]], [[0.59614996252439401, 0.10854457585530342], 
[-0.40106217365323915, -0.018574034975204601]], [[0.82914280755218428, 0.93917580979105675], [-0.79396858050831365, 
0.67231168747369185]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[0.30167181991238623, -0.22424642575008313], [0.22362997834625409, -0.56036430367132684]], 
[[-0.97715398812929877, -0.27725740964875811], [-0.37223469869999937, -0.22034279818830593]], [[0.55591160845086618, 
-0.34324889592637953], [0.52710589636781058, -0.95689563918660414]], [[-0.35168289005407094, -0.37669604385487077], 
[0.20958004500538463, -0.96259570122393878]], [[-0.95970777368582993, 0.29930529415460616], [-0.80159170475370045, 
0.44257345067806053]], [[-0.70842609919647059, -0.13999217181597801], [0.038354321743574671, 
-0.16029281241528892]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.095707658665982187, -0.87800332198596709], [-0.38791154579167308, 
-0.56036430367132684]], [[-0.97715398812929877, -0.12809397165941694], [-0.30813298039883286, -0.22034279818830593]], 
[[-0.13276652586452298, -0.32695352913627795], [0.52710589636781058, -0.76794400899403348]], [[0.46784168392430514, 
-0.34446586394572476], [0.53553620170355121, -0.85637309040467691]], [[-0.32356541060801969, 0.10854457585530342], 
[-0.80159170475370045, -0.018574034975204601]], [[0.0599796427701631, -0.13999217181597801], [-0.79396858050831365, 
-0.064582972070557521]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank0_taggedData_rank4(self):
      arg0=numpy.array(0.0678969292101)
      arg1=Data(numpy.array([[[[-0.12665265383896052, 0.25645593020479551, 0.98760363728441125, -0.18916947363341063], 
[0.80211392746016208, -0.79655807258798306, -0.58940645847190098, -0.49724875807858249], [-0.49500991266219541, 
0.4602060738670728, 0.21436040437842396, 0.5661338119377286]], [[0.087848762076911946, -0.50498475926115383, 
0.84587065458694211, 0.098644104468930349], [-0.78596542284390747, -0.67651479618492383, 0.10511315770912644, 
-0.19058258127001615], [0.71641996302637323, 0.63500785746291033, 0.81560895563646962, -0.59288037729291543]]], 
[[[0.2559593657056265, -0.37776660101224069, -0.95946916347307143, 0.15501490738840618], [0.16730518685751106, 
-0.46222139471612178, -0.63791622168448936, 0.20954454673259826], [-0.31793841550339241, -0.45385595920233235, 
-0.68622140724374336, -0.039171204912008539]], [[-0.62590780736051577, -0.41267150556102661, 0.4768706255626276, 
-0.91372615915324573], [-0.65014259107618155, -0.94415768054557736, 0.71668785435302085, -0.16420336939602187], 
[-0.66667350748883991, 0.72442477148466233, 0.54733099686881515, 0.28594946193801762]]], [[[0.54837453405049308, 
-0.93876707187428177, -0.57240842469036468, -0.26885541497656629], [-0.12384857635906665, -0.23387533676354044, 
-0.40390939361228617, -0.89507651739575844], [-0.11173676719683701, -0.62400270312509165, -0.51527960825827956, 
-0.86726950428358274]], [[0.46650544732571664, -0.74581061151918226, -0.20094665381988297, -0.99385158149006902], 
[0.51290463590382118, 0.15844344795084497, -0.68281260892041074, 0.14035919323554125], [0.0042193218663537646, 
0.71982120156389917, -0.3961301129880983, 0.92524756976901368]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.60594331810038815, -0.82528461892326299, -0.34523965416013458, 
-0.51839527724377676], [0.23847259856350234, -0.71361105356000376, 0.95103756502770009, 0.3132100477743327], 
[-0.96543739925905081, -0.98078990960070889, 0.79371150626242803, 0.19900802312419308]], [[-0.20741867984073736, 
-0.1355593671812001, 0.70720477617899968, -0.67009016018287726], [0.50556642629294268, 0.64847345373171117, 
-0.12039461947822128, -0.095057096271936148], [-0.37861447735464449, -0.81630928257989033, -0.28923776870741502, 
0.78536425705695234]]], [[[0.71946535762817243, -0.73368866219465101, -0.78017140263003371, 0.65343711363699319], 
[-0.71060680699782086, 0.88716578588330663, -0.48188227213878809, -0.083661183023939545], [-0.82465736514301047, 
0.31698524984234822, 0.44713834865060331, 0.098832471217584539]], [[0.63880848368962817, -0.15931797160423744, 
-0.3022343212279921, 0.13865738073465161], [0.34577961964212833, -0.42755843429704132, -0.55798716039458274, 
-0.34281486893970281], [0.085488423746791664, -0.50719763831744058, -0.94171316695839735, 0.90523404823880793]]], 
[[[-0.61887974101427501, -0.82508208104588832, -0.24641757627910255, -0.10216537470750242], [-0.39078188266722047, 
-0.12211127955934731, 0.65552463471164568, -0.38886408605624045], [-0.86372074296908941, 0.67650399227357116, 
0.5313399570645958, 0.85485736331270434]], [[0.65752196932455664, -0.24628725268356577, -0.94477726363991676, 
-0.8923716031253266], [0.26769998715815624, -0.44356036711993219, 0.088773352375644654, 0.71558079112410322], 
[0.7857844239268903, -0.7781317703863746, -0.41540878674797521, -0.64162521327287636]]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.12665265383896052, 0.067896929210103041, 0.067896929210103041, -0.18916947363341063], 
[0.067896929210103041, -0.79655807258798306, -0.58940645847190098, -0.49724875807858249], [-0.49500991266219541, 
0.067896929210103041, 0.067896929210103041, 0.067896929210103041]], [[0.067896929210103041, -0.50498475926115383, 
0.067896929210103041, 0.067896929210103041], [-0.78596542284390747, -0.67651479618492383, 0.067896929210103041, 
-0.19058258127001615], [0.067896929210103041, 0.067896929210103041, 0.067896929210103041, -0.59288037729291543]]], 
[[[0.067896929210103041, -0.37776660101224069, -0.95946916347307143, 0.067896929210103041], [0.067896929210103041, 
-0.46222139471612178, -0.63791622168448936, 0.067896929210103041], [-0.31793841550339241, -0.45385595920233235, 
-0.68622140724374336, -0.039171204912008539]], [[-0.62590780736051577, -0.41267150556102661, 0.067896929210103041, 
-0.91372615915324573], [-0.65014259107618155, -0.94415768054557736, 0.067896929210103041, -0.16420336939602187], 
[-0.66667350748883991, 0.067896929210103041, 0.067896929210103041, 0.067896929210103041]]], [[[0.067896929210103041, 
-0.93876707187428177, -0.57240842469036468, -0.26885541497656629], [-0.12384857635906665, -0.23387533676354044, 
-0.40390939361228617, -0.89507651739575844], [-0.11173676719683701, -0.62400270312509165, -0.51527960825827956, 
-0.86726950428358274]], [[0.067896929210103041, -0.74581061151918226, -0.20094665381988297, -0.99385158149006902], 
[0.067896929210103041, 0.067896929210103041, -0.68281260892041074, 0.067896929210103041], [0.0042193218663537646, 
0.067896929210103041, -0.3961301129880983, 0.067896929210103041]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.067896929210103041, -0.82528461892326299, -0.34523965416013458, 
-0.51839527724377676], [0.067896929210103041, -0.71361105356000376, 0.067896929210103041, 0.067896929210103041], 
[-0.96543739925905081, -0.98078990960070889, 0.067896929210103041, 0.067896929210103041]], [[-0.20741867984073736, 
-0.1355593671812001, 0.067896929210103041, -0.67009016018287726], [0.067896929210103041, 0.067896929210103041, 
-0.12039461947822128, -0.095057096271936148], [-0.37861447735464449, -0.81630928257989033, -0.28923776870741502, 
0.067896929210103041]]], [[[0.067896929210103041, -0.73368866219465101, -0.78017140263003371, 0.067896929210103041], 
[-0.71060680699782086, 0.067896929210103041, -0.48188227213878809, -0.083661183023939545], [-0.82465736514301047, 
0.067896929210103041, 0.067896929210103041, 0.067896929210103041]], [[0.067896929210103041, -0.15931797160423744, 
-0.3022343212279921, 0.067896929210103041], [0.067896929210103041, -0.42755843429704132, -0.55798716039458274, 
-0.34281486893970281], [0.067896929210103041, -0.50719763831744058, -0.94171316695839735, 0.067896929210103041]]], 
[[[-0.61887974101427501, -0.82508208104588832, -0.24641757627910255, -0.10216537470750242], [-0.39078188266722047, 
-0.12211127955934731, 0.067896929210103041, -0.38886408605624045], [-0.86372074296908941, 0.067896929210103041, 
0.067896929210103041, 0.067896929210103041]], [[0.067896929210103041, -0.24628725268356577, -0.94477726363991676, 
-0.8923716031253266], [0.067896929210103041, -0.44356036711993219, 0.067896929210103041, 0.067896929210103041], 
[0.067896929210103041, -0.7781317703863746, -0.41540878674797521, -0.64162521327287636]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_array_rank4_taggedData_rank4(self):
      arg0=numpy.array([[[[-0.82936405041301198, -0.79396490383160967, 0.71763317639219992, 0.96552909548859644], 
[0.54476402156674797, 0.55433955587666683, -0.83248342605335224, -0.28702074227398566], [-0.13594783805294353, 
-0.58220301976404976, 0.89353208934671025, -0.33429954370388071]], [[0.20204071761968589, 0.58012236136665196, 
0.94096401613735048, 0.77295041558343369], [0.70737919697043994, -0.94790815752100333, -0.6965481922929313, 
-0.10988610634726115], [0.42600259185457467, 0.84197958429976594, 0.34992874815030905, -0.1431861373350618]]], 
[[[-0.50292196020669744, 0.69406359516293459, -0.41311122847738813, 0.086707933043925989], [0.62081477697702314, 
-0.96503623119960991, -0.23089539183725338, 0.6128591741885816], [-0.38852475730206293, 0.64462729579766687, 
0.54052477597763926, 0.25342473949324718]], [[-0.602588623382597, -0.20927787345329585, -0.67212591676516631, 
0.32544187840330063], [0.30534939438050901, 0.9016343691426032, -0.26378576746521953, -0.96197872690176855], 
[0.093503831128672665, 0.29518020570069381, 0.6523991287635591, 0.54195123564830583]]], [[[-0.93313898366342074, 
0.59672302663341048, 0.85516405738901713, -0.56702462819811239], [0.26877702315015761, -0.95424920677748548, 
0.61717749112828346, -0.033322171153286062], [-0.014658844772328594, 0.17674457482801187, 0.7088883411053688, 
-0.96309807860029184]], [[-0.64718507871800468, -0.72384145624825158, -0.44945623150692637, 0.97411016582708587], 
[-0.2159683668777399, -0.7833017421119941, 0.32287182895004896, -0.32095554899946199], [-0.66944198289298384, 
0.6528642104020832, 0.099479295648528643, -0.36823506607666356]]]])
      arg1=Data(numpy.array([[[[-0.54267306431995266, 0.087096750455018856, 0.37413578645680068, -0.59505148835111599], 
[0.069536118226759314, 0.99811334424129861, 0.45374232819065852, -0.33402980252348935], [0.43760248978614347, 
-0.19904906666925259, -0.7882022134905986, -0.082680693938014471]], [[-0.71591504696031172, 0.60484151747213377, 
-0.93307268942836163, 0.56587975826783032], [0.86599225417952175, -0.25371687247719499, 0.35010513899659323, 
0.87149021307851582], [0.67747548970568472, -0.88681093776877362, 0.92515472388692421, 0.86913096121478972]]], 
[[[-0.52366880386216152, 0.82344423657223209, 0.26913922353373909, -0.34111415773856146], [0.50273314032539296, 
0.58269243851260377, 0.95797032613915278, 0.19864109290731968], [-0.97431626128799009, 0.42434532984991713, 
0.14335377279814132, -0.66915292104013568]], [[-0.33166570371753346, 0.032358507047952623, -0.67786155293815686, 
-0.93816498910878932], [0.20790849876514783, 0.53581063584175204, -0.069286880997705413, -0.93272841004996776], 
[-0.79913201135173084, -0.11062579251708482, -0.31293477845041284, -0.86451828989102331]]], [[[0.77387968120402162, 
0.78894667886635039, 0.40442752879506472, -0.92685000952569352], [0.31688048445289541, -0.0059031014227473033, 
-0.099614279304440556, 0.58742793340853217], [-0.4274257220012434, 0.33517925091190692, 0.36607972859673787, 
0.99889872122706813]], [[-0.20018638128390887, 0.59736816863175624, -0.37530716252853646, -0.77848082905042659], 
[0.48700042671809918, 0.54261968016751028, -0.0024921982843681967, -0.40430306778837388], [0.27038329607724232, 
0.24397833296585514, -0.14780775183372574, 0.047581284322311124]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[0.39416800876734559, 0.99014338653708278, 0.15903122923314017, 
-0.35839797282431118], [0.40558089757631177, 0.8025581070129495, -0.35878538018143313, 0.93333384290537746], 
[0.40133276352205427, -0.015044367001857184, -0.28582498369942133, 0.18779258589737369]], [[-0.016090078310134093, 
0.21616355497575501, 0.49426732010846619, -0.30892878282542413], [0.89875385300581434, -0.2868551786584046, 
-0.55731671132643945, -0.44660017090531134], [0.18137885565777268, -0.11848520809481, -0.28913384022425448, 
-0.64033385839366574]]], [[[0.45571924402024222, 0.21473831724468928, 0.42864453842635308, 0.5223409639104557], 
[0.65325121860405, -0.8057567709966591, 0.45127248421604627, 0.28175869398807385], [0.54569449100623868, -0.32032205858278262, 
0.91593090123037246, 0.58863408387256055]], [[-0.42467124696344771, 0.9101396153519079, -0.79865017526445303, 
0.70425425954435106], [-0.20695890594024746, -0.67163190975173648, -0.23733882318159805, 0.53689843977592844], 
[0.10995989031814291, 0.46119956351074332, 0.34819273977962717, -0.75598125930932247]]], [[[-0.56665623717604463, 
-0.17224727213003233, 0.053415864384508627, -0.050592664933241771], [0.51015093886792817, -0.9754302705114164, 
0.091870806978513242, 0.41755275851280671], [0.19511977812160053, -0.97855216381010446, -0.90245021466194242, 
-0.40345947249795566]], [[-0.92403115745746733, 0.54858218537017089, -0.4822958075473629, -0.53222278948330182], 
[-0.4472614626168665, 0.89424670938908868, 0.10917902674712177, 0.98847119270078121], [-0.61871843988925113, 
0.4910058506238939, -0.6899594042743955, 0.049800633462321109]]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.82936405041301198, -0.79396490383160967, 0.37413578645680068, -0.59505148835111599], 
[0.069536118226759314, 0.55433955587666683, -0.83248342605335224, -0.33402980252348935], [-0.13594783805294353, 
-0.58220301976404976, -0.7882022134905986, -0.33429954370388071]], [[-0.71591504696031172, 0.58012236136665196, 
-0.93307268942836163, 0.56587975826783032], [0.70737919697043994, -0.94790815752100333, -0.6965481922929313, 
-0.10988610634726115], [0.42600259185457467, -0.88681093776877362, 0.34992874815030905, -0.1431861373350618]]], 
[[[-0.52366880386216152, 0.69406359516293459, -0.41311122847738813, -0.34111415773856146], [0.50273314032539296, 
-0.96503623119960991, -0.23089539183725338, 0.19864109290731968], [-0.97431626128799009, 0.42434532984991713, 
0.14335377279814132, -0.66915292104013568]], [[-0.602588623382597, -0.20927787345329585, -0.67786155293815686, 
-0.93816498910878932], [0.20790849876514783, 0.53581063584175204, -0.26378576746521953, -0.96197872690176855], 
[-0.79913201135173084, -0.11062579251708482, -0.31293477845041284, -0.86451828989102331]]], [[[-0.93313898366342074, 
0.59672302663341048, 0.40442752879506472, -0.92685000952569352], [0.26877702315015761, -0.95424920677748548, 
-0.099614279304440556, -0.033322171153286062], [-0.4274257220012434, 0.17674457482801187, 0.36607972859673787, 
-0.96309807860029184]], [[-0.64718507871800468, -0.72384145624825158, -0.44945623150692637, -0.77848082905042659], 
[-0.2159683668777399, -0.7833017421119941, -0.0024921982843681967, -0.40430306778837388], [-0.66944198289298384, 
0.24397833296585514, -0.14780775183372574, -0.36823506607666356]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.82936405041301198, -0.79396490383160967, 0.15903122923314017, 
-0.35839797282431118], [0.40558089757631177, 0.55433955587666683, -0.83248342605335224, -0.28702074227398566], 
[-0.13594783805294353, -0.58220301976404976, -0.28582498369942133, -0.33429954370388071]], [[-0.016090078310134093, 
0.21616355497575501, 0.49426732010846619, -0.30892878282542413], [0.70737919697043994, -0.94790815752100333, 
-0.6965481922929313, -0.44660017090531134], [0.18137885565777268, -0.11848520809481, -0.28913384022425448, 
-0.64033385839366574]]], [[[-0.50292196020669744, 0.21473831724468928, -0.41311122847738813, 0.086707933043925989], 
[0.62081477697702314, -0.96503623119960991, -0.23089539183725338, 0.28175869398807385], [-0.38852475730206293, 
-0.32032205858278262, 0.54052477597763926, 0.25342473949324718]], [[-0.602588623382597, -0.20927787345329585, 
-0.79865017526445303, 0.32544187840330063], [-0.20695890594024746, -0.67163190975173648, -0.26378576746521953, 
-0.96197872690176855], [0.093503831128672665, 0.29518020570069381, 0.34819273977962717, -0.75598125930932247]]], 
[[[-0.93313898366342074, -0.17224727213003233, 0.053415864384508627, -0.56702462819811239], [0.26877702315015761, 
-0.9754302705114164, 0.091870806978513242, -0.033322171153286062], [-0.014658844772328594, -0.97855216381010446, 
-0.90245021466194242, -0.96309807860029184]], [[-0.92403115745746733, -0.72384145624825158, -0.4822958075473629, 
-0.53222278948330182], [-0.4472614626168665, -0.7833017421119941, 0.10917902674712177, -0.32095554899946199], 
[-0.66944198289298384, 0.4910058506238939, -0.6899594042743955, -0.36823506607666356]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank0_taggedData_rank0(self):
      arg0=Data(0.225877171103,self.functionspace)
      arg1=Data(0.277305889136,self.functionspace)
      arg1.setTaggedValue(1,0.66791281927)
      res=minimum(arg0,arg1)
      ref=Data(0.225877171103,self.functionspace)
      ref.setTaggedValue(1,0.225877171103)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank1_taggedData_rank0(self):
      arg0=Data(numpy.array([0.4526998409249432, 0.32537826522840341]),self.functionspace)
      arg1=Data(0.713411860532,self.functionspace)
      arg1.setTaggedValue(1,0.142616706755)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([0.4526998409249432, 0.32537826522840341]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.1426167067546118, 0.1426167067546118]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank2_taggedData_rank0(self):
      arg0=Data(numpy.array([[-0.017762685918934595, -0.65181297652129633, -0.49733240109013366, -0.88631086286368244, 
-0.8454558226997968], [0.57102714448249614, 0.89336290874654778, -0.39163691397751044, 0.51411401850728522, 
0.48802132742208237], [-0.58255359903259518, -0.53253742168488616, -0.51802594144131309, -0.53697035710467667, 
-0.4584068226271707], [0.99632312492130204, 0.34436152053392988, -0.44253712317691773, 0.20888549431701131, 
-0.66953394694454893]]),self.functionspace)
      arg1=Data(-0.321551617675,self.functionspace)
      arg1.setTaggedValue(1,-0.654155919415)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.32155161767549134, -0.65181297652129633, -0.49733240109013366, -0.88631086286368244, 
-0.8454558226997968], [-0.32155161767549134, -0.32155161767549134, -0.39163691397751044, -0.32155161767549134, 
-0.32155161767549134], [-0.58255359903259518, -0.53253742168488616, -0.51802594144131309, -0.53697035710467667, 
-0.4584068226271707], [-0.32155161767549134, -0.32155161767549134, -0.44253712317691773, -0.32155161767549134, 
-0.66953394694454893]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.65415591941517426, -0.65415591941517426, -0.65415591941517426, 
-0.88631086286368244, -0.8454558226997968], [-0.65415591941517426, -0.65415591941517426, -0.65415591941517426, 
-0.65415591941517426, -0.65415591941517426], [-0.65415591941517426, -0.65415591941517426, -0.65415591941517426, 
-0.65415591941517426, -0.65415591941517426], [-0.65415591941517426, -0.65415591941517426, -0.65415591941517426, 
-0.65415591941517426, -0.66953394694454893]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank3_taggedData_rank0(self):
      arg0=Data(numpy.array([[[0.031723292372289214, -0.1506263871166682], [-0.22779103617395546, -0.54349746406888233]], 
[[-0.80148557145288257, -0.7028254311980966], [0.20832259219510441, 0.6713746877610689]], [[0.17017680743903241, 
0.97848340856909766], [0.6836520777852988, -0.84964512894200528]], [[0.061507465574485831, -0.39320300833499533], 
[0.39128006043477503, -0.44588218784834344]], [[-0.67536030623801513, -0.025100348735637601], [-0.13683640484877535, 
0.61282973884995884]], [[0.13654655950821049, -0.56261207716072348], [-0.48988063259609316, 
-0.0086999274602632681]]]),self.functionspace)
      arg1=Data(-0.367208660034,self.functionspace)
      arg1.setTaggedValue(1,-0.240670242886)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[-0.36720866003361019, -0.36720866003361019], [-0.36720866003361019, -0.54349746406888233]], 
[[-0.80148557145288257, -0.7028254311980966], [-0.36720866003361019, -0.36720866003361019]], [[-0.36720866003361019, 
-0.36720866003361019], [-0.36720866003361019, -0.84964512894200528]], [[-0.36720866003361019, -0.39320300833499533], 
[-0.36720866003361019, -0.44588218784834344]], [[-0.67536030623801513, -0.36720866003361019], [-0.36720866003361019, 
-0.36720866003361019]], [[-0.36720866003361019, -0.56261207716072348], [-0.48988063259609316, 
-0.36720866003361019]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.24067024288599703, -0.24067024288599703], [-0.24067024288599703, 
-0.54349746406888233]], [[-0.80148557145288257, -0.7028254311980966], [-0.24067024288599703, -0.24067024288599703]], 
[[-0.24067024288599703, -0.24067024288599703], [-0.24067024288599703, -0.84964512894200528]], [[-0.24067024288599703, 
-0.39320300833499533], [-0.24067024288599703, -0.44588218784834344]], [[-0.67536030623801513, -0.24067024288599703], 
[-0.24067024288599703, -0.24067024288599703]], [[-0.24067024288599703, -0.56261207716072348], [-0.48988063259609316, 
-0.24067024288599703]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank4_taggedData_rank0(self):
      arg0=Data(numpy.array([[[[-0.44273623340708412, 0.031200100081486326, -0.6688661674042129, 0.93974417945553412], 
[-0.72425921940420768, 0.36034383514892876, -0.27543629504900191, -0.065543623552129393], [-0.17356597893246728, 
0.30017263946155937, -0.78722197498044255, 0.21445388610207949]], [[-0.48296053351683121, -0.2613301983288312, 
-0.92334566768053961, 0.71170761734845933], [0.88640386493040024, -0.41576566080311039, -0.73352921425900841, 
-0.74042399421284588], [-0.56192464945171983, -0.78170005647772278, 0.50922733279721566, 0.2413797499462409]]], 
[[[0.56357511617837885, -0.47647712447302593, 0.29030365384426382, -0.2442692684034915], [-0.74553519617033337, 
-0.44024461057976993, 0.76040476745339158, -0.2207613382664575], [-0.26170733416867087, -0.76364717987425301, 
0.32456453759685444, -0.79090530278295601]], [[-0.83219931906179934, 0.75890458689379514, 0.23083756972806069, 
-0.070971785006897603], [-0.28236830361646126, 0.64669880762152099, 0.0099612083946185592, 0.34162192303551087], 
[-0.70311666770736969, 0.90545868104691629, 0.76170490816030667, -0.60689158810650201]]], [[[-0.56232216544028413, 
-0.38014803124247365, 0.16523034113185564, -0.49427414694104699], [-0.014577293838089922, 0.96202839649336092, 
0.46765689633930485, 0.91544095792133118], [0.76953360395215831, -0.14659631399903894, 0.72067780986266983, 
-0.65041116138940236]], [[-0.31731440155455282, 0.27850816056703454, -0.77334131962394936, 0.71012557119454489], 
[-0.002373090596369698, 0.89848769974573317, 0.81394880569766803, -0.079059271207052584], [0.87236029512718227, 
-0.095527152237600887, -0.47929944271646385, -0.79443745585933057]]]]),self.functionspace)
      arg1=Data(-0.620140348567,self.functionspace)
      arg1.setTaggedValue(1,0.964538500719)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.62014034856657996, -0.62014034856657996, -0.6688661674042129, -0.62014034856657996], 
[-0.72425921940420768, -0.62014034856657996, -0.62014034856657996, -0.62014034856657996], [-0.62014034856657996, 
-0.62014034856657996, -0.78722197498044255, -0.62014034856657996]], [[-0.62014034856657996, -0.62014034856657996, 
-0.92334566768053961, -0.62014034856657996], [-0.62014034856657996, -0.62014034856657996, -0.73352921425900841, 
-0.74042399421284588], [-0.62014034856657996, -0.78170005647772278, -0.62014034856657996, -0.62014034856657996]]], 
[[[-0.62014034856657996, -0.62014034856657996, -0.62014034856657996, -0.62014034856657996], [-0.74553519617033337, 
-0.62014034856657996, -0.62014034856657996, -0.62014034856657996], [-0.62014034856657996, -0.76364717987425301, 
-0.62014034856657996, -0.79090530278295601]], [[-0.83219931906179934, -0.62014034856657996, -0.62014034856657996, 
-0.62014034856657996], [-0.62014034856657996, -0.62014034856657996, -0.62014034856657996, -0.62014034856657996], 
[-0.70311666770736969, -0.62014034856657996, -0.62014034856657996, -0.62014034856657996]]], [[[-0.62014034856657996, 
-0.62014034856657996, -0.62014034856657996, -0.62014034856657996], [-0.62014034856657996, -0.62014034856657996, 
-0.62014034856657996, -0.62014034856657996], [-0.62014034856657996, -0.62014034856657996, -0.62014034856657996, 
-0.65041116138940236]], [[-0.62014034856657996, -0.62014034856657996, -0.77334131962394936, -0.62014034856657996], 
[-0.62014034856657996, -0.62014034856657996, -0.62014034856657996, -0.62014034856657996], [-0.62014034856657996, 
-0.62014034856657996, -0.62014034856657996, -0.79443745585933057]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.44273623340708412, 0.031200100081486326, -0.6688661674042129, 
0.93974417945553412], [-0.72425921940420768, 0.36034383514892876, -0.27543629504900191, -0.065543623552129393], 
[-0.17356597893246728, 0.30017263946155937, -0.78722197498044255, 0.21445388610207949]], [[-0.48296053351683121, 
-0.2613301983288312, -0.92334566768053961, 0.71170761734845933], [0.88640386493040024, -0.41576566080311039, 
-0.73352921425900841, -0.74042399421284588], [-0.56192464945171983, -0.78170005647772278, 0.50922733279721566, 
0.2413797499462409]]], [[[0.56357511617837885, -0.47647712447302593, 0.29030365384426382, -0.2442692684034915], 
[-0.74553519617033337, -0.44024461057976993, 0.76040476745339158, -0.2207613382664575], [-0.26170733416867087, 
-0.76364717987425301, 0.32456453759685444, -0.79090530278295601]], [[-0.83219931906179934, 0.75890458689379514, 
0.23083756972806069, -0.070971785006897603], [-0.28236830361646126, 0.64669880762152099, 0.0099612083946185592, 
0.34162192303551087], [-0.70311666770736969, 0.90545868104691629, 0.76170490816030667, -0.60689158810650201]]], 
[[[-0.56232216544028413, -0.38014803124247365, 0.16523034113185564, -0.49427414694104699], [-0.014577293838089922, 
0.96202839649336092, 0.46765689633930485, 0.91544095792133118], [0.76953360395215831, -0.14659631399903894, 
0.72067780986266983, -0.65041116138940236]], [[-0.31731440155455282, 0.27850816056703454, -0.77334131962394936, 
0.71012557119454489], [-0.002373090596369698, 0.89848769974573317, 0.81394880569766803, -0.079059271207052584], 
[0.87236029512718227, -0.095527152237600887, -0.47929944271646385, -0.79443745585933057]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank0_taggedData_rank1(self):
      arg0=Data(0.178325716243,self.functionspace)
      arg1=Data(numpy.array([0.78505630297659312, -0.55243718784943407]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.30444848322455487, -0.2336982668991745]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([0.17832571624310667, -0.55243718784943407]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.30444848322455487, -0.2336982668991745]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank1_taggedData_rank1(self):
      arg0=Data(numpy.array([-0.20118572938121337, 0.19418231779243156]),self.functionspace)
      arg1=Data(numpy.array([-0.9899901196266836, 0.1871346991081082]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([-0.19602944903841113, -0.84912485105429125]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([-0.9899901196266836, 0.1871346991081082]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.20118572938121337, -0.84912485105429125]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank0_taggedData_rank2(self):
      arg0=Data(0.975190032401,self.functionspace)
      arg1=Data(numpy.array([[-0.86202636130621668, -0.59544876315837914, -0.78980194648289581, 0.59091403106639495, 
-0.59559768209928743], [0.9685191045495003, -0.71579615267260688, -0.1934965786539653, 0.99578308216468669, 
0.97663049576553007], [-0.28712850522824152, 0.19288383161517908, 0.81313940002605167, 0.37864539171222522, 
0.95145482397392511], [0.87190319774491232, 0.6446733353605647, 0.45453615370108946, -0.11342366980612506, 
0.83492988105300081]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[-0.56389253228196856, 0.87860161520032509, -0.25874394357260044, 0.77073198973677903, 
0.74977569035775038], [0.93561263862699739, 0.40798833393540224, 0.55403115060408648, 0.65439435064311069, 
0.068896274015486592], [0.14324617510242366, -0.014258913345540414, 0.31458116965964478, 0.79897131943450694, 
-0.58958708998853848], [-0.94029383517472676, 0.17611923939042984, 0.80375993775575427, 0.72231563608658655, 
-0.75408423629815147]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.86202636130621668, -0.59544876315837914, -0.78980194648289581, 0.59091403106639495, 
-0.59559768209928743], [0.9685191045495003, -0.71579615267260688, -0.1934965786539653, 0.97519003240141355, 
0.97519003240141355], [-0.28712850522824152, 0.19288383161517908, 0.81313940002605167, 0.37864539171222522, 
0.95145482397392511], [0.87190319774491232, 0.6446733353605647, 0.45453615370108946, -0.11342366980612506, 
0.83492988105300081]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.56389253228196856, 0.87860161520032509, -0.25874394357260044, 0.77073198973677903, 
0.74977569035775038], [0.93561263862699739, 0.40798833393540224, 0.55403115060408648, 0.65439435064311069, 
0.068896274015486592], [0.14324617510242366, -0.014258913345540414, 0.31458116965964478, 0.79897131943450694, 
-0.58958708998853848], [-0.94029383517472676, 0.17611923939042984, 0.80375993775575427, 0.72231563608658655, 
-0.75408423629815147]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank2_taggedData_rank2(self):
      arg0=Data(numpy.array([[0.34947114089707232, -0.28103463468130929, -0.76400894039365186, 0.73581392557737035, 
0.69640346875155545], [0.36552595835240198, -0.17490051696032549, 0.40693635231492609, 0.3084192407845967, 
-0.22803385698531464], [-0.7405552467921912, 0.48757279567462786, 0.058167944473768829, -0.3285259953857258, 
-0.38854576897745097], [-0.53463794089403138, 0.43048527918852564, -0.22030156380522659, 0.062738926404989348, 
-0.88092939482795396]]),self.functionspace)
      arg1=Data(numpy.array([[-0.99122857259120889, -0.99998817165650244, 0.045329460657784315, 0.20415516786226395, 
-0.30165493637145402], [-0.35138947790302288, 0.28888041693288091, -0.9752764220549881, -0.73056851909800691, 
0.86300882343314234], [-0.61635589258460421, -0.71881055281358019, 0.54434424080432398, 0.43677525028770003, 
-0.22013330813454357], [-0.57579033587058781, 0.38540999087821826, -0.62653671660229304, 0.20432264985624227, 
0.37966972969155899]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[0.23317573056993957, 0.70069357774036711, 0.039590050093938789, -0.36973464200852924, 
0.55605024662037983], [-0.20432721308644375, -0.80781775939607381, -0.82619858671897961, 0.88259251218704526, 
-0.14342875068832073], [0.71227111372480501, 0.23587970276917125, 0.1007049938833795, -0.45140064573178273, 
-0.73090075823332001], [0.91652562507916824, 0.097278630645751818, 0.91391125218666547, 0.12692481776378872, 
-0.90225676538054556]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.99122857259120889, -0.99998817165650244, -0.76400894039365186, 0.20415516786226395, 
-0.30165493637145402], [-0.35138947790302288, -0.17490051696032549, -0.9752764220549881, -0.73056851909800691, 
-0.22803385698531464], [-0.7405552467921912, -0.71881055281358019, 0.058167944473768829, -0.3285259953857258, 
-0.38854576897745097], [-0.57579033587058781, 0.38540999087821826, -0.62653671660229304, 0.062738926404989348, 
-0.88092939482795396]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.23317573056993957, -0.28103463468130929, -0.76400894039365186, -0.36973464200852924, 
0.55605024662037983], [-0.20432721308644375, -0.80781775939607381, -0.82619858671897961, 0.3084192407845967, 
-0.22803385698531464], [-0.7405552467921912, 0.23587970276917125, 0.058167944473768829, -0.45140064573178273, 
-0.73090075823332001], [-0.53463794089403138, 0.097278630645751818, -0.22030156380522659, 0.062738926404989348, 
-0.90225676538054556]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank0_taggedData_rank3(self):
      arg0=Data(-0.736854513353,self.functionspace)
      arg1=Data(numpy.array([[[-0.57448211499847024, 0.59139977207765204], [0.64092977291854702, -0.84271342538037364]], 
[[0.25291717252569024, 0.019346410146011461], [0.047447114311803817, 0.87546898954072416]], [[0.36466836303202999, 
0.30366122371319682], [-0.6259606483402036, -0.92828339668338589]], [[-0.57593607705092587, 0.43123907776972481], 
[-0.49316509031352407, 0.043393040792496729]], [[-0.41322651778640229, -0.084792175509268874], [-0.4234085673109349, 
-0.5325960310828699]], [[-0.73673021386826121, 0.45902782603580428], [0.92647540430535669, 
-0.77939917236563727]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[0.70177055752925654, -0.80268579816478214], [0.86835194900335133, 
0.94271987041164573]], [[0.7088856665145693, -0.62689665163020925], [-0.50886796926723732, -0.017796650137382075]], 
[[-0.38700196429580402, -0.40380125003689571], [0.29566043187492252, 0.035892260158582712]], [[0.35942043896170595, 
-0.52300218242620633], [-0.36632233898566846, -0.86500367346400897]], [[-0.26140476563667714, 0.24214775028887559], 
[-0.31627433835416463, -0.52230607748165325]], [[-0.57756134554965999, -0.30743335773523595], [-0.32706191089311454, 
-0.73094041155494627]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[-0.73685451335255903, -0.73685451335255903], [-0.73685451335255903, -0.84271342538037364]], 
[[-0.73685451335255903, -0.73685451335255903], [-0.73685451335255903, -0.73685451335255903]], [[-0.73685451335255903, 
-0.73685451335255903], [-0.73685451335255903, -0.92828339668338589]], [[-0.73685451335255903, -0.73685451335255903], 
[-0.73685451335255903, -0.73685451335255903]], [[-0.73685451335255903, -0.73685451335255903], [-0.73685451335255903, 
-0.73685451335255903]], [[-0.73685451335255903, -0.73685451335255903], [-0.73685451335255903, 
-0.77939917236563727]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.73685451335255903, -0.80268579816478214], [-0.73685451335255903, 
-0.73685451335255903]], [[-0.73685451335255903, -0.73685451335255903], [-0.73685451335255903, -0.73685451335255903]], 
[[-0.73685451335255903, -0.73685451335255903], [-0.73685451335255903, -0.73685451335255903]], [[-0.73685451335255903, 
-0.73685451335255903], [-0.73685451335255903, -0.86500367346400897]], [[-0.73685451335255903, -0.73685451335255903], 
[-0.73685451335255903, -0.73685451335255903]], [[-0.73685451335255903, -0.73685451335255903], [-0.73685451335255903, 
-0.73685451335255903]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank3_taggedData_rank3(self):
      arg0=Data(numpy.array([[[-0.42224967641267841, -0.04953280771110391], [0.28806498070366815, 0.73432982083316545]], 
[[0.88749165792498474, 0.85700506909446017], [0.25352612405207298, 0.61681617867896099]], [[0.52587016273739851, 
-0.58292193049311103], [-0.97016342390909238, 0.76925781921946301]], [[-0.042862467008675909, -0.29668144504061322], 
[-0.19848034188904573, 0.24227846596969793]], [[-0.52918092844978326, 0.42953959438772404], [-0.15263980316438475, 
0.14679279133389733]], [[-0.36635672708816958, -0.4330281646931915], [0.86615460943203648, 
-0.48772899445274054]]]),self.functionspace)
      arg1=Data(numpy.array([[[-0.87811552860092634, 0.36140980047466797], [0.81247226396752126, -0.64774597349686225]], 
[[0.9013118667475879, -0.7389952362864094], [-0.12712688498775493, 0.92309007938391407]], [[-0.33438585866235471, 
-0.58091360898060929], [-0.15766596904656693, 0.79576969879358428]], [[0.81830004813464852, 0.41331524413531628], 
[-0.58446102824871549, -0.17268541412774474]], [[0.87702192346698649, 0.30348596410238216], [-0.53014858765787243, 
0.041020623311175841]], [[0.66613383509296908, -0.8197163346181835], [0.97278499838840515, 
0.67571300136099532]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[-0.76380314111852177, 0.63430485167627371], [-0.53014076775893737, 
0.31390473384742212]], [[0.23376211898602151, 0.80042689496248354], [-0.71008948849055642, -0.94955686688141783]], 
[[0.21775509615833788, -0.97915924536092347], [-0.71331893534808311, 0.21462144216924184]], [[-0.75841034634074744, 
0.15706435884015768], [-0.81625051692502337, -0.63663558122654496]], [[-0.98840679975469814, 0.21121116436505916], 
[0.82396977494910217, -0.67736735834709472]], [[-0.030535851188874918, 0.81187897336059911], [0.78917855676896709, 
0.59825079982625895]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[-0.87811552860092634, -0.04953280771110391], [0.28806498070366815, -0.64774597349686225]], 
[[0.88749165792498474, -0.7389952362864094], [-0.12712688498775493, 0.61681617867896099]], [[-0.33438585866235471, 
-0.58292193049311103], [-0.97016342390909238, 0.76925781921946301]], [[-0.042862467008675909, -0.29668144504061322], 
[-0.58446102824871549, -0.17268541412774474]], [[-0.52918092844978326, 0.30348596410238216], [-0.53014858765787243, 
0.041020623311175841]], [[-0.36635672708816958, -0.8197163346181835], [0.86615460943203648, 
-0.48772899445274054]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.76380314111852177, -0.04953280771110391], [-0.53014076775893737, 
0.31390473384742212]], [[0.23376211898602151, 0.80042689496248354], [-0.71008948849055642, -0.94955686688141783]], 
[[0.21775509615833788, -0.97915924536092347], [-0.97016342390909238, 0.21462144216924184]], [[-0.75841034634074744, 
-0.29668144504061322], [-0.81625051692502337, -0.63663558122654496]], [[-0.98840679975469814, 0.21121116436505916], 
[-0.15263980316438475, -0.67736735834709472]], [[-0.36635672708816958, -0.4330281646931915], [0.78917855676896709, 
-0.48772899445274054]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank0_taggedData_rank4(self):
      arg0=Data(0.847767123794,self.functionspace)
      arg1=Data(numpy.array([[[[0.36810927235948121, -0.80671144788254434, -0.66806969421287654, 0.96350370310202194], 
[-0.076353218541952428, 0.27382054762836572, -0.05634042392452665, -0.86969476594725692], [0.90771485822678777, 
0.095835380995942909, 0.32990358518042728, -0.84027346607851894]], [[0.067455007620340179, -0.026361926908042621, 
0.66395387277662343, 0.89760372310798386], [-0.63370768209853168, 0.064171295378978144, -0.062586935479216788, 
-0.41223643432067325], [0.72880129876769306, 0.82430136418267819, 0.29936858462655902, 0.32630325443405583]]], 
[[[-0.33111254031149207, 0.86438719782065121, -0.13154158653081161, 0.95488508443685238], [0.67997591122688883, 
-0.73276466281010766, 0.16996609978000699, 0.23481440962914335], [-0.48776476040425543, 0.94148175245649512, 
-0.38755431700446263, -0.23397457069580962]], [[-0.076467713904021517, 0.19363980924169666, -0.46451128223172855, 
0.51827901190044035], [-0.1191187161970062, 0.19226606090617082, 0.61977648103609706, -0.34984255475987247], 
[0.6721992022178227, -0.73880638292794765, -0.17481573662248207, -0.71944858428616243]]], [[[-0.81421015840800681, 
-0.71331485323994115, 0.93354950174181672, -0.14734566813318617], [-0.044187407286275571, -0.26037624880879706, 
0.599342520726448, -0.67018767853815109], [0.95327278975622254, 0.015572812335825192, 0.43837001685019161, 
-0.73328549079867256]], [[0.76323757582530627, 0.78753733663268721, -0.078422094304839485, -0.91467258221103309], 
[-0.32506659779255154, -0.12044881052964262, -0.71354718322594657, 0.83907499522120443], [0.26325131309829675, 
0.40615773584577197, 0.7855071196698451, -0.62872663953173924]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.46816486312575312, 0.17310723401254857, 0.49314598293092304, 
0.92577264251688662], [0.44450319663365234, -0.94410536826156233, -0.028186907618339863, -0.051155304183805095], 
[0.52621824025321562, 0.38951773563722458, 0.010815159737161073, 0.80621013632253868]], [[0.52755950228226589, 
-0.6829279905193284, -0.4994157102198431, -0.035907358310534931], [-0.67853870891518198, -0.27263843566282331, 
-0.91483008039087066, 0.22996686639648223], [0.98924328660034111, -0.41800782162541994, 0.37582545499827336, 
0.32772929982355437]]], [[[-0.93639356591049827, -0.23675850485482375, 0.10394465343526327, 0.74044550568601064], 
[-0.54199581864081092, -0.43885808074710408, 0.21007362346165159, 0.95714334434505721], [-0.82292281508903131, 
-0.78352963010998633, 0.53894872311819242, -0.84075746010250962]], [[0.25640280239967872, -0.49837822472419013, 
0.020337858179311441, 0.49709006122450861], [-0.63656870118109321, -0.29577437438241616, -0.13132345310005755, 
0.80754917164541684], [0.67124865023074642, 0.44757637012118301, 0.60505187998087906, -0.56669979869520515]]], 
[[[-0.74381742322364608, -0.17445109670160575, -0.16610221033694628, -0.98379243559988039], [-0.74866956265381468, 
-0.64365369738150902, 0.85596487240248598, -0.73747004098303215], [-0.53918597292349246, 0.94803116317502734, 
0.36535162404104993, 0.94471847934807274]], [[-0.69301552250715215, 0.088492630592122534, -0.70925782539362436, 
-0.079756478328453007], [-0.23848517031492866, 0.44157210742186748, -0.57930743748572011, 0.054209975561518187], 
[-0.59689550329872021, -0.65945034106471367, -0.85477576502839803, -0.5403314055875188]]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[0.36810927235948121, -0.80671144788254434, -0.66806969421287654, 0.84776712379415953], 
[-0.076353218541952428, 0.27382054762836572, -0.05634042392452665, -0.86969476594725692], [0.84776712379415953, 
0.095835380995942909, 0.32990358518042728, -0.84027346607851894]], [[0.067455007620340179, -0.026361926908042621, 
0.66395387277662343, 0.84776712379415953], [-0.63370768209853168, 0.064171295378978144, -0.062586935479216788, 
-0.41223643432067325], [0.72880129876769306, 0.82430136418267819, 0.29936858462655902, 0.32630325443405583]]], 
[[[-0.33111254031149207, 0.84776712379415953, -0.13154158653081161, 0.84776712379415953], [0.67997591122688883, 
-0.73276466281010766, 0.16996609978000699, 0.23481440962914335], [-0.48776476040425543, 0.84776712379415953, 
-0.38755431700446263, -0.23397457069580962]], [[-0.076467713904021517, 0.19363980924169666, -0.46451128223172855, 
0.51827901190044035], [-0.1191187161970062, 0.19226606090617082, 0.61977648103609706, -0.34984255475987247], 
[0.6721992022178227, -0.73880638292794765, -0.17481573662248207, -0.71944858428616243]]], [[[-0.81421015840800681, 
-0.71331485323994115, 0.84776712379415953, -0.14734566813318617], [-0.044187407286275571, -0.26037624880879706, 
0.599342520726448, -0.67018767853815109], [0.84776712379415953, 0.015572812335825192, 0.43837001685019161, 
-0.73328549079867256]], [[0.76323757582530627, 0.78753733663268721, -0.078422094304839485, -0.91467258221103309], 
[-0.32506659779255154, -0.12044881052964262, -0.71354718322594657, 0.83907499522120443], [0.26325131309829675, 
0.40615773584577197, 0.7855071196698451, -0.62872663953173924]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.46816486312575312, 0.17310723401254857, 0.49314598293092304, 
0.84776712379415953], [0.44450319663365234, -0.94410536826156233, -0.028186907618339863, -0.051155304183805095], 
[0.52621824025321562, 0.38951773563722458, 0.010815159737161073, 0.80621013632253868]], [[0.52755950228226589, 
-0.6829279905193284, -0.4994157102198431, -0.035907358310534931], [-0.67853870891518198, -0.27263843566282331, 
-0.91483008039087066, 0.22996686639648223], [0.84776712379415953, -0.41800782162541994, 0.37582545499827336, 
0.32772929982355437]]], [[[-0.93639356591049827, -0.23675850485482375, 0.10394465343526327, 0.74044550568601064], 
[-0.54199581864081092, -0.43885808074710408, 0.21007362346165159, 0.84776712379415953], [-0.82292281508903131, 
-0.78352963010998633, 0.53894872311819242, -0.84075746010250962]], [[0.25640280239967872, -0.49837822472419013, 
0.020337858179311441, 0.49709006122450861], [-0.63656870118109321, -0.29577437438241616, -0.13132345310005755, 
0.80754917164541684], [0.67124865023074642, 0.44757637012118301, 0.60505187998087906, -0.56669979869520515]]], 
[[[-0.74381742322364608, -0.17445109670160575, -0.16610221033694628, -0.98379243559988039], [-0.74866956265381468, 
-0.64365369738150902, 0.84776712379415953, -0.73747004098303215], [-0.53918597292349246, 0.84776712379415953, 
0.36535162404104993, 0.84776712379415953]], [[-0.69301552250715215, 0.088492630592122534, -0.70925782539362436, 
-0.079756478328453007], [-0.23848517031492866, 0.44157210742186748, -0.57930743748572011, 0.054209975561518187], 
[-0.59689550329872021, -0.65945034106471367, -0.85477576502839803, -0.5403314055875188]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_constData_rank4_taggedData_rank4(self):
      arg0=Data(numpy.array([[[[0.17658848092916402, 0.72611496603204095, -0.12003607061228783, -0.36770287107434774], 
[0.28973497990121033, 0.22518180135245847, 0.43287818653532883, 0.49232120352800735], [0.71807823602733944, 
-0.83529407871982708, 0.49199165512787846, -0.90147701130057722]], [[0.50905759812376639, -0.76014675098579021, 
-0.17639917586657461, 0.071989168652567992], [0.52904428253495173, 0.055593670562822384, 0.10943531222677616, 
0.48478185344624758], [0.6604633944713254, 0.77262203707746591, -0.83140001519365536, -0.12628555789837193]]], 
[[[0.72759933775834873, -0.84367102760971546, 0.46175687127974041, -0.12419749685394321], [-0.54932269044276638, 
-0.57598013647750212, 0.94925870164772097, -0.9794000990291567], [-0.96741242065027833, 0.30780826831468544, 
-0.70116205986572488, -0.099131128223903353]], [[0.15935419946152418, -0.71425213510583485, -0.92953439771813473, 
0.19327756021431197], [-0.58756073454898705, 0.73903786914127867, -0.35311379116286523, 0.34779275694379042], 
[0.28917206431935671, -0.021591141296280814, -0.58635063235124818, 0.50644751204285288]]], [[[0.8465186149226025, 
-0.16358061947838154, 0.2779090586812123, 0.068668106711446075], [-0.67386336623273291, 0.3128208384508504, 
0.76194650412067944, 0.43747336548438298], [0.73625177392476804, -0.7495607604848471, -0.56800788254494083, 
-0.21872256780052912]], [[-0.020171371846024355, -0.053474168902059693, -0.22976475535323448, -0.88423120626481722], 
[-0.040653798000751307, -0.44411643272954526, 0.34104818913182444, -0.43774325557443095], [-0.63772621378240313, 
-0.62716680888538012, 0.37190569881118485, -0.6471128780763804]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[0.49028030216193974, 0.97627908987000467, -0.98012074159342344, -0.50378841107515915], 
[-0.82058718641293837, -0.61763548942523072, -0.39360229619580056, 0.02987260071764597], [-0.71862518098130934, 
-0.55200839867474749, 0.21011857938323408, 0.34025623453918841]], [[0.80670721104646081, 0.076638092924578682, 
0.92837834430181188, -0.53357840878544471], [-0.61437678217369096, 0.39121265292112484, 0.55322489364974392, 
0.53529126663663362], [-0.73920282509879409, 0.49143700377041055, -0.40865396066293513, -0.17977553855909711]]], 
[[[-0.42426813009871611, 0.46166677689333602, 0.40803650174676198, -0.59996295066310235], [0.064319465345361593, 
-0.48222088720071188, 0.56509801226989209, 0.16080361304840718], [0.70062380925413303, -0.26605957417872617, 
-0.52595756366230706, -0.49146030000794316]], [[-0.92391501814347565, 0.37478637672458714, 0.74465458064044499, 
-0.49229024123865694], [-0.23767119231094269, 0.63606341558140578, 0.99482091887981738, 0.59150594322590222], 
[-0.8724909475232141, 0.94758420189409365, -0.18504496070170973, -0.73496956976834116]]], [[[-0.064525264603144228, 
0.17334146745311885, -0.78572437896002989, -0.84535375953269631], [0.58068679256826861, 0.32617695391349888, 
0.12803431085631445, 0.30636060755411543], [0.22285096781269376, 0.92875808418025985, -0.4409002950218337, 
0.2698105481477584]], [[-0.85714021078849134, 0.95521870463328429, -0.41218460636540999, -0.17681380836277927], 
[-0.92503459537252297, -0.21429191853551899, 0.31821390547393169, 0.74923514989235462], [-0.86955570790220293, 
-0.95495170565382126, -0.20213982144119891, -0.39440676624488979]]]]),self.functionspace)
      arg1.setTaggedValue(1,numpy.array([[[[-0.28862928299329682, 0.48590108567348045, 0.46605976515374281, 
-0.35946207364451643], [0.6650170156774573, -0.68075399763227695, -0.19529981973413069, -0.67426281971694335], 
[-0.2514355300316502, -0.5000707149762551, -0.7328233841228462, 0.8330689087865697]], [[0.70667235451348565, 
0.58611205001689437, -0.55960816824941895, -0.016741842320867839], [-0.25034734635022704, -0.7328128546049586, 
-0.59864910128336168, -0.42046330565806134], [0.56692305299242274, -0.63051462632264599, -0.0031203701385391636, 
0.83719814932509262]]], [[[0.8699596083458081, -0.81105179506874792, 0.60241281214475229, -0.83246658198999812], 
[-0.20756577294201395, 0.48810301047269089, -0.51665526386034077, -0.32383211400398126], [0.9185136362933366, 
-0.73527855280084475, 0.91377290047985671, -0.61543591322725377]], [[0.98212430865886824, 0.31282795926394447, 
-0.66128711536317097, 0.98314848913752928], [0.91594338119477903, -0.52556995222432956, -0.99685808901461948, 
-0.98778335723963928], [-0.30062319495804557, -0.24605493384479682, -0.58246298695451038, 0.87291243466509849]]], 
[[[0.78959249839731482, 0.84260080794387826, 0.42275169162240123, 0.98318328055446802], [-0.75007395746815764, 
-0.18143501692910413, 0.32297783902112731, -0.97140942270435193], [-0.61910824066486869, -0.53189896305780815, 
0.2164475143489264, 0.24374100804485521]], [[0.68426183106069161, -0.44413732222456659, 0.57947580725984471, 
-0.32785721592768313], [-0.82910779952335067, 0.58658336307958425, -0.98789637388487406, 0.34716944534757688], 
[-0.053114351743224208, 0.68406544348072718, -0.025870209593289673, 0.44605230106198812]]]]))
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[0.17658848092916402, 0.72611496603204095, -0.98012074159342344, -0.50378841107515915], 
[-0.82058718641293837, -0.61763548942523072, -0.39360229619580056, 0.02987260071764597], [-0.71862518098130934, 
-0.83529407871982708, 0.21011857938323408, -0.90147701130057722]], [[0.50905759812376639, -0.76014675098579021, 
-0.17639917586657461, -0.53357840878544471], [-0.61437678217369096, 0.055593670562822384, 0.10943531222677616, 
0.48478185344624758], [-0.73920282509879409, 0.49143700377041055, -0.83140001519365536, -0.17977553855909711]]], 
[[[-0.42426813009871611, -0.84367102760971546, 0.40803650174676198, -0.59996295066310235], [-0.54932269044276638, 
-0.57598013647750212, 0.56509801226989209, -0.9794000990291567], [-0.96741242065027833, -0.26605957417872617, 
-0.70116205986572488, -0.49146030000794316]], [[-0.92391501814347565, -0.71425213510583485, -0.92953439771813473, 
-0.49229024123865694], [-0.58756073454898705, 0.63606341558140578, -0.35311379116286523, 0.34779275694379042], 
[-0.8724909475232141, -0.021591141296280814, -0.58635063235124818, -0.73496956976834116]]], [[[-0.064525264603144228, 
-0.16358061947838154, -0.78572437896002989, -0.84535375953269631], [-0.67386336623273291, 0.3128208384508504, 
0.12803431085631445, 0.30636060755411543], [0.22285096781269376, -0.7495607604848471, -0.56800788254494083, 
-0.21872256780052912]], [[-0.85714021078849134, -0.053474168902059693, -0.41218460636540999, -0.88423120626481722], 
[-0.92503459537252297, -0.44411643272954526, 0.31821390547393169, -0.43774325557443095], [-0.86955570790220293, 
-0.95495170565382126, -0.20213982144119891, -0.6471128780763804]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.28862928299329682, 0.48590108567348045, -0.12003607061228783, 
-0.36770287107434774], [0.28973497990121033, -0.68075399763227695, -0.19529981973413069, -0.67426281971694335], 
[-0.2514355300316502, -0.83529407871982708, -0.7328233841228462, -0.90147701130057722]], [[0.50905759812376639, 
-0.76014675098579021, -0.55960816824941895, -0.016741842320867839], [-0.25034734635022704, -0.7328128546049586, 
-0.59864910128336168, -0.42046330565806134], [0.56692305299242274, -0.63051462632264599, -0.83140001519365536, 
-0.12628555789837193]]], [[[0.72759933775834873, -0.84367102760971546, 0.46175687127974041, -0.83246658198999812], 
[-0.54932269044276638, -0.57598013647750212, -0.51665526386034077, -0.9794000990291567], [-0.96741242065027833, 
-0.73527855280084475, -0.70116205986572488, -0.61543591322725377]], [[0.15935419946152418, -0.71425213510583485, 
-0.92953439771813473, 0.19327756021431197], [-0.58756073454898705, -0.52556995222432956, -0.99685808901461948, 
-0.98778335723963928], [-0.30062319495804557, -0.24605493384479682, -0.58635063235124818, 0.50644751204285288]]], 
[[[0.78959249839731482, -0.16358061947838154, 0.2779090586812123, 0.068668106711446075], [-0.75007395746815764, 
-0.18143501692910413, 0.32297783902112731, -0.97140942270435193], [-0.61910824066486869, -0.7495607604848471, 
-0.56800788254494083, -0.21872256780052912]], [[-0.020171371846024355, -0.44413732222456659, -0.22976475535323448, 
-0.88423120626481722], [-0.82910779952335067, -0.44411643272954526, -0.98789637388487406, -0.43774325557443095], 
[-0.63772621378240313, -0.62716680888538012, -0.025870209593289673, -0.6471128780763804]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank0_float_rank0(self):
      arg0=Data(-0.667442478856,self.functionspace)
      arg0.setTaggedValue(1,-0.971360620043)
      arg1=-0.316390523151
      res=minimum(arg0,arg1)
      ref=Data(-0.667442478856,self.functionspace)
      ref.setTaggedValue(1,-0.971360620043)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank1_float_rank0(self):
      arg0=Data(numpy.array([0.73755562095077587, -0.078581728544633522]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.75869482157336776, 0.80029665233453717]))
      arg1=-0.0119653926614
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([-0.011965392661381857, -0.078581728544633522]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.75869482157336776, -0.011965392661381857]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[-0.075851199846881601, 0.088479024337185574, -0.20175236883406278, -0.23409621844831796, 
-0.50022432242802206], [-0.19458852808813076, -0.06827107878164318, 0.35397092585520018, 0.99755578098940134, 
0.36267334931554629], [0.065230569877553712, -0.44034579314783828, 0.67444615005481978, -0.72003737434899917, 
-0.35931497196936779], [-0.73789730385477803, 0.15705088168556669, 0.20571567851824546, -0.54048060121952068, 
-0.51903628888777797]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.30417027639257399, 0.73576532696162356, 0.95180759709207963, 0.6085197579435806, 
-0.34296186490748215], [0.71722168991111768, 0.83080157720272174, 0.86586561035264231, -0.55846463844230665, 
-0.60659289984021569], [-0.97089886134062553, 0.15252144565496462, 0.2278294815639923, -0.73411761774090811, 
0.11660625542498537], [0.20073946219200844, -0.17056120253222051, 0.93257284856405254, -0.84464260303668959, 
0.067569782156308378]]))
      arg1=-0.269425035974
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.26942503597426382, -0.26942503597426382, -0.26942503597426382, -0.26942503597426382, 
-0.50022432242802206], [-0.26942503597426382, -0.26942503597426382, -0.26942503597426382, -0.26942503597426382, 
-0.26942503597426382], [-0.26942503597426382, -0.44034579314783828, -0.26942503597426382, -0.72003737434899917, 
-0.35931497196936779], [-0.73789730385477803, -0.26942503597426382, -0.26942503597426382, -0.54048060121952068, 
-0.51903628888777797]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.26942503597426382, -0.26942503597426382, -0.26942503597426382, 
-0.26942503597426382, -0.34296186490748215], [-0.26942503597426382, -0.26942503597426382, -0.26942503597426382, 
-0.55846463844230665, -0.60659289984021569], [-0.97089886134062553, -0.26942503597426382, -0.26942503597426382, 
-0.73411761774090811, -0.26942503597426382], [-0.26942503597426382, -0.26942503597426382, -0.26942503597426382, 
-0.84464260303668959, -0.26942503597426382]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[0.78014064764525903, -0.022066619623113626], [-0.12647479080439017, 0.32996026199563477]], 
[[0.63152949513678691, 0.26884341823747238], [-0.0488086156035894, -0.19959833856629605]], [[-0.88150216946654902, 
0.1439700091372762], [-0.2877639880368863, 0.36048790086527616]], [[0.78058757638133658, 0.96793424211390744], 
[0.80683738576332842, 0.10665577652377523]], [[-0.99959344908994163, 0.15910797585814107], [-0.85343907283523124, 
0.56098524884934009]], [[0.94086643092500655, -0.30472017653688788], [-0.00015330292429460712, 
-0.10114863290697129]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-0.50924114644619167, 0.44445664673581997], [0.096469702165010052, 
-0.13771313267267726]], [[-0.242342574703869, -0.25782038289779607], [0.70169334565828656, -0.2693577214471834]], 
[[-0.15822206377518366, 0.33013943298044857], [0.81291783508677629, -0.82607219259490416]], [[-0.53686174517249841, 
-0.45589953783993598], [0.9543535873399922, 0.29832347367529266]], [[0.80960984004873859, -0.7135123966290986], 
[-0.42238226258644174, 0.14415461189902845]], [[-0.44062594460287063, -0.33740064063488662], [-0.013629950394130619, 
0.65069118522321268]]]))
      arg1=-0.83429301912
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[-0.83429301911994624, -0.83429301911994624], [-0.83429301911994624, -0.83429301911994624]], 
[[-0.83429301911994624, -0.83429301911994624], [-0.83429301911994624, -0.83429301911994624]], [[-0.88150216946654902, 
-0.83429301911994624], [-0.83429301911994624, -0.83429301911994624]], [[-0.83429301911994624, -0.83429301911994624], 
[-0.83429301911994624, -0.83429301911994624]], [[-0.99959344908994163, -0.83429301911994624], [-0.85343907283523124, 
-0.83429301911994624]], [[-0.83429301911994624, -0.83429301911994624], [-0.83429301911994624, 
-0.83429301911994624]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.83429301911994624, -0.83429301911994624], [-0.83429301911994624, 
-0.83429301911994624]], [[-0.83429301911994624, -0.83429301911994624], [-0.83429301911994624, -0.83429301911994624]], 
[[-0.83429301911994624, -0.83429301911994624], [-0.83429301911994624, -0.83429301911994624]], [[-0.83429301911994624, 
-0.83429301911994624], [-0.83429301911994624, -0.83429301911994624]], [[-0.83429301911994624, -0.83429301911994624], 
[-0.83429301911994624, -0.83429301911994624]], [[-0.83429301911994624, -0.83429301911994624], [-0.83429301911994624, 
-0.83429301911994624]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[-0.099202529792016492, -0.66176463043825517, -0.60473832393328264, -0.064795093040386664], 
[0.37631249491125196, 0.92334288346665261, -0.34017586254663179, -0.75501459388260916], [-0.97829515840377379, 
-0.0053347400745056284, -0.44822215951712385, 0.21628862760691892]], [[0.35375058641833435, 0.51221107665036114, 
0.13667125593603835, -0.82070373817977194], [0.98404604010482055, -0.022316698330294704, -0.98082878064924151, 
0.7630741093213762], [0.21550572645399235, 0.74697799437832635, -0.032467010500929039, -0.61540284516468668]]], 
[[[0.14238812954331936, -0.47129791386499531, 0.089495207271984345, 0.61419746167244127], [0.38509037706248628, 
0.85863800095009735, 0.46770654808452861, -0.65480196830448323], [-0.17075090018003736, -0.47769952250856251, 
0.52689760978106226, -0.55920878528675755]], [[0.76079226147472134, -0.27993125619618731, 0.95673376511094887, 
-0.040963210533838446], [-0.016926262480632115, -0.14447504279129286, 0.98586226019350409, 0.47147902210678549], 
[0.92638983397434393, -0.87847151881659102, 0.42157191467012711, 0.62664047281332635]]], [[[-0.69380305399032305, 
0.65534706998125558, 0.13481337574973029, 0.5203170879008101], [0.064189740965153952, 0.1617852874372292, 0.31130027994265452, 
-0.23762789967444742], [0.83084071973735574, 0.28947076140857764, -0.80980981119368423, -0.19236789408516142]], 
[[0.57762617183716336, -0.90890315620745432, 0.071305470942769844, -0.34737537851451261], [0.31798963235310262, 
-0.74482665976122808, 0.47004090467462678, 0.12786028345541434], [0.68109017784366133, -0.49693844015697874, 
-0.77807718596987496, -0.55122486057767195]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.18971879142667047, -0.96503204415646371, 0.93795741597993332, 
0.48106387625479652], [-0.44265794064266939, 0.14819852136622713, 0.87946034020190256, -0.27461308383462923], 
[-0.54903311465082005, -0.21192127038978104, -0.89233014041781944, 0.72624672224575404]], [[-0.16359905429141985, 
0.52131680241933376, 0.49446827358764667, 0.12143597732415068], [0.5300862786347873, -0.8366225404098111, -0.90375536282406155, 
0.84695904335817707], [0.69424651766708201, 0.47952347683582741, 0.4236652600658759, -0.90633701595345983]]], 
[[[-0.40039879031507697, 0.93157425443241859, 0.29952045251982895, -0.64746877862368502], [-0.34098675052449079, 
0.67776387723613429, 0.11524871919318858, -0.21728049807758265], [0.30497547065211439, -0.69716859399874975, 
0.40182819916980295, 0.97161882372408015]], [[-0.15891225934243502, -0.45574844163846318, -0.46787245863047056, 
0.85565903159964174], [-0.22910245777422289, -0.32644961681144724, -0.62120502825887436, -0.783430513001655], 
[0.95869637155765886, 0.70552970595856368, -0.15358355460290651, 0.16895269012852299]]], [[[-0.25340622647305056, 
0.13961016939476911, -0.66423651640726322, 0.031812218639510226], [0.97888825428566562, 0.75339476515895365, 
-0.85951212981063296, 0.77999552778788184], [0.19338594510546958, -0.49347392118131128, 0.52986089241122269, 
-0.7978167765039601]], [[0.030548591336885922, -0.5118767216564355, -0.79782912538526718, 0.73062218357850095], 
[-0.49370821423630717, -0.77333066884774793, 0.34512737321345655, -0.01951598609803451], [-0.089920519683897249, 
-0.83707509441892824, 0.04341884732484802, 0.2014353871746315]]]]))
      arg1=0.454407591055
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.099202529792016492, -0.66176463043825517, -0.60473832393328264, -0.064795093040386664], 
[0.37631249491125196, 0.45440759105514661, -0.34017586254663179, -0.75501459388260916], [-0.97829515840377379, 
-0.0053347400745056284, -0.44822215951712385, 0.21628862760691892]], [[0.35375058641833435, 0.45440759105514661, 
0.13667125593603835, -0.82070373817977194], [0.45440759105514661, -0.022316698330294704, -0.98082878064924151, 
0.45440759105514661], [0.21550572645399235, 0.45440759105514661, -0.032467010500929039, -0.61540284516468668]]], 
[[[0.14238812954331936, -0.47129791386499531, 0.089495207271984345, 0.45440759105514661], [0.38509037706248628, 
0.45440759105514661, 0.45440759105514661, -0.65480196830448323], [-0.17075090018003736, -0.47769952250856251, 
0.45440759105514661, -0.55920878528675755]], [[0.45440759105514661, -0.27993125619618731, 0.45440759105514661, 
-0.040963210533838446], [-0.016926262480632115, -0.14447504279129286, 0.45440759105514661, 0.45440759105514661], 
[0.45440759105514661, -0.87847151881659102, 0.42157191467012711, 0.45440759105514661]]], [[[-0.69380305399032305, 
0.45440759105514661, 0.13481337574973029, 0.45440759105514661], [0.064189740965153952, 0.1617852874372292, 0.31130027994265452, 
-0.23762789967444742], [0.45440759105514661, 0.28947076140857764, -0.80980981119368423, -0.19236789408516142]], 
[[0.45440759105514661, -0.90890315620745432, 0.071305470942769844, -0.34737537851451261], [0.31798963235310262, 
-0.74482665976122808, 0.45440759105514661, 0.12786028345541434], [0.45440759105514661, -0.49693844015697874, 
-0.77807718596987496, -0.55122486057767195]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.18971879142667047, -0.96503204415646371, 0.45440759105514661, 
0.45440759105514661], [-0.44265794064266939, 0.14819852136622713, 0.45440759105514661, -0.27461308383462923], 
[-0.54903311465082005, -0.21192127038978104, -0.89233014041781944, 0.45440759105514661]], [[-0.16359905429141985, 
0.45440759105514661, 0.45440759105514661, 0.12143597732415068], [0.45440759105514661, -0.8366225404098111, 
-0.90375536282406155, 0.45440759105514661], [0.45440759105514661, 0.45440759105514661, 0.4236652600658759, 
-0.90633701595345983]]], [[[-0.40039879031507697, 0.45440759105514661, 0.29952045251982895, -0.64746877862368502], 
[-0.34098675052449079, 0.45440759105514661, 0.11524871919318858, -0.21728049807758265], [0.30497547065211439, 
-0.69716859399874975, 0.40182819916980295, 0.45440759105514661]], [[-0.15891225934243502, -0.45574844163846318, 
-0.46787245863047056, 0.45440759105514661], [-0.22910245777422289, -0.32644961681144724, -0.62120502825887436, 
-0.783430513001655], [0.45440759105514661, 0.45440759105514661, -0.15358355460290651, 0.16895269012852299]]], 
[[[-0.25340622647305056, 0.13961016939476911, -0.66423651640726322, 0.031812218639510226], [0.45440759105514661, 
0.45440759105514661, -0.85951212981063296, 0.45440759105514661], [0.19338594510546958, -0.49347392118131128, 
0.45440759105514661, -0.7978167765039601]], [[0.030548591336885922, -0.5118767216564355, -0.79782912538526718, 
0.45440759105514661], [-0.49370821423630717, -0.77333066884774793, 0.34512737321345655, -0.01951598609803451], 
[-0.089920519683897249, -0.83707509441892824, 0.04341884732484802, 0.2014353871746315]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank0_array_rank0(self):
      arg0=Data(-0.513870332965,self.functionspace)
      arg0.setTaggedValue(1,-0.663142101825)
      arg1=numpy.array(-0.638085896136)
      res=minimum(arg0,arg1)
      ref=Data(-0.638085896136,self.functionspace)
      ref.setTaggedValue(1,-0.663142101825)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank1_array_rank0(self):
      arg0=Data(numpy.array([-0.83056185841012931, 0.71970256318466652]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.20919342024271859, 0.30125340008791412]))
      arg1=numpy.array(-0.880014313586)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([-0.88001431358572146, -0.88001431358572146]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.88001431358572146, -0.88001431358572146]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[-0.64144266770546077, -0.39012122865785837, -0.1991442350769288, 0.7487927796057201, 
0.47426891445834962], [0.85458880922226554, 0.18513690881109679, 0.34530252685133567, -0.002476007958266635, 
0.67027039822016832], [0.96820729039098397, 0.094152728632094229, 0.12208892206957755, -0.69408087607183822, 
-0.658032599929955], [-0.69159516571149915, -0.10806252520513504, 0.093139545761151687, -0.1691036297663111, 
-0.57628161607838213]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.40166987857676828, -0.53572647352931702, 0.9379672663964902, -0.43682288234908961, 
0.67409935248718411], [-0.87442575975951842, -0.48999660149037916, 0.68123702866819058, 0.25375161421877745, 
0.5829095853954307], [0.54342923377166996, 0.44774355315315151, 0.47137912154641426, 0.44760199967503511, 
-0.09088096452355221], [0.46841833035451463, 0.094505462528938322, -0.31426126803051502, 0.68574329908780474, 
0.020706643276932057]]))
      arg1=numpy.array(-0.492086966282)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.64144266770546077, -0.49208696628230753, -0.49208696628230753, -0.49208696628230753, 
-0.49208696628230753], [-0.49208696628230753, -0.49208696628230753, -0.49208696628230753, -0.49208696628230753, 
-0.49208696628230753], [-0.49208696628230753, -0.49208696628230753, -0.49208696628230753, -0.69408087607183822, 
-0.658032599929955], [-0.69159516571149915, -0.49208696628230753, -0.49208696628230753, -0.49208696628230753, 
-0.57628161607838213]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.49208696628230753, -0.53572647352931702, -0.49208696628230753, 
-0.49208696628230753, -0.49208696628230753], [-0.87442575975951842, -0.49208696628230753, -0.49208696628230753, 
-0.49208696628230753, -0.49208696628230753], [-0.49208696628230753, -0.49208696628230753, -0.49208696628230753, 
-0.49208696628230753, -0.49208696628230753], [-0.49208696628230753, -0.49208696628230753, -0.49208696628230753, 
-0.49208696628230753, -0.49208696628230753]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[-0.17605995414413944, 0.15420090127144026], [-0.13852500898712594, -0.13346890295080871]], 
[[0.70630799764581664, 0.87718192024708608], [0.49591204296245461, -0.37361420248017829]], [[0.40361527977561029, 
-0.35593196837221441], [-0.58586571706866031, -0.0044014923625939062]], [[0.038797433657362435, -0.92007112682390724], 
[-0.093054077101611732, -0.42395264412075373]], [[0.037322533325175433, 0.1699477535280236], [0.62608092873977061, 
-0.23235826620069777]], [[-0.13688663377052479, 0.81308230931969838], [-0.0039879410438432572, 
0.111020892017341]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.026877214965413065, 0.58216588107538669], [0.35383191861408259, 
-0.98617637072854292]], [[0.024622130194704717, 0.62942526986416425], [0.62878686824531482, 0.10890352163437322]], 
[[0.23254086363823467, 0.21215675267248213], [-0.68703998910496633, 0.94353576170304798]], [[-0.69095893261178176, 
0.023752482929776608], [0.58759514988960126, 0.96890111227320541]], [[-0.64179572747904579, -0.40015274245501398], 
[0.72385729120447695, -0.74111547099330788]], [[-0.53994347868078951, -0.80280253366603627], [0.96649085954806258, 
0.87031081336574645]]]))
      arg1=numpy.array(0.735656060786)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[-0.17605995414413944, 0.15420090127144026], [-0.13852500898712594, -0.13346890295080871]], 
[[0.70630799764581664, 0.73565606078621104], [0.49591204296245461, -0.37361420248017829]], [[0.40361527977561029, 
-0.35593196837221441], [-0.58586571706866031, -0.0044014923625939062]], [[0.038797433657362435, -0.92007112682390724], 
[-0.093054077101611732, -0.42395264412075373]], [[0.037322533325175433, 0.1699477535280236], [0.62608092873977061, 
-0.23235826620069777]], [[-0.13688663377052479, 0.73565606078621104], [-0.0039879410438432572, 
0.111020892017341]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.026877214965413065, 0.58216588107538669], [0.35383191861408259, 
-0.98617637072854292]], [[0.024622130194704717, 0.62942526986416425], [0.62878686824531482, 0.10890352163437322]], 
[[0.23254086363823467, 0.21215675267248213], [-0.68703998910496633, 0.73565606078621104]], [[-0.69095893261178176, 
0.023752482929776608], [0.58759514988960126, 0.73565606078621104]], [[-0.64179572747904579, -0.40015274245501398], 
[0.72385729120447695, -0.74111547099330788]], [[-0.53994347868078951, -0.80280253366603627], [0.73565606078621104, 
0.73565606078621104]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_taggedData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[0.2047676390152684, -0.51366423675699258, 0.73348098532392103, 0.40396804405503373], 
[-0.99516441752247009, 0.88016237431672173, -0.30452442849974748, -0.81798163741934093], [-0.9253118419026658, 
0.33038973620517531, 0.40656417414740353, 0.074526348348877614]], [[0.15425818343499809, -0.52621930956714724, 
0.4494916200184067, 0.22329564571570448], [0.28023895540611954, -0.50822594175105928, -0.70554576315955919, 
0.75790674439895067], [0.8640935408469792, 0.43178610302111098, -0.034864338043917975, 0.3898148174295053]]], 
[[[-0.6630354479234104, 0.69277025770215706, -0.35972989418459234, 0.36775629807464605], [-0.3561000700866237, 
0.44839508770882919, 0.63293536874246858, 0.37064437544784234], [-0.24498891228450281, 0.2158529444048447, 
-0.015784051298041391, -0.85277704393977238]], [[0.9115942146572853, -0.75843063573434555, -0.032769417350635432, 
-0.94094687001931221], [0.9960445659969257, 0.39070240966455283, -0.46297635984320196, -0.60040536586341298], 
[0.59354852210010489, 0.47949169441162143, 0.95685697611728315, -0.39130176536359551]]], [[[-0.52999430760839306, 
0.598191428157266, -0.6325778056411957, 0.11553057190169191], [0.34278702138450634, 0.086469059326779618, -0.34750762960161841, 
-0.12564523513665882], [0.77927577322825026, -0.64603450918280503, 0.54520149761315029, 0.48920986931343413]], 
[[-0.33562837295883363, -0.20798065757317863, -0.85195331127183538, 0.49571321797877554], [0.51668607477478634, 
0.0077847962447119556, 0.95597768941646377, 0.46047158090414175], [0.34277516780143258, -0.82409580053985043, 
0.77700104914659351, -0.49775812874417702]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.21786294342522927, 0.4387799839492259, -0.56381373742049612, 
0.52229972564516425], [-0.64302572769645328, -0.33605379329898777, 0.60498507069727214, 0.12310484544546685], 
[0.12857125988632956, 0.47070054344033041, -0.47485074965633478, -0.34516793499048637]], [[0.83910049645679408, 
0.29369101742038817, 0.82896603594825335, 0.4954901273348693], [0.83205511476008431, 0.57404852269270124, 
-0.020554010032003633, 0.024264571645504995], [-0.878344915986216, 0.8602921817759086, -0.61160716168120799, 
0.7686728183295144]]], [[[-0.19148443461549092, 0.92065803733451346, 0.44071277644321483, 0.83300221795550633], 
[-0.45929788655542425, 0.067320617397752969, -0.022956308268418102, -0.53882833487390136], [0.28281214418840883, 
0.52787200331522355, 0.71200306460413554, -0.24078059029800647]], [[0.53075173238499862, -0.37051637288983996, 
0.49527491913745347, -0.7285434046861019], [-0.065624194443034689, -0.89811569737974972, 0.53072245562265108, 
-0.5669225083950642], [0.66729514696789005, 0.97598752364769403, 0.35435865815651457, 0.28355620125225589]]], 
[[[0.29281934164144086, 0.90779978926320815, -0.72167564074762125, 0.4421311203903302], [0.35716067192901946, 
0.040794453371597816, -0.92311527655763204, -0.44007113384869245], [-0.90923643970824464, 0.16164578934446827, 
0.52076869530570802, 0.6043486282348749]], [[-0.45314040151585977, -0.47501767202633971, 0.081569836060568335, 
-0.89922933100476965], [-0.085135409753766655, -0.88190914745145532, -0.20530353975559112, 0.11021787159190288], 
[-0.578655825768712, -0.16646435873100751, -0.42476392490292381, -0.7965750933366087]]]]))
      arg1=numpy.array(0.126230321493)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[[0.12623032149265745, -0.51366423675699258, 0.12623032149265745, 0.12623032149265745], 
[-0.99516441752247009, 0.12623032149265745, -0.30452442849974748, -0.81798163741934093], [-0.9253118419026658, 
0.12623032149265745, 0.12623032149265745, 0.074526348348877614]], [[0.12623032149265745, -0.52621930956714724, 
0.12623032149265745, 0.12623032149265745], [0.12623032149265745, -0.50822594175105928, -0.70554576315955919, 
0.12623032149265745], [0.12623032149265745, 0.12623032149265745, -0.034864338043917975, 0.12623032149265745]]], 
[[[-0.6630354479234104, 0.12623032149265745, -0.35972989418459234, 0.12623032149265745], [-0.3561000700866237, 
0.12623032149265745, 0.12623032149265745, 0.12623032149265745], [-0.24498891228450281, 0.12623032149265745, 
-0.015784051298041391, -0.85277704393977238]], [[0.12623032149265745, -0.75843063573434555, -0.032769417350635432, 
-0.94094687001931221], [0.12623032149265745, 0.12623032149265745, -0.46297635984320196, -0.60040536586341298], 
[0.12623032149265745, 0.12623032149265745, 0.12623032149265745, -0.39130176536359551]]], [[[-0.52999430760839306, 
0.12623032149265745, -0.6325778056411957, 0.11553057190169191], [0.12623032149265745, 0.086469059326779618, 
-0.34750762960161841, -0.12564523513665882], [0.12623032149265745, -0.64603450918280503, 0.12623032149265745, 
0.12623032149265745]], [[-0.33562837295883363, -0.20798065757317863, -0.85195331127183538, 0.12623032149265745], 
[0.12623032149265745, 0.0077847962447119556, 0.12623032149265745, 0.12623032149265745], [0.12623032149265745, 
-0.82409580053985043, 0.12623032149265745, -0.49775812874417702]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.21786294342522927, 0.12623032149265745, -0.56381373742049612, 
0.12623032149265745], [-0.64302572769645328, -0.33605379329898777, 0.12623032149265745, 0.12310484544546685], 
[0.12623032149265745, 0.12623032149265745, -0.47485074965633478, -0.34516793499048637]], [[0.12623032149265745, 
0.12623032149265745, 0.12623032149265745, 0.12623032149265745], [0.12623032149265745, 0.12623032149265745, 
-0.020554010032003633, 0.024264571645504995], [-0.878344915986216, 0.12623032149265745, -0.61160716168120799, 
0.12623032149265745]]], [[[-0.19148443461549092, 0.12623032149265745, 0.12623032149265745, 0.12623032149265745], 
[-0.45929788655542425, 0.067320617397752969, -0.022956308268418102, -0.53882833487390136], [0.12623032149265745, 
0.12623032149265745, 0.12623032149265745, -0.24078059029800647]], [[0.12623032149265745, -0.37051637288983996, 
0.12623032149265745, -0.7285434046861019], [-0.065624194443034689, -0.89811569737974972, 0.12623032149265745, 
-0.5669225083950642], [0.12623032149265745, 0.12623032149265745, 0.12623032149265745, 0.12623032149265745]]], 
[[[0.12623032149265745, 0.12623032149265745, -0.72167564074762125, 0.12623032149265745], [0.12623032149265745, 
0.040794453371597816, -0.92311527655763204, -0.44007113384869245], [-0.90923643970824464, 0.12623032149265745, 
0.12623032149265745, 0.12623032149265745]], [[-0.45314040151585977, -0.47501767202633971, 0.081569836060568335, 
-0.89922933100476965], [-0.085135409753766655, -0.88190914745145532, -0.20530353975559112, 0.11021787159190288], 
[-0.578655825768712, -0.16646435873100751, -0.42476392490292381, -0.7965750933366087]]]]))
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape o