///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/characteristic.h"
#include "rheolef/piola.h"
#include "rheolef/field_evaluate.h"
#include "rheolef/rheostream.h"
#include "rheolef/band.h"

namespace rheolef { 

// ===========================================================================
// externs
// ===========================================================================
template<class T, class M>
void
interpolate_pass1_symbolic (
    const geo_basic<T,M>&                   omega,
    const array<point_basic<T>,M>&          x,
    const array<geo_element::size_type,M>&  ix2dis_ie, // x has been already localized in K
          array<index_set,M>&               ie2dis_ix, // K -> list of ix
          array<point_basic<T>,M>&          hat_y);

template<class T, class M>
void
interpolate_pass2_valued (
    const field_basic<T,M>&                uh,
    const array<point_basic<T>,M>&         x,
    const array<index_set,M>&              ie2dis_ix, // K -> list of ix
    const array<point_basic<T>,M>&         hat_y,     // ix -> hat_y
          array<T,M>&                      ux);

#ifdef TO_CLEAN
// ===========================================================================
// characteristic_element:
// compute an elementary lk = riesz(compose(field,characteristic)) on K
// suitable for form_assembly
// ===========================================================================
template<class T, class M>
class characteristic_element {
public:
  typedef geo_element::size_type size_type;
  typedef typename array<T,M>::const_iterator const_iterator;
  characteristic_element (
    const space_basic<T,M>&       Xh,
    const quadrature<T>&          quad,
    const basis_on_pointset<T>&   piola_on_quad,
    const_iterator                uq_K);
  void operator() (const geo_element& K, const int&, std::vector<T>& lk) const;

  // for compatibility with form_assembly
  bool is_on_band() const { return false; }
  band_basic<T,M> get_band() const { return band_basic<T,M>(); }
protected:
// pointers on more global variables:
  mutable const_iterator          _uq_K;
  const geo_basic<T,M>&           _omega;
  basis_basic<T>                  _u_basis;
  const quadrature<T>&            _quad;
  const basis_on_pointset<T>&     _piola_on_quad;
// data:
  basis_on_pointset<T>            _u_basis_on_quad;
// usefull infos:
  size_type                       _n_comp;
  size_type                       _dim;
  size_type                       _map_dim;
  space_constant::coordinate_type _sys_coord;
// local variables:
  mutable tensor_basic<T>         _DF;
  mutable std::vector<size_type>  _dis_inod;
};
// allocator:
template<class T, class M>
characteristic_element<T,M>::characteristic_element (
    const space_basic<T,M>&       Xh,
    const quadrature<T>&          quad,
    const basis_on_pointset<T>&   piola_on_quad,
    const_iterator                uq_K)
  : _uq_K            (uq_K),
    _omega           (Xh.get_geo()),
    _u_basis         (Xh.get_numbering().get_basis()),
    _quad            (quad),
    _piola_on_quad   (piola_on_quad),
    _u_basis_on_quad (quad, _u_basis),
    _n_comp          (Xh.valued_tag() == space_constant::scalar ? 1 : Xh.size()),
    _dim             (_omega.dimension()),
    _map_dim         (_omega.map_dimension()),
    _sys_coord       (_omega.coordinate_system()),
    _DF              (),
    _dis_inod        ()
{
  switch (Xh.valued_tag()) {
    case space_constant::scalar:
    case space_constant::vector: break; // tensor & cie: the code should also work, but not yet tested
    default: error_macro ("characteristic: unsupported "<<Xh.valued()<<"-valued field");
  }
}
// interface for field_assembly:
template<class T, class M>
void
characteristic_element<T,M>::operator() (const geo_element& K, const int&, std::vector<T>& lk) const
{
  _omega.dis_inod (K, _dis_inod);
  size_type nloc_comp_idof = _u_basis.size(K);
  size_type nloc_idof      = _n_comp*nloc_comp_idof;
  size_type nloc_comp_q    = _quad.size(K);
  size_type comp_nq        = _omega.size()*nloc_comp_q;
  lk.resize (nloc_idof);
  std::fill (lk.begin(), lk.end(), T(0.));
  size_type comp_q = 0;
  for (typename quadrature<T>::const_iterator
               first_quad = _quad.begin(K),
                last_quad = _quad.end(K);
    	       first_quad != last_quad; first_quad++, comp_q++) {
    jacobian_piola_transformation (_omega, _piola_on_quad, K, _dis_inod, comp_q, _DF);
    T det_DF = det_jacobian_piola_transformation (_DF, _dim, _map_dim);
    T wq = 1;
    if (_sys_coord != space_constant::cartesian) {
      point_basic<T> xq = piola_transformation (_omega, _piola_on_quad, K, _dis_inod, comp_q);
      wq *= weight_coordinate_system (_sys_coord, xq);
    }
    wq *= det_DF;
    wq *= (*first_quad).w;
    size_type iloc_comp_idof = 0;
    for (typename basis_on_pointset<T>::const_iterator
	   	first_phi = _u_basis_on_quad.begin (K, comp_q),
           	last_phi  = _u_basis_on_quad.end   (K, comp_q);
                first_phi != last_phi; first_phi++, iloc_comp_idof++) {
      for (size_type i_comp = 0; i_comp < _n_comp; i_comp++) {
        size_type iloc_idof = iloc_comp_idof + i_comp*nloc_comp_idof;
        size_type q         = comp_q         + i_comp*comp_nq;
        Float fwq = _uq_K[q]*wq;
        lk [iloc_idof] += (*first_phi)*fwq;
      }
    }
  }
  _uq_K += nloc_comp_q;
}
#endif // TO_CLEAN
// ===========================================================================
// riesz(Xh,compose (phi_h,X)) returns a field:
// => specialisation of the riesz function:
// It done in two passes:
// - the first one is symbolic: localize the moved quadrature points yq=xq+dh(xq)
//   this is done one time for all 
// - the second pass is valued: compte uh(yq) and then the Riesz representer
// ===========================================================================
template <class T, class M>
void
riesz_pass1_symbolic (
 // input :
    const space_basic<T,M>&             Xh,
    const field_basic<T,M>&             dh,
 // i/o :
          quadrature_option_type&       qopt,
 // output :
          quadrature<T>&                quad,
          basis_on_pointset<T>&         piola_on_quad,
          array<index_set,M>&           ie2dis_ix,
          array<point_basic<T>,M>&      hat_y,
          array<point_basic<T>,M >&     yq)
{
  // ----------------------------------------------------------------------
  // 1) set the quadrature formulae
  // ----------------------------------------------------------------------
  typedef typename space_basic<T,M>::size_type size_type;
  quad.set_order  (qopt.get_order());
  quad.set_family (qopt.get_family());
trace_macro ("quadrature : " << quad.get_family_name() << " order " << quad.get_order());
  const geo_basic<T,M>& omega = Xh.get_geo();
  check_macro (omega == dh.get_geo(), "characteristic: unsupported different meshes for flow ("
		<< dh.get_geo().name() << ") and convected field (" << omega.name() << ")");
  if (omega.order() > 1) {
    warning_macro ("characteristic: high-order curved elements not yet supported (treated as first order)");
  }
  piola_on_quad.set (quad, omega.get_piola_basis());
  // ----------------------------------------------------------------------
  // 2) size of the the array of all quadrature nodes
  // ----------------------------------------------------------------------
  // NOTE: since these nodes are moved (convected) by the flow associated to the characteristic
  // they can change from partition and go to an element managed by another proc
  // thus, in order to group comms, a whole array of quadrature point is allocated:
  // In sequential mode (or with only one proc), this can do in an element by element way
  // with less memory.
  size_type dim     = omega.dimension();
  size_type map_dim = omega.map_dimension();
  size_type     sz = 0;
  size_type dis_sz = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(map_dim);
                 variant < reference_element:: last_variant_by_dimension(map_dim); variant++) {
    distributor ige_ownership = omega.sizes().ownership_by_variant [variant];
    if (ige_ownership.dis_size() == 0) continue;
    reference_element hat_K (variant);
    size_type nq = piola_on_quad.size (hat_K);
        sz += nq*ige_ownership.size();
    dis_sz += nq*ige_ownership.dis_size();
  }
  distributor xq_ownership (dis_sz, omega.comm(), sz);
  // ----------------------------------------------------------------------
  // 3) build the array of all quadrature nodes; xq
  // ----------------------------------------------------------------------
  array<point_basic<T>,M> xq (xq_ownership);
  std::vector<size_type> dis_inod;
  for (size_type ixq = 0, ie = 0, ne = omega.size(); ie < ne; ie++) {
    const geo_element& K = omega[ie];
    omega.dis_inod (K, dis_inod);
    for (size_type q = 0, nq = piola_on_quad.size(K); q < nq; q++, ixq++) {
      xq[ixq] = piola_transformation (omega, piola_on_quad, K, dis_inod, q);
    }
  }
  // ----------------------------------------------------------------------
  // 4) interpolate dh on the xq nodes: dq(i)=dh(xq(i))
  // ----------------------------------------------------------------------
  dh.dis_dof_update();
  array<point_basic<T>,M> dq        (xq_ownership);
  array<size_type,M>      ix2dis_ie (xq_ownership);
  std::vector<size_type> d_dis_idof;
  const space_basic<T,M>& Dh = dh.get_space();
  const numbering<T,M>& d_fem = Dh.get_numbering();
  basis_on_pointset<T> d_basis_on_quad (quad, d_fem.get_basis());
  for (size_type ixq = 0, ie = 0, ne = omega.size(); ie < ne; ie++) {
    const geo_element& K = omega[ie];
    Dh.dis_idof (K, d_dis_idof);
    for (size_type q = 0, nq = piola_on_quad.size(K); q < nq; q++, ixq++) {
      dq[ixq] = vector_field_evaluate (dh, d_basis_on_quad, K, d_dis_idof, q);
      ix2dis_ie[ixq] = ie; // will be a guest for locate(yq)
    }
  }
  // ----------------------------------------------------------------------
  // 5) build the array of all moved quadrature nodes yq(i) = xq(i)+dq(i)
  //    with the constraint to remain inside Omega
  // ----------------------------------------------------------------------
  yq.resize (xq_ownership);
  omega.trace_move (xq, dq, ix2dis_ie, yq);
  // ----------------------------------------------------------------------
  // 6.a) symbolic interpolation-like pass: localize the yq nodes
  // ----------------------------------------------------------------------
  interpolate_pass1_symbolic (omega, yq, ix2dis_ie, ie2dis_ix, hat_y);
}
#ifdef TO_CLEAN
template <class T, class M>
field_basic<T,M>
riesz_pass2_valued (
    const space_basic<T,M>&             Xh,
    const field_basic<T,M>&             uh,
    const quadrature<T>&                quad,
    const basis_on_pointset<T>&         piola_on_quad,
    const array<index_set,M>&           ie2dis_ix,
    const array<point_basic<T>,M>&      hat_y,
    const array<point_basic<T>,M >&     yq)
{
  const geo_basic<T,M>& omega = Xh.get_geo();
  // ----------------------------------------------------------------------
  // 6.b) valued pass: evaluate uh(yq)
  // ----------------------------------------------------------------------
  array<T,M> uq (yq.ownership());
  interpolate_pass2_valued   (uh, yq, ie2dis_ix, hat_y, uq);
  // ----------------------------------------------------------------------
  // 7) field_assembly: build the riesz representer
  // ----------------------------------------------------------------------
  characteristic_element<T,M> charac_e (Xh, quad, piola_on_quad, uq.begin());
  int dummy_fct;
  field_basic<T,M> lh(Xh,0);
  field_assembly (omega, charac_e, dummy_fct, lh);
  return lh;
}
#endif // TO_CLEAN
// ========================================================================
// symbolic pass 1 is stored one time for all in the characteristic class
// ========================================================================
template<class T, class M>
const characteristic_on_quadrature<T,M>&
characteristic_rep<T,M>::get_pre_computed (
    const space_basic<T,M>&       Xh,
    const field_basic<T,M>&       dh,
    const quadrature_option_type& qopt1) const
{
  // get a good quadrature if not set
  typedef typename space_basic<T,M>::size_type size_type;
  quadrature_option_type qopt = qopt1;
  if (qopt.get_order() == 0 ||
      qopt.get_order() == std::numeric_limits<quadrature_option_type::size_type>::max()) {
    size_type quad_order = Xh.get_numbering().get_basis().degree();
    if (Xh.get_geo().coordinate_system() != space_constant::cartesian) quad_order++;
    qopt.set_order (quad_order);
  }
  if (qopt.get_family() == quadrature_option_type::max_family) {
    qopt.set_family (quadrature_option_type::gauss_lobatto);
  }
  // search if already used & memorized
  std::string label = qopt.get_family_name() + "(" + itos(qopt.get_order()) + ")";
  typename map_type::const_iterator iter = _coq_map.find (label);
  if (iter != _coq_map.end()) {
    return (*iter).second;
  }
  // build a new one & memorize it
  characteristic_on_quadrature<T,M> coq (qopt);
  characteristic_on_quadrature_rep<T,M>& coq_r = coq.data();
  riesz_pass1_symbolic (Xh, dh,
    coq_r._qopt,
    coq_r._quad,
    coq_r._piola_on_quad,
    coq_r._ie2dis_ix,
    coq_r._hat_y, 
    coq_r._yq);
  std::pair <typename map_type::iterator,bool> iter_b = _coq_map.insert (std::make_pair(label,coq));
  typename map_type::iterator iter2 = iter_b.first;
  return (*iter2).second;
}
#ifdef TO_CLEAN
// ========================================================================
// valued pass 2 is called by the riesz function
// ========================================================================
template <class T, class M>
field_basic<T,M>
riesz (
    const space_basic<T,M>&             Xh,
    const field_o_characteristic<T,M>&  f,
    quadrature_option_type              qopt)
{
  const field_basic<T,M>& dh = f.get_characteristic().get_displacement();
  const characteristic_on_quadrature<T,M>& coq = f.get_characteristic().get_pre_computed (Xh, dh, qopt);
  const characteristic_on_quadrature_rep<T,M>& coq_r = coq.data();
  return riesz_pass2_valued   (Xh, f.get_field(), 
		             coq_r._quad, coq_r._piola_on_quad, coq_r._ie2dis_ix, coq_r._hat_y, coq_r._yq);
}
#endif // TO_CLEAN
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T,M)                             \
template class characteristic_rep<T,M>;				\

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI

}// namespace rheolef
