#!/usr/bin/perl -w

##############################################################################
# File       : mc.pl
# Description: Message Text File translator
#  Copyright (C) Intel Corporation, 2006 - 2007.
##############################################################################

use strict;

# Define command type constants
my (    $TYPE_COMMENT,
	$TYPE_HEADER,
	$TYPE_MESSAGE) = (1,2,3);

my (	$CATEGORY,
        $EVENT)	= (1,2);
# Read arguments from the command line.
usage() unless ($#ARGV == 1);

my ($fullFileName, $fileName);
my $CATEGORY_CNT = $ARGV[0];

#taking the file name without the mc extension
if ($ARGV[1] =~ m/(.*)\.mc/) {
  $fullFileName = $ARGV[1];
  $fileName     = $1;
}
else {
  print "Input must have mc extension\n";
  usage();
}

###############################################################################
# Main starts here
###############################################################################

# Open the mc file for reading
open(FILE, "$fullFileName") || die "Failed to open: $fullFileName";

# First pass: Read the file into the memory, Remove comment, empty lines
my ($command, @commands);
while (<FILE>)
{
  chomp;

  $command = $_;

  # Ignore if empty line
  next if ($command =~ m/^\s*$/);

  push(@commands, $command);
}
close(FILE);

my @headerOutput;         # Here we write the compiled file.
my @srcOutput;

# Second pass: Go over each command, and parse each line.
my ($catCounter, $evtCounter)=(0,0);
my (@events, @categories);

my ($type, $jump) = (undef, 1);
my $i=0;
while ($i <= $#commands){

  # Parse the current line
  $type = parseLine($commands[$i]);
  $jump = 1;
  if (! defined $type) {
  	$i++;
        next;
  }
  if ($type == $TYPE_COMMENT) {
     if ($commands[$i] =~ m/^\s*;(.*)\s*$/i) {
     	push(@headerOutput, $1);
     }
  }
  elsif ($type == $TYPE_MESSAGE) {
  	my %curMsg;
 	($jump, %curMsg) = parseMsg($i, @commands);
        if ( $catCounter < $CATEGORY_CNT) {
	        push @headerOutput, writeMsgHeader(\%curMsg, $CATEGORY);
                push @categories, {%curMsg};
                $catCounter++;
        }
        else {
	        push @headerOutput, writeMsgHeader(\%curMsg, $EVENT);
        	push @events, { %curMsg};
                $evtCounter++;
        }
  }
  $i += $jump;
}

# write the src file.
push @srcOutput, "//  Copyright (C) Intel Corporation, 2005 - 2007";
push @srcOutput, "//  Auto generated Linux Event Messages\n\n";
push @srcOutput, "#include \"EventMessages.h\"";
push @srcOutput, "#include <syslog.h>";
push @srcOutput, "#include \"" . $fileName . "Linux.h\"\n";
push @srcOutput, "EventMessages::EventMessages() {";
push @srcOutput, "\n\t// Initializing categories";
my $ncat = $#categories + 1;
push @srcOutput, "\t_categories.resize($ncat);";
for (my $i=0; $i <= $#categories ; $i++) {
	push @srcOutput, writeCatagory($categories[$i]);
}
my $nevt = $#events + 1;
push @srcOutput, "\n\t// Initializing events";
push @srcOutput, "	_events.resize($nevt);";
for (my $i=0; $i <= $#events ; $i++) {
	push @srcOutput, writeEvent( $events[$i]);
}
push @srcOutput, "}";

# Open output files and write the hack code.
open(HEAD_OUT_FILE,">$fileName"."Linux.h") || die "Can't open output $fileName.h";
print HEAD_OUT_FILE join("\n",@headerOutput);
print HEAD_OUT_FILE "\n";
close(HEAD_OUT_FILE);

open(SRC_OUT_FILE,">EventMessages.cpp") || die "Can't open output EventMessages.cpp";
print SRC_OUT_FILE join("\n",@srcOutput);
print SRC_OUT_FILE "\n";
close(SRC_OUT_FILE);

###############################################################################
# Sub        : parseLine
# Description: Decide if the line is comment or message.
###############################################################################
sub parseLine {
  my $command = shift;

  # Check for comment
  return $TYPE_COMMENT if ($command =~ m/^\s*;(.*)\s*$/i);

  # Check for message
  return $TYPE_MESSAGE if ($command =~ m/^\s*MessageId=.*$/i);

  # Error wrong syntax
  return (undef, undef);
}

###############################################################################
# Sub        : parseMsg
# Description: Get the msg.
# Return     : The message, and the jump for i
###############################################################################
sub parseMsg {
    my $index = shift;
    my @commands = @_;
    my %msg = ( Severity => "Success",
    		Facility => "Application",
                Language => "English",
                Text	 => "" );
    my $j = 0;
    while (1) {
    	my $c = $commands[$index+$j];
        die "MessageId=[+number] is not supportes" if ($c =~ m/^\s*MessageId=\+.*$/i);
        if ($c =~ m/^\s*MessageId=(\d*)\s*$/i) {
     		$msg{"MessageId"} = $1;
	}
	elsif ($c =~ m/^\s*Severity=(.*)\s*$/i) {
  		$msg{"Severity"} = $1;
  	}
  	elsif ($c =~ m/^\s*Facility=(.*)\s*$/i) {
  		$msg{"Facility"} = $1;
	}
        elsif ($c =~ m/^\s*SymbolicName=(.*)\s*$/i) {
  		$msg{"SymbolicName"} = $1;
  	}
  	elsif ($c =~ m/^\s*OutputBase=(.*)\s*$/i) {
  		# ignore output base, work with decimal only.
  	}
  	elsif ($c =~ m/^\s*Language=(.*)\s*$/i) {
 		# Support only English
        	print "Language $1 is not supported.\n" if ($1 ne "English");
  	}
  	elsif ($c =~ m/^\s*\./i) {
        	last;
        }
        else {
        	$msg{"Text"} .= $c;
        }
        $j++;
    }
    return ($j,%msg);
}

###############################################################################
# Sub        : writeMsgHeader
# Description: Write the header defines of the message
###############################################################################
sub writeMsgHeader {
  my $msg = shift;
  my $type = shift;

  if ($type == $CATEGORY) {
	return "#define\t$msg->{SymbolicName}\t((unsigned short)$catCounter)\n";
  }
  elsif ($type == $EVENT) {
        return "#define\t$msg->{SymbolicName}\t((unsigned long)$evtCounter)\n";
  }
}

###############################################################################
# Sub        : writeCatagory
# Description: Add the category to the categories.
###############################################################################
sub writeCatagory {
  my $m = shift;

  return "\t_categories[$m->{SymbolicName}] = \"$m->{Text}\";";
}

###############################################################################
# Sub        : writeEvent
# Description: Add event.
###############################################################################
sub writeEvent {
  my $m = shift;

  my $priority = "LOG_INFO";
  $priority = "LOG_NOTICE"	if ($m->{Severity} =~ m/Success/i);
  $priority = "LOG_INFO"	if ($m->{Severity} =~ m/Information/i);
  $priority = "LOG_WARNING"	if ($m->{Severity} =~ m/Warning/i);
  $priority = "LOG_ERR" 	if ($m->{Severity} =~ m/Error/i);

  my $message = $m->{Text};
  $message =~ s/%%/%/g;
  $message =~ s/%\./\./g;
  return "\t_events[$m->{SymbolicName}] = SingleEvent( $priority, $m->{MessageId}, \"$message\");";
}
###############################################################################
# Sub        : usage
# Description: Print a usage message.
###############################################################################
sub usage {
  print "Usage: mc.pl <category count> file_name.mc\n";
  exit;
}
