#! /bin/bash
#
# This script contains all the fixups run when systemd package is installed or
# updated.
#

warn() {
	echo >&2 "warning: $@"
}

#
# /etc/machine-id might have been created writeable incorrectly (boo#1092269).
#
# Note: run at each package update.
#
fix_machine_id_perms() {
	if [ "$(stat -c%a /etc/machine-id)" != 444 ]; then
		echo "Incorrect file mode bits for /etc/machine-id which should be 0444, fixing..."
		chmod 444 /etc/machine-id
	fi
}

#
# v228 wrongly set world writable suid root permissions on timestamp files used
# by permanent timers. Fix the timestamps that might have been created by the
# affected versions of systemd (bsc#1020601).
#
# Note: run at each package update.
#
fix_bsc_1020601() {
	for stamp in $(ls /var/lib/systemd/timers/stamp-*.timer 2>/dev/null); do
		chmod 0644 $stamp
	done

	# Same for user lingering created by logind.
	for username in $(ls /var/lib/systemd/linger/* 2>/dev/null); do
		chmod 0644 $username
	done
}

#
# Due to the fact that DynamicUser= was turned ON during v235 and then switched
# back to off in v240, /var/lib/systemd/timesync might be a symlink pointing to
# /var/lib/private/systemd/timesync, which is inaccessible for systemd-timesync
# user as /var/lib/private is 0700 root:root, see
# https://github.com/systemd/systemd/issues/11329 for details.
#
# Note: only TW might be affected by this bug.
# Note: run at each package update.
#
fix_issue_11329() {
	if [ -L /var/lib/systemd/timesync ]; then
		rm /var/lib/systemd/timesync
		mv /var/lib/private/systemd/timesync /var/lib/systemd/timesync
	fi
}

#
# We don't ship after-local.service anymore however as a courtesy we install a
# copy in /etc for users who are relying on it.
#
# Note: should run only once since it is conditionalized on the presence of
# %{_unitdir}/after-local.service
#
drop_after_local_support() {
	if [ -x /etc/init.d/after.local ] &&
	   [ -f /usr/lib/systemd/system/after-local.service ]; then
		echo "after-local.service is no more provided by systemd but a copy has been installed in /etc"
		cp /usr/lib/systemd/system/after-local.service /etc/systemd/system/
		ln -s ../after-local.service /etc/systemd/system/multi-user.target.wants/after-local.service
	fi
}

#
# We have stopped shipping the main config files in /etc but we don't try to
# clean them up automatically as it can have unexepected side effects
# (bsc#1226415). Instead we simply suggest users to convert them (if they exist)
# into drop-ins.
#
# Note: run at each package update
#
check_config_files () {
	config_files=(systemd/journald.conf systemd/logind.conf systemd/system.conf systemd/user.conf
		      systemd/pstore.conf systemd/sleep.conf systemd/timesyncd.conf systemd/coredump.conf
		      systemd/journal-remote.conf systemd/journal-upload.conf systemd/networkd.conf
		      systemd/resolved.conf systemd/oomd.conf udev/iocost.conf udev/udev.conf)

	for f in ${config_files[*]}; do
		[ -e /etc/$f ] || continue

		cat >&2 <<EOF
Main configuration files are deprecated in favor of drop-ins.
Hence, we suggest that you remove /etc/$f if it doesn't contain any customization, or convert it into drop-in otherwise.
For more details, please visit https://en.opensuse.org/Systemd#Configuration.
EOF
	done
}

# On systems using the old btrfs layout, where /var was not a separate
# subvolume, convert /var/lib/machines into a subvolume suitable for snapper to
# perform snapshots, rollbacks, see bsc#992573.
#
# The conversion might only be problematic for openSUSE distros (TW/Factory)
# where the subvolume was created at the wrong place (via tmpfiles for example)
# and it got populated before we had time to fix it. In this case we'll let the
# user fix it manually.
#
# On SLE12 this subvolume was only introduced during the upgrade from v210 to
# v228 (ie SLE12-SP[01] -> SLE12-SP2+ when we added this workaround hence no
# user should had time to populate it. Note that the subvolume is still created
# at the wrong place due to the call to tmpfiles_create macro in the %post
# section however it's empty so again we shouldn't face any issue to convert it.
#
# In order to avoid ugly dependencies added in systemd package, this function
# should only be called during package updates when mksubvolume(8) is
# available. During installation, /var/lib/machines is supposed to be created by
# the installer now.
#
# See bsc#992573.
#
is_btrfs_subvolume() {
	# On btrfs subvolumes always have the inode 256
	test $(stat --format=%i "$1") -eq 256
}

# This assumes the directory/subvol to remove is empty.
rm_subvolume_or_directory() {
	is_btrfs_subvolume "$1" && {
		btrfs subvolume delete "$1"
		return
	}
	rmdir "$1"
}

fix_machines_subvol() {
	local tagfile=/var/lib/systemd/rpm/container-machines_subvol

	#
	# This test is probably redundant with the ones below but we don't want
	# to run this on transactional systems since /var has always been a
	# separate subvolume on such systems.
	#
	if [ -n "${TRANSACTIONAL_UPDATE}" ]; then
		return 0
	fi

	if [ -e $tagfile ]; then
		return 0
	fi
	touch $tagfile

	#
	# If there's already an entry in fstab for /var/lib/machines, it
	# means that:
	#
	#   - the installer initialized /var/lib/machines correctly (default)
	#   - we already fixed it
	#   - the sysadmin added it manually
	#
	# In any cases we should return.
	#
	# Note: we can't simply check if /var/lib/machines has been mounted
	# because an offline update might be in progress (see bsc#1030290).
	#
	if mount --fake /var/lib/machines 2>/dev/null; then
		return
	fi

	#
	# If there is already an entry in fstab for /var, it means that:
	#
	#   - the system has a seperate /var subvolume (default from Feb 2018)
	#   - the system has a seperate /var partition
	#
	# In any case we should return.
	#
	if mount --fake /var 2>/dev/null; then
		return
	fi

	#
	# If something is already mounted don't try to fix anything, it's been
	# done manually by the sysadmin.
	#
	if mountpoint -q /var/lib/machines; then
		return
	fi

	#
	# Let's try to figure out if the current filesystem uses a Snapper
	# BTRFS specific layout. Note that TW uses a different layout than
	# SLE...
	#
	# FIXME: not sure if it's correct, reliable or optimal.
	#
	case $(findmnt -nr -t btrfs -o FSROOT / 2>/dev/null) in
	*.snapshots/*/snapshot*)
		;;
	*)
		return 0
	esac

	if test -d /var/lib/machines; then
		#
		# Ok, we're on a system supporting rollbacks and
		# /var/lib/machines is not a subvolume remotely mounted so it
		# cannot be suitable for systems supporting rollback. Fix it.
		#
		echo "Making /var/lib/machines suitable for rollbacks..."

		type mksubvolume >/dev/null 2>&1 || {
			warn "mksubvolume(8) is not installed, aborting."
			return 1
		}
		test "$(ls -A /var/lib/machines/)" && {
			warn "/var/lib/machines is not empty, aborting."
			return 1
		}

		echo "Deleting empty /var/lib/machines directory/subvolume"
		rm_subvolume_or_directory /var/lib/machines || {
			warn "fail to delete /var/lib/machines"
			return 1
		}
	fi

	# At this point /var/lib/machines shouldn't exist.
	echo "Creating /var/lib/machines subvolume suitable for rollbacks."
	mksubvolume /var/lib/machines
}


r=0
fix_machine_id_perms || r=1
fix_bsc_1020601 || r=1
fix_issue_11329 || r=1
drop_after_local_support || r=1

if [ $1 -gt 1 ]; then
	# During upgrade
	fix_machines_subvol || {
		warn "Please fix the initialization of /var/lib/machines manually."
		r=1
	}
fi

exit $r
